/*
 * ESE, a HyperText Transfer Protocol server
 * Copyright (C) 1996-2001 Akira Higuchi <a-higuti@math.sci.hokudai.ac.jp>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "esehttpd.h"

static void
eh_cgi_set_env (eh_strhash_t *ht, const char *key, const char *val)
{
  char *k;
  eh_strhash_entry_t *ent;
  k = x_strdup (key);
  ent = eh_strhash_find_create (ht, k);
  if (ent->key != k) {
    /* don't overwrite */
    x_free (k);
    return;
  }
  ent->key = k;
  ent->val = val ? x_strdup (val) : x_strdup ("");
}


static void
eh_cgi_pass_env (eh_strhash_t *ht, const char *name)
{
  char *v;
  v = getenv (name);
  if (v)
    eh_cgi_set_env (ht, name, v);
}

static void
eh_cgi_pass_env_all (eh_strhash_t *ht, eh_config_vhost_t *ecv)
{
  char **ps;
  if (ecv->passenv == NULL)
    return;
  for (ps = ecv->passenv; *ps; ps++) {
    eh_cgi_pass_env (ht, *ps);
  }
}

static const struct {
  const char *varname, *header;
} pass_headers[] = {
  { "HTTP_ACCEPT", "accept" },
  { "HTTP_ACCEPT_CHARSET", "accept-charset" },
  { "HTTP_ACCEPT_ENCODING", "accept-encoding" },
  { "HTTP_ACCEPT_LANGUAGE", "accept-language" },
  { "HTTP_CONNECTION", "connection" },
  { "HTTP_COOKIE", "cookie" },
  { "HTTP_COOKIE2", "cookie2" },
  { "HTTP_HOST", "host" },
  { "HTTP_PRAGMA", "pragma" },
  { "HTTP_REFERER", "referer" },
  { "HTTP_USER_AGENT", "user-agent" },
};

static const char *const pass_envs[] = {
  "PATH", "SHELL",
};

static void
eh_cgi_set_http_headers_env_one (const char *key, const char *val, void *data)
{
  eh_strhash_t *ht = (eh_strhash_t *)data;
  char *varname;
  size_t i;
  if (strcmp (key, "authorization") == 0 ||
      strcmp (key, "proxy-authorization") == 0)
    return;
  varname = x_strdup_printf ("HTTP_%s", key);
  for (i = 5; varname[i]; i++) {
    if (varname[i] == '-') {
      varname[i] = '_';
    } else if (isalpha(varname[i]) || isdigit(varname[i])) {
      varname[i] = toupper (varname[i]);
    } else {
      x_free (varname);
      return;
    }
  }
  eh_cgi_set_env (ht, varname, val);
  x_free (varname);
}

static void
eh_cgicommon_ht_on_delete_entry (eh_strhash_entry_t *ent)
{
  x_free (ent->key);
  x_free (ent->val);
}

static void
eh_cgicommon_append_strarr_one (eh_strhash_entry_t *ent, void *data)
{
  char ***pp = (char ***)data;
  **pp = x_strdup_printf ("%s=%s", (char *)ent->key, (char *)ent->val);
  (*pp)++;
}

static char **
eh_cgicommon_make_env_strarr (eh_strhash_t *ht)
{
  char **envp, **p;
  int size;
  size = ht->num_ent_total;
  envp = (char **)x_malloc ((size + 1) * sizeof (*envp));
  p = envp;
  eh_strhash_foreach (ht, eh_cgicommon_append_strarr_one, &p);
  assert (p == (envp + size));
  p[0] = NULL;
  return envp;
}

char **
eh_cgicommon_make_env (const eh_request_t *er, const eh_connection_t *ec)
{
  char *str;
  eh_strhash_t *ht;
  char **strarr;
  int i;
  unsigned long raddr;
  unsigned short rport;
  raddr = ntohl (ec->client_addr.sin_addr.s_addr);
  rport = ntohs (ec->client_addr.sin_port);
  
  ht = eh_strhash_new (53, eh_cgicommon_ht_on_delete_entry);

  eh_cgi_set_env (ht, "ESEHTTPD_VERSION", VERSION);
  
  /* environment variables defined in CGI/1.1 specification */
  eh_cgi_set_env (ht, "SERVER_SOFTWARE", ESEHTTPD_NAME);
  if (er->econf_vhost_ref->servername)
    eh_cgi_set_env (ht, "SERVER_NAME", er->econf_vhost_ref->servername);
  else
    eh_cgi_set_env (ht, "SERVER_NAME", "localhost"); /* use gethostname? */
  eh_cgi_set_env (ht, "GATEWAY_INTERFACE", "CGI/1.1");
  str = x_strdup_printf ("HTTP/1.%d", er->http_version_minor);
  eh_cgi_set_env (ht, "SERVER_PROTOCOL", str);
  x_free (str);
  str = x_strdup_printf ("%d", er->econf_vhost_ref->port);
  eh_cgi_set_env (ht, "SERVER_PORT", str);
  x_free (str);
  eh_cgi_set_env (ht, "REQUEST_METHOD", er->method_str);
  /* PATH_INFO */
  /* PATH_TRANSLATED */
  eh_cgi_set_env (ht, "SCRIPT_NAME", er->url);
  if (er->query_string)
    eh_cgi_set_env (ht, "QUERY_STRING", er->query_string);
  else
    eh_cgi_set_env (ht, "QUERY_STRING", "");
  /* REMOTE_HOST */
  str = x_strdup_printf ("%lu.%lu.%lu.%lu",
			 (raddr >> 24) & 0xff,
			 (raddr >> 16) & 0xff,
			 (raddr >>  8) & 0xff,
			 (raddr >>  0) & 0xff);
  eh_cgi_set_env (ht, "REMOTE_ADDR", str);
  x_free (str);
  str = x_strdup_printf ("%u", rport);
  eh_cgi_set_env (ht, "REMOTE_PORT", str);
  x_free (str);
  if (er->remote_user) {
    eh_cgi_set_env (ht, "AUTH_TYPE", "Basic");
    eh_cgi_set_env (ht, "REMOTE_USER", er->remote_user);
  }
  /* REMOTE_IDENT */
  if (er->headers.predef.content_type)
    eh_cgi_set_env (ht, "CONTENT_TYPE", er->headers.predef.content_type);
  if (er->headers.predef.content_length)
    eh_cgi_set_env (ht, "CONTENT_LENGTH", er->headers.predef.content_length);

  /* apache sets the following environment variables also */
  /* DOCUMENT_ROOT */
  /* REQUEST_URI */
  /* SCRIPT_FILENAME */
  /* SERVER_ADDR */
  if (er->econf_vhost_ref->serveradmin)
    eh_cgi_set_env (ht, "SERVER_ADMIN", er->econf_vhost_ref->serveradmin);
  else
    eh_cgi_set_env (ht, "SERVER_ADMIN", "");
  /* SERVER_SIGNATURE */

  /* set HTTP_* environment variables */
  eh_headers_forall (&er->headers, eh_cgi_set_http_headers_env_one, ht);
  
#if 0
  for (i = 0; i < sizeof (pass_headers) / sizeof (pass_headers[0]); i++) {
    char *val;
    val = eh_headers_strdup_find (&er->headers, pass_headers[i].header);
    eh_cgi_set_env (ht, pass_headers[i].varname, val);
    if (val)
      x_free (val);
  }
#endif

  eh_cgi_pass_env_all (ht, er->econf_vhost_ref);
  for (i = 0; (size_t)i < sizeof (pass_envs) / sizeof (pass_envs[0]); i++) {
    eh_cgi_pass_env (ht, pass_envs[i]);
  }
  
  strarr = eh_cgicommon_make_env_strarr (ht);
  eh_strhash_delete (ht);
  return strarr;
}

void
eh_cgicommon_reply (eh_connection_t *ec, eh_method_t method,
		    char *buf, size_t buflen)
{
  char *p;
  eh_headers_t eh;
  if ((p = eh_find_end_of_header (buf, buflen)) != NULL) {
    assert (p[-1] == '\0');
    memset (&eh, 0, sizeof (eh));
    eh_headers_initnc (&eh, buf, p);
    if (eh.predef.location) {
      char *headers_str;
      eh_debug ("redirect location = %s", eh.predef.location);
      headers_str = eh_headers_strdup_getall (&eh);
      eh_debug ("redirect: headers_str = [%s]", headers_str);
      eh_connection_append_wvec_response (ec, method, "302",
					  headers_str, NULL, 0);
      eh_connection_set_request_events (ec, 0x444);
      x_free (headers_str);
    } else {
      char status_str[10] = "200";
      int status_str_len;
      status_str_len = eh.predef.status ?
	strcspn (eh.predef.status, " \t") : 0;
      if (status_str_len > 3) {
	eh_log (EH_LOG_WARNING,
		"CGI program returns an unknown status code: %s",
		eh.predef.status);
	eh_connection_append_wvec_response (ec, method, "500",
					    NULL, NULL, 0);
	eh_connection_set_request_events (ec, 0x445);
      } else {
	/* sane reply */
	char *headers_str;
	size_t body_len, content_length;
	if (eh.predef.status) {
	  memcpy (status_str, eh.predef.status, status_str_len);
	  status_str[status_str_len] = '\0';
	}
	eh_debug ("body = %s", p);
	body_len = buflen - (p - buf);
	content_length = 0;
	if (eh.predef.content_length) {
	  eh_parse_sizestr (eh.predef.content_length, &content_length);
	}
	eh_headers_remove (&eh, "status");
	eh_headers_remove (&eh, "connection");
	eh_headers_remove (&eh, "content-length");
#ifdef DEBUG_COOKIE
	{
	  char *c;
	  c = eh_headers_strdup_find (&eh, "set-cookie");
	  if (c) {
	    eh_log (EH_LOG_INFO, "set-cookie: %s", c);
	    x_free (c);
	  }
	}
#endif
	headers_str = eh_headers_strdup_getall (&eh);
	eh_connection_append_wvec_response_with_content_length
	  (ec, method, status_str, headers_str, p, body_len, content_length);
	eh_connection_set_request_events (ec, 0x446);
	x_free (headers_str);
      }
    }
    eh_headers_discard (&eh);
  } else {
    /* header not found */
    char *s, *str;
    str = x_strdup ("CGI program returns a corrupt message\n--------\n");
    s = x_strndup (buf, buflen);
    x_str_append (&str, s);
    eh_log (EH_LOG_INFO, "%s:\n%s", ec->current_request.filename, str);
    x_free (s);
    if (ec->app_backref->econf->sendcgierrors) {
      eh_connection_append_wvec_response (ec, method, "500",
					  "Content-Type: text/plain\r\n",
					  str, strlen (str));
    } else {
      eh_connection_append_wvec_response (ec, method, "500", NULL, NULL, 0);
    }
    x_free (str);
    eh_connection_set_request_events (ec, 0x447);
  }
}

int
eh_cgicommon_chdir (const char *filename)
{
  char *fname, *p;
  fname = x_strdup (filename);
  p = strrchr (fname, '/');
  if (p) {
    p[0] = '\0';
    if (chdir (fname) < 0) {
      x_free (fname);
      eh_log_perror (EH_LOG_FATAL, fname);
      return -1;
    }
  }
  x_free (fname);
  return 0;
}

void
eh_cgicommon_response_500 (eh_connection_t *ec, eh_method_t method,
			   const char *body)
{
  eh_log (EH_LOG_INFO, "%s:\n%s",
	  ec->current_request.filename, body);
  if (ec->app_backref->econf->sendcgierrors) {
    eh_connection_append_wvec_response (ec, method, "500",
					"Content-Type: text/plain\r\n",
					body, strlen (body));
  } else {
    eh_connection_append_wvec_response (ec, method, "500", NULL, NULL, 0);
  }
}
