/*
 * ESE, a HyperText Transfer Protocol server
 * Copyright (C) 1996-2001 Akira Higuchi <a-higuti@math.sci.hokudai.ac.jp>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "esehttpd.h"

#define LOCAL_DEBUG_DO(x)

static void eh_connection_on_event (eh_fd_t *ef);
static void eh_connection_on_timer (eh_fd_t *ef, int graceful_shutdown);

static void
eh_connection_accesslog_log (eh_connection_t *ec)
{
  eh_config_vhost_t *vhost;
  eh_request_t *req = &ec->current_request;
  static char debugstr[81];
#ifdef DEBUG
  snprintf (debugstr, 80, "%sf%d/e%lu/r%lu/w%lu",
	    ec->sslcon ? "S/" : "",
	    ec->client_pfd->pfd.fd,
	    (unsigned long)ec->event_count,
	    (unsigned long)ec->pollin_count,
	    (unsigned long)ec->pollout_count);
  debugstr[80] = '\0';
#else
  debugstr[0] = ec->sslcon ? 'S' : '-';
  debugstr[1] = '\0';
#endif
  vhost = req->econf_vhost_ref
    ? req->econf_vhost_ref : ec->vserver_backref->default_vhost;
  eh_accesslog_log (&ec->accesslog, vhost->access_log, vhost->log_combined,
		    vhost->nolog_ht,
		    req->headers.predef.user_agent,
		    req->headers.predef.referer,
		    debugstr);
}

static void
eh_connection_on_delete (eh_fd_t *ef)
{
  size_t i;
  eh_connection_t *ec = (eh_connection_t *)ef->user_data;
  eh_debug ("%d %p", ef->pfd.fd, ef);
#ifdef DEBUG
  if (ec->wrote_bytes == 0) {
    eh_fd_debug_trace (ef, 0x2f0000);
  }
#endif
  for (i = 0; i < ec->wvec_len; i++)
    if (ec->wvec[i].destructor)
      (*(ec->wvec[i].destructor))(ec->wvec[i].context);
  ec->wvec = (eh_wvec_t *)x_realloc (ec->wvec, 0);
  ec->wvec_len = 0;
  ec->ssl_writev_buffer = x_realloc (ec->ssl_writev_buffer, 0);
  ec->ssl_writev_buffer_alloclen = 0;
  ec->ssl_writev_buffer_len = 0;
  if (ec->rhandler)
    eh_rhandler_delete (ec);
  if (ec->current_request.not_finished)
    eh_request_discard (&ec->current_request);
  eh_strbuf_discard (&ec->readbuf);
  eh_accesslog_discard (&ec->accesslog);
  if (ec->sslcon) {
    eh_debug ("SSL_free");
    SSL_free (ec->sslcon);
  }
  ec->app_backref = NULL;
  x_free (ec);
}

static void
eh_connection_timeout_reset (eh_connection_t *ec)
{
  ec->timeout = eh_fd_get_time () + ec->connection_timeout_sec;
}

static short
eh_connection_get_state (eh_connection_t *ec)
{
  short events = 0;
  if (ec->shutdown_flag) {
    events = 0;
  } else if (ec->wvec_len > 0) {
    events = POLLOUT;
  } else {
    if (ec->rhandler == NULL) {
      if (eh_strbuf_readok (&ec->readbuf))
	events = POLLIN;
    } else {
      if (ec->rhandler->body_length_left > 0)
	events = POLLIN;
    }
  }
  eh_debug ("%d", events);
  return events;
}

static void
eh_connection_ssl_accept_finished (eh_connection_t *ec)
{
  X509 *peer;
  long errnum;
  ec->ssl_renegotiate_is_in_progress = 0;
  ec->ssl_peer = NULL;
  ec->ssl_x509_v_ok = 0;
  if ((peer = SSL_get_peer_certificate (ec->sslcon)) != NULL) {
    ec->ssl_peer = peer;
    eh_debug ("SSL_get_peer_certificate returns %p", peer);
    if ((errnum = SSL_get_verify_result (ec->sslcon)) == X509_V_OK) {
      ec->ssl_x509_v_ok = 1;
      eh_debug ("SSL_get_verify_result returns X509_V_OK");
    } else {
      eh_debug ("SSL_get_verify_result failed: %s",
		X509_verify_cert_error_string (errnum));
    }
  } else {
    eh_debug ("SSL_get_peer_certificate returns NULL");
  }
}

static int
eh_connection_ssl_error (eh_connection_t *ec, eh_fd_t *ef, int retval)
{
  int err;
  long err_long;
  eh_debug ("SSL_get_error");
  err = SSL_get_error (ec->sslcon, retval);
  switch (err) {
  case SSL_ERROR_WANT_READ:
    eh_debug ("WANT_READ");
    eh_fd_clear_revents (ef, POLLIN);
    ec->ssl_want_read = 1;
    break;
  case SSL_ERROR_WANT_WRITE:
    eh_debug ("WANT_WRITE");
    eh_fd_clear_revents (ef, POLLOUT);
    ec->ssl_want_write = 1;
    break;
  case SSL_ERROR_NONE:
    break;
  case SSL_ERROR_ZERO_RETURN:
    return 0;
  case SSL_ERROR_SYSCALL:
    eh_debug ("ERROR_SYSCALL");
    err_long = ERR_get_error ();
    if (err_long == 0) {
      eh_debug ("SSL EOF");
      return -1;
    }
    eh_debug ("ERR_get_error returns %ld", err_long);
    eh_log_ssl_error (EH_LOG_DEBUG, "IO");
    return -1;
  default:
    eh_log (EH_LOG_DEBUG, "err = %d", err);
    eh_log_ssl_error (EH_LOG_DEBUG, "IO");
    return -1;
  }
  LOCAL_DEBUG_DO (ec->last_ssl_error = err);
  return 1;
}

void
eh_connection_set_request_events (eh_connection_t *ec, int debug_info)
{
  eh_fd_t *ef = ec->client_pfd;
  short events;
  
  events = eh_connection_get_state (ec);

  if (ec->sslcon == NULL) {
    /* not SSL */
    if (ec->shutdown_flag == 0 && events == 0 && ec->rhandler == NULL) {
      eh_fd_debug_trace (ef, 0x210000 + debug_info);
      if (ec->is_http_1_0 == 0 ||
	  ec->last_request_has_body ||
	  ec->app_backref->econf->forcelingeringclose) {
	if (!ec->eof_flag) {
	  /* doing lingering-close */
	  if (shutdown (ef->pfd.fd, SHUT_WR) < 0) {
	    ec->eof_flag = 1;
	  }
	}
      } else {
	/* close connection without lingering-close */
	ec->eof_flag = 1;
      }
      ec->shutdown_flag = 1;
      ec->dummy_read_limit = ec->app_backref->econf->lingeringcloselimit;
    }
    if (ec->shutdown_flag) {
      if (ec->eof_flag) {
	eh_fd_delete_request (ef);
	return;
      }
      events = POLLIN;
    }
    eh_debug ("new request event %d", events);
#if 0
    if (!ec->eof_flag)
      events |= POLLIN;
#endif
    eh_fd_request_events (ef, events);
  } else {
    /* SSL */
    short request_events;
    if (events == 0 && ec->rhandler == NULL) {
      int r;
      ec->ssl_shutdown_is_in_progress = 1;
      eh_connection_timeout_reset (ec);
      /* 1: bidirectional shutdown has done,
	 0: close notify is sent,
	 -1: wouldblock or other errors */
      r = SSL_shutdown (ec->sslcon);
      eh_debug ("SSL_shutdown: %d", r);
      if (r == 0) {
	/* try bidirectional shutdown */
	r = SSL_shutdown (ec->sslcon);
	eh_debug ("SSL_shutdown(2nd): %d", r);
      }
      if (r <= 0) {
	if (eh_connection_ssl_error (ec, ec->client_pfd, r) <= 0) {
	  eh_fd_delete_request (ec->client_pfd);
	  return;
	}
	/* cont */
      } else {
	eh_fd_delete_request (ef);
	return;
      }
    }
    if (ec->ssl_want_read) {
      request_events = POLLIN;
    } else if (ec->ssl_want_write) {
      request_events = POLLOUT;
    } else {
      request_events = events;
    }
    eh_fd_request_events (ef, request_events);
  }
}

#ifdef DEBUG
static void
eh_connection_show_status (eh_fd_t *ef, char **strp)
{
  eh_connection_t *ec = (eh_connection_t *)ef->user_data;
  x_append_printf (strp, "wvec=%d shut=%c eof=%c pin=%d pout=%d",
		   ec->wvec_len,
		   ec->shutdown_flag ? 'Y' : 'N',
		   ec->eof_flag ? 'Y' : 'N',
		   ec->pollin_count, ec->pollout_count);
}
#endif

void
eh_connection_new (int fd, eh_app_t *app, eh_vserver_t *vs,
		   struct sockaddr_in *client_addr,
		   int ssl_flag)
{
  eh_connection_t *ec;
  SSL *sslcon = NULL;
  eh_debug ("");
  if (ssl_flag) {
    eh_debug ("SSL: SSL_new");
    sslcon = SSL_new (vs->sslctx);
    if (sslcon == NULL) {
      eh_log (EH_LOG_WARNING, "SSL_new failed");
      close (fd);
      return;
    }
  }
  ec = (eh_connection_t *)x_malloc (sizeof (*ec));
  if (ec == NULL) {
    eh_debug ("SSL: SSL_free");
    SSL_free (sslcon);
    return;
  }
  memset (ec, 0, sizeof (*ec));
  ec->rconn = eh_rconn_close;
  ec->client_addr = *client_addr;
  ec->app_backref = app;
  ec->vserver_backref = vs;
  eh_strbuf_init (&ec->readbuf, (size_t)-1);
  ec->client_pfd = eh_fd_new (fd, eh_connection_on_event,
			      eh_connection_on_timer,
			      eh_connection_on_delete,
			      (void *)ec);
#ifdef DEBUG
  eh_fd_set_show_status_callback (ec->client_pfd, eh_connection_show_status);
#endif

  if (sslcon) {
    ec->sslcon = sslcon;
    SSL_set_fd (ec->sslcon, ec->client_pfd->pfd.fd);
    SSL_set_accept_state (ec->sslcon);
  }
  ec->connection_timeout_sec = vs->connection_timeout;
  eh_accesslog_init (&ec->accesslog, client_addr);
  eh_fd_debug_trace (ec->client_pfd, 0x200000);
  eh_connection_timeout_reset (ec);
  eh_connection_set_request_events (ec, 0x10);
  return;
}

static void
eh_connection_read_finish (eh_connection_t *ec)
{
  eh_debug ("");
  eh_strbuf_set_read_limit (&ec->readbuf, 0);
  if (ec->rhandler) {
    eh_rhandler_delete (ec);
  }
}

static void
eh_connection_request_discard (eh_connection_t *ec)
{
  eh_connection_accesslog_log (ec);
  eh_request_discard (&ec->current_request);
  if (ec->delayed_strbuf_remove_len > 0) {
    eh_strbuf_remove (&ec->readbuf, ec->delayed_strbuf_remove_len);
    ec->delayed_strbuf_remove_len = 0;
  }
}

void
eh_connection_request_finish (eh_connection_t *ec)
{
  eh_debug ("");
  eh_connection_request_discard (ec);
  if (ec->rconn != eh_rconn_keepalive) {
    eh_connection_read_finish (ec);
  } else {
    if (ec->rhandler) {
      eh_rhandler_delete (ec);
    }
  }
}

void
eh_connection_write_finish (eh_connection_t *ec)
{
  size_t i;
  eh_debug ("");
  if (ec->current_request.not_finished)
    eh_connection_request_discard (ec);
  eh_strbuf_set_read_limit (&ec->readbuf, 0);
  if (ec->rhandler) {
    eh_rhandler_delete (ec);
  }
  for (i = 0; i < ec->wvec_len; i++)
    if (ec->wvec[i].destructor)
      (*(ec->wvec[i].destructor))(ec->wvec[i].context);
  ec->wvec = (eh_wvec_t *)x_realloc (ec->wvec, 0);
  ec->wvec_len = 0;
  ec->wvec_alloclen = 0;
}

void
eh_connection_append_wvec (eh_connection_t *ec, void *buf, size_t blen,
			   void (*destructor)(void *context),
			   void *context)
{
  size_t i;
  assert (buf);
  if (ec->wvec_len + 1 >= ec->wvec_alloclen) {
    ec->wvec_alloclen += 10;
    ec->wvec = (eh_wvec_t *)
      x_realloc (ec->wvec, (ec->wvec_alloclen + 1) * sizeof (ec->wvec[0]));
  }
  i = ec->wvec_len;
  ec->wvec[i].wvec.iov_base = buf;
  ec->wvec[i].wvec.iov_len = blen;
  ec->wvec[i].destructor = destructor;
  ec->wvec[i].context = context;
  ec->wvec_len++;
  eh_debug ("append_wvec: new wvec_len = %d", ec->wvec_len);
  return;
}

static int
eh_connection_ssl_writev (eh_connection_t *ec)
{
  int r;
  assert (ec->sslcon);
  if (ec->ssl_writev_buffer_len == 0) {
    /* this is the first try to SSL_write */
    eh_wvec_t *wvec;
    size_t len, l;
    size_t i;
    wvec = ec->wvec;
    for (len = 0, i = 0; i < ec->wvec_len; i++)
      len += wvec[i].wvec.iov_len;
    if (ec->ssl_writev_buffer_alloclen < len) {
      ec->ssl_writev_buffer_alloclen = len;
      ec->ssl_writev_buffer = x_realloc (ec->ssl_writev_buffer,
					 ec->ssl_writev_buffer_alloclen);
    }
    ec->ssl_writev_buffer_len = len;
    for (l = 0, i = 0; i < ec->wvec_len; i++) {
      memcpy ((char *)ec->ssl_writev_buffer + l,
	      wvec[i].wvec.iov_base, wvec[i].wvec.iov_len);
      l += wvec[i].wvec.iov_len;
    }
  }
  eh_debug ("SSL_write %p %d", ec->ssl_writev_buffer,
	    (int)ec->ssl_writev_buffer_len);
  r = SSL_write (ec->sslcon, (const char *)ec->ssl_writev_buffer,
		 ec->ssl_writev_buffer_len);
  return r;
}

static void
eh_connection_on_write (eh_connection_t *ec, eh_fd_t *ef)
{
  size_t i, wlen;
  int r;

#ifdef DEBUG
  ec->pollout_count++;
#endif
  
  if (ec->wvec_len == 0) {
    return;
  }

  LOCAL_DEBUG_DO (ec->last_action = 2);
  if (ec->sslcon) {
    struct iovec *iv;
    iv = &ec->wvec[0].wvec;
    if (ec->wvec_len == 1) {
      eh_debug ("SSL_write");
      r = SSL_write (ec->sslcon, (const char *)iv->iov_base, iv->iov_len);
    } else {
      r = eh_connection_ssl_writev (ec);
    }
    if (r <= 0) {
      /* EWOULDBLOCK or fatal */
      LOCAL_DEBUG_DO (ec->last_action_result = r);
      if (eh_connection_ssl_error (ec, ef, r) <= 0) {
	eh_fd_delete_request (ef);
	return;
      }
      eh_connection_timeout_reset (ec);
      goto finish_cont;
    }
    ec->ssl_writev_buffer_len = 0;
  } else {
    if (ec->wvec_len == 1) {
      struct iovec *iv;
      iv = &ec->wvec[0].wvec;
      r = write (ec->client_pfd->pfd.fd, iv->iov_base, iv->iov_len);
    } else {
      struct iovec *iv;
      iv = (struct iovec *)x_alloca (ec->wvec_len * sizeof (*iv));
      for (i = 0; i < ec->wvec_len; i++)
	iv[i] = ec->wvec[i].wvec;
      r = writev (ec->client_pfd->pfd.fd, iv, ec->wvec_len);
      x_alloca_free (iv);
    }
    LOCAL_DEBUG_DO (ec->last_action_result = r);
    if (r <= 0) {
      if (r < 0 && (errno == EWOULDBLOCK || errno == EAGAIN)) {
	eh_debug ("write/writev: would block");
	eh_fd_clear_revents (ef, POLLOUT);
	goto finish_cont;
      } else if (r < 0 && errno == EINTR) {
	goto finish_cont;
      } else {
	eh_log_perror (EH_LOG_DEBUG, "write");
	// TODO: close connection gracefully
	eh_fd_delete_request (ef);
	return;
      }
    }
  }
  LOCAL_DEBUG_DO (ec->last_action_result = r);
  eh_debug ("wrote %d bytes", r);
#ifdef DEBUG
  ec->wrote_bytes += r;
#endif
  wlen = r;
  for (i = 0; i < ec->wvec_len; i++) {
    if (wlen >= ec->wvec[i].wvec.iov_len) {
      /* ok, delete this */
      if (ec->wvec[i].destructor)
	(*(ec->wvec[i].destructor))(ec->wvec[i].context);
      wlen -= ec->wvec[i].wvec.iov_len;
    } else {
      /* this buffer has not yet written completely */
      eh_debug ("wvec[%u]: %u -> %u",
		i,
		ec->wvec[i].wvec.iov_len,
		ec->wvec[i].wvec.iov_len - wlen);
      (char *)ec->wvec[i].wvec.iov_base += wlen;
      ec->wvec[i].wvec.iov_len -= wlen;
      break;
    }
  }
  if (i > 0) {
    size_t j;
    for (j = i; j < ec->wvec_len; j++)
      ec->wvec[j - i] = ec->wvec[j];
  }
  ec->wvec_len -= i;
  eh_debug ("new wvec_len = %u", ec->wvec_len);
  
 finish_cont:
  return;
}

static void
eh_connection_on_read (eh_connection_t *ec, eh_fd_t *ef)
{
  int r, len_accepted;
  char *rbuf, *p, *q;
  size_t rbuflen;
  eh_config_global_t *ecg;

#ifdef DEBUG
  ec->pollin_count++;
#endif

  ecg = ec->app_backref->econf;
  
  if (ec->rhandler) {
    char buffer[EH_READ_MAX];
    size_t rlen;
    rlen = ec->rhandler->body_length_left;
    if (rlen > EH_READ_MAX) rlen = EH_READ_MAX;
    if (rlen == 0) {
      eh_fd_debug_trace (ef, 0x240000);
      return;
    }
    if (ec->sslcon) {
      eh_debug ("SSL_read");
      if ((r = SSL_read (ec->sslcon, buffer, rlen)) <= 0)
	goto ssl_error;
    } else {
      if ((r = read (ec->client_pfd->pfd.fd, buffer, rlen)) <= 0)
	goto read_error;
    }
    eh_rhandler_on_read_request_body (ec, buffer, r);
    return;
  }
  LOCAL_DEBUG_DO (ec->last_action = 1);
  if (ec->sslcon) {
    if (ec->current_request.not_finished) {
      /* we expect renegotiation will occur. we must not call
	 eh_strbuf_ssl_read_append here because ec->readbuf.buffer
	 must not changed. in ec->current_request, there are many
	 pointers refering the buffer. */
      char c;
      eh_debug ("SSL_read: dummy");
      if ((r = SSL_read (ec->sslcon, &c, 1)) <= 0)
	goto ssl_error;
      eh_log (EH_LOG_WARNING,
	      "SSL renegotiation is expected. closing connection");
      eh_fd_delete_request (ef);
      return;
    } else {
      if ((r = eh_strbuf_ssl_read_append (&ec->readbuf, ec->sslcon)) <= 0)
	goto ssl_error;
    }
  } else {
    if ((r = eh_strbuf_read_append (&ec->readbuf, ec->client_pfd->pfd.fd))
	<= 0)
      goto read_error;
  }
  LOCAL_DEBUG_DO (ec->last_action_result = r);
  eh_debug ("read %d rbuflen = %d", r, ec->readbuf.buffer_len);
  eh_fd_debug_trace (ef, 0x250000 + r);
  eh_debug ("buffer: %s", ec->readbuf.buffer);
  eh_connection_timeout_reset (ec);

  rbuf = ec->readbuf.buffer;
  rbuflen = ec->readbuf.buffer_len;
  for (p = rbuf, len_accepted = 0;
       rbuf + rbuflen - p > 0 &&
	 (q = eh_find_end_of_header (p, rbuf + rbuflen - p)) != NULL;
       p = q) {
    /* the range [p..q) contains the request headers */
    size_t len;
    assert (q[-1] == '\0');
    num_requests++;
    LOCAL_DEBUG_DO (ec->last_action = 9);
    if (q - p > ecg->requestheaderlimit) {
      /* the request header is too long. close immediately. */
      eh_debug ("delete 1");
      eh_fd_delete_request (ef);
      return;
    }
    len = eh_connection_reply (ec, p - rbuf, q - p, rbuf + rbuflen - q);
    /* len may be longer than 'q - p' if it has a request-body. */
    len_accepted += len;
    q = p + len;
    if (ec->ssl_renegotiate_is_in_progress)
      break;
    if (ec->rhandler)
      break;
    if (ec->readbuf.read_limit == 0) {
      /* already finished to read */
      break;
    }
  }
  eh_debug ("accepted length: %d\n", len_accepted);
  if (len_accepted) {
    if (ec->current_request.not_finished) {
      /* some 'const char *' fields in eh_request_t refers the readbuf,
	 and therefore we can't remove the region now. */
      ec->delayed_strbuf_remove_len = len_accepted;
    } else {
      eh_strbuf_remove (&ec->readbuf, len_accepted);
      if (ec->readbuf.buffer_len > (size_t)ecg->requestheaderlimit) {
	/* the request header is too long. close immediately. */
	eh_debug ("delete 2");
	eh_fd_delete_request (ef);
	return;
      }
    }
  }

  if (ec->wvec_len > 0) {
#ifndef DEBUG_FILECACHE_DELAYED_REMOVE
    /* try to write */
    eh_connection_on_write (ec, ef);
    return;
#endif
  }
  
  return;

 read_error:
  LOCAL_DEBUG_DO (ec->last_action_result = r);
  if (r < 0 && (errno == EAGAIN || errno == EWOULDBLOCK)) {
    eh_debug ("%d wouldblock", ef->pfd.fd);
    eh_fd_debug_trace (ef, 0xf00003);
    eh_fd_clear_revents (ef, POLLIN);
    return;
  } else if (r < 0 && (errno == EINTR)) {
    return;
  } else if (r < 0) {
    eh_log_perror (EH_LOG_DEBUG, "read");
    eh_fd_delete_request (ef);
    return;
  }
  /* got EOF */
  eh_fd_debug_trace (ef, 0x260000 + r);
  ec->eof_flag = 1;
  eh_connection_read_finish (ec);
  return;

 ssl_error:
  LOCAL_DEBUG_DO (ec->last_action_result = r);
  r = eh_connection_ssl_error (ec, ef, r);
  eh_debug ("ssl_error %d", r);
  if (r < 0) {
    eh_fd_delete_request (ef);
    return;
  } else if (r == 0) {
    /* got EOF */
    eh_connection_timeout_reset (ec);
    ec->eof_flag = 1;
    eh_connection_read_finish (ec);
    return;
  }
  /* wouldblock */
  eh_connection_timeout_reset (ec);
  return;
}

static void
eh_connection_dummy_read (eh_fd_t *ef, size_t *read_limit)
{
  /* we are in the 'lingering-close' state */
  char buffer[1024];
  int r;
  r = *read_limit > 1024 ? 1024 : *read_limit;
  while (1) {
    r = read (ef->pfd.fd, buffer, r);
    if (r < 0) {
      if (errno == EWOULDBLOCK || errno == EAGAIN) {
	eh_fd_debug_trace (ef, 0xf00004);
        eh_fd_clear_revents (ef, POLLIN);
      	break;
      }
      if (errno == EINTR)
        continue;
      eh_fd_delete_request (ef);
      return;
    } else if (r == 0) {
      eh_fd_delete_request (ef);
      return;
    } else {
      *read_limit -= r;
      if (*read_limit == 0) {
	eh_fd_delete_request (ef);
	return;
      }
    }
    break;
  }
  eh_fd_request_events (ef, POLLIN);
}

static void
eh_connection_on_ssl_accept (eh_fd_t *ef)
{
  eh_connection_t *ec = (eh_connection_t *)ef->user_data;
  int r;
  /* 1: success,  0: unknown error,  -1: wouldblock or others */
  r = SSL_accept (ec->sslcon);
  eh_debug ("SSL_accept: %d", r);
  if (r < 0) {
    if (eh_connection_ssl_error (ec, ec->client_pfd, r) <= 0) {
      eh_fd_delete_request (ec->client_pfd);
      return;
    }
    /* possibly wouldblock */
  } else if (r == 0) {
    eh_debug ("failed to accept a SSL connection");
    eh_fd_delete_request (ec->client_pfd);
    return;
  } else {
    eh_connection_ssl_accept_finished (ec);
    if (ec->current_request.not_finished) {
      /* we reach here because of renegotiation */
      r = eh_connection_reply_after_renegotiation (ec);
      if (r) {
	eh_strbuf_remove (&ec->readbuf, r);
      }
    }
  }
}

static void
eh_connection_on_event (eh_fd_t *ef)
{
  eh_connection_t *ec = (eh_connection_t *)ef->user_data;

#ifdef DEBUG
  ec->event_count++;
#endif
  
  if (ec->sslcon == NULL) {
    short revents;
    revents = ef->pfd.revents;
    if (ec->shutdown_flag) {
      /* we're doing lingering close */
      eh_connection_dummy_read (ef, &ec->dummy_read_limit);
    } else if (ec->wvec_len > 0 && (revents & POLLOUT)) {
      eh_connection_on_write (ec, ef);
    } else if ((revents & POLLIN)) {
      eh_connection_on_read (ec, ef);
    }
  } else {
    eh_debug ("SSL");
    ec->ssl_want_read = 0;
    ec->ssl_want_write = 0;
    eh_debug ("SSL_is_init_finished: %d", SSL_is_init_finished (ec->sslcon));
    if (!SSL_is_init_finished (ec->sslcon)) {
      eh_connection_on_ssl_accept (ef);
    } else {
      short events;
      events = eh_connection_get_state (ec);
      if (events == POLLIN) {
	eh_connection_on_read (ec, ef);
      } else if (events == POLLOUT) {
	eh_connection_on_write (ec, ef);
      }
    }
  }
  eh_connection_set_request_events (ec, 0x0000);
}

static void
eh_connection_graceful_shutdown (eh_connection_t *ec)
{
  eh_debug ("GRACEFUL SHUTDOWN");
  if (ec->shutdown_flag) {
    /* we're doing lingering close */
    return;
  } else if (ec->wvec_len > 0) {
    /* more thigs to write */
    return;
  } else if (ec->rhandler) {
    /* handler is running */
    return;
  } else if (ec->rconn == eh_rconn_keepalive) {
    /* waiting for a next request. do lingering close */
    eh_connection_write_finish (ec);
  }
}

static void
eh_connection_on_timer (eh_fd_t *ef, int graceful_shutdown)
{
  eh_connection_t *ec = (eh_connection_t *)ef->user_data;
  time_t now;
  now = eh_fd_get_time ();
  eh_debug ("on_timeout");
  if (now >= ec->timeout) {
    if (ec->shutdown_flag || ec->ssl_shutdown_is_in_progress) {
      /* lingering close is timed out */
      eh_fd_delete_request (ef);
      return;
    } else {
      /* do lingering close if necessary */
      eh_connection_write_finish (ec);
    }
  } else if (graceful_shutdown) {
    eh_connection_graceful_shutdown (ec);
  }
  eh_connection_set_request_events (ec, 0x101);
}

