/*
 * ESE, a HyperText Transfer Protocol server
 * Copyright (C) 1996-2001 Akira Higuchi <a-higuti@math.sci.hokudai.ac.jp>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
  simpleclient.c, a simple TCP client.
  
  Usage:
    $ echo -e "GET / HTTP/1.0\n\n" | ./simpleclient  localhost 80

*/

#include <sys/socket.h>
#include <netinet/in.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <netinet/tcp.h>
#include <sys/time.h>
#include <signal.h>
#include <netdb.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define Syscall_perror(x, y, z) \
  if ((x) < 0) {                \
  perror (y);                   \
  (z);                          \
  }

static void
get_sockaddr (const char *hostname, int portnum, struct sockaddr_in *addr)
{
  struct hostent *hp;
  
  if ((hp = gethostbyname (hostname)) == NULL) {
    herror ("gethostbyname");
    fprintf (stderr, "hostnme = %s", hostname);
    exit (1);
  }
  memset (addr, 0, sizeof (struct sockaddr_in));
  addr->sin_family = AF_INET;
  addr->sin_port = htons (portnum);
  memcpy (&addr->sin_addr, hp->h_addr, hp->h_length);
}

static int
init_client (void)
{
  int sd, v;

  Syscall_perror ((sd = socket (AF_INET, SOCK_STREAM, 0)),
		  "socket", exit (1));
  v = 1;
  Syscall_perror (setsockopt (sd, IPPROTO_TCP, TCP_NODELAY, &v, sizeof (v)),
		  "setsockopt SO_NODELAY", exit (1));
  v = 1;
  Syscall_perror (setsockopt (sd, SOL_SOCKET, SO_KEEPALIVE, &v, sizeof (v)),
		  "setsockopt SO_KEEPALIVE", exit (1));
  v = 16384;
  Syscall_perror (setsockopt (sd, SOL_SOCKET, SO_SNDBUF, &v, sizeof (v)),
		  "setsockopt SO_SNDBUF", exit (1));
  return sd;
}

static void
do_client (int sd)
{
  struct timeval starttime;
  fd_set rfds;
  int r, send_done = 0;
  char buffer[4096];

  signal (SIGPIPE, SIG_IGN);
  
  gettimeofday (&starttime, NULL);
  while (1) {
    FD_ZERO (&rfds);
    FD_SET (sd, &rfds);
    if (!send_done)
      FD_SET (0, &rfds);
    Syscall_perror (select (sd + 1, &rfds, NULL, NULL, NULL),
		    "select", exit (1));
    if (!send_done && FD_ISSET (0, &rfds)) {
      Syscall_perror ((r = read (0, buffer, 4096)),
		      "read from 0", exit (1));
      if (r == 0) {
	send_done = 1;
      } else {
	{
	  /* note: sd is in blocking mode */
	  Syscall_perror (write (sd, buffer, r),
			  "write to sd", send_done = 1);
	}
#if 1
	usleep (1);
#endif
      }
    }
    if (FD_ISSET (sd, &rfds)) {
      Syscall_perror ((r = read (sd, buffer, 1024)),
		      "read from sd", exit (1) );
      if (r < 0)
	continue;
      if (r == 0) {
	int i;
	fprintf (stderr, "\nread done. trying more write.\n");
	memset (buffer, ' ', 4096);
	for (i = 0; i < 32; i++) {
	  r = write (sd, buffer, 4096);
	  if (r < 0) {
	    perror ("write");
	    fprintf (stderr, "(wrote %d bytes)\n", 4096 * i);
	    fprintf (stderr, "the server has not done lingering close\n");
	    exit (-1);
	  }
	}
	fprintf (stderr, "the server is doing lingering close\n");
	exit (0);
      }
      Syscall_perror (write (1, buffer, r),
		      "write to 1", exit (1));
    }
  }
}

int main (int argc, char **argv)
{
  struct sockaddr_in servaddr;
  int sd;
  if (argc < 3) {
    printf ("Usage: %s HOST PORT\n", argv[0]);
    return 1;
  }
  get_sockaddr (argv[1], atoi (argv[2]), &servaddr);
  sd = init_client ();
  Syscall_perror (connect (sd, (const struct sockaddr *)&servaddr,
			   sizeof (servaddr)),
		  "connect", exit (1));
  do_client (sd);
  return 0;
}
