/*
 * ESE, a HyperText Transfer Protocol server
 * Copyright (C) 1996-2001 Akira Higuchi <a-higuti@math.sci.hokudai.ac.jp>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "esehttpd.h"

eh_strhash_t *
eh_strhash_new (size_t table_size, void (*on_destroy)(eh_strhash_entry_t *))
{
  eh_strhash_t *sh;
  sh = (eh_strhash_t *)x_malloc (sizeof (*sh));
  sh->num_bucket = table_size;
  sh->num_ent_total = 0;
  sh->on_destroy = on_destroy;
  sh->bucket = (eh_strhash_bucket_t *)x_malloc (sh->num_bucket *
						sizeof (sh->bucket[0]));
  memset (sh->bucket, 0, sh->num_bucket * sizeof (sh->bucket[0]));
  return sh;
}

void
eh_strhash_delete (eh_strhash_t *sh)
{
  size_t i, j;
  for (i = 0; i < sh->num_bucket; i++) {
    eh_strhash_bucket_t *b = sh->bucket + i;
    if (sh->on_destroy) {
      for (j = 0; j < b->num_ent; j++) {
	(*sh->on_destroy)(&b->ent[j]);
      }
    }
    if (b->ent) x_free (b->ent);
  }
  x_free (sh->bucket);
  x_free (sh);
}

unsigned int
eh_strhash_hash (const char *key)
{
  unsigned int h;
  unsigned char *p;
  for (p = (unsigned char *)key, h = 0; *p; p++)
    h = (h << 5) - h + p[0];
  return h;
}

static int
eh_strhash_find_internal (eh_strhash_t *sh, const char *key,
			  unsigned int hash_val,
			  size_t *bucket_pos_r, size_t *ent_pos_r)
{
  size_t bpos, epos;
  eh_strhash_bucket_t *b;
  bpos = (size_t) (hash_val % sh->num_bucket);
  b = sh->bucket + bpos;
  for (epos = 0; epos < b->num_ent; epos++)
    if (strcmp (key, b->ent[epos].key) == 0)
      break;
  *bucket_pos_r = bpos;
  *ent_pos_r = epos;
  return (epos < b->num_ent); /* TRUE if found */
}

void *
eh_strhash_find_fast (eh_strhash_t *sh, const char *key, unsigned int hash_val,
		      void *notfound_val)
{
  size_t bpos, epos;
  if (eh_strhash_find_internal (sh, key, hash_val, &bpos, &epos)) {
    return sh->bucket[bpos].ent[epos].val;
  } else {
    return notfound_val;
  }
}

void *
eh_strhash_find (eh_strhash_t *sh, const char *key, void *notfound_val)
{
  return eh_strhash_find_fast (sh, key, eh_strhash_hash (key), notfound_val);
}
  
void
eh_strhash_remove (eh_strhash_t *sh, const char *key)
{
  size_t bpos, epos;
  if (eh_strhash_find_internal (sh, key, eh_strhash_hash (key), &bpos, &epos)) {
    size_t j;
    eh_strhash_bucket_t *b = sh->bucket + bpos;
    if (sh->on_destroy)
      (*sh->on_destroy)(b->ent + epos);
    for (j = epos; j < b->num_ent - 1; j++)
      b->ent[j] = b->ent[j + 1];
    b->num_ent--;
    sh->num_ent_total--;
#ifdef DEBUG
    for (j = 0; j < b->num_ent; j++) {
      assert (b->ent[j].key);
    }
#endif
  }
}

eh_strhash_entry_t *
eh_strhash_find_create_fast (eh_strhash_t *sh, char *key,
			     unsigned int hash_val)
{
  size_t bpos, epos;
  if (eh_strhash_find_internal (sh, key, hash_val, &bpos, &epos)) {
    return &sh->bucket[bpos].ent[epos];
  } else {
    eh_strhash_bucket_t *b;
    eh_strhash_entry_t *e;
    b = &sh->bucket[bpos];
    b->num_ent++;
    sh->num_ent_total++;
    b->ent = (eh_strhash_entry_t *)x_realloc (b->ent,
					      b->num_ent * sizeof (b->ent[0]));
    e = &b->ent[epos];
    e->key = key;
    e->val = NULL;
    assert (key);
    return e;
  }
}

eh_strhash_entry_t *
eh_strhash_find_create (eh_strhash_t *sh, char *key)
{
  return eh_strhash_find_create_fast (sh, key, eh_strhash_hash (key));
}

void
eh_strhash_foreach (eh_strhash_t *sh,
		    void (*func)(eh_strhash_entry_t *ent, void *data),
		    void *data)
{
  size_t i, j;
  for (i = 0; i < sh->num_bucket; i++) {
    eh_strhash_bucket_t *b = sh->bucket + i;
    for (j = 0; j < b->num_ent; j++) {
      (*func)(&b->ent[j], data);
    }
  }
}

void
eh_strhash_dump (eh_strhash_t *sh)
{
  size_t i, j;
  for (i = 0; i < sh->num_bucket; i++) {
    eh_strhash_bucket_t *b = sh->bucket + i;
    for (j = 0; j < b->num_ent; j++) {
      eh_debug ("(%d, %d)[%p] %p(%s): %p(%s)", i, j, b->ent + j,
		b->ent[j].key, b->ent[j].key,
		b->ent[j].val, (char *)b->ent[j].val);
    }
  }
}
