/*
 * ESE, a HyperText Transfer Protocol server
 * Copyright (C) 1996-2001 Akira Higuchi <a-higuti@math.sci.hokudai.ac.jp>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "esehttpd.h"

static RSA *
eh_vserver_tmp_rsa_callback (SSL *ssl, int is_export, int keylength)
{
  static RSA *rsa = NULL;
  eh_debug ("is_export = %d, keylength = %d", is_export, keylength);
  if (keylength != 512)
    keylength = 1024;
  if (rsa)
    RSA_free (rsa);
  rsa = RSA_generate_key (keylength, RSA_F4, NULL, NULL);
  return rsa;
}

static DH *
eh_vserver_load_dhparam (const char *filename)
{
  BIO *bio;
  DH *dh;
  if ((bio = BIO_new_file (filename, "r")) == NULL) {
    eh_log (EH_LOG_FATAL, "failed to open %s", filename);
    return NULL;
  }
  dh = PEM_read_bio_DHparams (bio, NULL, NULL, NULL);
  BIO_free (bio);
  if (dh == NULL) {
    eh_log (EH_LOG_WARNING, "%s does not contain DH parameters",
	    filename);
  }
  return dh;
}

static int
eh_vserver_verify_callback (int state, X509_STORE_CTX *ctx)
{
  if (state == 0) {
    eh_debug ("Verify Peer: depth %d: %s", ctx->error_depth,
	      X509_verify_cert_error_string (ctx->error));
  } else {
    eh_debug ("Verify Peer: depth %d: OK", ctx->error_depth);
  }
  return state;
}

void
eh_vserver_discard (eh_vserver_t *ev)
{
  eh_mimetype_discard (&ev->mt);
  eh_statuscode_table_discard (&ev->statuscode_table);
  if (ev->sd >= 0)
    close (ev->sd);
  if (ev->sd_ssl >= 0)
    close (ev->sd_ssl);
  if (ev->sslctx)
    SSL_CTX_free (ev->sslctx);
  if (ev->primary_hostname)
    x_free (ev->primary_hostname);
}

void
eh_vserver_init (eh_vserver_t *ev, eh_app_t *app, eh_config_vhost_t *vhost,
		 const char *hostname)
{
  memset (ev, 0, sizeof (*ev));
  ev->app_backref = app;
  ev->default_vhost = vhost;
  ev->primary_hostname = hostname ? x_strdup (hostname) : NULL;
  ev->sd = -1;
  ev->sd_ssl = -1;
  ev->sslctx = 0;
  ev->servaddr = vhost->servaddr;
  ev->namevirtualhost = 0;
}

void
eh_vserver_init_ssl (eh_vserver_t *ev)
{
  char *certfile, *certkeyfile, *dhparamfile;
  char *mimetype_file, *errordocumentdir;
  const char *default_type;
  const char *hostname;
  eh_config_vhost_t *vhost;
  eh_config_global_t *ecg;

  hostname = ev->primary_hostname ? ev->primary_hostname : "*";
  vhost = ev->default_vhost;
  ecg = vhost->econf_global_ref;

  mimetype_file = eh_config_get_fullpath (ecg, vhost->typesconfig,
					  ESEHTTPD_TYPESCONFIG);
  errordocumentdir = eh_config_get_fullpath (ecg, vhost->errordocumentdir,
					     NULL);
  certfile = eh_config_get_fullpath (ecg, vhost->sslcertificatefile, NULL);
  certkeyfile = eh_config_get_fullpath (ecg, vhost->sslcertificatekeyfile,
					NULL);
  dhparamfile = eh_config_get_fullpath (ecg, vhost->ssldhparamfile, NULL);
  
  default_type = vhost->default_dir->defaulttype;
  if (default_type == NULL)
    default_type = ESEHTTPD_DEFAULT_TYPE;
  eh_mimetype_init (&ev->mt, mimetype_file, default_type);

  eh_statuscode_table_init (&ev->statuscode_table, errordocumentdir);
  
  if (certfile) {
    DH *dh;
    ev->sslctx = SSL_CTX_new (SSLv23_server_method ());
    if (ev->sslctx == NULL) {
      eh_log_ssl_error (EH_LOG_FATAL, "SSL_CTX_new");
      exit (-1);
    }
    if (!SSL_CTX_use_certificate_chain_file (ev->sslctx, certfile)) {
      eh_log_ssl_error (EH_LOG_FATAL, certfile);
      exit (-1);
    }
    if (!SSL_CTX_use_PrivateKey_file (ev->sslctx,
				      certkeyfile ? certkeyfile : certfile,
				      SSL_FILETYPE_PEM)) {
      eh_log_ssl_error (EH_LOG_FATAL, certkeyfile ? certkeyfile : certfile);
      exit (-1);
    }
    if (dhparamfile) {
      dh = eh_vserver_load_dhparam (dhparamfile ? dhparamfile : certfile);
      if (dh != NULL) {
	if (!SSL_CTX_set_tmp_dh (ev->sslctx, dh)) {
	  eh_log_ssl_error (EH_LOG_FATAL, dhparamfile);
	}
      }
      eh_log (EH_LOG_INFO, "%s: DH enabled", hostname);
    }
    if (vhost->sslciphersuite) {
      if (!SSL_CTX_set_cipher_list (ev->sslctx, vhost->sslciphersuite)) {
	eh_log_ssl_error (EH_LOG_FATAL, vhost->sslciphersuite);
	exit (-1);
      }
    }
    if (vhost->sslcacertificatefile) {
      char *cacert;
      STACK_OF (X509_NAME) *cert_names;
      cacert = eh_config_get_fullpath (ecg, vhost->sslcacertificatefile, NULL);
      eh_log (EH_LOG_INFO, "%s: CA Certificate File: %s", hostname, cacert);
      if (!SSL_CTX_load_verify_locations (ev->sslctx, cacert, NULL)) {
	eh_log_ssl_error (EH_LOG_FATAL, cacert);
	exit (-1);
      }
      cert_names = SSL_load_client_CA_file (cacert);
      if (cert_names != NULL) {
	SSL_CTX_set_client_CA_list (ev->sslctx, cert_names);
      } else {
	eh_log_ssl_error (EH_LOG_FATAL, cacert);
	exit (-1);
      }
      x_free (cacert);
    }
    SSL_CTX_set_tmp_rsa_callback (ev->sslctx, eh_vserver_tmp_rsa_callback);
    SSL_CTX_set_verify (ev->sslctx, SSL_VERIFY_NONE,
			eh_vserver_verify_callback);
    SSL_CTX_set_session_cache_mode (ev->sslctx, SSL_SESS_CACHE_OFF);
  }
  x_free (mimetype_file); /* != NULL */
  if (errordocumentdir)
    x_free (errordocumentdir);
  if (certfile)
    x_free (certfile);
  if (certkeyfile)
    x_free (certkeyfile);
  if (dhparamfile)
    x_free (dhparamfile);
}

void
eh_vserver_attach_namevhost (eh_vserver_t *ev, eh_config_vhost_t *vhost,
			     const char *hostname)
{
  ev->namevirtualhost = 1;
  if (strcmp (ev->primary_hostname, hostname) < 0)
    return;
  x_free (ev->primary_hostname);
  ev->primary_hostname = x_strdup (hostname);
  ev->default_vhost = vhost;
}

static int
eh_vserver_create_socket (int sendbuf_size, int recvbuf_size)
{
  int sd;
  int v;
  Syscall_perror ((sd = socket (AF_INET, SOCK_STREAM, 0)),
		  "socket", exit (1));
  v = 1;
  Syscall_perror (setsockopt (sd, IPPROTO_TCP, TCP_NODELAY, &v, sizeof (v)),
  		  "setsockopt SO_NODELAY", exit (1));
  v = 1;
  Syscall_perror (setsockopt (sd, SOL_SOCKET, SO_KEEPALIVE, &v, sizeof (v)),
		  "setsockopt SO_KEEPALIVE", exit (1));
  v = 1;
  Syscall_perror (setsockopt (sd, SOL_SOCKET, SO_REUSEADDR, &v, sizeof (v)),
		  "setsockopt SO_REUSEADDR", exit (1));
  v = sendbuf_size;
  Syscall_perror (setsockopt (sd, SOL_SOCKET, SO_SNDBUF, &v, sizeof (v)),
  		  "setsockopt SO_SNDBUF", exit (1));
  v = recvbuf_size;
  Syscall_perror (setsockopt (sd, SOL_SOCKET, SO_RCVBUF, &v, sizeof (v)),
  		  "setsockopt SO_RCVBUF", exit (1));
  return sd;
}

void
bind_listen(int sd, const char *hostname, struct sockaddr_in servaddr,
	    unsigned int portnum, int listenbacklog_size)
{
  servaddr.sin_family = AF_INET;
  servaddr.sin_port = htons (portnum);
  eh_log (EH_LOG_INFO, "%s: binding %08x:%d",
	  hostname, servaddr.sin_addr.s_addr, portnum);
  Syscall_perror (bind (sd, (const struct sockaddr *)&servaddr,
			sizeof (servaddr)),
		  "bind", exit (1));
  Syscall_perror (listen (sd, listenbacklog_size),
		  "listen", exit (1));
}

void
eh_vserver_start_servers (eh_vserver_t *ev)
{
  eh_config_vhost_t *vhost;
  eh_config_global_t *econf;
  int portnum, portnum_ssl;
  const char *hostname;

  hostname = ev->primary_hostname ? ev->primary_hostname : "*";
  vhost = ev->default_vhost;
  econf = ev->app_backref->econf;

  portnum = vhost->port;
  if (portnum) {
    eh_log (EH_LOG_INFO, "%s: port: %d", hostname, portnum);
  }

  portnum_ssl = 0;
  if (ev->sslctx) {
    portnum_ssl = vhost->sslport;
    if (portnum_ssl) {
      eh_log (EH_LOG_INFO, "%s: ssl port: %d", hostname, portnum_ssl);
    }
  }

  if (portnum == 0 && portnum_ssl == 0) {
    eh_log (EH_LOG_WARNING, "no port is specified");
    return;
  }

  eh_log (EH_LOG_INFO, "%s: send buffer size: %d",
	  hostname, econf->sendbuffersize);
  eh_log (EH_LOG_INFO, "%s: receive buffer size: %d",
	  hostname, econf->recvbuffersize);
  eh_log (EH_LOG_INFO, "%s: listen backlog size: %d",
	  hostname, econf->listenbacklog);

  {
    eh_debug ("ip address %x, vhost %p", ev->servaddr.sin_addr.s_addr, vhost);
    if (portnum > 0) {
      ev->sd = eh_vserver_create_socket (econf->recvbuffersize,
					 econf->listenbacklog);
    }
    if (portnum_ssl > 0) {
      ev->sd_ssl = eh_vserver_create_socket (econf->sendbuffersize,
					     econf->recvbuffersize);
    }
  }

  ev->connection_timeout = vhost->timeout;
  if (ev->connection_timeout < 10)
    ev->connection_timeout = 300;
  eh_log (EH_LOG_INFO, "%s: connection timeout: %d sec.",
	  hostname, ev->connection_timeout);

  ev->ef_socket = ev->ef_socket_ssl = NULL;
  if (ev->sd >= 0) {
    ev->ef_socket = eh_fd_new (ev->sd, NULL, NULL, NULL, NULL);
    bind_listen (ev->sd, hostname, ev->servaddr, portnum, econf->listenbacklog);
  }
  if (ev->sd_ssl >= 0) {
    ev->ef_socket_ssl = eh_fd_new (ev->sd_ssl, NULL, NULL, NULL, NULL);
    bind_listen (ev->sd_ssl, hostname, ev->servaddr, portnum_ssl,
		 econf->listenbacklog);
  }
}

static void
eh_vserver_on_event_worker (eh_fd_t *ef_socket)
{
  eh_vserver_t *ev = (eh_vserver_t *)ef_socket->user_data;
  struct sockaddr_in cliaddr;
  while (1) {
    int fd, clilen = sizeof (cliaddr);
    fd = accept (ef_socket->pfd.fd, (struct sockaddr *)&cliaddr,
		 (socklen_t *)&clilen);
    eh_debug ("accept %d", fd);
    if (fd >= ev->app_backref->rlimit_nofile - 10) {
      eh_debug ("file limit exceeded");
      close (fd);
    } else if (fd >= 0) {
      eh_connection_new (fd, ev->app_backref, ev, &cliaddr,
			 (ef_socket->pfd.fd == ev->sd_ssl));
    } else {
      break;
    }
  }
  eh_fd_clear_revents (ef_socket, POLLIN);
}

static void
eh_vserver_on_timeout (eh_fd_t *ef_socket, int graceful_shutdown)
{
  eh_debug ("");
  if (graceful_shutdown) {
    eh_fd_delete_request (ef_socket);
  }
}

static void
eh_vserver_on_delete (eh_fd_t *ef_socket)
{
  eh_vserver_t *ev = (eh_vserver_t *)ef_socket->user_data;
  if (ef_socket->pfd.fd == ev->sd_ssl) {
    ev->ef_socket_ssl = NULL;
    ev->sd_ssl = -1;
  } else {
    ev->ef_socket = NULL;
    ev->sd = -1;
  }
}

void
eh_vserver_set_worker (eh_vserver_t *ev)
{
  if (ev->ef_socket) {
    eh_fd_reset_pid(ev->ef_socket);
    eh_fd_set_data (ev->ef_socket,
		    eh_vserver_on_event_worker,
		    eh_vserver_on_timeout,
		    eh_vserver_on_delete,
		    (void *)ev);
    eh_fd_request_events (ev->ef_socket, POLLIN);
  }
  if (ev->ef_socket_ssl) {
    eh_fd_reset_pid(ev->ef_socket_ssl);
    eh_fd_set_data (ev->ef_socket_ssl,
		    eh_vserver_on_event_worker,
		    eh_vserver_on_timeout,
		    eh_vserver_on_delete,
		    (void *)ev);
    eh_fd_request_events (ev->ef_socket_ssl, POLLIN);
  }
}

static void
eh_vserver_on_event_manager (eh_fd_t *ef_socket)
{
  static int child_num = 0;
  union sigval sval;
  pid_t pid;
  int retry_count = 0;
  int delegate = 1;
  eh_vserver_t *ev = (eh_vserver_t *)ef_socket->user_data;
  eh_app_t *app = ev->app_backref;
  eh_debug ("");
  for (retry_count = 0; retry_count < app->num_child && delegate;
       retry_count++) {
    pid = app->child[child_num];
    eh_debug ("sending rtsig to %u", pid);
    sval.sival_int = ef_socket->pfd.fd;
    if ((delegate = eh_fd_delegate (pid, sval)) < 0) {
      if (errno == ESRCH) {
	/* no such process. fork a new worker */
	eh_log (EH_LOG_WARNING, "forking a new worker process");
	if ((pid = eh_app_fork_worker (app, child_num)) == 0) {
	  /* child */
	  return;
	}
      }
      /* We can ignore EPERM because it can be occured only on startup. EAGAIN
         can be ignored also because the receiver should get SIGIO (TODO:
         assert that). */
    }
    /* round robin */
    if (++child_num >= app->num_child)
      child_num = 0;
    eh_fd_clear_revents (ef_socket, POLLIN);
  }
}

void
eh_vserver_set_manager (eh_vserver_t *ev, int handle_event)
{
  
  if (ev->ef_socket) {
    eh_fd_set_data (ev->ef_socket,
		    handle_event ? eh_vserver_on_event_manager : NULL,
		    eh_vserver_on_timeout,
		    eh_vserver_on_delete,
		    (void *)ev);
    if (handle_event)
      eh_fd_request_events (ev->ef_socket, POLLIN);
  }
  if (ev->ef_socket_ssl) {
    eh_fd_set_data (ev->ef_socket_ssl,
		    handle_event ? eh_vserver_on_event_manager : NULL,
		    eh_vserver_on_timeout,
		    eh_vserver_on_delete,
		    (void *)ev);
    if (handle_event)
      eh_fd_request_events (ev->ef_socket_ssl, POLLIN);
  }
}

