#pragma once

#include <string>
#include <boost/iterator/iterator_facade.hpp>
#include <boost/iterator/reverse_iterator.hpp>

//f[^ǂݍݗpC^[tFCX
class IReader
{
public:
	enum Result{
		success,
		error,
		eof,
	};
	virtual ~IReader() {}
	virtual Result Read( ulong offset, ulong size, uchar *buffer, ulong& readsize ) = 0;
	virtual ulong Size() const = 0;
};
typedef boost::shared_ptr<IReader> Reader_ptr;

//f[^ݗpC^[tFCX
class IWriter
{
public:
	virtual ~IWriter() {}
	virtual bool Write( ulong offset, ulong size, uchar *buffer ) = 0;
};
typedef boost::shared_ptr<IWriter> Writer_ptr;

//f[^nhOpC^[tFCX
class IBuffer
{
public:
	virtual ~IBuffer() {}
	virtual ulong Get( ulong offset, ulong size, uchar *buffer ) = 0; //return readsize
	virtual bool Insert( ulong offset, ulong size, uchar *buffer ) = 0;
	virtual bool OverStrike( ulong offset, ulong size, uchar *buffer ) = 0;
	virtual bool Remove( ulong offset, ulong size ) = 0;
	virtual ulong Size() const = 0;

	virtual const uchar* Begin() = 0;
	virtual const uchar* End() = 0;
	virtual const uchar* Increment(const uchar*, ulong n) = 0;
	virtual const uchar* Decrement(const uchar*, ulong n) = 0;
	virtual bool IsEqual(const uchar*, const uchar*) = 0;

	virtual bool Load( Reader_ptr reader ) = 0;
	virtual bool Save( Writer_ptr writer ) = 0;
};
typedef boost::shared_ptr<IBuffer> Buffer_ptr;

//IBuffer̃f[^𑖍Ce[^
class buffer_iterator
	: public boost::iterator_facade<
		buffer_iterator,
		uchar,
		boost::random_access_traversal_tag,
		const uchar&>
{
public:
	buffer_iterator( IBuffer *buffer )
		: buffer_(buffer), p_(0)
	{
	}
	buffer_iterator( IBuffer *buffer, const uchar* p)
		: buffer_(buffer), p_(p)
	{
	}

	void increment()         { p_ = buffer_->Increment(p_,1); }
	void decrement()         { p_ = buffer_->Decrement(p_,1); }
	const uchar& dereference() const { return *p_; }
	bool equal( const buffer_iterator& rhs ) const { return buffer_->IsEqual(p_,rhs.p_); }
	void advance( difference_type n ) { p_ = buffer_->Increment(p_,n); }
	difference_type distance_to( const buffer_iterator& rhs ) const { return rhs.p_-p_; }

private:
	const uchar *p_;
	IBuffer *buffer_;
};
//tCe[^
typedef boost::reverse_iterator<buffer_iterator> reverse_buffer_iterator;


//t@CΏۂƂ郊[_
class FileReader : public IReader
{
public:
	FileReader( const std::string& filepath );
	IReader::Result Read( ulong offset, ulong size, uchar *buffer, ulong& readsize );
	ulong Size() const;

private:
	class Impl;
	boost::shared_ptr<Impl> pimpl_;
};

//eXgpŃ_~[f[^񋟂郊[_
class DummyReader : public IReader
{
public:
	DummyReader();
	IReader::Result Read( ulong offset, ulong size, uchar *buffer, ulong& readsize );
	ulong Size() const;
};

//ǂݍ܂Ȃ[_
class NullReader : public IReader
{
public:
	IReader::Result Read( ulong offset, ulong size, uchar *buffer, ulong& readsize );
	ulong Size() const;
};

//t@CΏۂƂ郉C^
class FileWriter : public IWriter
{
public:
	FileWriter( const std::string& filepath );
	bool Write( ulong offset, ulong size, uchar *buffer );

private:
	class Impl;
	boost::shared_ptr<Impl> pimpl_;
};

//t@CŜǂݍރobt@
class FullFileBuffer : public IBuffer
{
public:
	FullFileBuffer();
	ulong Get( ulong offset, ulong size, uchar *buffer);
	bool Insert( ulong offset, ulong size, uchar *buffer);
	bool OverStrike( ulong offset, ulong size, uchar *buffer);
	bool Remove( ulong offset, ulong size );
	ulong Size() const;

	const uchar* Begin();
	const uchar* End();
	const uchar* Increment(const uchar*, ulong n);
	const uchar* Decrement(const uchar*, ulong n);
	bool IsEqual(const uchar*, const uchar*);

	bool Load( Reader_ptr reader );
	bool Save( Writer_ptr writer );

private:
	class Impl;
	boost::shared_ptr<Impl> pimpl_;
};

//FILE̎N[Yp[eBeBNX(RAII)
class FileHandler
{
public:
	FileHandler(FILE *file) : file_(file) {}
	~FileHandler() { if(file_) fclose(file_); } //fcloses邱Ƃ͂Ƃ肠lȂ
	FILE* get() { return file_; }
private:
	FILE* file_;
};
