/*
Copyright (c) 2004, Dennis M. Sosnoski.
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice, this
   list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.
 * Neither the name of JiBX nor the names of its contributors may be used
   to endorse or promote products derived from this software without specific
   prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

package org.jibx.runtime.impl;

import java.io.IOException;
import java.util.Stack;

import org.jibx.runtime.IXMLWriter;

/**
 * Base implementation of XML writer interface. This provides common handling of
 * namespaces and indentation that can be used for all forms of text output.
 *
 * @author Dennis M. Sosnoski
 * @version 1.0
 */

public abstract class XMLWriterBase implements IXMLWriter
{
    /** URIs for namespaces. */
    protected String[] m_uris;
    
    /** Flag for current element has text content. */
    protected boolean m_textSeen;
    
    /** Flag for current element has content. */
    protected boolean m_contentSeen;
    
    /** Depth of nested tags. */
    protected int m_nestingDepth;
    
    /** Prefixes currently defined for namespaces. */
    protected String[] m_prefixes;
    
    /** Stack of information for namespace declarations. */
    private Stack m_namespaceStack;
    
    /** Depth of top namespace declaration level. */
    private int m_namespaceDepth;
    
    /** Extension namespace URIs (<code>null</code> if not in use). */
    private String[][] m_extensionUris;
    
    /** Extension namespace prefixes (<code>null</code> if not in use). */
    private String[][] m_extensionPrefixes;
    
    /**
     * Constructor.
     *
     * @param uris ordered array of URIs for namespaces used in document (must
     * be constant; the value in position 0 must always be the empty string "",
     * and the value in position 1 must always be the XML namespace
     * "http://www.w3.org/XML/1998/namespace")
     */
    
    public XMLWriterBase(String[] uris) {
        m_uris = uris;
        m_prefixes = new String[uris.length];
        m_prefixes[0] = "";
        m_prefixes[1] = "xml";
        m_namespaceStack = new Stack();
        m_namespaceDepth = -1;
    }
    
    /**
     * Write markup text to output. Markup text can be written directly to the
     * output without the need for any escaping, but still needs to be properly
     * encoded.
     *
     * @param text markup text to be written
     * @throws IOException if error writing to document
     */
    
    protected abstract void writeMarkup(String text) throws IOException;
    
    /**
     * Write markup character to output. Markup text can be written directly to
     * the output without the need for any escaping, but still needs to be
     * properly encoded.
     *
     * @param chr markup character to be written
     * @throws IOException if error writing to document
     */
    
    protected abstract void writeMarkup(char chr) throws IOException;
    
    /**
     * Report that namespace has been defined.
     *
     * @param index namespace URI index number
     * @param prefix prefix used for namespace
     * @throws IOException if error writing to document
     */
    
    protected abstract void defineNamespace(int index, String prefix)
        throws IOException;
    
    /**
     * Report that namespace has been undefined.
     *
     * @param index namespace URI index number
     */
    
    protected abstract void undefineNamespace(int index);
    
    /**
     * Write namespace prefix to output. This internal method is used to throw
     * an exception when an undeclared prefix is used.
     *
     * @param index namespace URI index number
     * @throws IOException if error writing to document
     */
    
    protected abstract void writePrefix(int index) throws IOException;
    
    /**
     * Write attribute text to output. This needs to write the text with any
     * appropriate escaping.
     *
     * @param text attribute value text to be written
     * @throws IOException if error writing to document
     */
    
    protected abstract void writeAttributeText(String text) throws IOException;
        
    /**
     * Write XML declaration to document. This can only be called before any
     * other methods in the interface are called.
     *
     * @param version XML version text
     * @param encoding text for encoding attribute (unspecified if
     * <code>null</code>)
     * @param standalone text for standalone attribute (unspecified if
     * <code>null</code>)
     * @throws IOException on error writing to document
     */

    public void writeXMLDecl(String version, String encoding, String standalone)
        throws IOException {
        writeMarkup("<?xml version=\"");
        writeAttributeText(version);
        if (encoding != null) {
            writeMarkup("\" encoding=\"");
            writeAttributeText(encoding);
        }
        if (standalone != null) {
            writeMarkup("\" standalone=\"");
            writeAttributeText(standalone);
        }
        writeMarkup("\"?>");
    }
    
    /**
     * Generate open start tag. This allows attributes to be added to the start
     * tag, but must be followed by a {@link #closeStartTag} call.
     *
     * @param index namespace URI index number
     * @param name unqualified element name
     * @throws IOException on error writing to document
     */

    public void startTagOpen(int index, String name) throws IOException {
        indent();
        writeMarkup('<');
        writePrefix(index);
        writeMarkup(name);
    }
    
    /**
     * Set prefix for namespace.
     *
     * @param index namespace URI index number
     * @return prefix text for namespace, or <code>null</code> if the namespace
     * is no longer mapped
     */
    
    private void setNamespacePrefix(int index, String prefix) {
        if (index < m_prefixes.length) {
            m_prefixes[index] = prefix;
        } else if (m_extensionUris != null) {
            index -= m_prefixes.length;
            for (int i = 0; i < m_extensionUris.length; i++) {
                int length = m_extensionUris[i].length;
                if (index < length) {
                    m_extensionPrefixes[i][index] = prefix;
                    break;
                } else {
                    index -= length;
                }
            }
        }
    }
    
    /**
     * Generate start tag for element with namespaces. This creates the actual
     * start tag, along with any necessary namespace declarations. Previously
     * active namespace declarations are not duplicated. The tag is
     * left incomplete, allowing other attributes to be added.
     *
     * @param index namespace URI index number
     * @param name element name
     * @param nums array of namespace indexes defined by this element (must
     * be constant, reference is kept until end of element)
     * @param prefs array of namespace prefixes mapped by this element (no
     * <code>null</code> values, use "" for default namespace declaration)
     * @return this context (to allow chained calls)
     * @throws IOException on error writing to document
     */

    public void startTagNamespaces(int index, String name,
        int[] nums, String[] prefs) throws IOException {
        
        // find the number of namespaces actually being declared
        int count = 0;
        for (int i = 0; i < nums.length; i++) {
            if (!prefs[i].equals(getNamespacePrefix(nums[i]))) {
                count++;
            }
        }
        
        // check if there's actually any change
        int[] deltas = null;
        if (count > 0) {
            
            // get the set of namespace indexes that are changing
            String[] priors = new String[count];
            if (count == nums.length) {
                
                // replace the full set, tracking the prior values
                deltas = nums;
                for (int i = 0; i < count; i++) {
                    int slot = deltas[i];
                    priors[i] = getNamespacePrefix(slot);
                    setNamespacePrefix(slot, prefs[i]);
                    defineNamespace(slot, prefs[i]);
                }
                
            } else {
                
                // replace only changed ones, tracking both indexes and priors
                int fill = 0;
                deltas = new int[count];
                for (int i = 0; i < nums.length; i++) {
                    int slot = nums[i];
                    String curr = getNamespacePrefix(slot);
                    if (!prefs[i].equals(curr)) {
                        deltas[fill] = slot;
                        priors[fill++] = curr;
                        setNamespacePrefix(slot, prefs[i]);
                        defineNamespace(slot, prefs[i]);
                    }
                }
            }
            
            // set up for undeclaring namespaces on close of element
            m_namespaceStack.push
                (new DeclarationInfo(m_nestingDepth, deltas, priors));
            m_namespaceDepth = m_nestingDepth;
        }
        
        // create the start tag for element
        startTagOpen(index, name);
        
        // add namespace declarations to open element
        for (int i = 0; i < count; i++) {
            int slot = deltas[i];
            String prefix = getNamespacePrefix(slot);
            if (prefix.length() > 0) {
                writeMarkup(" xmlns:");
                writeMarkup(prefix);
                writeMarkup("=\"");
            } else {
                writeMarkup(" xmlns=\"");
            }
            String uri = null;
            if (slot < m_uris.length) {
                uri = m_uris[slot];
            } else if (m_extensionUris != null) {
                slot -= m_uris.length;
                for (int j = 0; j < m_extensionUris.length; j++) {
                    int length = m_extensionUris[j].length;
                    if (slot < length) {
                        uri = m_extensionUris[j][slot];
                        break;
                    } else {
                        slot -= length;
                    }
                }
            }
            writeAttributeText(uri);
            writeMarkup('"');
        }
    }
    
    /**
     * Add attribute to current open start tag. This is only valid after a call
     * to {@link #startTagOpen} or {@link #startTagNamespaces} and before the
     * corresponding call to {@link #closeStartTag}.
     *
     * @param index namespace URI index number
     * @param name unqualified attribute name
     * @param value text value for attribute
     * @throws IOException on error writing to document
     */

    public void addAttribute(int index, String name, String value)
        throws IOException {
        writeMarkup(" ");
        writePrefix(index);
        writeMarkup(name);
        writeMarkup("=\"");
        writeAttributeText(value);
        writeMarkup('"');
    }
    
    /**
     * Ends the current innermost set of nested namespace definitions. Reverts
     * the namespaces involved to their previously-declared prefixes, and sets
     * up for ending the new innermost set.
     */
    
    public void closeNamespaces() {
        
        // revert prefixes for namespaces included in declaration
        DeclarationInfo info = (DeclarationInfo)m_namespaceStack.pop();
        int[] deltas = info.m_deltas;
        String[] priors = info.m_priors;
        for (int i = 0; i < deltas.length; i++) {
            int index = deltas[i];
            undefineNamespace(index);
            if (index < m_prefixes.length) {
                m_prefixes[index] = null;
            } else if (m_extensionUris != null) {
                index -= m_prefixes.length;
                for (int j = 0; j < m_extensionUris.length; j++) {
                    int length = m_extensionUris[j].length;
                    if (index < length) {
                        m_extensionPrefixes[j][index] = null;
                    } else {
                        index -= length;
                    }
                }
            }
        }
        
        // set up for clearing next nested set
        if (m_namespaceStack.empty()) {
            m_namespaceDepth = -1;
        } else {
            m_namespaceDepth =
                ((DeclarationInfo)m_namespaceStack.peek()).m_depth;
        }
    }
    
    /**
     * Close the current open start tag. This is only valid after a call to
     * {@link #startTagOpen}.
     *
     * @throws IOException on error writing to document
     */

    public void closeStartTag() throws IOException {
        writeMarkup('>');
        m_nestingDepth++;
        m_textSeen = m_contentSeen = false;
    }
    
    /**
     * Close the current open start tag as an empty element. This is only valid
     * after a call to {@link #startTagOpen}.
     *
     * @throws IOException on error writing to document
     */

    public void closeEmptyTag() throws IOException {
        writeMarkup("/>");
        if (m_nestingDepth == m_namespaceDepth) {
            closeNamespaces();
        }
        m_contentSeen = true;
    }
    
    /**
     * Generate closed start tag. No attributes or namespaces can be added to a
     * start tag written using this call.
     *
     * @param index namespace URI index number
     * @param name unqualified element name
     * @throws IOException on error writing to document
     */

    public void startTagClosed(int index, String name) throws IOException {
        indent();
        writeMarkup('<');
        writePrefix(index);
        writeMarkup(name);
        writeMarkup('>');
        m_nestingDepth++;
        m_textSeen = m_contentSeen = false;
    }
    
    /**
     * Generate end tag.
     *
     * @param index namespace URI index number
     * @param name unqualified element name
     * @throws IOException on error writing to document
     */

    public void endTag(int index, String name) throws IOException {
        --m_nestingDepth;
        if (m_contentSeen && !m_textSeen) {
            indent();
        }
        writeMarkup("</");
        writePrefix(index);
        writeMarkup(name);
        writeMarkup('>');
        if (m_nestingDepth == m_namespaceDepth) {
            closeNamespaces();
        }
        m_textSeen = false;
        m_contentSeen = true;
    }
    
    /**
     * Write comment to document.
     *
     * @param text comment text
     * @throws IOException on error writing to document
     */

    public void writeComment(String text) throws IOException {
        writeMarkup("<!--");
        writeMarkup(text);
        writeMarkup("-->");
    }
    
    /**
     * Write entity reference to document.
     *
     * @param name entity name
     * @throws IOException on error writing to document
     */

    public void writeEntityRef(String name) throws IOException {
        writeMarkup('&');
        writeMarkup(name);
        writeMarkup(';');
        m_contentSeen = true;
    }
    
    /**
     * Write DOCTYPE declaration to document.
     *
     * @param name root element name
     * @param sys system ID (<code>null</code> if none, must be
     * non-<code>null</code> for public ID to be used)
     * @param pub public ID (<code>null</code> if none)
     * @param subset internal subset (<code>null</code> if none)
     * @throws IOException on error writing to document
     */

    public void writeDocType(String name, String sys, String pub, String subset)
        throws IOException {
        indent();
        writeMarkup("<!DOCTYPE ");
        writeMarkup(name);
        writeMarkup(' ');
        if (sys != null) {
            if (pub == null) {
                writeMarkup("SYSTEM \"");
                writeMarkup(sys);
            } else {
                writeMarkup("PUBLIC \"");
                writeMarkup(pub);
                writeMarkup("\" \"");
                writeMarkup(sys);
            }
            writeMarkup('"');
        }
        if (subset != null) {
            writeMarkup('[');
            writeMarkup(subset);
            writeMarkup(']');
        }
        writeMarkup('>');
    }
    
    /**
     * Write processing instruction to document.
     *
     * @param target processing instruction target name
     * @param data processing instruction data
     * @throws IOException on error writing to document
     */

    public void writePI(String target, String data) throws IOException {
        indent();
        writeMarkup("<?");
        writeMarkup(target);
        writeMarkup(' ');
        writeMarkup(data);
        writeMarkup("?>");
        m_contentSeen = true;
    }
    
    /**
     * Close document output. Completes writing of document output, including
     * closing the output medium.
     *
     * @throws IOException on error writing to document
     */

    public abstract void close() throws IOException;
    
    /**
     * Reset to initial state for reuse. The context is serially reusable,
     * as long as this method is called to clear any retained state information
     * between uses. It is automatically called when output is set.
     */
    
    public void reset() {
        m_textSeen = m_contentSeen = false;
        m_nestingDepth = 0;
        m_namespaceDepth = -1;
        m_namespaceStack.clear();
        m_extensionUris = null;
        m_extensionPrefixes = null;
    }
    
    /**
     * Get namespace URIs for mapping. This gets the full ordered array of
     * namespaces known in the binding used for this marshalling, where the
     * index number of each namespace URI is the namespace index used to lookup
     * the prefix when marshalling a name in that namespace. The returned array
     * must not be modified.
     *
     * @return array of namespaces
     */
    
    public String[] getNamespaces() {
        return m_uris;
    }
    
    /**
     * Get current prefix defined for namespace.
     *
     * @param index namespace URI index number
     * @return current prefix text, or <code>null</code> if the namespace is not
     * currently mapped
     */
    
    public String getNamespacePrefix(int index) {
        if (index < m_prefixes.length) {
            return m_prefixes[index];
        } else if (m_extensionUris != null) {
            index -= m_prefixes.length;
            for (int i = 0; i < m_extensionUris.length; i++) {
                int length = m_extensionUris[i].length;
                if (index < length) {
                    return m_extensionPrefixes[i][index];
                } else {
                    index -= length;
                }
            }
        }
        return null;
    }
    
    /**
     * Get index of namespace mapped to prefix. This can be an expensive
     * operation with time proportional to the number of namespaces defined, so
     * it should be used with care.
     * 
     * @param prefix text to match
     * @return index namespace URI index number mapped to prefix
     */
    
    public int getPrefixIndex(String prefix) {
        if (m_extensionPrefixes != null) {
            for (int i = m_extensionPrefixes.length-1; i >= 0; i--) {
                String[] prefixes = m_extensionPrefixes[i];
                for (int j = prefixes.length-1; j >= 0; j--) {
                    if (prefix.equals(prefixes[j])) {
                        int index = j + m_prefixes.length;
                        for (int k = i-1; k >= 0; k--) {
                            index += m_extensionPrefixes[k].length;
                        }
                        return index;
                    }
                }
            }
        }
        for (int i = m_prefixes.length-1; i >= 0; i--) {
            if (prefix.equals(m_prefixes[i])) {
                return i;
            }
        }
        return -1;
    }
    
    /**
     * Grow array of array of strings.
     *
     * @param base array to be grown (<code>null</code> is treated as zero
     * length)
     * @param items array of strings to be added at end of base array
     * @return array with added array of items
     */
    
    protected static String[][] growArray(String[][] base, String[] items) {
        if (base == null) {
            return new String[][] { items };
        } else {
            int length = base.length;
            String[][] grow = new String[length+1][];
            System.arraycopy(base, 0, grow, 0, length);
            grow[length] = items;
            return grow;
        }
    }
    
    /**
     * Shrink array of array of strings.
     *
     * @param base array to be shrunk
     * @return array with last set of items eliminated (<code>null</code> if
     * empty)
     */
    
    protected static String[][] shrinkArray(String[][] base) {
        int length = base.length;
        if (length == 1) {
            return null;
        } else {
            String[][] shrink = new String[length-1][];
            System.arraycopy(base, 0, shrink, 0, length-1);
            return shrink;
        }
    }
    
    /**
     * Append extension namespace URIs to those in mapping.
     *
     * @param uris namespace URIs to extend those in mapping
     */
    
    public void pushExtensionNamespaces(String[] uris) {
        m_extensionUris = growArray(m_extensionUris, uris);
        m_extensionPrefixes =
            growArray(m_extensionPrefixes, new String[uris.length]);
    }
    
    /**
     * Remove extension namespace URIs. This removes the last set of
     * extension namespaces pushed using {@link #pushExtensionNamespaces}.
     */
    
    public void popExtensionNamespaces() {
        m_extensionUris = shrinkArray(m_extensionUris);
        m_extensionPrefixes = shrinkArray(m_extensionPrefixes);
    }
    
    /**
     * Get extension namespace URIs added to those in mapping. This gets the
     * current set of extension definitions. The returned arrays must not be
     * modified.
     *
     * @return array of arrays of extension namespaces (<code>null</code> if
     * none)
     */
    
    public String[][] getExtensionNamespaces() {
        return m_extensionUris;
    }
    
    /**
     * Namespace declaration tracking information. This tracks all information
     * associated with an element that declares namespaces.
     */
    
    private static class DeclarationInfo
    {
        /** Depth of element making declaration. */
        public final int m_depth;
        
        /** Indexes of namespaces included in declarations. */
        public final int[] m_deltas;
        
        /** Prior prefixes for namespaces. */
        public final String[] m_priors;
        
        /** Simple constructor. */
        public DeclarationInfo(int depth, int[] deltas, String[] priors) {
            m_depth = depth;
            m_deltas = deltas;
            m_priors = priors;
        }
    }
}