/*
 * $Id: Cache.java 245 2008-05-13 03:56:59Z sugimotokenichi $
 * Copyright (C) 2004-2006 SUGIMOTO Ken-ichi
 * 作成日: 2006/04/30
 */
package feat2.impl;

import java.lang.ref.SoftReference;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;

/**
 * 指定の時間だけオブジェクトを保持するキャッシュ。put, get, cleanメソッドは
 * Cacheインスタンス自身で同期される。
 * @author SUGIMOTO Ken-ichi
 */
public class Cache {

    private HashMap buf;
    private long cleanTime;
    private long cleanupTotalTime;
    private int hits;
    private int requests;
    private int putCount;
    private int cleanCount;
    private long startTime;

    public Cache() {
        buf = new HashMap();
        cleanTime = 0;
        startTime = System.currentTimeMillis();
    }


    /**
     * オブジェクトを保存する。
     * @param key キー
     * @param obj 保存するオブジェクト
     * @param ttl オブジェクトを保管する最大時間。
     * ヒープが不足したときはこの時間より早くオブジェクトが破棄されることがある
     */
    synchronized public void put(Object key, Object obj, long ttl) {


        CacheEntry entry = new CacheEntry(obj, ttl);
        buf.put(key, entry);

        if ( ++putCount >= 10000 ) {
            clean(1000, 10000);
            putCount = 0;
        }
    }


    /**
     * キーに関連付けられたエントリを削除する。
     * @param key
     */
    synchronized public void remove(Object key) {
        buf.remove(key);
    }


    /**
     * 保管されたオブジェクトを返す。
     * @param key キー
     * @return オブジェクトが保管されていればその参照。オブジェクトが見つからないか、破棄されていたらnull
     */
    synchronized public Object get(Object key) {

        requests++;

        CacheEntry entry = (CacheEntry)buf.get(key);
        if ( entry != null ) {

            Object ret = entry.get();
            if ( ret == null )
                buf.remove(key);
            else
                hits++;
            return ret;

        }

        return null;

    }


    /**
     * キャッシュの不要なオブジェクトを開放する。ただし、ガーベッジコレクションは行わない。
     * @param interval 実行間隔。前回の呼び出しからこの時間が経過していなければ処理を行わない
     * @param limit 実行時間制限。この時間を超過するとまだ処理が終わっていなくてもリターンする
     */
    synchronized public void clean(long interval, long limit) {

        if ( cleanTime+interval < System.currentTimeMillis() ) {

            long start = System.currentTimeMillis();
            ArrayList removeList = new ArrayList();
            for(Iterator it = buf.keySet().iterator(); it.hasNext(); ) {

                Object key = it.next();
                CacheEntry entry = (CacheEntry)buf.get(key);

                if ( entry != null ) {

                    Object obj = entry.get();
                    if ( obj == null )
                        removeList.add(key);

                    if ( start+limit < System.currentTimeMillis() )
                        break;
                }
                else
                    removeList.add(key);

            }

            for(int i=0; i<removeList.size(); i++) {
                buf.remove(removeList.get(i));
            }

            cleanTime = System.currentTimeMillis();
            cleanupTotalTime += cleanTime - start;
            cleanCount++;

        }

    }


    synchronized public String getCacheStatus() {

        long running = System.currentTimeMillis() - startTime;

        int alive = 0;
        int purged = 0;
        int expired = 0;
        int empty = 0;
        for(Iterator it = buf.keySet().iterator(); it.hasNext(); ) {

            Object key = it.next();
            CacheEntry entry = (CacheEntry)buf.get(key);

            if ( entry == null )
                empty++;
            else if ( entry.isPurged() )
                purged++;
            else if ( entry.isExpired() )
                expired++;
            else
                alive++;

        }

        int rate = requests > 0 ? (hits*100/requests) : 0;

        return
            "entries "+buf.size()+
            " : alive "+alive+
            " : purged "+purged+
            " : expired "+expired+
            " : requests "+requests+" : hits "+hits+
            " : rate "+rate+
            " : clean "+cleanCount+
            " : clean up time "+cleanupTotalTime+"ms"+
            " : running time "+(running/3600000)+"h ("+(running/60000)+"m)";

    }

    private class CacheEntry {

        private long time;
        private long ttl;
        private SoftReference ref;

        private CacheEntry(Object obj, long ttl) {
            ref = new SoftReference(obj);
            this.ttl = ttl;
            this.time = System.currentTimeMillis();
        }

        private boolean isExpired() {
            return time+ttl < System.currentTimeMillis();
        }

        private boolean isPurged() {
            return ref.get() == null;
        }

        private Object get() {
            if ( time+ttl < System.currentTimeMillis() ) {
                return null;
            }
            else {
                return ref.get();
            }
        }

    }
}
