/*
 * $Revision: 1.4 $ $Date: 2005/04/13 13:54:44 $
 * Copyright (C) 2004-2007 SUGIMOTO Ken-ichi
 *
 * 作成日:2007/6/18
 */

package feat2.impl;

import java.text.SimpleDateFormat;
import java.util.Date;

import feat2.StringUtil;


/**
 * XMLを書き出すためのユーティリティクラス。
 * ドキュメントは次のトップレベル要素を一つだけ持つことができる。<br>
 * ・object<br>
 * ・list<br>
 * ・property<br>
 *
 * object, listの各要素に含むことができる子要素はそれぞれ<br>
 * object<br>
 * 　　object<br>
 * 　　list<br>
 * 　　property<br>
 * list<br>
 * 　　object<br>
 * 　　list<br>
 *     property<br>
 * TODO StringBufferからStringに変換してレスポンスに出力、では効率が悪いのでコンストラクタの引数にWriterをとって直接Writerに出力できるようにする。
 */
public class XMLWriter {

    private SimpleDateFormat dateFormat;
    private StringBuffer buf;
    private boolean close;


    public XMLWriter() {
        buf = new StringBuffer("<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\r\n<response status=\"success\">");
        close = false;
    }


    /**
     * object要素を開始する。
     * nameは親要素がobjectの場合のみ意味がある。
     * @param name String
     * @param type String
     */
    public void startObject(String name, String type) {
        buf.append("<object name=\"").append(name);
        if ( type != null )
            buf.append("\" type=\"").append(type);
        buf.append("\">");
    }


    public void endObject() {
        buf.append("</object>\r\n");
    }


    /**
     * list要素を開始する。
     * nameは親要素がobjectの場合のみ意味がある。
     * @param name String
     */
    public void startList(String name) {
        buf.append("<list name=\"").append(name).append("\">");
    }


    public void endList() {
        buf.append("</list>\r\n");
    }


    public void endDocument() {
        if ( !close ) {
            buf.append("</response>");
            close = true;
        }
    }


    public void writeBoolean(String name, boolean value) {
        writeProperty(name, value ? "true" : "false", "boolean");
    }


    public void writeNumber(String name, int value) {
        writeProperty(name, Integer.toString(value), "number");
    }


    public void writeNumber(String name, short value) {
        writeProperty(name, Short.toString(value), "number");
    }


    public void writeNumber(String name, long value) {
        writeProperty(name, Long.toString(value), "number");
    }


    public void writeNumber(String name, float value) {
        writeProperty(name, Float.toString(value), "number");
    }


    public void writeNumber(String name, double value) {
        writeProperty(name, Double.toString(value), "number");
    }


    public void writeNumber(String name, Number value) {
        writeProperty(name, value.toString(), "number");
    }


    public void writeString(String name, String value) {
        if ( value == null ) {
            writeProperty(name, null, "string");
        }
        else {
            writeProperty(name, StringUtil.escapeXMLText(value), "string");
        }
    }


    public void writeDate(String name, Date value) {

        if ( value == null ) {
            writeProperty(name, null, "date");
        }
        else {
            if ( dateFormat == null )
                dateFormat = new SimpleDateFormat("yyyy.MM.dd.HH.mm.ss.SSS");
            writeProperty(name, dateFormat.format(value), "date");
        }

    }


    private void writeProperty(String name, String value, String type) {
        buf.append("<property name=\"").append(name);
        buf.append("\" type=\"").append(type);
        if ( value == null ) {
            buf.append("\" isNull=\"true");
        }
        buf.append("\">").append(value).append("</property>");
    }


    /**
     * エラーを表すXMLを作る。
     * このメソッドを呼ぶ前に書いたものはなくなる。
     * @param errorCode
     */
    public void setError(String errorCode) {
        buf = new StringBuffer("<?xml version=\"1.0\" ?>\r\n<response status=\"");
        buf.append(errorCode).append("\" >");
        endDocument();
    }


    /**
     * エラーを表すXMLを作る。
     * このメソッドを呼ぶ前に書いたものはなくなる。
     * @param errorCode
     * @param message クライアントに送るメッセージ
     */
    public void setError(String errorCode, String message) {
        String encoded = StringUtil.escapeXMLText(message);
        buf = new StringBuffer("<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\r\n");
        buf.append("<response status=\"").append(errorCode).append("\" ");
        buf.append("message=\"").append(encoded).append("\">");
        endDocument();
    }


    /**
     * ドキュメントをクローズしてXMLの文字列を返す。
     * @return String
     */
    public String toString() {
        endDocument();
        return buf.toString();
    }

}
