/**
 * feedblog SearchScript
 *
 * @copyright 2013 feedblog Project (http://sourceforge.jp/projects/feedblog/)
 * @author Kureha Hisame (http://lunardial.sakura.ne.jp/) & Yui Naruse (http://airemix.com/)
 * @since 2009/02/27
 * @version　4.2.0.0
 */

// ブログ本体のHTMLファイルのURL
var mainPageUrl;

// 検索用ページURL
var searchPageUrl;

// 最新の記事を示すパスへの文字列
var latestXml;

// ログのリストが書かれたXMLのファイルパス
var logXmlUrl;

// 一画面あたりの表示記事数
var showLength;

/**
 * XMLファイルから読み込んだファイルのバリデートモード
 * 0 = 改行コード部分に<br/>を挿入
 * 1 = 改行コード部分に<br/>を挿入しない
 */
var validateMode;

// 検索結果をメモリ上に保持する変数です
var loadedEntries;

// fetchEntries 用のセマフォ
var fetchEntriesSemaphore = new Semaphore();

// 現在の検索語のキャッシュ
var currentSearchWords;

// ログのファイルリストを格納するグローバル変数です
var logData;

// コンボボックスのオブジェクトを格納するグローバル変数です
var comboBox;

// URL末尾用文字列（スクリプトを開いた瞬間のミリ秒を記録）
var urlSuffix;

/**
 * 記事を実際に生成します。この部分を編集することでデザインを変更可能です。
 * @param {Entry} entry 記事の情報が代入されたEntryオブジェクト
 * @param {String} drawitem 「本文」を描画すべきパネルのDIV要素のid
 * @param {String} renderto 「タイトル・更新日時・本文」の1日分の記事データを最終的に描画すべきパネルのDIV要素のid
 * @param {String} closed (Ext jsパネルオプション)記事をクローズ状態で生成するか否か
 */
function generatePanel(entry, drawitem, renderto, closed) {
	// プラグインを実行
	if ( typeof ($("#" + renderto).feedblog_contents_plugin) == "function") {
		$("#" + renderto).feedblog_contents_plugin({
			entry : entry
		});
	}

	// HTML用の配列を用意する
	var htmlBuffer = [];

	// 内部的に描画先IDを生成
	var feedblogContentId = "" + renderto + "_content_div";

	// 各要素をオブジェクトに描画
	$("#" + drawitem).html(entry.content);

	// ヘッダパネルを生成 class= .feedblog_header
	htmlBuffer.push("<div class='feedblog_header' onclick='closePanel(\"" + feedblogContentId + "\")'><span>" + entry.title + "</span></div>" +

	// 本体記事を作成 class= .feedblog_content
	"<div class='feedblog_content' id='" + feedblogContentId + "'><span>" + document.getElementById(renderto).innerHTML + "</span></div>");

	// 最終描画実施
	$("#" + renderto).html(htmlBuffer.join(""));
}

/**
 * システム表示画面を実際に生成します。この部分を編集することでデザインを変更可能です。
 * @param {Entry} entry 記事の情報が代入されたEntryオブジェクト
 * @param {String} drawitem パネルの本文を格納したDIV要素のid
 * @param {String} renderto 「タイトル・更新日時・本文」の1日分の記事データを焼き付けるDIV要素のid
 * @param {String} closed (Ext jsパネルオプション)記事をクローズ状態で生成するか否か
 */
function generateSystemPanel(entry, drawitem, renderto, closed) {
	// HTMLを生成する
	var htmlBuffer = [];

	// 描画先IDを生成
	var feedblogContentId = "" + renderto + "_content_div";

	// 各要素をオブジェクトに描画
	$("#" + drawitem).html(entry.content);

	// ヘッダパネルを生成 class= .feedblog_header
	htmlBuffer.push("<div class='feedblog_header' onclick='closePanel(\"" + feedblogContentId + "\")'><span>" + entry.title + "</span></div>" +

	// 本体記事を作成 class= .feedblog_content
	"<div class='feedblog_content' id='" + feedblogContentId + "'><span>" + document.getElementById(renderto).innerHTML + "</span></div>");

	$("#" + renderto).html(htmlBuffer.join(""));
}

/**
 * 検索フォーム及び結果表示フォームを生成するメソッドです。
 */
function generateForm() {
	var formBuffer = [];
	formBuffer.push("<form class='feedblog_searchform' onsubmit='javascript: searchDiary(); return false;'>▼ 検索語句を入力してください (語句を半角で区切るとAND、|で区切るとORで検索します)<br/>");
	formBuffer.push("<input type='text' id='feedblog_searchword' class='feedblog_searchword'><input type='submit' id='feedblog_execsearch' value='検索'><br/>");
	formBuffer.push("<input type='checkbox' id='feedblog_regexpOptionI' class='feedblog_regexpOptionI' checked='checked'/><label class='feedblog_searchform' for='feedblog_regexpOptionI'>大文字、小文字を区別しない</label><br/>");
	formBuffer.push("<br/>");
	formBuffer.push("▼ 検索対象ログ選択<br/>");
	formBuffer.push("<div id='feedblog_logselecter'></div>");
	formBuffer.push("<input type='checkbox' id='feedblog_allsearchcheck' class='feedblog_allsearchcheck' checked='checked'/>");
	formBuffer.push("<label class='feedblog_searchform' for='feedblog_allsearchcheck'>すべてのログに対して検索を行う</label>");
	formBuffer.push("<br/>");
	formBuffer.push("</form>");
	$("#feedblog_searchform").html(formBuffer.join(""));

	var resultAreaBuffer = "<div class='feedblog_result_status'></div>";

	$("#feedblog_resultwritearea").html(resultAreaBuffer);
}

/**
 * 全ての定数を取得・セットします
 */
function initialize() {
	// 初期値をhiddenパラメータより読み込みます
	mainPageUrl = $("#feedblog_mainpageurl").val();
	searchPageUrl = $("#feedblog_searchpageurl").val();
	latestXml = $("#feedblog_latestxml").val();
	logXmlUrl = $("#feedblog_loglistxmlurl").val();
	showLength = parseInt($("#feedblog_showlength").val());
	if (isNaN(showLength)) {
		showLength = 1;
	}
	validateMode = $("#feedblog_validatemode").val();
	
	// 初期値を設定します
	urlSuffix = +new Date();

	// 必要な環境を確認します
	var errorBuf = [];
	// 変数確認
	if (mainPageUrl === undefined) {
		errorBuf.push("設定値「feedblog_mainpageurl」が欠落しています。");
	}
	if (searchPageUrl === undefined) {
		errorBuf.push("設定値「feedblog_searchpageurl」が欠落しています。");
	}
	if (latestXml === undefined) {
		errorBuf.push("設定値「feedblog_latestxml」が欠落しています。");
	}
	if (logXmlUrl === undefined) {
		errorBuf.push("設定値「feedblog_loglistxmlurl」が欠落しています。");
	}
	if (showLength === undefined) {
		errorBuf.push("設定値「feedblog_showlength」が欠落しています。");
	}
	if (validateMode === undefined) {
		errorBuf.push("設定値「feedblog_validatemode」が欠落しています。");
	}
	// SHA-1関数確認
	try {
		if ( typeof (CryptoJS.SHA1) != "function") {
			errorBuf.push("crypt-jsモジュール(hmac-sha1.js)が読み込まれていません。");
		}
	} catch (ex) {
		errorBuf.push("crypt-jsモジュール(hmac-sha1.js)が読み込まれていません。");
	}

	// 描画エリアチェック
	if ($("#feedblog_searchform").length == 0) {
		errorBuf.push("描画エリア「feedblog_searchform」が存在しません。");
	}
	if ($("#feedblog_resultwritearea").length == 0) {
		errorBuf.push("描画エリア「feedblog_resultwritearea」が存在しません。");
	}
	if ($("#feedblog_writearea").length == 0) {
		errorBuf.push("描画エリア「feedblog_writearea」が存在しません。");
	}

	// エラーがある場合は以降の処理を継続しない
	if (errorBuf.length > 0) {
		alert("初期設定値に誤りがあります。\n詳細：\n" + errorBuf.join("\n"));
		return false;
	}

	return true;
}

/**
 * jQueryへのイベント登録です
 */
$(document).ready(function() {
	// 初期処理を実施
	if (!initialize()) {
		return false;
	}

	// ログ一覧のXMLをロードします
	logXMLLoader();

	// 各種パネルを生成します
	generateForm();
});

/**
 * jQueryでのパネル開閉を制御します
 */
function closePanel(id) {
	$("#" + id).slideToggle();
}

/**
 * 記事クラス
 * @param {Object} obj entry 要素の DOM オブジェクト
 */
function Entry(obj) {
	this.title = $("title:first", obj).text();
	if (this.title == "")
		requiredElementError(obj, "title");
	this.title = validateText(this.title);
	this.content = $("content:first", obj).text();
	this.content = validateText(this.content);
	this.id = $("id:first", obj).text();
	if (this.id == "")
		requiredElementError(obj, "id");
	this.date = $("updated:first", obj).text();
	if (this.date == "")
		requiredElementError(obj, "updated");
	this.date = validateData(this.date);
	this.category = $("category", obj);
}

/**
 * システム用記事クラス
 * @param {Object} obj entry 要素の DOM オブジェクト
 */
function SystemEntry(obj) {
	this.title = $("title:first", obj).text();
	this.title = validateText(this.title);
	this.content = $("content:first", obj).text();
	this.content = validateText(this.content);
	this.id = $("id:first", obj).text();
	this.date = $("updated:first", obj).text();
	this.date = validateData(this.date);
	this.category = $("category", obj);
}

/**
 * 記事内が単語群を全て含んでいるか
 * @param {Array} keywords 単語群
 * @param {String} regexpType 正規表現の検索モードを示す文字列
 * @return {boolean} bool 全て含んでいれば true、さもなくば false
 */
Entry.prototype.hasKeywords = function(keywords, regexpType) {
	// 正規表現が一致するかという判定"のみ"を行います
	for (var i = 0; i < keywords.length; i++) {
		// 正規表現チェック用のオブジェクトを用意します(OR条件は一時的に条件を置換)
		var reg = new RegExp('(?:' + keywords[i] + ')(?![^<>]*>)', regexpType);
		// 一致しなかったらその時点で脱出
		if (!reg.test(this.content) && !reg.test(this.title))
			return false;
	}
	return true;
};

/**
 * 呼び出すとDIV:id名:feedblog_writearea上のHTMLを削除し、ロードエフェクトを表示します
 */
function loadingEffect() {
	$("#feedblog_writearea").html('<div id="feedblog_drawpanel" class="feedblog_drawpanel"><div id="feedblog_drawitem" class="feedblog_drawitem">&nbsp;<\/div><\/div>');

	// ロード表示用のパネルを生成
	var systemEntry = new SystemEntry();
	systemEntry.title = "Now Loading .....";
	systemEntry.content = '<br/>長時間画面が切り替わらない場合はページをリロードしてください。<br/><br/>';
	generateSystemPanel(systemEntry, "feedblog_drawitem", "feedblog_drawpanel", false);

	// 結果表示エリアをリセット
	$("#feedblog_resultwritearea").html("<div class='feedblog_result_status'></div>");
}

/**
 * 記事データのエラー時の処理を行います
 */
function showError() {
	$("#feedblog_writearea").html('<div id="feedblog_drawpanel" class="feedblog_drawpanel"><div id="feedblog_drawitem" class="feedblog_drawitem">&nbsp;<\/div><\/div>');

	// エラー内容をパネルに描画
	var systemEntry = new SystemEntry();
	systemEntry.title = "エラー";
	var errorContent = [];
	errorContent.push('<br/>記事ファイル（XML）の取得に失敗しました。以下のような原因が考えられます。<br/><br/>');
	errorContent.push('・設定値「feedblog_latestxml」に正しいパスが設定されていない。<br/>');
	errorContent.push('・設定値「feedblog_loglistxmlurl」に正しいパスが設定されていない。<br/>');
	errorContent.push('・ローカル環境で起動している（必ずサーバにアップロードして実行してください）。<br/>');
	errorContent.push('<br/>');
	systemEntry.content = errorContent.join("\n");
	generateSystemPanel(systemEntry, "feedblog_drawitem", "feedblog_drawpanel", false);

	// 結果表示エリアをリセット
	$("#feedblog_resultwritearea").html("<div class='feedblog_result_status'></div>");
}

/**
 * 記事データのエラー時の処理を行います
 */
function notFoundError() {
	$("#feedblog_writearea").html('<div id="feedblog_drawpanel" class="feedblog_drawpanel"><div id="feedblog_drawitem" class="feedblog_drawitem">&nbsp;<\/div><\/div>');
	$("#feedblog_drawitem").html('<br/>検索条件に一致する記事は見つかりませんでした。<br/><br/>');

	// エラー内容をパネルに描画
	var systemEntry = new SystemEntry();
	systemEntry.title = "検索結果";
	systemEntry.content = '<br/>検索条件に一致する記事は見つかりませんでした。<br/><br/>';
	generateSystemPanel(systemEntry, "feedblog_drawitem", "feedblog_drawpanel", false);

	// 結果表示エリアをリセット
	$("#feedblog_resultwritearea").html("<div class='feedblog_result_status'></div>");
}

/**
 * 記事データのエラー時の処理を行います
 */
function requiredElementError(parent, name) {
	alert(parent.ownerDocument.URL + ": 必須な要素 " + name + " が存在しないか空な " + parent.tagName + " 要素が存在します");
}

/**
 * 日付のHTML表示用バリデーション処理を行います
 * @param {String} data RFC3339形式のdate-time文字列
 */
function validateData(data) {
	data = data.replace(/T/g, " ");

	// 秒数の小数点以下の部分はカットする
	data = data.substring(0, 19);

	return data;
}

/**
 * 記事本文のバリデーション処理を行います
 * @param {String} contents 記事の本文が格納されている文字列
 */
function validateText(contents) {
	// <br/>タグを挿入する
	if (validateMode == 0) {
		contents = contents.replace(/[\n\r]|\r\n/g, "<br />");
	}

	return contents;
}

/**
 * XML用に要素をエスケープします
 * @param {String} str エスケープを行いたい文字列
 */
function xmlAttrContentEscape(str) {
	// return str.replace(/&/g, '&amp;').replace(/</g, '&lt;').replace(/>/g, '&gt;').replace(/"/g, '&quot;');
	return str.replace(/&/g, '&amp;').replace(/</g, '&lt;').replace(/>/g, '&gt;').replace(/"/g, '&quot;').replace(/^[ ]+/mg, "&nbsp;").replace(/^[\t]+/mg, "");
}

/**
 * 記事本文に日付を付加します
 * @param {String} contents 記事の本文が格納されている文字列
 * @param {String} id 記事の初公開日を示す日付文字列
 */
function contentsWithid(contents, id) {
	// リンク用文末作成
	var hashTag = '<br/><div class="feedblog_content_footer"><a href="' + xmlAttrContentEscape(mainPageUrl) + '#' + xmlAttrContentEscape(id) + '" target="_blank">- この日の記事にリンクする -<\/a><\/div>';
	return contents + hashTag;
}

/**
 * 強調タグを追加します
 * @param {String} word 強調したい語句
 */
function emphasizeWord(word) {
	return '<span style="background-color: red;">' + word + '</span>';
}

/**
 * 長い順に並べるための比較関数です
 * @param {String} a 比較対象(1)
 * @param {String} b 比較対象(2)
 */
function compareLengthDecrease(a, b) {
	a = a.length;
	b = b.length;
	return a > b ? -1 : a < b ? 1 : 0;
}

/**
 * セマフォ制御用のオブジェクトです
 */
function Semaphore() {
	this.id = null;
	this.count = 0;
	this.buf = [];
	this.xhrs = [];
}

/**
 * セマフォ初期化用の関数です
 */
Semaphore.prototype.init = function() {
	while (this.xhrs.length > 0) {
		this.xhrs.shift().abort();
	}
	this.id = Math.random();
	this.count = 0;
	this.buf = [];
};

/**
 * ログファイル選択用のコンボボックスをid名:feedblog_logselecterに生成します
 */
function logXMLLoader() {
	// ログ用のXMLを読み込みます
	jQuery.ajax({
		url : logXmlUrl + '?time=' + urlSuffix,
		method : "GET",
		error : showError,
		success : function(xmlData) {
			var separateTag = xmlData.getElementsByTagName("file");
			logData = new Array(separateTag.length);

			// 読み込んだ要素をStoreに格納して表示
			var boxBuffer = [];
			boxBuffer.push("<select class='feedblog_logselecter' id='feedblog_logbox'>");
			for (var i = 0; i < separateTag.length; i++) {
				boxBuffer.push("<option value='" + separateTag[i].getElementsByTagName("path")[0].firstChild.nodeValue + "'>" + separateTag[i].getElementsByTagName("display")[0].firstChild.nodeValue + "</option>");
				logData[i] = separateTag[i].getElementsByTagName("path")[0].firstChild.nodeValue;
			}
			boxBuffer.push("</select>");

			// コンボボックス要素を生成
			$("#feedblog_logselecter").html(boxBuffer.join(""));
		}
	});
}

/**
 * 検索単語を取得します
 */
function getSearchWords() {
	var searchWord = document.getElementById("feedblog_searchword").value;
	if (searchWord == "")
		return null;
	var searchWords = [];

	// 検索単語をサニタイジングします
	// HTMLのメタ文字
	searchWord = xmlAttrContentEscape(searchWord);
	// 正規表現のメタ文字
	searchWord = searchWord.replace(/([$()*+.?\[\\\]^{}])/g, '\\$1');
	// 半角スペースで配列に分割
	searchWords = searchWord.replace(/^\s+|\+$/g, '').split(/\s+/);
	// 正規表現の選択を長い順に並び替えます(AND条件)
	searchWords.sort(compareLengthDecrease);

	return searchWords.length == 0 ? null : searchWords;
}

/**
 * 文章内を特定の単語で検索し、一致した部分を強調表示タグで置き換えます
 * @param {String} searchWord 探索する単語
 * @param {String} plainText 探索を行う文章
 * @param {String} regexpType 正規表現の検索モードを示す文字列
 */
function complexEmphasize(searchWord, plainText, regexpType) {
	// 正規表現の選択を長い順に並び替える
	searchWord = searchWord.split('|').sort(compareLengthDecrease).join('|');
	// タグの内側でないことを確認する正規表現を追加
	var pattern = new RegExp('(?:' + searchWord + ')(?![^<>]*>)', regexpType);

	var result = [];
	var currentIndex = -1;
	// 現在マッチしている部分文字列の開始位置
	var currentLastIndex = -1;
	// 現在マッチしている部分文字列の、現在の末尾
	var m;
	// 正規表現マッチの結果配列
	while ( m = pattern.exec(plainText)) {
		if (m.index > currentLastIndex) {
			// 新しい部分文字列へのマッチが始まったので、そこまでの文字列をバッファに書き出す
			if (currentIndex < currentLastIndex)
				result.push(emphasizeWord(plainText.substring(currentIndex, currentLastIndex)));
			result.push(plainText.substring(currentLastIndex, m.index));
			// 開始位置の更新
			currentIndex = m.index;
		}
		// 末尾位置を更新
		currentLastIndex = pattern.lastIndex;
		// 次の正規表現マッチは今マッチした文字の次の文字から
		pattern.lastIndex = m.index + 1;
	}
	// 残った文字列を書き出す
	if (currentIndex < currentLastIndex)
		result.push(emphasizeWord(plainText.substring(currentIndex, currentLastIndex)));
	result.push(plainText.substring(currentLastIndex));

	// 結合して返す
	return result.join('');
}

/**
 * 検索結果を分割して表示します(2回目以降呼び出し)
 * @param {int} showLength 一回の画面に表示する記事数
 * @param {int} startIndex 表示を開始する記事のインデックス
 */
function showEntriesRange(showLength, startIndex) {
	// メモリ上から記事データをロード
	var entries = loadedEntries;

	// 表示インデックスが範囲外の場合はエラーパネルを表示して終了
	if (startIndex < 0 || entries.length <= startIndex) {
		showError();
		return;
	}

	var stringBuffer = [];

	// リミッターを設定する
	var loopLimit = (showLength + startIndex > entries.length) ? entries.length : showLength + startIndex;
	var indexShowEntries = loopLimit + 1;

	for (var i = startIndex; i < loopLimit; i++) {
		stringBuffer.push('<div class="feedblog_drawpanel" id="feedblog_drawpanel');
		stringBuffer.push(i);
		stringBuffer.push('"><div class="feedblog_drawitem" id="feedblog_drawitem');
		stringBuffer.push(i);
		stringBuffer.push('"><\/div><\/div>');
	}
	$("#feedblog_writearea").html(stringBuffer.join(''));

	stringBuffer.length = 0;
	for ( i = startIndex; i < loopLimit; i++) {
		var entry = entries[i];
		generatePanel(entry, "feedblog_drawitem" + i, "feedblog_drawpanel" + i, false);
	}

	// メニュー表示用バッファ
	var menuBuffer = [];
	menuBuffer.push("<div class='feedblog_pager_wrapper'>");
	menuBuffer.push("<ul class='feedblog_pager'>");

	// ブランクエリアを挟む
	menuBuffer.push("<li class='feedblog_pager_blank'></li>");

	// 左パネルの表示制御
	if (startIndex - showLength >= 0) {
		menuBuffer.push("\<li class='feedblog_pager_goback'><span class='feedblog_pager_goback' onclick='showEntriesRange(" + showLength + ", " + (startIndex - showLength) + "); return false;'>\< 前の" + showLength + "件を表示</span\></li>");
	} else {
		menuBuffer.push("\<li class='feedblog_pager_goback'>\< 前の" + showLength + "件を表示</a\></li>");
	}

	// 中央のパネルの表示制御
	menuBuffer.push("<li class='feedblog_pager_center'>[ ");
	var menuNumbers = Math.ceil(entries.length / showLength);
	for ( i = 0; i < menuNumbers; i++) {
		if (startIndex / showLength == i) {
			menuBuffer.push(i + " ");
		} else {
			menuBuffer.push("<span class='feedblog_pager_center' onclick='showEntriesRange(" + showLength + ", " + (i * showLength) + "); return false;'>");
			menuBuffer.push(i);
			menuBuffer.push("</span> ");
		}
	}
	menuBuffer.push("]</li>");

	// 右パネルの表示制御
	if (entries.length > startIndex + showLength) {
		menuBuffer.push("\<li class='feedblog_pager_gonext'><span class='feedblog_pager_gonext' onclick='showEntriesRange(" + showLength + ", " + (startIndex + showLength) + "); return false;'>次の" + showLength + "件を表示 \></span\></li>");
	} else {
		menuBuffer.push("\<li class='feedblog_pager_gonext'>次の" + showLength + "件を表示 \></li>");
	}

	// ブランクエリアを挟む
	menuBuffer.push("<li class='feedblog_pager_blank'></li>");

	menuBuffer.push("</ul></div>");

	// 検索結果を表示します
	$("#feedblog_resultwritearea").html("<div class='feedblog_result_status'>" + entries.length + "件の記事が該当しました /  " + (startIndex + 1) + "～" + loopLimit + "件目までを表示中<br/></div>" + menuBuffer.join(""));
}

/**
 * 検索時のjQuery.ajaxのcallback関数
 */
function fetchEntries(xmlData) {
	// 大文字小文字を区別するかを取得します
	var regexpOptionI = document.getElementById("feedblog_regexpOptionI");
	var regexpType = regexpOptionI ? "ig" : "g";

	// entry要素のみを切り出します
	var entries = xmlData.getElementsByTagName("entry");

	// entry要素の回数だけ実行します
	for (var j = 0; j < entries.length; j++) {
		var entry = new Entry(entries[j]);

		// 正規表現が一致した場合は、強調表現処理を行います
		if (entry.hasKeywords(currentSearchWords, regexpType)) {
			// 強調表現を実行します
			entry.title = complexEmphasize(currentSearchWords.join("|"), entry.title, regexpType);
			entry.content = complexEmphasize(currentSearchWords.join("|"), entry.content, regexpType);

			fetchEntriesSemaphore.buf.push(entry);
		}
	}

	// セマフォのカウンタを減少させます (Ajaxとの同期のため)
	fetchEntriesSemaphore.count--;

	// 全てのログを読み終わったら表示
	if (fetchEntriesSemaphore.count == 0) {
		var entries = fetchEntriesSemaphore.buf;

		// 一軒も検索にヒットしなかった場合は専用のパネルを表示して終了
		if (entries.length == 0) {
			notFoundError();
			return;
		}

		// entryを更新時間でソート
		entries = entries.sort(function(a, b) {
			a = a.updated;
			b = b.updated;
			return a > b ? -1 : a < b ? 1 : 0;
		});

		loadedEntries = entries;

		// 表示ロジック呼び出し
		showEntriesRange(showLength, 0);
	}
}

/**
 * 「探索」ボタンを押されたときに呼び出されるメソッドです
 */
function searchDiary() {
	// 検索結果フィールドをクリアします
	document.getElementById("feedblog_writearea").innerHTML = "";

	// 探索したい単語を取得します
	currentSearchWords = getSearchWords();
	if (!currentSearchWords) {
		alert("検索対象の単語が入力されていません");
		// 検索結果の欄をリセットします
		$("#feedblog_resultwritearea").html("<div class='feedblog_result_status'></div>");
		return;
	}

	// ロードエフェクトを表示します
	loadingEffect();

	// 全チェックを取得します
	var allCheckedFlag = document.getElementById("feedblog_allsearchcheck").checked;

	// セマフォを初期化
	fetchEntriesSemaphore.init();
	// 記事が全検索モードか否かをチェックします
	var urls = null;
	if (allCheckedFlag == true) {
		// 全記事検索なので全てのログのURL
		urls = logData;
	} else {
		// 単独記事探索なので、選んだログのURL
		urls = [document.getElementById("feedblog_logbox").options[document.getElementById("feedblog_logbox").selectedIndex].value];
	}
	fetchEntriesSemaphore.urls = urls;
	fetchEntriesSemaphore.count = urls.length;
	for ( i = 0; i < urls.length; i++) {
		var xhr = new jQuery.ajax({
			url : urls[i] + '?time=' + urlSuffix,
			method : "GET",
			async : true,
			success : fetchEntries,
			error : showError
		});
		fetchEntriesSemaphore.xhrs.push(xhr);
	}
}
