<?php
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2007 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.beans.annotation
 */
/**
 * @file BeanConstantAnnotationAccessor.php
 * @brief bean annotation accesor
 * @author <a href="mailto:kent@guarana.cc">ISHITOYA Kentaro</a>
 * @version $Id: BeanConstantAnnotationAccessor.php 2 2007-07-11 10:37:48Z ishitoya $
 */

require_once "ficus/beans/Bean.php";
require_once "ficus/lang/reflect/annotation/ReflectionAnnotationProperty.php";
require_once "ficus/lang/reflect/annotation/ReflectionAnnotationClass.php";

/**
 * @class Ficus_BeanConstantAnnotationAccessor
 */
class Ficus_BeanConstantAnnotationAccessor{
    const METHOD_SIGNATURE = '/^(get|has)([a-zA-Z][a-zA-Z0-9_]*)/';

    /**
     * @var $annotations
     */
    protected $annotations = null;

    /*
     * construct with bean
     * @param $annotations Ficus_ReflectionAnnotation
     */
    protected function __construct($annotations){
        $this->annotations = $annotations;
    }

    /**
     * get accessor
     * @param $bean Ficus_Bean bean to parse
     * @throw Ficus_IllegalBeanException illegal bean.
     */
    public static final function getAccessor($bean){
        Ficus_Assert::isInstanceOf($bean, "Ficus_Bean");
        
        $class = new Ficus_ReflectionClass($bean);
        $annotations = array_merge($class->getConstants(),
                                   $class->getStaticProperties());

        return new
            Ficus_BeanConstantAnnotationAccessor($annotations);
    }

    /**
     * get constant
     * @param $name string of constant to get
     * @return mixed value of constant
     * @throw Ficus_ConstantNotFoundException when name was not exist
     */ 
    public function get($name){
        if($this->has($name)){
            return $this->annotations[$name];
        }else{
            throw new Ficus_ConstantNotFoundException("$name constant is not found");
        }
    }

    /**
     * check for constant existans
     * @param $name string name of constant
     * @return boolean true if exists
     */
    public function has($name){
        if(isset($this->annotations[$name])){
            return true;
        }
        return false;
     }
    
    /**
     * get Property with name
     * @return Ficus_ReflectionAnnotationProperty annotations
     */
    public function getAnnotations(){
        return $this->annotations;
    }

    /**
     * syntax suger
     * @param $name string name of function
     * @param $argument array of argument
     */
    public function __call($name, $argument){
        if(preg_match(self::METHOD_SIGNATURE, $name, $matches)){
            $operator = $matches[1];

            $constant = $this->getProperName($matches[2]);
            switch($operator){
              case "has" :
                return $this->has($constant);
            }

            if(is_null($constant)){
                throw new Ficus_ConstantNotFoundException("$name constant is not found");
            }

            switch($operator){
                case "get" :
                    return $this->get($constant);
            }
        }
        $constant = $this->getProperName($name);
        if(is_null($constant)){
            throw new Ficus_ConstantNotFoundException("$name constant is not found");
        }
        return $this->get($constant);
    }

    /**
     * get proper name from given string
     * @param $name string name of constant
     * @return string proper name
     */
    protected function getProperName($name){
        $constant = strtolower($name[0]) . substr($name, 1);
        if($this->has($constant)){
            return $constant;
        }
        $constant = ucfirst($name);
        if($this->has($constant)){
            return $constant;
        }
        return null;
    }
}
?>
