<?php 
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2007 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.collection
 */
/**
 * @file SimpleMap.php
 * @brief This class is implementation of interface Map.
 * @author <a href="mailto:kent@guarana.cc">ISHITOYA Kentaro</a>
 * @author <a href="mailto:sumi@wakhok.ac.jp">SUMI Masafumi</a>
 * @version $Id: SimpleMap.php 2 2007-07-11 10:37:48Z ishitoya $
 */

require_once("ficus/collection/Map.php");
require_once("ficus/exception/IllegalArgumentException.php");

/**
 * @class Ficus_SimpleMap
 * Simple implementation of Ficus_Map
 */
class Ficus_SimpleMap implements Ficus_Map{

    /**
     * @var $elements array elements
     */
    protected $elements = array();

    /**
     * constructor
     * @param $keys Array key array of elements
     * @param $elements Array elements to add this map
     * @throw Ficus_IllegalArgumentException illegal argument.
     */
    public function __construct($keys = array(), $elements = array()){
        if(is_array($elements) == false || is_array($elements) == false){
            throw new Ficus_IllegalArgumentException("passed parameter is not an array");
        }
        if(count($elements) !== count($keys)){
            throw new Ficus_IllegalArgumentException("passed keys and elements is not same size");
        }

        while($key = current($keys)){
            $this->put($key, current($elements));
            next($keys);
            next($elements);
        }
    }
    
    /**
     * add element to Collection
     * @param $key key of element to add
     * @param $element Element to add the Map
     * @return value returns old value if the key found in Map
     * @throw Ficus_IllegalArgumentException illegal argument.
     */
    public function put($key, $element){
        if(is_null($key)){
            throw new Ficus_IllegalArgumentException("passed key is null");
        }
        
        $oldvalue = null;
        if(isset($this->elements[$key])){
            $oldvalue = $this->elements[$key];
        }
        $this->elements[$key] = $element;
        return $oldvalue;
    }
    
    /**
     * clear all elements in the Map.
     */
    public function clear(){
        $this->elements = array();
    }

    /**
     * remove element from the Map
     * @param $key string element to remove
     * @return element returns old value if key exists
     */
    public function remove($key){
        $oldvalue = null;
        if(isset($this->elements[$key])){
            $oldvalue = $this->elements[$key];
            unset($this->elements[$key]);
        }
        return $oldvalue;
    }

    /**
     * get element from the Map
     * @param $key string element to get
     * @return element
     */
    public function get($key){
        if(isset($this->elements[$key])){
            return $this->elements[$key];
        }
        return null;
    }
    
    /**
     * get size of the Map 
     * @return integer size of this Map 
     */
    public function size(){
        return count($this->elements);
    }

    /**
     * to array
     * @return Array array presentation of Map
     */
    public function toArray(){
        return $this->elements;
    }

    /**
     * returns true if Map is empty
     * @return boolean true if the Map is empty
     */
    public function isEmpty(){
        return empty($this->elements);
    }

    /**
     * returns key array from the Map
     * @return Array array of the keys of Map
     */
    public function keys(){
        return array_keys($this->elements);
    }
    
    /**
     * return value array from the Map
     * @return Array array of the values of Map
     */
    public function values(){
        return array_values($this->elements);
    }

    /**
     * rewind elements
     */
    public function rewind() {
        reset($this->elements);
    }

    /**
     * return value of current array pointer position
     * @return Object value of current array pointer position
     */
    public function current() {
        return current($this->elements);
    }

    /**
     * return key of current array pointer position
     * @return string key of current position
     */
    public function key() {
        return key($this->elements);
    }

    /**
     * move to next array pointer position and return next var
     * @return Object next pointer position value
     */
    public function next() {
        return next($this->elements);
    }

    /**
     * check is current position is legal array pointer position
     * @return boolean returns true if its valid
     */
    public function valid() {
        //current key is NOT null
        return !(is_null(key($this->elements)));
    }
}
