<?php 
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2007 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.lang
 */
/**
 * @file FatalErrorHandler.php
 * @brief handles fatal error and write it in log file
 * @author <a href="mailto:kent@guarana.cc">ISHITOYA Kentaro</a>
 * @author <a href="mailto:sumi@wakhok.ac.jp">SUMI Masafumi</a>
 * @version $Id: FatalErrorHandler.php 2 2007-07-11 10:37:48Z ishitoya $
 */

require_once("ficus/logging/Logger.php");
require_once("ficus/io/PropertyFileReader.php");
require_once("ficus/exception/NotReadyException.php");
/**
 * @class Ficus_FatalErrorHandler
 */
class Ficus_FatalErrorHandler {

	const CONFIG_FILE = "logger";
    const LOGGING_STARTED = 0;
    const LOGGING_NOTSTARTED = 1;
    const INI_DISPLAY_ERRORS = "display_errors";
    const INI_OUTPUT_BUFFERING = "output_buffering";
    const INI_ERROR_LOG = "error_log";
    const ERROR_FILE = "error_file";
    const ERROR_BACKTRACE = "error_backtrace";
    
    /**
     * @var $status integer status 
     */
    private static $status = null;
    
    /**
     * @var $filename string logfile 
     */
    private static $filename = null;
    
    /**
     * @var $backtrace boolean backtrace 
     */
    private static $backtrace = false;
    
    /**
     * @var $isInitialized boolean isInitialized
     */
    private static $isInitialized = false;
    
    /** 
     * @var $isAvailable boolean isAvailable
     */
    private static $isAvailable = false;
    
    /** 
     * @var $oldValues array oldValues
     */
    private static $oldValues = array();

    /**
     * initialize parameters
     */
    private static function initialize(){
        $ini = Ficus_PropertyFileReader::read(self::CONFIG_FILE);
        self::$filename = dirname(__FILE__) . "/../" . $ini[self::ERROR_FILE];
        self::$backtrace = (bool)$ini[self::ERROR_BACKTRACE];
        
        if((self::isOptionAvailable(self::INI_DISPLAY_ERRORS) == false &&
            self::$oldValues[self::INI_DISPLAY_ERRORS] == false) ||
           self::isOptionAvailable(self::INI_ERROR_LOG) == false){
            self::$isAvailable = false;
        }else{
            self::$isAvailable = true;
        }
        self::$status = self::LOGGING_NOTSTARTED;
        self::$isInitialized = true;
    }

    /**
     * checkINIOption is available
     * @param $option string ini option
     * @return boolean true if available
     */
    private static function isOptionAvailable($option){
        if(is_string($option) == false || empty($option)){
            return false;
        }
        self::$oldValues[$option] = ini_get($option);
        $result = ini_set($option, "true");
        $result2 = ini_set($option, self::$oldValues[$option]);
        
        if((self::$oldValues[$option] !== false && $result === false) ||
           (self::$oldValues[$option] === false && $result === false &&
            $result2 === false)){
            return false;
        }
        return true;
    }
    
    /**
     * check ini_set is available
     * @return boolean true if avaliable
     */
    public static function isAvailable(){
        if(self::$isInitialized == false){
            self::initialize();
        }
        return self::$isAvailable;
    }
    
    /**
     * start logging
     *
     * @throw Ficus_NotReadyException error handler is not ready.
     */
    public static function startLogging(){
        if(self::isAvailable() == false){
            throw new Ficus_NotReadyException("Fatal Error Handler is not Ready.");
        }else if(self::$status == self::LOGGING_NOTSTARTED){
            ini_set(self::INI_DISPLAY_ERRORS, true);
            ini_set(self::INI_ERROR_LOG, self::$filename);

            ob_start(array("Ficus_FatalErrorHandler", "fatal_error_handler"));
            self::$status = self::LOGGING_STARTED;
        }
    }

    /**
     * end logging
     *
     * @throw Ficus_NotReadyException error handler is not ready.
     */
    public static function endLogging(){
        if(self::isAvailable() == false){
            throw new Ficus_NotReadyException("Fatal Error Handler is not Ready.");
        }else if(self::$status == self::LOGGING_STARTED){
            ob_end_flush();

            ini_set(self::INI_DISPLAY_ERRORS,
                    self::$oldValues[self::INI_DISPLAY_ERRORS]);
            ini_set(self::INI_ERROR_LOG,
                    self::$oldValues[self::INI_ERROR_LOG]);
            self::$status = self::LOGGING_NOTSTARTED;
        }
    }

    /**
     * fatal error handler checks buffer all the time
     * @param $buffer string output buffer
     */
    public static function fatal_error_handler($buffer) {
        if(preg_match("/<b>Fatal error<\/b>.*/", $buffer, $match)){
            $log = "\n" . trim($match[0]);

            if(self::$backtrace){
                $buf = explode("\n", $buffer);
                foreach($buf as $backtrace){
                    if(preg_match("/^#/", $backtrace)){
                        $log .= "\n" . $backtrace;
                    }
                }
            }
            $log .= "\n";
            error_log($log);
        }
        return $buffer;
    }
}
?>
