<?php
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2007 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.lang
 */
/**
 * @file Types.php
 * @brief Types for php
 * @author <a href="mailto:sumi@wakhok.ac.jp">SUMI Masafumi</a>
 * @version $Id: Types.php 2 2007-07-11 10:37:48Z ishitoya $
 * 
 * Util Type utility.
 */
require_once("ficus/lang/reflect/ReflectionClass.php");
require_once("ficus/exception/TypeMismatchException.php");
/**
 * @class Ficus_Types
 */
class Ficus_Types
{
    /**
     * cast var to specified type
     * @param $value mixed target value
     * @param $type string type to cast
     * @param $type casted value
     */
    public static function cast($var, $type){
        switch($type){
        case("int") : return (int)$var;
        case("string") : return (string) $var;
        case("array") : return (array) $var;
        case("double") : return (double) $var;
        case("float") : return (float) $var;
        case("boolean") : return (boolean) $var;
        }

        throw new Ficus_ClassCastException("not supported type :$type");
    }
    
    /**
     * to string of object.
     *
     * @param $obj mixed scalar or object defined toString or String.
     * @return String of $obj.
     * @throw Ficus_TypeMismatchException $obj has no toString().
     */
    public static function toStringOf($obj) {
        if (is_string($obj)) {
            return $obj;
        } else if (is_scalar($obj)) {
            return strval($obj);
        } else if (method_exists($obj, "toString")) {
            return $obj->toString();
        } else if (method_exists($obj, "__toString")) {
			return $obj->__toString();
		} else if ($obj === NULL) {
            return '';
		} else {
            throw new Ficus_TypeMismatchException("toString not defined in "
                                                . gettype($obj));
        }
    }

    /**
     * compare same class of string.
     *
     * @param $left mixed left argument.
     * @param $right mixed right argument.
     * @return int compare left and right.
     */
    public static function compareByToString($left, $right) {
		$left = self::toStringOf($left);
		$right = self::toStringOf($right);
        return strcmp($left, $right);
		/* this code cause of seg fault!*/
        //return strcmp(self::toStringOf($left), self::toStringOf($right));
    }

    /**
     * equals same class of string.
     *
     * @param $left mixed left argument.
     * @param $right mixed right argument.
     * @return int left equals right.
     */
    public static function equalsByToString($left, $right) {
        return (self::toStringOf($left) == self::toStringOf($right));
    }

    /**
     * to array form of object.
     *
     * @param $obj mixed object defined toArray
     * @return array form of $obj.
     * @throw Ficus_TypeMismatchException $obj has no toArray().
     */
    public static function toArrayOf($obj) {
        if (is_array($obj) || $obj instanceof stdClass) {
            $ret = array();
            foreach($obj as $key => $element){
                if($element instanceof stdClass){
                    $casted = (array)$element;
                    foreach($casted as $castedKey => $castedElement){
                        if(is_object($castedElement) ||
                           is_array($castedElement)){
                            $casted[$castedKey] =
                                self::toArrayOf($castedElement);
                        }
                    }
                    $ret[$key] = $casted;
                }else if(is_object($element) || is_array($element)){
                    $ret[$key] = self::toArrayOf($element);
                }else{
                    $ret[$key] = $element;
                }
            }
            return $ret;
        } else if (method_exists($obj, "toArray")) {
            return $obj->toArray();
        } else if (is_object($obj)) {
			return self::objectToArray($obj);
		} else {
            throw new Ficus_TypeMismatchException("toArray does not defined in "
                                                . gettype($obj) . ":" . $obj);
        }
    }

	/**
	 * convert object to array
	 * @param $object mixed object to convert
	 * @return array array of object array form
	 */
	private function objectToArray($object){
		$result = array();
		$class = new Ficus_ReflectionClass($object);
		$props = $class->getAllProperties();

		foreach($props as $prop){
			$name = $prop->getName();
            $value = null;
			if($prop->isPublic()){
				$value = $prop->getValue($object);
			}else{
				$getMethod = "get" . ucfirst($name);
				
				if(method_exists($object, $getMethod)){
					$method = $class->getMethod($getMethod);
				}else if(method_exists($object, $name)){
					$method = $class->getMethod($name);
				}else{
					continue;
				}

				$value = $object->{$method->getName()}();
			}
            if(is_object($value)){
                $value = self::objectToArray($value);
            }else if(is_array($value)){
                $value = self::toArrayOf($value);
            }
            $result[$name] = $value;
		}
		return $result;
	}
}
?>
