/****************************************************************************
 *
 *	Copyright (c) 1999-2009, Watanabe Lab, School of Media Science,
 *	Tokyo University of Technology, All rights reserved.
 *
 *	Redistribution and use in source and binary forms,
 *	with or without modification, are permitted provided that the
 *	following conditions are met:
 *
 *		- Redistributions of source code must retain the above
 *			copyright notice, this list of conditions and the
 *			following disclaimer.
 *
 *		- Redistributions in binary form must reproduce the above
 *			copyright notice, this list of conditions and the
 *			following disclaimer in the documentation and/or
 *			other materials provided with the distribution.
 *
 *		- Neither the name of the copyright holders nor the names
 *			of its contributors may be used to endorse or promote
 *			products derived from this software without specific
 *			prior written permission.
 *
 *	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *	"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *	LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 *	FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 *	COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 *	INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 *	(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 *	SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 *	HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 *	STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 *	IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 *	POSSIBILITY OF SUCH DAMAGE. 
 *
 ****************************************************************************/
/****************************************************************************
 *
 *	Copyright (c) 1999-2009, Watanabe Lab, School of Media Science,
 *	Tokyo University of Technology, All rights reserved.
 *
 *	本ソフトウェアおよびソースコードのライセンスは、基本的に
 *	「修正 BSD ライセンス」に従います。以下にその詳細を記します。
 *
 *	ソースコード形式かバイナリ形式か、変更するかしないかを問わず、
 *	以下の条件を満たす場合に限り、再頒布および使用が許可されます。
 *
 *	- ソースコードを再頒布する場合、上記の著作権表示、本条件一覧、
 *		および下記免責条項を含めること。
 *
 *	- バイナリ形式で再頒布する場合、頒布物に付属のドキュメント等の
 *		資料に、上記の著作権表示、本条件一覧、および下記免責条項を
 *		含めること。
 *
 *	- 書面による特別の許可なしに、本ソフトウェアから派生した製品の
 *		宣伝または販売促進に、本ソフトウェアの著作権者の名前または
 *		コントリビューターの名前を使用してはならない。
 *
 *	本ソフトウェアは、著作権者およびコントリビューターによって「現
 *	状のまま」提供されており、明示黙示を問わず、商業的な使用可能性、
 *	および特定の目的に対する適合性に関す暗黙の保証も含め、またそれ
 *	に限定されない、いかなる保証もないものとします。著作権者もコン
 *	トリビューターも、事由のいかんを問わず、損害発生の原因いかんを
 *	問わず、かつ責任の根拠が契約であるか厳格責任であるか(過失その
 *	他の)不法行為であるかを問わず、仮にそのような損害が発生する可
 *	能性を知らされていたとしても、本ソフトウェアの使用によって発生
 *	した(代替品または代用サービスの調達、使用の喪失、データの喪失、
 *	利益の喪失、業務の中断も含め、またそれに限定されない)直接損害、
 *	間接損害、偶発的な損害、特別損害、懲罰的損害、または結果損害に
 *	ついて、一切責任を負わないものとします。
 *
 ****************************************************************************/
#include <FK/BSplCurve.h>

using namespace std;

fk_BSplCurve::fk_BSplCurve(void)
{
	setObjectType(FK_BSPLCURVE);
	setOrdinate(4);
	return;
}

fk_BSplCurve::~fk_BSplCurve()
{
	ctrlPos.clear();
	knotVec.clear();
	return;
}

bool fk_BSplCurve::IsSame(double a, double b)
{
	return ((a + FK_EPS) > b && (a - FK_EPS) < b);
}

bool fk_BSplCurve::IsBig(double a, double b)
{
	return ((a + FK_EPS) >= b);
}

double fk_BSplCurve::PosBasis(int i, int k, double t)
{
	double	p, q;

	if(IsSame(t, 0.0)) {
		if(i == 0) return 1.0;
		else return 0.0;
	}

	if(IsSame(t, 1.0)) {
		if(i == int(knotVec.size())-k-1) return 1.0;
		else return 0.0;
	}

	if(k == 1) {
		if(IsSame(knotVec[i], t)) return 1.0;
		else if(IsSame(knotVec[i], knotVec[i+1])) return 0.0;
		else if(IsSame(knotVec[i+1], t)) return 0.0;
		else if(knotVec[i] < t && t < knotVec[i+1]) return 1.0;
		else return 0.0;
	}

	if(IsSame(knotVec[i+k-1], knotVec[i]) || IsSame(t, knotVec[i])) {
		p = 0.0;
	} else {
		p = ((t - knotVec[i])/(knotVec[i+k-1] - knotVec[i])) *
			PosBasis(i, k-1, t);
	}

	if(IsSame(knotVec[i+k], knotVec[i+1]) || IsSame(knotVec[i+k], t)) {
		q = 0.0;
	} else {
		q = ((knotVec[i+k] - t)/(knotVec[i+k] - knotVec[i+1])) *
			PosBasis(i+1, k-1, t);
	}

	return (p + q);
}

double fk_BSplCurve::DiffBasis(int i, int k, double t)
{
	double	a, b, p, q;

	if(k == 1) return 0.0;

	if(IsSame(knotVec[i+k-1], knotVec[i])) {
		p = 0.0;
	} else {
		a = PosBasis(i, k-1, t) + (t - knotVec[i])*DiffBasis(i, k-1, t);
		p = a/(knotVec[i+k-1] - knotVec[i]);
	}

	if(IsSame(knotVec[i+k], knotVec[i+1])) {
		q = 0.0;
	} else {
		b = (knotVec[i+k] - t)*DiffBasis(i+1, k-1, t) - PosBasis(i+1, k-1, t);
		q = b/(knotVec[i+k] - knotVec[i+1]);
	}

	return (p + q);
}

bool fk_BSplCurve::UpdateKnot(void)
{
	int				ctrlNum;
	int				i;
	double			knot;
	double			d;

	ctrlNum = int(ctrlPos.size());
	if(ordinate > ctrlNum) return false;
	knotVec.clear();

	for(i = 0; i < ordinate; i++) {
		knotVec.push_back(0.0);
	}

	d = double(ctrlNum - ordinate + 1);

	knot = 0.0;
	for(i = 0; i < ctrlNum - ordinate; i++) {
		knot += 1.0/d;
		knotVec.push_back(knot);
	}

	for(i = 0; i < ordinate; i++) {
		knotVec.push_back(1.0);
	}

	return true;
}

void fk_BSplCurve::init(void)
{
	ctrlPos.clear();
	knotVec.clear();
	setOrdinate(3);
	return;
}

bool fk_BSplCurve::setOrdinate(int argOrd)
{
	if(argOrd < 3 || argOrd > 5) {
		return false;
	}

	ordinate = argOrd;
	if(int(ctrlPos.size()) < ordinate) {
		ctrlPos.resize(ordinate);
	}
	UpdateKnot();
	changeFlg = true;

	return true;
}

bool fk_BSplCurve::setCtrl(int argID, const fk_Vector &argCtrl)
{
	if(argID < 0) {
		return false;
	}

	if(argID >= int(ctrlPos.size())) {
		ctrlPos.resize(argID+1);
		UpdateKnot();
	}

	ctrlPos[argID] = argCtrl;
	changeFlg = true;
	return true;
}

bool fk_BSplCurve::setCtrl(vector<fk_Vector> *argCtrlArray)
{
	if(int(argCtrlArray->size()) < ordinate) {
		return false;
	}

	ctrlPos.clear();
	ctrlPos = *argCtrlArray;
	changeFlg = true;
	return true;
}

int fk_BSplCurve::getOrdinate(void)
{
	return ordinate;
}

fk_Vector fk_BSplCurve::getCtrl(int argID)
{
	fk_Vector	dummy(0.0, 0.0, 0.0);

	if(argID < 0 || argID >= ordinate) {
		return dummy;
	}

	return ctrlPos[argID];
}

fk_Vector fk_BSplCurve::pos(double t)
{
	fk_Vector	retPos(0.0, 0.0, 0.0);
	int			i, num;

	num = int(ctrlPos.size());
	for(i = 0; i < num; i++) {
		retPos += PosBasis(i, ordinate, t) * ctrlPos[i];
	}

	return retPos;
}

fk_Vector fk_BSplCurve::diff(double t)
{
	fk_Vector	retVec(0.0, 0.0, 0.0);
	int			i, num;

	num = int(ctrlPos.size());
	if(IsSame(t, 0.0)) {
		retVec = double(ordinate-1) * (ctrlPos[1] - ctrlPos[0]);
	} else if(IsSame(t, 1.0)) {
		retVec = double(ordinate-1) * (ctrlPos[num-1] - ctrlPos[num-2]);
	} else {
		for(i = 0; i < num; i++) {
			retVec += DiffBasis(i, ordinate, t) * ctrlPos[i];
		}
	}

	return retVec;
}
