/****************************************************************************
 *
 *	Copyright (c) 1999-2009, Fine Kernel Project, All rights reserved.
 *
 *	Redistribution and use in source and binary forms,
 *	with or without modification, are permitted provided that the
 *	following conditions are met:
 *
 *		- Redistributions of source code must retain the above
 *			copyright notice, this list of conditions and the
 *			following disclaimer.
 *
 *		- Redistributions in binary form must reproduce the above
 *			copyright notice, this list of conditions and the
 *			following disclaimer in the documentation and/or
 *			other materials provided with the distribution.
 *
 *		- Neither the name of the copyright holders nor the names
 *			of its contributors may be used to endorse or promote
 *			products derived from this software without specific
 *			prior written permission.
 *
 *	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *	"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *	LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 *	FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 *	COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 *	INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 *	(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 *	SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 *	HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 *	STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 *	IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 *	POSSIBILITY OF SUCH DAMAGE. 
 *
 ****************************************************************************/
/****************************************************************************
 *
 *	Copyright (c) 1999-2009, Fine Kernel Project, All rights reserved.
 *
 *	本ソフトウェアおよびソースコードのライセンスは、基本的に
 *	「修正 BSD ライセンス」に従います。以下にその詳細を記します。
 *
 *	ソースコード形式かバイナリ形式か、変更するかしないかを問わず、
 *	以下の条件を満たす場合に限り、再頒布および使用が許可されます。
 *
 *	- ソースコードを再頒布する場合、上記の著作権表示、本条件一覧、
 *		および下記免責条項を含めること。
 *
 *	- バイナリ形式で再頒布する場合、頒布物に付属のドキュメント等の
 *		資料に、上記の著作権表示、本条件一覧、および下記免責条項を
 *		含めること。
 *
 *	- 書面による特別の許可なしに、本ソフトウェアから派生した製品の
 *		宣伝または販売促進に、本ソフトウェアの著作権者の名前または
 *		コントリビューターの名前を使用してはならない。
 *
 *	本ソフトウェアは、著作権者およびコントリビューターによって「現
 *	状のまま」提供されており、明示黙示を問わず、商業的な使用可能性、
 *	および特定の目的に対する適合性に関す暗黙の保証も含め、またそれ
 *	に限定されない、いかなる保証もないものとします。著作権者もコン
 *	トリビューターも、事由のいかんを問わず、損害発生の原因いかんを
 *	問わず、かつ責任の根拠が契約であるか厳格責任であるか(過失その
 *	他の)不法行為であるかを問わず、仮にそのような損害が発生する可
 *	能性を知らされていたとしても、本ソフトウェアの使用によって発生
 *	した(代替品または代用サービスの調達、使用の喪失、データの喪失、
 *	利益の喪失、業務の中断も含め、またそれに限定されない)直接損害、
 *	間接損害、偶発的な損害、特別損害、懲罰的損害、または結果損害に
 *	ついて、一切責任を負わないものとします。
 *
 ****************************************************************************/
#include <FK/Image.h>
#include <FK/Error.H>

#ifdef _FREEBSD_
#include <FL/images/png.h>
#else
#include <libpng/png.h>
#endif

using namespace std;

extern "C" {
	#define XMD_H
	#ifdef WIN32
		#define HAVE_BOOLEAN
	#endif
	#ifdef _FREEBSD_
		#include <FL/images/jpeglib.h>
	#else
		#include "jpeg/jpeglib.h"
	#endif
}

bool fk_Image::IsBmpFile(const string argFName) const
{
	FILE		*fp;
	fk_ImType	buf[2];

	if((fp = fopen(argFName.c_str(), "rb")) == (FILE *)NULL) {
		return false;
	}

	if(fread(buf, sizeof(fk_ImType), 2, fp) != sizeof(fk_ImType)*2) {
		fclose(fp);
		return false;
	}

	fclose(fp);

	return IsBmpData(buf);
}

bool fk_Image::IsBmpData(fk_ImType *argData) const
{
	if(argData[0] == 'B' && argData[1] == 'M') return true;
	return false;
}


bool fk_Image::GetBmpFileHeader(FILE *argFP, fk_ImType *argHeader)
{
	size_t	imSize = sizeof(fk_ImType);

	if(fread(argHeader, imSize, 14, argFP) != imSize*14) {
		return false;
	}
	return true;
}

bool fk_Image::GetBmpInfoHeader(FILE *argFP, fk_ImType *argHeader)
{
	size_t	imSize = sizeof(fk_ImType);

	if(fread(argHeader, imSize, 40, argFP) != imSize*40) {
		return false;
	}
	return true;
}

fk_Dimension fk_Image::GetBmpSize(fk_ImType *argHeader)
{
	fk_Dimension	retDim;

	retDim.w = int(ChgUInt(argHeader, 4));
	retDim.h = int(ChgUInt(argHeader, 8));

	return retDim;
}

fk_ImageStatus fk_Image::LoadBmpFile(const string argFName)
{
	FILE				*fp;
	int					tmpSize;
	unsigned int		bmpType;
	fk_ImType			bmpFileHeader[14];
	fk_ImType			bmpInfoHeader[40];
	fk_Dimension		bmpSize;
	vector<fk_ImType>	rgbQuad;
	vector<fk_ImType>	tmpBuffer;
	int					x, y;
	unsigned int		paletteSize;

	if((fp = fopen(argFName.c_str(), "rb")) == (FILE *)NULL) {
		return FK_IMAGE_OPENERROR;
	}

	if(GetBmpFileHeader(fp, bmpFileHeader) == false) {
		fclose(fp);
		return FK_IMAGE_DATAERROR;
	}

	if(GetBmpInfoHeader(fp, bmpInfoHeader) == false) {
		fclose(fp);
		return FK_IMAGE_DATAERROR;
	}

	paletteSize = ChgUInt(bmpFileHeader, 10) - 54;
	bmpSize = GetBmpSize(bmpInfoHeader);
	CreateBuffer(bmpSize);

	bmpType = ChgUShort(bmpInfoHeader, 14);

	if(bmpType <= 8) {
		rgbQuad.resize(paletteSize);
		if(fread(&rgbQuad[0], sizeof(fk_ImType), paletteSize, fp) !=
		   size_t(paletteSize)) {
			rgbQuad.clear();
			return FK_IMAGE_DATAERROR;
		}
	} else {
		rgbQuad.clear();
	}

	tmpSize = GetOneBufferSize(bmpType, bmpSize.w);
	if(tmpSize < 0) return FK_IMAGE_DATAERROR;

	tmpBuffer.resize(tmpSize);

	for(y = bmpSize.h - 1; y >= 0; y--) {
		if(int(fread(&tmpBuffer[0], sizeof(fk_ImType),
					 tmpSize, fp)) != tmpSize) {
			fclose(fp);
			rgbQuad.clear();
			tmpBuffer.clear();
			return FK_IMAGE_DATAERROR;
		}

		for(x = 0; x < bmpSize.w; x++) {
			SetRGBA4Bmp(x, y, &tmpBuffer[0], bmpType, rgbQuad);
		}
	}

	fclose(fp);
	rgbQuad.clear();
	tmpBuffer.clear();

	return FK_IMAGE_OK;
}

fk_ImageStatus fk_Image::LoadBmpData(fk_ImType *argBuffer)
{
	int					tmpSize, count;
	fk_ImType			*bmpFileHeader;
	fk_ImType			*bmpInfoHeader;
	vector<fk_ImType>	rgbQuad;
	fk_ImType			*tmpImageBuf;
	unsigned int		bmpType;
	fk_Dimension		bmpSize;
	int					x, y;
	unsigned int		i;
	unsigned int		paletteSize;
	

	bmpFileHeader = argBuffer;
	bmpInfoHeader = argBuffer + 14;

	paletteSize = ChgUInt(bmpFileHeader, 10) - 54;
	bmpSize = GetBmpSize(bmpInfoHeader);
	CreateBuffer(bmpSize);

	bmpType = ChgUShort(bmpInfoHeader, 14);

	if(bmpType <= 8) {
		rgbQuad.resize(paletteSize);
		for(i = 0; i < paletteSize; i++) rgbQuad[i] = argBuffer[i+54];
		tmpImageBuf = argBuffer + paletteSize + 54;
	} else {
		rgbQuad.clear();
		tmpImageBuf = argBuffer + 54;
	}

	tmpSize = GetOneBufferSize(bmpType, bmpSize.w);
	if(tmpSize < 0) return FK_IMAGE_DATAERROR;

	for(y = bmpSize.h - 1, count = 0; y >= 0; y--, count++) {

		for(x = 0; x < bmpSize.w; x++) {
			SetRGBA4Bmp(x, y,
						tmpImageBuf + count * tmpSize, bmpType, rgbQuad);
		}
	}

	return FK_IMAGE_OK;
}

void fk_Image::SetRGBA4Bmp(int argX, int argY,
						   fk_ImType *argBuffer, int argType,
						   vector<fk_ImType> &argQuadBuffer)
{
	fk_ImType	tmp;
	int			pb;

	switch(argType) {
	  case 1:
		if((argBuffer[argX/8] & (0x80 >> argX % 8)) != 0) {
			setRGB(argX, argY,
				   (int)argQuadBuffer[4],
				   (int)argQuadBuffer[5],
				   (int)argQuadBuffer[6]);
		} else {
			setRGB(argX, argY,
				   (int)argQuadBuffer[0],
				   (int)argQuadBuffer[1],
				   (int)argQuadBuffer[2]);
		}

		break;

	  case 4:
		if((argX % 2) == 0) {
			tmp = (fk_ImType)((int)argBuffer[argX/2] >> 4);
			tmp &= 0x0f;
		} else {
			tmp = (fk_ImType)((int)argBuffer[argX/2] & 0x0f);
		}

		setRGB(argX, argY,
			   argQuadBuffer[2+4*tmp],
			   argQuadBuffer[1+4*tmp],
			   argQuadBuffer[4*tmp]);

		break;

	  case 8:
		pb = (int)argBuffer[argX];

		setRGB(argX, argY,
			   (int)argQuadBuffer[2+4*pb],
			   (int)argQuadBuffer[1+4*pb],
			   (int)argQuadBuffer[4*pb]);
		break;

	  case 24:

		setRGB(argX, argY,
			   (int)argBuffer[argX*3+2],
			   (int)argBuffer[argX*3+1],
			   (int)argBuffer[argX*3]);
		break;

	  case 32:

		setRGBA(argX, argY,
				(int)argBuffer[argX*4+2],
				(int)argBuffer[argX*4+1],
				(int)argBuffer[argX*4],
				255 - (int)argBuffer[argX*4+3]);
		break;

	  default:
		break;
	}

	return;
}

fk_ImageStatus fk_Image::SaveBmpFile(string argFName, bool argTransFlg)
{
	FILE				*fp;
	int					wSize, hSize;
	vector<fk_ImType>	bmpFileHeader;
	vector<fk_ImType>	bmpInfoHeader;
	vector<fk_ImType>	bmpBuffer;
	int					bmpBufSize, y;
	int					bitSize;

	if(imageBuf.empty() == true) return FK_IMAGE_DATAERROR;

	if((fp = fopen(argFName.c_str(), "wb")) == (FILE *)NULL) {
		return FK_IMAGE_OPENERROR;
	}

	wSize = imageSize.w;
	hSize = imageSize.h;

	bitSize = (argTransFlg == true) ? 4 : 3;

	MakeBmpFileHeader(wSize, hSize, bitSize, bmpFileHeader);
	MakeBmpInfoHeader(wSize, hSize, bitSize*8, bmpInfoHeader);

	bmpBufSize = wSize * bitSize;

	while(bmpBufSize % 4 != 0) {
		bmpBufSize++;
	}

	bmpBuffer.resize(bmpBufSize);

	fwrite(&bmpFileHeader[0], sizeof(fk_ImType), bmpFileHeader.size(), fp);
	fwrite(&bmpInfoHeader[0], sizeof(fk_ImType), bmpInfoHeader.size(), fp);

	for(y = hSize-1; y >= 0; y--) {
		MakeBmpBuffer(y, wSize, argTransFlg, &bmpBuffer[0]);
		fwrite(&bmpBuffer[0], sizeof(fk_ImType), bitSize*wSize, fp);
	}

	fclose(fp);

	bmpFileHeader.clear();
	bmpInfoHeader.clear();
	bmpBuffer.clear();

	return FK_IMAGE_OK;
}

void fk_Image::MakeBmpFileHeader(int argW, int argH, int argBitSize,
								 vector<fk_ImType> &argHeader)
{
	int			offset = 0;

	argHeader.resize(14);
	argHeader[0] = (fk_ImType)'B';
	argHeader[1] = (fk_ImType)'M';
	offset += 2;

	SetLong2Byte(14 + 40 + ((long)argW * (long)argH * (long)argBitSize),
				 &argHeader[0], offset);
	offset += 4;

	SetInt2Byte(0, &argHeader[0], offset);
	offset += 2;

	SetInt2Byte(0, &argHeader[0], offset);
	offset += 2;

	SetLong2Byte(14 + 40, &argHeader[0], offset);

	return;
}

void fk_Image::MakeBmpInfoHeader(int argW, int argH, int argBitSize,
								 vector<fk_ImType> &argInfo)
{
	int			offset = 0;

	argInfo.resize(40);

	// .biSize
	SetLong2Byte(40, &argInfo[0], offset);
	offset += 4;

	// .biWidth
	SetLong2Byte((long)argW, &argInfo[0], offset);
	offset += 4;

	// .biHeight
	SetLong2Byte((long)argH, &argInfo[0], offset);
	offset += 4;

	// .biPlanes
	SetInt2Byte(1, &argInfo[0], offset);
	offset += 2;

	// .biBitCount
	SetInt2Byte(argBitSize, &argInfo[0], offset);
	offset += 2;

	// .biCompression
	SetLong2Byte(0, &argInfo[0], offset);
	offset += 4;

	// .biSizeImage
	SetLong2Byte(0, &argInfo[0], offset);
	offset += 4;

	// .biXPelsPerMeter
	SetLong2Byte(0, &argInfo[0], offset);
	offset += 4;

	// .biYPelsPerMeter
	SetLong2Byte(0, &argInfo[0], offset);
	offset += 4;

	// .biClrUsed
	SetLong2Byte(0, &argInfo[0], offset);
	offset += 4;

	// .biClrImportant
	SetLong2Byte(0, &argInfo[0], offset);

	return;
}

void fk_Image::MakeBmpBuffer(int argY, int argW,
							 bool argTransFlg, fk_ImType *argBuffer)
{
	int		x;
	int		offset;

	offset = (argTransFlg == true) ? 4 : 3;

	for(x = 0; x < argW; x++) {
		argBuffer[x * offset] = getB(x, argY);
		argBuffer[x * offset + 1] = getG(x, argY);
		argBuffer[x * offset + 2] = getR(x, argY);
		if(argTransFlg == true) {
			argBuffer[x * offset + 3] = 255 - getA(x, argY);
		}
	}
}

bool fk_Image::readBMP(const string argFName)
{
	if(CreateImg(argFName) != FK_IMAGE_OK) {
		return false;
	}

	return true;
}

bool fk_Image::readBMPData(fk_ImType *argBuffer)
{
	if(CreateImg(argBuffer) != FK_IMAGE_OK) {
		return false;
	}

	return true;
}

bool fk_Image::writeBMP(const string argFName, const bool argTransFlg)
{
	if(SaveBmpFile(argFName, argTransFlg) != FK_IMAGE_OK) {
		return false;
	}

	return true;
}


bool fk_Image::IsPngFile(const string argFName) const
{
	FILE		*fp;
	char		sig[4];

	if((fp = fopen(argFName.c_str(), "rb")) == (FILE *)NULL) {
		return false;
	}

	// ヘッダの読込(4バイト)
	if(fread(sig, 1, 4, fp) != 4) {
		fclose(fp);
		return false;
	}

	// PNGファイルであるかどうかをチェック
	if(!png_check_sig((png_bytep)sig, 4)) {
		fclose(fp);
		return false;
	}

	fclose(fp);
	return true;
}

void _fk_Image_PngReadFunc(png_struct *argPng, png_bytep argBuf,
						   png_size_t argSize)
{
	unsigned char **p = (unsigned char **)png_get_io_ptr(argPng);
	memcpy(argBuf, *p, argSize);
	*p += (int)argSize;
	return;
}


fk_ImageStatus fk_Image::LoadPngData(fk_ImType *argBuffer)
{
	png_structp		png_ptr;
	png_infop		info_ptr;
	unsigned int	trueX, trueY;
	png_bytepp		tmpBuffer;

	if(!png_check_sig((png_bytep)argBuffer, 4)) return FK_IMAGE_DATAERROR;

	// PNG構造体の初期化
	png_ptr = png_create_read_struct(PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);

	if(png_ptr == NULL) {
		return FK_IMAGE_DATAERROR;
	}

	// PNG情報構造体の初期化
	info_ptr = png_create_info_struct(png_ptr);
	if(info_ptr == NULL) {
		png_destroy_read_struct(&png_ptr, (png_infopp)NULL, (png_infopp)NULL);
		return FK_IMAGE_DATAERROR;
	}

	if(setjmp(png_ptr->jmpbuf)) {
		png_destroy_read_struct(&png_ptr, &info_ptr, (png_infopp)NULL);
		return FK_IMAGE_DATAERROR;
	}

	png_set_read_fn(png_ptr, (png_voidp)&argBuffer,
					(png_rw_ptr)_fk_Image_PngReadFunc);

	// PNG情報構造体の取得
	png_read_info(png_ptr, info_ptr);

	// PNGデータの情報取得
	// PNG_COLOR_TYPE_GRAY (ビット深度 1, 2, 4, 8, 16)
	// PNG_COLOR_TYPE_GRAY_ALPHA (ビット深度 8, 16)
	// PNG_COLOR_TYPE_PALETTE (ビット深度 1, 2, 4, 8)
	// PNG_COLOR_TYPE_RGB (ビット深度 8, 16)
	// PNG_COLOR_TYPE_RGB_ALPHA (ビット深度 8, 16)
	int type     = info_ptr->color_type;					// 色タイプ
	int depth    = info_ptr->bit_depth;						// 色深度(1,2,4,8,16)
	// 16bit-depthのPNGは除外
	if(depth == 16){
		png_destroy_read_struct(&png_ptr, &info_ptr, (png_infopp)NULL);
		return FK_IMAGE_DATAERROR;
	}

	// 画像サイズの取得
	trueX = png_get_image_width(png_ptr, info_ptr);
	trueY = png_get_image_height(png_ptr, info_ptr);

	// fk_Image 本体のバッファを作成
	CreateBuffer(trueX, trueY);

	// テンポラリイメージデータのメモリを取得
	if((tmpBuffer = (png_bytepp)malloc(trueY * sizeof(png_bytep))) == NULL) {
		png_destroy_read_struct(&png_ptr, &info_ptr, (png_infopp)NULL);
		return FK_IMAGE_READERROR;
	}
	for(png_uint_32 i = 0; i < trueY; i++) {
		// 1 ラインのテンポラリメモリを取得
		tmpBuffer[i] = (png_bytep)malloc(png_get_rowbytes(png_ptr, info_ptr));
		if (tmpBuffer[i] == NULL) {
			for (png_uint_32 j = 0; j < i; j++) { free(tmpBuffer[j]); }
			free(tmpBuffer);
			png_destroy_read_struct(&png_ptr, &info_ptr, (png_infopp)NULL);
			return FK_IMAGE_READERROR;
		}
	}
	// pngの内容を読み込み
	png_read_image(png_ptr, tmpBuffer);

	// pngの読込終了
	png_read_end(png_ptr, info_ptr);

	fk_ImageStatus result = FK_IMAGE_OK;

	// フルカラー(RGB)
	if(type == PNG_COLOR_TYPE_RGB) {
		// 内容の移し変え
		unsigned int offset;

		for(unsigned long j = 0; j < trueY; j++){
			for(unsigned long i = 0; i < trueX; i++){
				offset = GetOffset(i, j);
				imageBuf[offset]   = tmpBuffer[j][i*3];
				imageBuf[offset+1] = tmpBuffer[j][i*3+1];
				imageBuf[offset+2] = tmpBuffer[j][i*3+2];
				imageBuf[offset+3] = 255;
			}
		}
	}
	// フルカラー+アルファチャンネル(RGBA)
	else if(type == PNG_COLOR_TYPE_RGB_ALPHA) {
		// 内容の移し変え
		int	lineSize = png_get_rowbytes(png_ptr, info_ptr);

		for(unsigned long j = 0; j < trueY; j++) {
			memcpy(&(imageBuf[GetOffset(0, j)]), tmpBuffer[j], lineSize);
		}
	}
	// グレースケール(8bit)
	else if(type == PNG_COLOR_TYPE_GRAY && depth == 8) {
		// 内容の移し変え
		unsigned int offset;

		for(unsigned long j = 0; j < trueY; j++){
			for(unsigned long i = 0; i < trueX; i++){
				offset = GetOffset(i, j);
				imageBuf[offset]   = tmpBuffer[j][i];
				imageBuf[offset+1] = tmpBuffer[j][i];
				imageBuf[offset+2] = tmpBuffer[j][i];
				imageBuf[offset+3] = 255;
			}
		}
	}
	// グレースケール(1bit)
	else if(type == PNG_COLOR_TYPE_GRAY && depth == 1){
		// 内容の移し変え
		unsigned char	color;
		unsigned int	offset, x, y;

		for(unsigned long j = 0; j < trueY; j++){
			for(unsigned long i = 0; i < trueX/8+1; i++){
				color = tmpBuffer[j][i];

				for(int k = 0; k < 8; k++){
					x = i*8+k;
					y = j;
					offset = GetOffset(x, y);
					
					if(x < trueX){
						imageBuf[offset]   = 255 * ((color >> (7-k)) % 2);
						imageBuf[offset+1] = 255 * ((color >> (7-k)) % 2);
						imageBuf[offset+2] = 255 * ((color >> (7-k)) % 2);
						imageBuf[offset+3] = 255;
					}
				}
			}
		}
	}
	// パレット(8bit)
	else if(type == PNG_COLOR_TYPE_PALETTE && depth == 8){
		png_colorp		palette;		// パレットデータ
		int				num_palette;	// パレット数
		unsigned int	offset;
		unsigned char	num;
		
		// パレットの取得
		png_get_PLTE(png_ptr, info_ptr, &palette, &num_palette);

		// 内容の移し変え
		for(unsigned long j = 0; j < trueY; j++){
			for(unsigned long i = 0; i < trueX; i++){
				offset = GetOffset(i, j);
				// パレット番号の取得
				num = tmpBuffer[j][i];
				imageBuf[offset]   = palette[num].red;
				imageBuf[offset+1] = palette[num].green;
				imageBuf[offset+2] = palette[num].blue;
				imageBuf[offset+3] = 255;
			}
		}	
	}
	// 想定外フォーマット
	else {
		result = FK_IMAGE_DATAERROR;
	}

	// メモリの解放
	for(png_uint_32 i = 0; i < trueY; i++) free(tmpBuffer[i]);
	free(tmpBuffer);

	// PNGメモリの解放
	png_destroy_read_struct(&png_ptr, &info_ptr, (png_infopp)NULL);

	return result;
}

fk_ImageStatus fk_Image::LoadPngFile(const string argFName)
{
	FILE			*fp;
	char			sig[4];
	png_structp		png_ptr;
	png_infop		info_ptr;
	unsigned int	trueX, trueY;
	png_bytepp		tmpBuffer;

	// ファイルの読込
	if((fp = fopen(argFName.c_str(), "rb")) == NULL) {
		return FK_IMAGE_OPENERROR;
	}

	// ヘッダの読込(4バイト)
	if(fread(sig, 1, 4, fp) != 4) {
		fclose(fp);
		return FK_IMAGE_DATAERROR;
	}

	// PNGファイルであるかどうかをチェック
	if(!png_check_sig((png_bytep)sig, 4)) {
		fclose(fp);
		return FK_IMAGE_DATAERROR;
	}

	// PNG構造体の初期化
	png_ptr = png_create_read_struct(PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);
	if(png_ptr == NULL) {
		fclose(fp);
		return FK_IMAGE_DATAERROR;
	}

	// PNG情報構造体の初期化
	info_ptr = png_create_info_struct(png_ptr);
	if(info_ptr == NULL) {
		png_destroy_read_struct(&png_ptr, (png_infopp)NULL, (png_infopp)NULL);
		fclose(fp);
		return FK_IMAGE_DATAERROR;
	}

	if(setjmp(png_ptr->jmpbuf)) {
		png_destroy_read_struct(&png_ptr, &info_ptr, (png_infopp)NULL);
		fclose(fp);
		return FK_IMAGE_DATAERROR;
	}

	// ファイルポインタのセット
	png_init_io(png_ptr, fp);
	// 事前にシグネチャを読込確認済なら
	// ファイル先頭から読み飛ばしているバイト数を知らせる
	png_set_sig_bytes(png_ptr, 4);
	// PNG情報構造体の取得
	png_read_info(png_ptr, info_ptr);

	// PNGデータの情報取得
	// PNG_COLOR_TYPE_GRAY (ビット深度 1, 2, 4, 8, 16)
	// PNG_COLOR_TYPE_GRAY_ALPHA (ビット深度 8, 16)
	// PNG_COLOR_TYPE_PALETTE (ビット深度 1, 2, 4, 8)
	// PNG_COLOR_TYPE_RGB (ビット深度 8, 16)
	// PNG_COLOR_TYPE_RGB_ALPHA (ビット深度 8, 16)
	int type     = info_ptr->color_type;					// 色タイプ
	int depth    = info_ptr->bit_depth;						// 色深度(1,2,4,8,16)
	// 16bit-depthのPNGは除外
	if(depth == 16){
		png_destroy_read_struct(&png_ptr, &info_ptr, (png_infopp)NULL);
		fclose(fp);
		return FK_IMAGE_DATAERROR;
	}

	// 画像サイズの取得
	trueX = png_get_image_width(png_ptr, info_ptr);
	trueY = png_get_image_height(png_ptr, info_ptr);

	// fk_Image 本体のバッファを作成
	CreateBuffer(trueX, trueY);

	// テンポラリイメージデータのメモリを取得
	if((tmpBuffer = (png_bytepp)malloc(trueY * sizeof(png_bytep))) == NULL) {
		png_destroy_read_struct(&png_ptr, &info_ptr, (png_infopp)NULL);
		fclose(fp);
		return FK_IMAGE_READERROR;
	}
	for(png_uint_32 i = 0; i < trueY; i++) {
		// 1 ラインのテンポラリメモリを取得
		tmpBuffer[i] = (png_bytep)malloc(png_get_rowbytes(png_ptr, info_ptr));
		if (tmpBuffer[i] == NULL) {
			for (png_uint_32 j = 0; j < i; j++) { free(tmpBuffer[j]); }
			free(tmpBuffer);
			png_destroy_read_struct(&png_ptr, &info_ptr, (png_infopp)NULL);
			fclose(fp);
			return FK_IMAGE_READERROR;
		}
	}
	// pngの内容を読み込み
	png_read_image(png_ptr, tmpBuffer);

	// pngの読込終了
	png_read_end(png_ptr, info_ptr);

	// ファイルを閉じる
	fclose(fp);

	fk_ImageStatus result = FK_IMAGE_OK;

	// フルカラー(RGB)
	if(type == PNG_COLOR_TYPE_RGB) {
		// 内容の移し変え
		unsigned int offset;

		for(unsigned long j = 0; j < trueY; j++){
			for(unsigned long i = 0; i < trueX; i++){
				offset = GetOffset(i, j);
				imageBuf[offset]   = tmpBuffer[j][i*3];
				imageBuf[offset+1] = tmpBuffer[j][i*3+1];
				imageBuf[offset+2] = tmpBuffer[j][i*3+2];
				imageBuf[offset+3] = 255;
			}
		}
	}
	// フルカラー+アルファチャンネル(RGBA)
	else if(type == PNG_COLOR_TYPE_RGB_ALPHA) {
		// 内容の移し変え
		int	lineSize = png_get_rowbytes(png_ptr, info_ptr);

		for(unsigned long j = 0; j < trueY; j++) {
			memcpy(&imageBuf[GetOffset(0, j)], tmpBuffer[j], lineSize);
		}
	}
	// グレースケール(8bit)
	else if(type == PNG_COLOR_TYPE_GRAY && depth == 8) {
		// 内容の移し変え
		unsigned int offset;

		for(unsigned long j = 0; j < trueY; j++){
			for(unsigned long i = 0; i < trueX; i++){
				offset = GetOffset(i, j);
				imageBuf[offset]   = tmpBuffer[j][i];
				imageBuf[offset+1] = tmpBuffer[j][i];
				imageBuf[offset+2] = tmpBuffer[j][i];
				imageBuf[offset+3] = 255;
			}
		}
	}
	// グレースケール(1bit)
	else if(type == PNG_COLOR_TYPE_GRAY && depth == 1){
		// 内容の移し変え
		unsigned char	color;
		unsigned int	offset, x, y;

		for(unsigned long j = 0; j < trueY; j++){
			for(unsigned long i = 0; i < trueX/8+1; i++){
				color = tmpBuffer[j][i];

				for(int k = 0; k < 8; k++){
					x = i*8+k;
					y = j;
					offset = GetOffset(x, y);
					
					if(x < trueX){
						imageBuf[offset]   = 255 * ((color >> (7-k)) % 2);
						imageBuf[offset+1] = 255 * ((color >> (7-k)) % 2);
						imageBuf[offset+2] = 255 * ((color >> (7-k)) % 2);
						imageBuf[offset+3] = 255;
					}
				}
			}
		}
	}
	// パレット(8bit)
	else if(type == PNG_COLOR_TYPE_PALETTE && depth == 8){
		png_colorp		palette;		// パレットデータ
		int				num_palette;	// パレット数
		unsigned int	offset;
		unsigned char	num;
		
		// パレットの取得
		png_get_PLTE(png_ptr, info_ptr, &palette, &num_palette);

		// 内容の移し変え
		for(unsigned long j = 0; j < trueY; j++){
			for(unsigned long i = 0; i < trueX; i++){
				offset = GetOffset(i, j);
				// パレット番号の取得
				num = tmpBuffer[j][i];
				imageBuf[offset]   = palette[num].red;
				imageBuf[offset+1] = palette[num].green;
				imageBuf[offset+2] = palette[num].blue;
				imageBuf[offset+3] = 255;
			}
		}	
	}
	// 想定外フォーマット
	else {
		result = FK_IMAGE_DATAERROR;
	}

	// メモリの解放
	for(png_uint_32 i = 0; i < trueY; i++) free(tmpBuffer[i]);
	free(tmpBuffer);

	// PNGメモリの解放
	png_destroy_read_struct(&png_ptr, &info_ptr, (png_infopp)NULL);

	return result;
}

bool fk_Image::readPNG(const string argFName)
{
	if(LoadPngFile(argFName) != FK_IMAGE_OK) {
		return false;
	}

	return true;
}

bool fk_Image::readPNGData(fk_ImType *argBuffer)
{
	if(LoadPngData(argBuffer) != FK_IMAGE_OK) {
		return false;
	}

	return true;
}

bool fk_Image::writePNG(const string fileName, const bool argTransFlg)
{
	FILE			*fp;
	png_structp		png_ptr;
	png_infop		info_ptr;
	unsigned int	trueX, trueY, lineSize;
	png_bytepp		tmpBuffer;

	// バイナリ書き込みモードでファイルをオープン
	fp = fopen(fileName.c_str(), "wb");
	if(fp == NULL) return false;

	// 書き込み用構造体を作成
	png_ptr = png_create_write_struct(PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);
	if(png_ptr == NULL) {
		fclose(fp);
		return false;
	}

	// PNG 情報構造体を作成
	info_ptr = png_create_info_struct(png_ptr);
	if(info_ptr == NULL) {
	   png_destroy_write_struct(&png_ptr, NULL);
	   fclose(fp);
	   return false;
	}

	// ファイルポインタのセット
	png_init_io(png_ptr, fp);

	// ヘッダに情報を設定(幅、高さ、256(8bit)階調、RGB(A)、インタレースなし)
	if(argTransFlg == true) {
		png_set_IHDR(png_ptr, info_ptr, getWidth(), getHeight(), 8,
					 PNG_COLOR_TYPE_RGB_ALPHA, PNG_INTERLACE_NONE,
					 PNG_COMPRESSION_TYPE_DEFAULT, PNG_FILTER_TYPE_BASE);
	} else {
		png_set_IHDR(png_ptr, info_ptr, getWidth(), getHeight(), 8,
					 PNG_COLOR_TYPE_RGB, PNG_INTERLACE_NONE,
					 PNG_COMPRESSION_TYPE_DEFAULT, PNG_FILTER_TYPE_BASE);
	}
	png_write_info(png_ptr, info_ptr);

	// 構造体にセットしてから幅と高さを取得し直す
	trueX = png_get_image_width(png_ptr, info_ptr);
	trueY = png_get_image_height(png_ptr, info_ptr);
	lineSize = png_get_rowbytes(png_ptr, info_ptr);

	// テンポラリイメージデータのメモリを取得
	if((tmpBuffer = (png_bytepp)malloc(trueY * sizeof(png_bytep))) == NULL) {
		png_destroy_write_struct(&png_ptr, &info_ptr);
		fclose(fp);
		return false;
	}
	for(png_uint_32 i = 0; i < trueY; i++) {
		// 1 ラインのテンポラリメモリを取得
		tmpBuffer[i] = (png_bytep)malloc(lineSize);
		if(tmpBuffer[i] == NULL) {
			for(png_uint_32 j = 0; j < i; j++) {
				free(tmpBuffer[j]);
			}
			free(tmpBuffer);
			png_destroy_write_struct(&png_ptr, &info_ptr);
			fclose(fp);
			return false;
		}
	}

	// 内容の移し変え
	if(argTransFlg == true) {
		for(unsigned int j = 0; j < trueY; j++) {
			memcpy(tmpBuffer[j], &imageBuf[GetOffset(0, j)], lineSize);
		}
	} else {
		int offset;
		for(unsigned int j = 0; j < trueY; j++) {
			for(unsigned int i = 0; i < trueX; i++) {
				offset = GetOffset(i, j);
				tmpBuffer[j][i * 3 + 0] = imageBuf[offset + 0];
				tmpBuffer[j][i * 3 + 1] = imageBuf[offset + 1];
				tmpBuffer[j][i * 3 + 2] = imageBuf[offset + 2];
			}
		}
	}

	// 画像の書き込み
	png_write_image(png_ptr, tmpBuffer);

	// 書き込み終了
	png_write_end(png_ptr, info_ptr);
	fclose(fp);

	// 構造体の解体
	png_destroy_write_struct(&png_ptr, &info_ptr);

	return true;
}

bool fk_Image::readJPG(const string fileName)
{
	struct jpeg_decompress_struct cinfo;	// JPEG情報構造体
	struct jpeg_error_mgr jerr;				// エラーハンドラ
	FILE*  infile;

	// ファイルを開く
	if((infile = fopen(fileName.c_str(), "rb")) == NULL) {
		return false;
	}

	// JPEGオブジェクトの初期化
	cinfo.err = jpeg_std_error(&jerr);

	// 独自のエラーハンドラを設定
	cinfo.client_data   = const_cast<char *>("fk_Image JPEG Proc");
	jerr.error_exit     = NULL;
	jerr.output_message = NULL;

	jpeg_create_decompress(&cinfo);

	// 標準出力のソースがinfileであることを設定
	jpeg_stdio_src(&cinfo, infile);

	// ヘッダの読み込み
	jpeg_read_header(&cinfo, TRUE);

	// 展開の開始
	jpeg_start_decompress(&cinfo);

	// 幅と高さの取得
	int wid = cinfo.output_width;
	int hgt = cinfo.output_height;

	// イメージを保持するメモリ領域の確保と初期化
	JSAMPARRAY img = (JSAMPARRAY)malloc(sizeof(JSAMPROW) * hgt);
	for(int i = 0; i < hgt; i++) {
		img[i] = (JSAMPROW)calloc(sizeof(JSAMPLE), 3 * wid);
	}

	// 全イメージデータを取得	
	while(cinfo.output_scanline < cinfo.output_height) {
		jpeg_read_scanlines(&cinfo, img + cinfo.output_scanline, cinfo.output_height - cinfo.output_scanline);
	}

	// 展開の終了
	jpeg_finish_decompress(&cinfo);

	// JPEGオブジェクトの破棄
	jpeg_destroy_decompress(&cinfo);

	// ファイルを閉じる
	fclose(infile);

	// バッファの生成
	CreateBuffer(wid, hgt);

	// 内容の移し変え
	int offset;

	for(int j = 0; j < hgt; j++){
		for(int i = 0; i < wid; i++){
			offset = GetOffset(i, j);
			imageBuf[offset + 0] = img[j][i * 3 + 0];
			imageBuf[offset + 1] = img[j][i * 3 + 1];
			imageBuf[offset + 2] = img[j][i * 3 + 2];
			imageBuf[offset + 3] = 255;
		}
	}

	// イメージデータを保持するメモリ領域を開放
	for(int i = 0; i < hgt; i++){
		free(img[i]);
	}
	free(img);

	return true;
}

bool fk_Image::writeJPG(const string fileName, int quality)
{
	if(imageBuf.empty()) {
		return false;
	}

	struct jpeg_compress_struct		cinfo;
	struct jpeg_error_mgr			jerr;
	FILE							*outfile;
	JSAMPARRAY						img;

	// クオリティ値の調整
	if(quality < 0) {
		quality = 0;
	} 
	if(quality > 100) {
		quality = 100;
	}

	// ファイルを開く
	if((outfile = fopen(fileName.c_str(), "wb")) == NULL) {
		return false;
	}

	int wid = getWidth();
	int hgt = getHeight();
	int offset;

	// イメージを保持するメモリ領域の確保と初期化
	img = (JSAMPARRAY)malloc(sizeof(JSAMPROW) * hgt);
	for(int j = 0; j < hgt; j++) {
		img[j] = (JSAMPROW)malloc(sizeof(JSAMPLE) * 3 * wid);
		for(int i = 0; i < wid; i++) {
			offset = GetOffset(i, j);
			img[j][i * 3 + 0] = imageBuf[offset + 0];
			img[j][i * 3 + 1] = imageBuf[offset + 1];
			img[j][i * 3 + 2] = imageBuf[offset + 2];
		}
	}

	// JPEGオブジェクトの初期化
	cinfo.err = jpeg_std_error(&jerr);

	// 独自のエラーハンドラを設定
	cinfo.client_data   = const_cast<char *>("fk_Image JPEG Proc");
	jerr.error_exit     = NULL;
	jerr.output_message = NULL;
	
	jpeg_create_compress(&cinfo);

	// 標準出力をoutfile
	jpeg_stdio_dest(&cinfo, outfile);
	
	// 圧縮のクオリティを設定
	jpeg_set_quality(&cinfo, quality, TRUE);

	// パラメータの設定
	cinfo.image_width      = wid;
	cinfo.image_height     = hgt;
	cinfo.input_components = 3;
	cinfo.in_color_space   = JCS_RGB;

	// デフォルト値の設定
	jpeg_set_defaults(&cinfo);

	// 圧縮の開始
	jpeg_start_compress(&cinfo, TRUE);

	// 全イメージデータを出力
	jpeg_write_scanlines(&cinfo, img, hgt);

	// 圧縮の終了
	jpeg_finish_compress(&cinfo);

	// JPEGオブジェクトの破棄
	jpeg_destroy_compress(&cinfo);

	// ファイルを閉じる
	fclose(outfile);	

	// イメージデータを保持するメモリ領域を開放
	for(int i = 0; i < hgt; i++) {
		free(img[i]);
	}
	free(img);

	return true;
}
