/****************************************************************************
 *
 *	Copyright (c) 1999-2009, Fine Kernel Project, All rights reserved.
 *
 *	Redistribution and use in source and binary forms,
 *	with or without modification, are permitted provided that the
 *	following conditions are met:
 *
 *		- Redistributions of source code must retain the above
 *			copyright notice, this list of conditions and the
 *			following disclaimer.
 *
 *		- Redistributions in binary form must reproduce the above
 *			copyright notice, this list of conditions and the
 *			following disclaimer in the documentation and/or
 *			other materials provided with the distribution.
 *
 *		- Neither the name of the copyright holders nor the names
 *			of its contributors may be used to endorse or promote
 *			products derived from this software without specific
 *			prior written permission.
 *
 *	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *	"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *	LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 *	FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 *	COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 *	INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 *	(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 *	SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 *	HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 *	STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 *	IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 *	POSSIBILITY OF SUCH DAMAGE. 
 *
 ****************************************************************************/
/****************************************************************************
 *
 *	Copyright (c) 1999-2009, Fine Kernel Project, All rights reserved.
 *
 *	本ソフトウェアおよびソースコードのライセンスは、基本的に
 *	「修正 BSD ライセンス」に従います。以下にその詳細を記します。
 *
 *	ソースコード形式かバイナリ形式か、変更するかしないかを問わず、
 *	以下の条件を満たす場合に限り、再頒布および使用が許可されます。
 *
 *	- ソースコードを再頒布する場合、上記の著作権表示、本条件一覧、
 *		および下記免責条項を含めること。
 *
 *	- バイナリ形式で再頒布する場合、頒布物に付属のドキュメント等の
 *		資料に、上記の著作権表示、本条件一覧、および下記免責条項を
 *		含めること。
 *
 *	- 書面による特別の許可なしに、本ソフトウェアから派生した製品の
 *		宣伝または販売促進に、本ソフトウェアの著作権者の名前または
 *		コントリビューターの名前を使用してはならない。
 *
 *	本ソフトウェアは、著作権者およびコントリビューターによって「現
 *	状のまま」提供されており、明示黙示を問わず、商業的な使用可能性、
 *	および特定の目的に対する適合性に関す暗黙の保証も含め、またそれ
 *	に限定されない、いかなる保証もないものとします。著作権者もコン
 *	トリビューターも、事由のいかんを問わず、損害発生の原因いかんを
 *	問わず、かつ責任の根拠が契約であるか厳格責任であるか(過失その
 *	他の)不法行為であるかを問わず、仮にそのような損害が発生する可
 *	能性を知らされていたとしても、本ソフトウェアの使用によって発生
 *	した(代替品または代用サービスの調達、使用の喪失、データの喪失、
 *	利益の喪失、業務の中断も含め、またそれに限定されない)直接損害、
 *	間接損害、偶発的な損害、特別損害、懲罰的損害、または結果損害に
 *	ついて、一切責任を負わないものとします。
 *
 ****************************************************************************/
#include <FK/UniCode.h>
#include <FK/Error.H>
#include <imdkcv/imdkcvlib.h>

using namespace std;

static fk_StrConverter		strConverter;

fk_UniChar::fk_UniChar(int argCode)
	: fk_BaseObject(FK_UNICHAR)
{
	setCode(argCode);
	return;
}

fk_UniChar::fk_UniChar(const fk_UniChar &argUC)
	: fk_BaseObject()
{
	code = argUC.code;
	return;
}

fk_UniChar & fk_UniChar::operator =(const fk_UniChar &argUC)
{
	code = argUC.code;

	return *this;
}

bool fk_UniChar::readFile(FILE *argFP)
{
	unsigned char	buffer[3];

	if(argFP == (FILE *)NULL) return false;
	if(fread(buffer, sizeof(unsigned char), 2, argFP) !=
	   2*sizeof(unsigned char)) {
		return false;
	}

	setBuffer(buffer[0], buffer[1]);

	return true;
}

void fk_UniChar::setBuffer(unsigned char argC1, unsigned char argC2)
{
	code = int(argC1) + int(argC2)*256;
	return;
}

void fk_UniChar::setCode(int argCode)
{
	code = argCode;
	return;
}

int fk_UniChar::getCode(void)
{
	return code;
}

bool fk_UniChar::isAscii(void)
{
	if(code < 128) {
		return true;
	}
	return false;
}

char fk_UniChar::getAscii(void)
{
	return char(code);
}

fk_UniStr::fk_UniStr(void)
	: fk_BaseObject(FK_UNISTR)
{
	clear();
	return;
}

fk_UniStr::~fk_UniStr()
{
	clear();
	return;
}

fk_UniStr::fk_UniStr(const fk_UniStr &argStr)
	: fk_BaseObject()
{
	unsigned int	i;
	clear();
	for(i = 0; i < argStr.uniStr.size(); i++) {
		push_back(argStr.uniStr[i]);
	}

	mark = argStr.mark;

	return;
}

fk_UniStr & fk_UniStr::operator =(const fk_UniStr &argStr)
{
	unsigned int	i;

	clear();
	for(i = 0; i < argStr.uniStr.size(); i++) {
		push_back(argStr.uniStr[i]);
	}
	mark = argStr.mark;

	return *this;
}

void fk_UniStr::clear(void)
{
	uniStr.clear();
	markClear();
	return;
}

int fk_UniStr::getLength(void) const
{
	return int(uniStr.size());
}

bool fk_UniStr::fgetLine(FILE *argFP, fk_StringCode argCode)
{
	char		buffer[512];

	if(argFP == (FILE *)NULL) return false;
	if(feof(argFP) != 0) return false;

	if(argCode == FK_STR_UTF16) {
		return fgetLine_UTF16(argFP);
	}

	if(fgets(buffer, 511, argFP) == (char *)NULL) {
		return false;
	}

	buffer[511] = 0;

	convert(buffer, argCode);
	return true;
}

void fk_UniStr::convert(const string &argStr, fk_StringCode argCode)
{
	unsigned int		i;
	fk_UniChar			uniChar;

	switch(argCode) {
	  case FK_STR_UTF16:
		clear();
		for(i = 0; i < argStr.size(); i += 2) {
			uniChar.setBuffer((unsigned char)argStr[i],
							  (unsigned char)argStr[i+1]);
			push_back(&uniChar);
		}
		break;

	  case FK_STR_JIS:
		strConverter.convertJIS(argStr, this);
		break;

	  case FK_STR_SJIS:
		strConverter.convertSJIS(argStr, this);
		break;

	  case FK_STR_EUC:
		strConverter.convertEUC(argStr, this);
		break;

	  default:
		break;
	}

	return;
}

bool fk_UniStr::fgetLine_UTF16(FILE *argFP)
{
	unsigned char	buffer[3];
	fk_UniChar		uniChar;

	if(argFP == (FILE *)NULL) return false;

	if(fread((char *)buffer, sizeof(unsigned char), 2, argFP) != 2) {
		return false;
	}

	do {

		uniChar.setBuffer(buffer[0], buffer[1]);
		if(uniChar.getCode() == 10) {	// CR
			if(getLength() != 0) {
				if(getCode(getLength()-1) == 13) {
					pop_back();
				}
			}
			push_back(uniChar.getCode());
			break;
		}

		if(uniChar.getCode() != 65279) {
			push_back(uniChar.getCode());
		}

	} while(fread((char *)buffer, sizeof(unsigned char), 2, argFP) == 2);

	return true;
}

bool fk_UniStr::readFile(string argFileName, fk_StringCode argCode)
{
	FILE	*fp;

	if((fp = fopen(argFileName.c_str(), "rb")) == (FILE *)NULL) {
		fk_PutError("fk_UniStr", "readFile", 1);
		return false;
	}

	clear();

	while(fgetLine(fp, argCode) == true) {};
	fclose(fp);

	return true;
}

int fk_UniStr::getCode(int argID) const
{
	if(argID < 0 || argID >= getLength()) {
		return -1;
	}

	return uniStr[argID];
}

void fk_UniStr::printf(fk_StringCode argCode, const char *argFormat, ...)
{
	va_list		ap;
	char		buffer[65536];
	string		str;

	va_start(ap, argFormat);
	vsnprintf(buffer, 65535, argFormat, ap);
	va_end(ap);
	str = buffer;
	convert(buffer, argCode);
	return;
}

int fk_UniStr::find(int argStart, int argCode)
{
	int				i;

	if(argStart < 0 || argStart >= getLength()) {
		return -1;
	}

	for(i = argStart; i < getLength(); i++) {
		if(argCode == uniStr[i]) {
			return i;
		}
	}

	return -1;
}

int fk_UniStr::find(int argStart, unsigned char argC1, unsigned char argC2)
{
	int		code;

	code = int(argC1) * int(argC2)*256;
	return find(argStart, code);
}

void fk_UniStr::pop_back(void)
{
	if(getLength() == 0) return;

	uniStr.pop_back();
	return;
}

void fk_UniStr::push_back(fk_UniChar *argUC)
{
	if(argUC == (fk_UniChar *)NULL) return;

	uniStr.push_back(argUC->getCode());
	return;
}

void fk_UniStr::push_back(fk_UniChar argUC)
{
	uniStr.push_back(argUC.getCode());
	return;
}

void fk_UniStr::push_back(int argUC)
{
	uniStr.push_back(argUC);
	return;
}

void fk_UniStr::copyStr(fk_UniStr *argStr)
{
	int			i;

	if(argStr == (fk_UniStr *)NULL) return;

	clear();
	for(i = 0; i < int(argStr->uniStr.size()); i++) {
		uniStr.push_back(argStr->uniStr[i]);
	}

	return;
}

void fk_UniStr::Print(string argStr)
{
	string	prefix;
	int		i;

	if(argStr.length() == 0) {
		prefix = "str";
	} else {
		prefix = argStr;
	}

	for(i = 0; i < getLength(); i++) {
		fk_Printf("%s[%d] = %d\n", prefix.c_str(), i, getCode(i));
	}
	return;
}

void fk_UniStr::markClear(void)
{
	mark = 0;
	return;
}

bool fk_UniStr::getLine(fk_UniStr *argStr)
{
	if(mark == getLength()) return false;

	if(argStr == (fk_UniStr *)NULL) return false;

	argStr->clear();

	for(;mark < getLength(); mark++) {
		if(getCode(mark) == 10) {
			mark++;
			break;
		}

		argStr->push_back(uniStr[mark]);
	}

	return true;
}


fk_StrConverter::fk_StrConverter(void)
{
	jisBase = (ImdKanjiCodeConverter *)NULL;
	sjisBase = (ImdKanjiCodeConverter *)NULL;
	eucBase = (ImdKanjiCodeConverter *)NULL;
	buffer = (char *)NULL;
	bufSize = 0;

	return;
}

fk_StrConverter::~fk_StrConverter()
{
	delete jisBase;
	delete sjisBase;
	delete eucBase;
	delete [] buffer;

	return;
}

ImdKcvEncoder_UCS * fk_StrConverter::MakeEncoder(void)
{
	ImdKcvEncoder_UCS		*encoder;

	encoder = new ImdKcvEncoder_UCS;
	encoder->mode = 16;
	encoder->endian = 1;
	encoder->_feff = 1;
	return encoder;
}

void fk_StrConverter::RemakeBuffer(ImdKanjiCodeConverter *argKCV)
{
	if(argKCV == (ImdKanjiCodeConverter *)NULL) return;

	if(bufSize < argKCV->MaxOutputSize()) {
		delete [] buffer;
		buffer = new char[argKCV->MaxOutputSize()];
	}
	return;
}

void fk_StrConverter::InitJISBase(void)
{

	delete jisBase;

	jisBase = new ImdKanjiCodeConverter();
	jisBase->SetDecoder(new ImdKcvDecoder_JIS);
	jisBase->ConcatinateModule(new Imd_JIStoUCS);
	jisBase->SetEncoder(MakeEncoder());
	RemakeBuffer(jisBase);
	return;
}

void fk_StrConverter::InitSJISBase(void)
{
	delete sjisBase;

	sjisBase = new ImdKanjiCodeConverter();
	sjisBase->SetDecoder(new ImdKcvDecoder_SJIS);
	sjisBase->ConcatinateModule(new Imd_SJISNtoUCS);
	sjisBase->SetEncoder(MakeEncoder());
	RemakeBuffer(sjisBase);
	return;
}

void fk_StrConverter::InitEUCBase(void)
{
	delete eucBase;

	eucBase = new ImdKanjiCodeConverter();
	eucBase->SetDecoder(new ImdKcvDecoder_EUC);
	eucBase->ConcatinateModule(new Imd_EUCtoJIS);
	eucBase->ConcatinateModule(new Imd_JIStoUCS);
	eucBase->SetEncoder(MakeEncoder());
	RemakeBuffer(eucBase);
	return;
}

void fk_StrConverter::CommonConvert(ImdKanjiCodeConverter *argKCV,
									const string &argStr, fk_UniStr *outStr)
{
	int			i, j, dummy, size;
	fk_UniChar	uniChar;

	//	outStr->clear();

	for(i = 0; i < int(argStr.size()); i++) {
		dummy = 0;
		size = argKCV->execute(argStr[i], buffer, &dummy);
		for(j = 0; j < size; j += 2) {
			uniChar.setBuffer(buffer[j], buffer[j+1]);
			outStr->push_back(&uniChar);
		}
	}
	
	return;
}

void fk_StrConverter::convertJIS(const string &argStr, fk_UniStr *outStr)
{
	if(jisBase == (ImdKanjiCodeConverter *)NULL) {
		InitJISBase();
	}

	CommonConvert(jisBase, argStr, outStr);
	return;
}

void fk_StrConverter::convertSJIS(const string &argStr, fk_UniStr *outStr)
{
	if(sjisBase == (ImdKanjiCodeConverter *)NULL) {
		InitSJISBase();
	}

	CommonConvert(sjisBase, argStr, outStr);
	return;
}

void fk_StrConverter::convertEUC(const string &argStr, fk_UniStr *outStr)
{
	if(eucBase == (ImdKanjiCodeConverter *)NULL) {
		InitEUCBase();
	}

	CommonConvert(eucBase, argStr, outStr);
	return;
}
