package jp.sourceforge.foolishmerge.text;

import org.apache.commons.jrcs.diff.Diff;
import org.apache.commons.jrcs.diff.DifferentiationFailedException;
import org.apache.commons.jrcs.diff.Revision;
import org.apache.commons.jrcs.util.ToString;

/**
 * 編集前、編集後のテキストを表す抽象クラス。
 */
public abstract class AbstractText {

  /**
   * テキストの内容
   */
  protected String[] content = null;

  /**
   * オブジェクトを生成する。
   * @param text テキスト
   */
  AbstractText(String text) {
    // テキストを文字列配列に変換して、フィールドにセット。
    content = ToString.stringToArray(text);
  }

  /**
   * テキストを取得する。
   * @return テキスト
   */
  public String getText() {
    // フィールドの文字列配列を文字列に変換。
    String text = ToString.arrayToString(content);

    // テキストを返す。
    return text;
  }

  /**
   * 文字列配列を取得する。
   * @return テキストの文字列配列
   */
  protected String[] getTextArray() {
    // フィールドの文字列配列を返す。
    return content;
  }

  /**
   * 差分を取得する。
   * @param others このテキストと比較されるテキスト
   * @throws DifferentiationFailedException 差分の取得に失敗した場合
   */
  public Revision diff(AbstractText others)
    throws DifferentiationFailedException {
    // 比較するテキストの内容を取得。
    String[] othersContent = others.getTextArray();

    // 自身の内容と比較して差分を取得。
    Revision rev = Diff.diff(this.content, othersContent);

    // 差分を返す。
    return rev;
  }

  /**
   * 指摘されたテキストと比較する。
   * @param obj このテキストと比較されるテキスト
   * @return 等しい場合はtrue
   */
  public boolean equals(Object obj) {
    // オブジェクトがテキストクラスでない場合はfalseを返す。
    if (!(obj instanceof AbstractText)) {
      return false;
    }

    // オブジェクトをテキストクラスにキャスト。
    AbstractText others = (AbstractText) obj;
    // 比較するテキストを取得。
    String othersText = others.getText();

    // 比較結果を返す。
    return (this.getText().equals(othersText));
  }

}
