package jp.sourceforge.foolishmerge.diff;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;

import jp.sourceforge.foolishmerge.FoolishMergeUtils;

/**
 * 文書差分クラス。
 */
public class DocDifference {

	/**
	 * 通過経路
	 */
	private List passed = new ArrayList();

	/**
	 * 反転フラグ
	 */
	private boolean reverse = false;

	/**
	 * 現在座標
	 */
	private LineDifference current = null;

	/**
	 * 行の差分情報
	 */
	private LineDifference[] lines = null;

	/**
	 * 編集元文書の差分情報
	 */
	private LineDifference[] original = null;

	/**
	 * 編集後文書の差分情報
	 */
	private LineDifference[] modified = null;

	/**
	 * 配列A
	 */
	private String[] A = null;

	/**
	 * 配列B
	 */
	private String[] B = null;

	/**
	 * 編集元の文書
	 */
	private String[] orgDoc = null;

	/**
	 * 編集後の文書
	 */
	private String[] modDoc = null;

	/**
	 * 差分
	 */
	private Delta[] deltas = null;

	/**
	 * LCS
	 */
	private int lcs = Integer.MIN_VALUE;

	/**
	 * 文書差分を構築する。
	 */
	public DocDifference() {
	}

	/**
	 * 指定された文書を使用して、文書差分を構築する。
	 * 
	 * @param org 編集元文書
	 * @param mod 編集後文書
	 */
	public DocDifference(String[] org, String[] mod) {
		// 文書をフィールドにセット。
		this.orgDoc = org;
		this.modDoc = mod;

		if (mod.length >= org.length) {
			// 編集後 ≧ 編集元の場合、配列Aに編集元、配列Bに編集後をセット。
			A = org;
			B = mod;
		} else {
			// 編集後 ＜ 編集元の場合、配列Bに編集元、配列Aに編集後をセット。
			A = mod;
			B = org;
			// 反転フラグをtrueにする。
			reverse = true;
		}
	}

	/**
	 * 行差分を取得する。
	 * 
	 * @return 行差分
	 */
	public LineDifference[] getLines() {
		// 行差分を返す。
		return lines;
	}

	/**
	 * 編集元文書の行差分(削除分)を取得する。
	 * 
	 * @return 行差分
	 */
	public LineDifference[] getOriginal() {
		// 行差分を返す。
		return original;
	}

	/**
	 * 編集後文書の行差分(追加分)を取得する。
	 * 
	 * @return 行差分
	 */
	public LineDifference[] getModified() {
		// 行差分を返す。
		return modified;
	}

	/**
	 * 差分を取得する。
	 * 
	 * @return 差分
	 */
	public Delta[] getDeltas() {
		// 差分を返す。
		return deltas;
	}

	/**
	 * 文書にパッチを当てる。
	 * 
	 * @param doc 文書
	 * @param offset オフセット
	 * @return パッチを当てた文書
	 */
	public String[] patch(String[] doc, int offset) {
		// 文書をリストに変換。
		List docList = new LinkedList(Arrays.asList(doc));

		// 文書にパッチを当てる。
		for (int i = 0; i < deltas.length; i++) {
			offset += deltas[i].patch(docList, offset);
		}

		// パッチを当てた文書を返す。
		return (String[]) docList.toArray(new String[docList.size()]);
	}

	/**
	 * 同じ編集元文書から構築された文書差分をマージする。
	 * 
	 * @param docDiff 同じ編集元文書から構築された文書差分
	 * @return マージされた文書
	 */
	public MergedDocument merge(DocDifference docDiff) {
		// マージされた文書を生成。
		MergedDocument merged = new MergedDocument(orgDoc);

		// 文書差分から差分を取得。
		Delta[] deltas1 = deltas;
		Delta[] deltas2 = docDiff.getDeltas();

		int offset = 0;
		int i1 = 0, i2 = 0;

		// 編集元文書に、それぞれの文書の差分をパッチとして当てる。
		while (i1 < deltas1.length || i2 < deltas2.length) {
			if (i1 < deltas1.length && deltas2.length <= i2) {
				// 差分2を取得できない場合は、差分1をパッチとして当てる。
				offset += merged.patch(deltas1[i1], offset);
				i1++;
			} else if (deltas1.length <= i1 && i2 < deltas2.length) {
				// 差分1を取得できない場合は、差分2をパッチとして当てる。
				offset += merged.patch(deltas2[i2], offset);
				i2++;
			} else if (deltas1[i1].isConflict(deltas2[i2])) {
				// 差分1と差分2がコンフリクトしている場合は
				// 差分をマージしてパッチを当てる。
				offset += merged.merge(deltas1[i1], deltas2[i2], offset);
				i1++;
				i2++;
			} else if (deltas1[i1].compareTo(deltas2[i2]) <= 0) {
				// 差分1 <= 差分2 の場合は差分1をパッチとして当てる。
				offset += merged.patch(deltas1[i1], offset);
				i1++;
			} else {
				// 差分1 > 差分2 の場合は差分2をパッチとして当てる。
				offset += merged.patch(deltas2[i2], offset);
				i2++;
			}
		}

		// マージされた文書を返す。
		return merged;
	}

	/**
	 * 文章の差分情報の文字列表現を返す。
	 * 
	 * @return 文書の差分情報の文字列表現
	 */
	public String toString() {
		// 文字列表現を返す。
		return FoolishMergeUtils.arrayToString(deltas);
	}

	/**
	 * 比較した文書に含まれる同一行の割合を出す。<BR>
	 * <CODE>
	 * ( <I>LCS</I> * 2 ) / ( <I>文書Aの行数</I> + <I>文書Bの行数</I> )
	 * </CODE>
	 * @return 同一行の割合(パーセンテージ)
	 */
	public int getSameLineRate() {
		BigDecimal a_b = new BigDecimal(A.length + B.length);
		BigDecimal lcs_2 = new BigDecimal(lcs * 2);
		BigDecimal rate = lcs_2.divide(a_b, 2, BigDecimal.ROUND_HALF_UP);
		return rate.multiply(new BigDecimal(100)).intValue();
	}

	/**
	 * LCSを取得する。
	 * 
	 * @return LCS
	 */
	public int getLCS() {
		// LCSを返す。
		return lcs;
	}

	// package private ----------------------------------------------

	/**
	 * 配列Aの長さを取得する。
	 * 
	 * @return 配列Aの長さ
	 */
	int getM() {
		// 配列Aを返す。
		return A.length;
	}

	/**
	 * 配列Bの長さを取得する。
	 * 
	 * @return 配列Bの長さ
	 */
	int getN() {
		// 配列Bを返す。
		return B.length;
	}

	/**
	 * 現在の座標でクローズする。
	 */
	void close() {
		// バッファを生成。
		LinkedList merged = new LinkedList();
		LinkedList org = new LinkedList();
		LinkedList mod = new LinkedList();
		LinkedList chunk = new LinkedList();
		LinkedList dlt = new LinkedList();

		// 通過経路を探索。
		for (LineDifference pos = current; pos != null; pos = pos.getPrev()) {
			// 経路をバッファに格納。
			switch (pos.getState()) {
				case LineDifference.ADD :
					// 追加行
					merged.addFirst(pos);
					mod.addFirst(pos);
					chunk.addFirst(pos);
					break;

				case LineDifference.DEL :
					// 削除行
					merged.addFirst(pos);
					org.addFirst(pos);
					chunk.addFirst(pos);
					break;

				case LineDifference.EQ :
					// 差分がない行
					merged.addFirst(pos);
					org.addFirst(pos);
					mod.addFirst(pos);

					if (chunk.size() != 0) {
						Delta delta = new Delta(chunk);
						dlt.addFirst(delta);
						chunk.clear();
					}

					break;

				default :
					// 状態がない場合は、バッファに格納しない。
					break;
			}
		}

		if (chunk.size() != 0) {
			Delta delta = new Delta(chunk);
			dlt.addFirst(delta);
		}

		// バッファをフィールドにセット。
		lines =
			(LineDifference[]) merged.toArray(
				new LineDifference[merged.size()]);
		original =
			(LineDifference[]) org.toArray(new LineDifference[org.size()]);
		modified =
			(LineDifference[]) mod.toArray(new LineDifference[mod.size()]);
		deltas = (Delta[]) dlt.toArray(new Delta[dlt.size()]);
	}

	/**
	 * 編集元ドキュメントと編集後ドキュメントが等しいものとして、LCS/SEDの経路を取得する。
	 * 
	 * @param len 長さ
	 * @return 文書の差分情報
	 */
	static DocDifference getEquals(int len) {
		// 差分情報を生成。
		DocDifference doc = new DocDifference();

		// 対角線の経路を生成する。
		LineDifference[] lines = new LineDifference[len + 1];

		// 座標を経路に格納。
		for (int i = 0; i < doc.lines.length; i++) {
			LineDifference line = new LineDifference(i, i);
			line.setState(LineDifference.EQ);
			lines[i] = line;
		}

		// 行の差分情報をフィールドにセット。
		doc.lines = lines;
		doc.original = lines;
		doc.modified = lines;

		// 差分情報を返す
		return doc;
	}

	/**
	 * LCSをセットする。
	 * 
	 * @param lcs LCS
	 */
	void setLCS(int lcs) {
		// LCSをフィールドに
		this.lcs = lcs;
	}

	/**
	 * エディットグラフを移動する。
	 * 
	 * @param k 指標
	 * @param y Y座標
	 */
	int snake(int k, int y) {
		int x = y - k;
		// 座標を移動。(距離1)
		move(x, y);

		// 対角線を移動。(距離0)
		while ((x < A.length) && (y < B.length) && (A[x].equals(B[y]))) {
			x += 1;
			y += 1;
			moveAslant(x, y);
		}

		return y;
	}

	// private ------------------------------------------------------

	/**
	 * 座標を移動する。(距離1)
	 * 
	 * @param x X座標
	 * @param y Y座標
	 */
	private void move(int x, int y) {
		// 反転フラグがtrueの場合、X座標とY座標を反転。
		if (reverse) {
			int tmp = x;
			x = y;
			y = tmp;
		}

		// 現在座標を生成。
		current = new LineDifference(x, y);
		// 移動元を検索して、現在座標にセット。
		setPrev(current, x, y - 1, LineDifference.ADD);
		setPrev(current, x - 1, y, LineDifference.DEL);
		// 現在座標に行をセット。
		setLine(current);
		// 通過経路に現在座標をセット。
		passed.add(current);
	}

	/**
	 * 座標を移動する。(距離0)
	 * 
	 * @param x X座標
	 * @param y Y座標
	 */
	private void moveAslant(int x, int y) {
		// 反転フラグがtrueの場合、X座標とY座標を反転。
		if (reverse) {
			int tmp = x;
			x = y;
			y = tmp;
		}

		// 現在座標を生成。
		current = new LineDifference(x, y);
		// 移動元を検索して、現在座標にセット。
		setPrev(current, x - 1, y - 1, LineDifference.EQ);
		// 現在座標に行をセット。
		setLine(current);
		// 通過経路に現在座標をセット。
		passed.add(current);
	}

	/**
	 * 現在座標に移動元をセットする。
	 * 
	 * @param current 現在座標
	 * @param x 移動元のX座標
	 * @param y 移動元のY座標
	 * @param state 状態
	 */
	private void setPrev(LineDifference current, int x, int y, int state) {
		// 既に移動元がセットされている場合は、処理を抜ける。
		if (current.getPrev() != null) {
			return;
		}

		// 通過経路から座標が一致する行の差分情報を検索。
		for (int i = 0; i < passed.size(); i++) {
			LineDifference prev = (LineDifference) passed.get(i);

			if (prev.getX() == x && prev.getY() == y) {
				// 座標が一致した場合、現在座標に移動元、状態、行をセット。
				current.setPrev(prev);
				current.setState(state);
				break;
			}
		}
	}

	/**
	 * 現在座標に移動元をセットする。
	 * 
	 * @param current 現在座標
	 * @param x 移動元のX座標
	 * @param y 移動元のY座標
	 */
	private void setLine(LineDifference current) {
		// 座標を取得。
		int x = current.getX();
		int y = current.getY();

		// 現在座標から状態を取得。
		int state = current.getState();

		// 状態がない場合、処理を抜ける。
		if (state == LineDifference.NO_STATE) {
			return;
		}

		if (state == LineDifference.ADD) {
			// 追加の場合、配列Bを行にセット。
			current.setLine(modDoc[y - 1]);
		} else {
			// 追加の場合、配列Aを行にセット。
			current.setLine(orgDoc[x - 1]);
		}

	}

}
