/* Copyright (c) 2002 Speech and Acoustics Processing Lab., NAIST */
/*   All rights reserved   */

#include "japi.h"

#if defined(_WIN32) && !defined(__CYGWIN32__)
#define WINSOCK
#endif

#ifdef WINSOCK
int winsock_initialized = 0;
#endif

/* connect to server.  return socket descriptor */
int
do_connect(char *hostname, int portnum)
{
  static struct hostent *hp;
  static struct sockaddr_in sin;
  int sd;

#ifdef WINSOCK
  /* init winsock */
  if (winsock_initialized == 0) {
	  WSADATA data;
	  WSAStartup(0x1010, &data);
	  winsock_initialized = 1;
  }
#endif
  /* get host entry */
  if ((hp = gethostbyname(hostname)) == NULL) {
    fprintf(stderr, "Error: host \"%s\" not found\n", hostname);
    exit(1);
  }
  /* open socket */
#ifdef WINSOCK
  if((sd = socket(PF_INET, SOCK_STREAM, 0)) == INVALID_SOCKET){
    perror("Error: socket()") ;
    printf("Error code: %d\n", WSAGetLastError());
    switch(WSAGetLastError()) {
    case WSANOTINITIALISED: printf("A successful WSAStartup must occur before using this function.\n"); break;
    case WSAENETDOWN: printf("The network subsystem or the associated service provider has failed.\n"); break;
    case WSAEAFNOSUPPORT: printf("The specified address family is not supported. \n"); break;
    case WSAEINPROGRESS: printf("A blocking Windows Sockets 1.1 call is in progress, or the service provider is still processing a callback function. \n"); break;
    case WSAEMFILE: printf("No more socket descriptors are available. \n"); break;
    case WSAENOBUFS: printf("No buffer space is available. The socket cannot be created. \n"); break;
    case WSAEPROTONOSUPPORT: printf("The specified protocol is not supported. \n"); break;
    case WSAEPROTOTYPE: printf("The specified protocol is the wrong type for this socket. \n"); break;
    case WSAESOCKTNOSUPPORT: printf("The specified socket type is not supported in this address family. \n"); break;
    }
    exit(1);
  }
#else  /* ~WINSOCK */
  /* create socket */
  if((sd = socket(PF_INET, SOCK_STREAM, 0)) < 0){
    perror("Error: socket()") ;
    exit(1);
  }
#endif /* ~WINSOCK */

  /* connect */
  memset((char *)&sin, 0, sizeof(sin));
  memcpy(&sin.sin_addr, hp->h_addr, hp->h_length);
  sin.sin_family = hp->h_addrtype;
  sin.sin_port = htons(portnum);
  fprintf(stderr, "connecting to %s:%d...", hostname, portnum);
  if (connect(sd, (struct sockaddr *)&sin, sizeof(sin)) < 0) {
    perror("Error");
    exit(1);
  }
  fprintf(stderr, "done\n");
  return(sd);
}

/* disconnect */
void
do_disconnect(int sd)
{
#ifdef WINSOCK
  closesocket(sd);
#else
  if (close(sd) < 0) {
    fprintf(stderr,"Error: close() failed");
  }
#endif
}

/* send data to server (with printf format) */
void
do_sendf(int sd, char *fmt, ...)
{
  static char buf[MAXLINELEN];
  va_list ap;
  int n;
  
  va_start(ap, fmt);
  vsnprintf(buf, MAXLINELEN, fmt, ap);
  n = send(sd, buf, strlen(buf),0);
  if (n < 0) {
    perror("Error: do_sendf");
  }
  va_end(ap);
}

/* send data to server */
void
do_send(int sd, char *buf)
{
  int n;
  
  n = send(sd, buf, strlen(buf),0);
  if (n < 0) {
    perror("Error: do_send");
  }
}

/* reveive data from server (return at newline or EOM) */
/* strip newline */
/* return pointer to buf, or NULL on error */
char *
do_reveive(int sd, char *buf, int maxlen)
{
  int cnt;
  char *p;
  
  p = buf;
  while(1) {
    cnt = recv(sd, p, 1, 0);
    if (cnt <= 0) return NULL;		/* eof or error */
    if (*p == '\n' && p > buf) {
      *p = '\0';
      break;
    } else {
      if (++p >= buf + maxlen) {
	fprintf(stderr,"Error: do_receive: line too long (> %d)\n", maxlen);
	exit(1);
      }
    }
  }
  return buf;
}
