/* Copyright (c) 1991-2002 Doshita Lab. Speech Group, Kyoto University */
/* Copyright (c) 2000-2002 Speech and Acoustics Processing Lab., NAIST */
/*   All rights reserved   */

/* server-client.c --- functions for TCP/IP server & client */

/* $Id: server-client.c,v 1.4 2002/11/06 08:37:06 ri Exp $ */

#include <sent/stddefs.h>
#include <sent/tcpip.h>

#if defined(_WIN32) && !defined(__CYGWIN32__)
#define WINSOCK
#endif

#ifdef WINSOCK
boolean winsock_initialized = FALSE;
#endif

/* Ready for receiving request from client. */
/* Return value: -1 ... fail to create socket descriptor
                 -2 ... fail to assign name to the socket
		 -3 ... fail to start listening
		 other ... socket descriptor ready for accept */
int
ready_as_server(int port_num)
{
  struct sockaddr_in sin;
  int sd;
  int optval;
  int optlen;

#ifdef WINSOCK
  /* init winsock */
  if (!winsock_initialized) {
	  WSADATA data;
	  WSAStartup(0x1010, &data);
	  winsock_initialized = TRUE;
  }
#endif

  /* create socket */
#ifdef WINSOCK
  if((sd = socket(PF_INET, SOCK_STREAM, 0)) == INVALID_SOCKET){
    perror("ready_as_server: socket() error") ;
    printf("Error code: %d\n", WSAGetLastError());
    switch(WSAGetLastError()) {
    case WSANOTINITIALISED: printf("A successful WSAStartup must occur before using this function.\n"); break;
    case WSAENETDOWN: printf("The network subsystem or the associated service provider has failed.\n"); break;
    case WSAEAFNOSUPPORT: printf("The specified address family is not supported. \n"); break;
    case WSAEINPROGRESS: printf("A blocking Windows Sockets 1.1 call is in progress, or the service provider is still processing a callback function. \n"); break;
    case WSAEMFILE: printf("No more socket descriptors are available. \n"); break;
    case WSAENOBUFS: printf("No buffer space is available. The socket cannot be created. \n"); break;
    case WSAEPROTONOSUPPORT: printf("The specified protocol is not supported. \n"); break;
    case WSAEPROTOTYPE: printf("The specified protocol is the wrong type for this socket. \n"); break;
    case WSAESOCKTNOSUPPORT: printf("The specified socket type is not supported in this address family. \n"); break;
    }
    return -1;
  }
#else  /* ~WINSOCK */
  /* create socket */
  if((sd = socket(PF_INET, SOCK_STREAM, 0)) < 0){
    perror("ready_as_server: socket() error") ;
    return -1;
  }
#endif /* ~WINSOCK */

  /* set socket to allow reuse of local address at bind() */
  /* this option prevent from "error: Address already in use" */
  optval = 1;
  optlen = sizeof(int);
  if (setsockopt(sd, SOL_SOCKET, SO_REUSEADDR, (char *)&optval, optlen) != 0) {
    perror("ready_as_server: setsockopt() error");
    return -2;
  }

  /* assign name(address) to socket */
  memset((char *)&sin, 0, sizeof(sin));
  sin.sin_family = AF_INET;
  sin.sin_port = htons((unsigned short)port_num);
  if(bind(sd, (struct sockaddr *)&sin, sizeof(sin)) < 0){
    perror("ready_as_server: bind() error") ;
    return -2;
  }
  /* begin to listen */
  if (listen(sd, 5) < 0) {
    perror("ready_as_server: listen() error");
    return -3;
  }
  return(sd);
}

#ifndef WINSOCK
/* Ready for receiving request from client. */
/* Created socket will be non-blocking. */
/* Return value: -1 ... fail to create socket descriptor
                 -2 ... fail to assign name to the socket
		 -3 ... fail to start listening
		 -4 ... fail to set socket non-blocking
		 other ... socket descriptor ready for accept */
int
ready_as_server_nonblock(int port_num)
{
  struct sockaddr_in sin;
  int sd;

  memset((char *)&sin, 0, sizeof(sin));
  sin.sin_family = AF_INET;
  sin.sin_port = htons(port_num);
  if((sd = socket(PF_INET, SOCK_STREAM, 0)) < 0){
    perror("ready_as_server_nonblock: socket() error") ;
    return -1;
  }
  if(bind(sd, (struct sockaddr *)&sin, sizeof(sin)) < 0){
    perror("ready_as_server_nonblock: bind() error") ;
    return -2;
  }
  if((fcntl(sd, F_SETFL , fcntl(sd,F_GETFL,0)|O_NDELAY ))== -1){
    perror("ready_as_server_nonblock: fcntl() error");
    return -4;
  }
  if (listen(sd, 5) < 0) {
    perror("ready_as_server_nonblock: listen() error");
    return -3;
  }
  return(sd);
}
#endif /* ~WINSOCK */

/* Wait for request from client. */
/* If blocking, wait until request comes. */
/* If non-blocking, no request in queue is an error (negative value returns) */
/* returns accepted socket descriptor */
static struct sockaddr_in from;
#ifdef HAVE_SOCKLEN_T
static socklen_t nbyte;
#else  
static int nbyte;
#endif /* HAVE_SOCKLEN_T */

int
accept_from(int sd)
{
  int asd;
  
  nbyte = sizeof(struct sockaddr_in);
  asd = accept(sd, (struct sockaddr *)&from, &nbyte);
  if (asd < 0) {		/* error */
#ifdef WINSOCK
    switch(WSAGetLastError()) {
    case WSANOTINITIALISED: printf("A successful WSAStartup must occur before using this FUNCTION. \n"); break;
    case WSAENETDOWN: printf(" The network subsystem has failed. \n"); break;
    case WSAEFAULT: printf(" The addrlen parameter is too small or addr is not a valid part of the user address space. \n"); break;
    case WSAEINTR: printf(" A blocking Windows Sockets 1.1 call was canceled through WSACancelBlockingCall. \n"); break;
    case WSAEINPROGRESS: printf(" A blocking Windows Sockets 1.1 call is in progress, or the service provider is still processing a callback function. \n"); break;
    case WSAEINVAL: printf(" The listen function was not invoked prior to accept. \n"); break;
    case WSAEMFILE: printf(" The queue is nonempty upon entry to accept and there are no descriptors available. \n"); break;
    case WSAENOBUFS: printf(" No buffer space is available. \n"); break;
    case WSAENOTSOCK: printf(" The descriptor is not a socket. \n"); break;
    case WSAEOPNOTSUPP: printf(" The referenced socket is not a type that supports connection-oriented service. \n"); break;
    case WSAEWOULDBLOCK: printf(" The socket is marked as nonblocking and no connections are present to be accepted. \n"); break;
    }
#else  /* ~WINSOCK */
    perror("accept_from: accept() error");
#endif
    j_error("failed to accept connection\n");
  }
  j_printf("connect from %s\n", inet_ntoa(from.sin_addr));
  return asd;
}
  

/* Send connection request to a server. */
/* Return value:
   -1 ... fail to create socket descripter
   -2 ... fail to connect
   -3 ... no such host */
int
make_connection(char *hostname, int port_num)
{
  static struct hostent *hp;
  static struct sockaddr_in	sin;
  int sd;

#ifdef WINSOCK
  /* init winsock */
  if (!winsock_initialized) {
	  WSADATA data;
	  WSAStartup(0x1010, &data);
	  winsock_initialized = TRUE;
  }
#endif

  /* host existence check */
  if ((hp  = gethostbyname(hostname)) == NULL) {
    j_printerr("make_connection: host not found: %s\n",hostname);
    return -3;
  }

 /* create socket */
#ifdef WINSOCK
  if((sd = socket(PF_INET, SOCK_STREAM, 0)) == INVALID_SOCKET){
    perror("make_connection: socket() error") ;
    printf("Error code: %d\n", WSAGetLastError());
    switch(WSAGetLastError()) {
    case WSANOTINITIALISED: printf("A successful WSAStartup must occur before using this function.\n"); break;
    case WSAENETDOWN: printf("The network subsystem or the associated service provider has failed.\n"); break;
    case WSAEAFNOSUPPORT: printf("The specified address family is not supported. \n"); break;
    case WSAEINPROGRESS: printf("A blocking Windows Sockets 1.1 call is in progress, or the service provider is still processing a callback function. \n"); break;
    case WSAEMFILE: printf("No more socket descriptors are available. \n"); break;
    case WSAENOBUFS: printf("No buffer space is available. The socket cannot be created. \n"); break;
    case WSAEPROTONOSUPPORT: printf("The specified protocol is not supported. \n"); break;
    case WSAEPROTOTYPE: printf("The specified protocol is the wrong type for this socket. \n"); break;
    case WSAESOCKTNOSUPPORT: printf("The specified socket type is not supported in this address family. \n"); break;
    }
    return -1;
  }
#else  /* ~WINSOCK */
  /* create socket */
  if((sd = socket(PF_INET, SOCK_STREAM, 0)) < 0){
    perror("make_connection: socket() error") ;
    return -1;
  }
#endif /* ~WINSOCK */

  /* try to connect */
    memset((char *)&sin, 0, sizeof(sin));
    memcpy(&sin.sin_addr, hp->h_addr, hp->h_length);
    sin.sin_family = hp->h_addrtype;
    sin.sin_port = htons((unsigned short)port_num);
    if (connect(sd, (struct sockaddr *)&sin, sizeof(sin)) < 0) {
      /* failure */
      perror("make_connection");
      j_printerr("make_connection: failed to connect to $s:%d\n", hostname, port_num);
      return -2;
    }

  return sd;
}

/* close socket */
int
close_socket(int sd)
{
#ifdef WINSOCK
  return(closesocket(sd));
#else
  return(close(sd));
#endif
}

/* clean-up socket at program exit */
void
cleanup_socket()
{
#ifdef WINSOCK
  if (winsock_initialized) {
    WSACleanup();
  }
#endif
}
