/* Copyright (c) 1991-2002 Doshita Lab. Speech Group, Kyoto University */
/* Copyright (c) 2000-2002 Speech and Acoustics Processing Lab., NAIST */
/*   All rights reserved   */

/* mybmalloc.c ---  malloc by large segment */
/* this will help when huge number of small memories will be allocated for
   LM and AM */

/* !!! NOTIVE !!! you CANNOT FREET the memories allocated by these functions */

/* $Id: mybmalloc.c,v 1.4 2002/09/11 22:01:50 ri Exp $ */


#undef DEBUG			/* output debug message */

#define MYBMALLOC_LINKED 1

#include <sent/stddefs.h>

static boolean initialized = FALSE;
static char *current = NULL, *nowp = NULL, *endp = NULL;
static unsigned int blocksize;
static int align;
static unsigned int align_mask;

#ifdef MYBMALLOC_LINKED
typedef struct __linked_buffer__ {
  unsigned long size;
  char *buffer;
  char *nowp;
  struct __linked_buffer__ *next;
} *LINKED_BUFFER;

static LINKED_BUFFER first_linked_buffer = NULL;
static LINKED_BUFFER last_linked_buffer = NULL;
#endif


/* set block size and memory alignment factor */
void
mybmalloc_set_param()
{
  long pagesize, blockpagenum;

  /* block size should be rounded up by page size */
  pagesize = getpagesize();
  blockpagenum = (MYBMALLOC_BLOCK_SIZE + (pagesize - 1)) / pagesize;
  blocksize = pagesize * blockpagenum;

  /* alignment by a word (= pointer size?) */
#ifdef NO_ALIGN_DOUBLE
  align = sizeof(void *);
#else
  /* better for floating points */
  align = sizeof(double);
#endif
  align_mask = ~(align - 1);	/* assume power or 2 */
#ifdef DEBUG
  j_printerr("pagesize=%d blocksize=%d align=%d (bytes)\n", (int)pagesize, blocksize, align);
#endif
  
  initialized = TRUE;
}

/* malloc specified size and return the pointer */
void *
mybmalloc(int size)
{
  void *allocated;
  if (!initialized) mybmalloc_set_param();  /* initialize if not yet */
  /* malloc segment should be aligned to a word boundary */
  size = (size + align - 1) & align_mask;
  if (!current || nowp + size >= endp) {
#ifndef MYBMALLOC_LINKED
    if (size > blocksize) {
      /* large block, fall to normal malloc */
      return(mymalloc(size));
    }
    /* allocate next block */
    current = (char *)mymalloc(blocksize);
    endp = current + blocksize;
    nowp = current;
#else
    unsigned long current_size;
    LINKED_BUFFER linked_buffer = NULL;
    static LINKED_BUFFER prev_linked_buffer = NULL;
    
    if (first_linked_buffer != NULL) {
      LINKED_BUFFER next_lb;

      if (prev_linked_buffer != NULL) {
        prev_linked_buffer->nowp = nowp;
      }

      /* search the buffer having left space */
      next_lb = first_linked_buffer;
      while (next_lb != NULL) {
        if (next_lb->nowp + size <= next_lb->buffer + next_lb->size) {
	  linked_buffer = next_lb;
	  break;
	}
        next_lb = next_lb->next;
      }
    }

    if (linked_buffer != NULL) {
      current = linked_buffer->nowp;
      endp = linked_buffer->buffer + linked_buffer->size;
    } else {
      current_size = (unsigned long)blocksize;
      while (current_size < (unsigned long)size) {
	current_size += (unsigned long)blocksize;
      }
    
      linked_buffer = (LINKED_BUFFER)mymalloc(sizeof(struct __linked_buffer__));
      linked_buffer->size = current_size;
      linked_buffer->buffer = (char *)mymalloc(linked_buffer->size);
      linked_buffer->nowp = linked_buffer->buffer;
      linked_buffer->next = NULL;
    
      if (first_linked_buffer == NULL) {
        first_linked_buffer = linked_buffer;
      } else if (last_linked_buffer != NULL) {
	last_linked_buffer->next = linked_buffer;
      }
      last_linked_buffer = linked_buffer;
      current = linked_buffer->buffer;
      endp = current + current_size;
    }
    prev_linked_buffer = linked_buffer;
    
    nowp = current;
#endif    
  }
  /* return current pointer */
  allocated = nowp;
  nowp += size;
  return(allocated);
}

void mybmalloc_free(void)
{
#ifdef MYBMALLOC_LINKED
  if (first_linked_buffer != NULL) {
    LINKED_BUFFER curr_lb, next_lb;

    next_lb = first_linked_buffer;
    while (next_lb != NULL) {
      curr_lb = next_lb;
      next_lb = curr_lb->next;
      free(curr_lb->buffer);
      free(curr_lb);
    }
  }
  first_linked_buffer = NULL;
  last_linked_buffer = NULL;
  current = nowp = endp = NULL;
#endif
  
  return;
}
    
/* duplicate string using mybmalloc */
char *
mybstrdup(char *s)
{
  char *allocated;
  int size = strlen(s) + 1;
  allocated = mybmalloc(size);
  memcpy(allocated, s, size);
  return(allocated);
}

/* new block malloc: keep base pointers on outside to enable free later */
void *
mybmalloc2(int size, BMALLOC_BASE **list)
{
  void *allocated;
  BMALLOC_BASE *new;
  if (!initialized) mybmalloc_set_param();  /* initialize if not yet */
  /* malloc segment should be aligned to a word boundary */
  size = (size + align - 1) & align_mask;
  if (*list == NULL || (*list)->now + size >= (*list)->end) {
    new = (BMALLOC_BASE *)mymalloc(sizeof(BMALLOC_BASE));
    if (size > blocksize) {
      /* large block, allocate a whole block */
      new->base = mymalloc(size);
      new->end = (char *)new->base + size;
    } else {
      /* allocate per blocksize */
      new->base = mymalloc(blocksize);
      new->end = (char *)new->base + blocksize;
    }
    new->now = (char *)new->base;
    new->next = (*list);
    *list = new;
  }
  /* return current pointer */
  allocated = (*list)->now;
  (*list)->now += size;
  return(allocated);
}

/* free all base pointer */
void
mybfree2(BMALLOC_BASE **list)
{
  BMALLOC_BASE *b, *btmp;
  b = *list;
  while (b) {
    btmp = b->next;
    free(b->base);
    free(b);
    b = btmp;
  }
  *list = NULL;
}
