;;;
;;; fastcgi.scm - FastCGI interface
;;;
;;;   Copyright (c) 2004 Kimura Fuyuki, All rights reserved.
;;;
;;;   Redistribution and use in source and binary forms, with or without
;;;   modification, are permitted provided that the following conditions
;;;   are met:
;;;
;;;   1. Redistributions of source code must retain the above copyright
;;;      notice, this list of conditions and the following disclaimer.
;;;
;;;   2. Redistributions in binary form must reproduce the above copyright
;;;      notice, this list of conditions and the following disclaimer in the
;;;      documentation and/or other materials provided with the distribution.
;;;
;;;   3. Neither the name of the authors nor the names of its contributors
;;;      may be used to endorse or promote products derived from this
;;;      software without specific prior written permission.
;;;
;;;   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
;;;   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
;;;   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
;;;   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
;;;   OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
;;;   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
;;;   TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
;;;   PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
;;;   LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
;;;   NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
;;;   SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
;;;
;;;  $Id: fastcgi.scm,v 1.1 2006/01/14 04:18:31 shirok Exp $
;;;

(define-module www.fastcgi
  (use www.cgi)
  (use gauche.parameter)
  (export call-with-fastcgi with-fastcgi))
(select-module www.fastcgi)
(dynamic-load "fastcgi")

(define (error? obj) (is-a? obj <error>))

(define (error->string obj)
  (and (is-a? obj <error>) (ref obj 'message)))

(define (call-with-fastcgi proc)
  (cond ((fcgx-is-cgi)
         (proc (current-input-port)
               (current-output-port)
               (current-error-port)
               '()))
        (else
         ;; rush into the infinity..
         (let loop ()
           ;; leave these signals to libfcgi
           (set-signal-handler! SIGTERM #t)
           (set-signal-handler! SIGHUP #t)
           (set-signal-handler! SIGUSR1 #t)
           ;; mod_fastcgi requires this!
           (set-signal-handler! SIGPIPE (lambda (k) (loop)))

           (receive (in out err env) (fcgx-accept)
             (let ((iport (fcgx-stream->port in))
                   (oport (fcgx-stream->port out))
                   (eport (fcgx-stream->port err))
                   (mvs (map (lambda (s)
                               (receive r (string-scan s #\= 'both) r))
                             env)))

               (with-error-handler
                 (lambda (e)
                   (display (if (error? e) (error->string e) e) eport)
                   (newline eport)
                   (close-input-port iport)
                   (close-output-port oport)
                   (close-output-port eport)
                   (fcgx-finish)
                   (raise e))
                 (lambda ()
                   (proc iport oport eport mvs)
                   (close-input-port iport)
                   (close-output-port oport)
                   (close-output-port eport)))))
           (loop)))))

(define (with-fastcgi thunk)
  (call-with-fastcgi
    (lambda (iport oport eport mvs)
      (with-input-from-port iport
        (lambda ()
          (with-output-to-port oport
            (lambda ()
              (with-error-to-port eport
                (lambda ()
                  (parameterize ((cgi-metavariables mvs))
                    (thunk)))))))))))

(provide "www/fastcgi")
