
/*	
	GameboyAdvanceVM 
		- Nintendo GameboyAdvance Emulator
	Copyright 2002 Y_N y_n@users.sourceforge.jp
	Homepage https://sourceforge.jp/projects/gbaemu/
*/


#define	SOREG		((opcode) & 0xF)		/*2nd operand register*/
#define	DEREG		((opcode>>12) & 0xF)	/*Destination register*/
#define	FOREG		((opcode>>16) & 0xF)	/*1st operand register*/

#define	REG_0_4		((opcode) & 0xF)
#define	REG_12_4	((opcode>>12) & 0xF)
#define	REG_16_4	((opcode>>16) & 0xF)

#define	GETCOND		((opcode>>28) & 0x0F)	/*RfBVR[h*/

#define	IMM_OPERAND	(opcode & BIT_25_)			/*ltO*/

#define	ARM_MODE_USR	0x10					/*CPU[htO*/
#define	ARM_MODE_FIQ	0x11
#define	ARM_MODE_IRQ	0x12
#define	ARM_MODE_SVR	0x13
#define	ARM_MODE_ABT	0x17
#define	ARM_MODE_UND	0x1B
#define	ARM_MODE_SYS	0x1F

_inline void SetCpuMode(u32 mode)
{
	CPSR |= mode;
	CPSR &= 0xFFFFFF10 | (~mode);
}

#ifdef	MACRO_EXPAND

#define SetSubFlag(a, b, c)	\
{	/*sub,rsb,rsc,cmp*/\
	if(!c)SF(Z_); else RF(Z_);\
	if(c & BIT_31_)SF(N_); else RF(N_);\
	if(((a & ~b) | (a & ~c) | (~b & ~c)) & BIT_31_)SF(C_); else RF(C_);\
	if(((a & ~(b | c)) | ((b & c) & ~a)) & BIT_31_)SF(V_); else RF(V_);\
}

#define SetAddFlag(a, b, c)	\
{	/*add,adc,cmn*/\
	if(!c)SF(Z_); else RF(Z_);\
	if(c & BIT_31_)SF(N_); else RF(N_);\
	if(((a & b) | (a & ~c) | (b & ~c)) & BIT_31_)SF(C_); else RF(C_);\
	if(((a & b & ~c) | (~a & ~b & c)) & BIT_31_)SF(V_); else RF(V_);\
}

#define SetDpFlags(a)	\
{\
	if(!a)SF(Z_); else RF(Z_);\
	if(a & BIT_31_)SF(N_); else RF(N_);\
}

#else	/*MACRO_EXPAND*/

_inline void SetSubFlag(u32 a, u32 b, u32 c)
{	/*sub,rsb,rsc,cmp*/
	if(!c)SF(Z_); else RF(Z_);
	if(c & BIT_31_)SF(N_); else RF(N_);
	if(((a & ~b) | (a & ~c) | (~b & ~c)) & BIT_31_)SF(C_); else RF(C_);
	if(((a & ~(b | c)) | ((b & c) & ~a)) & BIT_31_)SF(V_); else RF(V_);
}

_inline void SetAddFlag(u32 a, u32 b, u32 c)
{	/*add,adc,cmn*/
	if(!c)SF(Z_); else RF(Z_);
	if(c & BIT_31_)SF(N_); else RF(N_);
	if(((a & b) | (a & ~c) | (b & ~c)) & BIT_31_)SF(C_); else RF(C_);
	if(((a & b & ~c) | (~a & ~b & c)) & BIT_31_)SF(V_); else RF(V_);
}

_inline void SetDpFlags(u32 a)	
{	/*muls,mlas, and,eor,tst,teq,orr,mov,bic,mvn*/
	if(!a)SF(Z_); else RF(Z_);
	if(a & BIT_31_)SF(N_); else RF(N_);
}

#endif	/*MACRO_EXPAND*/


u32 imm_shift(u32 opcode)
{	/*Vtgl*/
	u32	bit, temp32;
	u32	shift, rm;
	u8	shift_type;

	rm	= SOREG;
	shift_type = (u8)((opcode>>5) & 3);

	if(opcode & 0x08){	/*Rs(8-11)*/
		shift = (opcode>>8) & 0xF;
	}else{				/*Rs(7-11)*/
		shift = (opcode>>7) & 0x1F;
	}

	switch(shift_type){
	case 0:	/*00 = logical left(lsl) _IVtg 0*/
		return ((arm.reg[rm])<<shift);
	case 1:	/*01 = logical right(lsr) _IEVtg 0*/
		return ((arm.reg[rm])>>shift);
	case 2:	/*10 = arithmetic right(asr) ZpIEVtg 31rbg̒l*/
		temp32 = ((arm.reg[rm])>>shift);
		if(arm.reg[rm] & BIT_31_){
			for(bit = BIT_31_; shift; bit>>=1){
				temp32 |= bit;
				shift--;
			}
		}
		return temp32;
	case 3:	/*11 = rotate right(ror) lւ*/
		return ((arm.reg[rm])>>shift) | ((arm.reg[rm])<<(32 - shift));
	}

	return 0;
}

u32 imm_rotate(u32 opcode)
{	/*E[e[gl*/
	u32	shift, imm_val;

	shift	= ((opcode>>8) & 0xF)<<1;
	imm_val	= opcode & 0xFF;

	return (imm_val>>shift) | ((imm_val)<<(32 - shift));
}

_inline void arm_adc()
{	/*Rd := Rn + Op2 + Carry <Data Processing> (L[tZ)*/
	u32	opcode, operand;
	u32	rd, rn;

	opcode	= OPCODE;
	rd		= DEREG;
	rn		= FOREG;

	if(IMM_OPERAND){	/*Immediate Operand*/
		operand = imm_rotate(opcode);
	}else{
		operand = imm_shift(opcode);
	}

	operand += LSB_FC;
	/*if((arm.reg[rn] + operand) > 0xFFFFFFFF)SF(C_); else RF(C_);*/

	arm.reg[rd] = arm.reg[rn] + operand;

	SetAddFlag(arm.reg[rn], operand, arm.reg[rd]);

	arm.cycle = 1;
}

_inline void arm_add()
{	/*Rd := Rn + Op2 <Data Processing> (Z)*/
	u32	opcode, operand;
	u32	rd, rn;

	opcode	= OPCODE;
	rd		= DEREG;
	rn		= FOREG;

	if(IMM_OPERAND){	/*Immediate Operand*/
		operand = imm_rotate(opcode);
	}else{
		operand = imm_shift(opcode);
	}

	arm.reg[rd] = arm.reg[rn] + operand;

	SetAddFlag(arm.reg[rn], operand, arm.reg[rd]);

	arm.cycle = 1;
}

_inline void arm_and()
{	/*Rd := Rn AND Op2 <Data Processing> (_)*/
	u32	opcode, operand;
	u32	rd, rn;

	opcode	= OPCODE;
	rd		= DEREG;
	rn		= FOREG;
	
	if(IMM_OPERAND){	/*Immediate Operand*/
		operand = imm_rotate(opcode);
	}else{
		operand = imm_shift(opcode);
	}

	arm.reg[rd] = arm.reg[rn] & operand;

	SetDpFlags(arm.reg[rd]);

	arm.cycle = 1;
}

_inline void arm_b()
{	/*R15 := address ()*/
	u32	opcode,	offset, address;

	opcode	= OPCODE;
	/*l͂S{*/
	offset	= ((opcode & 0x00FFFFFF)<<2);

	address = PC + offset + 4;

	/*ItZbg͈̔͂MSB^Ȃ}CiX Zl{̒l*/
	if(offset & BIT_25_)address -= BIT_25_<<1;	/*tItZbg*/

	PC = address;

	arm.cycle = 3;
}

_inline void arm_bic()
{	/*Rd := Rn AND NOT Op2 <Data Processing> (rbgNA)*/
	u32	opcode, operand;
	u32	rd, rn;

	opcode	= OPCODE;
	rd		= DEREG;
	rn		= FOREG;

	if(IMM_OPERAND){
		operand = imm_rotate(opcode);
	}else{
		operand = imm_shift(opcode);
	}

	arm.reg[rd] = arm.reg[rn] & ~operand;

	SetDpFlags(arm.reg[rd]);

	arm.cycle = 1;
}

_inline void arm_bl()
{	/*R14 := R15, R15 := address (ƃN)*/
	u32	opcode,	offset, address;

	opcode	= OPCODE;
	/*l͂S{*/
	offset	= ((opcode & 0x00FFFFFF)<<2);

	R14 = PC;// + 4;
	address = PC + offset + 4;

	/*ItZbg͈̔͂MSB^Ȃ}CiX Zl{̒l*/
	if(offset & BIT_25_)address -= BIT_25_<<1;	/*tItZbg*/

	PC = address;

	arm.cycle = 3;
}

_inline void arm_bx()
{	/*R15 := Rn (Ɩ߃Xe[g̕ύX)*/
	u32	opcode;
	u32 rn;
	
	opcode	= OPCODE;
	rn		= REG_0_4;
	R15		= arm.reg[rn];

	if(R15 & BIT_1_){
		CPSR |= T_;		/*ŉʃrbgPȂTHUMBXe[gɕύX*/
	}else{
		CPSR &= ~T_;	/*ŉʃrbgOȂARMXe[gɕύX*/
	}

	arm.cycle = 3;
}

_inline void arm_cdp()
{	/*(Coprocessor-specific)*/

	arm.cycle = 2;
}

_inline void arm_cmn()
{	/*CPSR flags := Rn + Op2 <Data Processing> (RvZbTf[^)*/
	u32	opcode, operand;
	u32	temp;
	u32	rn;

	opcode	= OPCODE;
	rn		= FOREG;

	if(IMM_OPERAND){
		operand = imm_rotate(opcode);
	}else{
		operand = imm_shift(opcode);
	}

	temp = arm.reg[rn] + operand;

#ifdef	INLINE_FLAG
	if(!temp)SF(Z_); else RF(Z_);
	if((arm.reg[rn] - operand) > 0xFFFFFFFF)SF(C_); else RF(C_);
	if(temp & BIT_31_)SF(N_); else RF(N_);
#else
	SetAddFlag(arm.reg[rn], operand, temp);
#endif	/*INLINE_FLAG*/

	arm.cycle = 1;
}

_inline void arm_cmp()
{	/*CPSR flags := Rn - Op2 <Data Processing> (r)*/
	u32	opcode, operand;
	u32	temp;
	u32	rn;

	opcode	= OPCODE;
	rn		= FOREG;

	if(IMM_OPERAND){	/*Immediate Operand*/
		operand = imm_rotate(opcode);
	}else{
		operand = imm_shift(opcode);
	}

	temp = arm.reg[rn] - operand;

#ifdef	INLINE_FLAG
	if((arm.reg[rn] - operand) > 0)SF(C_); else RF(C_);
	if(!temp)SF(Z_); else RF(Z_);
	if(temp & BIT_31_)SF(N_); else RF(N_);
#else
	SetSubFlag(arm.reg[rn], operand, temp);
#endif	/*INLINE_FLAG*/

	arm.cycle = 1;
}

_inline void arm_eor()
{	/*Rd := (Rn AND NOT Op2) OR (op2 AND NOT Rn) <Data Processing> (rI_a)*/
	u32	opcode, operand;
	u32	rd, rn;

	opcode	= OPCODE;
	rd		= DEREG;
	rn		= FOREG;
	
	if(IMM_OPERAND){	/*Immediate Operand*/
		operand = imm_rotate(opcode);
	}else{
		operand = imm_shift(opcode);
	} 

	arm.reg[rd] = arm.reg[rn] ^ operand;

	SetDpFlags(arm.reg[rd]);

	arm.cycle = 1;
}

_inline void arm_ldc()
{	/*Coprocessor load (RvZbTւ̓])*/
	u32	opcode;
	u32	rd, rn, cp;
	u8	offset;

	opcode	= OPCODE;
	rd		= (opcode>>12) & 0x0F;
	rn		= (opcode>>16) & 0x0F;
	cp		= (opcode) & 0x0F;
	offset	= (u8)(opcode);

	arm.cycle = 3;
}

_inline void arm_ldm()
{	/*Stack manipulation (̃WX^̃[h,POP)*/
	u32	opcode,	address, bit;
	u32	rn, lst;
	u32 fprepost;

	arm.cycle = 3;

	opcode	= OPCODE;
	rn		= (opcode>>16) & 0x0F;	/*16-19 4bit*/

	address = arm.reg[rn];

	fprepost = (opcode & BIT_24_)?TRUE:FALSE;

	for(bit = BIT_15_, lst = 15; bit; bit >>= 1,lst--){
		if(opcode & bit){//TRACE("%2d,%08X\n",lst, bit);
			arm.cycle++;
			if(fprepost){	/*Pre: ]OɃItZbgǉ*/
				if(opcode & BIT_23_){	/*Up: ItZbgx[XɉZ*/
					address += 4;
				}else{					/*Down: ItZbgx[X猸Z*/
					address -= 4;
				}
			}
			arm.reg[lst] = agb_read_mem32(address);
			if(!fprepost){	/*Post: ]ɃItZbgǉ*/
				if(opcode & BIT_23_){	/*Up: ItZbgx[XɉZ*/
					address += 4;
				}else{					/*Down: ItZbgx[X猸Z*/
					address -= 4;
				}
			}
		}
	}

	if((opcode & BIT_21_) || !fprepost){	/*Write-back: SP(rn)̓eXV*/
		arm.reg[rn] = address;
	}
}

_inline void arm_ldr()
{	/*Rd := (address) (烌WX^ւ̓ǂݍ)*/
	u32	opcode;
	u32	offset, address;
	u32	rd, rn;
	u32 fprepost;

	opcode = OPCODE;
	rd     = DEREG;
	rn     = (opcode>>16) & 0xF;	/*AhX̃x[XWX^*/
	
	address = arm.reg[rn];

	if(rn==15)address += 8;	/*ΏۂPC̏ꍇ*/

	fprepost = (opcode & BIT_24_)?TRUE:FALSE;

	if(opcode & BIT_25_){	/*1 = ItZbg̓WX^[*/
		offset = imm_shift(opcode);
	}else{					/*0 = ItZbg𑦒l*/
		offset = opcode & 0xFFF;
	}

	if(fprepost){	/*1 = Pre: ItZbgǉOɓ]*/
		if(opcode & BIT_23_){	/*1 = Up: ItZbgx[XɉZ*/
			address += offset;
		}else{					/*0 = Down: ItZbgx[X猸Z*/
			address -= offset;
		}
	}

	arm.reg[rd] = (opcode & BIT_22_)?agb_read_mem8(address):agb_read_mem32(address);

	if(!fprepost){	/*0 = Post: ItZbgǉ]*/
		if(opcode & BIT_23_){	/*1 = Up: ItZbgx[XɉZ*/
			address += offset;
		}else{					/*0 = Down: ItZbgx[X猸Z*/
			address -= offset;
		}
	}

	if((opcode & BIT_21_) || !fprepost){	/*Write-back: ݐ̃AhXcĂ*/
		arm.reg[rn] = address;
	}

	arm.cycle = 3;
}

_inline void arm_ldrs()
{	/*Rd := (address) (WX^ւ̃[h) Register offset*/
	u32	opcode, address;
	u32	rd, rn, rm;
	u32	fprepost;

	opcode	= OPCODE;
	rd		= DEREG;					/*ΏۃWX^*/
	rn		= (opcode>>16) & 0xF;	/*AhX̃x[XWX^*/
	rm		= opcode & 0xF;

	address = arm.reg[rn];

	fprepost = (opcode & BIT_24_)?TRUE:FALSE;

	if(fprepost){	/*1 = Pre: ]OɃItZbgǉ*/
		if(opcode & BIT_23_){	/*Up: ItZbgx[XɉZ*/
			address += arm.reg[rm];
		}else{					/*Down: ItZbgx[X猸Z*/
			address -= arm.reg[rm];
		}
	}

	if(opcode & BIT_5_){	/*Halfwords*/
		if(opcode & BIT_6_){	/*Signed*/
			/*11: tn[t[h*/
			arm.reg[rd] = agb_read_mem16(address);
		}else{
			/*01: n[t[h*/
			arm.reg[rd] = agb_read_mem16(address);
		}
	}else{
		if(opcode & BIT_6_){	/*Signed*/
			/*10: toCg*/
			arm.reg[rd] = agb_read_mem8(address);
		}else{
			/*00: SWP߂Œ`ς*/
		}
	}

	if(!fprepost){	/*0 = Post: ]ɃItZbgǉ*/
		if(opcode & BIT_23_){	/*Up: ItZbgx[XɉZ*/
			address += arm.reg[rm];
		}else{					/*Down: ItZbgx[X猸Z*/
			address -= arm.reg[rm];
		}
	}

	if((opcode & BIT_21_) || !fprepost){	/*Write-back: ݐ̃AhXcĂ*/
		arm.reg[rn] = address;
	}

	arm.cycle = 3;
}

_inline void arm_ldrs_imm()
{	/*Rd := (address) (WX^ւ̃[h) Immidiate offset*/
	u32	opcode,	offset, address;
	u32	rd, rn;
	u32	fprepost;

	opcode	= OPCODE;
	rd		= DEREG;					/*ΏۃWX^*/
	rn		= (opcode>>16) & 0xF;	/*AhX̃x[XWX^*/
	offset	= (opcode & 0xF) | ((opcode>>4) & 0xF0);

	address = arm.reg[rn];

	fprepost = (opcode & BIT_24_)?TRUE:FALSE;

	if(fprepost){	/*1 = Pre: ]OɃItZbgǉ*/
		if(opcode & BIT_23_){	/*Up: ItZbgx[XɉZ*/
			address += offset;
		}else{					/*Down: ItZbgx[X猸Z*/
			address -= offset;
		}
	}

	if(opcode & BIT_5_){	/*Halfwords*/
		if(opcode & BIT_6_){	/*Signed*/
			/*Signed Halfwords*/
			arm.reg[rd] = agb_read_mem16(address);
		}else{
			/*Unsigned Halfwords*/
			arm.reg[rd] = agb_read_mem16(address);
		}
	}else{					/*Bytes*/
		if(opcode & BIT_6_){	/*Signed*/
			/*Signed byte*/
			arm.reg[rd] = agb_read_mem8(address);
		}else{
			/*SWP instruction*/
		}
	}

	if(!fprepost){	/*0 = Post: ]ɃItZbgǉ*/
		if(opcode & BIT_23_){	/*Up: ItZbgx[XɉZ*/
			address += offset;
		}else{					/*Down: ItZbgx[X猸Z*/
			address -= offset;
		}
	}

	if((opcode & BIT_21_) || !fprepost){	/*Write-back: ݐ̃AhXcĂ*/
		arm.reg[rn] = address;
	}

	arm.cycle = 3;
}

_inline void arm_mcr()
{	/*cRn := rRn {<op>cRm} (CPUWX^RvZbTWX^ւ̓])*/
	u32	opcode;
	u32	rn, rd, rm;

	opcode	= OPCODE;
	rn		= (opcode>>16) & 0x0F;
	rd		= (opcode>>12) & 0x0F;
	rm		= (opcode) & 0x0F;
	
	arm.cycle = 3;
}

_inline void arm_mla()
{	/*Rd := (Rm * Rs) + Rn (ϘaZ)*/
	u32	opcode;
	u32	rd, rm, rs, rn;

	opcode	= OPCODE;
	
	rd		= (opcode>>16) & 0x0F;
	rs		= (opcode>>8) & 0x0F;
	rm		= (opcode) & 0x0F;
	rn		= (opcode>>12) & 0x0F;

	/*if(((arm.reg[rm] * arm.reg[rs]) + arm.reg[rn]) > 0xFFFFFFFF)SF(C_); else RF(C_);*/

	arm.reg[rd] = (arm.reg[rm] * arm.reg[rs]) + arm.reg[rn];

	SetDpFlags(arm.reg[rd]);//

	arm.cycle = 2;
}

_inline void arm_mlal()
{	/*RdHigh, RdLow := (Rm * Rs) + Rn (64bitϘaZ)*/
	u32	opcode;
	u32	rdh, rdl, rm, rs;

	opcode	= OPCODE;
	rdh		= (opcode>>16) & 0x0F;
	rdl		= (opcode>>12) & 0x0F;
	rs		= (opcode>>8) & 0x0F;
	rm		= (opcode) & 0x0F;

	arm.reg[rdl] = arm.reg[rm] * arm.reg[rs] + arm.reg[rdl];
	arm.reg[rdh] = (u32)((unsigned __int64)arm.reg[rm] * arm.reg[rs] + arm.reg[rdl]) >> 32;

	if(!arm.reg[rdl] && !arm.reg[rdh])SF(Z_); else RF(Z_);
	if(arm.reg[rdh] & BIT_31_)SF(N_); else RF(N_);
/*
MULL	RL,RH,Rm,Rs

UMULL	R1,R4,R2,R3	; R4,R1:=R2*R3
UMLALS	R1,R5,R2,R3	; R5,R1:=R2*R3+R5,R1 also setting
*/
	arm.cycle = 3;
}

_inline void arm_mov()
{	/*Rd : = Op2 <Data Processing> (WX^͒萔̑)*/
	u32	opcode, operand;
	u32	rd;

	opcode	= OPCODE;
	rd		= DEREG;

	if(IMM_OPERAND){	/*Immediate Operand*/
		operand = imm_rotate(opcode);
	}else{
		operand = imm_shift(opcode);
	}

	arm.reg[rd] = operand;

	SetDpFlags(arm.reg[rd]);

	arm.cycle = 1;
}

_inline void arm_mrc()
{	/*Rn := cRn {<op>cRm} (RvZbTWX^CPUWX^ւ̓])*/
	u32	opcode;
	u32	rn, rd, rm;

	opcode	= OPCODE;
	rn		= (opcode>>16) & 0x0F;
	rd		= (opcode>>12) & 0x0F;
	rm		= (opcode) & 0x0F;

	arm.cycle = 3;
}

_inline void arm_mrs()
{	/*Rn := PSR (Xe[^XWX^烌WX^ւ̓])*/
	u32	opcode,	spsr;
	u32	rd;

	opcode	= OPCODE;
	rd		= (opcode>>12) & 0x0F;
	spsr	= ((opcode>>22) & 0x01);	/*src PSRtO*/

	arm.reg[rd] = spsr?arm.reg[SR]:arm.reg[CPSR];
	
	arm.cycle = 3;
}

_inline void arm_msr()
{	/*PSR := Rm (WX^Xe[^XWX^ւ̓])*/
	u32	opcode, dpsr;
	u32	rm;

	opcode	= OPCODE;
	rm		= (opcode) & 0x0F;
	dpsr	= (opcode & BIT_22_);	/*dest PSRtO*/

	arm.reg[dpsr?SR:CPSR] = arm.reg[rm];
	
	arm.cycle = 3;
}

_inline void arm_mul()
{	/*Rd := Rm * Rs (ώZ)*/
	u32	opcode;
	u32	rd, rm, rs;

	opcode	= OPCODE;
	rd		= (opcode>>16) & 0x0F;
	rs		= (opcode>>8) & 0x0F;
	rm		= (opcode) & 0x0F;

	/*if((arm.reg[rm] * arm.reg[rs]) > 0xFFFFFFFF)SF(C_); else RF(C_);*/

	arm.reg[rd] = arm.reg[rm] * arm.reg[rs];

	SetDpFlags(arm.reg[rd]);

	arm.cycle = 3;
}

_inline void arm_mull()
{	/*RdHigh, RdLow := Rm * Rs (64bitώZ)*/
	u32	opcode;
	u32	rdh, rdl, rm, rs;

	opcode	= OPCODE;
	rdh		= (opcode>>16) & 0x0F;
	rdl		= (opcode>>12) & 0x0F;
	rs		= (opcode>>8) & 0x0F;
	rm		= (opcode) & 0x0F;

	arm.reg[rdl] = arm.reg[rm] * arm.reg[rs];
	arm.reg[rdh] = (u32)((unsigned __int64)arm.reg[rm] * arm.reg[rs]) >> 32;

	if(!arm.reg[rdl] && !arm.reg[rdh])SF(Z_); else RF(Z_);
	if(arm.reg[rdh] & BIT_31_)SF(N_); else RF(N_);

	arm.cycle = 3;
}

_inline void arm_mvn()
{	/*Rd := 0xFFFFFFFF EOR Op2 <Data Processing> (1̕␔)*/
	u32	opcode, operand;
	u32	rd;

	opcode	= OPCODE;
	rd		= DEREG;
	
	if(IMM_OPERAND){	/*Immediate Operand*/
		operand = imm_rotate(opcode);
	}else{
		operand = imm_shift(opcode);
	}

	arm.reg[rd] = 0xFFFFFFFF ^ operand;

	SetDpFlags(arm.reg[rd]);
	
	arm.cycle = 1;
}

_inline void arm_orr()
{	/*Rd := Rn OR Op2 <Data Processing> (_a)*/
	u32	opcode, operand;
	u32	rd, rn;

	opcode	= OPCODE;
	rd		= DEREG;
	rn		= FOREG;

	if(IMM_OPERAND){	/*Immediate Operand*/
		operand = imm_rotate(opcode);
	}else{
		operand = imm_shift(opcode);
	}

	arm.reg[rd] = arm.reg[rn] | operand;

	SetDpFlags(arm.reg[rd]);

	arm.cycle = 1;
}

_inline void arm_rsb()
{	/*Rd := Op2 - Rn <Data Processing> (tZ)*/
	u32	opcode, operand;
	u32	rd, rn;

	opcode	= OPCODE;
	rd		= DEREG;
	rn		= FOREG;

	if(IMM_OPERAND){
		operand = imm_rotate(opcode);
	}else{
		operand = imm_shift(opcode);
	}

	arm.reg[rd] = operand - arm.reg[rn];

	SetSubFlag(arm.reg[rn], operand, arm.reg[rd]);

	arm.cycle = 1;
}

_inline void arm_rsc()
{	/*Rd := Op2 - Rn - 1 + Carry <Data Processing> (L[ttZ)*/
	u32	opcode, operand;
	u32	rd, rn;

	opcode	= OPCODE;
	rd		= DEREG;
	rn		= FOREG;
	
	if(IMM_OPERAND){
		operand = imm_rotate(opcode);
	}else{
		operand = imm_shift(opcode);
	}

	operand +=  - 1 + LSB_FC;

	arm.reg[rd] = operand - arm.reg[rn];

	SetSubFlag(arm.reg[rn], operand, arm.reg[rd]);

	arm.cycle = 1;
}

_inline void arm_sbc()
{	/*Rd := Rn - Op2 - 1 + Carry <Data Processing> (L[tZ)*/
	u32	opcode, operand;
	u32	rd, rn;

	opcode	= OPCODE;
	rd		= DEREG;
	rn		= FOREG;

	if(IMM_OPERAND){
		operand = imm_rotate(opcode);
	}else{
		operand = imm_shift(opcode);
	}

	operand += - 1 + LSB_FC;

	arm.reg[rd] = arm.reg[rn] - operand;

	SetSubFlag(arm.reg[rn], operand, arm.reg[rd]);

	arm.cycle = 1;
}

_inline void arm_stc()
{	/*address := CRn (RvZbTWX^̓e֊i[)*/
	u32	opcode;
	u32	rd, rn, cp;
	u8	offset;

	opcode	= OPCODE;
	rd		= (opcode>>12) & 0x0F;
	rn		= (opcode>>16) & 0x0F;
	cp		= (opcode) & 0x0F;
	offset	= (u8)(opcode);

	arm.cycle = 3;
}

_inline void arm_stm()
{	/*Stack manipulation (̃WX^֊i[,Push)*/
	u32	opcode,	address, bit;
	u32	rn;		/*base register*/
	u32	lst;	/*register list*/
	u32	fprepost;

	arm.cycle = 2;

	opcode	= OPCODE;
	rn		= (opcode>>16)&0x0F;	/*16-19 4bit*/

	/*base address*/
	address = arm.reg[rn];

	fprepost = (opcode & BIT_24_)?TRUE:FALSE;

	for(bit=1,lst=0; bit!=BIT_16_; bit<<=1,lst++){
		if(opcode & bit){
			arm.cycle++;
			if(fprepost){	/*Pre: ItZbg̒ǉOɓ]*/
				if(opcode & BIT_23_){	/*Up: ItZbgx[XɉZ*/
					address += 4;
				}else{					/*Down: ItZbgx[X猸Z*/
					address -= 4;
				}
			}
			agb_write_mem32(address, arm.reg[lst]);
			if(!fprepost){	/*Post: ItZbgǉɓ]*/
				if(opcode & BIT_23_){	/*Up: ItZbgx[XɉZ*/
					address += 4;
				}else{					/*Down: ItZbgx[X猸Z*/
					address -= 4;
				}
			}
		}
	}

	if((opcode & BIT_21_) || !fprepost){	/*Write-back sp(rn)̓eXV*/
		arm.reg[rn] = address;
	}
}

_inline void arm_str()
{	/*<address> := Rd, [rn] (WX^֊i[) lItZbg*/
	u32	opcode,offset, address;
	u32	rd, rn;
	u32	fprepost;

	opcode	= OPCODE;
	rd		= DEREG;
	rn		= (opcode>>16) & 0xF;	/*AhX̃x[XWX^*/

	address = arm.reg[rn];

	if(rn==15)address += 8;	/*ΏۂPC̏ꍇ*/

	fprepost = (opcode & BIT_24_)?TRUE:FALSE;

	if(opcode & BIT_25_){	/*1 = ItZbg̓WX^*/
		offset = imm_shift(opcode);
	}else{					/*0 = ItZbg͑l*/
		offset = opcode & 0xFFF;
	}

	if(fprepost){	/*1 = Pre: ItZbg̒ǉOɓ]*/
		if(opcode & BIT_23_){	/*1 = Up: ItZbgx[XɉZ*/
			address += offset;
		}else{					/*0 = Down: ItZbgx[X猸Z*/
			address -= offset;
		}
	}

	if(opcode & BIT_22_){	/*Byte ]*/
		agb_write_mem8(address, arm.reg[rd]);
		//agb_write_mem8(address&~1, arm.reg[rd]);
	}else{					/*Double word ]*/
		agb_write_mem32(address, arm.reg[rd]);
	}

	if(!fprepost){	/*0 = Post: ItZbg̒ǉɓ]*/
		if(opcode & BIT_23_){	/*1 = Up: ItZbgx[XɉZ*/
			address += offset;
		}else{					/*0 = Down: ItZbgx[X猸Z*/
			address -= offset;
		}
	}

	if((opcode & BIT_21_) || !fprepost){	/*Write-back: ݐ̃AhXcĂ*/
		arm.reg[rn] = address;
	}

	arm.cycle = 2;
}

_inline void arm_strs()
{	/*<address> := Rd (WX^֊i[) Register offset*/
	u32	opcode, address;
	u32	rd, rn, rm;
	u32	fprepost;

	opcode	= OPCODE;
	rd		= DEREG;					/*ΏۃWX^*/
	rn		= (opcode>>16) & 0xF;	/*AhX̃x[XWX^*/
	rm		= opcode & 0xF;

	address = arm.reg[rn];

	fprepost = (opcode & BIT_24_)?TRUE:FALSE;

	if(fprepost){	/*1 = Pre: ]OɃItZbgǉ*/
		if(opcode & BIT_23_){	/*Up: ItZbgx[XɉZ*/
			address += arm.reg[rm];
		}else{					/*Down: ItZbgx[X猸Z*/
			address -= arm.reg[rm];
		}
	}

	if(opcode & BIT_5_){	/*Halfword*/
		if(opcode & BIT_6_){	/*Sign*/
			/*11: Signed Halfwords*/
			agb_write_mem16(address, arm.reg[rd]);
		}else{
			/*01: Unsigned Halfwords*/
			agb_write_mem16(address, arm.reg[rd]);
		}
	}else{
		if(opcode & BIT_6_){	/*Sign*/
			/*10: Signed byte*/
			agb_write_mem8(address, arm.reg[rd]);
		}else{
			/*00: SWP instruction*/
		}
	}

	if(!fprepost){	/*0 = Post: ]ɃItZbgǉ*/
		if(opcode & BIT_23_){	/*Up: ItZbgx[XɉZ*/
			address += arm.reg[rm];
		}else{					/*Down: ItZbgx[X猸Z*/
			address -= arm.reg[rm];
		}
	}

	if((opcode & BIT_21_) || !fprepost){	/*Write-back: ݐ̃AhXcĂ*/
		arm.reg[rn] = address;
	}

	arm.cycle = 3;
}

_inline void arm_strs_imm()
{	/*<address> := Rd (WX^֊i[) Immidiate offset*/
	u32	opcode,	offset, address;
	u32	rd, rn;
	u32	fprepost;

	opcode	= OPCODE;
	rd		= DEREG;					/*ΏۃWX^*/
	rn		= (opcode>>16) & 0xF;	/*AhX̃x[XWX^*/
	offset	= (opcode & 0xF) | ((opcode>>4) & 0xF0);

	address = arm.reg[rn];

	fprepost = (opcode & BIT_24_)?TRUE:FALSE;

	if(fprepost){	/*1 = Pre: ]OɃItZbgǉ*/
		if(opcode & BIT_23_){	/*Up: ItZbgx[XɉZ*/
			address += offset;
		}else{					/*Down: ItZbgx[X猸Z*/
			address -= offset;
		}
	}

	if(opcode & BIT_5_){	/*Halfwords*/
		if(opcode & BIT_6_){	/*Sign*/
			/*Signed Halfwords*/
			agb_write_mem16(address, arm.reg[rd]);
		}else{
			/*Unsigned Halfwords*/
			agb_write_mem16(address, arm.reg[rd]);
		}
	}else{
		if(opcode & BIT_6_){	/*Sign*/
			/*Signed byte*/
			agb_write_mem8(address, arm.reg[rd]);
		}else{
			/*SWP instruction*/
		}
	}

	if(!fprepost){	/*0 = Post: ]ɃItZbgǉ*/
		if(opcode & BIT_23_){	/*Up: ItZbgx[XɉZ*/
			address += offset;
		}else{					/*Down: ItZbgx[X猸Z*/
			address -= offset;
		}
	}

	if((opcode & BIT_21_) || !fprepost){	/*Write-back: ݐ̃AhXcĂ*/
		arm.reg[rn] = address;
	}

	arm.cycle = 3;
}

_inline void arm_sub()
{	/*Rd := Rn - Op2 <Data Processing> (Z)*/
	u32	opcode, operand;
	u32	rd, rn;

	opcode	= OPCODE;
	rd		= DEREG;
	rn		= FOREG;

	if(IMM_OPERAND){
		operand = imm_rotate(opcode);
	}else{
		operand = imm_shift(opcode);
	}

	/*if((arm.reg[rn] - operand) > 0)SF(C_); else RF(C_);*/

	arm.reg[rd] = arm.reg[rn] - operand;

	SetSubFlag(arm.reg[rn], operand, arm.reg[rd]);

	arm.cycle = 1;
}

_inline void arm_swi()
{	/*OS call (\tgEFA荞)*/
	u32	opcode, comment;

	opcode	= OPCODE;
	comment	= opcode & 0x00FFFFFF;

	switch((u8)comment){
		case 0x00:bios_soft_reset();break;
		case 0x01:bios_register_ram_reset();break;
		case 0x02:bios_halt();break;
		case 0x03:bios_stop();break;
		case 0x04:bios_intr_wait();break;
		case 0x05:bios_vblank_intr_wait();break;
		case 0x06:bios_div();break;
		case 0x07:bios_div_arm();break;
		case 0x08:bios_sqrt();break;
		case 0x09:bios_arc_tan();break;
		case 0x0A:bios_arc_tan2();break;
		default:console_print("Undefined BIOS Call.");
	}

	arm.cycle = 3;
}

_inline void arm_swp()
{	/*Rd := [Rn], [Rn] := Rm (Pf[^̌)*/
	u32	opcode;
	u32	rd, rn, rm;

	opcode	= OPCODE;
	rd		= (opcode>>12) & 0x0F;
	rn		= (opcode>>16) & 0x0F;
	rm		= (opcode) & 0x0F;

	arm.reg[rd] = arm.reg[rn];
	arm.reg[rn] = arm.reg[rm];

	arm.cycle = 4;
}

_inline void arm_teq()
{	/*CPSR flags := Rn EOR Op2 <Data Processing> (rbgp^[̔r)*/
	u32	opcode,	operand, temp;
	u32	rn;

	opcode	= OPCODE;
	rn		= FOREG;

	if(IMM_OPERAND){
		operand = imm_rotate(opcode);
	}else{
		operand = imm_shift(opcode);
	}

	/*CPSR = arm.reg[rn] ^ operand;*/
	temp = arm.reg[rn] ^ operand;

	SetDpFlags(temp);

	arm.cycle = 1;
}

_inline void arm_tst()
{	/*CPSR flags := Rn AND Op2 <Data Processing> (w肵rbg̃eXg)*/
	u32	opcode,	operand, temp;
	u32	rn;

	opcode	= OPCODE;
	rn		= FOREG;

	if(IMM_OPERAND){
		operand = imm_rotate(opcode);
	}else{
		operand = imm_shift(opcode);
	}

	/*CPSR = arm.reg[rn] & operand;*/
	temp = arm.reg[rn] & operand;

	SetDpFlags(temp);

	arm.cycle = 1;
}

void CMainFrame::exec_arm_clk(s32 clk)
{
	do{
		if(CPSR & T_){
			//exec_thumb_state();
		}else{
			exec_arm_state();
		}
		clk -= arm.cycle;
	}while(clk > 0);
}

int CMainFrame::exec_arm_state()
{
	u32	opcode, cond_enable;
	u8	opcode_24_4, cond;
	char message[64];

	opcode		= OPCODE;
	opcode_24_4	= (u8)((opcode>>24) & 0x0F);	/*24-27 4bit*/
	cond		= GETCOND;
	cond_enable	= FALSE;

	switch(cond){	/*s - Condition Field*/
	case 0x0:
		if(FZ)cond_enable = TRUE;
		break;		/*if Z set		- equal(==)*/
	case 0x1:
		if(!FZ)cond_enable = TRUE;
		break;		/*if Z clear	- not equal(!=)*/
	case 0x2:
		if(FC)cond_enable = TRUE;
		break;		/*if C set		- unsigned higher or same(>=)*/
	case 0x3:
		if(!FC)cond_enable = TRUE;
		break;		/*if C clear	- unsigne lower(<)*/
	case 0x4:
		if(FN)cond_enable = TRUE;
		break;		/*if N set		- negative(<0)*/
	case 0x5:
		if(!FN)cond_enable = TRUE;
		break;		/*if N clear	- positive or zero(>=0)*/
	case 0x6:
		if(FV)cond_enable = TRUE;
		break;		/*if V set		- overflow(>0xFFFFFFFF)*/
	case 0x7:
		if(!FV)cond_enable = TRUE;
		break;		/*if V clear	- no overflow<=0xFFFFFFFF*/
	case 0x8:
		if(FC && !FZ)cond_enable = TRUE;
		break;		/*if C set and Z clear	- unsigned higher(>)*/
	case 0x9:
		if(!FC || FZ)cond_enable = TRUE;
		break;		/*if C clear and Z set	- unsigned lower or same(<=)*/
	case 0xA:
		if((FN && FV) || (!FN && !FV))cond_enable = TRUE;
		break;		/*if ((N set and V set) or (N clear and V clear))	- greater or equal*/
	case 0xB:
		if((FN && !FV) || (!FN && FV))cond_enable = TRUE;
		break;		/*if ((N set and V clear) or (N clear and V set))	- less than*/
	case 0xC:
		if((!FZ && (FN || FV)) || (!FZ && (!FN || !FV)))cond_enable = TRUE;
		break;		/*if (Z clear and(N or V set) or (N or V clear))	- greater than*/
	case 0xD:
		if((FZ || (FN && !FV)) || (FZ || (!FN && FV)))cond_enable = TRUE;
		break;		/*if (Z set or(N set and V clear) or (N clear and V set))- less than or equal*/
	default:
		cond_enable = TRUE;
		break;		/*(ignored)	- always */
	}

	if(cond_enable){
		switch(opcode_24_4){
		case 0x0:	/*0000*/
			if(((opcode>>4)&0x0F)==0x09){
				if(opcode & BIT_23_){	/*Multiply Long*/
					if(opcode & BIT_21_){	/*Accumelate*/
						arm_mlal();
						break;
					}else{
						arm_mull();
						break;
					}
				}else{					/*Multiply*/
					if(opcode & BIT_21_){	/*Accumelate*/
						arm_mla();
						break;
					}else{
						arm_mul();
						break;
					}
				}
			}
		case 0x1:	/*0001*/
			if(((opcode&0xFF)>>4)==0x09){	/*00001001*/
				arm_swp();	/*Single Data Swap*/
				break;
			}
			if((opcode&0x0FFFFFF0)==0x012FFF10){
				arm_bx();	/*Branch and Exchange*/
				break;
			}
			if(opcode & BIT_4_){	/*Halfword data Transfer:*/
				if(opcode & BIT_22_){	/*immdiate offset*/
					if(opcode & BIT_20_){	/*Load from memory*/
						arm_ldrs_imm();
					}else{					/*Store to memory*/
						arm_strs_imm();
					}
					break;
				}else{					/*register offset*/
					if(opcode & BIT_20_){	/*Load from memory*/
						arm_ldrs();
					}else{					/*Store to memory*/
						arm_strs();
					}
					break;
				}
			}
		case 0x2:	/*0010*/
		case 0x3:	/*0011*/
		/*Data Processing*/
			switch((u8)((opcode>>21)&0x0F)){
			case 0x0: arm_and();break;
			case 0x1: arm_eor();break;
			case 0x2: arm_sub();break;
			case 0x3: arm_rsb();break;
			case 0x4: arm_add();break;
			case 0x5: arm_adc();break;
			case 0x6: arm_sbc();break;
			case 0x7: arm_rsc();break;
			case 0x8: arm_tst();break;
			case 0x9: arm_teq();break;
			case 0xA: arm_cmp();break;
			case 0xB: arm_cmn();break;
			case 0xC: arm_orr();break;
			case 0xD: arm_mov();break;
			case 0xE: arm_bic();break;
			case 0xF: arm_mvn();break;
			}
			break;
		/*Single Data Transfer*/
		case 0x4:	/*0100*/
		case 0x5:	/*0101*/
		case 0x6:	/*0110*/
		case 0x7:	/*0111*/
			if(opcode & BIT_20_){	/*Load from memory*/
				arm_ldr();
				break;
			}else{					/*Store to memory*/
				arm_str();
				break;
			}
			if(opcode & BIT_4_){
		/*`R[h*/
				sprintf(message, "Undefine ARM opcode \"0x%08X\" at 0x%08X", opcode, PC);
				/*MessageBox(message, NULL, MB_ICONSTOP);
				CPUIsRunning = FALSE;*/
				console_print(message);
				SetCpuMode(ARM_MODE_UND);/*UND[h*/
				break;
			}
		/*Block Data Transfer*/
		case 0x8:	/*1000*/
		case 0x9:	/*1001*/
			if(opcode & BIT_20_){	/*Load from */
				arm_ldm();
			}else{
				arm_stm();
			}
			break;
		/*Branch*/
		case 0xA:	/*1010*/
			arm_b();
			break;
		case 0xB:	/*1011*/
			arm_bl();	/*branch with Link*/
			break;
		/*Coprocesser Data Transfer*/
		case 0xC:	/*1100*/
		case 0xD:	/*1101*/
			if(opcode & BIT_20_){	/*Load from */
				arm_ldc();
			}else{
				arm_stc();
			}
			break;
		case 0xE:	/*1110*/
			if(opcode & BIT_4_){
		/*Coprocesser Resgister Transfer*/
				if(opcode & BIT_20_){	/*Load from Co-Processer*/
					arm_mrc();
				}else{					/*Store to Co-Processer*/
					arm_mcr();
				}
			}else{
		/*Coprocesser Data Operation*/
				arm_cdp();
			}
			break;
		/*Software interrupt*/
		case 0xF:	/*1111*/
			arm_swi();
			break;
		default:
			sprintf(message, "Unknow ARM opcode \"0x%08X\" at 0x%08X", opcode, PC);
			MessageBox(message, NULL, MB_ICONSTOP);
			console_print(message);
			CPUIsRunning = FALSE;
			break;
		}
	}
	PC += 4;

	return 0;
}

