#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
#  phrasemanager.py - new phrase manager
#  Copyright (C) 2005 by Atzm WATANABE <sitosito@p.chan.ne.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#
# $Id: phrasemanager.py,v 1.9 2005/09/21 15:29:20 atzm Exp $
#

__version__ = '$Revision: 1.9 $'
__author__ = '$Author: atzm $'

import os, sys
import gtk, gobject

try:
	from cStringIO import StringIO
except ImportError:
	from StringIO import StringIO

from common import APP, open_bottlecase, open_error_dialog
import config

from viewer.ghostmanager import OrderedDictionary

from xml.sax import saxexts, saxutils
from xml.sax.handler import ContentHandler
from xml.parsers import expat

class PhraseParser:
	def __init__(self, func=lambda x, y: None):
		self._func = func
		self.clear()

	def clear(self):
		self.parser = expat.ParserCreate()
		self.parser.StartElementHandler = self.startElement
		self.parser.EndElementHandler = self.endElement

		self._xml = []
		self._actions = []

		self._tree = OrderedDictionary()
		self.__parent = []

	def parse(self, content):
		self.parser.Parse(content)

	def get_names(self):
		return list(zip(*self._actions)[0])

	def get_accels(self):
		return list(zip(*self._actions)[3])

	def get_actions(self):
		return self._actions[:]

	def get_xml(self):
		return '\n'.join(self._xml)

	def get_tree(self):
		return self._tree.copy()

	def _append_tree(self, name, attrs):
		if name == 'menubar':
			self._tree[attrs['name']] = OrderedDictionary()
			self.__parent = [attrs['name']]
		elif name == 'menu':
			parent = self._tree
			for i in self.__parent:
				parent = parent[i]
			parent[attrs['name']] = OrderedDictionary()
			self.__parent.append(attrs['name'])
		elif name == 'menuitem':
			parent = self._tree
			for i in self.__parent:
				parent = parent[i]

			try: v1 = attrs['name']
			except KeyError: v1 = ''
			try: v2 = attrs['insert']
			except KeyError: v2 = ''
			try: v3 = attrs['accel']
			except KeyError: v3 = ''

			parent[attrs['name']] = (v1, v2, v3)
			self.__parent.append(attrs['name'])

	def startElement(self, name, attrs):
		self._append_tree(name, attrs)

		keys = attrs.keys()

		if 'name' in keys:
			keys.append('action')
			attrs['action'] = attrs['name']

			action = [attrs['name'], '', attrs['name']]

			if 'accel' in keys:
				action.append(attrs['accel'])
			else:
				action.append('')

			action.append(attrs['name'])

			if 'insert' in keys:
				action.append(lambda x: self._func(x, attrs['insert'][:]))

			self._actions.append(tuple(action))

		try: keys.remove('insert')
		except ValueError: pass
		try: keys.remove('accel')
		except ValueError: pass

		xml = ['%s="%s"' % (k, attrs[k]) for k in keys]
		xml = '<%s %s>' % (name, ' '.join(xml))
		self._xml.append(xml)

	def endElement(self, name):
		self.__parent.pop()
		self._xml.append('</%s>' % name)

class PhraseManager:
	DEFAULT_PHRASE = r'''
<menubar name="PhraseBar">
	<menu name="セッション(_S)">
		<menuitem name="タイムクリティカル" insert="\t" accel="&lt;control&gt;&lt;alt&gt;t"/>
		<menuitem name="クイック" insert="\_q" accel="&lt;control&gt;&lt;alt&gt;q"/>
		<menuitem name="シンクロナイズド" insert="\_s" accel="&lt;control&gt;&lt;alt&gt;s"/>
		<menuitem name="さくらセッション" insert="\h" accel="&lt;control&gt;&lt;alt&gt;h"/>
		<menuitem name="うにゅうセッション" insert="\u" accel="&lt;control&gt;&lt;alt&gt;u"/>
	</menu>
	<menu name="さくら">
		<menuitem name="[0] 素" insert="\s[0]" accel="&lt;control&gt;0"/>
		<menuitem name="[1] 照れ" insert="\s[1]" accel="&lt;control&gt;1"/>
		<menuitem name="[2] 驚き" insert="\s[2]" accel="&lt;control&gt;2"/>
		<menuitem name="[3] 不安" insert="\s[3]" accel="&lt;control&gt;3"/>
		<menuitem name="[4] 落込み" insert="\s[4]" accel="&lt;control&gt;4"/>
		<menuitem name="[5] 微笑み" insert="\s[5]" accel="&lt;control&gt;5"/>
		<menuitem name="[6] 目閉じ" insert="\s[6]" accel="&lt;control&gt;6"/>
		<menuitem name="[7] 怒り" insert="\s[7]" accel="&lt;control&gt;7"/>
		<menuitem name="[8] 冷笑" insert="\s[8]" accel="&lt;control&gt;8"/>
	</menu>
	<menu name="うにゅう">
		<menuitem name="[10] 素" insert="\s[10]"/>
		<menuitem name="[11] 刮目" insert="\s[11]"/>
	</menu>
	<menu name="その他">
		<menuitem name="ウェイト(1.8秒)" insert="\w9\w9" accel="&lt;control&gt;w"/>
		<menuitem name="改行" insert="\n"/>
		<menuitem name="クリア" insert="\c"/>
		<menuitem name="URL" insert="\URL[http://]"/>
		<menuitem name="な〜。" insert="\t\u\s[10]\h\s[5]\_sな〜。\e"/>
	</menu>
</menubar>
	'''
	def __init__(self, filename=os.path.join(open_bottlecase(), 'phrasebar.xml'),
				 insert_func=lambda x, y: sys.stdout.write('%s %s\n' % (x, y))):
		self.__filename = filename
		self.__insert_func = insert_func
		self.__phrase_parser = PhraseParser(func=self.__insert_func)
		self.parse()

	def clear(self):
		self.__phrase_parser.clear()

	def parse(self, xmlstring=None):
		if xmlstring is None:
			if os.path.isfile(self.__filename):
				xmlstring = file(self.__filename).read().strip()
			else:
				xmlstring = self.DEFAULT_PHRASE
				file(self.__filename, 'w').write(xmlstring)
		xmlstring = unicode(xmlstring, 'utf-8')
		self.__phrase_parser.clear()
		self.__phrase_parser.parse(xmlstring)

	def get_actions(self):
		return self.__phrase_parser.get_actions()

	def get_names(self):
		return self.__phrase_parser.get_names()

	def get_accels(self):
		return self.__phrase_parser.get_accels()

	def get_tree(self):
		return self.__phrase_parser.get_tree()

	def get_xml(self):
		return self.__phrase_parser.get_xml()

	def get_xml_as_fileobj(self):
		return StringIO(self.__phrase_parser.get_xml())

	def put_xml(self, xmlstring):
		file(self.__filename, 'w').write(xmlstring)
		self.parse(xmlstring)

	def create_treestore(self):
		def _create_treestore(dic, treestore, titer=None, level=0):
			for k, v in dic.items():
				titer2 = treestore.append(titer, [k, '', ''])
				#print '%s%s' % ('  ' * level, k)
				if isinstance(v, dict):
					_create_treestore(v, treestore, titer2, level+1)
				else:
					treestore.remove(titer2)
					treestore.append(titer, v)
					#print '%s%s' % ('  ' * (level+1), v)

		treestore = gtk.TreeStore(*[gobject.TYPE_STRING for i in range(3)])
		_create_treestore(self.get_tree(), treestore)
		return treestore

	def create_menues(self, accel_group):
		def _create_menues(dic, menues, parent=None, level=0):
			for k, v in dic.items():
				menuitem = gtk.MenuItem(k)

				if isinstance(v, dict):
					menu = gtk.Menu()
					_create_menues(v, menues, menu, level+1)
					menuitem.set_submenu(menu)
				else:
					if v[1]:
						menuitem.connect('activate', self.__insert_func, v[1])
					if v[2]:
						key, mods = gtk.accelerator_parse(v[2])
						menuitem.add_accelerator('activate', accel_group, key, mods, gtk.ACCEL_VISIBLE)

				if parent is None:
					menues.append(menuitem)
				else:
					parent.append(menuitem)
			return menues

		parent = _create_menues(self.get_tree(), [])[0].get_submenu()
		if parent is not None:
			for item in parent.get_children():
				parent.remove(item)
				yield item

class PhraseBar(gtk.MenuBar):
	def __init__(self, phrase_manager):
		gtk.MenuBar.__init__(self)
		self.__phrase_manager = phrase_manager
		self.__accel_group = gtk.AccelGroup()

		self.__set_prev_actions()
		self.reload_ui()

	def reload_ui(self):
		def _remove_menues(menu):
			for menuitem in menu.get_children():
				index = self.__prev_actions[0].index(menuitem.get_child().get_label())
				accelstr = self.__prev_actions[1][index]
				if accelstr:
					menuitem.remove_accelerator(self.__accel_group, *gtk.accelerator_parse(accelstr))

				submenu = menuitem.get_submenu()
				if submenu is not None:
					_remove_menues(submenu)
				menu.remove(menuitem)

		_remove_menues(self)
		self.__set_prev_actions()

		for menuitem in self.__phrase_manager.create_menues(self.__accel_group):
			self.append(menuitem)
		self.show_all()

	def get_accel_group(self):
		return self.__accel_group

	def __set_prev_actions(self):
		self.__prev_actions = [
			self.__phrase_manager.get_names(),
			self.__phrase_manager.get_accels(),
			]

class PhraseEditor(gtk.Window):
	MENU_STRING = '''
<toolbar name="PhraseEditor">
    <toolitem name="New" action="New"/>
    <toolitem name="Append" action="Append"/>
    <toolitem name="Delete" action="Delete"/>
    <toolitem name="Down" action="Down"/>
</toolbar>
	'''

	def __init__(self, phrase_manager, callback_edited=None):
		self.phrase_manager = phrase_manager
		self.callback_edited = callback_edited

		treestore = self.phrase_manager.create_treestore()
		self.treeview = gtk.TreeView(treestore)
		self.treeview.set_rules_hint(True)
		self.treeview.set_search_column(True)
		self.treeview.set_enable_search(True)
		self.treeview.connect('cursor-changed', self.selected)
		self.treeview.connect('row-expanded', lambda *args: self.treeview.columns_autosize())
		self.treeview.connect('row-collapsed', lambda *args: self.treeview.columns_autosize())
		#self.treeview.connect('key-press-event', self.keypress)
		cols = [gtk.TreeViewColumn(_('Name')),
				gtk.TreeViewColumn(_('Insert string')),
				gtk.TreeViewColumn(_('Key accelerator'))]
		for i in range(len(cols)):
			cell = gtk.CellRendererText()
			cell.set_property('editable', True)
			cell.connect('edited', self.cell_edited, treestore, i)
			if i == 2:
				cell.connect('editing-started', self.cell_editing_started)
			cols[i].pack_start(cell, True)
			cols[i].add_attribute(cell, 'text', i)
			cols[i].set_resizable(True)
			self.treeview.append_column(cols[i])
		sw = gtk.ScrolledWindow()
		sw.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
		sw.set_shadow_type(gtk.SHADOW_IN)
		sw.add(self.treeview)

		self._uimanager  = gtk.UIManager()
		self._accelgroup = self._uimanager.get_accel_group()
		self._actiongroup = gtk.ActionGroup('PhraseEditor')
		self._actiongroup.add_actions([
			('PhraseEditor', None, _('PhraseEditor')),
			('New', gtk.STOCK_FILE, _('_New'), None, _('Insert new item'), self.new),
			('Append', gtk.STOCK_DIRECTORY, _('_Append'), None, _('Append submenu to item'), self.append),
			('Delete', gtk.STOCK_DELETE, _('_Delete'), None, _('Delete selected item'), self.delete),
            ('Down', gtk.STOCK_GO_DOWN, _('Down'), None, _('Go down selected item'), self.go_down),
			])
		self._uimanager.insert_action_group(self._actiongroup, 0)
		self.ui_merge_id = self._uimanager.add_ui_from_string(self.MENU_STRING)
		toolbar = self._uimanager.get_widget('/PhraseEditor')

		vbox = gtk.VBox()
		vbox.pack_start(toolbar, False, False)
		vbox.pack_start(sw)

		gtk.Window.__init__(self)
		self.set_title('%s: %s' % (APP, _('Phrase editor')))
		self.connect('delete-event', self.close_window)
		self.add(vbox)
		self.add_accel_group(self._accelgroup)
		self.set_default_size(config.get('phraseeditor', 'width', 'int'),
							  config.get('phraseeditor', 'height', 'int'))

	def go_down(self, widget=None, *data):
		model = self.treeview.get_model()

		titer = self.get_selected_iter()
		if titer is None: return True
		titer2 = model.iter_next(titer)
		if titer2 is None: return True

		model.swap(titer, titer2)
		self.update_notify()

	def new(self, widget=None, *data):
		titer = self.get_selected_iter()
		if titer is None: return True
		model = self.treeview.get_model()

		titer2 = model.iter_parent(titer)
		if titer2 is None: return True
		self._append(titer2)

	def append(self, widget=None, *data):
		titer = self.get_selected_iter()
		if titer is None: return True
		model = self.treeview.get_model()
		if model.get_value(titer, 1) or model.get_value(titer, 2):
			return True
		self._append(titer)

	def _append(self, titer):
		model = self.treeview.get_model()
		titer2 = model.append(titer, [_('New item') + '#%d' % titer.__hash__(), '', ''])
		self.treeview.expand_row(model.get_path(titer), False)
		self.treeview.set_cursor(model.get_path(titer2))
		self.update_notify()

	def delete(self, widget=None, data=None):
		titer = self.get_selected_iter()
		if titer is None: return True
		model = self.treeview.get_model()
		model.remove(titer)
		self.update_notify()

	def selected(self, widget=None, *data):
		titer = self.get_selected_iter()
		if titer is None:
			return True
		model = self.treeview.get_model()
		root = model.get_iter_root()
		if model.get_path(root) == model.get_path(titer):
			self._uimanager.get_widget('/PhraseEditor/New').set_sensitive(False)
			self._uimanager.get_widget('/PhraseEditor/Delete').set_sensitive(False)
		else:
			self._uimanager.get_widget('/PhraseEditor/New').set_sensitive(True)
			self._uimanager.get_widget('/PhraseEditor/Delete').set_sensitive(True)

		if model.get_value(titer, 1) or model.get_value(titer, 2):
			self._uimanager.get_widget('/PhraseEditor/Append').set_sensitive(False)
		else:
			self._uimanager.get_widget('/PhraseEditor/Append').set_sensitive(True)

	def get_selected_iter(self):
		return self.treeview.get_selection().get_selected()[1]

	#def keypress(self, widget, event=None):
	#	if type(widget) is gtk.TreeView:
	#		if event.keyval == 65535: # delete key
	#			self.delete()

	def cell_editing_started(self, widget, editable, path):
		if path == '0':
			return True
		treestore = self.treeview.get_model()
		if treestore.iter_has_child(treestore.get_iter(path)):
			return True

		def callback(keyaccel):
			editable.set_text(keyaccel)

		pkawindow = PhraseKeyAccelWindow(_('Please press any key to grab...'), self, callback)
		pkawindow.show_all()

	def cell_edited(self, cell, path, new_text, treestore, column):
		if path == '0':
			if treestore[path][column] != new_text or (column != 0 and new_text):
				open_error_dialog(_("Can't modify root menu."), self)
			return True
		if treestore.iter_has_child(treestore.get_iter(path)) and column != 0:
			if new_text:
				open_error_dialog(_("Can't add 'insert string' or 'key accelerator' to menu."), self)
			return True
		if column == 2:
			k, m = gtk.accelerator_parse(new_text)
			if k == 0 and m == 0 and new_text:
				open_error_dialog(_('Invalid key accelerator: %s') % new_text, self)
				return True
		if column == 0:
			try:
				self.phrase_manager.get_names().index(new_text)
			except ValueError:
				pass
			else:
				if treestore[path][column] != new_text:
					open_error_dialog(_('Item named %s already exists.') % new_text, self)
				return True

		treestore[path][column] = new_text
		self.update_notify()

	def update_notify(self):
		xmlstr = unicode(self.create_xml()).encode('utf-8')
		self.phrase_manager.put_xml(xmlstr)
		if callable(self.callback_edited):
			self.callback_edited()
		self.selected()
		self.treeview.columns_autosize()
		self.treeview.grab_focus()

	def open_window(self, widget=None, *args):
		self.show_all()
		self.treeview.grab_focus()
		return True

	def close_window(self, widget=None, *args):
		self.hide_all()
		return True

	def create_xml(self):
		def _create_xml(model, titer, xmlstr, level=0):
			while True:
				name, insert, accel = [model.get_value(titer, i) for i in range(3)]
				nest = '  ' * level
				if model.iter_has_child(titer):
					if name == 'PhraseBar':
						tagname = 'menubar'
					else:
						tagname = 'menu'
					xmlstr.append('%s<%s name=%s>' % (nest, tagname, saxutils.quoteattr(name)))
					_create_xml(model, model.iter_children(titer), xmlstr, level+1)
					xmlstr.append('%s</%s>' % (nest, tagname))
				else:
					if insert: insert = ' insert=%s' % saxutils.quoteattr(insert)
					else: insert = ''
					if accel: accel = ' accel=%s' % saxutils.quoteattr(accel)
					else: accel = ''
					xmlstr.append('%s<menuitem name=%s%s%s/>' % \
								  (nest, saxutils.quoteattr(name), insert, accel))
				titer = model.iter_next(titer)
				if titer is None:
					break
			return xmlstr

		xmlstr = []
		model = self.treeview.get_model()
		titer = model.get_iter_root()

		if titer is None:
			xmlstr.append('<menubar name="PhraseBar">')
			xmlstr.append('</menubar>')
		else:
			xmlstr = _create_xml(model, titer, xmlstr)

		return '\n'.join(xmlstr)

class PhraseKeyAccelWindow(gtk.Window):
	def __init__(self, title, parent, callback):
		self.callback = callback
		self.__string = ''

		label = gtk.Label(''.join([
			'<span weight="bold" size="large">',
			_('Please press any key to grab...'),
			'</span>',
			]))
		label.set_use_markup(True)

		button = gtk.Button(_('Cancel'))
		button.connect('clicked', lambda x: self.destroy())

		vbox = gtk.VBox()
		vbox.set_border_width(5)
		vbox.pack_start(label, True, True, 2)
		vbox.pack_start(button, True, True, 4)

		gtk.Window.__init__(self, gtk.WINDOW_POPUP)
		self.set_title(title)
		self.set_transient_for(parent)
		self.set_position(gtk.WIN_POS_CENTER_ON_PARENT)
		self.set_destroy_with_parent(True)
		self.set_modal(True)
		self.set_default_size(300, 50)
		self.add(vbox)
		self.connect('key-press-event', self.keypress)
		self.connect('key-release-event', self.keyrelease)

	def keypress(self, widget, event):
		self.__string = gtk.accelerator_name(event.keyval, event.state)

	def keyrelease(self, widget, event):
		self.callback(self.__string)
		self.destroy()

if __name__ == '__main__':
	import gettext
	gettext.install('gbottler')

	manager = PhraseManager()
	p = PhraseBar(manager)

	w = gtk.Window()
	w.connect('destroy', lambda x: gtk.main_quit())
	w.add_accel_group(p.get_accel_group())
	w.add(p)
	w.show_all()

	pe = PhraseEditor(manager, p.reload_ui)
	pe.open_window()

	gtk.main()
