# -*- coding: ascii -*-
#
#  preferences.py - New preference manipulator for GBottler
#  Copyright (C) 2005 by Atzm WATANABE <sitosito@p.chan.ne.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#
# $Id: preferences.py,v 1.6 2005/09/27 21:35:42 atzm Exp $
#

__author__ = '$Author: atzm $'
__version__ = '$Revision: 1.6 $'

import gtk, gobject

import common, config
from queryluid import LUIDInfo
from viewer.ghostmanager import OrderedDictionary

def create_frame(frame_name, dic):
	box = gtk.VBox()
	for widget in dic.values():
		widget.set_border_width(5)
		box.pack_start(widget, False, False, 0)
	frame = gtk.Frame(frame_name)
	frame.set_border_width(5)
	frame.add(box)
	return frame

class PreferenceDialog(gtk.Dialog):
	def __init__(self, callback_notify, parent=None, luid=''):
		self.callback_notify = callback_notify
		self.luid_info = LUIDInfo(luid)

		items = []
		items.append([_('General')] + self.__create_general_page())
		items.append([_('Logs')] + self.__create_log_page())
		items.append([_('Network')] + self.__create_network_page())
		items.append([_('Fonts')] + self.__create_font_page())
		items.append([_('GViewer')] + self.__create_gviewer_page())
		items.append([_('LUID'), self.__create_luid_page(), {}])

		self.__widgets_dict = {}
		self.__notebook = gtk.Notebook()
		self.__notebook.set_show_tabs(False)
		self.__notebook.set_show_border(False)
		self.__notebook.set_scrollable(True)

		liststore = gtk.ListStore(gobject.TYPE_STRING)
		for title, widget, dic in items:
			titlelabel = gtk.Label('<span weight="bold" size="x-large">%s</span>' % title)
			titlelabel.set_use_markup(True)
			hbox = gtk.HBox()
			hbox.pack_start(titlelabel, False, False, 5)
			vbox = gtk.VBox()
			vbox.pack_start(hbox, False, False, 5)
			vbox.pack_start(gtk.HSeparator(), False, False, 2)
			vbox.pack_start(widget)

			self.__notebook.append_page(vbox, gtk.Label(title))
			self.__widgets_dict.update(dic)
			liststore.append((title,))

		self.load_config()

		navisw = self.__create_navi(liststore)
		paned = gtk.HPaned()
		paned.pack1(navisw, False, False)
		paned.pack2(self.__notebook, True)

		gtk.Dialog.__init__(self, '%s: %s' % (common.APP, _('Preferences')), parent,
							gtk.DIALOG_DESTROY_WITH_PARENT|gtk.DIALOG_MODAL|gtk.DIALOG_NO_SEPARATOR,
							(gtk.STOCK_APPLY, gtk.RESPONSE_APPLY,
							 gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL,
							 gtk.STOCK_OK, gtk.RESPONSE_OK))
		self.set_border_width(5)
		self.vbox.pack_start(paned, True, True, 0)
		self.connect('delete-event', lambda x, y: self.emit('response', gtk.RESPONSE_CANCEL))

	def load_config(self, dic=None, section=''):
		if dic is None:
			dic = self.__widgets_dict
		for key, val in dic.items():
			if isinstance(val, dict):
				self.load_config(val, key)
			else:
				if isinstance(val, gtk.ToggleButton):
					val.set_active(config.get(section, key, 'boolean'))
				elif isinstance(val, gtk.ComboBox):
					val.set_active(config.get(section, key, 'int'))
				elif isinstance(val, gtk.SpinButton):
					val.set_value(config.get(section, key, 'int'))
				elif isinstance(val, gtk.Entry):
					val.set_text(config.get(section, key))
				elif isinstance(val, list):
					listdict = {}
					for i in val:
						listdict[i.get_label()] = i
						i.set_active(False)
					for i in config.get(section, key).split():
						listdict[i].set_active(True)
				else:
					raise RuntimeError, 'config[%s][%s] = %s' % (section, key, val)

	def save_config(self, dictionary=None):
		def _save_config(dic, section=''):
			for key, val in dic.items():
				if isinstance(val, dict):
					_save_config(val, key)
				else:
					if isinstance(val, gtk.ToggleButton) or isinstance(val, gtk.ComboBox):
						config.set(section, key, str(val.get_active()))
					elif isinstance(val, gtk.SpinButton):
						config.set(section, key, str(val.get_value_as_int()))
					elif isinstance(val, gtk.Entry):
						config.set(section, key, str(val.get_text()))
					elif isinstance(val, list):
						names = [i.get_label() for i in filter(lambda x: x.get_active(), val)]
						names.sort()
						config.set(section, key, ' '.join(names))
					else:
						raise RuntimeError, 'config[%s][%s] = %s' % (section, key, val)

		if dictionary is None:
			dictionary = self.__widgets_dict
		_save_config(dictionary)
		config.write()
		self.callback_notify()

	def open(self):
		self.show_all()

		while True:
			response = self.run()

			if response == gtk.RESPONSE_APPLY:
				self.save_config()
			elif response == gtk.RESPONSE_OK:
				self.save_config()
				self.close()
				break
			elif response == gtk.RESPONSE_CANCEL:
				self.close()
				break

	def close(self, *args):
		self.destroy()

	def _nav_selected(self, treeview, *args):
		selection = treeview.get_selection()
		liststore, treeiter = selection.get_selected()
		if treeiter:
			path = liststore.get_path(treeiter)[0]
			self.__notebook.set_current_page(path)

	def __create_navi(self, liststore):
		cell = gtk.CellRendererText()
		column = gtk.TreeViewColumn('')
		column.pack_start(cell)
		column.add_attribute(cell, 'text', 0)
		treeview = gtk.TreeView(liststore)
		treeview.append_column(column)
		treeview.set_rules_hint(True)
		treeview.set_headers_visible(False)
		treeview.connect('cursor-changed', self._nav_selected)
		treeview.connect('row-activated', self._nav_selected)
		navisw = gtk.ScrolledWindow()
		navisw.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
		navisw.set_shadow_type(gtk.SHADOW_IN)
		navisw.add(treeview)
		navisw.set_size_request(110, -1)
		return navisw

	def __create_general_page(self):
		genframe, gendic = self.__create_general_frame()
		sstpframe, sstpdic  = self.__create_sstp_frame()
		scriptframe, scriptdic = self.__create_initial_script_frame()

		general_page_dict = OrderedDictionary()
		general_page_dict['general'] = gendic
		general_page_dict['sstp_server'] = sstpdic
		general_page_dict['script'] = scriptdic

		general_page = gtk.VBox()
		for widget in [genframe, sstpframe, scriptframe]:
			general_page.pack_start(widget, False, False, 0)
		return [general_page, general_page_dict]

	def __create_general_frame(self):
		gendic = OrderedDictionary()
		gendic['startup_open_logwindow'] = gtk.CheckButton(_('Open log window when startup'))
		gendic['startup_connect'] = gtk.CheckButton(_('Connect bottle server when startup'))
		gendic['connect_all'] = gtk.CheckButton(_('Join all channels when connect'))
		gendic['auto_reset_editor'] = gtk.CheckButton(_('Reset editor after send'))
		gendic['logging_sent_script'] = gtk.CheckButton(_('Logging history of sent script'))

		genframe = create_frame(_('General'), gendic)
		return genframe, gendic

	def __create_sstp_frame(self):
		sstpdic = OrderedDictionary()
		sstpdic['forward_local'] = gtk.CheckButton(_('Forward'))
		sstpdic['forward_to'] = gtk.combo_box_new_text()
		sstpdic['forward_to'].append_text(_('Accept All'))
		sstpdic['forward_to'].append_text(_('Forward only listed Ghost'))
		sstpdic['forward_to'].set_active(0)
		sstpdic['forward_local'].connect('toggled', lambda x: sstpdic['forward_to'].set_sensitive(x.get_active()))

		hbox = gtk.HBox()
		hbox.set_border_width(5)
		hbox.pack_start(sstpdic['forward_to'])

		vbox = gtk.VBox()
		for item in [sstpdic['forward_local'], hbox]:
			vbox.set_border_width(5)
			vbox.pack_start(item)
		frame = gtk.Frame(_('SSTP server'))
		frame.set_border_width(5)
		frame.add(vbox)

		return frame, sstpdic

	def __create_initial_script_frame(self):
		scriptdic = OrderedDictionary()
		scriptdic['initial_script'] = gtk.Entry()

		hbox = gtk.HBox()
		hbox.set_border_width(5)
		hbox.pack_start(scriptdic['initial_script'])

		scriptbox = gtk.VBox()
		scriptbox.pack_start(hbox, False, False, 5)
		scriptbox.pack_start(gtk.Label(_('A first | is cursor position')), False, False, 5)

		scriptframe = gtk.Frame(_('Initial script'))
		scriptframe.set_border_width(5)
		scriptframe.add(scriptbox)
		return scriptframe, scriptdic

	def __create_log_page(self):
		loggingframe, loggingdic = self.__create_logging_frame()
		lognewtabframe, lognewwindic = self.__create_logwindow_frame()
		logtabframe, logwindic = self.__create_logwindow_tab_place_frame()

		log_page_dict = OrderedDictionary()
		log_page_dict['logging'] = loggingdic
		log_page_dict['logwindow'] = lognewwindic
		log_page_dict['logwindow'].update(logwindic)

		log_page = gtk.VBox()
		for widget in [loggingframe, lognewtabframe, logtabframe]:
			log_page.pack_start(widget, False, False, 0)
		return [log_page, log_page_dict]

	def __create_logging_frame(self):
		loggingdic = OrderedDictionary()
		loggingdic['save_current_when_disconnected'] = gtk.CheckButton(_("Save `Current' log as XML when disconnect"))
		loggingdic['save_current_when_close_tab'] = gtk.CheckButton(_("Save log as XML when close `Current' tab"))

		loggingdic['gzip'] = gtk.CheckButton(_('GZIP Compress'))
		loggingdic['gzip'].set_sensitive(False)

		loggingdic['filename'] = gtk.Entry()
		loggingdic['filename'].set_sensitive(False)

		def check_toggled(widget):
			state = loggingdic['save_current_when_disconnected'].get_active() or loggingdic['save_current_when_close_tab'].get_active()
			loggingdic['gzip'].set_sensitive(state)
			loggingdic['filename'].set_sensitive(state)
		loggingdic['save_current_when_disconnected'].connect('toggled', check_toggled)
		loggingdic['save_current_when_close_tab'].connect('toggled', check_toggled)

		hbox = gtk.HBox()
		hbox.set_border_width(5)
		hbox.pack_start(loggingdic['filename'])

		lbox = gtk.VBox()
		lbox.pack_start(hbox, False, False, 3)
		lbox.pack_start(gtk.Label(_('Can use time format e.g. %H, %M, %S and etc.')), False, False, 3)

		lframe = gtk.Frame(_('Filename'))
		lframe.set_shadow_type(gtk.SHADOW_ETCHED_OUT)
		lframe.set_border_width(5)
		lframe.add(lbox)

		logbox = gtk.VBox()
		for item in [loggingdic[t] for t in ['save_current_when_disconnected', 'save_current_when_close_tab', 'gzip']] + [lframe]:
			item.set_border_width(5)
			logbox.pack_start(item, False, False, 0)

		logframe = gtk.Frame(_('Logs'))
		logframe.set_border_width(5)
		logframe.add(logbox)
		return logframe, loggingdic

	def __create_logwindow_frame(self):
		logwindic = OrderedDictionary()
		logwindic['focus_new_page'] = gtk.CheckButton(_('Focus new page automatically'))
		logwindic['new_page_placement'] = gtk.combo_box_new_text()
		for text in [_("To current tab's next"), _("To current tab's previous"), _('As end tab'), _('As first tab')]:
			logwindic['new_page_placement'].append_text(text)
		logwindic['new_page_placement'].set_active(0)

		hbox = gtk.HBox()
		hbox.set_border_width(5)
		hbox.pack_start(logwindic['new_page_placement'])

		vbox = gtk.VBox()
		for item in [logwindic['focus_new_page'], hbox]:
			vbox.set_border_width(5)
			vbox.pack_start(item)
		frame = gtk.Frame(_('When open new page...'))
		frame.set_border_width(5)
		frame.add(vbox)

		return frame, logwindic

	def __create_logwindow_tab_place_frame(self):
		logwindic = OrderedDictionary()
		logwindic['tab_placement'] = gtk.combo_box_new_text()
		for text in [_('Top'), _('Bottom'), _('Left'), _('Right')]:
			logwindic['tab_placement'].append_text(text)
		logwindic['tab_placement'].set_active(0)

		hbox = gtk.HBox()
		hbox.set_border_width(5)
		hbox.pack_start(logwindic['tab_placement'])

		vbox = gtk.VBox()
		vbox.set_border_width(5)
		vbox.pack_start(hbox)
		frame = gtk.Frame(_('Tab position'))
		frame.set_border_width(5)
		frame.add(vbox)

		return frame, logwindic

	def __create_network_page(self):
		browserframe, browserdic = self.__create_browser_frame()
		downloadframe, downloaddic = self.__create_download_frame()

		network_page_dict = OrderedDictionary()
		network_page_dict['browser'] = browserdic
		network_page_dict['download'] = downloaddic

		network_page = gtk.VBox()
		for widget in [browserframe, downloadframe]:
			network_page.pack_start(widget, False, False, 0)
		return [network_page, network_page_dict]

	def __create_browser_frame(self):
		browdict = OrderedDictionary()
		browdict['browser_command'] = gtk.Entry()

		hbox = gtk.HBox()
		hbox.set_border_width(5)
		hbox.pack_start(browdict['browser_command'])

		browbox = gtk.VBox()
		browbox.pack_start(hbox, False, False, 5)
		browbox.pack_start(gtk.Label(_('Replace %s to URL')), False, False, 5)

		browframe = gtk.Frame(_('Command to run browser'))
		browframe.set_border_width(5)
		browframe.add(browbox)
		return browframe, browdict

	def __create_download_frame(self):
		dldict = OrderedDictionary()
		dldict['auto_download'] = gtk.CheckButton(_('Auto download when receive script contained URL'))
		dldict['logging_download_history'] = gtk.CheckButton(_('Logging history'))
		dldict['logging_download_history'].set_sensitive(False)

		# make table
		rows = 4
		ex_table = gtk.Table(7, rows)

		keys = common.DL_SUFFIX_PATTERNS.keys()
		keys.sort()

		length = len(keys) / rows + 1
		itemlist = [keys[length*i:length*(i+1)] for i in xrange(rows)]

		dldict['download_type'] = []
		for i in xrange(len(itemlist)):
			for j in xrange(len(itemlist[i])):
				button = gtk.CheckButton(itemlist[i][j])
				button.set_sensitive(False)
				dldict['download_type'].append(button)
				ex_table.attach(button, j, j+1, i, i+1, gtk.FILL, gtk.FILL, 3, 3)

		def check_toggled(widget):
			state = widget.get_active()
			for i in [dldict['logging_download_history']] + dldict['download_type']:
				i.set_sensitive(state)
		dldict['auto_download'].connect('toggled', check_toggled)

		ex_frame = gtk.Frame(_('Supported file types'))
		ex_frame.set_shadow_type(gtk.SHADOW_ETCHED_OUT)
		ex_frame.set_border_width(5)
		ex_frame.add(ex_table)

		dlbox = gtk.VBox()
		for widget in [dldict['auto_download'], dldict['logging_download_history'], ex_frame]:
			widget.set_border_width(5)
			dlbox.pack_start(widget, False, False, 0)

		dlframe = gtk.Frame(_('Download'))
		dlframe.set_border_width(5)
		dlframe.add(dlbox)
		return dlframe, dldict

	def __create_font_page(self):
		fontframe, fontdic = self.__create_font_frame()
		fgcolorframe, fgcolordic = self.__create_fgcolor_frame()

		font_page_dict = OrderedDictionary()
		font_page_dict['font'] = fontdic
		font_page_dict['fgcolor'] = fgcolordic

		font_page = gtk.VBox()
		for widget in [fontframe, fgcolorframe]:
			font_page.pack_start(widget, False, False, 0)
		return [font_page, font_page_dict]

	def __create_font_frame(self):
		def open_fontselection(widget=None, entry=None):
			fontsel = gtk.FontSelectionDialog(_('Chose font...'))
			fontsel.show()
			fontsel.set_font_name(entry.get_text())
			res = fontsel.run()
			fontsel.hide()
			if res == gtk.RESPONSE_OK:
				entry.set_text(fontsel.get_font_name())
			fontsel.destroy()

		fontdict = OrderedDictionary()

		fonttable = gtk.Table(2, 3)
		itemlist = [['general', _('general font')], ['script', _('script font')]]

		for i in xrange(len(itemlist)):
			fontdict[itemlist[i][0]] = gtk.Entry()
			fonttable.attach(gtk.Label(itemlist[i][1]), 0, 1, i, i+1, gtk.FILL, gtk.FILL, 3, 3)
			fonttable.attach(fontdict[itemlist[i][0]], 1, 2, i, i+1, gtk.FILL, gtk.FILL, 3, 3)
			btn = common.get_icon_button(gtk.STOCK_SELECT_FONT, size=gtk.ICON_SIZE_BUTTON, relief=gtk.RELIEF_NORMAL)
			btn.connect('clicked', open_fontselection, fontdict[itemlist[i][0]])
			fonttable.attach(btn, 2, 3, i, i+1, gtk.FILL, gtk.FILL, 0, 3)

		fontbox = gtk.VBox()
		fontbox.pack_start(fonttable, False, False, 0)

		fontframe = gtk.Frame(_('Font settings'))
		fontframe.set_border_width(5)
		fontframe.add(fontbox)
		return fontframe, fontdict

	def __create_fgcolor_frame(self):
		def open_colorselection(widget=None, entry=None):
			colorsel = gtk.ColorSelectionDialog(_('Chose color...'))
			colorsel.show()
			try:
				colorsel.colorsel.set_current_color(gtk.gdk.color_parse(entry.get_text()))
			except ValueError:
				pass
			res = colorsel.run()
			colorsel.hide()
			if res == gtk.RESPONSE_OK:
				color = colorsel.colorsel.get_current_color()
				color_str = '#%04x%04x%04x' % (color.red, color.green, color.blue)
				entry.set_text(color_str)
			colorsel.destroy()

		fgcdict = OrderedDictionary()

		colortable = gtk.Table(6, 3)
		itemlist = [
			['script', _('script')],
			['sakura', _('sakura')],
			['unyuu', _('kero')],
			['synchronized', _('sync')],
			['error', _('error')],
			['url', _('URL')],
			]
		for i in xrange(len(itemlist)):
			fgcdict[itemlist[i][0]] = gtk.Entry()
			colortable.attach(gtk.Label(itemlist[i][1]), 0, 1, i, i+1, gtk.FILL, gtk.FILL, 3, 3)
			colortable.attach(fgcdict[itemlist[i][0]], 1, 2, i, i+1, gtk.FILL, gtk.FILL, 3, 3)
			btn = common.get_icon_button(gtk.STOCK_SELECT_COLOR, size=gtk.ICON_SIZE_BUTTON, relief=gtk.RELIEF_NORMAL)
			btn.connect('clicked', open_colorselection, fgcdict[itemlist[i][0]])
			colortable.attach(btn, 2, 3, i, i+1, gtk.FILL, gtk.FILL, 0, 3)

		colorbox = gtk.VBox()
		colorbox.pack_start(colortable, False, False, 0)

		colorframe = gtk.Frame(_('Color settings'))
		colorframe.set_border_width(5)
		colorframe.add(colorbox)
		return colorframe, fgcdict

	def __create_gviewer_page(self):
		gvdict = OrderedDictionary()
		gvdict['use_gviewer'] = gtk.CheckButton(_('Use GViewer'))

		adj = gtk.Adjustment(1.0, 0.0, 500.0, 1.0, 5.0, 0.0)
		gvdict['wait_char'] = gtk.SpinButton(adj, 0, 0)

		adj = gtk.Adjustment(1.0, 0.0, 500.0, 1.0, 5.0, 0.0)
		gvdict['wait_w1'] = gtk.SpinButton(adj, 0, 0)

		gvdict['draw_number'] = gtk.CheckButton(_('Display surface number'))
		gvdict['sticky'] = gtk.CheckButton(_('Stick GViewer'))
		gvdict['auto_open'] = gtk.CheckButton(_('Auto open GViewer when play script'))
		gvdict['raise_before_play'] = gtk.CheckButton(_('Raise GViewer before play script'))
		gvdict['sink_after_play'] = gtk.CheckButton(_('Sink GViewer after play script'))

		adj = gtk.Adjustment(1.0, 0.0, 10000.0, 1.0, 5.0, 0.0)
		gvdict['sink_timer'] = gtk.SpinButton(adj, 0, 0)

		def check_toggled(widget):
			state = widget.get_active()
			for i in [gvdict[t] for t in ['wait_char', 'wait_w1', 'draw_number', 'sticky', 'auto_open', 'raise_before_play', 'sink_after_play']]:
				i.set_sensitive(state)
			state = state and gvdict['sink_after_play'].get_active()
			gvdict['sink_timer'].set_sensitive(state)
		gvdict['use_gviewer'].connect('toggled', check_toggled)
		check_toggled(gvdict['use_gviewer'])

		def check_toggled_sink(widget):
			state = widget.get_active() and gvdict['use_gviewer'].get_active()
			gvdict['sink_timer'].set_sensitive(state)
		gvdict['sink_after_play'].connect('toggled', check_toggled_sink)
		check_toggled_sink(gvdict['sink_after_play'])

		wait_table = gtk.Table(2, 3)
		wait_table.attach(gtk.Label(_('Wait for a character')), 0, 1, 0, 1, gtk.FILL, gtk.FILL, 3, 3)
		wait_table.attach(gvdict['wait_char'], 1, 2, 0, 1, gtk.FILL, gtk.FILL, 3, 3)
		wait_table.attach(gtk.Label(_('milli seconds')), 2, 3, 0, 1, gtk.FILL, gtk.FILL, 3, 3)
		wait_table.attach(gtk.Label(_(r'Wait for \w1')), 0, 1, 1, 2, gtk.FILL, gtk.FILL, 3, 3)
		wait_table.attach(gvdict['wait_w1'], 1, 2, 1, 2, gtk.FILL, gtk.FILL, 3, 3)
		wait_table.attach(gtk.Label(_('milli seconds')), 2, 3, 1, 2, gtk.FILL, gtk.FILL, 3, 3)

		sink_table = gtk.Table(1, 3)
		sink_table.attach(gtk.Label(_('Time from played through sinking GViewer')), 0, 1, 0, 1, gtk.FILL, gtk.FILL, 3, 3)
		sink_table.attach(gvdict['sink_timer'], 1, 2, 0, 1, gtk.FILL, gtk.FILL, 3, 3)
		sink_table.attach(gtk.Label(_('milli seconds')), 2, 3, 0, 1, gtk.FILL, gtk.FILL, 3, 3)

		viewbox = gtk.VBox()
		for widget in [gvdict['use_gviewer'], wait_table, gvdict['draw_number'], gvdict['sticky'], gvdict['auto_open'], gvdict['raise_before_play'], gvdict['sink_after_play'], sink_table]:
			widget.set_border_width(5)
			viewbox.pack_start(widget, False, False, 0)

		viewerframe = gtk.Frame(_('GViewer'))
		viewerframe.set_border_width(5)
		viewerframe.add(viewbox)

		viewerpage = gtk.VBox()
		viewerpage.pack_start(viewerframe, False, False, 0)

		gvpage_dict = OrderedDictionary()
		gvpage_dict['gviewer'] = gvdict

		return [viewerpage, gvpage_dict]

	def __create_luid_page(self):
		def set_columns(treeview, columns):
			for i in xrange(len(columns)):
				treeview.append_column(columns[i])
				cell = gtk.CellRendererText()
				columns[i].pack_start(cell, False)
				columns[i].add_attribute(cell, 'text', i)
				columns[i].set_resizable(True)
		def get_treeview():
			treeview = gtk.TreeView()
			treeview.set_rules_hint(True)
			treeview.set_search_column(True)
			treeview.set_reorderable(True)
			return treeview
		def get_scrolled_window(widget, width, height):
			sw = gtk.ScrolledWindow()
			sw.set_size_request(width, height)
			sw.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
			sw.set_border_width(5)
			sw.set_shadow_type(gtk.SHADOW_IN)
			sw.add(widget)
			return sw
		def get_luid_tree(key, luid_data):
			treeview = get_treeview()
			if luid_data.has_key(key) and luid_data[key]:
				items = [gobject.TYPE_STRING for i in xrange(len(luid_data[key][0]))]
				if not items: items = [gobject.TYPE_STRING]
				liststore = gtk.ListStore(*items)
				treeview.set_model(liststore)
				columns = [gtk.TreeViewColumn(name) for name in luid_data[key][0]]
				set_columns(treeview, columns)
			else:
				liststore = gtk.ListStore(gobject.TYPE_STRING)
				treeview.set_model(liststore)
				set_columns(treeview, [gtk.TreeViewColumn()])
			for line in luid_data[key][1:]:
				liststore.append(tuple(line))
			return treeview, liststore

		self.luid_data = self.luid_info.get_all()

		self.luidinfolist, self.luidinfostore = get_luid_tree(LUIDInfo.LUID_INFO_GENERAL, self.luid_data)
		self.luidinfosw = get_scrolled_window(self.luidinfolist, 300, 80)

		self.luidlist, self.luidstore = get_luid_tree(LUIDInfo.LUID_INFO_DETAIL, self.luid_data)
		self.luidsw = get_scrolled_window(self.luidlist, 300, 100)

		def refresh_luid_info(widget):
			self.luid_data = self.luid_info.get_all(True)

			self.luidinfosw.remove(self.luidinfolist)
			self.luidinfolist = get_luid_tree(LUIDInfo.LUID_INFO_GENERAL, self.luid_data)[0]
			self.luidinfosw.add(self.luidinfolist)

			self.luidsw.remove(self.luidlist)
			self.luidlist = get_luid_tree(LUIDInfo.LUID_INFO_DETAIL, self.luid_data)[0]
			self.luidsw.add(self.luidlist)
			self.show_all()

		luidbutton = gtk.Button(_('Update'))
		luidbutton.connect('clicked', refresh_luid_info)

		luidbox = gtk.VBox()
		luidbox.pack_start(self.luidinfosw, False, False, 0)
		luidbox.pack_start(self.luidsw, True, True, 0)
		luidbox.pack_end(luidbutton, False, False, 3)
		return luidbox

if __name__ == '__main__':
	import gettext, sys
	gettext.install('gbottler')
	config.init()
	def hoge(): pass
	p = PreferenceDialog(hoge)
	p.close = lambda *args: sys.exit()
	p.open()
