#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
#  gviewer.py - SSTP playing library for PyGTK
#  Copyright (C) 2004 by Atzm WATANABE <sitosito@p.chan.ne.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#
# $Id: gviewer.py,v 1.18 2005/09/11 12:15:55 atzm Exp $
#

import gtk, gobject, pango
import gc
from script import *
from viewercommon import *
import config

class GViewerQueue:
	# milli seconds
	POLLING_INTERVAL = 10

	def __init__(self, sakura, kero, window=None, s=None, cg=None, rest_scripts=None, force_garbage_collect=True,
				 polling_start_at_first=config.get('gviewer', 'use_gviewer', 'boolean')):
		self.sakura_frame, self.b1, self.sw1 = sakura
		self.kero_frame,   self.b2, self.sw2 = kero

		self._window = window
		self._lower_id = []

		self.sender_label = s
		self.channel_and_ghost_label = cg
		self.rest_scripts_label = rest_scripts

		self.force_garbage_collect = force_garbage_collect

		# parser
		self.parser = Parser('loose')

		# request queue
		self.queue = []
		self.playing = False
		self.timeouts = []

		# one request's each chunk queue
		self.chunk_queue = []
		self.chunk_playing = False
		self.chunk_timeouts = []

		# initialize
		self.super_id       = None
		self.super_chunk_id = None

		if polling_start_at_first:
			self.polling_start()

	def set_force_garbage_collect(self, flag):
		self.force_garbage_collect = bool(flag)

	def polling_start(self):
		if not self.super_id:
			self.super_id  = gobject.timeout_add(self.POLLING_INTERVAL, self._insert)
		if not self.super_chunk_id:
			self.super_chunk_id = gobject.timeout_add(self.POLLING_INTERVAL, self._chunk_insert)

	def polling_stop(self):
		self.cancel_all(True)

	def cancel(self):
		for i in self.chunk_timeouts:
			gobject.source_remove(i)
		self.chunk_timeouts = []

		self.chunk_queue = []
		self.b1.set_text('')
		self.b2.set_text('')

		self.clear_surface(self.sakura_frame)
		self.clear_surface(self.kero_frame)
		self.clear_information()

		self.chunk_playing = False
		self.playing = False

		self.super_chunk_id = gobject.timeout_add(self.POLLING_INTERVAL, self._chunk_insert)

	def cancel_all(self, stop=False):
		for i in self.chunk_timeouts + self.timeouts:
			gobject.source_remove(i)
		self.timeouts = []
		self.chunk_timeouts = []

		self.queue = []
		self.chunk_queue = []
		self.b1.set_text('')
		self.b2.set_text('')

		self.clear_surface(self.sakura_frame)
		self.clear_surface(self.kero_frame)
		self.clear_information()

		self.chunk_playing = False
		self.playing = False

		if stop:
			self.super_id       = None
			self.super_chunk_id = None
		else:
			self.super_id       = gobject.timeout_add(self.POLLING_INTERVAL, self._insert)
			self.super_chunk_id = gobject.timeout_add(self.POLLING_INTERVAL, self._chunk_insert)

	def clear_information(self):
		if self.sender_label:
			self.sender_label.set_text('')
		if self.channel_and_ghost_label:
			self.channel_and_ghost_label.set_text('')
		if self.rest_scripts_label:
			self.rest_scripts_label.set_text('')

	def clear_surface(self, container):
		for image in container.get_children():
			image.hide()
			container.remove(image)
			image.destroy()

		if self.force_garbage_collect:
			gc.collect()

	def update_rest(self):
		if self.rest_scripts_label:
			rest = str(len(self.queue))
			self.rest_scripts_label.set_text(rest)

	def enqueue(self, script, ghost, sender=None, channel=None, ifghost=None):
		self.queue.append([script, ghost, sender, channel, ifghost])
		self.update_rest()

	def _raise(self):
		if config.get('gviewer', 'raise_before_play', 'boolean'):
			if isinstance(self._window, gtk.Window) and isinstance(self._window.window, gtk.gdk.Window):
				if self._lower_id:
					for lid in self._lower_id:
						gobject.source_remove(lid)
					self._lower_id = []
				self._window.window.raise_()

	def _lower(self):
		if config.get('gviewer', 'sink_after_play', 'boolean'):
			if isinstance(self._window, gtk.Window) and isinstance(self._window.window, gtk.gdk.Window):
				self._window.window.lower()

	def _insert(self):
		self.timeouts.append(gobject.timeout_add(self.POLLING_INTERVAL, self._insert))
		if not self.queue or self.playing or self.chunk_queue or self.chunk_playing:
			return

		self._raise()

		self.playing = True
		self.b1.set_text('')
		self.b2.set_text('')
		text, ghost, sender, channel, ifghost = self.queue.pop(0)
		self.update_rest()

		#self.clear_surface(self.sakura_frame)
		#self.clear_surface(self.kero_frame)
		self.chunk_queue.append(['', 0,  SIDE_SAKURA, 0, [SESSION_QUICK],
								 ghost, sender, channel, ifghost])
		self.chunk_queue.append(['', 0, SIDE_KERO, 10, [SESSION_QUICK],
								 ghost, sender, channel, ifghost])

		chunks = self.parser.parse(text)

		sessions = []
		events   = []
		linefeed_buf = ''

		current_side = [SIDE_SAKURA]
		for chunk in chunks:
			if chunk[0] == TEXT_STRING:
				for n in xrange(len(chunk[1])):
					if chunk[1][n][0] in [SCRIPT_TEXT, TEXT_META]:
						text = chunk[1][n][1]

						if chunk[1][n][0] == TEXT_META:
							if ghost is not None and text == '%username':
								user = ghost.get_user()
								if user is not None:
									text = user
							elif text == '%selfname':
								try:
									sakuraname = ghost.get_sakura()
								except:
									sakuraname = ifghost
								if sakuraname:
									text = sakuraname
							elif ghost is not None and text == '%keroname':
								keroname = ghost.get_unyu()
								if keroname is not None:
									text = keroname

						text = linefeed_buf + text
						linefeed_buf = ''
						self.chunk_queue.append([text, 0, current_side[-1], None,
												 sessions[:], ghost, sender, channel, ifghost])

						while sessions.count(EVENT_CLEAR):
							sessions.remove(EVENT_CLEAR)

			elif chunk[0] == TEXT_META:
				name, args = chunk[1], chunk[2:]

				if name == r'\w':
					#if SESSION_QUICK not in sessions:
					self.chunk_queue.append(['', int(args[0]), current_side[-1], None,
											 sessions[:], ghost, sender, channel, ifghost])

				elif name == r'\s':
					num = range(len(current_side))
					num.reverse()
					for i in num:
						if current_side[i] in [SIDE_SAKURA, SIDE_KERO]:
							self.chunk_queue.append(['', 0, current_side[i], int(args[0]), sessions[:],
													 ghost, sender, channel, ifghost])
							break

				elif name == r'\n':
					linefeed_buf += '\n'

				elif name in [r'\0', r'\h', r'\1', r'\u', r'\_s']:
					self.chunk_queue.append([linefeed_buf, 0, current_side[-1], None,
											 sessions[:], ghost, sender, channel, ifghost])
					linefeed_buf = ''

					while sessions.count(EVENT_CLEAR):
						sessions.remove(EVENT_CLEAR)

					if name in [r'\0', r'\h']:
						if not current_side[-1] == SIDE_SYNC:
							current_side = [SIDE_SAKURA]

					elif name in [r'\1', r'\u']:
						if not current_side[-1] == SIDE_SYNC:
							current_side = [SIDE_KERO]

					elif name == r'\_s':
						if current_side[-1] == SIDE_SYNC and len(current_side) > 1:
							current_side.pop()
						else:
							current_side.append(SIDE_SYNC)

				elif name == r'\URL':
					self.chunk_queue.append(['\n', 0, current_side[-1], None,
											 sessions[:] + [SESSION_QUICK], ghost, sender,
											 channel, ifghost])
					for arg in args:
						for type, text in arg:
							self.chunk_queue.append([text + '\n', 0, current_side[-1], None,
													 sessions[:] + [SESSION_QUICK], ghost, sender,
													 channel, ifghost])

				elif name == r'\_q':
					if SESSION_QUICK in sessions:
						while sessions.count(SESSION_QUICK):
							sessions.remove(SESSION_QUICK)
					else:
						sessions.append(SESSION_QUICK)

				elif name == r'\c':
					sessions.append(EVENT_CLEAR)

				elif name == r'\e':
					self.chunk_queue.append([linefeed_buf, 5, current_side[-1], None,
											 sessions[:], ghost, sender, channel, ifghost])
					break

	def _chunk_insert(self):
		self.chunk_timeouts.append(gobject.timeout_add(self.POLLING_INTERVAL, self._chunk_insert))
		if not self.chunk_queue or self.chunk_playing:
			return
		self.chunk_playing = True

		chunk_max = len(self.chunk_queue)

		text, wait, side, surf, sessions, ghost, sender, channel, ifghost = self.chunk_queue.pop(0)

		if self.sender_label and sender:
			self.sender_label.set_text(sender)

		if self.channel_and_ghost_label:
			if channel is not None:
				self.channel_and_ghost_label.set_text(channel + ' / ')

			ghost_name = None
			if ghost is not None:
				ghost_name = ghost.get_sakura()

			if ghost_name is not None:
				self.channel_and_ghost_label.set_text(self.channel_and_ghost_label.get_text() +
													  ghost_name)
			elif ifghost:
				self.channel_and_ghost_label.set_text(self.channel_and_ghost_label.get_text() +
													  ifghost)

		event = None
		if EVENT_CLEAR in sessions:
			event = EVENT_CLEAR

		if text != '':
			if SESSION_QUICK in sessions:
				self.chunk_timeouts.append(gobject.timeout_add(0, self.chunk_insert, text, 0, 0,
														   chunk_max, side, surf, event, ghost))
			else:
				# for clear
				self.chunk_timeouts.append(gobject.timeout_add(wait * config.get('gviewer', 'wait_w1', 'int'),
															   self.chunk_insert, text[0],
															   1, len(text), chunk_max,
															   side, surf, event, ghost))
				for i in xrange(1, len(text)):
					self.chunk_timeouts.append(gobject.timeout_add(i * config.get('gviewer', 'wait_char', 'int') + \
															   wait * config.get('gviewer', 'wait_w1', 'int'),
															   self.chunk_insert, text[i],
															   i+1, len(text), chunk_max,
															   side, surf, None, ghost))
		elif EVENT_CLEAR in sessions:
			self.chunk_timeouts.append(gobject.timeout_add(0, self.chunk_insert, '',
													   0, 0, chunk_max, side, surf,
													   EVENT_CLEAR, ghost))
		else:
			self.chunk_timeouts.append(gobject.timeout_add(wait * config.get('gviewer', 'wait_w1', 'int'),
													   self.chunk_insert, '',
													   0, 0, chunk_max, side, surf,
													   event, ghost))

	def chunk_insert(self, c, i, max, chunk_max, side, surf, event, ghost, sequential_flag=False):
		if not sequential_flag:
			self.insert(c, chunk_max, side, surf, event, ghost)
		else:
			self.insert_ss(c, chunk_max, side, surf, event, ghost)
		if i == max:
			self.chunk_playing = False

	def insert(self, c, chunk_max, side, surf, event, ghost):
		self.set_surface(side, surf, ghost)

		if side == SIDE_SAKURA:
			insertee = [[self.b1, self.sw1]]
		elif side == SIDE_KERO:
			insertee = [[self.b2, self.sw2]]
		elif side == SIDE_SYNC:
			insertee = [[self.b1, self.sw1], [self.b2, self.sw2]]

		for buf, sw in insertee:
			if event == EVENT_CLEAR:
				buf.set_text('')
			buf.insert(buf.get_end_iter(), c)
			sw.emit('scroll-child', gtk.SCROLL_END, False)

		if chunk_max == 1:
			gobject.timeout_add(config.get('gviewer', 'wait_char', 'int') + self.POLLING_INTERVAL, self.clear_timeouts)

	### use Sequential Splite
	def insert_ss(self, c, chunk_max, side, surf, event, ghost):
		### if the surface is 'sequential='
		# if not exist 'sequqntial' call self.insert (are you all right?)
		if not ghost.check_ss(surf):
			self.insert(c, chunk_max, side, surf, event, ghost)
		# else if exist 'sequential' run in while self.insert
		else:
			## ss is a definition of Sequential Splite
			##		for example '('104','','100','',101,'','',102,'',103)'
			ss = ghost.get_ss_group(surf)
			# after here,
			# need parse ss and call set_surface with define wait_time as you like...
			stand = 0
			for i in xrange(len(ss)):
				if ss[i] == r'*':
					i = 0
					continue
				elif ss[i] == '':
					stand += config.get('gviewer', 'wait_w1', 'int')
				else:
					gobject.timeout_add(config.get('gviewer', 'wait_w1', 'int') * i + stand,
									self.set_surface, side, int(ss[i]), ghost, True)

					if side == SIDE_SAKURA:
						insertee = [[self.b1, self.sw1]]
					elif side == SIDE_KERO:
						insertee = [[self.b2, self.sw2]]
					elif side == SIDE_SYNC:
						insertee = [[self.b1, self.sw1], [self.b2, self.sw2]]

					for buf, sw in insertee:
						if event == EVENT_CLEAR:
							print i
							buf.set_text('')
							buf.insert(buf.get_end_iter(), c)
							sw.emit('scroll-child', gtk.SCROLL_END, False)

			gobject.timeout_add(config.get('gviewer', 'wait_char', 'int') + self.POLLING_INTERVAL, self.clear_timeouts)

	def set_number(self, pixbuf, surf_num, side):
		def _expose(drawingarea, event, pixmap, num, side):
			style = drawingarea.get_style()
			fg_gc = style.fg_gc[gtk.STATE_NORMAL]
			bg_gc = style.bg_gc[gtk.STATE_NORMAL]
			drawingarea.window.draw_drawable(bg_gc, pixmap, 0, 0, 0, 0, -1, -1)

			bg = style.bg[gtk.STATE_NORMAL]
			color = '#%04x%04x%04x' % (bg.red, bg.green, bg.blue)
			layout = pango.Layout(drawingarea.get_pango_context())
			layout.set_markup('<span size="x-small" background="%s">%d</span>' % (color, num))

			w, h = layout.get_pixel_size()
			h_pos = SIZE_SURFACE_V - h - 1
			if side == SIDE_SAKURA:
				w_pos = 0
			else:
				w_pos = SIZE_SURFACE_H - w - 3

			drawingarea.window.draw_rectangle(bg_gc, True, SIZE_SURFACE_H, SIZE_SURFACE_V, w, h)
			drawingarea.window.draw_layout(fg_gc, w_pos, h_pos, layout)

			return True

		pixmap = pixbuf.render_pixmap_and_mask()[0]
		drawingarea = gtk.DrawingArea()
		drawingarea.set_events(gtk.gdk.EXPOSURE_MASK)
		drawingarea.connect('expose-event', _expose, pixmap, surf_num, side)
		return drawingarea

	def set_surface(self, side, surf_num, ghost, sequential_flag=False):
		if surf_num is None:
			return

		if side == SIDE_SAKURA:
			surf_container = self.sakura_frame
		elif side == SIDE_KERO:
			surf_container = self.kero_frame
		else:
			return
		self.clear_surface(surf_container)

		if ghost:
			if not sequential_flag:
				pixitem = ghost.get_surface(surf_num, side)
				if config.get('gviewer', 'draw_number', 'boolean'):
					pixitem = self.set_number(pixitem, surf_num, side)
			else:
				pixitem = ghost.get_surface_s(surf_num, side) ### if use Sequential Splite

			if isinstance(pixitem, gtk.gdk.Pixbuf):
				photo = gtk.Image()
				photo.set_from_pixbuf(pixitem)
			elif isinstance(pixitem, gtk.DrawingArea):
				photo = pixitem
			else:
				photo = gtk.Label(str(surf_num))
		else:
			photo = gtk.Label(str(surf_num))

		photo.set_size_request(SIZE_SURFACE_H, SIZE_SURFACE_V)
		photo.show()

		photoframe = gtk.Frame()
		photoframe.set_border_width(0)
		photoframe.set_shadow_type(gtk.SHADOW_OUT)
		photoframe.set_size_request(SIZE_SURFACE_H+1, SIZE_SURFACE_V+1)
		photoframe.show()
		photoframe.add(photo)

		fcon = gtk.VBox()
		fcon.set_border_width(0)
		fcon.show()
		fcon.pack_start(photoframe, True, False)

		surf_container.add(fcon)

	def clear_timeouts(self):
		self.timeouts = []
		self.chunk_timeouts = []
		self.playing = False
		self._lower_id.append(gobject.timeout_add(config.get('gviewer', 'sink_timer', 'int'), self._lower))


if __name__ == "__main__":
	import random
	from ghost import Ghost
	from ghostmanager import GhostManager

	SCRIPT_PATTERNS = [
		unicode(r'\t\u\s[10]\h\s[5]みんな、\w5おはよう。\w9\w9\u５月２５日の今日の誕生石だ。\w9\w9\w9\h\_s\c\s[70]イーッ！\w9\w9\_s\_q\n\n\n\n　　　　　　　　　ダイアモンド\w1\c\n\n\n\n　　　　　　　　　　 ルビー\w1\c\n\n\n\n　　　　　　　　　　トパーズ\w1\c\n\n\n\n　　　　　　　　　　　翡翠\w1\c\n\n\n\n　　　　　　　　　　ジルコン\w1\c\n\n\n\n　　　　　　　　　 サファイア\w1\c\n\n\n\n　　　　　　　　　 ガーネット\w1\c\n\n\n\n　　　　　　　　 　アメジスト\w1\c\n\n\n\n　　　　　　　　　 クリスタル\w1\c\n\n\n\n　　　　　　　　　アクアマリン\w1\c\n\n\n\n　　　　　　　　　 エメラルド\w1\c\n\n\n\n　　　　　　　　　ダイアモンド\w1\c\n\n\n\n　　　　　　　　　　オパール\w1\c\n\n\n\n　　　　BTH 小っちゃいってことは便利だねっ\w1\c\n\n\n\n　　　　　　　　　　　真珠\w1\c\n\n\n\n　　　　　　　　　　トルコ石\w1\c\n\n\n\n　　　　　　　　 ムーンストーン\w1\c\n\n\n\n　　　　　　　　ブラッドストーン\w1\c\n\n\n\n　　　　　　　　　 ペリドット\w1\c\n\n\n\n　　　　　　　　　 トルマリン\w1\c\n\n\n\n　　　　　　　　　　　琥珀\w1\c\n\n\n\n　　　　　　　　　　クォーツ\w1\c\n\n\n\n　　　　　　　　　　プラチナ\w1\c\n\n\n\n　　　　　　　　 レッドジルコン\w1\c\n\n\n\n　　　　　　　　　　シルバー\w1\c\n\n\n\n　　　　　　　 　 ラピスラズリ\w9\w9\w9\u\n\n　　　　　　　　　 幸福を得る\w1\c\n\n　　　　　　　　　　　純粋\w1\c\n\n　　　　　　　　　　真実の愛\w1\c\n\n　　　　　　　　　  至福の時\w1\c\n\n　　　　　　　　　　　再開\w1\c\n\n　　　　　　　　　　夢の実現\w1\c\n\n　　　　　　　　　　　情熱\w1\c\n\n　　　　　　　　　　清浄無垢\w1\c\n\n　　　　　　　　　　　希望\w1\c\n\n　　　　　　　　　 夢見るころ\w1\c\n\n　　　　　スカートの中覗き放だ…ウベシッ！\w1\c\n\n　　　　　　　　　 慈愛・誠実\w1\c\n\n　　　　　　　　誰よりもやさしく\w1\c\n\n　　　　　　　　　 健康・長寿\w1\c\n\n　　　　　　　　　　恋の誘惑\w1\c\n\n　　　　　　　　　　　成功\w1\c\n\n　　　　　　　　　 夫婦の幸福\w1\c\n\n　　　　　　　　　　　聡明\w1\c\n\n　　　　　　　　私だけをみつめて\w1\c\n\n　　　　　　　　　　　幸運\w1\c\n\n　　　　　　　　　 永遠の誓い\w9\w9\w9\w9\_s\n\_q　　　　　　　　　　イーッ！\_s\w9\w9\w9\s[10]\c永遠はあるよ…\w9汁親父温泉にあるよ。\w9\w9\h\c\s[4]本当に永遠の世界に逝っちゃうよ…。\e', 'utf-8'),
		unicode(r'\t\u\s[10]\h\s[5]　それでは今日の誕生花です。\n\w9\w9\w9\c\n\n\n\n\s[9]　　　　　　ヒーリングサイバーッ！\w9\w9\w9\c\_q　　　　　　☆　　　　　　　★\n\n\n\n\n\n\n　　　　　　○　　　　　　　●\_q\w2\_q\c　　　　　　　　　　　　★\n　　　　　　☆\n\n\n\n\n　　　　　　　　　　　　　　●\n　　　　　　　○\_q\w2\_q\c\n　　　　　　　　　　　★\n\n　　　　　　　☆\n\n　　　　　　　　　　　　　●\n\n　　　　　　　　　○\n\_q\w2\_q\c\n\n　　　　　　　　　　★\n\n　　　　　　　　☆　　　●\n\n　　　　　　　　　　○\n\n\_q\w2\_q\c\n\n\n　　　　　　　　　★　●\n\n　　　　　　　　　☆　○\n\n\n\_q\w2\_q\c\n\n\n　　　　　　　　　　●○\n　　　　　　　　　　★☆\n\n\n\_q\w2\_q\c\n\n\n[half]　　　　　　　　　　○☆\n　　　　　　　　　　●★\n\n\n\_q\w2\_q\c\n\n　　　　　　　　　　☆★\n　　　　　　　　　　○●\n\n\n\_q\w2\_q\c\n\n[half]　　　　　　　　　　★●\n　　　　　　　　　　☆○\n\n\n\_q\w2\_q\c\n　　　　　　　　　　●○\n　　　　　　　　　　★☆\n\_q\w2\_q\c\n[half]　　　　　　　　　　○☆\n　　　　　　　　　　●★\n\_q\w2\_q\c　　　　　　　　　　☆★\n　　　　　　　　　　○●\n\_q\w2\_q\c　　　　　　　　　　★●\n　　　　　　　　　　☆○\n\_q\w2\_q\c　　　　　　　　　　●○\n　　　　　　　　　　★☆\n\_q\w2\_q\c　　　　　　　　　　○☆\n　　　　　　　　　　●★\n\_q\w2\_q\c　　　　　　　　　　☆★\n　　　　　　　　　　○●\n\_q\w2\_q\c　　　　　　　　　　☆★\n[half]　　　　　　　　　　○●\n\_q\w2\_q\c　　　　　　　　　　 ◎\_q\w9\w9\_q\c　　　　　　　　　　○○\n[half]　　　　　　　　　 ☆　☆\n\_q\w2\_q\c\n[half]　　　　　　　　　 ○　○\n　　　　　　　　 ☆　　　☆\n\_q\w2\_q\c\n　　　　　　　　  ○　　　○\n　　　　　　　 ☆ 　　　　　☆\n\_q\w2\_q\c\n\n[half]　　　　　　　 ○　　　　　○\n\n[half]　　　　　　 ☆　　　　　　　☆\n\_q\w2\_q\c\n\n　　　　　　　○ 　　　　　 ○\n\n　　　　　　☆ 　　　　　　　☆\n\_q\w2\_q\c\n\n\n[half]　　　　　　 ○　　　　　　○\n\n　　　　　 ☆　　　　　　　☆\n\_q\w2\_q\c\n\n\n　　　　　　　○  　　　　　○\n\n　　　　　　☆ 　　　　 　　　☆\n\_q\w2\_q\c\n\n\n　　　　　　　○  　　　　　○\n\n　　　　　　☆ 　　　　 　　　☆\n\_q\w9\w9\w9\_q\c\n\n\n　　　　　　　　○  　　　○\n\n　　　　　　　 ☆　　 　　　☆\n\_q\w2\_q\c\n\n\n　　　　　　　　　○  　○\n\n　　　　　　　　 ☆ 　 　☆\n\_q\w2\_q\c\n\n\n　　　　　　　　　　○○\n\n　　　　　　　　　　☆☆\n\_q\w2\_q\c\n\n\n　　　　　　　　　○,｡･:○\n\n　　　　　　　　　☆:･｡,☆\_q\w2\_q\c\n\n\n　　　　　　　　○★,｡･:*:○\n\n　　　　　　　　☆:*:･｡,★☆\_q\w2\_q\c\n\n\n　　　　　　　○*:･ﾟ★,｡･:*:･ﾟ○\n\n　　　　　　　☆ﾟ･:*:･｡,★ﾟ･:*☆\_q\w2\_q\c\n\n\n　　　　　　○･:*:･ﾟ★,｡･:*:･ﾟ☆○\n\n　　　　　　☆☆ﾟ･:*:･｡,★ﾟ･:*:･☆\_q\w2\_q\c\n\n\n　　　　　○　･:*:･ﾟ★,｡･:*:･ﾟ☆　○\n\n　　　　　☆　☆ﾟ･:*:･｡,★ﾟ･:*:･　☆\_q\w2\_q\c\n\n\n　　　　○　　･:*:･ﾟ★,｡･:*:･ﾟ☆　　○\n\n　　　　☆　　☆ﾟ･:*:･｡,★ﾟ･:*:･　　☆\_q\w2\_q\c\n\n\n　　　○　　　･:*:･ﾟ★,｡･:*:･ﾟ☆　　　○\n\n　　　☆　　　☆ﾟ･:*:･｡,★ﾟ･:*:･　　　☆\_q\w2\_q\c\n\n\n　　○　　　　･:*:･ﾟ★,｡･:*:･ﾟ☆　　　　○\n\n　　☆　　　　☆ﾟ･:*:･｡,★ﾟ･:*:･　　　　☆\_q\w9\s[5]\_q\c　　　　　　　　　えいっ♪\n\n\n　　○　　　　･:*:･ﾟ★,｡･:*:･ﾟ☆　　　　○\n\n　　☆　　　　☆ﾟ･:*:･｡,★ﾟ･:*:･　　　　☆\_q\w9\w9\_q\c　　　　　　　　　えいっ♪\n\n　　　　 〜〜〜〜今日の誕生花〜〜〜〜\n　　○　　　 『デルフィニュウム』 　　　○\n\n　　☆　 『人の心を読む、愛の伝達者』 　☆\_q\w9\w9\w9\n\n　　　　　　　　こんな感じです。\n\w9\w9\n[half]\s[105]　　皆様、\w9今日も一日張り切っていきましょうね。\n\e', 'utf-8'),
		unicode(r'\t\u\s[10]\h\s[5]みんな、\w5おはよう。\w9\w9\u理夢とヨンヨンの今日の誕生石だ。\w9\w9\h\n\n今日の誕生石はこれ！\w9\w9\w9\_s\c\s[70]イーッ！\w9\w9\_s\_q\n\n\n\n　　　　　　　　　ダイアモンド\w1\c\n\n\n\n　　　　　　　　　　 ルビー\w1\c\n\n\n\n　　　　　　　　　　トパーズ\w1\c\n\n\n\n　　　　　　　　　　　翡翠\w1\c\n\n\n\n　　　　　　　　　　ジルコン\w1\c\n\n\n\n　　　　　　　　　 サファイア\w1\c\n\n\n\n　　　　　　　　　 ガーネット\w1\c\n\n\n\n　　　　　　　　 　アメジスト\w1\c\n\n\n\n　　　　　　　　　 クリスタル\w1\c\n\n\n\n　　　　　　　　　アクアマリン\w1\c\n\n\n\n　　　　　　　　　 エメラルド\w1\c\n\n\n\n　　　　　　　　　ダイアモンド\w1\c\n\n\n\n　　　　　　　　　　オパール\w1\c\n\n\n\n　　　　　　　シュークリーム分が足りない\w1\c\n\n\n\n　　　　　　　　　　　真珠\w1\c\n\n\n\n　　　　　　　　　　トルコ石\w1\c\n\n\n\n　　　　　　　　 ムーンストーン\w1\c\n\n\n\n　　　　　　　　ブラッドストーン\w1\c\n\n\n\n　　　　　　　　　 ペリドット\w1\c\n\n\n\n　　　　　　　　　 トルマリン\w1\c\n\n\n\n　　　　　　　　　　　琥珀\w1\c\n\n\n\n　　　　　　　　　　クォーツ\w1\c\n\n\n\n　　　　　　　　　　プラチナ\w1\c\n\n\n\n　　　　　　　　 レッドジルコン\w1\c\n\n\n\n　　　　　　　　　　シルバー\w1\c\n\n\n\n　　　　　　　 　 ダイアモンド\w9\w9\w9\u\n\n　　　　　　　　　 幸福を得る\w1\c\n\n　　　　　　　　　　　純粋\w1\c\n\n　　　　　　　　　　真実の愛\w1\c\n\n　　　　　　　　　  至福の時\w1\c\n\n　　　　　　　　　　　再開\w1\c\n\n　　　　　　　　　　夢の実現\w1\c\n\n　　　　　　　　　　　情熱\w1\c\n\n　　　　ならば代わりにカレーを食べなさい！\w1\c\n\n　　　　　　　　　　　希望\w1\c\n\n　　　　　　　　　 夢見るころ\w1\c\n\n　　　　　　　　　  清浄無垢\w1\c\n\n　　　　　　　　　 慈愛・誠実\w1\c\n\n　　　　　　　　誰よりもやさしく\w1\c\n\n　　　　　　　　　 健康・長寿\w1\c\n\n　　　　　　　　　　恋の誘惑\w1\c\n\n　　　　　　　　　　　成功\w1\c\n\n　　　　　　　　　 夫婦の幸福\w1\c\n\n　　　　　　　　　　　聡明\w1\c\n\n　　　　　　　　私だけをみつめて\w1\c\n\n　　　　　　　　　 永遠の誓い\w1\c\n\n　　　　　　　　　  清浄無垢\w9\w9\w9\w9\_s\n\_q　　　　　　　　　　イーッ！\_s\w9\w9\w9\w9\u\s[10]\c以上、\w5今日の誕生石だ。\w9\w9\h\s[5]\cそれじゃあ、\w5楽しいボトルを！\w9\w9\s[1]\n今日も一日ドキドキだね。\e', 'utf-8'),
		unicode(r'\t\h\s[0]\u\s[10]\_q（カチャ）\h　　　　　　　　　　　■\n　　　　　　　　　　　■\n　　　　　　　　■■■■■■■\n　　　　　　　　　　　■\n　　　　　　　　　　　■\n　　　　　　　　　　　■\n　　　　　　　　　　　■\w4\u（カチャ）\h\c　　　　　　　　　■■■■■\n\n\n\n\n　　　　　　　■■■■■■■■■\w4\u（カチャ）\h\c　　　　　　　■■■■■■■■■\n[half]　　　　　　　■　　　　　　　■\n[half]　　　　　　　■　　　　　　　■\n[half]　　　　　　　■　■■■■■　■\n[half]　　　　　　　■　　　■　　　■\n[half]　　　　　　　■　　　■　　　■\n[half]　　　　　　　■　　■■■　　■\n[half]　　　　　　　■　　　■　　　■\n[half]　　　　　　　■　　　■　■　■\n[half]　　　　　　　■　　　■　　　■\n[half]　　　　　　　■　■■■■■　■\n[half]　　　　　　　■　　　　　　　■\n[half]　　　　　　　■　　　　　　　■\n[half]　　　　　　　■■■■■■■■■\w4\u（カチャ）\h\c　　　　　　　　　■　　■■■■\n[half]　　　　　　　　　■　　　　　■\n[half]　　　　　　　　　　　　　　　■\n[half]　　　　　　　■■■■■　　　■\n[half]　　　　　　　　　　　　　　　■\n[half]　　　　　　　　■■■　　　　■\n[half]　　　　　　　　　　　　■■■■\n[half]　　　　　　　　■■■　■\n[half]　　　　　　　　　　　　■\n[half]　　　　　　　 ■■■■ ■\n[half]　　　　　　　 ■　　■ ■\n[half]　　　　　　　 ■　　■ ■\n[half]　　　　　　　 ■　　■ ■　　　■\n[half]　　　　　　　 ■■■■ 　■■■\w4\c\u\c\_qチャラリララ―――\w9\w9\h\_q\n\n　　　　　　　　　　十二国記\n\n　　　　　　　月の影　影の海　八章\n\n―――――――（ＮＨＫ教育：開始）――――――\_q\uチャラ\w4チャラ\w4チャラ\w4チャラ\w2チャチャチャン！\e', 'utf-8'),
		unicode(r'\t\h\s[0]\u\s[10]\_q（カチャ）\h　　　　　　　　　　　■\n　　　　　　　　　　　■\n　　　　　　　　　　　■\n　　　　　　　　　　　■■\n　　　　　　　　　　　■　■\n　　　　　　　　　　　■　　■\n　　　　　　　　　　　■\n　　　　　　　　　　　■\n　　　　　　　　　　　■\w8\u（カチャ）\h\c　　　　　　　　　　　　　　■\n　　　　　　　　　■　　　　■\n　　　　　　　　　■　　　　■\n　　　　　　　　　■　　　　■\n　　　　　　　　　　　　　　■\n　　　　　　　　　　　　　 ■\n　　　　　　　　　　　　　■\n　　　　　　　　　　　　 ■\n　　　　　　　　　　　　■\w4\u（カチャ）\h\c　　　　　　　　　■　　　　■　■\n　　　　　　　　　■　　　　 ■　■\n　　　　　　　　　■\n[half]　　　　　　　　　　　　　■■\n[half]　　　　　　　　　■■■■\n　　　　　　　　　■\n　　　　　　　　　■\n　　　　　　　　　■\n　　　　　　　　　■\n　　　　　　　　　　■■■■■■\w4\u（カチャ）\h\c　　　　　　　　　■■■■■■■\n[half]\n[half]　　　　　　　　　　　　　　■\n[half]　　　　　　　　　　　 ■\n[half]　　　　　　　　　　　　■■\n　　　　　　　　　　　　■\n　　　　　　　　　　　　■\n　　　　　　　　　　　　■\n　　　　　　　　　　　　■\n　　　　　　　　　　　 ■\n　　　　　　　　　　　■\w4\u（カチャ）\h\c\n\n　　　　　　　　　　■■■\n[half]　　　　　　　 　 ■　■　■\n[half]　　　　　　 　　■ 　■　 ■\n[half]　　　　　　　　■　　■　　■\n[half]　　　　　 　　■ 　　■　　■\n[half]　　　 　　　　■ 　　■　　■\n[half]　　　　　　　　■　 ■　　 ■\n[half]　　　　　　　　 ■ ■　　 ■\n[half]　　　　　　　　　 ■　　 ■\w4\u（カチャ）\h\c　　　　　　　　　　　■\n[half]　　　　　　　　　　■\n[half]　　　　　　　　　■\n[half]　　　　　　　　■■■■■■■\n[half]　　　　　　　　■　　　　　■\n[half]　　　　　　　　■■■■■■■\n[half]　　　　　　　　■　　　　　■\n[half]　　　　　　　　■■■■■■■\n[half]\n[half]　　　　　　　　　　　■　　　■\n[half]　　　　　　　　　　　■　　■\n[half]　　　　　　　■■■■■■■\n[half]　　　　　　　　　　■■■\n[half]　　　　　　　　　 ■ ■ ■\n[half]　　　　　　　　　■　■　■\n[half]　　　　　　　　 ■ 　■　 ■\n[half]　　　　　　　 ■ 　　■　　 ■\n[half]　　　　　　 ■ 　　　■　　　 ■\w4\c\u\c\_qチャラリララ―――\w9\w9\h\_q\n\n　　　　　　　　　トリビアの泉\n\n　　　　　　　 素晴らしきムダ知識\n\n―――――――（フジテレビ：開始）――――――\_q\uチャラ\w4チャラ\w4チャラ\w4チャラ\w2チャチャチャン！\e', 'utf-8'),
		]
	SENDER_PATTERNS = [
		unicode(r'GBottler', 'utf-8'),
		unicode(r'SSTP Bottle Client', 'utf-8'),
		]
	CHANNEL_PATTERNS = [
		unicode(r'更新情報', 'utf-8'),
		unicode(r'駅前繁華街', 'utf-8'),
		unicode(r'海浜公園街', 'utf-8'),
		unicode(r'学園街', 'utf-8'),
		unicode(r'山の温泉街', 'utf-8'),
		]
	GHOST_PATTERNS = [
		unicode(r'理夢', 'utf-8'),
		unicode(r'せりこ', 'utf-8'),
		unicode(r'毒子', 'utf-8'),
		unicode(r'なる', 'utf-8'),
		unicode(r'ねここ', 'utf-8'),
		unicode(r'猫', 'utf-8'),
		]

	# sender, channel and ghost:
	# they must be gtk.Label
	s_l = gtk.Label()
	s_l.show()
	cg_l = gtk.Label()
	cg_l.show()
	r_l = gtk.Label()
	r_l.show()
	i_l = gtk.Label(' Items')
	i_l.show()
	scbox = gtk.HBox()
	scbox.show()
	scbox.pack_start(s_l)
	scbox.pack_start(cg_l)
	scbox.pack_start(r_l)
	scbox.pack_start(i_l)

	# sakura side
	sakura_frame = gtk.Frame()
	sakura_frame.set_shadow_type(gtk.SHADOW_NONE)
	sakura_frame.set_size_request(SIZE_SURFACE_H, SIZE_SURFACE_V)
	sakura_frame.show()

	t1 = gtk.TextView()
	t1.show()
	t1.set_wrap_mode(gtk.WRAP_CHAR)
	t1.set_editable(False)
	t1.set_cursor_visible(False)
	b1 = t1.get_buffer()
	sw1 = gtk.ScrolledWindow()
	sw1.show()
	sw1.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_ALWAYS)
	sw1.add(t1)
	f1 = gtk.Frame()
	f1.show()
	f1.set_shadow_type(gtk.SHADOW_IN)
	f1.add(sw1)

	sakurabox = gtk.HBox()
	sakurabox.show()
	sakurabox.pack_start(sakura_frame, False, False, 0)
	sakurabox.pack_start(f1)
	sakurabox.set_size_request(100, SIZE_SURFACE_V * 2 + 20)

	# kero side
	kero_frame = gtk.Frame()
	kero_frame.set_shadow_type(gtk.SHADOW_NONE)
	kero_frame.set_size_request(SIZE_SURFACE_H, SIZE_SURFACE_V)
	kero_frame.show()

	t2 = gtk.TextView()
	t2.show()
	t2.set_wrap_mode(gtk.WRAP_CHAR)
	t2.set_editable(False)
	t2.set_cursor_visible(False)
	b2 = t2.get_buffer()
	sw2 = gtk.ScrolledWindow()
	sw2.show()
	sw2.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_ALWAYS)
	sw2.add(t2)
	f2 = gtk.Frame()
	f2.show()
	f2.set_shadow_type(gtk.SHADOW_IN)
	f2.add(sw2)

	kerobox = gtk.HBox()
	kerobox.show()
	kerobox.pack_start(kero_frame, False, False, 0)
	kerobox.pack_start(f2)
	kerobox.set_size_request(100, SIZE_SURFACE_V + 20)

	# play button
	pbutton = gtk.Button(unicode('Test play', 'utf-8'))
	pbutton.show()

	# cancel button
	ibutton = gtk.Button(unicode('Cancel', 'utf-8'))
	ibutton.show()

	# cancel button
	cbutton = gtk.Button(unicode('Cancel all', 'utf-8'))
	cbutton.show()

	# buttons
	bbox = gtk.HBox()
	bbox.show()
	bbox.pack_start(pbutton)
	bbox.pack_start(ibutton)
	bbox.pack_start(cbutton)

	# all
	box = gtk.VBox()
	box.show()
	box.pack_start(scbox)
	box.pack_start(sakurabox)
	box.pack_start(kerobox)
	box.pack_start(bbox)

	# window
	w = gtk.Window()
	w.set_size_request(400, SIZE_SURFACE_V * 2 + 140)
	w.show()
	w.connect('destroy', lambda x: gtk.mainquit())
	w.add(box)

	gv = GViewerQueue([sakura_frame, b1, sw1],
					 [kero_frame, b2, sw2], w, s_l, cg_l, r_l)
	gm = GhostManager()

	pbutton.connect('clicked', lambda x: gv.enqueue(random.choice(SCRIPT_PATTERNS),
													gm.nominate_ghost(random.choice(GHOST_PATTERNS)),
													random.choice(SENDER_PATTERNS),
													random.choice(CHANNEL_PATTERNS)))
	ibutton.connect('clicked', lambda x: gv.cancel())
	cbutton.connect('clicked', lambda x: gv.cancel_all())

	gtk.main()
