#!/usr/bin/env python
#
#  bottlelib.py - SSTP Bottle (SLPP 2.0) client library
#				 Tamito KAJIYAMA <28 May 2001>
#				 Atzm WATANABE   <01 Feb 2004>

import httplib
import mimetools
import os
import select
import socket
import string
import StringIO
import sys
import re

AGENT_NAME = os.path.basename(sys.argv[0])

HOST = "bottle.mikage.to"
PORT = 9871

class BottleClientError(Exception):
	def __init__(self, message, detail=None):
		if detail is None:
			self.message = message
		else:
			self.message = "%s (%s)" % (message, detail)
	def __str__(self):
		return self.message
error = BottleClientError

class BottleClient:
	MessageClass = mimetools.Message

	def __init__(self, progress, kanjierror="replace"):
		self.progress = progress
		self.stoc = lambda x: unicode(x, "sjis")
		self.ctos = lambda x: x.encode("sjis")
		self.luid = ""
		self.initialized = self.connected = 0
		self.sock_buffer = []
		self.debug = 0

	###   CGI   ###
	def send_cgi_request(self, request, action="/bottle2.cgi", response=1):
		self.progress.set_text("Now sending CGI request...")
		try:
			http = httplib.HTTP(HOST)
			http.putrequest("POST", action)
			http.putheader("Host", HOST)
			http.putheader("From", AGENT_NAME)
			http.putheader("Accept", "text/html, text/plain")
			self.progress.set_fraction(self.progress.get_fraction() + 0.1)
			http.putheader("Content-type", "application/x-www-form-urlencoded")
			http.putheader("Content-length", str(len(request)))
			self.progress.set_fraction(self.progress.get_fraction() + 0.1)
			http.endheaders()
			self.progress.set_fraction(self.progress.get_fraction() + 0.1)
			http.send(request + "%0D%0AAgent%3A+" + AGENT_NAME)
			self.progress.set_fraction(self.progress.get_fraction() + 0.1)
		except socket.error, e:
			self.progress.set_fraction(0.0)
			self.progress.set_text("")
			raise BottleClientError("cannot send CGI request to the server",
									str(e))
		self.progress.set_fraction(self.progress.get_fraction() + 0.1)
		try:
			code, message, headers = http.getreply()
		except socket.error, e:
			self.progress.set_fraction(0.0)
			self.progress.set_text("Error: " + unicode(e[1], "euc-jp"))
			return

		self.progress.set_fraction(self.progress.get_fraction() + 0.1)
		if code != 200:
			self.progress.set_fraction(0.0)
			self.progress.set_text("")
			raise BottleClientError("unable to post CGI request",
									"%d %s" % (code, message))
		if not response:
			self.progress.set_fraction(0.0)
			self.progress.set_text("")
			return 0
		self.progress.set_fraction(self.progress.get_fraction() + 0.1)
		self.headers = self.MessageClass(http.getfile(), 0)
		self.progress.set_fraction(self.progress.get_fraction() + 0.1)
		rawheaders = self.headers.headers[:]
		self.progress.set_fraction(self.progress.get_fraction() + 0.1)
		for k, v in self.headers.items():
			self.headers[k] = self.stoc(v)
		self.progress.set_fraction(self.progress.get_fraction() + 0.1)
		self.headers.headers = map(self.stoc, rawheaders)
		self.progress.set_text(self.progress.get_text() + " Done!")
		return (self.headers["Result"] != "OK")
	def get_new_id(self):
		self.send_cgi_request("Command%3A+getNewId")
		self.luid = self.headers["NewID"]
	def get_channels(self):
		self.send_cgi_request("Command%3A+getChannels")
		self.channels = []
		for i in range(int(self.headers["Count"])):
			prefix = "CH%d_" % (i + 1)
			self.channels.append({
				"name":   self.headers[prefix + "name"],
				"info":   self.headers[prefix + "info"],
				"ghost":  self.headers[prefix + "ghost"],
				"count":  int(self.headers[prefix + "count"]),
				"nopost": int(self.headers[prefix + "nopost"]),
				})
	def set_channels(self, channels):
		self.in_channels = channels # may be an empty list
		message = ["Command%3A+setChannels", "LUID%3A+" + self.luid]
		i = 1
		for n in channels:
			name = self.ctos(self.channels[n]["name"])
			message.append("Ch%d" % i + "%3A+" + self.encode(name))
			i = i + 1
		return self.send_cgi_request(string.join(message, "%0D%0A"))
	def send_broadcast(self, channel, script, ghost=None):
		if self.channels[channel]["nopost"]:
			raise BottleClientError("posting to this channel is forbidden")
		name = self.channels[channel]["name"]
		message = [
			"Command%3A+sendBroadcast",
			"LUID%3A+" + self.luid,
			"Channel%3A+" + self.encode(self.ctos(name)),
			"Talk%3A+" + self.encode(self.ctos(script)),
			]
		if ghost:
			message.append("Ghost%3A+" + self.encode(self.ctos(ghost)))
		return self.send_cgi_request(string.join(message, "%0D%0A"))
	def vote_message(self, mid, type="Vote"):
		message = [
			"Command%3A+voteMessage",
			"LUID%3A+" + self.luid,
			"MID%3A+" + mid,
			"VoteType%3A+" + type,
			]
		return self.send_cgi_request(string.join(message, "%0D%0A"))
	def send_ghost_names(self, namelist):
		message = ["CCC=%88%A4", "LUID=" + self.luid]
		for name in namelist:
			message.append("GHOST=" + self.encode(self.ctos(name)))
		return self.send_cgi_request(string.join(message, "&"),
									 "/glog/bottleglog.cgi", 0)
	def encode(self, s):
		return string.join(map(lambda c: '%%%0x' % ord(c), s), '')

	###   Socket   ###
	def connect(self):
		try:
			self.sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
			self.sock.connect((HOST, PORT))
		except socket.error, e:
			raise BottleClientError("cannot connect to the server", str(e))
	def makefile(self, blocking=1):
		if self.debug & 2:
			retry = 0
		while 1:
			data = self.sock.recv(1024)
			self.sock_buffer.append(data)
			if len(data) < 1024:
				data = string.join(self.sock_buffer, '')
				if data[-4:] == "\r\n\r\n" or data[-2:] == "\n\n":
					if self.debug & 2 and retry:
						print "DONE"
					self.sock_buffer = []
					break
				elif not blocking:
					data = ""
					break
				if self.debug & 2 and not retry:
					retry = 1
					print "RETRY", "*" * 70
		if self.debug & 1:
			open("debug.out", "a").write(data)
		return StringIO.StringIO(data)
	def getreply(self):
		self.file = self.makefile()
		line = self.file.readline()
		[version, code, message] = string.split(line, None, 2)
		if version[:5] != "HTTP/":
			raise BottleClientError("Bad status line")
		if version not in ["HTTP/1.0", "HTTP/1.1"]:
			raise BottleClientError("Unknown protocol", version)
		return int(code), message, self.MessageClass(self.file)
	def send_luid(self):
		self.sock.send("POST /\r\n\r\n" + self.luid + "\r\n")
		code, message, headers = self.getreply()
		if code != 200:
			raise BottleClientError("unable to send LUID to the server",
									"%d %s" % (code, message))
		message = self.get_slpp_message()
		if message is None:
			raise BottleClientError("no expected SLPP message")
		command, headers = message
		if command != "allUsers":
			raise BottleClientError("unexpected SLPP message", command)
		self.users = int(headers["Num"])
	def get_slpp_message(self): # read from self.file
		# skip empty lines
		while 1:
			line = self.file.readline()
			if not line or self.strip_newline(line):
				break
		if not line:
			return None
		command = self.strip_newline(line)
		headers = self.MessageClass(self.file)
		rawheaders = headers.headers[:]
		for k, v in headers.items():
			headers[k] = self.stoc(v)
		headers.headers = map(self.stoc, rawheaders)
		return command, headers
	def get_sstp_message(self): # read from self.file
		buffer = []
		while 1:
			line = self.file.readline()
			if not line:
				break
			buffer.append(line)
			if not self.strip_newline(line):
				break
		msg  = string.join(buffer, '')

		charset = "Shift_JIS" # Default
		mres = re.search("Charset: ?(.+?)\r\n", msg)
		if mres is not None:
			charset = mres.group(1)

		result = ""
		try:
			result = unicode(msg, charset)
		except UnicodeDecodeError:
			print "Received illegal script (containing model dependence char?) ... ignoring."
			result = None

		return result
	def send_sstp_message(self, message, host="", port=9801):
		buffer = []
		message = message.encode("sjis")
		s = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
		try:
			s.connect((host, port))
			s.send(message)
			while 1:
				buffer.append(s.recv(1024))
				if not buffer[-1]:
					break
			s.close()
		except socket.error, (code, message):
			return None
		return string.join(buffer, '')
	def strip_newline(self, line):
		if line[-2:] == "\r\n":
			return line[:-2]
		elif line[-1:] == "\n":
			return line[:-1]
		else:
			return line

	###   API   ###
	def init(self):
		if not self.luid:
			self.get_new_id()
		self.connect()
		self.send_luid()
		self.get_channels()
		self.initialized = 1
	def join(self, channels):
		error = self.set_channels(channels)
		if not error:
			self.connected = 1
		return error
	def close(self):
		if not self.connected:
			return
		self.sock.close()
		self.initialized = self.connected = 0
	def run_forever(self):
		while 1:
			self.handle_event(None)
	def handle_event(self, timeout=0):
		r, w, e = select.select([self.sock], [], [], timeout)
		if not r:
			return 0
		# handle a series of messages
		self.file = self.makefile(0)
		handled = 0
		while 1:
			message = self.get_slpp_message()
			if message is None:
				break
			self.command, self.headers = message
			sstp_message = self.get_sstp_message()
			if self.command == "unicastMessage":
				if sstp_message is not None:
					self.handle_sstp_message(sstp_message, 1, 0)
			elif self.command == "broadcastMessage":
				if sstp_message is not None:
					self.handle_sstp_message(sstp_message, 0, 0)
			elif self.command == "forceBroadcastMessage":
				if sstp_message is not None:
					self.handle_sstp_message(sstp_message, 0, 1)
					message = self.headers.get("DialogMessage")
					if message:
						self.handle_dialog_message(message)
			elif self.command == "broadcastInformation":
				self.handle_broadcast_information(self.headers["Type"], 0)
			elif self.command == "forceBroadcastInformation":
				self.handle_broadcast_information(self.headers["Type"], 1)
			elif self.command == "allUsers":
				self.users = int(self.headers["Num"])
				self.handle_all_users()
			elif self.command == "channelUsers":
				name = self.headers["Channel"]
				for i in range(len(self.channels)):
					if self.channels[i]["name"] == name:
						self.channels[i]["count"] = int(self.headers["Num"])
						self.handle_channel_users(i)
			elif self.command == "closeChannel":
				name = self.headers["Channel"]
				for n in range(len(self.channels)):
					if self.channels[n]["name"] == name:
						del self.channels[n]
						if n in self.in_channels:
							self.in_channels.remove(n)
						for i in range(len(self.in_channels)):
							if self.in_channels[i] > n:
								self.in_channels[i] = self.in_channels[i] - 1
						self.handle_close_channel(name)
						break
			elif self.command == "channelList":
				self.handle_channel_list()
			else:
				raise BottleClientError("unexpected SLPP message",
										self.command)
			handled = 1
		return handled
	# callback functions
	def handle_sstp_message(self, message, unicast, forced):
		if message:
			self.send_sstp_message(message)
	def handle_dialog_message(self, message):
		pass
	def handle_broadcast_information(self, type, forced):
		pass
	def handle_all_users(self):
		pass
	def handle_channel_users(self, channel):
		pass
	def handle_close_channel(self, name):
		pass
	def handle_channel_list(self):
		pass

class BottleDumper(BottleClient):
	def handle_sstp_message(self, message, unicast, forced):
		print self.command
		print str(self.headers),
		if not forced or message:
			print self.stoc(message), # assume Charset: Shift_JIS
	def handle_broadcast_information(self, type, forced):
		print self.command
		print str(self.headers),
	def handle_all_users(self):
		print "Users:", self.users
	def handle_channel_users(self, n):
		print "Users in Channel #%d: %d" % (n, self.channels[n]["count"])
	def handle_close_channel(self, name):
		print "Closed the \"%s\" channel" % name
	def handle_channel_list(self):
		print self.command
		print string.join(self.headers.headers, ''),

USAGE = """\
Usage:
  %s [-d] [-k name] channels ...
  %s -c channel [-g ghost] script ...
  %s -l
Options:
  -d, --dump			run in the dumper mode (default: the forwarder mode)
  -k, --kanjicode NAME  the internal kanji code (default: EUC-JP)
  -c, --channel NUM	 a channel to which the script is sent
  -g, --ghost NAME	  a ghost name (to be sent by the IfGhost: header)
  -l, --list-channels   show a list of channels
  -D, --debug NUM	   debug code (default: 0)
""" % (AGENT_NAME, AGENT_NAME, AGENT_NAME)

def test():
	import getopt
	try:
		options, rest = getopt.getopt(sys.argv[1:], "c:g:dk:lD:", [
			"channel=", "ghost=", "dump", "kanjicode=", "list-channels",
			"debug="])
	except getopt.error, message:
		sys.stderr.write(USAGE)
		sys.exit(1)
	dumper = 0
	kanjicode = "EUC-JP"
	channel = None
	ghost = None
	list_channels = 0
	debug = 0
	for opt, val in options:
		if opt in ["-d", "--dump"]:
			dumper = 1
		elif opt in ["-k", "--kanjicode"]:
			kanjicode = val
		elif opt in ["-c", "--channel"]:
			channel = int(val)
		elif opt in ["-g", "--ghost"]:
			ghost = val
		elif opt in ["-l", "--list-channels"]:
			list_channels = 1
		elif opt in ["-D", "--debug"]:
			debug = int(val)
	if dumper:
		client = BottleDumper(kanjicode)
	else:
		client = BottleClient(kanjicode)
	client.debug = debug
	client.luid = os.environ.get("LUID", "")
	client.init()
	if list_channels or dumper:
		print "LUID:", client.luid
		print "Users:", client.users
	if list_channels:
		perm = {0: "", 1: " (receive only)"}
		for i in range(len(client.channels)):
			dict = client.channels[i]
			print "%d: %s%s" % (i, dict["name"], perm[dict["nopost"]])
			print "  ", dict["info"]
		return
	if channel is not None:
		if channel < 0 or channel >= len(client.channels):
			sys.stderr.write("Error: channel number out of range.\n")
			sys.exit(1)
		if client.send_broadcast(channel, string.join(rest), ghost):
			sys.stderr.write("Error: %s\n" % client.headers["ExtraMessage"])
			sys.exit(1)
		print client.headers["Result"]
		return
	client.join(map(int, rest))
	client.run_forever()

if __name__ == "__main__":
	test()
