#!/usr/bin/env python
# -*- coding: ascii -*-
#
#  gbottler.py - a SSTP Bottle Client
#  Copyright (C) 2001, 2002 by Tamito KAJIYAMA
#  Copyright (C) 2004 by Atzm WATANABE <sitosito@p.chan.ne.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#
# $Id: gbottler.py,v 1.92 2004/05/25 10:46:24 atzm Exp $
#

import os
import sys
import time
import string
import re
import mimetools
import StringIO
import gettext
import getopt
from types import *

import gtk

from bottlelib    import BottleClient, BottleClientError
from logs         import LogWindow
from phrases      import PhraseManager, PhraseWindow
from prefs        import PreferenceManager, PreferenceDialog
from textmanager  import TextManager
from downloader   import Downloader
from queryluid    import LUIDInfo
from bottleparser import *

from common import *

if os.name != 'posix':
	# FIXME /-P
	os.environ['LANG'] = 'ja'
	gettext.install('gbottler', 'locale')
else:
	gettext.install('gbottler')

class Bottler(BottleClient):
	def __init__(self, app):
		BottleClient.__init__(self)
		self.app = app

		self.downloader = Downloader(verbose=True, dic=self.app.prefs.rc,
									 logging=self.app.prefs.get('download_log'))

	def set_pattern(self, dic):
		self.downloader.set_pattern(dic)

	def set_dl_logging(self, logging):
		self.downloader.set_logging(logging)

	def handle_sstp_message(self, message, unicast, forced):
		# dispatch SSTP message to local SSTP server
		if not message:
			return

		# get unique values
		mid = self.headers.get('mid', '')
		sender = self.headers.get('sender', 'SSTP Bottle')
		receive_time = time.localtime(float(eval("0x%s" % mid[:8])))
		if unicast:
			channel = 'unicast'
		elif forced:
			channel = 'broadcast'
		else:
			channel = self.headers.get('channel', '')

		# parse SSTP message
		file = StringIO.StringIO(message)
		request = file.readline()
		headers = mimetools.Message(file)

		try:
			script = headers["script"]
		except KeyError:
			return
		charset = headers.get("charset", "Shift_JIS")
		ifghost = headers.get('ifghost', 'default')

		# forward local SSTP server?
		if self.app.prefs.get('forward'):

			# Forward all messages?
			if self.app.prefs.get('accept_all'):
				self.app.send_local_message(channel, ifghost, script, sender, charset)

			# Forward only installed ghosts?
			elif self.app.prefs.get('forward_listed'):
				if ifghost in self.app.local_ghost:
					self.app.send_local_message(channel, ifghost, script, sender, charset)

		# log message
		self.app.log_window.update(mid, channel, ifghost, script, receive_time)

		# download
		if self.app.prefs.get('autodl'):
			self.downloader.get_files(self.downloader.find_url_from_script(script))

	def handle_dialog_message(self, message):
		self.app.monitor_clear()
		self.app.monitor_insert(message)

	def handle_broadcast_information(self, type, forced):
		BottleClient.handle_broadcast_information(self, type, forced)
		if type in [VOTE, AGREE]:
			self.app.log_votes(self.headers["mid"], type, int(self.headers["num"]))

	def handle_all_users(self):
		self.app.show_users()

	def handle_channel_users(self, channel):
		self.app.show_users_of(channel)

	def handle_close_channel(self, channel):
		self.app.close_channel(channel)

class Application:
	DEFAULT_TARGET = unicode(_("default"), "utf-8")
	to_sjis = lambda x: x.encode("sjis")
	sjis_to = lambda x: unicode(x, "sjis")

	def __init__(self, host="localhost", port=9801):
# === Definition of Basic Const === #
		self.host = host
		self.port = port

		self.prefs        = PreferenceManager()
		self.parser       = BottleParser('strict') # using check script
		self.color_parser = BottleParser('loose')  # using coloring

		accel_group = gtk.AccelGroup()

# === Create Base Window === #
		self.main_vbox = gtk.VBox(gtk.FALSE, 0)
		self.main_vbox.show()

		self.window = gtk.Window(gtk.WINDOW_TOPLEVEL)
		self.window.connect("destroy", self.quit, "WM destroy")
		self.window.set_title(unicode(_(APP), "utf-8"))
		self.window.add_accel_group(accel_group)
		self.window.add(self.main_vbox)

# === Create Menubar === #
		self.set_self_messages()
		menu_items = self.get_menu_items()
		self.item_factory = gtk.ItemFactory(gtk.MenuBar, "<main>", accel_group)
		self.item_factory.create_items(menu_items)

		self.menu_list = {}
		for m in menu_items:
			# keeping ref to each items
			key = re.sub("_", "", m[0])
			self.menu_list[key] = self.item_factory.get_item(key)

		for m in ["/%s/%s" % (self.msg["file"][1], self.msg["part"]),
				  "/%s" % self.msg["channel"][1],
				  "/%s" % self.msg["ghost"][1]]:
			# part, channel and ghost are not sensitive at startup
			self.menu_list[m].set_sensitive(gtk.FALSE)			

		self.menubar = self.item_factory.get_widget("<main>")
		self.menubar.show()

# === Create Main Editor === #
		self.main_text = gtk.TextView()
		self.main_text.show()
		self.main_text.set_size_request(500,100)
		self.main_text.set_wrap_mode(gtk.WRAP_CHAR)

		self.textmanager = TextManager(self, self.main_text, self.prefs)
		self.textmanager.set_style_script()

		self.editor = self.main_text.get_buffer()
		self.editor.connect('end-user-action', self.editor_modified)

		sw = gtk.ScrolledWindow()
		sw.show()
		sw.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
		sw.add(self.main_text)

		self.frame = gtk.Frame()
		self.frame.show()
		self.frame.set_shadow_type(gtk.SHADOW_IN)
		self.frame.add(sw)

# === Create Buttons === #
		self.tooltips = gtk.Tooltips()

		checkButton = get_icon_button(gtk.STOCK_SPELL_CHECK,
									  gtk.ICON_SIZE_MENU)
		testButton = get_icon_button(gtk.STOCK_REFRESH,
									 gtk.ICON_SIZE_MENU)
		self.send_button = get_icon_button(gtk.STOCK_OK,
										   gtk.ICON_SIZE_MENU)
		self.clear_button = get_icon_button(gtk.STOCK_CLEAR,
											gtk.ICON_SIZE_MENU)

		checkButton.connect("clicked", self.check, None)
		testButton.connect("clicked", self.test, None)
		self.send_button.connect("clicked", self.send_question, None)
		self.clear_button.connect("clicked", self.edit_clear, None)

		checkButton.connect("focus-in-event", lambda x, y: self.main_text.grab_focus())
		testButton.connect("focus-in-event", lambda x, y: self.main_text.grab_focus())
		self.send_button.connect("focus-in-event", lambda x, y: self.main_text.grab_focus())
		self.clear_button.connect("focus-in-event", lambda x, y: self.main_text.grab_focus())

		self.tooltips.set_tip(checkButton, unicode(_("Check"), "utf-8"))
		self.tooltips.set_tip(testButton, unicode(_("Test"), "utf-8"))
		self.tooltips.set_tip(self.send_button, unicode(_("Send"), "utf-8"))
		self.tooltips.set_tip(self.clear_button, unicode(_("Clear"), "utf-8"))

		self.send_button.set_sensitive(gtk.FALSE) # at startup

# === Create Labels, e.g: channels, ghosts and users === #
		channelLabel = gtk.Label(unicode(_("Channel") + ": ", "utf-8"))
		channelLabel.show()

		self.channel = gtk.Label()
		self.channel.show()
		self.c_e_box = gtk.EventBox()
		self.c_e_box.show()
		self.c_e_box.set_events(gtk.gdk.BUTTON_PRESS_MASK)
		self.c_e_box.add(self.channel)

		self.channel_ghost = gtk.Label()
		self.channel_ghost.show()
		self.g_e_box = gtk.EventBox()
		self.g_e_box.show()
		self.g_e_box.set_events(gtk.gdk.BUTTON_PRESS_MASK)
		self.g_e_box.add(self.channel_ghost)

		usersLabel = gtk.Label(unicode(_("Users") + ": ", "utf-8"))
		usersLabel.show()
		self.users = gtk.Label()
		self.users.show()

# === Bring Buttons and Labels together to HBox, Frame === #
		self.toolbar = gtk.HBox(gtk.FALSE, 1)
		self.toolbar.show()

		self.toolbar.pack_start(checkButton, gtk.FALSE, gtk.TRUE, 0)
		self.toolbar.pack_start(testButton, gtk.FALSE, gtk.TRUE, 0)
		self.toolbar.pack_start(self.send_button, gtk.FALSE, gtk.TRUE, 0)
		self.toolbar.pack_start(self.clear_button, gtk.FALSE, gtk.TRUE, 0)

		vs = gtk.VSeparator()
		vs.show()
		self.toolbar.pack_start(vs, gtk.FALSE, gtk.TRUE, 3)

		self.toolbar.pack_start(channelLabel, gtk.FALSE, gtk.TRUE, 0)
		self.toolbar.pack_start(self.c_e_box, gtk.FALSE, gtk.TRUE, 0)
		self.toolbar.pack_start(self.g_e_box, gtk.FALSE, gtk.TRUE, 0)

		vs = gtk.VSeparator()
		vs.show()
		self.toolbar.pack_start(vs, gtk.FALSE, gtk.TRUE, 3)

		self.toolbar.pack_start(usersLabel, gtk.FALSE, gtk.TRUE, 0)
		self.toolbar.pack_start(self.users, gtk.FALSE, gtk.TRUE, 0)

		self.toolframe = gtk.Frame()
		self.toolframe.show()
		self.toolframe.set_border_width(0)
		self.toolframe.set_shadow_type(gtk.SHADOW_OUT)
		self.toolframe.add(self.toolbar)

# === Create Popup Menu from Main Editor === #
		pmsg = self.get_popup_menu_messages()
		popup_menu_items = self.get_popup_menu_items(pmsg)

		self.popup_item_factory = gtk.ItemFactory(gtk.Menu, "<main>", accel_group)
		self.popup_item_factory.create_items(popup_menu_items)
		self.popup_menu = self.popup_item_factory.get_widget("<main>")

		self.main_text.connect("button-press-event", self.popup)

# === Create Progressbar === #
		self.pbar = gtk.ProgressBar()
		self.pbar.show()
		self.pbar.set_size_request(380, 20)

		self.pframe = gtk.Frame()
		self.pframe.show()
		self.pframe.set_shadow_type(gtk.SHADOW_IN)
		self.pframe.set_border_width(0)
		self.pframe.add(self.pbar)

# === Create Statusbar as window resize grip === #
		self.sbar = gtk.Statusbar()
		self.sbar.show()
		self.sbar.set_has_resize_grip(gtk.TRUE)

# === Bring Progressbar and Statusbar together to HBox === #
		self.barbox = gtk.HBox()
		self.barbox.show()
		self.barbox.pack_start(self.pframe)
		self.barbox.pack_start(self.sbar)

# === Bring Main Editor and Progressbar together to VPaned === #
		self.paned = gtk.VPaned()
		self.paned.show()
		self.paned.pack1(self.frame, gtk.TRUE)
		self.paned.pack2(self.barbox, gtk.FALSE, gtk.FALSE)

# === Bring all items together to Main Window Finally === #
		self.main_vbox.pack_start(self.menubar, gtk.FALSE, gtk.TRUE, 0)
		self.main_vbox.pack_start(self.toolframe, gtk.FALSE, gtk.TRUE, 0)
		self.main_vbox.pack_start(self.paned, gtk.TRUE, gtk.TRUE, 0) # main editor and progressbar

		self.window.show()
		self.main_text.grab_focus()

# === Initialization of Application === #
		[str, pos] = self.get_initial_script()
		self.textmanager.insert_with_color_from(str, pos)
		self.editor_modified()

		self.client = Bottler(self)
		self.client.debug = 2

		self.luid_info = None
		try:
			luid = open(os.path.join(open_bottlecase(), "luid")).read()
		except IOError:
			luid = None
		if luid and len(luid) == 82:
			self.client.luid = luid
			self.luid_info = LUIDInfo(luid)

		self.log_window = LogWindow(self)
		self.phrase_manager = PhraseManager()
		self.phrase_window = PhraseWindow(self, self.phrase_manager)

		self.menu_list["/%s/%s" % (self.msg["window"][1],
								   self.msg["logs"])].connect("activate", self.log_window.open, None)

		self.menu_list["/%s/%s" % (self.msg["window"][1],
								   self.msg["phrases"])].connect("activate", self.phrase_window.open, None)

		self.about = None

		self.execute_command = 'GetNames'
		self.target_ghost = ""
		self.sakuraname = ""

		self.local_ghost = []
		self.local_ghost_last_notified = 0
		self.send_ghost_names()

		self.c_e_id = 1 # channel label eventbox handler id
		self.g_e_id = 1
		self.disable_channel()

		self.timeout_id = None

# === Initialization from Preferences === #
		if self.prefs.get('auto_open_log'):
			self.log_window.open(None, None)

		if self.prefs.get('auto_connect'):
			self.join(None, None)

	def get_initial_script(self):
		scr = self.prefs.get('initialscript')
		n = string.find(scr, '|')

		# | founds
		if n != -1:
			n -= len(scr)

			# | founds at rightmost
			if n != -1:
				scr = scr[:n] + scr[n+1:]
			else:
				scr = scr[:n]

		return [scr, n]

	def editor_modified(self, widget=None, event=None):
		self.textmanager.coloring(widget, event)
		[si, ei] = self.editor.get_bounds()
		script = self.editor.get_text(si, ei)
		self.sbar.pop(0)
		self.sbar.push(0, str(len(script.encode('sjis'))) + unicode(_('bytes'), 'utf-8'))

	def edit_cut(self, widget, data=None):
		self.main_text.emit("cut-clipboard")

	def edit_copy(self, widget, data=None):
		self.main_text.emit("copy-clipboard")

	def edit_paste(self, widget, data=None):
		self.main_text.emit("paste-clipboard")

	def edit_delete(self, widget, data=None):
		self.main_text.emit("delete-from-cursor", gtk.DELETE_CHARS, 1)

	def open_preference(self, widget=None, data=None):
		PreferenceDialog(self, self.prefs, self.luid_info)

	def notify_log_current_tab_closed(self):
		if self.prefs.get('logging_current_tab_closed'):
			self.log_window.autologging(self.prefs.get('logging_path'))

	def notify_preferences_changed(self, prefs):
		# notify BottleClient
		self.client.set_pattern(prefs)
		self.client.set_dl_logging(prefs['download_log'])

		# notify LogWindow
		self.log_window.tab_pos_changed(int(prefs['logtabpos']))
		self.log_window.color_changed(prefs['script'], prefs['sakura'], prefs['kero'],
									  prefs['sync'], prefs['error'], prefs['URL'])

		# notify self attributes
		self.textmanager.color_changed(prefs['script'], prefs['sakura'], prefs['kero'],
									   prefs['sync'], prefs['error'], prefs['URL'])
		self.textmanager.coloring()

 	def popup(self, widget=None, event=None):
 		if event.button == 3:
 			self.popup_menu.popup(None, widget, None, event.button, 0)

	def join(self, widget=None, data=None):
		if not self.client.initialized:
			try:
				self.client.init()
				self.luid_info = LUIDInfo(self.client.luid)
			except BottleClientError, e:
				return
			filename = os.path.join(open_bottlecase(), "luid")
			try:
				open(filename, "w").write(self.client.luid)
			except IOError:
				self.monitor_insert("\n" + unicode(_("Warning"), "utf-8") + \
									": cannot write " + filename)

		if self.prefs.get('auto_join'):
			channels = range(len(self.client.channels))
		else:
			d = ChannelDialog(self.window, self.client.channels)
			d.run()
			channels = d.selection
			d.destroy()

		self.pbar.set_fraction(0.0)
		self.monitor_clear()

		if channels is None:
			return
		if self.client.join(channels):
			return

		self.pbar.set_fraction(0.0)
		self.monitor_clear()
		self.enable_channel()
		self.timeout_id = gtk.timeout_add(100, self.handle_event)

	def button_popup(self, widget, event, sender):
		if event.button in [1, 3]:
			sender.popup(None, widget, None, event.button, event.time)

	def enable_channel(self):
		channel_item = self.menu_list["/%s" % self.msg["channel"][1]]
		channel_item.deselect()
		channel_item.set_sensitive(gtk.FALSE)
		channel_item.remove_submenu()

		channels = []
		for n in self.client.in_channels:
			name = self.client.channels[n]["name"]
			channels.append(name)
			if not self.channel.get_text():
				self.channel.set_text(name)

		channel_menu = []
		for n in range(len(channels)):
			if n == 0:
				channel_menu.append(("/"+channels[n], None, None, 0, "<RadioItem>"))
			else:
				channel_menu.append(("/"+channels[n], None, None, 0, "/"+channels[n-1]))
		channel_menu = tuple(channel_menu)

		self.channel_factory = gtk.ItemFactory(gtk.Menu, "<main>")
		self.channel_factory.create_items(channel_menu)

		for n in range(len(channels)):
			self.channel_factory.get_item(channel_menu[n][0]).connect("toggled",
																	  self.update_channel,
																	  channels[n])

		channel_item.set_submenu(self.channel_factory.get_widget("<main>"))
		channel_item.set_sensitive(gtk.TRUE)

		if not self.c_e_box.handler_is_connected(self.c_e_id):
			self.c_e_id = self.c_e_box.connect("button-press-event", self.button_popup,
											   self.channel_factory.get_widget("<main>"))

		self.rebuild_ghostmenu()
		self.menu_list["/%s/%s" % (self.msg["file"][1], self.msg["part"])].set_sensitive(gtk.TRUE)
		self.menu_list["/%s/%s" % (self.msg["file"][1], self.msg["join"])].set_sensitive(gtk.FALSE)
		self.log_window.join()
		self.show_users()

	def update_channel(self, widget=None, data=None):
		self.channel.set_text(data)
		self.show_users_of(self.get_channel_id())
		self.check_channel_permission()

	def close_channel(self, name):
		if self.channel.get_text() == name:
			self.channel.set_text("")
		self.enable_channel()
		self.monitor_clear()
		self.monitor_insert(name + unicode(_('channel was closed.'), 'utf-8'))

	def rebuild_ghostmenu(self):
		ghost_item = self.menu_list["/%s" % self.msg["ghost"][1]]
		ghost_item.deselect()
		ghost_item.set_sensitive(gtk.FALSE)

		if self.g_e_box.handler_is_connected(self.g_e_id):
			self.g_e_box.disconnect(self.g_e_id)

		current = self.target_ghost
		ghosts = [self.DEFAULT_TARGET] + self.local_ghost
		name = ""
		for n in range(len(self.client.channels)):
			name = self.client.channels[n]["ghost"]
			if name and name not in ghosts:
				ghosts.append(name)

		ghost_menu = []
		for n in range(len(ghosts)):
			if n == 0:
				ghost_menu.append(("/"+ghosts[n], None, None, 0, "<RadioItem>"))
			else:
				ghost_menu.append(("/"+ghosts[n], None, None, 0, "/"+ghosts[n-1]))
		ghost_menu = tuple(ghost_menu)

		self.g_ifact = gtk.ItemFactory(gtk.Menu, "<main>")
		self.g_ifact.create_items(ghost_menu)

		all_ghost_items = {}
		for n in range(len(ghosts)):
			all_ghost_items[ghosts[n]] = self.g_ifact.get_item(ghost_menu[n][0])
			all_ghost_items[ghosts[n]].connect("toggled", self.update_ghost, ghosts[n])

		# reselect ghost it's selected by user
		if current in ghosts:
			all_ghost_items[current].set_active(gtk.TRUE)
			all_ghost_items[current].toggled()
		else:
			if current != self.DEFAULT_TARGET and current != name:
				all_ghost_items[self.sakuraname].set_active(gtk.TRUE)
				all_ghost_items[self.sakuraname].toggled()
			else:
				all_ghost_items[self.DEFAULT_TARGET].set_active(gtk.TRUE)
				all_ghost_items[self.DEFAULT_TARGET].toggled()

		ghost_item.set_submenu(self.g_ifact.get_widget("<main>"))
		ghost_item.set_sensitive(gtk.TRUE)

		if not self.g_e_box.handler_is_connected(self.g_e_id):
			self.g_e_id = self.g_e_box.connect("button-press-event",
											   self.button_popup, self.g_ifact.get_widget("<main>"))
		self.check_channel_permission()

	def update_ghost(self, widget=None, data=None):
		self.target_ghost = data
		self.channel_ghost.set_text("(" + self.target_ghost + ")")

	def check_channel_permission(self):
		channel = self.get_channel_id()
		if self.client.channels[channel]["nopost"]:
			self.send_button.set_sensitive(gtk.FALSE)
		else:
			self.send_button.set_sensitive(gtk.TRUE)

	def handle_event(self):
		if not self.client.connected:
			return
		handled = 0
		try:
			handled = self.client.handle_event()
		except BottleClientError, e:
			self.client.close()
			self.disable_channel()
			return

		now = time.time()
		if now - self.local_ghost_last_notified > 30: # about 30 seconds
			self.local_ghost_last_notified = now
			self.send_ghost_names()
			self.rebuild_ghostmenu()
		self.timeout_id = gtk.timeout_add(1000, self.handle_event)

	SSTP_EXECUTE = (
		"EXECUTE SSTP/1.0\r\n"
		"Sender: %s\r\n"
		"Command: %s\r\n"
		"Option: notranslate\r\n"
		"Charset: Shift_JIS\r\n"
		"\r\n")
	def get_execute(self, command):
		if command == 'GetNames':
			default_returns = []
		else:
			default_returns = [None, None]
			
		try:
			data = self.client.send_sstp_message(self.SSTP_EXECUTE % (APP, command),
												 self.host, self.port)
		except BottleClientError, e:
			return default_returns

		file = StringIO.StringIO(data)
		request = file.readline()

		if request[:12] != "SSTP/1.0 200":
			return default_returns

		names = []
		charset = ''
		while True:
			line = file.readline()
			if not line:
				break
			line = string.strip(line, '\r\n\r\n')
			if not line:
				continue
			elif string.lower(line[:9]) == 'charset: ':
				charset = line[9:]
				continue
			names.append(line)

		if not charset:
			charset = 'sjis'

		for i in range(len(names)):
			names[i] = unicode(names[i], charset)

		if command == 'GetNames':
			return names
		else:
			return re.split('\s*,\s*', names[0], 1)

	def send_ghost_names(self):
		ghosts =  self.get_execute('GetNames')
		sakura, kero = self.get_execute('GetName')

		if ghosts:
			self.local_ghost = ghosts
			self.sakuraname  = sakura
		elif sakura:
			self.local_ghost = [sakura]
			self.sakuraname  = sakura
		else:
			self.local_ghost = []
			self.sakuraname  = self.DEFAULT_TARGET

	def monitor_clear(self):
		self.pbar.set_text("")

	def monitor_insert(self, text):
		setted = ""
		if self.pbar.get_text():
			setted = self.pbar.get_text()
		self.pbar.set_text(setted + text)

	def disable_channel(self):
		self.menu_list["/%s/%s" % (self.msg["file"][1], self.msg["part"])].set_sensitive(gtk.FALSE)
		self.menu_list["/%s/%s" % (self.msg["file"][1], self.msg["join"])].set_sensitive(gtk.TRUE)
		self.log_window.part()

		if self.c_e_box.handler_is_connected(self.c_e_id):
			self.c_e_box.disconnect(self.c_e_id)
			self.c_e_id = 1
		if self.g_e_box.handler_is_connected(self.g_e_id):
			self.g_e_box.disconnect(self.g_e_id)
			self.g_e_id = 1

		self.send_button.set_sensitive(gtk.FALSE)
		self.channel.set_text("")
		self.channel_ghost.set_text("")
		self.target_ghost = self.DEFAULT_TARGET
		self.users.set_text("")

		for m in ["/%s" % self.msg["ghost"][1],
				  "/%s" % self.msg["channel"][1]]:
			menu = self.menu_list[m]
			menu.deselect()
			menu.set_sensitive(gtk.FALSE)
			menu.remove_submenu()

	def get_channel_id(self):
		name = self.channel.get_text()
		for id in self.client.in_channels:
			if self.client.channels[id]["name"] == name:
				return id
		raise RuntimeError, "unknown channel name (%s)" % name

	def show_users(self):
		channel = self.get_channel_id()
		count = self.client.channels[channel]["count"]
		self.users.set_text("%d / %d " % (count, self.client.users))

	def show_users_of(self, channel):
		count = self.client.channels[channel]["count"]
		self.users.set_text("%d / %d " % (count, self.client.users))

	def part(self, data, widget):
		if not self.client.connected:
			return
		if self.timeout_id:
			gtk.timeout_remove(self.timeout_id)
			self.timeout_id = None
		self.client.close()
		self.monitor_clear()
		self.monitor_insert(unicode(_("Connection closed."), "utf8"))
		self.disable_channel()

		if self.prefs.get('logging'):
			self.log_window.autologging(self.prefs.get('logging_path'))

	def quit(self, data, widget):
		self.part(data, widget)
		gtk.mainquit()

	def edit_clear(self, widget=None, data=None):
		[script, pos] = self.get_initial_script()

		self.edit_clear_all(widget, data)
		self.textmanager.insert_with_color_from(script, pos)
		self.sbar.pop(0)
		self.sbar.push(0, str(len(script.encode('sjis'))) + unicode(_('bytes'), 'utf-8'))

	def edit_clear_all(self, widget=None, data=None):
		self.monitor_clear()
		self.editor.set_text('')
		self.sbar.pop(0)
		self.sbar.push(0, '0' + unicode(_('bytes'), 'utf-8'))

	def check(self, widget=None, data=None):
		self.main_text.grab_focus()
		self.monitor_clear()
		self.monitor_insert(unicode(_("Now checking script..."), "utf8"))
		[si, ei] = self.editor.get_bounds()
		lines = string.split(self.editor.get_text(si, ei), "\n")

		error = 0
		buffer = []
		i = 1.0 / len(lines)
		for n in range(len(lines)):
			if not lines[n]:
				break
			lines[n] = self.check_line(lines[n], n)
			if lines[n] is None:
				error = 1
				break
			buffer.extend(lines[n])
			self.pbar.set_fraction(self.pbar.get_fraction() + i)
		if error or self.check_tag_abuse(buffer):
			return 1
		else:
			self.pbar.set_fraction(0.0)
			self.monitor_insert(unicode(_("No error."), "utf8"))
			return 0

	def check_line(self, script, lineno):
		try:
			return self.parser.parse(script)
		except ParserError, e:
			self.pbar.set_fraction(0.0)
			self.monitor_insert(unicode(_("Error"), "utf8"))

			d = gtk.Dialog(title="Script Parser Error", parent=self.window, flags=1)
			d.set_has_separator(gtk.FALSE)
			d.connect("destroy", lambda w,data:d.destroy(), "WM destroy")
			d.vbox.set_border_width(10)

			h = gtk.HBox()
			i = gtk.Image()
			i.set_from_stock(gtk.STOCK_DIALOG_ERROR, gtk.ICON_SIZE_LARGE_TOOLBAR)
			i.show()
			h.pack_start(i, gtk.FALSE, gtk.TRUE, 0)
			l = gtk.Label("Script Parser Error")
			l.show()
			h.pack_start(l, gtk.FALSE, gtk.TRUE, 0)
			h.show()
			d.vbox.add(h)
			s = gtk.HSeparator()
			s.show()
			d.vbox.add(s)

			s = gtk.ScrolledWindow()
			s.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
			t = gtk.TextView()
			t.set_cursor_visible(gtk.TRUE)
			t.set_size_request(300, 100)
			b = t.get_buffer()
			b.set_text('')
			tag = b.create_tag()
			tag.set_property('font', 'Monospace')

			b.insert_with_tags(b.get_end_iter(), unicode(_("Line"), "utf8") + \
							   " %d: %s\n%s\n" % (lineno, e.message, script) +
							   " " * (e.column) + "^" * (e.length or 1), tag)

			t.set_editable(gtk.FALSE)
			t.set_wrap_mode(gtk.WRAP_CHAR)
			s.add(t)

			f = gtk.Frame()
			f.set_shadow_type(gtk.SHADOW_IN)
			f.add(s)
			f.show()

			d.vbox.add(f)

			ok = gtk.Button(_("Close"), gtk.STOCK_CLOSE)
			ok.connect("clicked", lambda w,data:d.destroy(), None)
			ok.show()
			d.action_area.pack_start(ok, gtk.TRUE, gtk.TRUE, 0)

			t.show()
			s.show()
			d.show()

			return None

	RESERVED_SURFACES = (0, 1, 2, 3, 4, 5, 6, 7, 8, 10, 11, 19, 20, 25)

	def check_tag_abuse(self, script):
		scope = error = None
		surface_usage = [0, 10]

		msg_or  = unicode(_("or"), "utf8")
		msg_tag = unicode(_("tag"), "utf8")
		for node in script:
			if node[0] == SCRIPT_TAG:
				name, args = node[1], node[2:]
				if name == r"\h":
					scope = 0
				elif name == r"\u":
					scope = 1
				elif name == r"\s":
					try:
						id = int(args[0])
					except ValueError:
						error = unicode(_('bad use of surfaces'), 'utf-8')
						break
					if id in self.RESERVED_SURFACES:
						target = id / 10 % 2
					else:
						target = None
					if scope is None:
						if target is None:
							tag = r"\h %s \u %s" % (msg_or, msg_tag)
						elif target == 0:
							tag = r"\h %s" % msg_tag
						else:
							tag = r"\u %s" % msg_tag
						error = r"\s[%d] " % id + \
								unicode(_("is used without a preceding"), "utf8") + \
								" %s" % tag
						break
					if id in self.RESERVED_SURFACES:
						surface_usage[scope] = id
			elif node[0] == SCRIPT_TEXT:
				if scope is None:
					error = r"\h %s \u " % msg_or + \
							unicode(_("is needed before characters"), "utf8")
					break
		else:
			if surface_usage[0] / 10 % 2 == 1 or surface_usage[0] == 20 or \
			   surface_usage[1] / 10 % 2 == 0:
				error = unicode(_("bad use of surfaces"), "utf8")
		if error is not None:
			self.pbar.set_fraction(0.0)
			self.monitor_clear()
			self.monitor_insert(unicode(_("Error"), "utf8") + ": %s " % error)
			return 1
		return 0

	def test(self, widget, data):
		self.main_text.grab_focus()
		if self.check(widget, data):
			return
		self.send_local_message('[' + unicode(_("Test"), "utf-8") + ']', self.target_ghost, self.get_script())

	def send_question(self, widget, data):
		self.main_text.grab_focus()
		if self.check(widget, data):
			return

		d = get_simple_yes_or_no_dialog(unicode(_("Really?"), 'utf-8'),
										unicode(_("Send to "), "utf8") + \
										self.channel.get_text() + " / " + \
										self.target_ghost + unicode(_(", OK?"), "utf8"),
										self.window)
		d.show()
		res = d.run()
		d.destroy()

		if not res or res != gtk.RESPONSE_ACCEPT:
			return

		channel = self.get_channel_id()
		script = self.get_script()
		ghost = self.target_ghost
		if ghost == self.DEFAULT_TARGET:
			ghost = None
		gtk.timeout_add(0, self.send, widget, data, channel, script, ghost)

	def send(self, widget, data, channel, script, ghost):
		self.monitor_clear()
		self.monitor_insert(unicode(_("Broadcasting..."), "utf8"))
		try:
			if self.client.send_broadcast(channel, script, ghost):
				error = self.client.headers["ExtraMessage"]
			else:
				error = None
		except BottleClientError, e:
			error = str(e)

		if error is not None:
			self.monitor_insert(unicode(_("failed!!")) + "\n" + \
								unicode(_("Error"), "utf8") + ": ")
			self.monitor_insert(unicode(error))

		if self.prefs.get('auto_clear'):
			self.edit_clear(data, widget)

		self.pbar.set_fraction(0.0)

	def get_script(self):
		[si, ei] = self.editor.get_bounds()
		script = string.replace(self.editor.get_text(si, ei), "\n", "")

		if string.rstrip(script)[-2:] != r"\e":
			script = script + r"\e"
		return unicode(script, "utf-8")

	def about(self, data, widget):
		if self.about:
			return

		def aboutdestroy(widget, data):
			self.about.destroy()
			self.about = None

		self.about = gtk.Dialog(unicode(_("About"), "utf-8"), parent=self.window,
								flags=gtk.DIALOG_NO_SEPARATOR)
		self.about.connect("destroy", aboutdestroy, "WM destroy")
		self.about.set_border_width(10)

		h = gtk.HBox()
		i = gtk.Image()
		i.set_from_stock(gtk.STOCK_DIALOG_INFO, gtk.ICON_SIZE_LARGE_TOOLBAR)
		i.show()
		h.pack_start(i, gtk.FALSE, gtk.TRUE, 0)
		l = gtk.Label(unicode(_("About"), 'utf-8'))
		l.show()
		h.pack_start(l, gtk.FALSE, gtk.TRUE, 0)
		h.show()
		self.about.vbox.add(h)
		s = gtk.HSeparator()
		s.show()
		self.about.vbox.add(s)

		frame = gtk.Frame(label=" %s %s " % (APP, VER))
		l = gtk.Label("\n"+
					  " Copyright(C) 2001-'04, Tamito KAJIYAMA \n"+
					  " Copyright(C) 2003-'04, Atzm WATANABE \n")
		l.set_justify(gtk.JUSTIFY_RIGHT)
		l.show()
		frame.add(l)
		frame.show()
		self.about.vbox.add(frame)

		frame = gtk.Frame(label=unicode(_(" License "), "utf-8"))
		l = gtk.Label("\n"+
					  " GNU General Public License version 2 or later \n")
		l.set_justify(gtk.JUSTIFY_RIGHT)
		l.show()
		frame.add(l)
		frame.show()
		self.about.vbox.add(frame)

		b = gtk.Button(unicode(_("OK"), "utf8"), gtk.STOCK_OK)
		b.connect("clicked", aboutdestroy, None)
		b.show()
		self.about.action_area.pack_start(b, gtk.TRUE, gtk.TRUE, 0)

		self.about.show()

	def log_votes(self, mid, type, num):
		self.log_window.log_votes(mid, type, num)

	SSTP_SEND = (
		"SEND SSTP/1.4\r\n"
		"Sender: %s / %s / %s\r\n"
		"IfGhost: %s\r\n"
		"Script: %s\r\n"
		"Option: notranslate\r\n"
		"Charset: %s\r\n"
		"\r\n")
	def send_local_message(self, channel, ifghost, script, sender=APP, charset='Shift_JIS'):
		result = self.client.send_sstp_message(self.SSTP_SEND % (sender, channel, ifghost, ifghost, script, charset),
											   self.host, self.port)
		#print result

	def vote_message(self, mid, type):
		if self.client.vote_message(mid, type):
			open_error_dialog(self.client.headers["ExtraMessage"], self.log_window.window)

		self.monitor_clear()
		self.pbar.set_fraction(0.0)

	def insert(self, pre, post=""):
		self.main_text.emit("insert-at-cursor", pre)
		if post:
			self.main_text.emit("insert-at-cursor", post)
			self.main_text.emit("move-cursor", gtk.MOVEMENT_LOGICAL_POSITIONS,
								-1 * len(post), gtk.FALSE)

	def insert_h(self, data, widget): self.insert("\n\\h")
	def insert_u(self, data, widget): self.insert("\n\\u")
	def insert_s0(self, data, widget): self.insert(r"\s[0]")
	def insert_s1(self, data, widget): self.insert(r"\s[1]")
	def insert_s2(self, data, widget): self.insert(r"\s[2]")
	def insert_s3(self, data, widget): self.insert(r"\s[3]")
	def insert_s4(self, data, widget): self.insert(r"\s[4]")
	def insert_s5(self, data, widget): self.insert(r"\s[5]")
	def insert_s6(self, data, widget): self.insert(r"\s[6]")
	def insert_s7(self, data, widget): self.insert(r"\s[7]")
	def insert_s8(self, data, widget): self.insert(r"\s[8]")
	def insert_s20(self, data, widget): self.insert(r"\s[20]")
	def insert_s25(self, data, widget): self.insert(r"\s[25]")
	def insert_s10(self, data, widget): self.insert(r"\s[10]")
	def insert_s11(self, data, widget): self.insert(r"\s[11]")
	def insert_s19(self, data, widget): self.insert(r"\s[19]")
	def insert_username(self, data, widget): self.insert("%username")
	def insert_selfname(self, data, widget): self.insert("%selfname")
	def insert_selfname2(self, data, widget): self.insert("%selfname2")
	def insert_keroname(self, data, widget): self.insert("%keroname")
	def insert_url(self, data, widget): self.insert(r"\URL[http://", "]")
	def insert_c(self, data, widget): self.insert(r"\c")
	def insert_n(self, data, widget): self.insert("\\n\n")
	def insert_w2(self, data, widget): self.insert(r"\w2")
	def insert_w4(self, data, widget): self.insert(r"\w4")
	def insert_w6(self, data, widget): self.insert(r"\w6")
	def insert_w8(self, data, widget): self.insert(r"\w8")
	def insert_w10(self, data, widget): self.insert(r"\w8\w2")
	def insert_w12(self, data, widget): self.insert(r"\w8\w4")
	def insert_w18(self, data, widget): self.insert(r"\w9\w9")
	def insert_w20(self, data, widget): self.insert(r"\w8\w8\w4")
	def insert_w24(self, data, widget): self.insert(r"\w8\w8\w8")
	def insert_w30(self, data, widget): self.insert(r"\w9\w9\w9\w3")
	def insert_w40(self, data, widget): self.insert(r"\w9\w9\w9\w9\w4")
	def insert_w50(self, data, widget): self.insert(r"\w9\w9\w9\w9\w9\w5")
	def insert_w60(self, data, widget): self.insert(r"\w9\w9\w9\w9\w9\w9\w6")
	def insert_t(self, data, widget): self.insert(r"\t")
	def insert_q(self, data, widget): self.insert(r"\_q")
	def insert_s(self, data, widget): self.insert(r"\_s")
	def insert_e(self, data, widget): self.insert(r"\e")

	def main(self):
		gtk.main()
		return 0

	def get_popup_menu_messages(self):
		pmsg = {
     		"sessions":	    unicode(_("Sessions"), "utf-8"),
     		"timecritical": unicode(_("TimeCritical"), "utf-8"),
     		"quick":		unicode(_("Quick"), "utf-8"),
     		"syncronized":  unicode(_("Syncronized"), "utf-8"),
     
     		"sakura":       unicode(_("Sakura"), "utf-8"),
     		"countenance":  unicode(_("Countenance"), "utf-8"),
     		"ordinary":	    unicode(_("Ordinary"), "utf-8"),
     		"shy":		    unicode(_("Shy"), "utf-8"),
     		"surprised":	unicode(_("Surprised"), "utf-8"),
     		"angst":		unicode(_("Angst"), "utf-8"),
     		"ease off":	    unicode(_("Ease off"), "utf-8"),
     		"smile":		unicode(_("Smile"), "utf-8"),
     		"closed eyes":  unicode(_("Closed eyes"), "utf-8"),
     		"angry":		unicode(_("Angry"), "utf-8"),
     		"scorned":	    unicode(_("Scorned"), "utf-8"),
     		"signboard":	unicode(_("Signboard"), "utf-8"),
     		"singing":	    unicode(_("Singing"), "utf-8"),
     
     		"unyu":		    unicode(_("Unyu"), "utf-8"),
     		"opened eyes":  unicode(_("Opened Eyes"), "utf-8"),
     
     		"wait":		    unicode(_("Wait"), "utf-8"),
     		"sec":		    unicode(_(" sec"), "utf-8"),
     
     		"meta string":  unicode(_("Meta String"), "utf-8"),
     		"user name":	unicode(_("User name"), "utf-8"),
     		"self name":	unicode(_("Self name"), "utf-8"),
     		"true":		    unicode(_("True"), "utf-8"),
     		"false":		unicode(_("False"), "utf-8"),
     		"kero name":	unicode(_("Kero name"), "utf-8"),
     
     		"new line":	    unicode(_("New line"), "utf-8"),
     		"clear":		unicode(_("Clear"), "utf-8"),
     
     		"terminate":	unicode(_("Terminate"), "utf-8"),
     		}
		return pmsg
	
	def get_popup_menu_items(self, pmsg):     
		popup_menu_items = (
     		( "/sep0", None, None, 0, "<Separator>" ),
     		( "/%s" % pmsg["sessions"], None, None, 0, "<Branch>" ),
     		( "/%s/%s" % (pmsg["sessions"], pmsg["timecritical"]), None, self.insert_t, 0, None ),
     		( "/%s/%s" % (pmsg["sessions"], pmsg["quick"]), None, self.insert_q, 0, None ),
     		( "/%s/%s" % (pmsg["sessions"], pmsg["syncronized"]), None, self.insert_s, 0, None ),
     		( "/sep1", None, None, 0, "<Separator>" ),
     
     		( "/%s" % pmsg["sakura"], None, self.insert_h, 0, None ),
     		( "/%s" % pmsg["countenance"], None, None, 0, "<Branch>" ),
     		( "/%s/[0]  %s" % (pmsg["countenance"], pmsg["ordinary"]),
			  "<control><alt>0",self.insert_s0,  0, None, ),
     		( "/%s/[1]  %s" % (pmsg["countenance"], pmsg["shy"]),
			  "<control><alt>1", self.insert_s1, 0, None, ),
     		( "/%s/[2]  %s" % (pmsg["countenance"], pmsg["surprised"]),
			  "<control><alt>2", self.insert_s2, 0, None, ),
     		( "/%s/[3]  %s" % (pmsg["countenance"], pmsg["angst"]),
			  "<control><alt>3", self.insert_s3, 0, None, ),
     		( "/%s/[4]  %s" % (pmsg["countenance"], pmsg["ease off"]),
			  "<control><alt>4", self.insert_s4, 0, None, ),
     		( "/%s/[5]  %s" % (pmsg["countenance"], pmsg["smile"]),
			  "<control><alt>5", self.insert_s5, 0, None, ),
     		( "/%s/[6]  %s" % (pmsg["countenance"], pmsg["closed eyes"]),
			  "<control><alt>6", self.insert_s6, 0, None, ),
     		( "/%s/[7]  %s" % (pmsg["countenance"], pmsg["angry"]),
			  "<control><alt>7", self.insert_s7, 0, None, ),
     		( "/%s/[8]  %s" % (pmsg["countenance"], pmsg["scorned"]),
			  "<control><alt>8", self.insert_s8, 0, None, ),
     		( "/%s/[20] %s" % (pmsg["countenance"], pmsg["signboard"]), None, self.insert_s20, 0, None, ),
     		( "/%s/[25] %s" % (pmsg["countenance"], pmsg["singing"]), None, self.insert_s25, 0, None, ),
     		( "/sep2", None, None, 0, "<Separator>" ),
     
     		( "/%s" % pmsg["unyu"], None, self.insert_u, 0, None ),
     		( "/%s" % pmsg["countenance"], None, None, 0, "<Branch>" ),
     		( "/%s/[10] %s" % (pmsg["countenance"], pmsg["ordinary"]),
			  "<control><alt><shift>g", self.insert_s10, 0, None ),
     		( "/%s/[11] %s" % (pmsg["countenance"], pmsg["opened eyes"]), None, self.insert_s11, 0, None ),
     		( "/%s/[19] %s" % (pmsg["countenance"], pmsg["singing"]), None, self.insert_s19, 0, None ),
     		( "/sep3", None, None, 0, "<Separator>" ),
     
     		( "/%s" % pmsg["wait"], None, None, 0, "<Branch>" ),
     		( "/%s/0.1%s" % (pmsg["wait"], pmsg["sec"]), None, self.insert_w2, 0, None ),
     		( "/%s/0.2%s" % (pmsg["wait"], pmsg["sec"]), None, self.insert_w4, 0, None ),
     		( "/%s/0.3%s" % (pmsg["wait"], pmsg["sec"]), None, self.insert_w6, 0, None ),
     		( "/%s/0.4%s" % (pmsg["wait"], pmsg["sec"]), None, self.insert_w8, 0, None ),
     		( "/%s/0.5%s" % (pmsg["wait"], pmsg["sec"]), None, self.insert_w10, 0, None ),
     		( "/%s/0.6%s" % (pmsg["wait"], pmsg["sec"]), None, self.insert_w12, 0, None ),
     		( "/%s/0.9%s" % (pmsg["wait"], pmsg["sec"]), "<control><alt>w", self.insert_w18, 0, None ),
     		( "/%s/1.0%s" % (pmsg["wait"], pmsg["sec"]), None, self.insert_w20, 0, None ),
     		( "/%s/1.2%s" % (pmsg["wait"], pmsg["sec"]), None, self.insert_w24, 0, None ),
     		( "/%s/1.5%s" % (pmsg["wait"], pmsg["sec"]), None, self.insert_w30, 0, None ),
     		( "/%s/2.0%s" % (pmsg["wait"], pmsg["sec"]), None, self.insert_w40, 0, None ),
     		( "/%s/2.5%s" % (pmsg["wait"], pmsg["sec"]), None, self.insert_w50, 0, None ),
     		( "/%s/3.0%s" % (pmsg["wait"], pmsg["sec"]), None, self.insert_w60, 0, None ),
     
     		( "/%s" % pmsg["meta string"], None, None, 0, "<Branch>" ),
     		( "/%s/%s" % (pmsg["meta string"], pmsg["user name"]), None, self.insert_username, 0, None ),
     		( "/%s/%s (%s)" % (pmsg["meta string"], pmsg["self name"], pmsg["true"]),
			  None, self.insert_selfname, 0, None ),
     		( "/%s/%s (%s)" % (pmsg["meta string"], pmsg["self name"], pmsg["false"]),
			  None, self.insert_selfname2, 0, None),
     		( "/%s/%s" % (pmsg["meta string"], pmsg["kero name"]), None, self.insert_keroname, 0, None ),
     		( "/sep4", None, None, 0, "<Separator>" ),
     
     		( "/%s" % pmsg["new line"], "<control><alt>n", self.insert_n, 0, None ),
     		( "/%s" % pmsg["clear"], None, self.insert_c, 0, None ),
     		( "/URL", None, self.insert_url, 0, None ),
     
			( "/%s" % pmsg["terminate"], "<control><alt>e", self.insert_e, 0, None ),
     		)
		return popup_menu_items
	
	def set_self_messages(self):
		self.msg = {
			"file":   [unicode(_("File(_F)"), "utf-8"),
					   unicode(_("File(F)"), "utf-8")],
			"join":   unicode(_("Join"), "utf-8"),
			"part":   unicode(_("Part"), "utf-8"),
			"quit":   unicode(_("Quit"), "utf-8"),
			"edit":   [unicode(_("Edit(_E)"), "utf-8"),
					   unicode(_("Edit(E)"), "utf-8")],
			"cut":	  unicode(_("Cut"), "utf-8"),
			"copy":   unicode(_("Copy"), "utf-8"),
			"paste":  unicode(_("Paste"), "utf-8"),
			"delete": unicode(_("Delete"), "utf-8"),
			"clear":  unicode(_("Clear"), "utf-8"),
			"ca":	  unicode(_("Clear All"), "utf-8"),
			"pref":   [unicode(_("Preferences(_P)"), "utf-8"),
					   unicode(_("Preferences(P)"), "utf-8")],
			"channel":[unicode(_("Channel(_C)"), "utf-8"),
					   unicode(_("Channel(C)"), "utf-8")],
			"ghost":  [unicode(_("Ghost(_G)"), "utf-8"),
					   unicode(_("Ghost(G)"), "utf-8")],
			"window": [unicode(_("Window(_W)"), "utf-8"),
					   unicode(_("Window(W)"), "utf-8")],
			"logs":    unicode(_("Logs"), "utf-8"),
			"phrases": unicode(_("Phrases"), "utf-8"),
			"help":   [unicode(_("Help(_H)"), "utf-8"),
					   unicode(_("Help(H)"), "utf-8")],
			"about":  unicode(_("About"), "utf-8"),
		}
	
	def get_menu_items(self):
		menu_items = (
			( "/%s" % self.msg["file"][0], None, None, 0, "<Branch>" ),
			( "/%s/%s" % (self.msg["file"][1], self.msg["join"]), "<control>J", self.join, 0, None ),
			( "/%s/%s" % (self.msg["file"][1], self.msg["part"]), "<control>P", self.part, 0, None ),
			( "/%s/sep1" % self.msg["file"][1], None, None, 0, "<Separator>" ),
			( "/%s/%s" % (self.msg["file"][1], self.msg["quit"]), "<control>Q", self.quit, 0, None ),

			( "/%s" % self.msg["edit"][0], None, None, 0, "<Branch>" ),
			( "/%s/%s" % (self.msg["edit"][1], self.msg["cut"]), "<control>X", self.edit_cut, 0, None ),
			( "/%s/%s" % (self.msg["edit"][1], self.msg["copy"]), "<control>C", self.edit_copy, 0, None ),
			( "/%s/%s" % (self.msg["edit"][1], self.msg["paste"]), "<control>V", self.edit_paste, 0, None ),
			( "/%s/%s" % (self.msg["edit"][1], self.msg["delete"]), "<control>D", self.edit_delete,0, None ),
			( "/%s/sep4" % self.msg["edit"][1], None, None, 0, "<Separator>" ),

			( "/%s/%s" % (self.msg["edit"][1], self.msg["clear"]), None, self.edit_clear, 0, None ),
			( "/%s/%s" % (self.msg["edit"][1], self.msg["ca"]), None, self.edit_clear_all, 0, None ),
			( "/%s/sep5" % self.msg["edit"][1], None, None, 0, "<Separator>" ),
			( "/%s/%s" % (self.msg["edit"][1], self.msg["pref"][0]), None, self.open_preference, 0, None ),

			( "/%s" % self.msg["channel"][0], None, None, 0, "<Branch>" ),
			( "/%s" % self.msg["ghost"][0], None, None, 0, "<Branch>" ),

			( "/%s" % self.msg["window"][0], None, None, 0, "<Branch>" ),
			( "/%s/%s" % (self.msg["window"][1], self.msg["logs"]), "<control>L", None, 0, None ),
			( "/%s/%s" % (self.msg["window"][1], self.msg["phrases"]), None, None, 0, None ),

			( "/%s" % self.msg["help"][0], None, None, 0, "<LastBranch>" ),
			( "/%s/%s" % (self.msg["help"][1], self.msg["about"]), None, self.about, 0, None ),
			)
	
		return menu_items

class ChannelDialog(gtk.Dialog):
	def __init__(self, master, channels):
		self.selection = []
		self.channels  = channels

		gtk.Dialog.__init__(self, unicode(_("Select Channels"), "utf-8"), master,
							gtk.DIALOG_DESTROY_WITH_PARENT|gtk.DIALOG_MODAL|gtk.DIALOG_NO_SEPARATOR,
							(gtk.STOCK_CANCEL, gtk.RESPONSE_REJECT,
							 gtk.STOCK_OK, gtk.RESPONSE_ACCEPT))
		self.connect("destroy", self.cancel, "WM destroy")
		self.connect('response', self.res_caught)

		cbs = []
		for n in range(len(self.channels)):
			cb = gtk.CheckButton(self.channels[n]["name"])
			cbs.append(cb)
			cb.connect("toggled", self.selected, [self.channels[n], n])
			self.vbox.pack_start(cb, gtk.TRUE, gtk.TRUE, 2)
			cb.show()

		self.vbox.show()

		self.buffer   = gtk.TextBuffer()
		self.textview = gtk.TextView(self.buffer)
		self.textview.set_editable(gtk.FALSE)
		self.textview.set_cursor_visible(gtk.FALSE)
		self.textview.show()

		self.f = gtk.Frame()
		self.f.set_shadow_type(gtk.SHADOW_IN)
		self.f.add(self.textview)
		self.f.show()

		self.vbox.pack_start(self.f, gtk.TRUE, gtk.TRUE, 2)

		cbs.reverse()
		for b in cbs:
			b.set_active(gtk.TRUE)

		ok_button = self.action_area.get_children()[0]
		ok_button.grab_focus()

	def selected(self, caller, data):
		n = data[1]
		data = data[0]
		buf = "info: " + data["info"] + "\n" + "ghost: " + data["ghost"] + \
			  "\n" + "count: " + str(data["count"])
		self.buffer.set_text(buf)

		if n in self.selection:
			self.selection.remove(n)
		else:
			self.selection.append(n)
			self.selection.sort()

	def res_caught(self, dialog, response_id):
		if response_id == gtk.RESPONSE_ACCEPT:
			self.hide()
		else:
			self.cancel()

	def cancel(self, widget=None, data=None):
		self.selection = None
		self.hide()


def usage():
	port = unicode(_("alternative port for forwarding SSTP messages"), 'utf-8').encode('euc-jp')
	help = unicode(_("show this message"), 'utf-8').encode('euc-jp')
	host = unicode(_("alternative host for forwarding SSTP messages"), 'utf-8').encode('euc-jp')

	usage = """\
	-p, --port NUM  %s
	--host HOST     %s
	-h, --help      %s\n""" % (port, host, help)

	sys.stderr.write(unicode(_("Usage"), 'utf-8').encode('euc-jp') + ": " + \
					 AGENT_NAME + "\n" + \
					 unicode(_("Options"), 'utf-8').encode('euc-jp') + ":\n" + usage)
	sys.exit(1)

def main():
	# parse command line arguments
	try:
		options, rest = getopt.getopt(sys.argv[1:], "p:h",
									  ["port=", "help", "host="])
	except getopt.error, e:
		sys.stderr.write(unicode(_("Error"), 'utf-8').encode('euc-jp') + ": %s\n" % str(e))
		usage()
	if rest:
		usage()

	port = 9801
	host = "localhost"   # default - localhost:9801

	# check environment variables
	val = os.environ.get("BOTTLER_PORT")
	if val:
		try:
			num = int(val)
		except ValueError:
			sys.stderr.write(unicode(_("Invalid BOTTLER_PORT number (ignored)") + \
									 "\n", 'utf-8').encode('euc-jp'))
		else:
			port = num

	# parse command line options
	for opt, val in options:
		if opt in ["-p", "--port"]:
			port = int(val)
		elif opt == "--host":
			host = val
		else:
			usage()
	# run
	app = Application(host, port)
	app.main()

if __name__ == "__main__":
	main()

