# -*- coding: ascii -*-
#
#  logs.py - Log manipulator for GBottler
#  Copyright (C) 2001, 2002 by Tamito KAJIYAMA
#  Copyright (C) 2004 by Atzm WATANABE <sitosito@p.chan.ne.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#
# $Id: logs.py,v 1.60 2004/05/29 11:07:51 atzm Exp $
#

import os, re, time
from types import *

from logfetch     import LogFetcher, LogFetchDialog
from textmanager  import TextManager
from xmllogs      import XMLLogger
from bottleparser import BottleParser

import gtk

from common import *

class LogWindow:
	def __init__(self, app):
# === Definition of Basic Const === #
		self.app    = app
		accel_group = gtk.AccelGroup()

		self.set_self_msg()

# === Create Base Window === #
		self.window = gtk.Window(gtk.WINDOW_TOPLEVEL)
		self.window.add_accel_group(accel_group)
		self.window.connect("delete_event", self.close)
		self.window.set_title(unicode(_(APP), "utf-8") + ":" + \
							  unicode(_("Logs"), "utf-8"))
		self.window.set_size_request(450,300)

# === Create Entry for Search === #
		self.search_entry = gtk.Entry()
		self.search_entry.show()
		self.search_entry.set_text(self.msg['search'])
		self.search_entry.connect("activate", self.search)
		self.search_entry.connect("key-press-event", self.entry_keypress)
		self.search_entry.connect("focus-in-event", self.set_entry_guide, 'in')
		self.search_entry.connect("focus-out-event", self.set_entry_guide, 'out')

		self.s_arrow = gtk.Arrow(gtk.ARROW_DOWN, gtk.SHADOW_NONE)
		self.s_arrow.show()

		self.s_a_b = gtk.ToggleButton()
		self.s_a_b.show()
		self.s_a_b.add(self.s_arrow)
		self.s_a_b.connect('clicked',
						   lambda x: self.s_arrow.set((not self.s_arrow.get_property('arrow-type')),
													  gtk.SHADOW_NONE))
		self.s_a_b.connect('focus-in-event', lambda x, y: self.search_entry.grab_focus())

		self.sbar = gtk.Statusbar()
		self.sbar.show()
		self.sbar.set_has_resize_grip(gtk.TRUE)

		self.rowsbar = gtk.Statusbar()
		self.rowsbar.show()
		self.rowsbar.set_has_resize_grip(gtk.FALSE)

		self.s_box = gtk.HBox()
		self.s_box.show()
		self.s_box.pack_start(self.s_a_b, gtk.FALSE, gtk.FALSE, 0)
		self.s_box.pack_start(self.search_entry)
		self.s_box.pack_start(self.rowsbar)
		self.s_box.pack_start(self.sbar)

# === Create Menubar === #
		menu_items = self.get_menu_items()
		# can't set menu_items before set entry

		self.item_factory = gtk.ItemFactory(gtk.MenuBar, "<main>", accel_group)
		self.item_factory.create_items(menu_items)

		self.menu_list = {}
		for m in menu_items:
			key = re.sub("_", "", m[0])
			self.menu_list[key] = self.item_factory.get_item(key)

		self.menubar = self.item_factory.get_widget("<main>")
		self.menubar.show()

# === Create Popup Menu for CList === #
		p_menu = (
			( "/%s" % unicode(_("Copy to main editor")), "<control>c", self.edit_copy, 0, None ),
			( "/%s" % unicode(_("Play")), "<control>p", self.play, 0, None ),
			( "/%s" % unicode(_("Vote")), "<control><alt>v", self.voting, 0, None ),
			( "/%s" % unicode(_("Agree")), "<control><alt>a", self.agreeing, 0, None ),
			)
		self.p_ifact = gtk.ItemFactory(gtk.Menu, "<main>", accel_group)
		self.p_ifact.create_items(p_menu)
		self.popup_menu = self.p_ifact.get_widget("<main>")

		# keep MenuItems, vote and agree
		self.vi = self.p_ifact.get_item("/%s" % unicode(_("Vote"), "utf-8"))
		self.ai = self.p_ifact.get_item("/%s" % unicode(_("Agree"), "utf-8"))

# === Create Log Viewer === #
		self.main_text = gtk.TextView()
		self.main_text.show()
		self.main_text.set_cursor_visible(gtk.FALSE)
		self.main_text.set_editable(gtk.FALSE)
		self.main_text.set_wrap_mode(gtk.WRAP_CHAR)
		self.main_text.set_border_width(0)
		self.main_text.connect("focus-in-event", self.focus_active_loglist)

		self.sw2 = gtk.ScrolledWindow()
		self.sw2.show()
		self.sw2.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
		self.sw2.set_border_width(0)
		self.sw2.add(self.main_text)

		self.frame = gtk.Frame()
		self.frame.show()
		self.frame.set_shadow_type(gtk.SHADOW_IN)
		self.frame.set_border_width(0)
		self.frame.add(self.sw2)

		self.monitor = self.main_text.get_buffer()
		self.textmanager = TextManager(self, self.main_text, self.app.prefs,
									   BottleParser('loose'))

# === Create Buttons === #
		self.fb = get_icon_button(gtk.STOCK_CONVERT, gtk.ICON_SIZE_MENU)
		self.ob = get_icon_button(gtk.STOCK_OPEN,    gtk.ICON_SIZE_MENU)
		self.sb = get_icon_button(gtk.STOCK_SAVE_AS, gtk.ICON_SIZE_MENU)

		self.pb = get_icon_button(gtk.STOCK_REFRESH, gtk.ICON_SIZE_MENU)
		self.vb = get_icon_button(gtk.STOCK_YES,     gtk.ICON_SIZE_MENU)
		self.ab = get_icon_button(gtk.STOCK_NO,      gtk.ICON_SIZE_MENU)

		self.fb.connect("clicked", self.fetch)
		self.ob.connect("clicked", self.open_xml_log)
		self.sb.connect("clicked", self.save_as_xml)

		self.pb.connect("clicked", self.play, None)
		self.vb.connect("clicked", self.voting, None)
		self.ab.connect("clicked", self.agreeing, None)

		self.vb.set_sensitive(gtk.FALSE)
		self.ab.set_sensitive(gtk.FALSE)

		self.fb.connect("focus-in-event", self.focus_active_loglist)
		self.ob.connect("focus-in-event", self.focus_active_loglist)
		self.sb.connect("focus-in-event", self.focus_active_loglist)
		self.pb.connect("focus-in-event", self.focus_active_loglist)
		self.vb.connect("focus-in-event", self.focus_active_loglist)
		self.ab.connect("focus-in-event", self.focus_active_loglist)

		self.tooltips = gtk.Tooltips()
		self.tooltips.set_tip(self.fb, self.msg['fetch'])
		self.tooltips.set_tip(self.ob, self.msg['open_xml'])
		self.tooltips.set_tip(self.sb, self.msg['save'])

		self.tooltips.set_tip(self.pb, unicode(_("Play"), "utf-8"))
		self.tooltips.set_tip(self.vb, unicode(_("Vote"), "utf-8"))
		self.tooltips.set_tip(self.ab, unicode(_("Agree"), "utf-8"))

# === Create Labels and style === #
		self.m = gtk.Label(unicode(_("Style"), "utf-8") + ": ")
		self.m.show()

		self.style_l = gtk.Label(self.msg['talk style'])
		self.style_l.show()

		self.s_e_box = gtk.EventBox()
		self.s_e_box.show()
		self.s_e_box.set_events(gtk.gdk.BUTTON_PRESS_MASK)
		self.s_e_box.add(self.style_l)
		self.s_e_box.connect("button-press-event", self.button_popup,
							 self.item_factory.get_widget("/%s/%s" % (self.msg["view"][1],
																	  self.msg["style"][1])))

# === Bring Buttons and Labels together to HBox, Frame === #
		self.hbox = gtk.HBox()
		self.hbox.show()

		self.hbox.pack_start(self.fb, gtk.FALSE, gtk.TRUE, 0)
		self.hbox.pack_start(self.ob, gtk.FALSE, gtk.TRUE, 0)
		self.hbox.pack_start(self.sb, gtk.FALSE, gtk.TRUE, 0)
		vs = gtk.VSeparator()
		vs.show()
		self.hbox.pack_start(vs, gtk.FALSE, gtk.TRUE, 3)

		self.hbox.pack_start(self.pb, gtk.FALSE, gtk.TRUE, 0)
		self.hbox.pack_start(self.vb, gtk.FALSE, gtk.TRUE, 0)
		self.hbox.pack_start(self.ab, gtk.FALSE, gtk.TRUE, 0)
		vs = gtk.VSeparator()
		vs.show()
		self.hbox.pack_start(vs, gtk.FALSE, gtk.TRUE, 3)

		self.hbox.pack_start(self.m, gtk.FALSE, gtk.TRUE, 0)
		self.hbox.pack_start(self.s_e_box, gtk.FALSE, gtk.TRUE, 0)

		self.bframe = gtk.Frame()
		self.bframe.show()
		self.bframe.set_border_width(0)
		self.bframe.set_shadow_type(gtk.SHADOW_OUT)
		self.bframe.add(self.hbox)

# === Create Notebook === #
		label = unicode(_('Current'), 'utf-8')
		self.logbook = LogBook(label)
		self.logbook.set_tab_pos(int(self.app.prefs.get('logtabpos')))
		self.logbook.connect('switch-page', self.tab_changed)

		self.create_tab(label)

# === Bring Log Entry and Log Viewer together to VPaned === #
		self.pane = gtk.VPaned()
		self.pane.show()
		self.pane.pack1(self.logbook, gtk.TRUE)
		self.pane.pack2(self.frame, gtk.FALSE)

# === Bring all items together to Main Window Finally === #
		self.main_vbox = gtk.VBox(gtk.FALSE, 0)
		self.main_vbox.show()

		self.main_vbox.pack_start(self.menubar, gtk.FALSE, gtk.TRUE, 0)
		self.main_vbox.pack_start(self.bframe, gtk.FALSE, gtk.TRUE, 0)
		self.main_vbox.pack_start(self.pane)
		self.main_vbox.pack_start(self.s_box, gtk.FALSE, gtk.FALSE, 0)

		self.window.add(self.main_vbox)

# === Initialization of LogWindow === #
		self.focus_active_loglist()

# === Open and Close === #
	def open(self, widget, data):
		self.window.show()

	def close(self, widget=None, e=None, data=None):
		self.window.hide()
		return gtk.TRUE

	def join(self):
		self.vb.set_sensitive(gtk.TRUE)
		self.ab.set_sensitive(gtk.TRUE)
		self.vi.set_sensitive(gtk.TRUE)
		self.ai.set_sensitive(gtk.TRUE)

	def part(self):
		self.vb.set_sensitive(gtk.FALSE)
		self.ab.set_sensitive(gtk.FALSE)
		self.vi.set_sensitive(gtk.FALSE)
		self.ai.set_sensitive(gtk.FALSE)

	def color_changed(self, script, sakura, kero, sync, error, URL):
		self.textmanager.color_changed(script, sakura, kero, sync, error, URL)
		list = self.logbook.get_active_loglist()
		if not list or not list.selection:
			return
		self.select(list, list.selection[0])

	def autologging(self, path):
		list = self.logbook.get_current_loglist()
		if not list:
			return
		filename = time.strftime(path, time.localtime(time.time()))

		dir = os.path.dirname(filename)
		if not os.path.exists(dir):
			try:
				os.makedirs(dir)
			except:
				return

		logger = XMLLogger()
		logger.save_as_xml_from(list.get_plain_array(), filename)

	def save_as_xml(self, widget=None, list=None):
		def file_ok_sel(w, list):
			filename = filew.get_filename()
			filew.destroy()

			if os.path.exists(filename):
				d = get_simple_yes_or_no_dialog(unicode(_("Really?"), 'utf-8'),
												unicode(_("File already exists."), 'utf-8') + " " + \
												unicode(_("overwrite?"), 'utf-8') + "\n\n" + \
												filename + '\n',
												self.window)
				d.show()
				res = d.run()
				d.destroy()

				if not res or res == gtk.RESPONSE_REJECT:
					return

			if not list or not isinstance(list, LogList):
				list = self.logbook.get_active_loglist()
			if not list:
				return

			lines = list.get_plain_array()
			logger = XMLLogger()
			if not logger.save_as_xml_from(lines, filename):
				open_error_dialog(unicode(_("Can't save file"), 'utf-8') + " '%s' " % filename,
								  self.window)

		filew = gtk.FileSelection()
		filew.set_select_multiple(gtk.FALSE)
		filew.connect("destroy", lambda x: filew.destroy())

		filew.set_filename(os.path.normpath(open_bottlecase() + '/*.xml'))
		filew.ok_button.connect("clicked", file_ok_sel, list)
		filew.cancel_button.connect("clicked", lambda x: filew.destroy())
		filew.show()

	def open_xml_log(self, widget=None, data=None):
		def file_ok_sel(w):
			filename = filew.get_filename()
			filew.destroy()

			logger = XMLLogger()
			logs = logger.open_xml_from(filename)

			if logs == -1:
				open_error_dialog(unicode(_("Can't open file"), 'utf-8') + " '%s' " % filename,
								  self.window)
				return

			self.create_tab(os.path.basename(filename), list=logs,
							focus=self.app.prefs.get('logtabfocus'),
							pos=self.app.prefs.get('logpagepos'))

		filew = gtk.FileSelection()
		filew.set_select_multiple(gtk.FALSE)
		filew.connect("destroy", lambda x: filew.destroy())

		filew.set_filename(os.path.normpath(open_bottlecase() + '/*.xml'))
		filew.ok_button.connect("clicked", file_ok_sel)
		filew.cancel_button.connect("clicked", lambda x: filew.destroy())
		filew.show()

	def fetch(self, widget=None, data=None):
		d = LogFetchDialog(self.window)
		if not d.request or not d.request_name:
			return

		fetcher = LogFetcher()
		logs    = fetcher.fetch(d.request)

 		self.create_tab(d.request_name, list=logs,
						focus=self.app.prefs.get('logtabfocus'),
						pos=self.app.prefs.get('logpagepos'))

	def popup_from_logtab(self, widget, event, sender):
		if event.button == 3:
			sender.popup(None, None, None, event.button, event.time)

	def tab_drag_data_get(self, widget, context, selection, targetType, eventTime, sender):
		n = self.logbook.page_num(sender)
		selection.set(selection.target, 8, str(n))

	def tab_drop_received(self, widget, context, x, y, selection, targetType, time, receiver):
		sender_n = int(selection.data)
		sender = self.logbook.get_nth_page(sender_n)

		receiver_n = self.logbook.page_num(receiver)

		self.logbook.reorder_child(sender, receiver_n)
		self.logbook.reorder_child(receiver, sender_n)

	def connect_for_logtab(self, list, child, eventbox):
		save_menuitem = gtk.MenuItem(self.msg["save"])
		save_menuitem.show()
		save_menuitem.connect("activate", self.save_as_xml, list)
		close_menuitem = gtk.MenuItem(self.msg["close"])
		close_menuitem.show()
		if self.logbook.current_is(list):
			close_menuitem.connect("activate", self.close_current_tab, child)
		else:
			close_menuitem.connect("activate", self.close_tab, child)

		logtab_popup_menu = gtk.Menu()
		logtab_popup_menu.show()
		logtab_popup_menu.append(save_menuitem)
		logtab_popup_menu.append(close_menuitem)
		eventbox.connect("button-press-event", self.popup_from_logtab, logtab_popup_menu)

		# DND
		eventbox.connect("drag-data-get", self.tab_drag_data_get, child)
		eventbox.drag_source_set(gtk.gdk.BUTTON1_MASK, [ ( "text/plain", 0, 80 ) ], gtk.gdk.ACTION_MOVE)

		eventbox.connect("drag-data-received", self.tab_drop_received, child)
		eventbox.drag_dest_set(gtk.DEST_DEFAULT_MOTION|gtk.DEST_DEFAULT_HIGHLIGHT|gtk.DEST_DEFAULT_DROP,
							   [ ( "text/plain", 0, 80 ) ], gtk.gdk.ACTION_MOVE)

	def create_tab(self, label_text, pos=None, focus=False, list=None):
		close_button, eventbox, child = self.logbook.create_tab(label_text, pos, focus, list)

		list = child.get_child()
		if self.logbook.get_n_pages() == 1:
			self.update_rows(list.rows)
			if list.rows > 0 and list.selection:
				self.select(list, list.selection[0])
		self.connect_list(list)

		if self.logbook.current_is(list):
			close_button.connect('clicked', self.close_current_tab, child)
		else:
			close_button.connect('clicked', self.close_tab, child)
		close_button.connect('focus-in-event', self.focus_active_loglist)

		self.connect_for_logtab(list, child, eventbox)

	def update(self, mid, channel, ghost, script, receive_time):
		datetime = time.strftime("%y/%m/%d %H:%M:%S", receive_time)
		close_button, eventbox, child = self.logbook.update(mid, channel, ghost, script, datetime,
															pos=self.app.prefs.get('logpagepos'),
															focus=self.app.prefs.get('logtabfocus'))

		if child and close_button and eventbox:
			# if Tab named `Current' Created
			close_button.connect('clicked', self.close_current_tab, child)
			close_button.connect('focus-in-event', self.focus_active_loglist)

			list = child.get_child()
			if list:
				self.connect_list(list)

				if list == self.logbook.get_active_loglist():
					if list.selection:
						self.select(list, list.selection[0])
					self.update_rows(list.rows)
			self.focus_active_loglist()
			self.connect_for_logtab(list, child, eventbox)

		elif self.logbook.current_is(self.logbook.get_active_loglist()):
			self.update_rows(self.logbook.get_active_loglist().rows)

	def close_current_tab(self, widget, child):
		self.app.notify_log_current_tab_closed()
		self.close_tab(widget, child)

	def close_tab(self, widget, child):
		n = self.logbook.close_tab(widget, child)
		if n == -1:
			self.remove_status()

	def connect_list(self, list):
		list.connect("select-row",         self.select)
		list.connect("key-press-event",    self.loglist_keypress)
		list.connect("button-press-event", self.popup)

	def update_rows(self, rows):
		self.rowsbar.pop(0)
		self.rowsbar.push(0, str(rows) + unicode(_('Messages'), 'utf-8'))

	def remove_status(self):
		self.sbar.pop(0)
		self.rowsbar.pop(0)
		self.textmanager.clear()

	def tab_changed(self, notebook, page, pagenum):
		list = self.logbook.tab_changed(notebook, page, pagenum)
		if list is None:
			self.remove_status()
		else:
			self.select(list, list.selection[0])
			self.update_rows(list.rows)

	def tab_pos_changed(self, pos):
		if self.logbook.get_tab_pos() != pos:
			self.logbook.set_tab_pos(pos)

	def search(self, widget=None, data=None):
		self.logbook.search(self.s_arrow.get_property('arrow-type'),
							self.search_entry.get_text(),
							widget, data)

	def play(self, widget=None, data=None):
		list = self.logbook.get_active_loglist()
		if not list or not list.selection:
			return
		script  = unicode(list.get_text(list.selection[0], 5), 'utf-8')
		ifghost = unicode(list.get_text(list.selection[0], 1), 'utf-8')
		channel = unicode(list.get_text(list.selection[0], 2), 'utf-8')
		if script is None or ifghost is None:
			return
		self.app.send_local_message(channel, ifghost, script)

	def log_votes(self, mid, type, num):
		self.logbook.log_votes(mid, type, num)

	def edit_copy(self, widget, data):
		script = self.logbook.edit_copy(widget, data)
		if not script:
			return
		script = re.sub(r'\\n', r'\\n\n', script)
		self.app.insert(script)

	def voting(self, widget, data):
		mid, type = self.logbook.voting(widget, data, self.window)
		if mid is None or type is None:
			return
		gtk.timeout_add(0, self.app.vote_message, mid, type)

	def agreeing(self, widget, data):
		mid, type = self.logbook.agreeing(widget, data, self.window)
		if mid is None or type is None:
			return
		gtk.timeout_add(0, self.app.vote_message, mid, type)

	def button_popup(self, widget, event, sender):
		if event.button in [1, 3]:
			sender.popup(None, widget, None, event.button, event.time)

	def set_entry_guide(self, widget, event, data):
		if data == 'in':
			if widget.get_text() == self.msg['search']:
				widget.set_text('')
		elif data == 'out':
			if not widget.get_text():
				widget.set_text(self.msg['search'])

	def popup(self, loglist, event):
		sel = loglist.get_selection_info(int(event.x), int(event.y))
		if sel:
			loglist.set_focus_row(sel[0])
			if event.button == 3:
				self.popup_menu.popup(None, loglist, None, event.button, event.time)
			elif event.type == 5:
				self.play(loglist, event)

	def select(self, loglist, row, col=None, event=None):
		self.textmanager.clear()
		script = unicode(loglist.get_text(row, 5), "utf-8", 'replace')
		self.textmanager.insert_with_color_from(script)
		self.sw2.emit('scroll-child', gtk.SCROLL_START, gtk.FALSE)
		self.sbar.pop(0)
		self.sbar.push(0, str(len(script.encode('sjis'))) + unicode(_('bytes'), 'utf-8'))

	def scroll(self, widget, event):
		if event.state == 1: # shift
			if event.keyval == 65362: # up
				self.sw2.emit('scroll-child', gtk.SCROLL_PAGE_UP, gtk.FALSE)
			elif event.keyval == 65364: # down
				self.sw2.emit('scroll-child', gtk.SCROLL_PAGE_DOWN, gtk.FALSE)
			return True
		return False

	def entry_keypress(self, widget, event):
		self.scroll(widget, event)

	def loglist_keypress(self, widget, event):
		if not self.scroll(widget, event):
			if event.keyval == 65293: # enter
				self.play()

	def set_style(self, data, widget):
		if data == STYLE_TALK:
			label = self.msg['talk style']
		elif data == STYLE_SCRIPT:
			label = self.msg['script style']
		elif data == STYLE_SCRIPT_WITH_LINEFEED:
			label = "%s(%s)" % (self.msg['script style'], self.msg['linefeed'])

		if label == self.style_l.get_text():
			return
		self.style_l.set_text(label)
		self.textmanager.set_style(data)

		list = self.logbook.get_active_loglist()
		if not list or not list.selection:
			return
		self.select(list, list.selection[0], None, None)

	def focus_active_loglist(self, widget=None, data=None):
		self.logbook.focus_active_loglist()

	def help(self, event=None):
		pass

	def get_menu_items(self):
		menu_items = (
			( "/%s" % self.app.msg["file"][0], None, None, 0, "<Branch>" ),
			( "/%s/%s" % (self.app.msg["file"][1], self.msg["open_xml"]), "<control>o", self.open_xml_log, 0, None ),
			( "/%s/%s" % (self.app.msg["file"][1], self.msg["save"]), "<control>s", self.save_as_xml, 0, None ),
			( "/%s/%s" % (self.app.msg["file"][1], self.msg["fetch"]), None, self.fetch, 0, None ),
			( "/%s/%s" % (self.app.msg["file"][1], self.msg["close"]), "<control>q", self.close, 0, None ),

			( "/%s" % self.app.msg["edit"][0], None, None, 0, "<Branch>" ),
			( "/%s/%s" % (self.app.msg["edit"][1], self.msg["copy to main editor"]),
			  "<control>c", self.edit_copy, 0, None ),
			( "/%s/%s" % (self.app.msg["edit"][1], self.msg["search"]), "<control>f",
			  lambda x, y: self.search_entry.grab_focus(), 0, None ),
			( "/%s/sep" % self.app.msg["edit"][1], None, None, 0, "<Separator>" ),
			( "/%s/%s" % (self.app.msg["edit"][1], self.app.msg["pref"][0]), None,
			  self.app.open_preference, 0, None ),

			( "/%s" % self.msg["view"][0], None, None, 0, "<Branch>" ),

			( "/%s/%s" % (self.msg["view"][1], self.msg["style"][0]), None, None, 0, "<Branch>" ),
			( "/%s/%s/%s" % (self.msg["view"][1], self.msg["style"][1], self.msg["talk style"]),
			  None, self.set_style, STYLE_TALK, "<RadioItem>" ),
			( "/%s/%s/%s" % (self.msg["view"][1], self.msg["style"][1], self.msg["script style"]),
			  None, self.set_style, STYLE_SCRIPT,
			  "/%s/%s/%s" % (self.msg["view"][1],
							 self.msg["style"][1],
							 self.msg["talk style"])),
			( "/%s/%s/%s(%s)" % (self.msg["view"][1], self.msg["style"][1],
								 self.msg["script style"], self.msg["linefeed"]),
			  None, self.set_style, STYLE_SCRIPT_WITH_LINEFEED,
			  "/%s/%s/%s" % (self.msg["view"][1],
							 self.msg["style"][1],
							 self.msg["script style"])),

			( "/%s" % self.app.msg["help"][0], None, None, 0, "<LastBranch>" ),
			( "/%s/%s" % (self.app.msg["help"][1], self.app.msg["about"]), None, self.app.about, 0, None ),
			)
		return menu_items

	def set_self_msg(self):
		self.msg = {
			"open_xml":  unicode(_("Open XML log"), "utf-8"),
			"save":  unicode(_("Save as XML"), "utf-8"),
			"fetch":  unicode(_("Fetch logs"), "utf-8"),
			"close":  unicode(_("Close"), "utf-8"),
			"copy to main editor":   unicode(_("Copy to main editor"), "utf-8"),
			"search":   unicode(_("Search"), "utf-8"),

			"view":		   [unicode(_("View(_V)"), "utf-8"),
							unicode(_("View(V)"), "utf-8")],
			"update":	   unicode(_("Update"), "utf-8"),
			"style":	   [unicode(_("Style(_S)"), "utf-8"),
							unicode(_("Style(S)"), "utf-8")],
			"talk style":  unicode(_("Talk style"), "utf-8"),
			"script style":unicode(_("Script style"), "utf-8"),
			"linefeed":	   unicode(_("with linefeed"), "utf-8"),
			}


class LogBook(gtk.Notebook):
	def __init__(self, label_text):
		self.current_label_text = label_text

		gtk.Notebook.__init__(self)
		self.set_scrollable(gtk.TRUE)
		self.set_tab_pos(gtk.POS_TOP)
		self.show()

		self.set_create_tab_position(POS_NEXT)

	def get_n_pages(self):
		# override for PyGTK 2.3 or earlier
		return len(self.get_children())

	def set_create_tab_position(self, pos):
		self.create_position = pos

	def get_create_tab_position(self, pos=None):
		if not pos:
			pos = self.create_position

		if pos == POS_NEXT:
			return self.get_current_page() + 1
		elif pos == POS_PREV:
			p = self.get_current_page()
			if p > 0:
				p -= 1
			return p
		elif pos == POS_FIRST:
			return 0
		elif pos == POS_END:
			return -1
		else:
			raise ValueError, 'chan to basho shitei site kure'

	def focus_active_loglist(self, widget=None, data=None):
		list = self.get_active_loglist()
		if not list:
			return
		list.grab_focus()

	def tab_changed(self, notebook, page, pagenum):
		tab  = notebook.get_nth_page(pagenum)
		list = tab.get_child()
		return list.tab_changed()

	def close_tab(self, widget, child):
		n = self.page_num(child)
		self.remove_page(n)
		return self.get_current_page()

	def create_tab(self, label_text, pos=None, focus=False, list=None):
		pos = self.get_create_tab_position(pos)

		new_tab = LogList()

		sw = gtk.ScrolledWindow()
		sw.show()
		sw.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
		sw.add(new_tab)

		hbox, eventbox, button = get_hbox_with_label_button(gtk.STOCK_CLOSE, label_text)
		hbox.show()

		self.insert_page(sw, hbox, pos)
		self.set_menu_label(sw, gtk.Label(label_text))

		if list:
			new_tab.set_local_log(list)

		if focus:
			self.set_current_page(pos)

		return button, eventbox, sw

	def get_active_loglist(self):
		# getting showing LogList
		n = self.get_current_page()
		if n == -1:
			return
		tab  = self.get_nth_page(n)
		list = tab.get_child()
		return list

	def get_current_loglist(self):
		# getting LogList named `Current'
		tabs = self.get_children()
		for tab in tabs:
			hbox = self.get_tab_label(tab)
			eventbox, button = hbox.get_children()
			label = eventbox.get_children()[0]

			label_text = label.get_text()
			if label_text == self.current_label_text:
				return tab.get_child()

	def current_is(self, list):
		return (self.get_current_loglist() == list)

	def get_current_button_and_child(self):
		# getting close button and ScrolledWindow inside LogList named `Current'
		tabs = self.get_children()
		for i in range(len(tabs)):
			hbox = self.get_tab_label(tabs[i])
			eventbox, button = hbox.get_children()
			label = eventbox.get_children()[0]

			label_text = label.get_text()
			if label_text == self.current_label_text:
				return button, self.get_nth_page(i)

	def search(self, arrow, text, widget=None, data=None):
		list = self.get_active_loglist()
		if not list:
			return

		list.search(arrow, text, widget, data)

	def update(self, mid, channel, ghost, script, datetime, pos=None, focus=False):
		current_tab = self.get_current_loglist()

		button   = None
		eventbox = None
		sw       = None
		if not current_tab:
			button, eventbox, sw = self.create_tab(self.current_label_text, pos=pos, focus=focus)
			current_tab = sw.get_child()

		current_tab.update(mid, channel, ghost, script, datetime)
		return button, eventbox, sw
		# returns close button, EventBox, and ScrolledWindow if create LogList
		# named `Current' (if not exists LogList named `Current')
		# otherwise, returns [None, None, None]

	def log_votes(self, mid, type, num):
		current_tab = self.get_current_loglist()
		if not current_tab:
			return
		current_tab.log_votes(mid, type, num)

	def voting(self, widget, data, parent):
		list = self.get_active_loglist()
		if not list:
			return None, None
		return list.voting(widget, data, parent)

	def agreeing(self, widget, data, parent):
		list = self.get_active_loglist()
		if not list:
			return None, None
		return list.agreeing(widget, data, parent)

	def edit_copy(self, widget, data):
		list = self.get_active_loglist()
		if not list or not list.selection:
			return
		script = list.get_text(list.selection[0], 5)
		if not script:
			return
		return script


class LogList(gtk.CList):
	def __init__(self):
		gtk.CList.__init__(self, 6, [unicode(_("Datetime"), "utf-8"),
									 unicode(_("Ghost"), "utf-8"),
									 unicode(_("Channel"), "utf-8"),
									 unicode(_("Votes"), "utf-8"),
									 unicode(_("Agrees"), "utf-8"),
									 unicode(_("Script"), "utf-8")])
		self.show()

		self.set_shadow_type(gtk.SHADOW_IN)
		self.set_selection_mode(gtk.SELECTION_BROWSE)
		self.set_reorderable(gtk.TRUE)
		self.set_size_request(450, 120)
		self.set_column_width(0, 100)
		self.column_titles_passive()

	def set_local_log(self, loglist):
		loglist.reverse()
		self.freeze()
		self.clear()
		for datetime, mid, votes, agrees, channel, ghost, script in loglist:
			date = ''

			# old format?
			try:
				date = time.strptime(datetime, '%Y%m%d%H%M%S')
			except ValueError:
				pass

			# Bottle Client for Win or new GBotter format?
			try:
				date = time.strptime(datetime, '%y/%m/%d %H:%M:%S')
			except ValueError:
				pass

			if not date:
				continue

			date = time.strftime("%y/%m/%d %H:%M:%S", date)
			votes  = str(votes)
			agrees = str(agrees)
			n = self.append((date, ghost, channel, votes, agrees, script))
			self.set_row_data(n, mid)
		self.thaw()

	def tab_changed(self):
		if self.selection:
			return self

	def search(self, arrow, text, widget=None, data=None):
		if not self.selection:
			return

		pat = re.compile(text, re.IGNORECASE)
		current = self.selection[0]
		rng = []

		if arrow == gtk.ARROW_UP:
			rng = range(0, current-1)
			rng.reverse()
		elif arrow == gtk.ARROW_DOWN:
			rng = range(current+1, self.rows)

		for i in rng:
			for j in range(self.columns):
				m = pat.search(self.get_text(i, j))
				if m:
					self.set_focus_row(row=i)
					return

	def update(self, mid, channel, ghost, script, datetime):
		self.freeze()
		n = self.prepend((datetime, ghost, channel, "%2d" % 0, "%2d" % 0, script))
		self.thaw()
		self.set_row_data(n, mid)
		self.set_focus_row()

	def log_votes(self, mid, type, num):
		col = -1
		if type == VOTE:
			col = 3
		elif type == AGREE:
			col = 4
		else:
			return

		# gtk.CList.find_row_from_data does not work, this is a serious bug in PyGTK!!
		#row = current_tab.find_row_from_data(mid)
		for i in range(self.rows):
			if self.get_row_data(i) == mid:
				self.freeze()
				self.set_text(i, col, "%2d" % num)
				self.thaw()
				return

	def voting(self, widget, data, parent):
		if not self.selection:
			return None, None
		mid = self.get_row_data(self.selection[0])
		if mid is None:
			return None, None

		d = get_simple_yes_or_no_dialog(unicode(_("Really?"), 'utf-8'),
										unicode(_("Vote this bottle?"), 'utf-8'),
										parent)
		d.show()
		res = d.run()
		d.destroy()

		if not res or res == gtk.RESPONSE_REJECT:
			return None, None
		elif res == gtk.RESPONSE_ACCEPT:
			return mid, VOTE
		return None, None

	def agreeing(self, widget, data, parent):
		if not self.selection:
			return None, None
		mid = self.get_row_data(self.selection[0])
		if mid is None:
			return None, None

		d = get_simple_yes_or_no_dialog(unicode(_("Really?"), 'utf-8'),
										unicode(_("Agree this bottle?"), 'utf-8'),
										parent)
		d.show()
		res = d.run()
		d.destroy()

		if not res or res == gtk.RESPONSE_REJECT:
			return None, None
		elif res == gtk.RESPONSE_ACCEPT:
			return mid, AGREE
		return None, None

	def set_focus_row(self, row=None, row_count=None):
		if row is None:
			row = self.selection[0]
		if row_count is None:
			row_count = self.rows

		if row_count == 0:
			return
		if row_count == 1:
			r = 0
		else:
			r = min((row+0.00001)/(row_count-1),1.0)
		self.emit("scroll-vertical", gtk.SCROLL_JUMP, r)

	def get_plain_array(self):
		lines = []
		for row in range(self.rows):
			line = []
			for col in range(self.columns):
				line.append(self.get_text(row, col))
			line.append(self.get_row_data(row))
			lines.append(line)
		return lines
