# -*- coding: ascii -*-
#
#  textmanager.py - TextBuffer manipulator part for GBottler
#  Copyright (C) 2004 by Atzm WATANABE <sitosito@p.chan.ne.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#
# $Id: textmanager.py,v 1.30 2004/05/24 23:48:40 atzm Exp $
#

import os
import sys
import gtk, pango

from bottleparser import *
from types import *
from common import *

class TextManager:
	FONT_SCRIPT = 'Monospace'

	def __init__(self, app, textview, prefs, parser=BottleParser('loose')):
		self.app = app
		self.textview = textview
		self.parser = parser
		self.view_style = STYLE_TALK
		self.prefs = prefs
		self.textbuffer = self.textview.get_buffer()

		# create tags
		self.textbuffer.create_tag('script')
		self.textbuffer.create_tag('sakura')
		self.textbuffer.create_tag('kero')
		self.textbuffer.create_tag('sync')
		self.textbuffer.create_tag('error')
		self.textbuffer.create_tag('overURL')

		url_tag = self.textbuffer.create_tag('URL')
		self.font_tag = self.textbuffer.create_tag('font')
		self.default_font = self.font_tag.get_property('font')

		# some connection
		self.p_flag = False
		url_tag.connect('event', self.receive_event)
		self.textview.connect('motion-notify-event', self.motion)
		self.app.window.connect('focus-out-event', self.focus_out)

		# initialize
		self.color_changed(self.prefs.get('script'), self.prefs.get('sakura'), self.prefs.get('kero'),
						   self.prefs.get('sync'), self.prefs.get('error'), self.prefs.get('URL'))

	def color_changed(self, script_color, sakura_color, kero_color,
					  sync_color, error_color, URL_color, over_color='light gray'):
		tag_table = self.textbuffer.get_tag_table()

		tag_table.lookup('script').set_property('foreground', script_color)
		tag_table.lookup('sakura').set_property('foreground', sakura_color)
		tag_table.lookup('kero').set_property('foreground', kero_color)
		tag_table.lookup('sync').set_property('foreground', sync_color)
		tag_table.lookup('error').set_property('foreground', error_color)

		url_tag = tag_table.lookup('URL')
		url_tag.set_property('foreground', URL_color)
		url_tag.set_property('underline', pango.UNDERLINE_SINGLE)

		tag_table.lookup('overURL').set_property('background', over_color)

	def receive_event(self, texttag, widget, event, iter):
		buf = iter.get_buffer()
		end = iter.copy()

		if not iter.begins_tag(texttag):
			iter.backward_to_tag_toggle(texttag) 

		if not end.ends_tag(texttag):
			end.forward_to_tag_toggle(texttag)

		if event.type == 4 and \
			   event.button and event.button == 1: # 4 left button pressed
			os.system(self.prefs.get('browser') % buf.get_text(iter, end) + ' &')
			buf.remove_tag_by_name('overURL', iter, end)
		elif event.type == 3: # 3 mouse over?
			self.p_flag = True
			buf.apply_tag_by_name('overURL', iter, end)
			event.window.set_cursor(gtk.gdk.Cursor(gtk.gdk.HAND1))

	def focus_out(self, widget, event):
		self.p_flag = False
		self.textbuffer.remove_tag_by_name('overURL', self.textbuffer.get_start_iter(),
										   self.textbuffer.get_end_iter())
		event.window.set_cursor(None)

	def motion(self, widget, event):
		event.window.get_pointer() # why I must call this method?
		if not self.p_flag:
			self.textbuffer.remove_tag_by_name('overURL', self.textbuffer.get_start_iter(),
											   self.textbuffer.get_end_iter())
			event.window.set_cursor(None)
		else:
			self.p_flag = False

	RE_NEWLINE = re.compile('\n+$')
	def coloring(self, widget=None, event=None):
		current = ['sakura'] # tag stack for closable (currently only synchronized) sessions
		[s, e] = self.textbuffer.get_bounds()

		string = unicode(self.textbuffer.get_text(s, e), 'utf-8')
		string = self.RE_NEWLINE.sub('', string)

		pos = self.textbuffer.get_iter_at_mark(self.textbuffer.get_insert()).get_offset() # cursor position
		end = self.textbuffer.get_end_iter().get_offset()

		#print 'string:', string
		chunks = self.parser.parse(string)
		#print 'start: ', chunks

		self.textbuffer.set_text("")
		for chunk in chunks:
			if chunk[0] == TEXT_STRING:
				for n in range(len(chunk[1])):
					if chunk[1][n][0] in [SCRIPT_TEXT, TEXT_META]:
						c = chunk[1][n][1]#.replace("\\", r"\\")
						#c = c.replace("%", r"\%")
						#c = c.replace(r"\]", "]")
						self.textbuffer.insert_with_tags_by_name(self.textbuffer.get_end_iter(),
															 c, current[-1])
					elif chunk[1][n][0] == PARSE_ERROR:
						c = chunk[1][n][1]#.replace("\\", r"\\")
						self.textbuffer.insert_with_tags_by_name(self.textbuffer.get_end_iter(),
															 c, 'error')

			elif chunk[0] in [TEXT_META, PARSE_ERROR]:
				tag = 'script'
				name, args = chunk[1], chunk[2:]

				if name in [r'\0', r'\h']:
					if not current[-1] == 'sync':
						current = ['sakura']
				elif name in [r'\1', r'\u']:
					if not current[-1] == 'sync':
						current = ['kero']
				elif name == r'\_s':
					if current[-1] == 'sync' and len(current) > 1:
						current.pop()
						# when synchronized session closed
					else:
						current.append('sync')

				if chunk[0] == PARSE_ERROR:
					tag = 'error'

				self.textbuffer.insert_with_tags_by_name(self.textbuffer.get_end_iter(),
													 name, tag)

				if name == r'\URL' and chunk[0] == PARSE_ERROR and len(args) >= 2:
					format = '[%s]'
				elif name == r'\w':
					format = '%s'
				elif name == r'\s' and chunk[0] == PARSE_ERROR and (not args[0] or args[0][0] == '['):
					format = '%s'
				else:
					format = '[%s]'

				for n in range(len(args)):
					if type(args[0]) is TupleType:
						try:
							if args[n][0][0] == PARSE_ERROR:# and name == r'\n':
								self.textbuffer.insert_with_tags_by_name(self.textbuffer.get_end_iter(),
																	 "%s" % args[n][0][1], 'error')
							else:
								self.textbuffer.insert_with_tags_by_name(self.textbuffer.get_end_iter(),
																	 format % args[n][0][1], tag)
						except IndexError:
							pass
					else:
						self.textbuffer.insert_with_tags_by_name(self.textbuffer.get_end_iter(),
															 format % args[n], tag)

		[si, ei] = self.textbuffer.get_bounds()
		self.textbuffer.apply_tag_by_name('font', si, ei)

		#self.textbuffer.insert(self.textbuffer.get_end_iter(), '\n')
		if end != pos:
			if end != 0:
				self.textview.emit('move-cursor', gtk.MOVEMENT_VISUAL_POSITIONS, end*-1, gtk.FALSE)
			if pos != 0:
				self.textview.emit('move-cursor', gtk.MOVEMENT_VISUAL_POSITIONS, pos, gtk.FALSE)

	def insert_with_color_from(self, string, pos=-3):
		current = ['sakura'] # tag stack for closable (currently only synchronized) sessions
		for chunk in self.parser.parse(string):
			if chunk[0] == TEXT_STRING:
				if chunk[1][0][0] in [SCRIPT_TEXT, TEXT_META]:
					c = chunk[1][0][1]
					if self.view_style == STYLE_TALK:
						c = c.replace(r"\\", "\\")
						c = c.replace(r"\%", "%")
						c = c.replace(r"\]", "]")
					self.textbuffer.insert_with_tags_by_name(self.textbuffer.get_end_iter(),
															 c, current[-1])

			elif chunk[0] in [TEXT_META, PARSE_ERROR]:
				tag = 'script'
				name, args = chunk[1], chunk[2:]

				if name in [r'\0', r'\h']:
					if not current[-1] == 'sync':
						current = ['sakura']
				elif name in [r'\1', r'\u']:
					if not current[-1] == 'sync':
						current = ['kero']
				elif name == r'\_s':
					if current[-1] == 'sync' and len(current) > 1:
						current.pop()
						# when synchronized session closed
					else:
						current.append('sync')

				if chunk[0] == PARSE_ERROR:
					tag = 'error'

				if self.view_style == STYLE_TALK:
					pre_end = self.textbuffer.get_end_iter()
					first = pre_end.backward_char()
					if name in [r'\0', r'\h', r'\1', r'\u', r'\_s'] and \
						   pre_end.get_char() != '\n' and first:
						if current[-1] != 'sync' or name == r'\_s':
							self.textbuffer.insert(self.textbuffer.get_end_iter(), '\n')
					elif name == r'\URL':
						for n in range(len(args)):
							if not n % 2 and len(args) != 1:
								self.textbuffer.insert_with_tags_by_name(self.textbuffer.get_end_iter(),
																		 "[%s]" % args[n][0][1], tag)
							else:
								self.textbuffer.insert_with_tags_by_name(self.textbuffer.get_end_iter(),
																		 "[", tag)
								self.textbuffer.insert_with_tags_by_name(self.textbuffer.get_end_iter(),
																		 "%s" % args[n][0][1], 'URL')
								self.textbuffer.insert_with_tags_by_name(self.textbuffer.get_end_iter(),
																		 "]", tag)

				else:
					if name in [r'\0',  r'\h', r'\1', r'\u', r'\_s',
								r'\_q', r'\t', r'\c', r'\e'] and \
								len(args) == 0:
						self.textbuffer.insert_with_tags_by_name(self.textbuffer.get_end_iter(),
																 name, tag)
						continue

					if name == r'\URL':
						self.textbuffer.insert_with_tags_by_name(self.textbuffer.get_end_iter(),
																 name, tag)
					if name == r'\w':
						format = '%s%s'
					else:
						format = '%s[%s]'

					for n in range(len(args)):
						if type(args[0]) is TupleType:
							if not n % 2 and len(args) != 1:
								self.textbuffer.insert_with_tags_by_name(self.textbuffer.get_end_iter(),
																		 '[%s]' % args[n][0][1], tag)
							else:
								if name == r'\URL':
									self.textbuffer.insert_with_tags_by_name(self.textbuffer.get_end_iter(),
																			 '[', tag)
									self.textbuffer.insert_with_tags_by_name(self.textbuffer.get_end_iter(),
																			 '%s' % args[n][0][1], 'URL')
									self.textbuffer.insert_with_tags_by_name(self.textbuffer.get_end_iter(),
																			 ']', tag)
								else:
									tmp = '%s[%s]' % (name, args[n][0][1])
									self.textbuffer.insert_with_tags_by_name(self.textbuffer.get_end_iter(),
																			 tmp, tag)
						else:
							self.textbuffer.insert_with_tags_by_name(self.textbuffer.get_end_iter(),
																	 format % (name, args[n]), tag)

					if self.view_style == STYLE_SCRIPT_WITH_LINEFEED and name == r'\n':
						self.textbuffer.insert_with_tags_by_name(self.textbuffer.get_end_iter(), '\\n\n', tag)

		self.textbuffer.insert(self.textbuffer.get_end_iter(), '\n')
		[si, ei] = self.textbuffer.get_bounds()
		self.textbuffer.apply_tag_by_name('font', si, ei)

		if pos != 0:
			self.textview.emit("move-cursor", gtk.MOVEMENT_VISUAL_POSITIONS, pos, gtk.FALSE)

	def set_style_talk(self):
		self.view_style = STYLE_TALK
		self.font_tag.set_property('font', self.default_font)

	def set_style_script(self):
		self.view_style = STYLE_SCRIPT
		self.font_tag.set_property('font', self.FONT_SCRIPT)

	def set_style_script_with_linefeed(self):
		self.view_style = STYLE_SCRIPT_WITH_LINEFEED
		self.font_tag.set_property('font', self.FONT_SCRIPT)

	def set_style(self, style):
		if type(style) is not IntType:
			raise ValueError("Invalid argument: " + str(style))

		if style == STYLE_TALK:
			self.set_style_talk()
		elif style == STYLE_SCRIPT:
			self.set_style_script()
		else:
			self.set_style_script_with_linefeed()

	def clear(self):
		self.textbuffer.set_text("")
