# -*- coding: ascii -*-
#
#  xmllogs.py - XML Log manipulator for GBottler
#  Copyright (C) 2004 by Atzm WATANABE <sitosito@p.chan.ne.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#
# $Id: xmllogs.py,v 1.2 2004/05/23 02:47:49 atzm Exp $
#

import re, string, StringIO, time
from xml.sax import saxexts, saxutils
from xml.sax.handler import ContentHandler

from common import *

XML_HEADER = (
	'<?xml version="1.0" encoding="Shift_JIS"?>\r\n'
	'<?xml-stylesheet type="text/xsl" href="http://bottle.mikage.to/xbtl.xsl"?>\r\n'
	'<!DOCTYPE bottlelog [\r\n'
	'\t<!ELEMENT bottlelog (message*)>\r\n'
	'\t<!ATTLIST bottlelog version CDATA #REQUIRED>\r\n'
	'\t<!ATTLIST bottlelog generator CDATA #REQUIRED>\r\n'
	'\t<!ATTLIST bottlelog saved CDATA #REQUIRED>\r\n'
	'\t<!ELEMENT message (date, channel, script, votes, agrees, ghost)>\r\n'
	'\t<!ATTLIST message mid CDATA #REQUIRED>\r\n'
	'\t<!ELEMENT date (#PCDATA)>\r\n'
	'\t<!ELEMENT channel (#PCDATA)>\r\n'
	'\t<!ELEMENT script (#PCDATA)>\r\n'
	'\t<!ELEMENT votes (#PCDATA)>\r\n'
	'\t<!ELEMENT agrees (#PCDATA)>\r\n'
	'\t<!ELEMENT ghost (#PCDATA)>\r\n'
	']>\r\n'
	)

XML_LINE = (
	'\t<message mid="%s">\r\n'
	'\t\t<date>%s</date>\r\n'
	'\t\t<channel>%s</channel>\r\n'
	'\t\t<script>%s</script>\r\n'
	'\t\t<votes>%s</votes>\r\n'
	'\t\t<agrees>%s</agrees>\r\n'
	'\t\t<ghost>%s</ghost>\r\n'
	'\t</message>\r\n'
	)

class XMLLogHandler(ContentHandler):
	def __init__(self):
		self.parsed = []
		self.current_ch = ''

	def startDocument(self):
		pass

	def endDocument(self):
		pass

	def startElement(self, name, attrs):
		if name == 'message':
			self.parsed.append({'mid': attrs.get('mid')})

	def endElement(self, name):
		if name not in ['message', 'bottlelog']:
			self.parsed[-1][name] = string.strip(self.current_ch)
			self.current_ch = ''

	def characters(self, ch, start, length):
		self.current_ch += ch

	def initialize(self):
		self.parsed = []
		self.current_ch = ''

class XMLLogger:
	def __init__(self):
		self.parser = saxexts.make_parser()
		self.handler = XMLLogHandler()
		self.parser.setDocumentHandler(self.handler)
		self.parser.setErrorHandler(saxutils.ErrorRaiser())
		self.parsed = []

	def parse(self, filename):
		try:
			file = open(filename, 'r')
			lines = file.readlines()[1:] # SAX is not supported Shift_JIS code
			file.close()

			lines = string.join(lines, '')
			lines = unicode(lines, 'sjis', 'replace').encode('utf-8')
			lines = re.sub('\r\n|\r', '\n', lines)

			self.parser.parseFile(StringIO.StringIO(lines))
			self.parsed = self.handler.parsed
			self.handler.initialize()
		except:
			return -1

		return self.parsed

	def conv_xml(self, elem):
		elem = re.sub('&', '&amp;', elem)
		elem = re.sub('<', '&lt;', elem)
		elem = re.sub('>', '&gt;', elem)
		elem = re.sub('"', '&quot;', elem)
		elem = string.strip(elem)
		return elem

	def deconv_xml(self, elem):
		elem = re.sub('&amp;', '&', elem)
		elem = re.sub('&lt;', '<', elem)
		elem = re.sub('&gt;', '>', elem)
		elem = re.sub('&quot;', '"', elem)
		elem = string.strip(elem)
		return elem

	def save_as_xml_from(self, lists, filename):
		try:
			file = open(filename, 'w')
		except:
			return False

		saved = time.strftime('%y/%m/%d %H:%M:%S', time.localtime(time.time()))
		generator = '%s ver %s' % (APP, VER)
		version = '1.0'

		opentag = '<bottlelog saved="%s" generator="%s" version="%s">\r\n' % (saved, generator, version)

		file.write(XML_HEADER)
		file.write(opentag)

		for line in lists:
			line = map(self.conv_xml, line)
			datetime, ghost, channel, votes, agrees, script, mid = line
			xmlline = (XML_LINE % (mid, datetime, channel, script, votes, agrees, ghost)).encode('sjis')
			file.write(xmlline)
		file.write('</bottlelog>\r\n')
		file.close()
		return True

	def open_xml_from(self, filename):
		parsed = self.parse(filename)
		if parsed == -1:
			return -1

		lines = []
		for dic in parsed:
			list = map(self.deconv_xml, [dic['date'], dic['mid'], dic['votes'], dic['agrees'],
										 dic['channel'], dic['ghost'], dic['script']])
			lines.insert(0, list)
		return lines
