/** @file
 *
 * コードの大部分は、id3v2から拝借した。
 */
#if defined(HAVE_CONFIG_H)
#  include "../../config.h"
#endif
#include <cstdio>

#include <boost/format.hpp>

#include <libgnomevfs/gnome-vfs-ops.h>

#include <id3/tag.h>
#include <id3/reader.h>
#include <id3/misc_support.h>

#include "id3_filter.hpp"


namespace gdestraier {
  namespace builder {
    namespace filter {

      from_id3::from_id3() { }
      from_id3::~from_id3() { }

      factory::extention_map_type from_id3::extentions_[] = {
        { "mp3", "audio/mpeg" },
        { "mpeg3", "audio/mpeg" },
        { "ogg", "application/ogg" },
        { 0, 0 }
      };


      factory const& from_id3::get_factory() {
        static factory f("id3 tag", &from_id3::create, from_id3::extentions_);
        return f;
      }

      abstract_filter* from_id3::create() { return new from_id3; }


      namespace {
        char *genre_table[] = {
          "Blues",
          "Classic Rock",
          "Country",
          "Dance",
          "Disco",
          "Funk",
          "Grunge",
          "Hip-Hop",
          "Jazz",
          "Metal",
          "New Age",
          "Oldies",
          "Other",
          "Pop",
          "R&B",
          "Rap",
          "Reggae",
          "Rock",
          "Techno",
          "Industrial",
          "Alternative",
          "Ska",
          "Death Metal",
          "Pranks",
          "Soundtrack",
          "Euro-Techno",
          "Ambient",
          "Trip-Hop",
          "Vocal",
          "Jazz+Funk",
          "Fusion",
          "Trance",
          "Classical",
          "Instrumental",
          "Acid",
          "House",
          "Game",
          "Sound Clip",
          "Gospel",
          "Noise",
          "Alt. Rock",
          "Bass",
          "Soul",
          "Punk",
          "Space",
          "Meditative",
          "Instrum. Pop",
          "Instrum. Rock",
          "Ethnic",
          "Gothic",
          "Darkwave",
          "Techno-Indust.",
          "Electronic",
          "Pop-Folk",
          "Eurodance",
          "Dream",
          "Southern Rock",
          "Comedy",
          "Cult",
          "Gangsta",
          "Top 40",
          "Christian Rap",
          "Pop/Funk",
          "Jungle",
          "Native American",
          "Cabaret",
          "New Wave",
          "Psychadelic",
          "Rave",
          "Showtunes",
          "Trailer",
          "Lo-Fi",
          "Tribal",
          "Acid Punk",
          "Acid Jazz",
          "Polka",
          "Retro",
          "Musical",
          "Rock & Roll",
          "Hard Rock",
          "Folk",
          "Folk/Rock",
          "National Folk",
          "Swing",
          "Fusion",
          "Bebob",
          "Latin",
          "Revival",
          "Celtic",
          "Bluegrass",
          "Avantgarde",
          "Gothic Rock",
          "Progress. Rock",
          "Psychadel. Rock",
          "Symphonic Rock",
          "Slow Rock",
          "Big Band",
          "Chorus",
          "Easy Listening",
          "Acoustic",
          "Humour",
          "Speech",
          "Chanson",
          "Opera",
          "Chamber Music",
          "Sonata",
          "Symphony",
          "Booty Bass",
          "Primus",
          "Porn Groove",
          "Satire",
          /* The following were added 1999 26 Apr by Ben Gertzfield <che@debian.org> 
           * as per the list at http://mp3.musichall.cz/id3master/faq.htm but with a 
           * few spell-checks confirmed by running 'strings' on the in_mp3.dll file 
           * from winamp 2.10 (sorry :)
           */
          "Slow Jam",
          "Club",
          "Tango",
          "Samba",
          "Folklore",
          "Ballad",
          "Power Ballad",
          "Rhythmic Soul",
          "Freestyle",
          "Duet",
          "Punk Rock",
          "Drum Solo",
          "A Capella",
          "Euro-House",
          "Dance Hall",
          "Goa",
          "Drum & Bass",
          "Club-House",
          "Hardcore",
          "Terror",
          "Indie",
          "BritPop",
          "Negerpunk",
          "Polsk Punk",
          "Beat",
          "Christian Gangsta Rap",	/* DJ JC in da house */
          "Heavy Metal",
          "Black Metal",
          "Crossover",
          "Contemporary Christian",
          "Christian Rock",
          /* winamp 1.91 genres */
          "Merengue",
          "Salsa",
          "Thrash Metal",
          /* winamp 1.92 genres */
          "Anime",
          "Jpop",
          "Synthpop",
        };

        const int genre_count = sizeof(genre_table) / sizeof(char*);

        const char *GetGenreFromNum(int genre_id)
        {
          if ((genre_id >= 0) && (genre_id < genre_count))
            return genre_table[genre_id];
          else
            return "Unknown";
        }
      }



      /**
       * id3 lib に gnome_vfs 経由でIOさせる為のリーダ
       */
      class gnomevfs_reader : public ID3_Reader
      {
      protected:
        ::GnomeVFSHandle* handle_;
        ::GnomeVFSFileSize fsize_;

        enum { BUFSIZE = 1024 };
        char* buffer_;
        ::GnomeVFSFileOffset cur_;
        char* tail_;
        char* read_ptr_;


      public:
        gnomevfs_reader(::GnomeVFSHandle* handle, ::GnomeVFSFileSize fsize) :
          handle_(handle),
          fsize_(fsize),
          cur_(0)
        {
          buffer_ = (char*)std::malloc(BUFSIZE);
          tail_ = buffer_;
          read_ptr_ = buffer_;
        }
        virtual ~gnomevfs_reader() { std::free((void*)buffer_); }

        virtual void close() { ::gnome_vfs_close(handle_); handle_ = 0; }

        virtual ID3_Reader::pos_type getCur() {
          return ID3_Reader::pos_type(cur_ + (read_ptr_ - buffer_));
        }

        virtual ID3_Reader::pos_type setCur(ID3_Reader::pos_type pos) {
          if (pos >= cur_ && pos < cur_ + (tail_ - buffer_) )
            read_ptr_ = buffer_ + (pos - cur_);
          else {
            cur_ = pos;
            tail_ = buffer_;
            read_ptr_ = buffer_;
          }
          return pos;
        }

        virtual ID3_Reader::pos_type getEnd() { return fsize_; }

        bool fill_buffer() {
          ::GnomeVFSResult res = ::gnome_vfs_seek(handle_, GNOME_VFS_SEEK_START, cur_);
          if (res != GNOME_VFS_OK) {
            ::g_warning("gnome_vfs_seek: %s", ::gnome_vfs_result_to_string(res));
            return false;
          }

          ::GnomeVFSFileSize bytes_read;
          res = ::gnome_vfs_read(handle_, buffer_, ::GnomeVFSFileSize(BUFSIZE), &bytes_read);
          if (res != GNOME_VFS_OK) {
            ::g_warning("gnome_vfs_read: %s", ::gnome_vfs_result_to_string(res));
            return false;
          }

          read_ptr_ = buffer_;
          tail_ = buffer_ + bytes_read;
          return true;
        }

        virtual ID3_Reader::int_type peekChar() {
          if (cur_ >= fsize_) return ID3_Reader::END_OF_READER; // ファイル末尾より後

          if (read_ptr_ >= tail_) {
            if (! fill_buffer()) return ID3_Reader::END_OF_READER;
          }
          return *read_ptr_;  // バッファ中に存在
        }

        virtual ID3_Reader::size_type readChars(ID3_Reader::char_type buf[], ID3_Reader::size_type len) {
          ID3_Reader::char_type* dst = buf;

          while (len > 0) {
            while (read_ptr_ < tail_) {
              *dst++ = *read_ptr_++;
              if (--len == 0) return dst - &buf[0];
            }

            cur_ += read_ptr_ - buffer_;
            if (! fill_buffer()) break;
          }
          return dst - &buf[0];
        }

        virtual ID3_Reader::size_type skipChars(ID3_Reader::size_type len) {
          if (len <= tail_ - read_ptr_)
            read_ptr_ += len;
          else {
            len = std::min(fsize_ - cur_, ::GnomeVFSFileSize(len));
            cur_ += len;
            read_ptr_ = tail_ = buffer_;
          }
          return len;
        }
      };



      bool from_id3::operator() (hyperestraier::local_document* doc,
                                 gdestraier::model::index_type const& index,
                                 ::GnomeVFSURI* uri,
                                 char const* text_ur,
                                 ::GnomeVFSFileInfo* info,
                                 char const* mime_type) const
      {
        ::GnomeVFSHandle* fh;
        if (::gnome_vfs_open_uri(&fh, uri, ::GNOME_VFS_OPEN_READ) != ::GNOME_VFS_OK)
          return false;

        doc->create();
        doc->set_attr(ESTDATTRTYPE, (mime_type? mime_type : "audio/mpeg"));


        ID3_Tag tag;
        gnomevfs_reader reader(fh, info->size);
        tag.Link(reader, ID3TT_ALL);
        ID3_Tag::Iterator* iter = tag.CreateIterator();

        if (tag.NumFrames() == 0) return true; // v2タグが無かった


        for (std::size_t nf = 0; nf < tag.NumFrames(); nf++) {
          ID3_Frame* frame = iter->GetNext();
          if (frame == 0) continue;


          char const* desc = frame->GetDescription();

          ID3_FrameID frameid = frame->GetID();
          switch (frameid) {
          case ID3FID_TITLE:
            {
              char* p = ID3_GetString(frame, ID3FN_TEXT);
              std::string tmp(index.str2utf8(p));
              delete[] p;

              doc->set_attr(ESTDATTRTITLE, tmp);
              doc->add_text(tmp, true);
            }
            break;

          case ID3FID_LEADARTIST:
            {
              char* p = ID3_GetString(frame, ID3FN_TEXT);
              std::string tmp(index.str2utf8(p));
              delete[] p;
                
              doc->set_attr(ESTDATTRAUTHOR, tmp);
              doc->add_text(tmp, true);
            }
            break;

          case ID3FID_SUBTITLE:
          case ID3FID_ALBUM:
          case ID3FID_YEAR:
          case ID3FID_BPM:
          case ID3FID_COMPOSER:
          case ID3FID_COPYRIGHT:
          case ID3FID_DATE:
          case ID3FID_PLAYLISTDELAY:
          case ID3FID_ENCODEDBY:
          case ID3FID_LYRICIST:
          case ID3FID_FILETYPE:
          case ID3FID_TIME:
          case ID3FID_CONTENTGROUP:
          case ID3FID_INITIALKEY:
          case ID3FID_LANGUAGE:
          case ID3FID_SONGLEN:
          case ID3FID_MEDIATYPE:
          case ID3FID_ORIGALBUM:
          case ID3FID_ORIGFILENAME:
          case ID3FID_ORIGLYRICIST:
          case ID3FID_ORIGARTIST:
          case ID3FID_ORIGYEAR:
          case ID3FID_FILEOWNER:
          case ID3FID_BAND:
          case ID3FID_CONDUCTOR:
          case ID3FID_MIXARTIST:
          case ID3FID_PARTINSET:
          case ID3FID_PUBLISHER:
          case ID3FID_TRACKNUM:
          case ID3FID_RECORDINGDATES:
          case ID3FID_NETRADIOSTATION:
          case ID3FID_NETRADIOOWNER:
          case ID3FID_SIZE:
          case ID3FID_ISRC:
          case ID3FID_ENCODERSETTINGS:
            {
              char *p = ID3_GetString(frame, ID3FN_TEXT);
              std::string tmp(index.str2utf8(p));
              delete [] p;

              doc->add_text(tmp);
              if (desc != 0)
                doc->set_attr(desc, tmp);
              break;
            }

          case ID3FID_CONTENTTYPE:
            {
              int genre_id = 255;
              char *p = ID3_GetString(frame, ID3FN_TEXT);
              std::sscanf(p, "(%d)", &genre_id);
              std::string tmp(index.str2utf8(genre_id == 255? p : GetGenreFromNum(genre_id)));
              delete [] p;

              doc->add_text(tmp);
              doc->set_attr("genre", tmp);
              break;
            }

          case ID3FID_USERTEXT:
            {
              char* p = ID3_GetString(frame, ID3FN_TEXT);
              char* desc = ID3_GetString(frame, ID3FN_DESCRIPTION);
              std::string tmp(desc);
              tmp += ':';
              tmp += p;
              delete [] p;
              delete [] desc;

              doc->add_text(index.str2utf8(tmp));
              break;
            }
          case ID3FID_COMMENT:
          case ID3FID_UNSYNCEDLYRICS:
            {
              char* lang = ID3_GetString(frame, ID3FN_LANGUAGE);
              if (index.language_.empty() || index.language_ == "Auto" ||
                  index.language_ == lang) {
                char* p = ID3_GetString(frame, ID3FN_TEXT);
                char* desc = ID3_GetString(frame, ID3FN_DESCRIPTION);
                std::string txt(index.str2utf8(p));
                std::string dsc(index.str2utf8(desc));
                delete [] p;
                delete [] desc;

                doc->set_attr(dsc.c_str(), txt);
                doc->add_text(txt);
              }
              delete [] lang;
            }
            break;

          case ID3FID_INVOLVEDPEOPLE:
            {
              size_t n = frame->Field(ID3FN_TEXT).GetNumTextItems();
              for (size_t i = 1; i <= n; i++) {
                char *p = ID3_GetString(frame, ID3FN_TEXT, i);
                std::string tmp(index.str2utf8(p));
                doc->add_text(tmp);
                delete [] p;
              }
            }
            break;


          case ID3FID_PLAYCOUNTER:
            doc->set_attr("play-counter", (boost::format("%d") % frame->Field(ID3FN_COUNTER).Get()).str());
            break;


          case ID3FID_PICTURE:
          case ID3FID_CRYPTOREG:
          case ID3FID_GROUPINGREG:
          case ID3FID_WWWAUDIOFILE:
          case ID3FID_WWWARTIST:
          case ID3FID_WWWAUDIOSOURCE:
          case ID3FID_WWWCOMMERCIALINFO:
          case ID3FID_WWWCOPYRIGHT:
          case ID3FID_WWWPUBLISHER:
          case ID3FID_WWWPAYMENT:
          case ID3FID_WWWRADIOPAGE:
          case ID3FID_WWWUSER:
          case ID3FID_GENERALOBJECT:
          case ID3FID_AUDIOCRYPTO:
          case ID3FID_UNIQUEFILEID:
          case ID3FID_POPULARIMETER:
          case ID3FID_EQUALIZATION:
          case ID3FID_EVENTTIMING:
          case ID3FID_CDID:
          case ID3FID_MPEGLOOKUP:
          case ID3FID_OWNERSHIP:
          case ID3FID_PRIVATE:
          case ID3FID_POSITIONSYNC:
          case ID3FID_BUFFERSIZE:
          case ID3FID_VOLUMEADJ:
          case ID3FID_REVERB:
          case ID3FID_SYNCEDLYRICS:
          case ID3FID_SYNCEDTEMPO:
          case ID3FID_METACRYPTO:
          default:
            break;
          }
        }

        return true;
      }



    }
  }
}
