/** @file
 */
#if !defined(__HYPERESTRAIER__COND__HPP__)
#define __HYPERESTRAIER__COND__HPP__

#include <estraier.h>

namespace hyperestraier {

  class condition {
  protected:
    ESTCOND* raw_handle_;

  public:

    /** @brief デフォルトコンストラクタ
     */
    condition() { raw_handle_ = ::est_cond_new(); }


    /** @brief 条件フレーズを元に構築します
     *  @param phrase 条件式
     */
    explicit condition(char const* phrase) {
      raw_handle_ = ::est_cond_new();
      set_phrase(phrase);
    }


    /** @brief 条件フレーズを元にオプション付きで構築します
     *  @param phrase 条件式
     *  @param options オプション
     */
    condition(char const* phrase, int options) {
      raw_handle_ = ::est_cond_new();
      set_phrase(phrase);
      set_options(options);
    }


    /** @brief デストラクタ
     *  @remkars 仮想デストラクタを実装していないので、このクラスを継承してはいけません。
     */
    ~condition()
    {
      if (raw_handle_ != 0) ::est_cond_delete(raw_handle_);
    }

    /** @brief ネイティブの条件オブジェクトを取得します
     *  @returns ネイティブの条件オブジェクト
     */
    inline ESTCOND* raw_handle() const { return raw_handle_; }


    /** @brief インスタンスがネイティブの条件オブジェクトに結び付いているか検査します
     *  @returns 結び付いている場合はtrue。いない場合はfalse。
     */
    inline operator bool() const { return raw_handle_ != 0; }

    /** @brief インスタンスがネイティブの条件オブジェクトに結び付いていないか検査します
     *  @returns 結び付いている場合はfalse。いない場合はtrue。
     */
    inline bool operator !() const { return raw_handle_ == 0; }


    /** @brief 条件フレーズを設定します
     *  @param phrase 条件フレーズ
     */
    inline void set_phrase(char const* phrase) { ::est_cond_set_phrase(raw_handle_, phrase); }

    /** @brief 属性条件を追加します
     *  @param expr 条件式
     */
    inline void add_attr(char const* expr) { ::est_cond_add_attr(raw_handle_, expr); }

    /** @brief 順序を設定します
     *  @param expr 順序式
     */
    inline void set_order(char const* expr) { ::est_cond_set_order(raw_handle_, expr); }


    /** @brief 条件によって取得する結果の最大数を設定します
     *  @param max 取得する最大数(デフォルトは無制限)
     */
    inline void set_max(int max) { ::est_cond_set_max(raw_handle_, max); }

    /** @brief オプションを設定します
     *  @param options オプション
     *  \argval \c ESTCONDSURE 全てのN-gramキーを比較します
     *  \argval \c ESTCONDUSU  N-gramキーを1つおきに比較します
     *  \argval \c ESTCONDFAXT N-gramキーを2つおきに比較します
     *  \argval \c ESTCONDAGIT N-gramキーを3つおきに比較します
     *  \argval \c ESTCONDNOIDF TF-IDF法による重み付けを行いません
     *  \argval \c ESTCONDSIMPLE 検索フレーズを簡便方のものとして処理します
     */
    inline void set_options(int options) { ::est_cond_set_options(raw_handle_, options); }


    /** @brief 直前の検索結果に含まれる文書のスコアを取得します
     *  @param index   DB::search で戻された検索結果配列のインデックス
     */
    inline int get_score(int index) { return ::est_cond_score(raw_handle_, index); }

  };
}


#endif

