/** @file
 */
#if defined(HAVE_CONFIG_H)
#  include "../../config.h"
#endif

#include <cstring>
#include <iconv.h>
#include <cerrno>
#include <cstring>

#include <boost/format.hpp>
#include "encodings.hpp"

namespace gdestraier {
  namespace model {

    encoding const encoding::encodings[] = {
      { "BIG-5", false },
      { "CP932", false },
      { "EUC-CN", false },
      { "EUC-JP-MS", false },
      { "EUC-CN", false },
      { "EUC-JP", false },
      { "EUC-KR", false },
      { "EUC-TW", false },
      { "ISO-2022-CN", false },
      { "ISO-2022-JP", false },
      { "ISO-2022-KR", false },
      { "ISO-8859-1", true },
      { "ISO-8859-2", false },
      { "ISO-8859-3", false },
      { "ISO-8859-4", false },
      { "ISO-8859-5", false },
      { "ISO-8859-6", false },
      { "ISO-8859-7", false },
      { "ISO-8859-8", false },
      { "ISO-8859-9", false },
      { "ISO-8859-10", false },
      { "ISO-8859-11", false },
      { "ISO-8859-12", false },
      { "ISO-8859-13", false },
      { "ISO-8859-14", false },
      { "ISO-8859-15", false },
      { "ISO-8859-16", false },
      { "ISO-10646", false },
      { "SJIS-OPEN", false },
      { "SJIS-WIN", false },
      { "Shift_JIS", false },
      { "UCS2", false },
      { "UCS4", false },
      { "UTF-7", false },
      { "UTF-8", true },
      { "UTF-16BE", false },
      { "UTF-16LE", false },
      { "UTF-32BE", false },
      { "UTF-32LE", false },
      { "US-ASCII", true }
    };

    std::size_t encoding::num_encodings = sizeof(encoding::encodings) / sizeof(encoding);

    encoding const* encoding::find(char const* enc)
    {
      for (encoding const* i = encodings; i != (encodings + num_encodings); i++)
        if (std::strcmp(i->id_, enc) == 0) return i;
      return 0;
    }


    std::string
    encoding::to_utf8(std::string const& src) const {
      std::size_t n = src.size();
      char const* p = src.c_str();
      return to_utf8(p, p + n);
    }


    std::string
    encoding::to_utf8(char const* src) const { return to_utf8(src, src + std::strlen(src)); }


    std::string encoding::to_utf8(char const* first, char const* last) const
    {
      if (first == last) return std::string(); // 空の場合はなにもせずに成功させる
      if (is_utf8_compatible_) return std::string(first, last); // 変換不要だったのでコピーして戻す

      char buf[1024];
      ::iconv_t cd = ::iconv_open("UTF-8", id_);
      if (cd == iconv_t(-1))
        throw std::runtime_error((boost::format("iconv_open(%1$s to %2$s): %3$s\n") %
                                  id_ % "UTF-8" % ::strerror_r(errno, buf, sizeof(buf))).str());

      std::string result;

      std::size_t inleft = last - first;
      char* inbuf = const_cast<char*>(first);
      do {
        char* outbuf = buf;
        std::size_t outleft = 1024;
        std::size_t sz = ::iconv(cd, &inbuf, &inleft,&outbuf, &outleft);
        if (sz != std::size_t(-1))
          result.append(buf, 1024 - outleft);
        else if (errno == E2BIG)
          result.append(buf, 1024 - outleft);
        else if (errno == EINVAL)
          break;
        else if (errno == EILSEQ) {
          inleft--;
          inbuf++;
        }

      } while (inleft > 0);

      ::iconv_close(cd);

      return result;
    }


  }
}

