/** @file
 */
#if defined(HAVE_CONFIG_H)
#  include "../config.h"
#endif

#include <cstdlib>
#include <cstring>
#include <ctime>
#include <iostream>
#include <sstream>
#include <algorithm>

#include <boost/assert.hpp>
#include <boost/scoped_ptr.hpp>
#include <boost/format.hpp>
#include <boost/bind.hpp>
#include <boost/regex.hpp>
#include <boost/date_time/posix_time/posix_time.hpp>
#include <glib/gmessages.h>
#include <gtk/gtksignal.h>

#include "../gettext.h"
#include <glibmm/i18n.h>

#include <gdkmm/pixbuf.h>
#include <gdkmm/event.h>
#include <gtkmm/stock.h>
#include <gtkmm/messagedialog.h>
#include <gtkmm/icontheme.h>
extern "C" {
#include <eel/eel-open-with-dialog.h>
}
#include <cabin.h>
#include <libgnomevfs/gnome-vfs-mime-handlers.h>

#include <libgnomeui/gnome-thumbnail.h>
#include <libgnomeui/gnome-icon-lookup.h>

#include <hyperestraier/hyperestraier.hpp>

#include "gdestraier.hpp"
#include "glade-helper.hpp"
#include "thumbnail_loader.hpp"
#include "result_document.hpp"
#include "resultview.hpp"

namespace gdestraier {
  namespace gui {



    namespace {


      class model_columns_type :
        public Gtk::TreeModel::ColumnRecord
      {
      public:
        enum { ROWTYPE_NODE, ROWTYPE_DOCUMENT };

        Gtk::TreeModelColumn<int>                        row_type_;
        Gtk::TreeModelColumn<Glib::ustring>              key_string_;
        Gtk::TreeModelColumn<std::time_t>                newest_modified_timestamp_;
        Gtk::TreeModelColumn<unsigned int>               num_documents_in_group_;
        Gtk::TreeModelColumn<result_document*>  document_;

      protected:
        model_columns_type() {
          add(row_type_);
          add(key_string_); add(num_documents_in_group_);
          add(newest_modified_timestamp_);
          add(document_);
        }

      public:
        static model_columns_type const& instance() {
          static model_columns_type inst;
          return inst;
        }
      };




      class result_model :
        public Gtk::TreeStore {
      protected:
        result_model(Gtk::TreeModel::ColumnRecord const& columns) :
          Gtk::TreeStore(columns) { }

      public:
        static Glib::RefPtr<Gtk::TreeStore>
        create( Gtk::TreeModelColumnRecord const& columns)
        {
          Gtk::TreeStore* obj = new result_model(columns);
          obj->reference();
          return Glib::RefPtr<Gtk::TreeStore>(obj);
        }

      protected:

        virtual bool row_draggable_vfunc(Gtk::TreeModel::Path const& path) const
        {
          iterator iter = const_cast<result_model*>(this)->get_iter(path); // FIXME: constなget_pathがまだサポートされていないので無理矢理通す
          if(! iter) return false;

          // ドラッグできるのは文書だけ
          model_columns_type const& columns = model_columns_type::instance();
          return (iter->get_value(columns.row_type_) == model_columns_type::ROWTYPE_DOCUMENT)? true : false;
        }

        virtual bool drag_data_get_vfunc(Gtk::TreeModel::Path const& path,
                                         Gtk::SelectionData& selection_data) const
        {
          model_columns_type const& columns = model_columns_type::instance();
          iterator iter = const_cast<result_model*>(this)->get_iter(path);
          selection_data.set("text/uri-list",
                             iter->get_value(columns.document_)->uri_);

          return true;
        }

        virtual bool drag_data_delete_vfunc(Gtk::TreeModel::Path const& path)
        {
          std::cerr << "ccc" << std::endl;
          return true;
        }
      };


      ::GnomeThumbnailFactory* thumbnail_factory() {
        struct factory_holder {
          ::GnomeThumbnailFactory* obj_;
          factory_holder() { obj_ = ::gnome_thumbnail_factory_new(GNOME_THUMBNAIL_SIZE_NORMAL); }
          ~factory_holder() { ::g_object_unref(G_OBJECT(obj_)); }
        };

        static factory_holder f;
        return f.obj_;
      }
    }




    resultview::resultview()
    {
    }



    resultview::resultview(GtkTreeView* cobject, ::GladeXML* glade)
      : Gtk::TreeView(cobject),
        group_key_(gdestraier::model::preferences::GROUP_BY_MIME_TYPE),
        sort_key_(gdestraier::model::preferences::SORT_BY_SCORE)
    {
      // Initialize view.
      model_columns_type const& columns = model_columns_type::instance();
      model_ = result_model::create(columns);
      model_->set_default_sort_func(sigc::mem_fun(this, &resultview::on_compare_row));
      set_model(model_);

      main_column_.pack_start(thumbnail_renderer_, false);
      main_column_.set_cell_data_func(thumbnail_renderer_, sigc::mem_fun(this, &resultview::on_thumbnail_cell_data));

      info_renderer_.property_yalign().set_value(0);
      info_renderer_.property_xalign().set_value(0);
      main_column_.pack_start(info_renderer_, true);
      main_column_.set_cell_data_func(info_renderer_, sigc::mem_fun(this, &resultview::on_info_cell_data) );
      append_column(main_column_);

      set_enable_search(true);
      //set_search_column(columns.snippet_);


      // DNDを有効にします
      std::list<Gtk::TargetEntry> target_entries;
      target_entries.push_back(Gtk::TargetEntry("text/uri-list", Gtk::TargetFlags(0), 0) );
      enable_model_drag_source(target_entries,
                               Gdk::MODIFIER_MASK,
                               Gdk::ACTION_COPY | Gdk::ACTION_MOVE | Gdk::ACTION_LINK | Gdk::ACTION_ASK);


      signal_button_press_event().connect_notify(sigc::mem_fun(this, &resultview::on_button_pressed));

      get_selection()->signal_changed().connect(boost::bind(&resultview::on_sel_changed, this));

      // Initialize popup context menu
      context_menu_.get_popup()->accelerate(*this);
      context_menu_.signal_open_.connect(boost::bind(&resultview::on_open_document_activate, this, _1));
      context_menu_.signal_open_with_.connect(boost::bind(&resultview::on_open_document_with_activate, this));
      context_menu_.signal_open_parent_.connect(boost::bind(&resultview::on_open_document_parent_activate, this));
      context_menu_.signal_properties_.connect(boost::bind(&resultview::on_document_properties_activate, this));
    }


    /** @brief デストラクタ
     */
    resultview::~resultview()
    {
      model_->unset_default_sort_func();
    }




    /** @brief 結果リストを消去します
     */
    void
    resultview::clear()
    {
      //model_columns_type const& columns = model_columns_type::instance();

      found_documents_ = 0;
      found_indexes_ = 0;
      total_indexes_ = 0;



      // Clear all keywords and release that memory.
      {
        keywords_.clear();
        keywords_type empty;
        keywords_.swap(empty);
      }

      results_.clear();  // Free all results;
      model_->clear();   // Clear all row in model.
    }



    /** @berif 検索を実行します
     *  検索を実行して、結果をリストに追加します
     *
     *  @param phrase  検索式
     *  @param target_index 検索対象のインデックス。0を指定すると、全てのインデックスから検索します。
     */
    void
    resultview::do_search(Glib::ustring const& phrase,
                          gdestraier::model::index_type const* target_index)
    {
      //
      // Initialize
      //
      clear();

      // Compile condition exp.
      hyperestraier::condition cond(phrase.c_str(), (ESTCONDUSUAL | ESTCONDSIMPLE | ESTCONDSCFB));


      if (target_index != 0)
        search_impl(cond, target_index); // Try to search for single index.
      else {
        //
        // Try to search for all indexes.
        //
        gdestraier::model::preferences const& pref = gdestraier::gui::get_preferences();
        for (gdestraier::model::preferences::indexes_type::const_iterator iindex = pref.indexes_.begin();
             iindex != pref.indexes_.end(); iindex++) {
          if (! iindex->active_) continue; // Ignore if inactive.

          search_impl(cond, &*iindex);
        }
      }

      // キーワードの色つけを決定します
      {
        unsigned int h = 0;
        for (keywords_type::iterator i = keywords_.begin(); i != keywords_.end(); i++)
          i->second.highlight_ = h++;
      }


      //
      // ツリーのモデルを更新します
      //
      update_tree();
    }



    /** @brief 単一のインデックスに対しての検索実装部
     *  @param cond 検索フレーズ
     *  @param target_index 目標のインデックス
     */
    void
    resultview::search_impl(hyperestraier::condition& cond,
                            gdestraier::model::index_type const* target_index)
    {
      if (target_index->get_max_documents() > 0)
        cond.set_max(target_index->get_max_documents());

      gdestraier::model::preferences const& pref = gdestraier::gui::get_preferences();


      unsigned int num_docs_in_index = 0;

      switch (target_index->database_location_) {
      case gdestraier::model::index_type::LOCAL_FILESYSTEM:
        {
          // Open index database
          typedef hyperestraier::local_database<hyperestraier::multi_threaded> db_type;
          db_type db(target_index->database_path_.c_str(), ESTDBREADER | ESTDBLCKNB);
          if (! db) { // Open filed
            Gtk::MessageDialog msg((boost::format(_("Could not open index database: %1$s on %2$s")) %
                                    db.error_message() % target_index->database_path_).str(),
                                   false, Gtk::MESSAGE_ERROR, Gtk::BUTTONS_CLOSE, true);
            msg.run();
            return ;
          }


          // 検索実行
          CBMAP* hints = ::cbmapopen();
          db_type::search_result_type result(db.search(cond, hints));

          // ヒントからキーワードのリストと、合致する文書数を抽出します
          ::cbmapiterinit(hints);
          CBLIST* words = ::cblistopen();
          char const* key;
          while ( (key = ::cbmapiternext(hints, 0)) != 0) {
            if (key[0] != '\0') {
              int num_docs = std::atoi(::cbmapget(hints, key, -1, 0));
              if (num_docs >= 0) {
                ::cblistpush(words, key, -1);
                keywords_[Glib::ustring(key)].num_docs_ += num_docs;
              }
            }
          }
          ::cbmapclose(hints);
      

          for (db_type::search_result_type::const_iterator i = result.begin();
               num_docs_in_index < target_index->get_max_documents() && i != result.end();
               i++) {
            hyperestraier::local_document doc(db.get_doc(*i, 0)); // 文書オブジェクトを取り出す
            if (! doc) continue; // インデックスが壊れていると起こり得る

            // 要約を取得します
            char* snippet = 0;
            if (target_index->get_use_snippet()) {
              snippet = doc.make_snippet(words, 480, 0, 66);
#if 0
              if (std::strchr(snippet, '\t')  == 0) {
                // snippet中にキーワードが無いという事はゴミ文書だったという事なので、無視する
                std::free(snippet);
                continue;
              }
#endif
            }

            num_docs_in_index++;
            add_result_doc(&doc, cond.get_score(i - result.begin()), snippet, target_index);
          }

          ::cblistclose(words);
        }
        break;


      case gdestraier::model::index_type::REMOTE_NODE:
        {
          // データベースをオープン
          typedef hyperestraier::node_database db_type;
          db_type db(target_index->database_path_.c_str());
          db.set_timeout(pref.network_.timeout_);
          db.set_auth(target_index->user_.c_str(), target_index->password_.c_str());

          std::string proxy_host;
          int         proxy_port;
          pref.get_proxy(&proxy_host, &proxy_port);
          if (! proxy_host.empty())
            db.set_proxy(proxy_host.c_str(), proxy_port);

          if (! target_index->get_use_snippet())
            db.set_snippet_width(0, 0, 0);
          else
            db.set_snippet_width(480, 0, 66);


          // 検索実行
          hyperestraier::node_result result(db.search(cond, target_index->get_depth()));
          if (! result) {
            Gtk::MessageDialog msg((boost::format(_("Failed to search on node database: %1$s")) %
                                    target_index->database_path_).str(),
                                   false, Gtk::MESSAGE_ERROR, Gtk::BUTTONS_CLOSE, true);
            msg.run();
            return ;
          }

          // キーワードとヒット数を計上する
          ::CBMAP* hints = result.hints();
          ::cbmapiterinit(hints);
          char const* key;
          while ((key = ::cbmapiternext(hints, 0)) != 0) {
            if (std::memcmp(key, "HINT#", 5) == 0) {
              char const* first = ::cbmapget(hints, key, -1, 0);
              if (first != 0) {
                char const* tail = std::strchr(first, '\t');
                if (tail != 0) {
                  int num_docs = std::atoi(tail + 1);
                  keywords_[Glib::ustring(first, tail)].num_docs_ += num_docs;
                }
              }
            }
          }

          // 結果に追加してまわる
          int docs = std::min(result.doc_num(), target_index->get_max_documents());
          num_docs_in_index += docs;
          for (int i = 0; i < docs; i++) {
            hyperestraier::node_document doc(result.get_doc(i));
            add_result_doc(&doc,
                           std::atoi(doc.get_attr("#nodescore", "-1")),
                           (target_index->get_use_snippet()? ::strdup(doc.make_snippet()) : 0),
                           target_index);
          }
        }
        break;

      default:
        BOOST_ASSERT(0 && "Unknown database location.");
        return ;
      }

      total_indexes_++;

      if (num_docs_in_index > 0) {
        found_indexes_++;
        found_documents_ += num_docs_in_index;
      }
    }
  



    void 
    resultview::add_result_doc(hyperestraier::document const* doc,
                               int score,
                               char* snippet,
                               gdestraier::model::index_type const* target_index)
    {
      boost::regex const& uri_replace_regex = target_index->get_uri_replace_regex_compiled();
      boost::regex const& title_replace_regex = target_index->get_title_replace_regex_compiled();


      // Add document to list of results.
      results_.push_back(result_document());
      result_document&  new_doc = results_.back();
      if (! target_index->is_valid_uri_replace_regex())
        new_doc.uri_ = doc->get_attr(ESTDATTRURI, "");
      else
        new_doc.uri_ = boost::regex_replace(std::string(doc->get_attr(ESTDATTRURI, "")),
                                            uri_replace_regex,
                                            target_index->get_uri_replace_to(),
                                            boost::match_default | boost::format_all);
      {
        char const* mime_type = doc->get_attr(ESTDATTRTYPE, 0);
        if (mime_type != 0) {
          char const* p;
          for (p = mime_type; *p != '\0' && *p != ';'; p++) ;
          new_doc.mime_type_.assign(mime_type, p);
        } else
          new_doc.mime_type_ = "application/octet-stream";
      }
      new_doc.mime_type_desc_   = ::gnome_vfs_mime_get_description(new_doc.get_mime_type().c_str());
      if (new_doc.mime_type_desc_ == 0) new_doc.mime_type_desc_ = new_doc.mime_type_.c_str();

      char const* title = doc->get_attr(ESTDATTRTITLE, 0);
      if (title == 0) title = doc->get_attr("_lfile", 0);
      if (title == 0) title = _("(no title)");

      if (! target_index->is_valid_uri_replace_regex())
        new_doc.title_ = title;
      else
        new_doc.title_ = boost::regex_replace(std::string(title),
                                              title_replace_regex,
                                              target_index->get_title_replace_to(),
                                              boost::match_default | boost::format_all);

      new_doc.author_             = doc->get_attr(ESTDATTRAUTHOR, _("unknown"));
      new_doc.language_           = doc->get_attr(ESTDATTRLANG, _("unknown"));

      // 更新日時の表記を正規化する
      {
        std::time_t mtime = ::cbstrmktime(doc->get_attr(ESTDATTRMDATE, "")); // エラーの時はそのまま -1 を突っ込む
        new_doc.last_modified_time_ = mtime;
        if (mtime < 0)
          new_doc.last_modified_ = "";
        else {
          char time_buf[128];
          struct tm tm;
          std::strftime(time_buf, 128, "%c", ::localtime_r(&mtime, &tm));
          new_doc.last_modified_      = time_buf;
        }
      }

      new_doc.index_              = target_index;
      new_doc.snippet_.reset(snippet);
      new_doc.score_              = score;
    }


    template <typename GetKeyProc>
    void
    resultview::update_tree_impl(GetKeyProc get_key)
    {
      typedef std::map<Glib::ustring, Gtk::TreeStore::iterator> iter_cache_type;;
      iter_cache_type iter_cache;

      model_columns_type const& columns = model_columns_type::instance();

      for (results_type::iterator idoc = results_.begin(); idoc != results_.end(); idoc++) {
        Glib::ustring key = get_key(&*idoc);

        // この文書の属するグループの親ノードを取得します
        typename iter_cache_type::iterator inode = iter_cache.find(key);
        Gtk::TreeStore::iterator node;
        if (inode != iter_cache.end()) {
          node = inode->second; // すでに挿入済みのグループ

          unsigned int n = node->get_value(columns.num_documents_in_group_);
          node->set_value(columns.num_documents_in_group_, n + 1);
        } else {  // 新しいグループ
          node = model_->append();
          node->set_value(columns.row_type_, int(model_columns_type::ROWTYPE_NODE));
          node->set_value(columns.key_string_, key);
          node->set_value(columns.num_documents_in_group_, static_cast<unsigned int>(1));

          iter_cache[key] = node;
        }

        // 文書を追加します
        Gtk::TreeStore::iterator leaf = model_->append(node->children());
        leaf->set_value(columns.row_type_, int(model_columns_type::ROWTYPE_DOCUMENT));
        leaf->set_value(columns.document_, &*idoc);
        idoc->tree_iter_ = leaf;
      }
    }



    /** @brief ツリーモデルを更新します
     *
     *  現在のグループ化の指定に従い、検索結果の文書をツリーモデルへ登録します。
     */
    void
    resultview::update_tree()
    {
      model_->clear();    // Clear all rows in model.

      // 追加中はソートの為の比較を抑止する
      model_->set_sort_column(GTK_TREE_SORTABLE_UNSORTED_SORT_COLUMN_ID,
                              Gtk::SORT_ASCENDING);

      switch (group_key_) {
      case gdestraier::model::preferences::GROUP_BY_NONE:
        {
          model_columns_type const& columns = model_columns_type::instance();
          for (results_type::iterator idoc = results_.begin(); idoc != results_.end(); idoc++) {
            Gtk::TreeStore::iterator leaf = model_->append();
            leaf->set_value(columns.row_type_, int(model_columns_type::ROWTYPE_DOCUMENT));
            leaf->set_value(columns.document_, &*idoc);
            idoc->tree_iter_ = leaf;
          }

          break;
        };

      case gdestraier::model::preferences::GROUP_BY_INDEX:
        update_tree_impl(std::mem_fun(&result_document::get_index_name));
        break;
      case gdestraier::model::preferences::GROUP_BY_MIME_TYPE:
        update_tree_impl(std::mem_fun(&result_document::get_mime_type_desc));
        break;
      case gdestraier::model::preferences::GROUP_BY_AUTHOR:
        update_tree_impl(std::mem_fun(&result_document::get_author));
        break;
      case gdestraier::model::preferences::GROUP_BY_LANGUAGE:
        update_tree_impl(std::mem_fun(&result_document::get_language));
        break;
      case gdestraier::model::preferences::GROUP_BY_LAST_MODIFIED:
        {
          model_columns_type const& columns = model_columns_type::instance();
          std::time_t now = std::time(NULL); // この瞬間の時刻を基準にする

          typedef std::map<Glib::ustring, Gtk::TreeStore::iterator> group_iters_type;
          group_iters_type group_iters;

          {
            // まずユニークなタイムスタンプを昇順に取得しておく
            std::vector<std::time_t> timestamps;
            timestamps.reserve(results_.size());
            for (results_type::iterator i = results_.begin(); i != results_.end(); i++)
              timestamps.push_back(i->get_last_modified_time());
            std::sort(timestamps.begin(), timestamps.end(), std::greater<std::time_t>());

            // 全てのグループヘッダをあらかじめ追加しておく
            for (std::vector<std::time_t>::iterator i = timestamps.begin(); i != timestamps.end(); i++) {
              Glib::ustring age = gdestraier::timestamp::to_age(now, *i);

              group_iters_type::iterator j = group_iters.find(age);
              if (j == group_iters.end()) {
                Gtk::TreeStore::iterator node = model_->append();
                node->set_value(columns.row_type_, int(model_columns_type::ROWTYPE_NODE));
                node->set_value(columns.key_string_, age);
                node->set_value(columns.num_documents_in_group_, static_cast<unsigned int>(0));
                node->set_value(columns.newest_modified_timestamp_, *i);

                group_iters[age] = node;
              } else {
                // このグループ内で最新の更新タイムスタンプを記録しておく
                std::time_t t = j->second->get_value(columns.newest_modified_timestamp_);
                if (t < *i)
                  j->second->set_value(columns.newest_modified_timestamp_, *i);
              }
            }
          }

          // 全ての文書を追加していく
          for (results_type::iterator idoc = results_.begin(); idoc != results_.end(); idoc++) {
            Glib::ustring age = idoc->get_age(now);

            // この文書の属するグループの親ノードを取得します
            Gtk::TreeStore::iterator node = group_iters[age];
            unsigned int n = node->get_value(columns.num_documents_in_group_);
            node->set_value(columns.num_documents_in_group_, n + 1);

            // 文書を追加します
            Gtk::TreeStore::iterator leaf = model_->append(node->children());
            leaf->set_value(columns.row_type_, int(model_columns_type::ROWTYPE_DOCUMENT));
            leaf->set_value(columns.document_, &*idoc);
          }
        }
        break;
      default:
        BOOST_ASSERT(0 && "Bad group key.");
        break;
      }

      // 全て追加し終わったので、ソートキーを設定する
      model_->set_sort_column(Gtk::TreeSortable::DEFAULT_SORT_COLUMN_ID, Gtk::SORT_ASCENDING);
    }




    /**
     * Default row comparator
     */
    int resultview::on_compare_row(Gtk::TreeModel::iterator const& l,
                                   Gtk::TreeModel::iterator const& r) const
    {
      model_columns_type const& cols = model_columns_type::instance();

      if (l->get_value(cols.row_type_) == model_columns_type::ROWTYPE_NODE ||
          r->get_value(cols.row_type_) == model_columns_type::ROWTYPE_NODE) {

        if (group_key_ == gdestraier::model::preferences::GROUP_BY_LAST_MODIFIED) {
          std::time_t ltime = l->get_value(cols.newest_modified_timestamp_);
          std::time_t rtime = r->get_value(cols.newest_modified_timestamp_);
          return (ltime == rtime)? 0 : ( (ltime > rtime)? -1 : 1); // あくまで新鮮度の比較なので時刻の大小関係を逆に取る
        }
        Glib::ustring lstr = l->get_value(cols.key_string_);
        Glib::ustring rstr = r->get_value(cols.key_string_);
        return (lstr == rstr)? 0 : ( (lstr < rstr)? -1 : 1);
      }

      result_document const* ldoc = l->get_value(cols.document_);
      result_document const* rdoc = r->get_value(cols.document_);

      if (ldoc == 0 || rdoc == 0) return 0;

      /* スコアは高い方を上に、更新日時は新しい方を上にするので、大小関係を逆に取る */
      switch (sort_key_) {
      case gdestraier::model::preferences::SORT_BY_SCORE:
        return (ldoc->score_ == rdoc->score_)? 0 : ((ldoc->score_ > rdoc->score_)? -1 : 1);
      case gdestraier::model::preferences::SORT_BY_LAST_MODIFIED:
        return (ldoc->get_last_modified_time() == rdoc->get_last_modified_time())? 0:
          ( (ldoc->get_last_modified_time() > rdoc->get_last_modified_time())? -1 : 1);
      case gdestraier::model::preferences::SORT_BY_TITLE:
        return (ldoc->title_ == rdoc->title_)? 0 : ((ldoc->title_ < rdoc->title_)? - 1: 1);
      }
      return 0;
    }




    /** Cell data callback for thumbnail.
     */
    void
    resultview::on_thumbnail_cell_data(Gtk::CellRenderer* renderer, Gtk::TreeModel::iterator const& iter) const
    {
      Gtk::CellRendererPixbuf* r = dynamic_cast<Gtk::CellRendererPixbuf*>(renderer);
      BOOST_ASSERT(r != 0 && "Dynamic cast failed on resultview::on_thumbnail_cell_data.");

      gdestraier::model::preferences const& pref = gdestraier::gui::get_preferences();

      model_columns_type const& columns = model_columns_type::instance();
      switch (iter->get_value(columns.row_type_)) {
      case model_columns_type::ROWTYPE_NODE:
        // Draw empty image.

        //r->property_stock_id().set_value(Gtk::Stock::DIRECTORY.id);
        r->property_visible().set_value(false);
        r->property_pixbuf().set_value(Glib::RefPtr<Gdk::Pixbuf>(0));
        break;

      case model_columns_type::ROWTYPE_DOCUMENT:
        {
          result_document* doc = iter->get_value(columns.document_);
          if (! doc->thumbnail_ && ! doc->cannt_thumbnail_) { // Load thumbnail if available.
            if (thumbnail_loader::get_default().load(&doc->thumbnail_,
                                                     doc->uri_.c_str(),
                                                     doc->get_last_modified_time(),
                                                     doc->get_mime_type().c_str()) < 0)
              doc->cannt_thumbnail_ = true;
          }

          if (!! doc->thumbnail_) {
            r->property_pixbuf().set_value(doc->thumbnail_);
          } else {
            // Use MIME-type icon insteed.
            Glib::RefPtr<Gtk::IconTheme> icon_theme(Gtk::IconTheme::get_default());
            GnomeIconLookupResultFlags result_flags; // ::gnome_icon_lookup should init.
            char* iconname = ::gnome_icon_lookup(icon_theme->gobj(),
                                                 0,
                                                 doc->uri_.c_str(),
                                                 0,
                                                 0,
                                                 doc->get_mime_type().c_str(),
                                                 ::GnomeIconLookupFlags(0),
                                                 &result_flags);
            r->property_pixbuf().set_value(icon_theme->load_icon(iconname, pref.thumbnail_size_, Gtk::IconLookupFlags(0)) );
            ::g_free(iconname);
          }
          r->set_fixed_size(pref.thumbnail_size_, -1);
          r->property_visible().set_value(true);
        }
      }
    }




    /** Cell data callback for markup
     */
    void
    resultview::on_info_cell_data(Gtk::CellRenderer* renderer, Gtk::TreeModel::iterator const& iter)
    {
      Gtk::CellRendererText* r = dynamic_cast<Gtk::CellRendererText*>(renderer);
      BOOST_ASSERT(r != 0 && "Dynamic cast failed on resultview::on_info_cell_data.");

      gdestraier::model::preferences const& pref = gdestraier::gui::get_preferences();
      model_columns_type const& columns = model_columns_type::instance();

      // Cache UTF-8 string for performance.
      static Glib::ustring txt_author(_("Author:"));
      static Glib::ustring txt_lang(_("Lang:"));
      static Glib::ustring txt_mime_type(_("File type:"));
      static Glib::ustring txt_in_index(_(" in index "));
      static Glib::ustring txt_last_modified(_("Last-modified:"));
      static Glib::ustring txt_score(_("Score:"));

      std::ostringstream markup;
      markup.imbue(std::locale::classic());

      switch (iter->get_value(columns.row_type_)) {
      case model_columns_type::ROWTYPE_NODE:
        pref.font_and_colors_.group_header_.write_pango_markup(markup, 
                                                               iter->get_value(columns.key_string_),
                                                               false);
        pref.font_and_colors_.group_header_
          .write_pango_markup(markup,
                              (boost::format(" (%1d)") % iter->get_value(columns.num_documents_in_group_)).str(),
                              false);
        r->property_cell_background().set_value(pref.font_and_colors_.background_.highlight_color_);
        r->property_markup().set_value(markup.str());
        break;

      case model_columns_type::ROWTYPE_DOCUMENT:
        {
          result_document* doc = iter->get_value(columns.document_);
          
          //
          // Build markup string if no cached.
          //
          if (doc->markup_cache_.empty()) {

            pref.font_and_colors_.title_.write_pango_markup(markup, doc->title_, false);

            markup << txt_in_index.raw();
            pref.font_and_colors_.index_.write_pango_markup(markup, doc->get_index_name(), false);
            markup.put('\n');


            pref.font_and_colors_.score_.write_pango_markup(markup,
                                                            txt_score + (boost::format("%d") % doc->get_score()).str(),
                                                            false);
            markup << ", ";

            pref.font_and_colors_.author_.write_pango_markup(markup,
                                                             txt_author + doc->get_author(),
                                                             false);
            markup << ", ";


            pref.font_and_colors_.language_.write_pango_markup(markup,
                                                               txt_lang + doc->get_language(),
                                                               false);
            markup << ", ";


            pref.font_and_colors_.file_type_.write_pango_markup(markup,
                                                                txt_mime_type + doc->mime_type_desc_,
                                                                false);
            markup << ", ";

            pref.font_and_colors_.last_modified_.write_pango_markup(markup,
                                                                    txt_last_modified + doc->last_modified_,
                                                                    false);
            markup.put('\n');

            // Build snppet
            if (doc->snippet_.get() != 0) {
              char* line = doc->snippet_.get();
              while (*line != '\0' && *line == '\n') line++;

              bool first_line = true;
              while (*line != '\0') {
                if (*line == '\n' && line[1] != '\0') {
                  if (! first_line) markup << "\n";
                  first_line = false;
                  markup << "...";
                  line++;
                } else {
                  char * p;
                  for (p = line; *p != '\0' && *p != '\t' && *p != '\n'; p++) ;
                  Glib::ustring const word(line, p);

                  if (*p == '\t') {
                    char* pp = p;
                    while (*pp != '\n' && *pp != '\0') pp++;
                    unsigned int hn = keywords_[Glib::ustring(p + 1, pp)].highlight_ % gdestraier::model::preferences::NUM_HIGHLIGHT;
                    pref.font_and_colors_.keyword_[hn].write_pango_markup(markup, word, false);
                    p = pp;
                  } else {
                    char* escaped = ::cbxmlescape(word.c_str());
                    markup << escaped;
                    std::free(static_cast<void*>(escaped));

                    while (*p != '\0' && *p != '\n') p++;
                  }

                  if (*p == '\0') break;
                  line = p + 1;
                }
              }
            }

            doc->markup_cache_ = markup.str();
          }

          r->property_cell_background().set_value(pref.font_and_colors_.background_.normal_color_);
          r->property_markup().set_value(doc->markup_cache_);
        }
        break;
      }

    }



    /** @brief 結果リストにある件数を取得します
        @returns 件数
     */
    resultview::size_type
    resultview::size() const
    {
      return model_->children().size();
    }



    /** @brief コンテキストメニューを更新します
     */
    bool
    resultview::update_context_menu()
    {
      context_menu_.clear();

      Glib::RefPtr<Gtk::TreeSelection> selection = get_selection();
      if (selection->count_selected_rows() != 1) return false;

      model_columns_type const& columns = model_columns_type::instance();
      Gtk::TreeModel::iterator selected_row = selection->get_selected();

      if (selected_row->get_value(columns.row_type_) != model_columns_type::ROWTYPE_DOCUMENT)
        return false;

      result_document* doc = selected_row->get_value(columns.document_);
      context_menu_.update(doc->uri_, doc->mime_type_);
      return true;
    }



    /** @brief ボタン押下イベントを override します
     *  @param ev イベント
     */
    void
    resultview::on_button_pressed(GdkEventButton *ev)
    {
      //bool result = Gtk::TreeView::on_button_press_event(ev);
      if (ev->type == Gdk::BUTTON_PRESS && ev->button == 3) {
        // ボタン3押下

        if (update_context_menu())
          context_menu_.get_popup()->popup(ev->button, ev->time);
      }
    }



    /** @brief 選択状態が変化した時に呼び出されます
     */
    void
    resultview::on_sel_changed()
    {
      context_menu_.get_popup()->popdown();
      context_menu_.clear();
    }



    /** @brief メニュー [Open with XXXX]
     */
    void
    resultview::on_open_document_activate(Glib::ustring const& appid)
    {
      // 選択されている文書のURIを列挙します
      ::GList* uris = 0;
        struct helper {
          static void add(GList** uris, Gtk::TreeModel::iterator i) {
            model_columns_type const& columns = model_columns_type::instance();
            if (i->get_value(columns.row_type_) == model_columns_type::ROWTYPE_DOCUMENT)
              *uris = ::g_list_append(*uris, const_cast<char*>(i->get_value(columns.document_)->uri_.c_str()) );
          }
        };
        get_selection()->selected_foreach_iter(boost::bind(&helper::add, &uris, _1));


        // 文書が選択されていればアプリケーションを起動します。
        if (uris != 0) {
#if defined(HAVE_GNOME_VFS_MIME_APPLICATION_NEW_FROM_DESKTOP_ID)
          ::GnomeVFSMimeApplication* app = ::gnome_vfs_mime_application_new_from_desktop_id(appid.c_str());
#else
          ::GnomeVFSMimeApplication* app = ::gnome_vfs_mime_application_new_from_id(appid.c_str());
#endif

          ::gnome_vfs_mime_application_launch(app, uris);

          ::gnome_vfs_mime_application_free(app);
          ::g_list_free(uris);
        }
    }



    /** @brief メニュー [Open with...]
     */
    void
    resultview::on_open_document_with_activate()
    {
      try {
        struct helper {
          static void on_application_selected(::EelOpenWithDialog* dialog,
                                              ::GnomeVFSMimeApplication* application,
                                              ::gpointer user_data) {
            ::GList* uris = ::g_list_append(0, const_cast<char*>(((Glib::ustring*)user_data)->c_str()) );
            ::gnome_vfs_mime_application_launch(application, uris);
            ::g_list_free(uris);
          }

          static void open(Gtk::TreeModel::iterator i) {
            model_columns_type const& columns = model_columns_type::instance();
            if (i->get_value(columns.row_type_) == model_columns_type::ROWTYPE_DOCUMENT) {
              result_document* doc = i->get_value(columns.document_);

              ::GtkWidget* w = ::eel_open_with_dialog_new(doc->mime_type_.c_str(),
                                                          doc->uri_.c_str());
              ::gtk_signal_connect(GTK_OBJECT(w), "application_selected",
                                   (void (*)())&helper::on_application_selected, (void*)&doc->uri_);
              ::gtk_dialog_run(GTK_DIALOG(w));
            }
          }
        };

        get_selection()->selected_foreach_iter(&helper::open);

      }
      catch (std::exception e) {
        Gtk::MessageDialog msg(e.what(), false, Gtk::MESSAGE_ERROR, Gtk::BUTTONS_CLOSE, true);
        msg.run();
      }
    }


    void
    resultview::on_open_document_parent_activate()
    {
      struct helper {
        static void open(Gtk::TreeModel::iterator i) {
          // TODO:
        }
      };

      get_selection()->selected_foreach_iter(&helper::open);
    }


    void
    resultview::on_document_properties_activate()
    {
      Gtk::MessageDialog msg(_("No implemented yet"), false, Gtk::MESSAGE_ERROR, Gtk::BUTTONS_CLOSE, true);
      msg.run();
    }


  }
}
