unit arenascript;
	{ This unit holds the scripting language for GearHead. }
	{ It's pretty similar to the scripts developed for DeadCold. }
	{ Basically, certain game events will add a trigger to the }
	{ list. In the main combat procedure, if there are any pending }
	{ triggers, they will be checked against the events list to }
	{ see if anything happens. }

	{ Both the triggers and the event scripts will be stored as }
	{ normal string attributes. }

	{ This unit also handles conversations with NPCs, since those }
	{ are written using the scripting language and may use any }
	{ commands available there + a few special commands. }
{
	GearHead2, a roguelike mecha CRPG
	Copyright (C) 2005 Joseph Hewitt

	This library is free software; you can redistribute it and/or modify it
	under the terms of the GNU Lesser General Public License as published by
	the Free Software Foundation; either version 2.1 of the License, or (at
	your option) any later version.

	The full text of the LGPL can be found in license.txt.

	This library is distributed in the hope that it will be useful, but
	WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
	General Public License for more details. 

	You should have received a copy of the GNU Lesser General Public License
	along with this library; if not, write to the Free Software Foundation,
	Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
}
{$LONGSTRINGS ON}

interface

uses sysutils,gears_base,gears,locale,
{$IFDEF ASCII}
	vidmenus,vidgfx;
{$ELSE}
	sdlmenus,sdl;
{$ENDIF}

const
	NAG_ScriptVar = 0;
	NAG_SkillCounter = -16;	{ Counter for skill tests. }
	Max_Plots_Per_Story = 5;

	NAG_ArenaData = -18;	{ Used to store information about mecha arena combat }
		NAS_ArenaState = 1;	{ Determines what exactly is happening at the arena now. }
			NAV_AS_Vacant = 0;	{ No fight now, no fight scheduled }
			NAV_AS_Ready = 1;	{ Start fight next time PC enters scene }
			NAV_AS_Battle = 2;	{ Battle in progress }
			NAV_AS_Win = 3;		{ PC has won the battle }
			NAV_AS_Loss = 4;	{ PC hass lost the battle }
		NAS_ArenaWins = 2;	{ # of times PC has won match }
		NAS_ArenaThreat = 4;	{ Threat value of enemy mecha }
		NAS_ArenaForces = 5;	{ % of generic enemies to fight }
		NAS_ChallengerID = 6;	{ NPC challenger present during battle }
		NAS_ChallengerHome = 7;	{ Where to return champion after fight }
		NAS_ArenaRecharge = 8;	{ Time when next fight can take place }

	{ When playing in arena mode, the following string attributes will be added to the scene }
	{ following battle. }
	ARENAREPORT_CharDied = 'AR_PCDIED';
	ARENAREPORT_CharRecovered = 'AR_PCRECOVERED';
	ARENAREPORT_MechaDestroyed = 'AR_MECHADIED';
	ARENAREPORT_MechaRecovered = 'AR_MECHARECOVERED';
	ARENAREPORT_MechaObtained = 'AR_MECHAOBTAINED';
	ARENAREPORT_Personal = 'AR_PERSONAL';

	TAG_SALVAGED_MAP = 'SalvagedMap';

var
	{ This gear pointer will be created if a dynamic scene is requested. }
	SCRIPT_DynamicEncounter: GearPtr;

	{ **************************************** }
	{ ***  INTERACTION  GLOBAL  VARIABLES  *** }
	{ **************************************** }
	{ These variables hold information that may be needed anywhere }
	{ while interaction is taking place, but are undefined if }
	{ interaction is not taking place. }
	{ IntMenu should let procedures know whether or not interaction }
	{ is currently happening or not- if IntMenu <> Nil, we're in the }
	{ middle of a conversation and all other interaction variables }
	{ should have good values. }
	IntMenu: RPGMenuPtr;	{ Interaction Menu }
	I_PC,I_NPC: GearPtr;	{ Pointers to the PC & NPC Chara gears }
	I_NPC_org: GearPtr;	{ Pointers to the backup NPC Chara gears }
				{  When ASLs "SetNPC E4 EMail d3" is executed, }
				{ original I_NPC is preserved in I_NPC_org }
				{ and E4 is set to I_NPC. }
				{  At the end of an ASL EMail, }
				{ the content of I_NPC_org returns to I_NPC. }
	I_Persona: GearPtr;	{ The conversation currently being used. }

	Grabbed_Gear: GearPtr;	{ This gear can be acted upon by }
				{ generic commands. }
	Grabbed_Gear_Brace: Boolean;	{ Inside of brace for GrabGear or not. }

	lancemate_tactics_persona: GearPtr;	{ Persona for setting lancemate tactics. }
	BLANK_PERSONA: GearPtr;			{ Simple persona for default NPCs. }
	rumor_leads: GearPtr;			{ Mini-conversations for finding rumors. }

	DestroyInvcomAtDelinkJJang: Boolean;

	NeedGC: Boolean;


Procedure SetLancemateOrders( GB: GameBoardPtr );
Function NumLancemateSlots( Adv,PC: GearPtr ): Integer;

Procedure BrowseMemoType( GB: GameBoardPtr; Tag: String );

Function YesNoMenu( GB: GameBoardPtr; Prompt,YesMsg,NoMsg: String ): Boolean;
Function YesNoMenu( GB: GameBoardPtr; Prompt,YesMsg,NoMsg: String; Choice_of_Default: Boolean ): Boolean;

Function BasicSkillTarget( Renown: Integer ): Integer;
Function HardSkillTarget( Renown: Integer ): Integer;
Function Calculate_Reward_Value( GB: GameBoardPtr; Renown,Percent: LongInt ): LongInt;

Function ScriptValue( var Event: String; GB: GameBoardPtr; Scene: GearPtr ): LongInt;

Function AS_GetString( Source: GearPtr; Key: String ): String;
Function ScriptMessage( msg_label: String; GB: GameBoardPtr; Source: GearPtr ): String;
Function NPCScriptMessage( const msg_label: String; GB: GameBoardPtr; NPC, Source: GearPtr ): String;

Procedure InvokeEvent( Event: String; GB: GameBoardPtr; Source: GearPtr; var Trigger: String );

Procedure AddLancemate( GB: GameBoardPtr; NPC: GearPtr );
Function AddLancemateFrontEnd( GB: GameBoardPtr; PC,NPC: GearPtr; CanCancel: Boolean ): Boolean;
Procedure RemoveLancemate( GB: GameBoardPtr; Mek: GearPtr; DoMessage: Boolean );

Procedure HandleInteract( GB: GameBoardPtr; PC,NPC,Persona: GearPtr; with_DialogMsg: String );
Procedure DoTalkingWIthNPC( GB: GameBoardPtr; PC,Mek: GearPtr; ByTelephone: Boolean );

Function TriggerGearScript( GB: GameBoardPtr; Source: GearPtr; var Trigger: String ): Boolean;
Function CheckTriggerAlongPath( var T: String; GB: GameBoardPtr; Plot: GearPtr; CheckAll: Boolean ): Boolean;
Procedure HandleTriggers( GB: GameBoardPtr );


Function StartRescueScenario( GB: GameBoardPtr; PC: GearPtr; Context: String ): Boolean;

Procedure PutAwayArenascript( const Mek: GearPtr );
Procedure InitArenascript;
Procedure DoScriptGC( GB: GameBoardPtr );


implementation

uses
     errmsg,
     utf8msg,action,arenacfe,ability,gearutil,ghchars,gearparser,ghmodule,backpack,
     ghprop,ghweapon,grabgear,interact,menugear,playwright,rpgdice,
     services,texutil,ui4gh,wmonster,narration,description,skilluse,
	ghintrinsic,movement,minigame,customization,aibrain,mpbuilder,
{$IFDEF ASCII}
	vidmap,vidinfo,colormenu;
{$ELSE}
	sdlgfx,sdlmap,sdlinfo,colormenu;
{$ENDIF}

const
	CMD_Chat = -2;
	CMD_Join = -3;
	CMD_Quit = -4;
	CMD_WhereAreYou = -5;
	CMD_AskAboutRumors = -6;

var
	DEBUG_cmd_org: String;	{ for DEBUG }
	DEBUG_cmd_org2: String;	{ for DEBUG }
	DEBUG_cmd_org3: String;	{ for DEBUG }
	Script_Macros,Value_Macros,Default_Scene_Scripts: SAttArrayPtr;
	ASRD_GameBoard: GameBoardPtr;
	ASRD_MemoMessage: String;
	local_triggers: SAttListPtr;
	ForceInteract_wo_SayAnything: Boolean;

Function Make_ErrorMessage_ASL( const Source: GearPtr; const cmd: String ): String;
var
	MsgOut: String;
begin
	MsgOut := 'In "' + GearName( Source ) + '", ';
	if '' <> cmd then begin
		MsgOut := MsgOut + 'ASL:"' + cmd + '", ';
	end;
	Make_ErrorMessage_ASL := MsgOut;
end;

Function Make_ErrorMessage_ASL_CONTEXT1( const Event: String ): String;
var
	MsgOut: String;
begin
	MsgOut := '';
	if ('' <> DEBUG_cmd_org2) then begin
		MsgOut := MsgOut + 'SUB-CONTEXT: ';
	end else begin
		MsgOut := MsgOut + 'CONTEXT: ';
	end;
	if '' <> Event then begin
		MsgOut := MsgOut + '"' + Event + '" ';
	end;
	Make_ErrorMessage_ASL_CONTEXT1 := MsgOut;
end;

Function Make_ErrorMessage_ASL_CONTEXT2( const Event: String ): String;
var
	MsgOut: String;
begin
	MsgOut := '';
	if ('' <> DEBUG_cmd_org2) then begin
		MsgOut := MsgOut + 'in <' + DEBUG_cmd_org2 + '>';
	end else begin
		MsgOut := MsgOut + 'in <' + DEBUG_cmd_org + '>';
	end;
	Make_ErrorMessage_ASL_CONTEXT2 := MsgOut;
end;

Function Make_ErrorMessage_ASL_CONTEXT3( const Event: String ): String;
var
	MsgOut: String;
begin
	MsgOut := '';
	if ('' <> DEBUG_cmd_org3) then begin
		MsgOut := MsgOut + ' as <' + DEBUG_cmd_org3 + '>.';
	end else begin
		MsgOut := MsgOut + '.';
	end;
	Make_ErrorMessage_ASL_CONTEXT3 := MsgOut;
end;

{  ******************************  }
{  ***   REDRAW  PROCEDURES   ***  }
{  ******************************  }

Procedure InteractRedraw;
	{ Redraw the screen for whatever interaction is going to go on. }
begin
	CombatDisplay( ASRD_GameBoard );
	SetupInteractDisplay( PlayerBlue );
	if I_NPC <> Nil then begin
		DisplayInteractStatus( ASRD_GameBoard , I_NPC , CHAT_React , I_Endurance );

	end;
	GameMsg( CHAT_Message , ZONE_InteractMsg , InfoHiLight );
end;

Procedure ArenaScriptReDraw;
	{ Redraw the combat screen for some menu usage. }
begin
	if ASRD_GameBoard <> Nil then CombatDisplay( ASRD_GameBoard );
end;

Procedure MemoPageReDraw;
	{ Redraw the combat screen for some menu usage. }
begin
	if ASRD_GameBoard <> Nil then CombatDisplay( ASRD_GameBoard );
	SetupMemoDisplay;
	GameMsg( ASRD_MemoMessage , ZONE_MemoText , StdWhite );
end;

Procedure ChoiceReDraw;
	{ Redraw the combat screen for some menu usage. }
begin
	if ASRD_GameBoard <> Nil then CombatDisplay( ASRD_GameBoard );
	SetupMemoDisplay;
	GameMsg( ASRD_MemoMessage , ZONE_MemoMenu , StdWhite );
end;



{  ****************************  }
{  ***   EVERYTHING  ELSE   ***  }
{  ****************************  }

Function BasicSkillTarget( Renown: Integer ): Integer;
	{ Return an appropriate target for skill rolls for someone of the listed renown. }
var
	it: Integer;
begin
	it := Renown div 8 + 3;
	if it < 5 then it := 5;
	BasicSkillTarget := it;
end;

Function HardSkillTarget( Renown: Integer ): Integer;
	{ Return a difficult target for skill rolls for someone of the listed renown. }
var
	it: Integer;
begin
	it := Renown div 7 + 10;
	if it < 9 then it := 9;
	HardSkillTarget := it;
end;

Function SocSkillTarget( GB: GameBoardPtr; Renown: Integer ): Integer;
	{ Return a social difficult target for skill rolls based on Renown and modified }
	{ by the relationship between I_PC and I_NPC. }
var
	it,react: Integer;
begin
	it := Renown div 7 + 7;
	if it < 10 then it := 10;

	{ If the PC and NPC exist, apply the special modifier. }
	if ( I_PC <> Nil ) and ( I_NPC <> Nil ) then begin
		react := ReactionScore( GB^.Scene , I_PC , I_NPC );
		it := it - ( react div 10 );
	end;

	SocSkillTarget := it;
end;


Function CreateRumorList( GB: GameBoardPtr; Rumor_Source: GearPtr; SkRoll: Integer; DoHarvest: Boolean; var Rumor_Error: Boolean; const RuMemo_Msg,Rumor_Head: String ): SAttListPtr;
	{ RUMOR_HEAD is the rumor type to collect: RUMORs or RUMEMOs. }
	{ RUMEMO_Msg is the leading and trailing message appended to a rumor if we're }
	{ harvesting them for RUMEMOs. }

	{ If DoHarvest is TRUE, all found rumors will be changed to RUMEMOs. }
	{ ...if FALSE, then SkRoll is ignored and all messages are returned. }

	{ Search through the adventure, revealing rumors as you go. When a rumor }
	{ is discovered, do the following: }
	{ - Check to see if SkRoll is high enough to reveal it. }
	{ - If so, add it to the list of rumors. }
	{ - Also convert the original rumor to a rumor memo. }
	{ If no rumors are found at all, set RUMOR_ERROR to TRUE; otherwise FALSE. }
var
	InfoList: SAttListPtr;

	Function RumorSKTarget( Part: gearPtr ): Integer;
		{ Return the skill roll needed to learn this rumor. }
	var
		Plot: GearPtr;
		it: Integer;
	begin
		{ If we're dealing with a plot-based rumor, get the difficulty number from the plot. }
		Plot := PlotMaster( GB , Part );
		if Plot <> Nil then begin
			{ Use the difficulty rating for this plot. }
			it := SocSkillTarget( GB , NAttValue( Plot^.NA , NAG_Narrative , NAS_DifficultyLevel ) );
		end else begin
			it := 5;
		end;

		{ In order to keep the process somewhat unpredictable, randomize the target }
		{ just a bit. }
		it := it - 5 + RollStep( 3 );
		if Random( 3 ) = 1 then it := it + Random( 4 );

		RumorSKTarget := it;
	end;

	Procedure AddThisRumor( Part: GearPtr; Rumor: SAttArrayElementPtr );
		{ Add this rumor to the list... maybe. }
	var
		msg,rh: String;
	begin
		msg := Rumor^.Info;
		if ( Length( msg ) <= 0 ) then exit;
		if DoHarvest then begin
			if SkRoll > RumorSkTarget( Part ) then begin
				{ This rumor has been retrieved. }
				if ( 0 < Length( msg ) ) then begin
					StoreSAttList( InfoList , msg );
				end;
				rh := rumor^.Code;
				msg := ReplaceHash( RuMemo_Msg , msg );
				msg := FormatChatStringByGender( msg, I_NPC );
				RemoveSAttArray( Part^.SA , Rumor );
				SetSAttArray( Part^.SA , 'RUMEMO' + Copy( rh , Length( Rumor_Head ) + 1 , Length( rh ) ) , msg );
			end;
		end else begin
			{ We aren't converting to RUMEMO right now. Ignore the skill roll }
			{ and simply return the message without changing anything. }
			if ( 0 < Length( msg ) ) then begin
				StoreSAttList( InfoList , msg );
			end;
		end;
		{ Set RUMOR_ERROR to FALSE, since we just found one. }
		RUMOR_ERROR := FALSE;
	end;

	Procedure GetRumorFromGear( P: GearPtr );
		{ Retrieve the rumor info from this gear, without caring about }
		{ what kind of gear it is. Well, for the most part, anyhow... }
	var
		Rumor: SAttArrayElementPtr;
		Level: LongInt;
	begin
		{ First add the basic rumor.  }
		Rumor := FindSAttArray( P^.SA , Rumor_Head );
		if Rumor <> Nil then AddThisRumor( P , Rumor );

		{ Next add the quest rumor. }
		Level := NAttValue( P^.NA , NAG_Narrative , NAS_PlotID );
		if Level < 0 then begin
			Rumor := FindSAttArray( P^.SA , Rumor_Head + BStr( NAttValue( FindRoot( GB^.Scene )^.NA , NAG_PlotStatus , Level ) ) );
			if Rumor <> Nil then AddThisRumor( P , Rumor );
		end else if (( P^.G = GG_Persona ) or ( P^.G = GG_MetaScene )) and ( P^.Parent <> Nil ) and ( P^.Parent^.G = GG_Plot ) then begin
			{ Finally add the plot rumor. }
			Rumor := FindSAttArray( P^.SA , Rumor_Head + BStr( NAttValue( FindRoot( GB^.Scene )^.NA , NAG_PlotStatus , Level ) ) );
			if Rumor <> Nil then AddThisRumor( P , Rumor );
		end;
	end;

	Procedure RumorWorkup( P: GearPtr ); Forward;
	Procedure ExtractData( P: GearPtr );
		{ Store all relevant info from PART. }
		{ If P is of certain types, we're gonna have to harvest the data from }
		{ its associated bits. Characters also need the data from their Personas, }
		{ gates to metascenes need to check there, and scenes get faction data. }
	var
		Persona: GearPtr;
	begin
		if ( P <> Rumor_Source ) and ( P^.G <> GG_Persona ) then begin
			if P <> GB^.Scene then GetRumorFromGear( P );
			if P^.G = GG_Character then begin
				Persona := SeekPersona( GB , NAttValue( P^.NA , NAG_Personal , NAS_CID ) );
				if Persona <> Nil then begin
					{ Previously we'd just collect the rumor from the persona }
					{ and be done with it, but since Quests have been introduced }
					{ the rumor associated with a given NPC can change depending }
					{ on quest state. }
					GetRumorFromGear( Persona );
				end;

			end else if ( P^.G = GG_MetaTerrain ) and ( P^.Stat[ STAT_Destination ] < 0 ) then begin
				{ Find the metascene, and do a complete rumor work-up of it. }
				Persona := FindActualScene( GB , P^.Stat[ STAT_Destination ] );
				if Persona <> Nil then begin
					RumorWorkup( Persona );
				end;
			end;
		end;
	end;
	Procedure RumorWorkup( P: GearPtr );
		{ Do a complete rumor workup on P, gathering info from it }
		{ and all its child gears. }
	var
		P2: GearPtr;
	begin
		if P = Nil then Exit;
		ExtractData( P );
		P2 := P^.SubCom;
		while P2 <> Nil do begin
			RumorWorkup( P2 );
			P2 := P2^.Next;
		end;
		P2 := P^.InvCom;
		while P2 <> Nil do begin
			RumorWorkup( P2 );
			P2 := P2^.Next;
		end;
	end;
var
	Part: GearPtr;

begin
	{ Assume an error until we find a rumor to prove us wrong. }
	Rumor_Error := TRUE;

	InfoList := CreateSAttList;

	Part := FindRootScene( GB^.Scene );
	RumorWorkup( Part );

	Part := GB^.Meks;
	while Part <> Nil do begin
		ExtractData( Part );
		Part := Part^.Next;
	end;

	CreateRumorList := InfoList;
end;

Function RevealRumors( GB: GameBoardPtr; NPC: GearPtr; SkRoll: Integer; var Rumor_Error: Boolean ): SAttListPtr;
	{ Reveal some rumors! Call the CreateRumorList procedure with a standard reveal. }
begin
	RevealRumors := CreateRumorList( GB, NPC, SkRoll, TRUE, Rumor_Error, ReplaceHash( UTF8_MsgString( 'RevealRumors' , '#1 said that #2' ) , PilotName( NPC ) , '#' ), 'RUMOR' );
end;

Function ReviewRumorMemos( GB: GameBoardPtr ): SAttListPtr;
	{ Create the list of rumor memos. }
var
	Rumor_Error: Boolean;	{ A dummy variable. }
begin
	ReviewRumorMemos := CreateRumorList( GB, Nil, 0, FALSE, Rumor_Error, '#', 'RUMEMO' );
end;

Procedure BrowseMemoType( GB: GameBoardPtr; Tag: String );
	{ Create a list, then browse the memos based upon this }
	{ TAG type. Possible options are MEMO, NEWS, and EMAIL. }
var
	MemoList: SAttListPtr;
	M: SAttListElementPtr;
	Adv: GearPtr;

	Procedure HarvestPlotMemos( LList: SAttArrayPtr );
		{ This list may contain plot memos. How to tell? The first four }
		{ characters will be "MEMO". It's just like harvesting the history. }
	var
		i: Longint;
		tmp: SAttArrayElementPtr;
	begin
		if 0 < NumSAttArray( LList ) then begin
			for i := 0 to ( LList^.size - 1 ) do begin
				tmp := LList^.AoH[i];
				if ( ( NIL <> tmp ) and ( tmp^.act ) ) then begin
					if Copy( tmp^.Code , 1 , 4 ) = 'MEMO' then begin
						if ( 0 < Length( tmp^.Info ) ) then begin
							StoreSAttList( MemoList , tmp^.Info );
						end;
					end;
				end;
			end;
		end;
	end;
	Procedure CreateMemoList( Part: GearPtr; Tag: String );
		{ Look through all gears in the structure recursively, }
		{ looking for MEMO string attributes to store in our list. }
	var
		msg: String;
		QID: LongInt;
	begin
		while Part <> Nil do begin
			if Part^.G <> GG_AbsolutelyNothing then begin
				if ( tag = 'MEMO' ) and ( Part^.G = GG_Plot ) then begin
					{ This is a plot. It may have subplot memos. These are }
					{ memos that have the PLOTID attached to their butts. Why? }
					{ Because I realized, somewhat late, that a plot which can }
					{ contain multiple narrative threads really needs multiple }
					{ memos as well. }
					HarvestPlotMemos( Part^.SA );
				end else begin
					{ Not a plot. Just do the regular harvesting work, then. }
					msg := SAttArrayValue( Part^.SA , Tag );
					if msg <> '' then begin
						if ( 0 < Length( msg ) ) then begin
							StoreSAttList( MemoList , msg );
						end;
					end;

					{ This part may also have a quest-related message attached }
					{ to it. See if that's so. }
					QID := NAttValue( Part^.NA , NAG_Narrative , NAS_PlotID );
					if ( QID <> 0 ) then begin
						msg := SAttArrayValue( Part^.SA , Tag + '_' + BStr( NAttValue( Adv^.NA , NAG_PlotStatus , Qid ) ) );
						if msg <> '' then begin
							if ( 0 < Length( msg ) ) then begin
								StoreSAttList( MemoList , msg );
							end;
						end;
					end;
				end;

				CreateMemoList( Part^.SubCom , Tag );
				CreateMemoList( Part^.InvCom , Tag );
			end; { if Part = AbsolutelyNothing }
			Part := Part^.Next;
		end;
	end;
	Procedure AddQuestMemos;
		{ Quest memos work a bit differently than other memos. First, }
		{ they only appear so long as the quest they're assigned to is }
		{ active (i.e. it has a nonnegative QID). }
	var
		i: Longint;
		SA: SAttArrayElementPtr;
		msg_head: String;
		qid: LongInt;
	begin
		if 0 < NumSAttArray( Adv^.SA ) then begin
			for i := 0 to ( Adv^.SA^.size - 1 ) do begin
				SA := Adv^.SA^.AoH[i];
				if ( ( NIL <> SA ) and ( SA^.act ) ) then begin

					{ If this string attribute is potentially a quest memo, }
					{ try to extract its QuestID. If this memo is no longer }
					{ valid then delete it. }
					if HeadMatchesString( 'MEMO_' , SA^.Code ) then begin
						msg_head := SA^.Code;
						msg_head := Copy( msg_head , 6 , Length( msg_head ) );
						qid := ExtractValue( msg_head );
						if ( qid <> 0 ) and ( NAttValue( Adv^.NA , NAG_PlotStatus , Qid ) > -1 ) then begin
							{ Add it to the list. }
							if ( 0 < Length( SA^.Info ) ) then begin
								StoreSAttList( MemoList , SA^.Info );
							end;
						end else begin
							{ Invalid quest memo. Get rid of it. }
							RemoveSAttArray( Adv^.SA , SA );
						end;
					end;
				end;
			end;
		end;
	end;
	Function PlaceMemoPhoneCall( PC: GearPtr; MMsg: String ): Boolean;
		{ We want to make a phone call to someone mentioned in this memo. }
		{ Search through all the NPCs on the game board and within the current city. }
		{ Add them to a menu. Then, query the menu, and do talking with the selected NPC. }
		{ Return TRUE if a conversation was had, or FALSE otherwise. }
		Procedure CheckAlongList( RPM: RPGMenuPtr; LList: GearPtr );
			{ Check along this list for NPCs mentioned in the memo, recursively }
			{ searching through children as well. }
			{ Add any good NPCs found to the menu, using their CID as the key. }
		var
			Name: String;
			CID: LongInt;
			UnknownName: Boolean;
			UnknownFace: Boolean;
			UnknownAddress: Boolean;
		begin
			while LList <> Nil do begin
				if LList^.G = GG_Character then begin
					{ This is a character. Is it somebody we're looking for? }
					Name := GearName( LList );
					CID := NAttValue( LList^.NA , NAG_Personal , NAS_CID );
					if ( CID <> 0 ) and ( Pos( Name , MMsg ) > 0 ) then begin
						if CanContactByPhone( GB , PC , LList , UnknownName , UnknownFace , UnknownAddress ) then begin
							if CHEAT_No_CheckPhonelist_Memo or ( not UnknownAddress ) then begin
								AddRPGMenuItem( RPM , GearName( LList ) , CID );
							end else if ( not UnknownName ) then begin
								AddRPGMenuItem( RPM , GearName( LList ) , 0 );
							end;
						end;
					end;
				end else begin
					{ Not a character. Recurse like mad! }
					CheckAlongList( RPM , LList^.SubCom );
					CheckAlongList( RPM , LList^.InvCom );
				end;

				LList := LList^.Next;
			end;
		end;
	var
		RPM: RPGMenuPtr;
		city,NPC: GearPtr;
		CID: LongInt;
	begin
		{ Step One- Create the menu. }
		RPM := CreateRPGMenu( MenuItem , MenuSelect , @ZONE_MemoMenu );
		CheckAlongList( RPM , GB^.Meks );
		City := FindRootScene( GB^.Scene );
		if City <> Nil then begin
			CheckAlongList( RPM , City^.SubCom );
			CheckAlongList( RPM , City^.InvCom );
		end;
		{ RPMSortAlpha( RPM ); } { In UTF-8, this sort causes a result that I cannot assume. }
		AlphaKeyMenu( RPM );
		if RPM^.NumItem > 0 then begin
			AddRPGMenuItem( RPM , MsgString( 'Cancel' ) , SELECTMENU_Cancel );
		end else begin
			AddRPGMenuItem( RPM , MsgString( 'MEMO_CALL_NoPeople' ) , SELECTMENU_Cancel );
		end;

		{ Step Two- Query the menu and locate the NPC. }
		CID := SelectMenu( RPM , @MemoPageRedraw );

		{ Step Three- Pass the request along to HandleInteract. }
		if ( 0 = CID ) then begin
			DialogMsg( ReplaceHash( UTF8_MsgString( 'PCTelephone' , 'UnknownAddress' ) , RPMLocateByPosition( RPM , RPM^.selectitem )^.msg ) );
		end else if ( SELECTMENU_Enable <= CID ) then begin
			NPC := GG_LocateNPC( CID , GB , GB^.Scene );
			if NPC <> Nil then begin
				DoTalkingWithNPC( GB , PC , NPC , True );
			end;
		end else NPC := Nil;

		DisposeRPGMenu( RPM );
		PlaceMemoPhoneCall := NPC <> Nil;
	end;
	Procedure BrowseList( NoMemo: Boolean );
		{ Actually browse the created list. }
	var
		RPM: RPGMenuPtr;
		N,D: Integer;
		PC: GearPtr;
		Msg: String;
	begin
		{ Locate the PC. We need it for the PComm capability. }
		PC := GG_LocatePC( GB );
		if 0 < NumSAttList( MemoList ) then begin
			RPM := CreateRPGMenu( MenuItem , MenuSelect , @ZONE_MemoMenu );
			AddRPGMenuItem( RPM , UTF8_MsgString( 'BrowseMemoType' , Tag + '_Next' ) , 1 );
			AddRPGMenuItem( RPM , UTF8_MsgString( 'BrowseMemoType' , Tag + '_Prev' ) , 2 );
			if ( PC <> Nil ) and HasPCommCapability( PC , PCC_Phone ) then AddRPGMenuItem( RPM , MsgString( 'MEMO_Call' ) , 3 );
			AddRPGMenuKey( RPM , RPK_Right ,  1 );
			AddRPGMenuKey( RPM , RPK_Left , 2 );
			AddRPGMenuKey( RPM , KeyMap[ KMC_East ].KCode , 1 );
			AddRPGMenuKey( RPM , KeyMap[ KMC_West ].KCode , 2 );
			AlphaKeyMenu( RPM );
			RPM^.Mode := RPMNoCleanup;
			N := 1;

			repeat
				M := RetrieveSAttList( MemoList , N );
				Msg := FormatGearNameString( M^.Info );
				ASRD_GameBoard := GB;
				if NoMemo then begin
					ASRD_MemoMessage := Msg;
				end else begin
					ASRD_MemoMessage := '(' + BStr( N ) + '/' + BStr( NumSAttList( MemoList ) ) + ') ' + Msg;
				end;
				SetFlag_PhoneList( GB , Msg , True , False , False , True );
				D := SelectMenu( RPM , @MemoPageRedraw );


				if D = 1 then begin
					N := N + 1;
					if N > NumSAttList( MemoList ) then N := 1;
				end else if D = 2 then begin
					N := N - 1;
					if N < 1 then N := NumSAttList( MemoList );
				end else if D = 3 then begin
					{ We want to place a phone call to someone mentioned }
					{ in this memo. Make it so. }
					PlaceMemoPhoneCall( PC , Msg );
					if not BrowseMemo_NoReturn_after_PhoneCall then begin
						D := SELECTMENU_Cancel;
					end;
				end;
			until ( SELECTMENU_Cancel = D ) or not KeepPlayingSC( GB );

			DisposeRPGMenu( RPM );
		end;
		DisposeSAttList( MemoList );

	end;
	Function NoMemoError: String;
		{ Return a string which will explain to the user that there are }
		{ no memos of the selected type. }
	var
		msg: String;
	begin
		msg := MsgString( 'MEMO_No_' + Tag );
		if msg = '' then msg := ReplaceHash( MsgString( 'MEMO_None' ) , LowerCase( Tag ) );
		NoMemoError := msg;
	end;
var
	NoMemo: Boolean;
begin
	{ Error check first - we need the GB and the scene for this. }
	if ( GB = Nil ) or ( GB^.Scene = Nil ) then Exit;
	tag := UpCase( Tag );
	Adv := FindRoot( GB^.Scene );
	if ( Tag = 'RUMEMO' ) then begin
		MemoList := ReviewRumorMemos( GB );
	end else begin
		MemoList := CreateSAttList;
		CreateMemoList( Adv , Tag );
	end;
	if Tag = 'MEMO' then AddQuestMemos;

	{ Sort the memo list. }
	NoMemo := False;
	if ( 0 = NumSAttList( MemoList ) ) then begin
		StoreSAttList( MemoList , NoMemoError );
		NoMemo := True;
	end;

	BrowseList( NoMemo );
end;

Function YesNoMenu( GB: GameBoardPtr; Prompt,YesMsg,NoMsg: String ): Boolean;
begin
	YesNoMenu := YesNoMenu( GB , Prompt , YesMsg , NoMsg , True );
end;

Function YesNoMenu( GB: GameBoardPtr; Prompt,YesMsg,NoMsg: String; Choice_of_Default: Boolean ): Boolean;
	{ This will open up a small window in the middle of the map }
	{ display, then prompt the user for a choice. }
	{ Return TRUE if the "yes" option was selected, or FALSE if }
	{ the "no" option was selected. }
	{ This function performs no screen cleanup. }
var
	rpm: RPGMenuPtr;
	N: Integer;
begin
	RPM := CreateRPGMenu( MenuItem , MenuSelect , @ZONE_MemoMenu );
	if ( 0 = Length( YesMsg ) ) and ( 0 = Length( NoMsg ) ) then begin
		AddRPGMenuItem( RPM , MsgString( 'Continue' ) , 1 );
	end else begin
		AddRPGMenuItem( RPM , YesMsg , 1 );
		if ( YesMsg <> NoMsg ) then begin
			AddRPGMenuItem( RPM , NoMsg , SELECTMENU_Cancel );
		end;
	end;
	if ( '' <> YesMsg ) then begin
		RPM^.Mode := RPMNoCancel;
	end;

	SetItemByValue( RPM , 1 );
	if not Choice_of_Default then begin
		SetItemByValue( RPM , SELECTMENU_Cancel );
	end;

	ASRD_GameBoard := GB;
	ASRD_MemoMessage := Prompt;
	N := SelectMenu( RPM , @MemoPageRedraw );

	DisposeRPGMenu( RPM );

	{ Do cleanup before branching. }
	CombatDisplay( GB );

	YesNoMenu := ( SELECTMENU_Cancel <> N );
end;

Procedure SetLancemateOrders( GB: GameBoardPtr );
	{ Go through the lancemates, and assign any orders they might need. }
var
	PCUID,DefOrder: LongInt;
	mek: gearPtr;
begin
	{ Step one- find the PC's UID. }
	mek := GB^.meks;
	while mek <> Nil do begin
		if ( NAttValue( mek^.NA , NAG_Location , NAS_Team ) = NAV_DefPlayerTeam ) and IsMasterGear( mek ) then begin
			{ This must be the PC. }
			PCUID := NAttValue( mek^.NA , NAG_EpisodeData , NAS_UID );
		end;
		mek := mek^.Next;
	end;

	{ Step two- look for the lancemates and set their orders. }
	mek := GB^.meks;
	while mek <> Nil do begin
		if ( NAttValue( mek^.NA , NAG_Location , NAS_Team ) = NAV_LancemateTeam ) and IsMasterGear( mek ) then begin
			DefOrder := NAttValue( mek^.NA , NAG_Personal , NAS_LancemateOrders );
			if DefOrder = NAV_Passive then begin
				SetNAtt( mek^.NA , NAG_EpisodeData , NAS_Orders , NAV_Passive );
			end else if DefOrder = NAV_Follow then begin
				SetNAtt( mek^.NA , NAG_EpisodeData , NAS_Orders , NAV_Follow );
				SetNAtt( mek^.NA , NAG_EpisodeData , NAS_ATarget , PCUID );
			end else begin
				SetNAtt( mek^.NA , NAG_EpisodeData , NAS_Orders , NAV_SeekAndDestroy );
			end;
		end;
		mek := mek^.Next;
	end;
end;

Function NumLancemateSlots( Adv,PC: GearPtr ): Integer;
	{ Return the number of freely-selected lancemates this PC can have. }
var
	N: Integer;
begin
	{ You get one lancemate for free. }
	N := 1;

	{ You can earn extra lancemates via merit badges. }
	if HasMeritBadge( Adv , NAS_MB_Lancemate2 ) then begin
		Inc( N );
		if HasMeritBadge( Adv, NAS_MB_Lancemate3 ) then Inc( N );
	end;

	{ Or, you can take a talent for one extra. }
	if HasTalent( PC , NAS_Entourage ) then begin
		Inc( N );
	end;

	NumLancemateSlots := N;
end;

Function CanJoinLance( GB: GameBoardPtr; PC,NPC: GearPtr ): Boolean;
	{ Return TRUE if NPC can join the lance right now, or FALSE otherwise. }
var
	ERen: Integer;	{ Lancemate Points needed, Effective Renown }
	CanJoin: Boolean;
begin
	ERen := NAttValue( PC^.NA , NAG_CharDescription , NAS_Renowned );
	if ERen < 15 then ERen := 15;
	ERen := ERen + CStat( PC , STAT_Charm );
	CanJoin := True;
	if ( NPC = Nil ) or ( NPC^.G <> GG_Character ) then begin
		CanJoin := False;
	end else if NAttValue( NPC^.NA , NAG_CharDescription , NAS_Renowned ) > ERen then begin
		CanJoin := False;
	end else if ( GB <> Nil ) and ( ReactionScore( GB^.Scene , PC , NPC ) < 10 ) then begin
		CanJoin := False;
	end else if ( GB <> Nil ) and not ( OnTheMap( GB , FindRoot( NPC ) ) and IsFoundAlongTrack( GB^.Meks , FindRoot( NPC ) ) ) then begin
		{ Can only join if in the same scene as the PC. }
		CanJoin := False;
	end else if PersonaUsedByQuest( FindRoot( GB^.Scene ) , NPC ) then begin
		CanJoin := False;
	end;
	CanJoinLance := CanJoin;
end;


Function SceneName( GB: GameBoardPtr; ID: Integer; Exact: Boolean ): String;
	{ Find the name of the scene with the given ID. If no such }
	{ scene can be found, return a value that should let the player }
	{ know an error has been commited. }
	{ If EXACT=TRUE, use the EXACT_NAME attribute instead of the }
	{ regular name. The reason for this is that sometimes there's }
	{ some ambiguity with the common name of a scene: if I say "Cayley }
	{ Rock", do I mean the city in general or the main station }
	{ specifically? }
var
	msg: String;
	Part: GearPtr;
begin
	if ( GB = Nil ) or ( GB^.Scene = Nil ) then begin
		SceneName := 'XXX';
	end else begin
		{ Look for the scene along the subcomponents of the }
		{ adventure. This is to make sure we don't accidentally }
		{ pick a dynamic scene with the right ID. }
		{ Also, if we have a metascene instead of a regular scene, then }
		{ we want the name of the entrance instead of the name of the }
		{ scene itself. }
		if ID > 0 then begin
			Part := FindActualScene( GB , ID );
		end else begin
			Part := FindSceneEntrance( FindRoot( GB^.Scene ) , GB , ID );
		end;
		if Exact then begin
			msg := SAttArrayValue( Part^.SA , 'EXACT_NAME_I18N' );
			if msg = '' then msg := SAttArrayValue( Part^.SA , 'EXACT_NAME' );
			if msg = '' then msg := GearName( Part );
			SceneName := msg;
		end else begin
			SceneName := GearName( Part );
		end;
	end;
end;

Function FindRandomMekID( GB: GameBoardPtr; Team: Integer ): LongInt;
	{ Locate a random mek belonging to TEAM. }
var
	NumMeks,N,T,MID: Integer;
	Mek: GearPtr;
begin
	{ Start out by finding out how many meks belong to this team }
	{ anyways. }
	NumMeks := NumOperationalMasters( GB , Team );
	MID := 0;

	{ If there are actually members on this team, select one randomly. }
	if NumMeks > 0 then begin
		{ Decide what mek to take, and initialize the }
		{ search variables. }
		N := Random( NumMeks ) + 1;
		T := 0;
		Mek := GB^.Meks;

		while Mek <> Nil do begin
			if ( NAttValue( Mek^.NA , NAG_Location , NAS_Team ) = Team ) and GearOperational( Mek ) then begin
				Inc( T );
				if T = N then MID := NAttValue( Mek^.NA , NAG_EpisodeData , NAS_UID );
			end;
			Mek := Mek^.Next;
		end;
	end;

	FindRandomMekID := MID;
end;

Function FindRandomPilotID( GB: GameBoardPtr; Team: Integer ): LongInt;
	{ Locate a random pilot belonging to TEAM. }
var
	NumMeks,N,T,MID: Integer;
	Mek,P: GearPtr;
begin
	{ Start out by finding out how many meks belong to this team }
	{ anyways. }
	NumMeks := NumOperationalMasters( GB , Team );
	MID := 0;

	{ If there are actually members on this team, select one randomly. }
	if NumMeks > 0 then begin
		{ Decide what mek to take, and initialize the }
		{ search variables. }
		N := Random( NumMeks ) + 1;
		T := 0;
		Mek := GB^.Meks;

		while Mek <> Nil do begin
			if ( NAttValue( Mek^.NA , NAG_Location , NAS_Team ) = Team ) and GearOperational( Mek ) then begin
				Inc( T );
				if T = N then begin
					P := LocatePilot( Mek );
					if P <> Nil then MID := NAttValue( P^.NA , NAG_EpisodeData , NAS_UID );
				end;
			end;
			Mek := Mek^.Next;
		end;
	end;

	FindRandomPilotID := MID;
end;

Function FindRootUID( GB: GameBoardPtr; ID: LongInt ): LongInt;
	{ Find the ID of the root of the specified gear. }
var
	Part: GearPtr;
begin
	{ First, find the part being pointed to. }
	Part := LocateMekByUID( GB , ID );

	{ Locate its root. }
	if Part <> Nil then begin
		Part := FindRoot( Part );

		{ Return the root's UID. }
		FindRootUID := NAttValue( Part^.NA , NAG_EpisodeData , NAS_UID );

	{ If there was an error locating the part, return 0. }
	end else FindRootUID := 0;
end;


Function NumPCMeks( GB: GameBoardPtr ): Integer;
	{ Return the number of mecha belonging to team 1. }
	{ It doesn't matter if they're on the board or not, nor whether or }
	{ not they are destroyed. }
var
	M: GearPtr;
	N: Integer;
begin
	N := 0;
	if GB <> Nil then begin
		M := GB^.Meks;
		while M <> Nil do begin
			{ If this is a mecha, and it belongs to team 1, }
			{ increment the counter. }
			if ( M^.G = GG_Mecha ) and ( NAttValue( M^.NA , NAG_Location , NAS_Team ) = NAV_DefPlayerTeam ) then Inc( N );
			M := M^.Next;
		end;
	end;
	NumPCMeks := N;
end;

Function FindPCScale( GB: GameBoardPtr ): Integer;
	{ Return the scale of the PC. Generally this will be 0 if the }
	{ PC is on foot, 1 or 2 if the PC is in a mecha, unless the PC }
	{ is a storm giant or a zentradi in which case anything goes. }
var
	PC: GearPtr;
begin
	PC := GG_LocatePC( GB );
	if PC <> Nil then begin
		FindPCScale := FindRoot( PC )^.Scale;
	end else begin
		FindPCScale := 0;
	end;
end;

Function Calculate_Reward_Value( GB: GameBoardPtr; Renown,Percent: LongInt ): LongInt;
	{ Return an appropriate reward value, based on the listed }
	{ threat level and percent scale. }
const
	Min_Reward_Value = 3000;
var
	RV: LongInt;
begin
	{ Calculate the base reward value. }
	RV := Calculate_Threat_Points( Renown + 10 , 100 ) div 64 * Percent div 100;
	if RV < Min_Reward_Value then RV := Min_Reward_Value;

	{ Modify this for the PC's talents. }
	if GB <> Nil then begin
		if TeamHasTalent( GB , NAV_DefPlayerTeam , NAS_BusinessSense ) then RV := ( RV * 5 ) div 4;
	end;

	Calculate_Reward_Value := RV;
end;

Function Calculate_Asking_Price( GB: GameBoardPtr; Renown,Percent: LongInt ): LongInt;
	{ Return an appropriate asking price, based on the listed }
	{ threat level, percent scale, and reaction score with I_NPC. }
const
	Min_Asking_Price = 3000;
var
	RV,React: LongInt;
begin
	{ Calculate the base reward value. }
	RV := Calculate_Threat_Points( Renown , 25 ) * Percent div 100;
	if RV < Min_Asking_Price then RV := Min_Asking_Price;

	{ Modify this for the reaction score. }
	if ( I_NPC <> Nil ) and ( I_PC <> Nil ) and ( GB <> Nil ) then begin
		React := ReactionScore( GB^.Scene , I_PC , I_NPC );
		if React < 0 then RV := RV * ( 100 - 2 * React ) div 100
		else if React > 20 then RV := RV * ( 220 - React ) div 200;
	end;

	Calculate_Asking_Price := RV;
end;

Function Count_Faction_Buddies( GB: GameBoardPtr; FacID: Integer ): Integer;
	{ Return the number of positive relationships the PC has with this }
	{ particular faction. }
	function CountBuddies( LList: GearPtr ): LongInt;
		{ Count the number of buddies along this path. }
	var
		N: LongInt;
	begin
		N := 0;
		while LList <> Nil do begin
			if ( LList^.G = GG_Character ) and ( NAttValue( LList^.NA , NAG_Relationship , 0 ) > 0 ) and ( GetFactionID( LList ) = FacID ) and NotDestroyed( LList ) then Inc( N );
			N := N + CountBuddies( LList^.SubCom );
			N := N + CountBuddies( LList^.InvCom );
			LList := LList^.Next;
		end;
		CountBuddies := N;
	end;
var
	Adv: GearPtr;
begin
	if ( GB = Nil ) or ( GB^.Scene = Nil ) then Exit( 0 );
	Adv := FindRoot( GB^.Scene );
	Count_Faction_Buddies := CountBuddies( GB^.Meks ) + CountBuddies( Adv^.SubCom ) + CountBuddies( Adv^.InvCom );
end;

Function FindLocalMacro( const cmd: String; GB: GameBoardPtr; Source: GearPtr ): String;
	{ Locate the local macro described by "cmd". }
var
	it: String;
	Plot: GearPtr;
begin
	it := SAttArrayValue( Source^.SA , cmd );
	if it = '' then begin
		Plot := PlotMaster( GB , Source );
		if Plot <> Nil then it := SAttArrayValue( Plot^.SA , cmd );
		if it = '' then begin
			Plot := StoryMaster( GB , Source );
			if Plot <> Nil then it := SAttArrayValue( Plot^.SA , cmd );
		end;
	end;
	if it = '' then begin
{$IFDEF DEBUG}
		ErrorMessage_fork( 'SCRIPT ERROR: Local macro ' + cmd + ' not found.' );
{$ENDIF DEBUG}
		DialogMsg( 'SCRIPT ERROR: Local macro ' + cmd + ' not found.' );
	end;
	FindLocalMacro := it;
end;

Procedure InitiateMacro( GB: GameBoardPtr; Source: GearPtr; var Event: String; ProtoMacro: String );
	{ Initialize the provided macro, and stick it to the head of }
	{ event. To initialize just go through it word by word, replacing }
	{ all question marks with words taken from EVENT. }
	function NeededParameters( cmd: String ): Integer;
		{ Return the number of parameters needed by this function. }
	const
		NumStandardFunctions = 20;
		StdFunName: Array [1..NumStandardFunctions] of string = (
			'-', 'GNATT', 'GSTAT', '?PILOT', '?MECHA',
			'SKROLL', 'THREAT', 'REWARD', 'ESCENE', 'RANGE',
			'FXPNEEDED','*','MAPTILE','PCSKILLVAL','CONCERT',
			'SKILLTAR', 'HARDSKILLTAR', 'SOCSKILLTAR', 'PRICE', 'FACBUDDIES'
		);
		StdFunParam: Array [1..NumStandardFunctions] of byte = (
			1,2,1,1,1,
			1,2,2,1,2,
			1,2,2,2,2,
			1,1,1,2,1
		);
	var
		it,T: Integer;
		Mac: String;
	begin
		it := 0;
		CMD := UpCase( CMD );
		{ If a hardwired function, look up the number of parameters }
		{ from the above list. }
		for t := 1 to NumStandardFunctions do begin 
			if CMD = StdFunName[ t ] then it := StdFunParam[ t ];
		end;

		{ If a macro function, derive the number of parameters }
		{ from the number of ?s in the macro. }
		if it = 0 then begin
			Mac := SAttArrayValue( Value_Macros , Cmd );
			if ( Mac = '' ) and ( Cmd <> '' ) and ( Cmd[1] = '&' ) then Mac := FindLocalMacro( cmd , GB , Source );
			if Mac <> '' then begin
				for t := 1 to Length( Mac ) do begin
					if Mac[ T ] = '?' then Inc( it );
				end;
			end;
		end;
		NeededParameters := it;
	end;
	function  GetSwapText: String;
		{ Grab the swap text from EVENT. If it's a function }
		{ that we just grabbed, better grab the parameters as }
		{ well. Oh, bother... }
	var
		it,cmd: String;
		N: Integer;
	begin
		{ Grab the beginning of the swap text. }
		it := ExtractWord( Event );

		{ Check to see if it's a function. Get rid of the - first }
		{ since it'll screw up our check. }
		cmd := it;
		if ( Length( cmd ) > 1 ) and ( cmd[1] = '-' ) then DeleteFirstChar( cmd );
		N := NeededParameters( cmd );
		While N > 0 do begin
			it := it + ' ' + GetSwapText();
			Dec( N );
		end;
		GetSwapText := it;
	end;
var
	Cmd,NewMacro,LastWord: String;
	DItS: Boolean;		{Do insert the space, or not.}
	CW_UTF8: Boolean;	{Is the current word UTF8 ?}
begin
	NewMacro := '';
	LastWord := '';

	while ProtoMacro <> '' do begin
		cmd := ExtractWordForParse( ProtoMacro, DItS, CW_UTF8 );
		if cmd = '?' then begin
			LastWord := GetSwapText;
			cmd := LastWord;
		end else if cmd = '!' then begin
			cmd := LastWord;
		end;
		if DItS then begin
			NewMacro := NewMacro + ' ' + cmd;
		end else begin
			NewMacro := NewMacro + cmd;
		end;
	end;

	Event := NewMacro + Event;
end;

Procedure InitiateLocalMacro( GB: GameBoardPtr; var Event , cmd: String; Source: GearPtr );
	{ Attempt to initiate a local macro. The local macro will be located }
	{ in Source, or its plot, or its story. }
var
	it: String;
begin
	it := FindLocalMacro( cmd , GB , Source );
	if it <> '' then InitiateMacro( GB , Source , Event , it );
end;


Function SV_Range( GB: GameBoardPtr; UID1,UID2: LongInt ): Integer;
	{ Locate the two gears pointed to by the UIDs, then calculate }
	{ the range between them. If one or both are NIL, or if one or }
	{ both are off the map, return 0. }
var
	M1,M2: GearPtr;
begin
	{ Error check. }
	if GB = Nil then Exit( 0 );

	M1 := LocateMekByUID( GB , UID1 );
	M2 := LocateMekByUID( GB , UID2 );

	if ( M1 <> Nil ) and OnTheMap( GB , M1 ) and ( M2 <> Nil ) and OnTheMap( GB , M2 ) then begin
		SV_Range := Range( GB , M1 , M2 );
	end else begin
		SV_Range := 0;
	end;
end;

Function SV_PCSkillVal( GB: GameBoardPtr; Skill,Stat: Integer ): Integer;
	{ Return the PC's base skill value. This used to be easy until those }
	{ stupid lancemates came along... Check all PCs and lancemates, and }
	{ return the highest value. }
var
	M: GearPtr;
	HiSkill,T: Integer;
begin
	{ Error check. }
	if GB = Nil then Exit( 0 );

	M := GB^.Meks;
	HiSkill := 0;
	while M <> Nil do begin
		T := NAttValue( M^.NA , NAG_Location , NAS_Team );
		if GearActive( M ) and ( ( T = NAV_DefPlayerTeam ) or ( T = NAV_LancemateTeam ) ) then begin
			T := SkillValue( M , Skill , Stat );
			if T > HiSkill then HiSkill := T;
		end;

		M := M^.Next;
	end;

	SV_PCSkillVal := HiSkill;
end;

Function SV_WorldID( GB: GameBoardPtr ): Integer;
	{ Return the ID number of the current world. }
var
	Scene: GearPtr;
begin
	Scene := FindWorld( GB , GB^.Scene );
	if Scene <> Nil then SV_WorldID := Scene^.S
	else SV_WorldID := 0;
end;



Function AV_ProcessConcert( GB: GameBoardPtr; AudienceSize,SkillTarget: Integer ): Integer;
	{ A concert is beginning! Yay! }
var
	PC: GearPtr;
begin
	PC := GG_LocatePC( GB );
	AV_ProcessConcert := DoConcert( GB , PC , AudienceSize , SkillTarget );
end;


Function ScriptValue( var Event: String; GB: GameBoardPtr; Scene: GearPtr ): LongInt;
	{ Normally, numerical values will be stored as constants. }
	{ Sometimes we may want to do algebra, or use the result of }
	{ scenario variables as the parameters for commands. That's }
	{ what this function is for. }
	{ *** IMPORTANT *** }
	{ When adding new functions, remember to add them to the InitiateMacro }
	{ procedure as well. }
	{ *** IMPORTANT *** }
var
	Old_Grabbed_Gear,PC: GearPtr;
	VCode,VC2: LongInt;
	SV: LongInt;
	SMsg,S2: String;
{$IFDEF DEBUG}
	TRACE_SMsg, TRACE_event: String;
{$ENDIF DEBUG}
begin
	{ Save the grabbed gear, to restore it later. }
	Old_Grabbed_Gear := Grabbed_Gear;

{$IFDEF DEBUG}
	TRACE_event := Event;
{$ENDIF DEBUG}
	SMsg := UpCase(ExtractWord( Event ));
{$IFDEF DEBUG}
	TRACE_SMsg := SMsg;
{$ENDIF DEBUG}
	SV := 0;

	{ Start by checking for value macros. }
	if SMsg = '' then begin
		{ An empty string is a bad thing. }
		SV := 0;
		DialogMsg( 'SCRIPT WARNING: Blank scriptvalue from ' + GearName( Scene ) );

	end else if SAttArrayValue( Value_Macros , SMsg ) <> '' then begin
		{ Install the macro, then re-call this procedure to get }
		{ the results. }
		InitiateMacro( GB , scene , Event , SAttArrayValue( Value_Macros , SMsg ) );
		SV := ScriptValue( Event , gb , scene );

	{ If the command starts with a &, this means it's a local macro. }
	end else if Smsg[1] = '&' then begin
		InitiateLocalMacro( GB , Event , SMsg , Scene );
		SV := ScriptValue( Event , gb , scene );

	end else if ( SMsg = 'GNATT' ) then begin
		{ Get a Numeric Attribute from the currently grabbed gear. }
		VCode := ScriptValue( Event , GB , Scene );
		VC2 := ScriptValue( Event , GB , Scene );
		if Grabbed_Gear <> Nil then begin
			SV := NAttValue( Grabbed_Gear^.NA , VCode , VC2 );
		end else begin
{$IFDEF DEBUG}
			ErrorMessage_fork( 'SCRIPT WARNING: Illigal GNATT. ' + Make_ErrorMessage_ASL(Scene,Event) );
			ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT1(Event) );
			ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT2(Event) );
			ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT3(Event) );
{$ENDIF DEBUG}
			DialogMsg( 'SCRIPT WARNING: Illigal GNATT. ' + Make_ErrorMessage_ASL(Scene,Event) );
		end;

	end else if ( SMsg = 'GSTAT' ) then begin
		{ Get a Numeric Attribute from the currently grabbed gear. }
		VCode := ScriptValue( Event , GB , Scene );
		if ( Grabbed_Gear <> Nil ) then begin
			if ( VCode >= 1 ) and ( VCode <= NumGearStats ) then begin
				SV := Grabbed_Gear^.Stat[ VCode ];
			end;
		end else begin
{$IFDEF DEBUG}
			ErrorMessage_fork( 'SCRIPT WARNING: Illigal GSTAT. ' + Make_ErrorMessage_ASL(Scene,Event) );
			ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT1(Event) );
			ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT2(Event) );
			ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT3(Event) );
{$ENDIF DEBUG}
			DialogMsg( 'SCRIPT WARNING: Illigal GSTAT. ' + Make_ErrorMessage_ASL(Scene,Event) );
		end;

	end else if ( SMsg = 'GV' ) then begin
		{ Get the V descriptor from the currently grabbed gear. }
		if ( Grabbed_Gear <> Nil ) then begin
			SV := Grabbed_Gear^.V;
		end else begin
{$IFDEF DEBUG}
			ErrorMessage_fork( 'SCRIPT WARNING: Illigal GV. ' + Make_ErrorMessage_ASL(Scene,Event) );
			ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT1(Event) );
			ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT2(Event) );
			ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT3(Event) );
{$ENDIF DEBUG}
			DialogMsg( 'SCRIPT WARNING: Illigal GV. ' + Make_ErrorMessage_ASL(Scene,Event) );
		end;

	end else if ( SMsg = 'GS' ) then begin
		{ Get the S descriptor from the currently grabbed gear. }
		if ( Grabbed_Gear <> Nil ) then begin
			SV := Grabbed_Gear^.S;
		end else begin
{$IFDEF DEBUG}
			ErrorMessage_fork( 'SCRIPT NOTICE: Illigal GS. ' + Make_ErrorMessage_ASL(Scene,Event) );
			ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT1(Event) );
			ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT2(Event) );
			ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT3(Event) );
			DialogMsg( 'SCRIPT NOTICE: Illigal GS. ' + Make_ErrorMessage_ASL(Scene,Event) );
{$ENDIF DEBUG}
			SV := 0;
		end;

	end else if ( SMsg = 'GSCENE' ) then begin
		{ Get the sceneID of the grabbed gear. }
		if ( Grabbed_Gear <> Nil ) then begin
			SV := FindSceneID( Grabbed_Gear , GB );
		end else begin
{$IFDEF DEBUG}
			ErrorMessage_fork( 'SCRIPT WARNING: Illigal GSCENE. ' + Make_ErrorMessage_ASL(Scene,Event) );
			ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT1(Event) );
			ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT2(Event) );
			ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT3(Event) );
{$ENDIF DEBUG}
			DialogMsg( 'SCRIPT WARNING: Illigal GSCENE. ' + Make_ErrorMessage_ASL(Scene,Event) );
		end;

	end else if ( SMsg = 'WORLDID' ) then begin
		{ Return the ID of the current world. }
		SV := SV_WorldID( GB );

	end else if ( SMsg = 'FXPNEEDED' ) then begin
		{ Return how many faction XP points needed for next level. }
		VCode := ScriptValue( Event , GB , Scene );
		SV := ( VCode + 1 ) * 5;

	end else if ( SMsg = 'CONCERT' ) then begin
		VCode := ScriptValue( Event , GB , Scene );
		VC2 := ScriptValue( Event , GB , Scene );
		SV := AV_ProcessConcert( GB , VCode , VC2 );

	end else if ( SMsg = 'MAPTILE' ) then begin
		{ Return the terrain of the requested map tile. }
		VCode := ScriptValue( Event , GB , Scene );
		VC2 := ScriptValue( Event , GB , Scene );
		if ( GB <> Nil ) and OnTheMap( GB , VCode , VC2 ) then begin
			SV := TileTerrain( GB , VCode , VC2 );
		end else begin
			SV := 0;
		end;

	end else if Attempt_Gear_Grab( SMsg , Event , GB , Scene ) then begin
		if ( NIL = Grabbed_Gear ) then begin
{$IFDEF DEBUG}
			ErrorMessage_fork( 'SCRIPT NOTICE: Gear-Grabbing command failed in ScriptValue.' + Make_ErrorMessage_ASL(Scene,Event) );
			ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT1(Event) );
			ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT2(Event) );
			ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT3(Event) );
			DialogMsg( 'SCRIPT NOTICE: Gear-Grabbing command failed in ScriptValue. ' + Make_ErrorMessage_ASL(Scene,Event) );
{$ENDIF DEBUG}
		end;
		{ The correct Grabbed_Gear was set by the above call, }
		{ so just recurse to find the value we want. Am I looming }
		{ closer to functional programming or something? }
		SV := ScriptValue( Event , gb , scene );

	end else if ( SMsg = 'COMTIME' ) then begin
		{ Return the current combat time. }
		if ( GB <> Nil ) then begin
			SV := GB^.ComTime;
		end;

	end else if ( SMsg = 'NEXTDAY' ) then begin
		{ Return the start of the next day. }
		if ( GB <> Nil ) then begin
			SV := GB^.ComTime + 86400 - GB^.ComTime mod 86400;
		end;

	end else if ( SMsg = 'PCSKILLVAL' ) then begin
		{ Return the PC team's highest skill value. }
		VCode := ScriptValue( Event , GB , Scene );
		VC2 := ScriptValue( Event , GB , Scene );
		SV := SV_PCSkillVal( GB , VCode , VC2 );

	end else if ( SMsg = 'SCENEID' ) then begin
		{ Return the current scene's unique ID. }
		{ Only do this if we're in the real scene! }
		SV := RealSceneID( GB^.Scene );

	end else if ( SMsg = 'REACT' ) then begin
		{ Return the current reaction score between PC & NPC. }
		if ( IntMenu <> Nil ) then begin
			if GB = Nil then begin
				SV := ReactionScore( Nil , I_PC , I_NPC );
			end else begin
				SV := ReactionScore( GB^.Scene , I_PC , I_NPC );
			end;
		end;

	end else if ( SMsg = 'SKROLL' ) then begin
		{ Return a skill roll from the PC. }
		VCode := ScriptValue( Event , GB , Scene );
		VC2 := ScriptValue( Event , GB , Scene );
		if ( VCode >= 1 ) and ( VCode <= NumSkill ) then begin
			SV := SkillRoll( GB , GG_LocatePC( GB ) , VCode , VC2 , 0 , 0 , True , True );
		end else begin
{$IFDEF DEBUG}
			ErrorMessage_fork( 'SCRIPT ERROR: Illegal Skill Roll ' + BStr( VCode ) + ': ' + Event );
{$ENDIF DEBUG}
			DialogMsg( 'SCRIPT ERROR: Illegal Skill Roll ' + BStr( VCode ) + ': ' + Event );
			SV := 0;
		end;
		PC := GG_LocatePC( GB );
		if PC <> Nil then DoleSkillExperience( PC , VCode , 5 );

	end else if SMsg = 'PCMEKS' then begin
		SV := NumPCMeks( GB );

	end else if ( SMsg = 'THREAT' ) then begin
		VCode := ScriptValue( Event , GB , Scene );
		VC2 := ScriptValue( Event , GB , Scene );
		SV := Calculate_Threat_Points( VCode , VC2 );

	end else if ( SMsg = 'FACBUDDIES' ) then begin
		{ Count the number of positive relationships the PC has }
		{ with members of the provided faction. }
		VCode := ScriptValue( Event , GB , Scene );
		SV := Count_Faction_Buddies( GB , VCode );

	end else if ( SMsg = '*' ) then begin
		VCode := ScriptValue( Event , GB , Scene );
		VC2 := ScriptValue( Event , GB , Scene );
		SV := VCode * VC2;

	end else if ( SMsg = 'REWARD' ) then begin
		VCode := ScriptValue( Event , GB , Scene );
		VC2 := ScriptValue( Event , GB , Scene );
		SV := Calculate_Reward_Value( GB , VCode , VC2 );

	end else if ( SMsg = 'PRICE' ) then begin
		VCode := ScriptValue( Event , GB , Scene );
		VC2 := ScriptValue( Event , GB , Scene );
		SV := Calculate_Asking_Price( GB , VCode , VC2 );

	end else if ( SMsg = 'RANGE' ) then begin
		VCode := ScriptValue( Event , GB , Scene );
		VC2 := ScriptValue( Event , GB , Scene );
		SV := SV_Range( GB , VCode , VC2 );

	end else if ( SMsg = 'SKILLTAR' ) then begin
		VCode := ScriptValue( Event , GB , Scene );
		SV := BasicSkillTarget( VCode );

	end else if ( SMsg = 'HARDSKILLTAR' ) then begin
		VCode := ScriptValue( Event , GB , Scene );
		SV := HardSkillTarget( VCode );

	end else if ( SMsg = 'SOCSKILLTAR' ) then begin
		VCode := ScriptValue( Event , GB , Scene );
		SV := SocSkillTarget( GB , VCode );

	end else if SMsg = 'PCSCALE' then begin
		SV := FindPCScale( GB );

	end else if ( SMsg = 'ESCENE' ) then begin
		{ Find out what element to find the scene for. }
		if ( Scene <> Nil ) then begin
			VCode := ExtractValue( Event );

			{ Find out what plot we're dealing with. }
			Scene := PlotMaster( GB , Scene );

			if ( Scene <> Nil ) and ( VCode >= 1 ) and ( VCode <= Num_Plot_Elements ) then begin
				SV := ElementLocation( FindRoot( Scene ) , Scene , VCode , GB );
			end;
		end;

	end else if ( SMsg[1] = 'D' ) then begin
		{ Roll a die, return the result. }
		DeleteFirstChar( SMsg );
		Event := SMsg + ' ' + Event;
		VCode := ScriptValue( Event , GB , Scene );;
		if VCode < 1 then VCode := 1;
		SV := Random( VCode ) + 1;

	{ As a last resort, see if the first character shows up in the }
	{ scripts file. If so, use that. }
	end else if SAttArrayValue( Value_Macros , SMsg[1] ) <> '' then begin
		{ Install the macro, then re-call this procedure to get }
		{ the results. }
		S2 := SMsg[ 1 ];
		DeleteFirstChar( SMsg );
		Event := SMsg + ' ' + Event;
		InitiateMacro( GB , scene , Event , SAttArrayValue( Value_Macros , S2 ) );
		SV := ScriptValue( Event , gb , scene );

	end else if ( SMsg = '?PILOT' ) then begin
		VCode := ScriptValue( Event , GB , Scene );
		SV := FindRandomPilotID( GB , VCode );

	end else if ( SMsg = '?MECHA' ) then begin
		VCode := ScriptValue( Event , GB , Scene );
		SV := FindRandomMekID( GB , VCode );

	end else if ( SMsg[1] = 'T' ) and ( gb <> Nil ) then begin
		{ Return the number of gears on the provided team. }
		DeleteFirstChar( SMsg );
		VCode := ExtractValue( SMsg );
		SV := NumActiveMasters( GB , VCode );

	end else if ( SMsg[1] = '@' ) and ( gb <> Nil ) then begin
		{ Return the root gear of the gear indicated by the }
		{ rest of this expression. Return 0 if there's an error. }
		DeleteFirstChar( SMsg );
		VCode := ScriptValue( SMsg , gb , scene );
		if VCode <> 0 then SV := FindRootUID( GB , VCode )
		else SV := 0;


	end else if SMsg[1] = '-' then begin
		{ We want the negative of the value to follow. }
		DeleteFirstChar( SMsg );
		event := SMsg + ' ' + event;
		SV := -ScriptValue( event , gb , scene );

	end else begin
		{ No command was given, so this must be a constant value. }
		S2 := SMsg;
		SV := ExtractValue( SMsg );
		if ( SV = 0 ) and ( S2 <> '' ) and ( S2 <> '0' ) then begin
{$IFDEF DEBUG}
			ErrorMessage_fork( 'SCRIPT WARNING: value ' + S2 + ' in "' + GearName( scene ) + '" . ' + Make_ErrorMessage_ASL(Scene,Event) );
			ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT1(Event) );
			ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT2(Event) );
			ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT3(Event) );
{$ENDIF DEBUG}
			DialogMsg( 'SCRIPT WARNING: value ' + S2 + ' in ' + GearName( scene ) );
			DialogMsg( 'CONTEXT: ' + event );
		end;
	end;
{$IFDEF DEBUG}
	ErrorMessage_fork('SCRIPT TRACE: SCRIPTVALUE ' + TRACE_SMsg + ' value ' + BStr( SV ) + ' in "' + GearName( scene ) + '", CONTEXT: <' + TRACE_event + '>' );
{$ENDIF DEBUG}

	{ Restore the grabbed gear before exiting. }
	Grabbed_Gear := Old_Grabbed_Gear;

	ScriptValue := SV;
end;

Function AS_GetString( Source: GearPtr; Key: String ): String;
	{ Check the SOURCE for a SAttList with the provided KEY. }
	{ If none can be found, search the default list for SOURCE's type. }
	{ Normally, getting a string attribute could be handled simply by the }
	{ SAttArrayValue function. But, I had some trouble with my doors getting so }
	{ #$#@%! huge, so I decided to write the function as a space-saver. }
var
	msg: String;
begin
	if Source <> Nil then begin
		msg := SAttArrayValue( Source^.SA , Key );
		if ( msg = '' ) and ( Source^.G = GG_MetaTerrain ) and ( Source^.S >= 1 ) and ( Source^.S <= NumBasicMetaTerrain ) then begin
			msg := SAttArrayValue( Meta_Terrain_Scripts[ Source^.S ] , Key );
		end else if ( msg = '' ) and ( ( Source^.G = GG_Scene ) or ( Source^.G = GG_MetaScene ) or ( Source^.G = GG_World ) ) then begin
			msg := SAttArrayValue( Default_Scene_Scripts , Key );
		end;
		if ( msg = '' ) and HeadMatchesString( 'CLUE_' , Key ) then begin
			msg := 'VMsg AS_YouHaveNoClue 0';
		end;
	end else begin
		msg := '';
	end;
	AS_GetString := msg;
end;

Function EncounterMapType( GB: GameBoardPtr; X,Y: Integer ): Integer;
	{ The PC has met an encounter at point X,Y on the current map. Return the }
	{ map generator which this encounter should use. }
begin
	if ( GB <> Nil ) and ( GB^.Scene <> Nil ) and AStringHasBString( SAttArrayValue( GB^.Scene^.SA , 'SPECIAL' ) , 'UNIFORM' ) then begin
		EncounterMapType := GB^.Scene^.Stat[ STAT_MapGenerator ];
	end else if GB <> Nil then begin
		EncounterMapType := TileTerrain( GB , X , Y )
	end else begin
		EncounterMapType := 1;
	end;
end;

Procedure AS_SetExit( GB: GameBoardPtr; RC: Integer );
	{ Several things need to be done when exiting the map. }
	{ This procedure should centralize most of them. }
var
	Dest,Src,SrcHome: GearPtr;
	T: Integer;
begin
	{ Only process this request if we haven't already set an exit. }
	if ( GB <> Nil ) and ( not GB^.QuitTheGame ) then begin
		GB^.QuitTheGame := True;
		GB^.ReturnCode := RC;
		if GB^.Scene <> Nil then begin
			if IsInvCom( GB^.SCene ) then begin
				SCRIPT_Gate_To_Seek := GB^.Scene^.S;
			end else begin
				SCRIPT_Gate_To_Seek := RealSceneID( GB^.Scene );
			end;

			Dest := FindActualScene( GB , RC );

			{ Set the return value for metascenes. }
			if ( Dest = Nil ) and ( SCRIPT_DynamicEncounter = Nil ) then begin
				{ If we've been asked to exit to a nonexistant scene, }
				{ try to re-enter the current scene again. }
				GB^.ReturnCode := FindGearScene( GB^.Scene , GB );

			end else if ( RC < 0 ) and ( Dest <> Nil ) then begin
				{ This is a metascene. Set the entrance value and the map generator }
				{ if appropriate. }
				if ( NAttValue( Dest^.NA , NAG_Narrative , NAS_EntranceScene ) = 0 ) and ( GB^.Scene^.G = GG_Scene ) then begin
					SetNAtt( Dest^.NA , NAG_Narrative , NAS_EntranceScene , GB^.Scene^.S );
				end;

				{ If the metascene has no map generator set, better set the map }
				{ generator from the tile the entrance is sitting upon. }
				if Dest^.Stat[ STAT_MapGenerator ] = 0 then begin
					Src := FindSceneEntrance( FindRoot( GB^.Scene ) , GB , RC );
					if ( Src <> Nil ) and OnTheMap( GB , Src ) and IsFoundAlongTrack( GB^.Meks , Src ) then begin
						Dest^.Stat[ STAT_MapGenerator ] := EncounterMapType( GB , NAttValue( Src^.NA , NAG_Location , NAS_X ) , NAttValue( Src^.NA , NAG_Location , NAS_Y ) );
						{ If this will make the encounter a space map, set the map-scroll tag. }
						if Dest^.Stat[ STAT_MapGenerator ] = TERRAIN_Space then Dest^.Stat[ STAT_SpaceMap ] := 1;
						SrcHome := GB^.Scene;
					end else begin
						{ Copy the entrance's map generator and hope for the best. }
						SrcHome := FindActualScene( GB , FindGearScene( Src , GB ) );
						if SrcHome = Nil then SrcHome := GB^.Scene;
						Dest^.Stat[ STAT_MapGenerator ] := SrcHome^.Stat[ STAT_MapGenerator ];
						{ If this will make the encounter a space map, set the map-scroll tag. }
						if Dest^.Stat[ STAT_MapGenerator ] = TERRAIN_Space then Dest^.Stat[ STAT_SpaceMap ] := 1;
					end;

					{ Also copy over the tileset + backdrop. }
					SetNAtt( Dest^.NA , NAG_SceneData , NAS_TileSet , NAttValue( SrcHome^.NA , NAG_SceneData , NAS_TileSet ) );
					SetNAtt( Dest^.NA , NAG_SceneData , NAS_Backdrop , NAttValue( SrcHome^.NA , NAG_SceneData , NAS_Backdrop ) );

					{ Copy the environmental effects from the parent scene. }
					for t := 1 to Num_Environment_Variables do begin
						SetNAtt( Dest^.NA , NAG_EnvironmentData , T , NAttValue( SrcHome^.NA , NAG_EnvironmentData , T ) );
					end;
				end;

			end else if ( Dest <> Nil ) and ( Dest^.G = GG_World ) then begin
				{ If we're exiting to the world, the gate to seek }
				{ should be the root scene. }
				Src := FindRootScene( GB^.Scene );
				if Src <> Nil then SCRIPT_Gate_To_Seek := Src^.S;
			end;
		end;
	end;
end;

Procedure ProcessExit( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ An exit command has been received. }
var
	SID: Integer;
begin
	SID := ScriptValue( Event , GB , Source );
	if ( SID < 0 ) and ( FindActualScene( GB , SID ) = Nil ) then begin
		DialogMsg( MsgString( 'AS_EXIT_NOMETASCENE' ) );
	end else begin
		AS_SetExit( GB , SID );
	end;
end;

Procedure ProcessForceExit( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ An exit command has been received and will not be denied. }
var
	SID: Integer;
begin
	SID := ScriptValue( Event , GB , Source );
	GB^.QuitTheGame := False;
	AS_SetExit( GB , SID );
end;

Procedure ProcessReturn( GB: GameBoardPtr );
	{ An exit command has been received. }
	Procedure ReturnToScene( DefaultSID: Integer );
		{ Return from the current scene to some place appropriate. If a }
		{ "ReturnTo" value has been stored, go there. Otherwise, return to }
		{ the default scene given as a parameter. }
	var
		RtS: Integer;	{ Return To Scene }
	begin
		if ( GB <> Nil ) and ( GB^.Scene <> Nil ) then begin
			RtS := NAttValue( GB^.Scene^.NA , NAG_Narrative , NAS_ReturnToScene );
		end else RtS := 0;
		if RtS <> 0 then begin
			SetNAtt( GB^.Scene^.NA , NAG_Narrative , NAS_ReturnToScene , 0 );
			AS_SetExit( GB , RtS );
		end else begin
			AS_SetExit( GB , DefaultSID );
		end;
	end;
begin
	if ( GB <> Nil ) and ( GB^.Scene <> Nil ) and ( FindROot( GB^.Scene )^.G = GG_Adventure ) and ( FindROot( GB^.Scene )^.S = GS_ArenaCampaign ) then begin
		{ In an arena campaign, return always returns with an exit value of 1. }
		GB^.QuitTheGame := True;
		GB^.ReturnCode := 1;
	end else if ( GB <> Nil ) and ( GB^.Scene <> Nil ) and ( GB^.Scene^.G = GG_MetaScene ) then begin
		ReturnToScene( NAttValue( GB^.Scene^.NA , NAG_Narrative , NAS_EntranceScene ) );

	end else if ( GB <> Nil ) and ( GB^.Scene <> Nil ) and IsInvCom( GB^.Scene ) then begin
		if GB^.Scene^.S <> 0 then begin
			AS_SetExit( GB , GB^.Scene^.S );
			GB^.Scene^.S := 0;
			{ Eliminated old error check that doesn't appear to do anything anymore. In case of error, roll back to v0.420 or so. }
		end;
	end else if ( GB <> Nil ) and ( GB^.Scene <> Nil ) and ( GB^.Scene^.Parent <> Nil ) and ( GB^.Scene^.Parent^.G = GG_Scene ) then begin
		ReturnToScene( GB^.Scene^.Parent^.S );
	end else begin
		ReturnToScene( 0 );
	end;
end;

Function FactionRankName( GB: GameBoardPtr; Source: GearPtr; FID,FRank: Integer ): String;
	{ Return the generic name of rank FRank from faction FID. }
var
	F: GearPtr;
	it: String;
begin
	{ First find the faction. }
	F := GG_LocateFaction( FID , GB , Source );

	{ Do range checking on the FRank score obtained. }
	if FRank < 0 then FRank := 0
	else if FRank > 8 then FRank := 8;

	{ If no faction was found, return the default rank title. }
	if F = Nil then begin
		it := MsgString( 'FacRank_' + BStr( FRank ) );
	end else begin
		{ First check the faction for a name there. }
		{ If the faction has no name set, use the default. }
		it := UTF8_Name_NoFailback('FacRank',BStr(FID) + '_' + BStr(FRank));
		if '' = it then begin
			it := SAttArrayValue( F^.SA , 'FacRank_' + BStr( FRank ) );
		end;
		if it = '' then it := MsgString( 'FacRank_' + BStr( FRank ) );
	end;

	FactionRankName := it;
end;

Function PCRankName( GB: GameBoardPtr; Source: GearPtr ): String;
	{ Return the name of the PC's rank with the PC's faction. }
	{ if the PC has no rank, return the basic string. }
var
	FID: Integer;
	A,F: GearPtr;
begin
	{ The PC's faction ID is located in the adventure gear, so }
	{ locate that first. }
	A := GG_LocateAdventure( GB , Source );
	if A <> Nil then begin
		{ The faction rank score is located in the faction itself. }
		{ So now, let's locate that. }
		FID := NAttValue( A^.NA , NAG_Personal , NAS_FactionID );
		F := GG_LocateFaction( FID , GB , Source );
		if F <> Nil then begin
			{ Call the general Faction Rank function. }
			PCRankName := FactionRankName( GB , Source , FID , NAttValue( A^.NA , NAG_Experience , NAS_FacLevel ) );
		end else begin
			{ No faction found. Return the default value. }
			PCRankName := MsgString( 'FACRANK_0' );
		end;
	end else begin
		{ Adventure not found. Return default "peon". }
		PCRankName := MsgString( 'FACRANK_0' );
	end;
end;

Function FormatMemoString( GB: GameBoardPtr; const Msg: String ): String;
	{ Add the name of the city to the memo. }
var
	RootScene: GearPtr;
begin
	RootScene := FindRootScene( GB^.Scene );
	if ( RootScene <> Nil ) and ( msg <> '' ) then begin
		FormatMemoString := GearName( RootScene ) + ': ' + msg;
	end else begin
		FormatMemoString := msg;
	end;
end;

Function PlotHintString( GB: GameBoardPtr; Source: GearPtr; N: LongInt; AddMemo: Boolean ): String;
	{ Determine hint string N for this plot. }
	{ Also store a rumor memo. }
	Function GenericHintString: String;
		{ We can't find a valid hint string for this plot. Just return }
		{ some generic advice. }
	begin
		GenericHintString := MsgString( 'GENERIC_HINT_STRING_' + BStr( Random( 6 ) + 1 ) );
	end;
var
	Plot: GearPtr;
	IsFound: Boolean;
	msg,memo_msg: String;
begin
	{ Find the plot. We need it if we are to proceed. }
	Plot := PlotMaster( GB , Source );
	if Plot = Nil then Exit( GenericHintString );

	{ Start looking for the correct message. }
	repeat
		msg := ScriptMessage( 'HINT_' + BStr( N ) , GB , Plot );
		DeleteWhiteSpace( msg );
		IsFound := True;
		if ( Length( msg ) > 1 ) and ( msg[1] = ':' ) then begin
			{ This is a redirect. }
			DeleteFirstChar( msg );
			N := ExtractValue( msg );
			IsFound := False;
		end;
	until IsFound;
	if msg = '' then PlotHintString := GenericHintString
	else begin
		{ We found a hint. Save the plot memo. }
		if AddMemo and ( Plot <> Nil ) and ( I_NPC <> Nil ) then begin
			memo_msg := ReplaceHash( UTF8_MsgString( 'PlotHintString', 'PLOT_HINT_MEMO' ) , GearName( I_NPC ) , msg );
			memo_msg := FormatMemoString( GB , memo_msg );
			memo_msg := FormatChatStringByGender( memo_msg, I_NPC );
			SetSAttArray( Plot^.SA , 'MEMO' + BStr( N ) , memo_msg );
		end;
		PlotHintString := msg;
	end;
end;

Procedure FormatMessageString( var msg: String; gb: GameBoardPtr; Scene: GearPtr );
	{ There are a number of formatting commands which may be used }
	{ in an arenascript message string. }
var
	S0,S1,w: String;
	ID,ID2: LongInt;
	Part: GearPtr;
	DItS: Boolean;		{Do insert the space, or not.}
	CW_UTF8: Boolean;	{Is the current word UTF8 ?}
begin
	S0 := msg;
	S1 := '';

	while S0 <> '' do begin
		w := ExtractWordForParse( S0, DItS, CW_UTF8 );

		if ( W <> '' ) and ( W[1] = '\' ) then begin
			W := UpCase( W );
			if W = '\MEK' then begin
				{ Insert the name of a specified gear. }
				ID := ScriptValue( S0 , GB , Scene );
				Part := LocateMekByUID( GB , ID );
				if Part <> Nil then begin
					W := GearName( Part );
				end else begin
{$IFDEF DEBUG}
					ErrorMessage_fork( 'SCRIPT ERROR: FormatMessageString \MEK ' + BStr( ID ) );
{$ENDIF DEBUG}
					W := 'ERROR!!!';
				end;

			end else if W = '\PILOT' then begin
				{ Insert the name of a specified gear. }
				ID := ScriptValue( S0 , GB , Scene );
				Part := LocateMekByUID( GB , ID );
				if Part <> Nil then begin
					W := PilotName( Part );
				end else begin
{$IFDEF DEBUG}
					ErrorMessage_fork( 'SCRIPT ERROR: FormatMessageString \PILOT ' + BStr( ID ) );
{$ENDIF DEBUG}
					W := 'ERROR!!!';
				end;

			end else if W = '\ELEMENT' then begin
				{ Insert the name of a specified plot element. }
				ID := ScriptValue( S0 , GB , Scene );
				Part := PlotMaster( GB , Scene );
				if Part <> Nil then begin
					W := ElementName( FindRoot( GB^.Scene ) , Part , ID , GB );
				end else begin
{$IFDEF DEBUG}
					ErrorMessage_fork( 'SCRIPT ERROR: FormatMessageString \ELEMENT ' + BStr( ID ) );
{$ENDIF DEBUG}
					W := 'ERROR!!!';
				end;

			end else if W = '\NARRATIVE' then begin
				{ Insert the name of a specified plot element. }
				ID := ScriptValue( S0 , GB , Scene );
				Part := StoryMaster( GB , Scene );
				if Part <> Nil then begin
					W := ElementName( FindRoot( GB^.Scene ) , Part , ID , GB );
				end else begin
{$IFDEF DEBUG}
					ErrorMessage_fork( 'SCRIPT ERROR: FormatMessageString \NARRATIVE ' + BStr( ID ) );
{$ENDIF DEBUG}
					W := 'ERROR!!!';
				end;

			end else if W = '\PERSONA' then begin
				{ Insert the name of a specified persona. }
				ID := ScriptValue( S0 , GB , Scene );
				Part := GG_LocateNPC( ID , GB , Scene );
				if Part <> Nil then begin
					W := GearName( Part );
				end else begin
{$IFDEF DEBUG}
					ErrorMessage_fork( 'SCRIPT ERROR: FormatMessageString \PERSONA ' + BStr( ID ) );
{$ENDIF DEBUG}
					W := 'ERROR!!!';
				end;

			end else if W = '\HINT' then begin
				{ Insert the hint of a specified plot element. }
				ID := ScriptValue( S0 , GB , Scene );
				W := PlotHintString( GB , Scene , ID , False );

			end else if W = '\HINT_MEMO' then begin
				{ Insert the hint of a specified plot element and store a memo about it. }
				ID := ScriptValue( S0 , GB , Scene );
				W := PlotHintString( GB , Scene , ID , True );

			end else if W = '\ITEM' then begin
				{ Insert the name of a specified item. }
				ID := ScriptValue( S0 , GB , Scene );
				Part := GG_LocateItem( ID , GB , Scene );
				if Part <> Nil then begin
					W := GearName( Part );
				end else begin
{$IFDEF DEBUG}
					ErrorMessage_fork( 'SCRIPT ERROR: FormatMessageString \ITEM ' + BStr( ID ) );
{$ENDIF DEBUG}
					W := 'ERROR!!!';
				end;

			end else if W = '\ITEM_DESC' then begin
				{ Insert the description of a specified item. }
				ID := ScriptValue( S0 , GB , Scene );
				Part := GG_LocateItem( ID , GB , Scene );
				if Part <> Nil then begin
					W := SAttArrayValue( Part^.SA , 'ITEM_DESC' )
				end else begin
{$IFDEF DEBUG}
					ErrorMessage_fork( 'SCRIPT ERROR: FormatMessageString \ITEM_DESC ' + BStr( ID ) );
{$ENDIF DEBUG}
					W := 'SCRIPT ERROR: can''t find ITEM_DESC for item ' + BStr( ID );
				end;

			end else if W = '\ITEM_HISTORY' then begin
				{ Insert the description of a specified item. }
				ID := ScriptValue( S0 , GB , Scene );
				Part := GG_LocateItem( ID , GB , Scene );
				if Part <> Nil then begin
					W := SAttArrayValue( Part^.SA , 'ITEM_HISTORY' )
				end else begin
{$IFDEF DEBUG}
					ErrorMessage_fork( 'SCRIPT ERROR: FormatMessageString \ITEM_HISTORY ' + BStr( ID ) );
{$ENDIF DEBUG}
					W := 'SCRIPT ERROR: can''t find ITEM_DESC for item ' + BStr( ID );
				end;

			end else if W = '\ITEM_USAGE' then begin
				{ Insert the description of a specified item. }
				ID := ScriptValue( S0 , GB , Scene );
				Part := GG_LocateItem( ID , GB , Scene );
				if Part <> Nil then begin
					W := SAttArrayValue( Part^.SA , 'ITEM_USAGE' )
				end else begin
{$IFDEF DEBUG}
					ErrorMessage_fork( 'SCRIPT ERROR: FormatMessageString \ITEM_USAGE ' + BStr( ID ) );
{$ENDIF DEBUG}
					W := 'SCRIPT ERROR: can''t find ITEM_DESC for item ' + BStr( ID );
				end;

			end else if W = '\SECRET' then begin
				{ Insert a plot secret here. }
				ID := ScriptValue( S0 , GB , Scene );
				Part := GG_LocateItem( ID , GB , Scene );
				if Part <> Nil then begin
					W := ScriptMessage( 'msg' , GB , Part )
				end else begin
{$IFDEF DEBUG}
					ErrorMessage_fork( 'SCRIPT ERROR: FormatMessageString \SECRET ' + BStr( ID ) );
{$ENDIF DEBUG}
					W := 'SCRIPT ERROR: can''t find MSG for secret ' + BStr( ID );
				end;

			end else if W = '\FACTION' then begin
				{ Insert the name of a specified faction. }
				ID := ScriptValue( S0 , GB , Scene );
				Part := GG_LocateFaction( ID , GB , Scene );
				W := GEarName( Part );

			end else if W = '\FACTION_DESIG' then begin
				{ Insert the name of a specified faction. }
				ID := ScriptValue( S0 , GB , Scene );
				Part := GG_LocateFaction( ID , GB , Scene );
				if Part <> Nil then begin
					W := GearDesig_org( Part );
				end else begin
{$IFDEF DEBUG}
					ErrorMessage_fork( 'SCRIPT ERROR: FormatMessageString \FACTION_DESIG ' + BStr( ID ) );
{$ENDIF DEBUG}
					W := 'SCRIPT ERROR: Faction_Not_Found';
				end;

			end else if W = '\SCENE' then begin
				{ Insert the name of a specified scene. }
				ID := ScriptValue( S0 , GB , Scene );
				W := SceneName( GB , ID , False );

			end else if W = '\EXACT_SCENE' then begin
				{ Insert the name of a specified scene. }
				ID := ScriptValue( S0 , GB , Scene );
				W := SceneName( GB , ID , True );

			end else if W = '\VAL' then begin
				{ Insert the value of a specified variable. }
				ID := ScriptValue( S0 , GB , Scene );
				W := BStr( ID );

			end else if W = '\PC' then begin
				{ The name of the PC. }
				W := GearName( LocatePilot( GG_LocatePC( GB ) ) );

			end else if W = '\PCJOB' then begin
				{ The name of the PC. }
				Part := LocatePilot( GG_LocatePC( GB ) );
				if Part <> Nil then begin
					W := UTF8_Name( 'JOB' , SAttArrayValue( Part^.SA , 'JOB' ) );
				end else begin
{$IFDEF DEBUG}
					ErrorMessage_fork( 'SCRIPT ERROR: FormatMessageString \PCJOB ' );
{$ENDIF DEBUG}
					W := 'SCRIPT ERROR: PCJOB';
				end;

			end else if W = '\CHATNPC' then begin
				{ The name of the Chat NPC. }
				W := GearName( I_NPC );

			end else if W = '\CHATNPCMECHA' then begin
				{ The name of the Chat NPC's mecha. }
				Part := FindRoot( I_NPC );
				if ( Part <> Nil ) and ( Part^.G = GG_Mecha ) then begin
					W := FullGearName( Part );
				end else begin
					W := MsgString( 'MECHA' );
				end;

			end else if W = '\SOURCE' then begin
				{ The name of the PC. }
				W := GearName( Scene );

			end else if W = '\OPR' then begin
				{ Object Pronoun }
				ID := ScriptValue( S0 , GB , Scene );
				if ID = 0 then begin
					Part := LocatePilot( GG_LocatePC( GB ) );
				end else begin
					Part := GG_LocateNPC( ID , GB , Scene );
				end;
				if Part <> Nil then begin
					W := MsgString( 'OPR_' + BStr( NAttValue( Part^.NA , NAG_CharDescription , NAS_Gender ) ) );
				end else begin
					W := 'it';
				end;

			end else if W = '\SPR' then begin
				{ Object Pronoun }
				ID := ScriptValue( S0 , GB , Scene );
				if ID = 0 then begin
					Part := LocatePilot( GG_LocatePC( GB ) );
				end else begin
					Part := GG_LocateNPC( ID , GB , Scene );
				end;
				if Part <> Nil then begin
					W := MsgString( 'SPR_' + BStr( NAttValue( Part^.NA , NAG_CharDescription , NAS_Gender ) ) );
				end else begin
					W := 'it';
				end;

			end else if W = '\PPR' then begin
				{ Object Pronoun }
				ID := ScriptValue( S0 , GB , Scene );
				if ID = 0 then begin
					Part := LocatePilot( GG_LocatePC( GB ) );
				end else begin
					Part := GG_LocateNPC( ID , GB , Scene );
				end;
				if Part <> Nil then begin
					W := MsgString( 'PPR_' + BStr( NAttValue( Part^.NA , NAG_CharDescription , NAS_Gender ) ) );
				end else begin
					W := 'its';
				end;

			end else if W = '\OFFSPRING' then begin
				{ Son or Daughter, depending on gender }
				ID := ScriptValue( S0 , GB , Scene );
				if ID = 0 then begin
					Part := LocatePilot( GG_LocatePC( GB ) );
				end else begin
					Part := GG_LocateNPC( ID , GB , Scene );
				end;
				if Part <> Nil then begin
					if NAttValue( Part^.NA , NAG_CharDescription , NAS_Gender ) = NAV_Male then begin
						W := UTF8_MsgString( 'FormatMessageString' , 'SON' )
					end else if NAttValue( Part^.NA , NAG_CharDescription , NAS_Gender ) = NAV_Female then begin
						W := UTF8_MsgString( 'FormatMessageString' , 'DAUGHTER' );
					end else if NAttValue( Part^.NA , NAG_CharDescription , NAS_Gender ) = NAV_Nonbinary then begin
						W := UTF8_MsgString( 'FormatMessageString' , 'CHILD' );
					end else if NAttValue( Part^.NA , NAG_CharDescription , NAS_Gender ) = NAV_Undefined then begin
						W := UTF8_MsgString( 'FormatMessageString' , 'CHILD' );
					end else begin
						W := UTF8_MsgString( 'FormatMessageString' , 'CHILD' );
					end;
				end else begin
					W := 'sprog';
				end;

			end else if W = '\SIBLING' then begin
				{ Sister or Brother, depending on gender }
				ID := ScriptValue( S0 , GB , Scene );
				if ID = 0 then begin
					Part := LocatePilot( GG_LocatePC( GB ) );
				end else begin
					Part := GG_LocateNPC( ID , GB , Scene );
				end;
				if Part <> Nil then begin
					if NAttValue( Part^.NA , NAG_CharDescription , NAS_Gender ) = NAV_Male then begin
						W := UTF8_MsgString( 'FormatMessageString' , 'BROTHER' )
					end else if NAttValue( Part^.NA , NAG_CharDescription , NAS_Gender ) = NAV_Female then begin
						W := UTF8_MsgString( 'FormatMessageString' , 'SISTER' )
					end else if NAttValue( Part^.NA , NAG_CharDescription , NAS_Gender ) = NAV_Nonbinary then begin
						W := UTF8_MsgString( 'FormatMessageString' , 'RELATIVE' )
					end else if NAttValue( Part^.NA , NAG_CharDescription , NAS_Gender ) = NAV_Undefined then begin
						W := UTF8_MsgString( 'FormatMessageString' , 'RELATIVE' )
					end else begin
						W := UTF8_MsgString( 'FormatMessageString' , 'RELATIVE' )
					end;
				end else begin
					W := 'sib';
				end;

			end else if W = '\RANK' then begin
				{ The faction rank of the PC. }
				W := PCRankName( GB , Scene );

			end else if W = '\FACRANK' then begin
				{ A generic faction rank, not nessecarilt belonging }
				{ to the PC. }
				ID := ScriptValue( S0 , GB , Scene );
				ID2 := ScriptValue( S0 , GB , Scene );

				W := FactionRankName( GB , Scene , ID , ID2 );

			end else if W = '\DATE' then begin
				ID := ScriptValue( S0 , GB , Scene );

				W := TimeString( ID );

			end else if UpCase( W ) = '\NAME2' then begin
				W := ExtractWord( S0 );
				W := UTF8_Name( W, ExtractWord( S0 ) );
			end else if UpCase( W ) = '\NAME' then begin
				W := UTF8_Name( ExtractWord( S0 ) );
			end;
		end;

		if ( W <> '' ) and ( S1 <> '' ) and ( IsPunctuation( W[1] ) or ( S1[Length(S1)] = '$' ) or ( S1[Length(S1)] = '@' ) ) then begin
			S1 := S1 + W;
		end else begin
			if DItS then begin
				S1 := S1 + ' ' + W;
			end else begin
				S1 := S1 + W;
			end;
		end;

	end;

	msg := S1;
end;

Function ConditionAccepted( Event: String; gb: GameBoardPtr; Source: GearPtr ): Boolean;
	{ Run a conditional script. }
	{ If it returns 'ACCEPT', this function returns true. }
var
	T: String;	{ The trigger to be used. }
begin
	{ Error check - an empty condition is always true. }
	if Event = '' then Exit( True );

	{ Generate the trigger. }
	T := 'null';

	{ Execute the provided event. }
	InvokeEvent( Event , GB , Source , T );

	{ If the trigger was changed, that counts as a success. }
	ConditionAccepted := T = 'ACCEPT';
end;

Function ScriptMessage( msg_label: String; GB: GameBoardPtr; Source: GearPtr ): String;
	{ Retrieve and format a message from the source. }
var
	N,T: Integer;
	C,msg: String;
	MList: SAttListPtr;
	M: SAttArrayElementPtr;
begin
	{ Create the list of possible strings. }
	MList := CreateSAttList;
	C := AS_GetString( Source , 'C' + msg_label );

	{ The master condition must be accepted in order to continue. }
	if ConditionAccepted( C , GB , Source ) and ( Source <> Nil ) then begin
		msg := AS_GetString( Source , msg_label );
		if msg <> '' then StoreSAttList( MList , msg );

		msg := msg_label + '_';
		N := NumHeadMatches( msg , Source^.SA );
		for t := 1 to N do begin
			M := FindHeadMatch( msg , Source^.SA , T );
			if ( 0 < Length( M^.Info ) ) then begin
				C := SAttArrayValue( Source^.SA , 'C' + M^.Code );
				if ConditionAccepted( C , GB , Source ) then begin
					StoreSAttList( MList , M^.Info );
				end;
			end;
		end;
	end;

	{ If any messages were found, pick one. }
	if 0 < NumSAttList( MList ) then begin
		msg := SelectRandomSAttList( MList )^.Info;
		FormatMessageString( msg , gb , source );
	end else begin
		msg := '';
	end;
	DisposeSAttList( MList );

	ScriptMessage := Msg;
end;

Function NPCScriptMessage( const msg_label: String; GB: GameBoardPtr; NPC, Source: GearPtr ): String;
	{ Get a script message, temporarily setting the I_NPC to the provided NPC. }
var
	Temp_NPC: GearPtr;
	msg: String;
begin
	Temp_NPC := I_NPC;
	I_NPC := NPC;
	msg := ScriptMessage( msg_label , GB , Source );
	I_NPC := Temp_NPC;
	NPCScriptMessage := Msg;
end;

Function GetTheMessage( head: String; idnum: Integer; GB: GameBoardPtr; Scene: GearPtr ): String;
	{ Just call the SCRIPTMESSAGE with the correct label. }
begin
	GetTheMessage := ScriptMessage( head + BStr( idnum ) , GB , Scene );
end;

Procedure ProcessSetNPC( var Event: String; GB: GameBoardPtr; Scene: GearPtr );
var
	CID: Integer;
begin
	CID := ScriptValue( Event, GB, Scene );
	if ( 0 = CID ) then begin
		I_NPC := NIL;
	end else if 0 < CID then begin
		if NIL = I_NPC_org then begin
			I_NPC_org := I_NPC;
		end;
		I_NPC := GG_LocateNPC( CID , GB , GB^.Scene );
	end else begin
{$IFDEF DEBUG}
		ErrorMessage_fork( 'SCRIPT ERROR: NPC not found. ' + Make_ErrorMessage_ASL(Scene,Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT1(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT2(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT3(Event) );
{$ENDIF DEBUG}
		DialogMsg( 'SCRIPT ERROR: NPC not found. ' + Make_ErrorMessage_ASL(Scene,Event) );
	end;
end;

Procedure ProcessPrint( var Event: String; GB: GameBoardPtr; Scene: GearPtr );
	{ Locate and then print the specified message. }
var
	msg: String;
	id: Integer;
begin
	id := ScriptValue( Event , GB , Scene );
	msg := getTheMessage( 'msg', id , GB , Scene );
	msg := FormatChatStringByGender( msg, I_NPC );
	if msg <> '' then DialogMsg( msg );
	SetFlag_PhoneList( GB , msg , True , True , False , True );
end;

Procedure ProcessAlert( var Event: String; GB: GameBoardPtr; Scene: GearPtr );
	{ Locate and then print the specified message. }
var
	id: Integer;
	msg: String;
begin
	id := ScriptValue( Event , GB , Scene );
	msg := getTheMessage( 'msg', id , GB , Scene );
	msg := FormatChatStringByGender( msg, I_NPC );
	if msg <> '' then begin
		YesNoMenu( GB , msg , '' , '' );
		DialogMsg( msg );
		SetFlag_PhoneList( GB , msg , True , True , False , True );
	end;
end;

Procedure ProcessPrintDesc( var Event: String; GB: GameBoardPtr; Scene: GearPtr );
var
	msg: String;
begin
	msg := FormatDescString( Scene );
	if ( '' <> msg ) then begin
		DialogMsg( msg );
	end;
end;

Procedure ProcessGMonologue( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ An NPC is about to say something. }
var
	id: Integer;	{ Message ID }
	msg: String;
	NPC: GearPtr;
begin
	{ Record the NPC, and find the message number. }
	NPC := Grabbed_Gear;
	id := ScriptValue( Event , GB , Source );

	{ Locate the NPC and the message. }
	if NPC <> Nil then begin
		msg := NPCScriptMessage( 'msg' + BStr( id ) , GB , NPC , Source );
		if ( msg <> '' ) then begin
			msg := FormatChatStringByGender( msg , NPC );
			Monologue( GB , NPC , msg );
			SetFlag_PhoneList( GB , msg , True , True , False , True );
			{ The monologue will do its own output. }
		end;
	end else begin
{$IFDEF DEBUG}
		ErrorMessage_fork( 'SCRIPT WARNING: Illigal GMonologue. ' + Make_ErrorMessage_ASL(Source,Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT1(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT2(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT3(Event) );
{$ENDIF DEBUG}
		DialogMsg( 'SCRIPT WARNING: Illigal GMonologue. ' + Make_ErrorMessage_ASL(Source,Event) );
	end;
end;

Procedure ProcessAddDebriefing( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ Add a message for a certain NPC to speak during the debriefing. }
	{ Note that the first word in the debriefing message will be the ID of the }
	{ character meant to deliver the line. }
var
	cid,id: Integer;	{ Character ID, Message ID }
	NPC: GearPtr;
	msg: String;
begin
	{ Find the two needed numeric valies. }
	cid := ScriptValue( Event , GB , Source );
	id := ScriptValue( Event , GB , Source );

	{ Locate the message. }
	NPC := GG_LocateNPC( CID , GB , Source );
	msg := NPCScriptMessage( 'msg' + BStr( id ) , GB , NPC , Source );
	if ( msg <> '' ) and ( GB <> Nil ) and ( GB^.Scene <> Nil ) then begin
		msg := FormatChatStringByGender( msg , NPC );
		SetSAttArray( GB^.Scene^.SA , ARENAREPORT_Personal , BStr( cid ) + ' ' + msg );
	end;
end;

Procedure ProcessMemo( var Event: String; GB: GameBoardPtr; Scene: GearPtr );
	{ Locate and then store the specified message. }
var
	id: Integer;
	msg: String;
begin
	id := ScriptValue( Event , GB , Scene );
	msg := FormatMemoString( GB , getTheMessage( 'msg', id , GB , Scene ) );
	msg := FormatChatStringByGender( msg, I_NPC );
	if ( Scene <> Nil ) then SetSAttArray( Scene^.SA , 'MEMO' , msg );
	SetFlag_PhoneList( GB , msg , True , False , False , True );
end;

Procedure ProcessPMemo( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ Locate and then store the specified message in the plot gear, }
	{ along with an ID number. }
var
	id,plotid: LongInt;
	msg: String;
begin
	plotid := ScriptValue( Event , GB , Source );
	id := ScriptValue( Event , GB , Source );
	msg := FormatMemoString( GB , getTheMessage( 'msg', id , GB , Source ) );
	msg := FormatChatStringByGender( msg, I_NPC );
	Source := PlotMaster( GB , Source );
	if ( Source <> Nil ) then SetSAttArray( Source^.SA , 'MEMO' + BStr( plotid ) , msg );
	SetFlag_PhoneList( GB , msg , True , False , False , True );
end;

Procedure ProcessSMemo( var Event: String; GB: GameBoardPtr; Scene: GearPtr );
	{ Locate and then store the specified message in the story gear. }
var
	id: Integer;
	msg: String;
begin
	id := ScriptValue( Event , GB , Scene );
	msg := FormatMemoString( GB , getTheMessage( 'msg', id , GB , Scene ) );
	msg := FormatChatStringByGender( msg, I_NPC );
	Scene := StoryMaster( GB , Scene );
	if ( Scene <> Nil ) then SetSAttArray( Scene^.SA , 'MEMO' , msg );
	SetFlag_PhoneList( GB , msg , True , False , False , True );
end;

Procedure ProcessQMemo( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ Locate and then store the specified message. }
var
	qid,mid: LongInt;
	msg: String;
begin
	{ Determine the Quest ID and the Message ID. }
	qid := ScriptValue( Event , GB , Source );
	mid := ScriptValue( Event , GB , Source );
	msg := FormatMemoString( GB , getTheMessage( 'msg', mid , GB , Source ) );
	msg := FormatChatStringByGender( msg, I_NPC );

	{ Store this message in the adventure. }
	Source := GG_LocateAdventure( GB , Source );

	{ Quest memos look like regular memos but their tag is followed by an underscore }
	{ and the Quest ID. }
	if ( Source <> Nil ) then SetSAttArray( Source^.SA , 'MEMO_' + BStr( qid ) , msg );
	SetFlag_PhoneList( GB , msg , True , False , False , True );
end;

Procedure ProcessEMail( var Event: String; GB: GameBoardPtr; Scene: GearPtr );
	{ Locate and then store the specified message. }
var
	msg: String;
	PC: GearPtr;
	id: Integer;
begin
	id := ScriptValue( Event , GB , Scene );
	if id <> 0 then begin
		msg := getTheMessage( 'msg' , id , GB , Scene );
		msg := FormatChatStringByGender( msg, I_NPC );
		if ( Scene <> Nil ) and ( msg <> '' ) then SetSAttArray( Scene^.SA , 'EMAIL' , msg );
		SetFlag_PhoneList( GB , GearName( I_NPC ) , True , False , True , True );
		PC := GG_LocatePC( GB );
		if ( PC <> Nil ) and HasPCommCapability( PC , PCC_EMail ) and ( msg <> '' ) then DialogMsg( MsgString( 'AS_EMail' ) );
	end else if Scene <> Nil then begin
		SetSAttArray( Scene^.SA , 'EMAIL' , '' );
	end;
	if NIL <> I_NPC_org then begin
		I_NPC := I_NPC_org;
		I_NPC_org := NIL;
	end;
end;


Procedure ProcessHistory( var Event: String; GB: GameBoardPtr; Scene: GearPtr );
	{ Locate and then store the specified message. }
var
	id: Integer;
	msg: String;
	Adv: GearPtr;
begin
	id := ScriptValue( Event , GB , Scene );
	msg := getTheMessage( 'msg' , id , GB , Scene );
	Adv := GG_LocateAdventure( GB , Scene );
	if ( msg <> '' ) and ( Adv <> Nil ) then AddSAttArray( Adv^.SA , 'HISTORY' , msg );
	SetFlag_PhoneList( GB , msg , True , True , False , True );
end;

Procedure ProcessVictory( GB: GameBoardPtr );
	{ Sum up the entire campaign in a list of SAtts, then print to }
	{ a file. }
const
	InvStr = '+';
	SubStr = '>';
	Num_Significant_Relationships = 5;
	Significant_Relationship: Array [1..Num_Significant_Relationships] of Integer = (
		NAV_ArchEnemy, NAV_ArchAlly, NAV_Friend, NAV_Family, NAV_Lover
	);
var
	VList: SAttListPtr;
	i: Longint;
	PC,Fac,Adv: GearPtr;
	T,V: LongInt;
	msg,fname: String;
	Procedure CheckAlongPath( Part: GearPtr; TabPos,Prefix: String );
		{ CHeck along the path specified, adding info to }
		{ the victory list. }
	var
		msg: String;
	begin
		while Part <> Nil do begin
			if ( Part^.G <> GG_AbsolutelyNothing ) then begin
				StoreSAttList( VList , tabpos + prefix + GearName( Part ) );
				msg := ExtendedDescription( GB , Part );
				if msg <> '' then StoreSAttList( VList , tabpos + ' ' + msg );
			end;
			if Part^.G <> GG_Cockpit then begin
				CheckAlongPath( Part^.InvCom , TabPos + '  ' , InvStr );
				CheckAlongPath( Part^.SubCom , TabPos + '  ' , SubStr );
			end;
			Part := Part^.Next;
		end;
	end;{CheckAlongPath}
var
	A: Char;
	HList: SAttListPtr;
	SA: SAttArrayElementPtr;
	tmp: SAttListElementPtr;
	s: String;
begin
	{ Initialize our list to NIL. }
	VList := CreateSAttList;

	DialogMsg( MsgString( 'HISTORY_AnnounceVictory' ) );

	repeat
		CombatDisplay( GB );
		DoFlip;
		A := RPGKey;
	Until IsMoreKey( A );

	{ Locate the PC, add PC-specific information. }
	PC := LocatePilot( GG_LocatePC( GB ) );
	Adv := FindRoot( GB^.Scene );
	if PC <> Nil then begin
		{ Store the  name. }
		fname := GearName( PC );
		StoreSAttList( VList , fname );
		StoreSAttList( VList , JobAgeGenderDesc( PC ) );
		StoreSAttList( VList , TimeString( GB^.ComTime ) );
		StoreSAttList( VList , ' ' );


		{ Store the stats. }
		for t := 1 to 8 do begin
			msg := UTF8_Name( 'StatName', BStr( t ) );
			while Length( msg ) < 20 do msg := msg + ' ';
			msg := msg + BStr( PC^.Stat[ T ] );
			V := ( PC^.Stat[ T ] + 2 ) div 3;
			if V > 7 then V := 7;
			msg := msg + '  (' + MsgString( 'STATRANK' + BStr( V ) ) + ')';
			StoreSAttList( VList , msg );
		end;
		StoreSAttList( VList , ' ' );

		{ Add info on the PC's XP and credits. }
		msg := MsgString( 'INFO_XP' );
		V := NAttVAlue( PC^.NA , NAG_Experience , NAS_TotalXP );
		msg := msg + ' ' + BStr( V );
		StoreSAttList( VList , msg );

		msg := MsgString( 'INFO_XPLeft' );
		V := V - NAttVAlue( PC^.NA , NAG_Experience , NAS_SpentXP );
		msg := msg + ' ' + BStr( V );
		StoreSAttList( VList , msg );

		msg := MsgString( 'INFO_Credits' );
		V := NAttVAlue( PC^.NA , NAG_Experience , NAS_Credits );
		msg := msg + ' ' + BStr( V );
		StoreSAttList( VList , msg );

		{ Store the faction and rank. }
		Fac := GG_LocateFaction( NAttValue( PC^.NA , NAG_Personal , NAS_FactionID ) , GB , Nil );
		if Fac <> Nil then begin
			msg := UTF8_MsgString( 'ProcessVictory', 'HISTORY_FACTION' );
			msg := ReplaceHash( msg , PCRankName( GB , Nil ) , GearName( Fac ) );
			StoreSAttList( VList , msg );
			StoreSAttList( VList , ' ' );
		end;

		{ Store the relationships and fatalities. }
		if Adv <> Nil then begin
			for t := 1 to Num_Significant_Relationships do begin
				V := CountGearsByIDTag( Adv , NAG_Relationship , 0 , Significant_Relationship[ t ] ) + CountGearsByIDTag( GB^.Meks , NAG_Relationship , 0 , Significant_Relationship[ t ] );
				if V > 1 then begin
					StoreSAttList( VList , ReplaceHash( MsgString( 'HISTORY_RELATIONS_' + BStr( T ) ) , BStr( V ) ) );
				end else if V = 1 then begin
					StoreSAttList( VList , MsgString( 'HISTORY_RELATION_' + BStr( T ) ) );
				end;
			end;

			for t := 1 to Num_Fatality_Types do begin
				V := NAttValue( Adv^.NA , NAG_Narrative , Fatality_Base + T );
				if V > 1 then begin
					StoreSAttList( VList , ReplaceHash( MsgString( 'HISTORY_FATALITIES_' + BStr( T ) ) , BStr( V ) ) );
				end else if V = 1 then begin
					StoreSAttList( VList , MsgString( 'HISTORY_FATALITY_' + BStr( T ) ) );
				end;
			end;
		end;
		StoreSAttList( VList , ' ' );

		{ Store the personality traits. }
		for t := 1 to Num_Personality_Traits do begin
			V := NATtValue( PC^.NA , NAG_CharDescription , -T );
			if V <> 0 then begin
				Msg := UTF8_MsgString( 'ProcessVictory', 'HISTORY_Traits' );
				Msg := ReplaceHash( msg , PersonalityTraitDesc( T , V ) , BStr( Abs( V ) ) );
				StoreSAttList( VList , msg );
			end;
		end;
		StoreSAttList( VList , ' ' );

		{ Store the talents. }
		V := 0;
		for t := 1 to NumTalent do begin
			if HasTalent( PC , T ) then begin
				msg := MsgString( 'TALENT' + BStr( T ) );
				StoreSAttList( VList , msg );
				msg := '  ' + MsgString( 'TALENTDESC' + BStr( T ) );
				StoreSAttList( VList , msg );
				inc( V );
			end;
		end;
		if V > 0 then StoreSAttList( VList , ' ' );

		{ Store the skill ranks. }
		for t := 1 to NumSkill do begin
			V := NATtValue( PC^.NA , NAG_Skill , T );
			if V > 0 then begin
				msg := UTF8_MsgString( 'ProcessVictory', 'HISTORY_Skills' );
				msg := ReplaceHash( msg , MsgString( 'SKILLNAME_' + BStr( T ) ) , BStr( V ) );
				StoreSAttList( VList , msg );
			end;
		end;
		StoreSAttList( VList , ' ' );

		{ Store info on the PC's body and equipment. }
		CheckAlongPath( PC^.InvCom , '  ' , '+' );
		CheckAlongPath( PC^.SubCom , '  ' , '>' );
		StoreSAttList( VList , ' ' );

	end else begin
		{ No PC found, so filename will be "out.txt". }
		fname := 'out';
	end;

	if Adv <> Nil then begin
		{ Once the PC wins, unlock the adventure. }
		Adv^.V := 1;
		HList := CreateSAttList;
		if 0 < NumSAttArray( Adv^.SA ) then begin
			for i := 0 to ( Adv^.SA^.size - 1 ) do begin
				SA := Adv^.SA^.AoH[i];
				if ( ( NIL <> SA ) and ( SA^.act ) ) then begin
					if Copy( SA^.Code , 1 , 7 ) = 'HISTORY' then begin
						if ( 9 <= Length( SA^.Code ) ) then begin
							s := Copy( SA^.Code , 9 , Length( SA^.Code ) );
							v := ExtractValue( s );
						end else begin
							v := 0;
						end;
						s := RightStr( '000000' + BStr( v ) , 6 );
						SetSAttList( HList , s , SA^.Info );
					end;
				end;
			end;
		end;
		if 0 < NumSAttList( HList ) then begin
			tmp := HList^.head;
			while ( NIL <> tmp ) do begin
				StoreSAttList( VList , tmp^.Info );
				tmp := tmp^.Next;
			end;
		end;
		DisposeSAttList( HList );
	end;

	{ Add info on the PC's mechas. }
	PC := GB^.Meks;
	while PC <> Nil do begin
		if ( NAttValue( PC^.NA , NAG_Location , NAS_Team ) = NAV_DefPlayerTeam ) and ( PC^.G = GG_Mecha ) then begin
			StoreSAttList( VList , FullGearName( PC ) );

			CheckAlongPath( PC^.InvCom , '  ' , '+' );
			CheckAlongPath( PC^.SubCom , '  ' , '>' );

			StoreSAttList( VList , ' ' );
		end;
		PC := PC^.Next;
	end;

	SaveStringList( Config_Directory + FName + '.txt' , VList );
{$IFDEF ASCII}
	MoreText( VList , 1 );
{$ELSE}
	ASRD_GameBoard := GB;
	MoreText( VList , 1 , @ArenaScriptReDraw );
{$ENDIF}
	DisposeSAttList( VList );
end;

Procedure ProcessNews( var Event: String; GB: GameBoardPtr; Scene: GearPtr );
	{ Locate and then store the specified message. }
var
	msg: String;
	id: Integer;
begin
	id := ScriptValue( Event , GB , Scene );
	msg := getTheMessage( 'msg' , id , GB , Scene );
	msg := FormatChatStringByGender( msg, I_NPC );
	if ( msg <> '' ) and ( Scene <> Nil ) then SetSAttArray( Scene^.SA , 'NEWS' , msg );
end;

Procedure ProcessValueMessage( var Event: String; GB: GameBoardPtr; Scene: GearPtr );
	{ Locate and then print the specified message. }
var
	msg: String;
	V: LongInt;
begin
	{ FInd the message we're supposed to print. }
	msg := ExtractWord( Event );
	msg := MsgString( msg );

	{ Find the value to insert. }
	V := ScriptValue( Event , GB , Scene );

	{ Insert the value. }
	msg := ReplaceHash( msg , BStr( V ) );

	if msg <> '' then DialogMsg( msg );
end;

Procedure ProcessSay( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ Locate and then print the specified message. }
var
	id: Integer;
	msg: String;
begin
	{ Error check- if not in a conversation, call the PRINT }
	{ routine instead. }
	if IntMenu = Nil then begin
{$IFDEF DEBUG}
		ErrorMessage_fork( 'SCRIPT ERROR: Illigal Say. ' + Make_ErrorMessage_ASL(Source,Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT1(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT2(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT3(Event) );
{$ENDIF DEBUG}
		DialogMsg( 'SCRIPT ERROR: Illigal Say. ' + Make_ErrorMessage_ASL(Source,Event) );
		id := ScriptValue( Event , GB , Source );
		ProcessPrint( Event , GB , Source );
		Exit;
	end;

	id := ScriptValue( Event , GB , Source );
	msg := getTheMessage( 'msg' , id , GB , Source );
	msg := FormatChatStringByGender( msg, I_NPC );
	if msg <> '' then begin
		CHAT_Message := msg;
		SetFlag_PhoneList( GB , CHAT_Message , True , True , False , True );
	end;
end;

Procedure ProcessSayPlotMsg( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ Locate and then print the specified message. }
	Procedure PauseConversation();
		{ Before moving on to the rest of the conversation, take a }
		{ minute to read this message. }
	var
		RPM: RPGMenuPtr;
	begin
		RPM := CreateRPGMenu( MenuItem , MenuSelect , @ZONE_InteractMenu );
		AddRPGMenuItem( RPM , MsgString( 'Continue' ), SELECTMENU_Cancel );
		ASRD_GameBoard := GB;
		CHAT_React := ReactionScore( GB^.Scene , I_PC , I_NPC );
		SelectMenu( RPM , @InteractRedraw );
		DisposeRPGMenu( RPM );
	end;
var
	id: Integer;
	plot: GearPtr;
	msg: String;
begin
	{ Error check- if not in a conversation, call the PRINT }
	{ routine instead. }
	if IntMenu = Nil then begin
{$IFDEF DEBUG}
		ErrorMessage_fork( 'SCRIPT ERROR: Illigal SayPlotMsg. ' + Make_ErrorMessage_ASL(Source,Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT1(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT2(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT3(Event) );
{$ENDIF DEBUG}
		DialogMsg( 'SCRIPT ERROR: Illigal SayPlotMsg. ' + Make_ErrorMessage_ASL(Source,Event) );
		id := ScriptValue( Event , GB , Source );
		ProcessPrint( Event , GB , Source );
		Exit;
	end;

	id := ScriptValue( Event , GB , Source );
	plot := PlotMaster( GB , Source );
	msg := getTheMessage( 'msg' , id , GB , plot );
	if msg <> '' then begin
		msg := FormatChatStringByGender( msg , I_NPC );
		YesNoMenu( GB , ReplaceHash( MsgString( 'Debrief_Intro' ) , PilotName( I_NPC ) ) , '' , '' );
		CHAT_Message := msg;
		SetFlag_PhoneList( GB , CHAT_Message , True , True , False , True );
		PauseConversation();
	end;
end;


Procedure ProcessAddChat( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ Add a new item to the IntMenu. }
var
	N: Integer;
	Msg: String;
begin
	{ Error check - this command can only work if the IntMenu is }
	{ already allocated. }
	if ( IntMenu <> Nil ) and ( Source <> Nil ) then begin
		{ First, determine the prompt number. }
		N := ScriptValue( Event , GB , Source );

		msg := getthemessage( 'PROMPT' , N , GB , Source );
		DeleteWhiteSpace( msg );
		msg := FormatChatStringByGender( msg, I_PC );
		if Msg <> '' then begin
			AddRPGMenuItem( IntMenu , Msg , N );
			{ RPMSortAlpha( IntMenu ); } { In UTF-8, this sort causes a result that I cannot assume. }
		end;
	end else begin
		N := ScriptValue( Event , GB , Source );
{$IFDEF DEBUG}
		ErrorMessage_fork( 'SCRIPT ERROR: Illigal AddChat. ' + Make_ErrorMessage_ASL(Source,Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT1(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT2(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT3(Event) );
{$ENDIF DEBUG}
		DialogMsg( 'SCRIPT ERROR: Illigal AddChat. ' + Make_ErrorMessage_ASL(Source,Event) );
	end;
end;

Procedure ProcessSayAnything( var Event: String; GB: GameBoardPtr );
	{ Print a random message in the interact message area. }
	{ New- if there are any memes floating about, select one of those for the message. }
	Procedure CleanMemes( City: GearPtr );
		{ Most memes come with a "best before" date. If they are not revealed to the player }
		{ by this time, they get deleted. This can be useful for things like news events. }
	var
		M,M2: GearPtr;
	begin
		M := City^.SubCom;
		while M <> Nil do begin
			M2 := M^.Next;
			if ( M^.G = GG_Meme ) and ( NAttValue( M^.NA , NAG_MemeData , NAS_MemeTimeLimit ) > 0 ) and ( NAttValue( M^.NA , NAG_MemeData , NAS_MemeTimeLimit ) < GB^.ComTime ) then begin
				RemoveGear( City^.SubCom , M );
			end;
			M := M2;
		end;
	end;
	Function NumMeme( City: GearPtr ): Integer;
		{ Return the number of active memes in this city. }
	var
		M: GearPtr;
		N: Integer;
	begin
		if City = Nil then Exit( 0 );
		M := City^.SubCom;
		N := 0;
		while M <> Nil do begin
			if ( M^.G = GG_Meme ) or ( M^.G = GG_CityMood ) then Inc( N );
			M := M^.Next;
		end;
		NumMeme := N;
	end;
	Function GetMeme( City: GearPtr; N: Integer ): GearPtr;
		{ Return the requested meme. }
	var
		S,M: GearPtr;
	begin
		S := City^.SubCom;
		M := Nil;
		while ( S <> Nil ) and ( N > 0 ) do begin
			if ( S^.G = GG_Meme ) or ( S^.G = GG_CityMood ) then begin
				Dec( N );
				if N = 0 then M := S;
			end;
			S := S^.Next;
		end;
		GetMeme := M;
	end;
var
	N: Integer;
	City,Meme: GearPtr;
	msg: String;
	NPC: GearPtr;
	IntScr: String;
	T: String;
begin
	City := FindRootScene( GB^.Scene );
	{ Before checking for memes, delete any expired memes that might still be kicking around. }
	if City <> Nil then CleanMemes( City );

	N := NumMeme( City );
	if ( I_NPC <> Nil ) and ( I_PC <> Nil ) and ( GB <> Nil ) and ( ReactionScore( GB^.Scene , I_PC , I_NPC ) <= -Random( 10 ) ) then begin
		{ If talking to someone who dislikes you, you're most likely to get }
		{ a brushoff. }
		msg := MsgString( 'BUZZOFF_' + BStr( Random( 5 ) + 1 ) );
		msg := FormatChatStringByGender( msg , I_NPC );

	end else if N > 0 then begin
		Meme := GetMeme( City , Random( N ) + 1 );

		msg := ScriptMessage( 'Msg' , GB , Meme );
		msg := FormatChatStringByGender( msg , I_NPC );

		if msg = '' then begin
			msg := IdleChatter( I_NPC );
		end else begin
			{ A message was successfully extracted from this meme. Increment the }
			{ message counter and maybe delete it. }
			AddNAtt( Meme^.NA , NAG_MemeData , NAS_NumViews , 1 );
			if ( Meme^.G = GG_Meme ) and ( NAttValue( Meme^.NA , NAG_MemeData , NAS_NumViews ) >= NAttValue( Meme^.NA , NAG_MemeData , NAS_MaxMemeViews ) ) then begin
				RemoveGear( City^.SubCom , Meme );
			end;
		end;
	end else begin
		msg := IdleChatter( I_NPC );
	end;

	CHAT_Message := msg;
	SetFlag_PhoneList( GB , CHAT_Message , True , True , False , True );

	if not CHEAT_DoNotConsider_LancemateTacticsPersona_in_SayAnything then begin
		NPC := I_NPC;
		if ( Length( Event ) <= 0 ) and ( I_Persona <> lancemate_tactics_persona ) and ( NAV_LancemateTeam = NAttValue( NPC^.NA , NAG_Location , NAS_Team ) ) and ( NAV_TempLancemate <> NAttValue( NPC^.NA , NAG_Chardescription , NAS_CharType ) ) then begin
			I_Persona := lancemate_tactics_persona;
			IntScr := AS_GetString( I_Persona , 'GREETING' );
			T := 'Greeting';
			InvokeEvent( IntScr , GB , I_Persona , T );
			CHAT_Message := msg + ' ' + CHAT_Message;
		end;
	end;
end;

Procedure ProcessActivateMeme( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ Activate a meme. The meme should be stored in SOURCE, the plot master of SOURCE, or the }
	{ story master of SOURCE. If the named meme cannot be found this will generate an error. }
var
	MemeID,SceneID: LongInt;
	Meme,Scene: GearPtr;
begin
	{ Get the meme ID and the scene ID. }
	MemeID := ScriptValue( event , GB , source );
	SceneID := ScriptValue( event , GB , source );
	Meme := GG_LocateItem( MemeID , GB , Source );
	Scene := FindRootScene( FindActualScene( GB , SceneID ) );

	if ( Meme = Nil ) or ( Meme^.G <> GG_Meme ) then begin
{$IFDEF DEBUG}
		ErrorMessage_fork( 'SCRIPT ERROR: Meme ' + BStr( MemeID ) + ' not found. Context: ' + Event );
{$ENDIF DEBUG}
		DialogMsg( 'SCRIPT ERROR: Meme ' + BStr( MemeID ) + ' not found. Context: ' + Event );
	end else if Scene = Nil then begin
{$IFDEF DEBUG}
		ErrorMessage_fork( 'SCRIPT ERROR: ActivateMeme failed, scene ' + BStr( SceneID ) + ' not found. Context: ' + Event );
{$ENDIF DEBUG}
		DialogMsg( 'SCRIPT ERROR: ActivateMeme failed, scene ' + BStr( SceneID ) + ' not found. Context: ' + Event );
	end else begin
		DelinkGearForMovement( GB , Meme );
		InsertSubCom( Scene , Meme );

		{ If this meme has a time limit, set that now. }
		if NAttValue( Meme^.NA , NAG_MemeData , NAS_MemeTimeLimit ) > 0 then begin
			AddNAtt( Meme^.NA , NAG_MemeData , NAS_MemeTimeLimit , GB^.ComTime );
		end;
	end;
end;

Procedure ProcessGSetNAtt( var Event: String; GB: GameBoardPtr; Scene: GearPtr );
	{ The script is going to assign a value to one of the scene }
	{ variables. }
var
	G,S: Integer;
	V: LongInt;
begin
	{ Find the variable ID number and the value to assign. }
	G := ScriptValue( event , GB , scene );
	S := ScriptValue( event , GB , scene );
	V := ScriptValue( event , GB , scene );
	if DEBUG_ON_arenascript then begin
{$IFDEF DEBUG}
		ErrorMessage_fork( 'GSetNAtt: ' + GearName( Grabbed_Gear ) + ' ' + BStr( G ) + '/' + BStr( S ) + '/' + BStr( V ) );
{$ENDIF DEBUG}
		DialogMsg( 'GSetNAtt: ' + GearName( Grabbed_Gear ) + ' ' + BStr( G ) + '/' + BStr( S ) + '/' + BStr( V ) );
	end;

	if Grabbed_Gear <> Nil then begin
		SetNAtt( Grabbed_Gear^.NA , G , S , V );
	end else begin
{$IFDEF DEBUG}
		ErrorMessage_fork( 'SCRIPT WARNING: Illigal GSetNAtt. ' + Make_ErrorMessage_ASL(Scene,Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT1(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT2(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT3(Event) );
{$ENDIF DEBUG}
		DialogMsg( 'SCRIPT WARNING: Illigal GSetNAtt. ' + Make_ErrorMessage_ASL(Scene,Event) );
	end;
end;

Procedure ProcessGAddNAtt( var Event: String; GB: GameBoardPtr; Scene: GearPtr );
	{ The script is going to add a value to one of the scene }
	{ variables. }
var
	G,S: Integer;
	V: LongInt;
begin
	{ Find the variable ID number and the value to assign. }
	G := ScriptValue( event , GB , scene );
	S := ScriptValue( event , GB , scene );
	V := ScriptValue( event , GB , scene );
	if DEBUG_ON_arenascript then begin
{$IFDEF DEBUG}
		ErrorMessage_fork( 'GAddNAtt: ' + GearName( Grabbed_Gear ) + ' ' + BStr( G ) + '/' + BStr( S ) + '/' + BStr( V ) );
{$ENDIF DEBUG}
		DialogMsg( 'GAddNAtt: ' + GearName( Grabbed_Gear ) + ' ' + BStr( G ) + '/' + BStr( S ) + '/' + BStr( V ) );
	end;

	if Grabbed_Gear <> Nil then begin
		AddNAtt( Grabbed_Gear^.NA , G , S , V );
	end else begin
{$IFDEF DEBUG}
		ErrorMessage_fork( 'SCRIPT WARNING: Illigal GAddNAtt. ' + Make_ErrorMessage_ASL(Scene,Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT1(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT2(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT3(Event) );
{$ENDIF DEBUG}
		DialogMsg( 'SCRIPT WARNING: Illigal GAddNAtt. ' + Make_ErrorMessage_ASL(Scene,Event) );
	end;
end;

Procedure ProcessGSetStat( var Event: String; GB: GameBoardPtr; Scene: GearPtr );
	{ The script is going to add a value to one of the scene }
	{ variables. }
var
	Slot,Value: Integer;
begin
	{ Find the variable ID number and the value to assign. }
	Slot := ScriptValue( event , GB , scene );
	Value := ScriptValue( event , GB , scene );

	if Grabbed_Gear <> Nil then begin
		Grabbed_Gear^.Stat[ Slot ] := Value;
		ResizeCharacter(Grabbed_Gear);
	end else begin
{$IFDEF DEBUG}
		ErrorMessage_fork( 'SCRIPT WARNING: Illigal GSetStat. ' + Make_ErrorMessage_ASL(Scene,Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT1(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT2(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT3(Event) );
{$ENDIF DEBUG}
		DialogMsg( 'SCRIPT WARNING: Illigal GSetStat. ' + Make_ErrorMessage_ASL(Scene,Event) );
	end;
end;

Procedure ProcessGAddStat( var Event: String; GB: GameBoardPtr; Scene: GearPtr );
	{ The script is going to add a value to one of the scene }
	{ variables. }
var
	Slot,Value: Integer;
begin
	{ Find the variable ID number and the value to assign. }
	Slot := ScriptValue( event , GB , scene );
	Value := ScriptValue( event , GB , scene );

	if Grabbed_Gear <> Nil then begin
		Grabbed_Gear^.Stat[ Slot ] := Grabbed_Gear^.Stat[ Slot ] + Value;
		ResizeCharacter(Grabbed_Gear);
	end else begin
{$IFDEF DEBUG}
		ErrorMessage_fork( 'SCRIPT WARNING: Illigal GAddStat. ' + Make_ErrorMessage_ASL(Scene,Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT1(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT2(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT3(Event) );
{$ENDIF DEBUG}
		DialogMsg( 'SCRIPT WARNING: Illigal GAddStat. ' + Make_ErrorMessage_ASL(Scene,Event) );
	end;
end;

Procedure ProcessGSetSAttArray( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ Store a string attribute in the grabbed gear. }
var
	Key,Info: String;
begin
	Key := ExtractWord( Event );
	Info := ExtractWord( Event );
	if Source <> Nil then Info := AS_GetString( Source , Info );
	FormatMessageString( info , GB , source );
	DeleteWhiteSpace( info );
	if Grabbed_Gear <> Nil then begin
		SetSAttArray( Grabbed_Gear^.SA , Key , Info );
	end else begin
{$IFDEF DEBUG}
		ErrorMessage_fork( 'SCRIPT WARNING: Illigal GSetSAttArray. ' + Make_ErrorMessage_ASL(Source,Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT1(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT2(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT3(Event) );
{$ENDIF DEBUG}
		DialogMsg( 'SCRIPT WARNING: Illigal GSetSAttArray. ' + Make_ErrorMessage_ASL(Source,Event) );
	end;
end;

Procedure IfSuccess( var Event: String; Scene: GearPtr );
	{ An IF call has generated a "TRUE" result. Just get rid of }
	{ any ELSE clause that the event string might still be holding. }
var
	cmd: String;
begin
	{ Extract the next word from the script. }
	cmd := ExtractWord( Event );

{$IFDEF DEBUG}
	ErrorMessage_fork('SCRIPT TRACE: IfSuccess ' + cmd + ' in "' + GearName( scene ) + '" to <' + Event + '>' );
{$ENDIF DEBUG}

	{ If the next word is ELSE, we have to also extract the label. }
	{ If the next word isn't ELSE, better re-assemble the line... }
	if UpCase( cmd ) = 'ELSE' then ExtractWord( Event )
	else Event := cmd + ' ' + Event;
end;

Procedure IfFailure( var Event: String; Scene: GearPtr );
	{ An IF call has generated a "FALSE" result. See if there's }
	{ a defined ELSE clause, and try to load the next line. }
var
	cmd: String;
begin
	{ Extract the next word from the script. }
	cmd := ExtractWord( Event );

{$IFDEF DEBUG}
	ErrorMessage_fork('SCRIPT TRACE: IfFailure ' + cmd + ' in "' + GearName( scene ) + '" to <' + Event + '>' );
{$ENDIF DEBUG}

	if UpCase( cmd ) = 'ELSE' then begin
		{ There's an else clause. Attempt to jump to the }
		{ specified script line. }
		cmd := ExtractWord( Event );
		Event := AS_GetString( Scene , CMD );
{$IFDEF DEBUG}
		ErrorMessage_fork('SCRIPT TRACE: ELSE ' + cmd + ' in "' + GearName( scene ) + '" to <' + Event + '>' );
{$ENDIF DEBUG}
		DEBUG_cmd_org2 := Event;	{ for DEBUG }

	end else begin
		{ There's no ELSE clause. Just cease execution of this }
		{ line by setting it to an empty string. }
		Event := '';
	end;
end;

Procedure ProcessIfGInPlay( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ Return true if the Grabbed_Gear is on the map and operational. }
	{ Return false otherwise. }
begin
	if ( Grabbed_Gear <> Nil ) and OnTheMap( GB , Grabbed_Gear ) and GearOperational( Grabbed_Gear ) and IsFoundAlongTrack( GB^.Meks , FindRoot( Grabbed_Gear ) ) then begin
		IfSuccess( Event , Source );
	end else begin
		IfFailure( Event , Source );
	end;
end;

Procedure ProcessIfGOK( var Event: String; Source: GearPtr );
	{ If the grabbed gear is OK, count as true. If it is destroyed, }
	{ or if it can't be found, count as false. }
begin
	if ( Grabbed_Gear <> Nil ) and NotDestroyed( Grabbed_Gear ) then begin
		IfSuccess( Event , Source );
	end else IfFailure( Event , Source );
end;

Procedure ProcessIfGDead( var Event: String; Source: GearPtr );
	{ If the grabbed gear is dead or nil, count as true. }
begin
	if ( Grabbed_Gear = Nil ) or Destroyed( Grabbed_Gear ) then begin
		IfSuccess( Event , Source );
	end else IfFailure( Event , Source );
end;

Procedure ProcessIfGSexy( var Event: String; gb: GameBoardPtr; Source: GearPtr );
	{ If the grabbed gear is sexy to the PC, count as true. If it is not, }
	{ or if it can't be found, count as false. }
var
	PC: GearPtr;
begin
	PC := GG_LOcatePC( GB );
	if ( Grabbed_Gear <> Nil ) and ( PC <> Nil ) and IsSexy( PC , Grabbed_Gear ) then begin
		IfSuccess( Event , Source );
	end else IfFailure( Event , Source );
end;

Procedure ProcessIfGSealed( var Event: String; Source: GearPtr );
	{ If the grabbed gear is EnviroSealed, count as true. If not, }
	{ count as false. }
begin
	if ( Grabbed_Gear <> Nil ) and IsEnviroSealed( Grabbed_Gear ) then begin
		IfSuccess( Event , Source );
	end else IfFailure( Event , Source );
end;

Procedure ProcessIfGArchEnemy( var Event: String; gb: GameBoardPtr; Source: GearPtr );
	{ If the grabbed gear is an enemy of the PC, or belongs to a faction that's }
	{ an enemy of the PC, count as true. }
var
	Adv: GearPtr;
begin
	Adv := GG_LOcateAdventure( GB , Source );
	if ( Grabbed_Gear <> Nil ) and ( Adv <> Nil ) and IsArchEnemy( Adv , Grabbed_Gear ) then begin
		IfSuccess( Event , Source );
	end else IfFailure( Event , Source );
end;

Procedure ProcessIfGArchAlly( var Event: String; gb: GameBoardPtr; Source: GearPtr );
	{ If the grabbed gear is an ally of the PC, or belongs to a faction that's }
	{ an ally of the PC, count as true. }
var
	Adv: GearPtr;
begin
	Adv := GG_LOcateAdventure( GB , Source );
	if ( Grabbed_Gear <> Nil ) and ( Adv <> Nil ) and IsArchAlly( Adv , Grabbed_Gear ) then begin
		IfSuccess( Event , Source );
	end else IfFailure( Event , Source );
end;

Procedure ProcessIfGHasSkill( var Event: String; gb: GameBoardPtr; Source: GearPtr );
	{ If the grabbed gear has the skill, count as true. If it doesn't, }
	{ or if it can't be found, count as false. }
var
	Skill: Integer;
begin
	Skill := ScriptValue( Event , GB , Source );
	if ( Grabbed_Gear <> Nil ) and HasSkill( Grabbed_Gear , Skill ) then begin
		IfSuccess( Event , Source );
	end else IfFailure( Event , Source );
end;

Procedure ProcessIfGCanJoinLance( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ Return true if the Grabbed_Gear can join the PC's lance, or FALSE otherwise. }
	{ Please note that the NPC must be in play in order to join the lance. }
var
	PC: GearPtr;
begin
	PC := GG_LocatePC( GB );
	if ( Grabbed_Gear <> Nil ) and OnTheMap( GB , Grabbed_Gear ) and GearOperational( Grabbed_Gear ) and IsFoundAlongTrack( GB^.Meks , FindRoot( Grabbed_Gear ) ) and CanJoinLance( GB , PC , Grabbed_Gear ) then begin
		IfSuccess( Event , Source );
	end else begin
		IfFailure( Event , Source );
	end;
end;

Procedure ProcessIfMechaCanEnterScene( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ Return true if the PC's mecha can enter the requested scene. }
	{ Return false otherwise. }
var
	PC,Mek,Scene: GearPtr;
	SceneID: Integer;
begin
	{ Locate the PC, the PC's mecha, and the target scene. }
	PC := FindRoot( GG_LocatePC( GB ) );
	if ( PC <> Nil ) then begin
		if PC^.G = GG_Mecha then Mek := PC
		else Mek := FindPilotsMecha( GB^.Meks , PC );
	end;
	SceneID := ScriptValue( Event , GB , Source );
	Scene := FindActualScene( GB , SceneID );

	if ( PC <> Nil ) and ( Mek <> Nil ) and ( Scene <> Nil ) and NotDestroyed( Mek ) and MekCanEnterScene( Mek , Scene ) then begin
		IfSuccess( Event , Source );
	end else begin
		IfFailure( Event , Source );
	end;
end;

Procedure ProcessIfMeritBadge( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ Return true if the PC has the requested merit badge. }
	{ Return false otherwise. }
var
	Adv: GearPtr;
	Badge: Integer;
begin
	Badge := ScriptValue( Event , GB , Source );

	Adv := GG_LocateAdventure( GB , Source );

	if ( Adv <> Nil ) and HasMeritBadge( Adv , Badge ) then begin
		IfSuccess( Event , Source );
	end else begin
		IfFailure( Event , Source );
	end;
end;

Procedure ProcessIfNoDesc( var Event: String; GB: GameBoardPtr; Source: GearPtr );
begin
	if ( Length( SAttArrayValue( Source^.SA , 'DESC' ) ) <= 0 ) then begin
		IfSuccess( Event , Source );
	end else begin
		IfFailure( Event , Source );
	end;
end;


Procedure ProcessIfTeamCanSeeGG( var Event: String; gb: GameBoardPtr; Source: GearPtr );
	{ If the grabbed gear can be seen by the requested team, return TRUE. }
var
	Team: Integer;
begin
	Team := ScriptValue( Event , GB , Source );
	if ( Grabbed_Gear <> Nil ) and TeamCanSeeTarget( GB , Team , Grabbed_Gear ) then begin
		IfSuccess( Event , Source );
	end else IfFailure( Event , Source );
end;

Procedure ProcessIfFaction( var Event: String; gb: GameBoardPtr; Source: GearPtr );
	{ Check to see if the requested faction is active or not. }
var
	FID: Integer;
	Fac: GearPtr;
begin
	{ Locate the requested Faction ID, and from there locate }
	{ the faction gear itself. }
	FID := ScriptValue( Event , GB , Source );
	Fac := GG_LocateFaction( FID , GB , Source );

	{ If the faction was found, see whether or not it's active. }
	if Fac <> Nil then begin
		if FactionIsInactive( Fac ) then IfFailure( Event , Source )
		else IfSuccess( Event , Source );

	{ If said faction cannot be found, it counts as a failure. }
	end else IfFailure( Event , Source );
end;

Procedure ProcessIfStoryless( var Event: String; Source: GearPtr );
	{ Return true if the SOURCE has no story linked. }
	{ Return false otherwise. }
var
	story: GearPtr;
begin
	if Source <> Nil then begin
		story := Source^.InvCom;
		while ( story <> Nil ) and ( story^.G <> GG_Story ) do story := story^.Next;

		if Story = Nil then begin
			IfSuccess( Event , Source );
		end else begin
			IfFailure( Event , Source );
		end;
	end else begin
		IfFailure( Event , Source );
	end;
end;

Procedure ProcessIfEqual( var Event: String; gb: GameBoardPtr; Source: GearPtr );
	{ Two values are supplied as the arguments for this procedure. }
	{ If they are equal, that's a success. }
var
	a,b: LongInt;
begin
	{ Determine the two values. }
	A := ScriptValue( Event , gb , Source );
	B := ScriptValue( Event , gb , Source );

{$IFDEF DEBUG}
	ErrorMessage_fork('SCRIPT TRACE: IfEqual(' + BStr(A) + ',' + BStr(B) + ') in "' + GearName( Source ) + '" to <' + Event + '>' );
{$ENDIF DEBUG}

	if A = B then IfSuccess( Event , Source )
	else IfFailure( Event , Source );
end;

Procedure ProcessIfNotEqual( var Event: String; gb: GameBoardPtr; Source: GearPtr );
	{ Two values are supplied as the arguments for this procedure. }
	{ If they are not equal, that's a success. }
var
	a,b: LongInt;
begin
	{ Determine the two values. }
	A := ScriptValue( Event , gb , Source );
	B := ScriptValue( Event , gb , Source );

{$IFDEF DEBUG}
	ErrorMessage_fork('SCRIPT TRACE: IfNotEqual(' + BStr(A) + ',' + BStr(B) + ') in "' + GearName( Source ) + '" to <' + Event + '>' );
{$ENDIF DEBUG}

	if A <> B then IfSuccess( Event , Source )
	else IfFailure( Event , Source );
end;

Procedure ProcessIfGreater( var Event: String; gb: GameBoardPtr; Source: GearPtr );
	{ Two values are supplied as the arguments for this procedure. }
	{ If A > B, that's a success. }
var
	a,b: LongInt;
begin
	{ Determine the two values. }
	A := ScriptValue( Event , gb , Source );
	B := ScriptValue( Event , gb , Source );

{$IFDEF DEBUG}
	ErrorMessage_fork('SCRIPT TRACE: IfGreater(' + BStr(A) + ',' + BStr(B) + ') in "' + GearName( Source ) + '" to <' + Event + '>' );
{$ENDIF DEBUG}

	if A > B then IfSuccess( Event , Source )
	else IfFailure( Event , Source );
end;

Procedure ProcessIfKeyItem( var Event: String; gb: GameBoardPtr; Source: GearPtr );
	{ Process TRUE if the specified key item is in the posession of the PC. }
	{ We'll define this as being in the posession of any member of team }
	{ one... Process FALSE if it isn't. }
var
	NID: Integer;
	FoundTheItem: Boolean;
	PC: GearPtr;
begin
	{ Start by assuming FALSE, then go looking for it. }
	FoundTheItem := False;

	{ Find out what Key Item we're looking for. }
	NID := ScriptValue( Event , GB , Source );

	if ( GB <> Nil ) and ( NID <> 0 ) then begin
		{ Search through every gear on the map. }
		PC := GB^.Meks;

		while PC <> Nil do begin
			{ If this gear belongs to the player team, check it }
			{ for the wanted item. }
			if NAttValue( PC^.NA , NAG_Location , NAS_Team ) = NAV_DefPlayerTeam then begin
				{ Set FOUNDTHEITEM to TRUE if the specified key item }
				{ is the PC gear itself, if it's in the subcoms of the PC, }
				{ or if it's somewhere in the inventory of the PC. }
				if NAttValue( PC^.NA , NAG_Narrative , NAS_NID ) = NID then FoundTheItem := True
				else if SeekGearByIDTag( PC^.SubCom , NAG_Narrative , NAS_NID , NID ) <> Nil then FoundTheItem := True
				else if SeekGearByIDTag( PC^.InvCom , NAG_Narrative , NAS_NID , NID ) <> Nil then FoundTheItem := True;
			end;

			{ Move to the next gear to check. }
			PC := PC^.Next;
		end;
	end;

	{ Finally, do something appropriate depending upon whether or not }
	{ the item was found. }
	if FoundTheItem then IfSuccess( Event , Source )
	else IfFailure( Event , Source );
end;

Procedure ProcessIfGHasItem( var Event: String; gb: GameBoardPtr; Source: GearPtr );
	{ Process TRUE if the specified key item is in the posession of the grabbed gear. }
var
	NID: Integer;
	FoundTheItem: Boolean;
begin
	{ Start by assuming FALSE, then go looking for it. }
	FoundTheItem := False;

	{ Find out what Key Item we're looking for. }
	NID := ScriptValue( Event , GB , Source );

	if ( Grabbed_Gear <> Nil ) and ( NID <> 0 ) then begin
		if NAttValue( Grabbed_Gear^.NA , NAG_Narrative , NAS_NID ) = NID then FoundTheItem := True
		else if SeekGearByIDTag( Grabbed_Gear^.SubCom , NAG_Narrative , NAS_NID , NID ) <> Nil then FoundTheItem := True
		else if SeekGearByIDTag( Grabbed_Gear^.InvCom , NAG_Narrative , NAS_NID , NID ) <> Nil then FoundTheItem := True;
	end;

	{ Finally, do something appropriate depending upon whether or not }
	{ the item was found. }
	if FoundTheItem then IfSuccess( Event , Source )
	else IfFailure( Event , Source );
end;

Procedure ProcessIfYesNo( var Event: String; gb: GameBoardPtr; Source: GearPtr );
	{ Two values are supplied as the arguments for this procedure. }
	{ If they are equal, that's a success. }
var
	Desc,YesPrompt,NoPrompt: String;
	it: Boolean;
	ID: Integer;
begin
	{ Find all the needed messages. }
	id := ScriptValue( Event , GB , Source );
	Desc := GetTheMessage( 'msg' , id , GB , Source );
	id := ScriptValue( Event , GB , Source );
	YesPrompt := GetTheMessage( 'msg' , id , GB , Source );
	id := ScriptValue( Event , GB , Source );
	NoPrompt := GetTheMessage( 'msg' , id , GB , Source );

	Desc      := FormatChatStringByGender( Desc     , I_NPC );
	YesPrompt := FormatChatStringByGender( YesPrompt, I_PC );
	NoPrompt  := FormatChatStringByGender( NoPrompt , I_PC );

	it := YesNoMenu( GB , Desc , YesPrompt , NoPrompt );

	if it then IfSuccess( Event , Source )
	else IfFailure( Event , Source );
end;

Procedure ProcessIfScene( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ Return TRUE if the current scene matches the provided }
	{ description, or FALSE otherwise. }
var
	Desc: String;
begin
	Desc := ExtractWord( Event );
	if Source <> Nil then Desc := AS_GetString( Source , Desc );

	if ( GB <> Nil ) and ( GB^.Scene <> Nil ) and PartMatchesCriteria( SceneDesc( GB^.Scene ) , Desc ) then begin
		IfSuccess( Event , Source );
	end else begin
		IfFailure( Event , Source );
	end;
end;

Procedure ProcessIfSafeArea( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ Return TRUE if the current scene is a Safe Area (as determined }
	{ by the function of the same name), or FALSE otherwise. }
begin
	if ( GB <> Nil ) and IsSafeArea( GB ) then begin
		IfSuccess( Event , Source );
	end else begin
		IfFailure( Event , Source );
	end;
end;

Procedure ProcessIfSkillTest( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ Return TRUE if the PC makes the requested skill roll, or FALSE otherwise. }
	{ If the PC has already attempted this skill roll, he can't succeed unless }
	{ he's improved his skill level. }
var
	PC: GearPtr;
	Skill,SkStat,SkTar,SkRank,SkRoll: Integer;
begin
	PC := GG_LocatePC( GB );
	Skill := ScriptValue( Event , GB , Source );
	SkStat := ScriptValue( Event , GB , Source );
	SkRank := SkillRank( PC , Skill ) + 1;
	SkTar := ScriptValue( Event , GB , Source );
	if ( Source <> Nil ) and ( SkRank <= NAttValue( Source^.NA , NAG_SkillCounter , Skill ) ) then begin
		IfFailure( Event , Source );
	end else begin
		SkRoll := SkillRoll( GB , PC , Skill , SkStat , SkTar , 0 , True , True );
		if ( SkRoll >= SkTar ) then begin
			IfSuccess( Event , Source );
		end else begin
			if Source <> Nil then SetNAtt( Source^.NA , NAG_SkillCounter , Skill , SkRank );
			IfFailure( Event , Source );
		end;
	end;
end;

Procedure ProcessIfUSkillTest( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ Return TRUE if the PC makes the requested skill roll, or FALSE otherwise. }
var
	Skill,SkStat,SkTar,SkRoll: Integer;
begin
	Skill := ScriptValue( Event , GB , Source );
	SkStat := ScriptValue( Event , GB , Source );
	SkTar := ScriptValue( Event , GB , Source );
	SkRoll := SkillRoll( GB , GG_LocatePC( GB ) , Skill , SkStat , SkTar , 0 , IsSafeArea( GB ) , True );
	if SkRoll >= SkTar then begin
		IfSuccess( Event , Source );
	end else begin
		IfFailure( Event , Source );
	end;
end;

Procedure ProcessIfNoObjections( var Event: String; gb: GameBoardPtr; Source: GearPtr );
	{ Run a trigger through the narrative gears. }
	{ If none of them BLOCK it, then count the result as TRUE. }
var
	T: String;	{ The trigger to be used. }
	Adv: GearPtr;
begin
	{ Generate the trigger, which is in the same format as for COMPOSE. }
	{ It's a trigger label plus a numeric value. }
	T := ExtractWord( Event );
	T := T + BStr( ScriptValue( Event, GB, Source ) );

	{ Check the trigger along the adventure's invcoms, }
	{ where all the narrative components should be located. }
	Adv := GG_LocateAdventure( GB , Source );
	if Adv <> Nil then begin
		CheckTriggerAlongPath( T, GB, Adv^.InvCom , False );
	end;

	{ If the trigger wasn't blocked, that counts as a success. }
	if T <> '' then IfSuccess( Event , Source )
	else IfFailure( Event , Source );
end;

Procedure ProcessTeamOrders( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ This procedure is used to assign a behavior type to }
	{ every master unit on the designated team. }
const
	OrderParams: Array [0..NumAITypes] of Byte = (
		0,2,1,0,0,1
	);
var
	Team: Integer;
	OrderName: String;
	T,OrderCode: Integer;
	Mek: GearPtr;
	P: Array [1..2] of Integer;
begin
	{ Record the team number. }
	Team := ScriptValue( Event , gb , Source );

	{ Figure out what order we're supposed to be assigning. }
	OrderName := UpCase( ExtractWord( Event ) );
	OrderCode := -1;
	for t := 0 to NumAITypes do begin
		if OrderName = AI_Type_Label[ t ] then OrderCode := T;
	end;

	{ If a valid order was received, process it. }
	if OrderCode > -1 then begin
		for t := 1 to OrderParams[ OrderCode ] do P[T] := ScriptValue( Event , gb , Source );

		{ Go through each of the meks and, if they are part }
		{ of the specified team, assign the specified order. }
		Mek := gb^.Meks;
		while Mek <> Nil do begin
			if NAttValue( Mek^.NA , NAG_Location , NAS_Team ) = Team then begin
				SetNAtt( Mek^.NA , NAG_EpisodeData , NAS_Orders , OrderCode );

				{ DEFAULT BEHAVIOR- If number of params = 1, assume it to be a mek ID. }
				{ If number of params = 2, assume it to be a map location. }
				if OrderParams[ OrderCode ] = 1 then begin
					SetNAtt( Mek^.NA , NAG_EpisodeData , NAS_ATarget , P[1] );
				end else if OrderParams[ OrderCode ] = 2 then begin
					SetNAtt( Mek^.NA , NAG_Location , NAS_GX , P[1] );
					SetNAtt( Mek^.NA , NAG_Location , NAS_GY , P[2] );
				end;
			end;
			Mek := Mek^.Next;
		end;
	end;
end;

Procedure ProcessCompose( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ A new item is going to be added to the scene list. }
var
	Trigger,Ev2: String;
	P: Integer;
begin
	if Source = Nil then begin
{$IFDEF DEBUG}
		ErrorMessage_fork( 'SCRIPT ERROR: Illigal Compose. ' + Make_ErrorMessage_ASL(Source,Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT1(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT2(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT3(Event) );
{$ENDIF DEBUG}
		DialogMsg( 'SCRIPT ERROR: Illigal Compose. ' + Make_ErrorMessage_ASL(Source,Event) );
		Trigger := ExtractWord( Event );
		P := ScriptValue( Event , GB , Source );
		Ev2 := AS_GetString( Source , ExtractWord( Event ) );
		exit;
	end;

	{ Extract the values we need. }
	Trigger := ExtractWord( Event );
	P := ScriptValue( Event , GB , Source );
	Ev2 := AS_GetString( Source , ExtractWord( Event ) );

	SetSAttArray( Source^.SA , Trigger + BStr( P ) , Ev2 );
end;

Procedure ProcessNewChat( const Event: String; GB: GameBoardPtr; const Source: GearPtr );
	{ Reset the dialog menu with the standard options. }
begin
	{ Error check - make sure the interaction menu is active. }
	if IntMenu = Nil then begin
{$IFDEF DEBUG}
		ErrorMessage_fork( 'SCRIPT ERROR: Illigal NewChat. ' + Make_ErrorMessage_ASL(Source,Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT1(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT2(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT3(Event) );
{$ENDIF DEBUG}
		DialogMsg( 'SCRIPT ERROR: Illigal NewChat. ' + Make_ErrorMessage_ASL(Source,Event) );
		Exit;

	{ If there are any menu items currently in the list, get rid }
	{ of them. }
	end else if IntMenu^.FirstItem <> Nil then begin
		ClearMenu( IntMenu );
	end;

	AddRPGMenuItem( IntMenu , MsgString( 'NEWCHAT_Goodbye' ) , SELECTMENU_Cancel );
	if ( GB <> Nil ) and OnTheMap( GB , FindRoot( I_NPC ) ) and IsFoundAlongTrack( GB^.Meks , FindRoot( I_NPC ) ) then begin
		{ Only add the JOIN command if this NPC is in the same scene as the PC. }
		if ( I_PC <> Nil ) and HasTalent( I_PC , NAS_Camaraderie ) then begin
			if ( I_NPC <> Nil ) and ( NAttValue( I_NPC^.NA , NAG_Relationship , 0 ) >= NAV_Friend ) and ( NAttValue( I_NPC^.NA , NAG_Location , NAS_Team ) <> NAV_LancemateTeam ) then AddRPGMenuItem( IntMenu , MsgString( 'NEWCHAT_Join' ) , CMD_Join );
		end else begin
			if ( I_NPC <> Nil ) and ( NAttValue( I_NPC^.NA , NAG_Relationship , 0 ) >= NAV_ArchAlly ) and ( NAttValue( I_NPC^.NA , NAG_Location , NAS_Team ) <> NAV_LancemateTeam ) then AddRPGMenuItem( IntMenu , MsgString( 'NEWCHAT_Join' ) , CMD_Join );
		end;
	end;
	if ( I_NPC <> Nil ) and ( NAttValue( I_NPC^.NA , NAG_Location , NAS_Team ) = NAV_LancemateTeam ) and ( NAttValue( I_NPC^.NA , NAG_CharDescription , NAS_CharType ) <> NAV_TempLancemate ) and IsSafeArea( GB ) and IsSubCom( GB^.Scene ) then AddRPGMenuItem( IntMenu , MsgString( 'NEWCHAT_QuitLance' ) , CMD_Quit );
	if not ( OnTheMap( GB , FindRoot( I_NPC ) ) and IsFoundAlongTrack( GB^.Meks , FindRoot( I_NPC ) ) ) then AddRPGMenuItem( IntMenu , MsgString( 'NewChat_WhereAreYou' ) , CMD_WhereAreYou );
	if ( NAttValue( I_NPC^.NA , NAG_Personal , NAS_RumorRecharge ) < GB^.ComTime ) and ( NAttValue( I_NPC^.NA , NAG_Location , NAS_Team ) <> NAV_LancemateTeam ) then AddRPGMenuItem( IntMenu , MsgString( 'NEWCHAT_AskAboutRumors' ) , CMD_AskAboutRumors );
	{ RPMSortAlpha( IntMenu ); } { In UTF-8, this sort causes a result that I cannot assume. }
end;

Procedure ProcessEndChat( const Event: String; const GB: GameBoardPtr; const Source: GearPtr );
	{ End this conversation by clearing the menu. }
begin
	{ Error check - make sure the interaction menu is active. }
	if IntMenu = Nil then begin
{$IFDEF DEBUG}
		ErrorMessage_fork( 'SCRIPT ERROR: Illigal EndChat. ' + Make_ErrorMessage_ASL(Source,Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT1(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT2(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT3(Event) );
{$ENDIF DEBUG}
		DialogMsg( 'SCRIPT ERROR: Illigal EndChat. ' + Make_ErrorMessage_ASL(Source,Event) );
		Exit;
	end else begin
		ClearMenu( IntMenu );
	end;
end;

Procedure RevertPersona( var Event: String; GB: GameBoardPtr; var Source: GearPtr );
	{ We don't wanna use this persona anymore. Attempt to locate }
	{ I_NPC's original persona, and start the GREETING event. }
var
	Adv,Persona2: GearPtr;
	CID: LongInt;
	errmsg: String;
	err: Boolean;
begin
	{ Error check- if there's no defined NPC, }
	{ we can't very well be expected to locate a persona. }
	if ( I_NPC = Nil ) or ( Source = Nil ) or ( Source^.G <> GG_Persona ) then begin
{$IFDEF DEBUG}
		ErrorMessage_fork( 'SCRIPT ERROR: Illigal Persona at RevertPersona. ' + Make_ErrorMessage_ASL(Source,Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT1(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT2(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT3(Event) );
{$ENDIF DEBUG}
		DialogMsg( 'SCRIPT ERROR: Illigal Persona at RevertPersona. ' + Make_ErrorMessage_ASL(Source,Event) );
		Event := '';
		Exit;
	end;

	{ Locate the adventure and hopefully the persona. }
	Adv := GG_LocateAdventure( GB , Source );
	if Adv <> Nil then begin
		CID := NAttValue( I_NPC^.NA , NAG_Personal , NAS_CID );
		Persona2 := SeekGear( Adv , GG_Persona , CID , False );

		if ( Persona2 <> Nil ) and ( Persona2 <> Source ) then begin
			{ Change the event script to the requested line. }
			Event := AS_GetString( Persona2 , 'GREETING' );
			I_Persona := Persona2;
			Source := Persona2;
		end else begin
			{ A different persona wasn't found. Do nothing. }
			err := False;
			if ( 0 = CID ) then begin
				errmsg := 'CID not found at RevertPersona.';
				err := True;
			end else if ( NIL = Persona2 ) then begin
				errmsg := 'Persona2:' + BStr(CID) + ' not found at RevertPersona.';
			end else begin
				errmsg := '( Persona2:' + BStr(CID) + ' = Source ) at RevertPersona.';
			end;
			if err then begin
{$IFDEF DEBUG}
				ErrorMessage_fork( 'SCRIPT ERROR: ' + errmsg + ' ' + Make_ErrorMessage_ASL(Source,Event) );
				ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT1(Event) );
				ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT2(Event) );
				ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT3(Event) );
{$ENDIF DEBUG}
				DialogMsg( 'SCRIPT ERROR: ' + errmsg + ' ' + Make_ErrorMessage_ASL(Source,Event) );
				Event := 'NewChat SayAnything';
			end else begin
{$IFDEF DEBUG}
				ErrorMessage_fork( 'SCRIPT TRACE: ' + errmsg + ' ' + Make_ErrorMessage_ASL(Source,Event) );
				ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT1(Event) );
				ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT2(Event) );
				ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT3(Event) );
{$ENDIF DEBUG}
				if not ForceInteract_wo_SayAnything then begin
					Event := 'NewChat SayAnything';
				end else begin
					Event := '';
				end;
			end;
		end;
	end else begin
		{ The adventure wasn't found. Do nothing. }
{$IFDEF DEBUG}
		ErrorMessage_fork( 'SCRIPT ERROR: Adv not found at RevertPersona. ' + Make_ErrorMessage_ASL(Source,Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT1(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT2(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT3(Event) );
{$ENDIF DEBUG}
		DialogMsg( 'SCRIPT ERROR: Adv not found at RevertPersona. ' + Make_ErrorMessage_ASL(Source,Event) );
		Event := 'NewChat SayAnything';
	end;
end;

Procedure ProcessGoto( var Event: String; Source: GearPtr );
	{ Attempt to jump to a different line of the script. }
	{ If no line label is provided, or if the label can't be }
	{ found, this procedure sets EVENT to an empty string. }
var
	destination: String;
begin
	{ Error check- if there's no defined source, we can't very }
	{ well jump to another line, can we? }
	if Source = Nil then begin
{$IFDEF DEBUG}
		ErrorMessage_fork( 'SCRIPT ERROR: Illigal Goto. ' + Make_ErrorMessage_ASL(Source,Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT1(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT2(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT3(Event) );
{$ENDIF DEBUG}
		DialogMsg( 'SCRIPT ERROR: Illigal Goto. ' + Make_ErrorMessage_ASL(Source,Event) );
		Event := '';
		Exit;
	end;

	destination := ExtractWord( Event );
	if destination <> '' then begin
		{ Change the event script to the requested line. }
		Event := AS_GetString( Source , destination );
		DEBUG_cmd_org2 := Event;	{ for DEBUG }
	end else begin
		{ No label was provided. Just return a blank line. }
		Event := '';
	end;
{$IFDEF DEBUG}
	ErrorMessage_fork('SCRIPT TRACE: GOTO ' + destination + ' in "' + GearName( Source ) + '" to <' + Event + '>' );
{$ENDIF DEBUG}
end;

Procedure ProcessSeekTerr( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ Assign a value to SCRIPT_Terrain_To_Seek. }
var
	Terrain: Integer;
begin
	Terrain := ScriptValue( event , GB , Source );
	SCRIPT_Terrain_To_Seek := Terrain;
end;

Procedure CantOpenBusiness( var Event: String; GB: GameBoardPtr );
	{ The business can't be opened. Print an error message and }
	{ cancel the rest of the event. }
var
	Scene: Integer;
	msg: String;
begin
	Event := '';
	Scene := FindSceneID( I_NPC , GB );
	if Scene <> 0 then begin
		msg := ReplaceHash( MsgString( 'CantOpenShop_WithScene' ) , SceneName( GB , Scene , True ) );
	end else begin
		msg := MsgString( 'CantOpenShop' );
	end;
	msg := FormatChatStringByGender( msg , I_NPC );
	CHAT_Message := msg;
end;

Procedure ProcessShop( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ Retrieve the WARES line, then pass it all on to the OpenShop }
	{ procedure. }
var
	Wares: String;
begin
	{ Retrieve the WARES string. }
	Wares := ExtractWord( Event );
	if Wares <> '' then begin
		{ Change the event script to the requested line. }
		Wares := AS_GetString( Source , Wares );
	end;

	{ Only open the shop if the NPC is on the current map. }
	if IsFoundAlongTrack( GB^.Meks , FindRoot( I_NPC ) ) then begin
		{ Pass all info on to the OPENSHOP procedure. }
		OpenShop( GB , I_PC , I_NPC , Wares );
	end else begin
		{ Call the error handler. }
		CantOpenBusiness( Event , GB );
	end;
end;

Procedure ProcessSchool( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ Retrieve the WARES line, then pass it all on to the OpenSchool }
	{ procedure. }
var
	Wares: String;
begin
	{ Retrieve the WARES string. }
	Wares := ExtractWord( Event );
	if Wares <> '' then begin
		{ Change the event script to the requested line. }
		Wares := AS_GetString( Source , Wares );
	end;

	{ Only open the shop if the NPC is on the current map. }
	if IsFoundAlongTrack( GB^.Meks , FindRoot( I_NPC ) ) then begin
		{ Pass all info on to the OPENSHOP procedure. }
		OpenSchool( GB , I_PC , I_NPC , Wares );
	end else begin
		{ Call the error handler. }
		CantOpenBusiness( Event , GB );
	end;
end;

Procedure ProcessExpressDelivery( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ Call the ExpressDelivery procedure. }
begin
	{ Only open the shop if the NPC is on the current map. }
	if IsFoundAlongTrack( GB^.Meks , FindRoot( I_NPC ) ) then begin
		{ Pass all info on to the ExpressDelivery procedure. }
		ExpressDelivery( GB , I_PC , I_NPC );
	end else begin
		{ Call the error handler. }
		CantOpenBusiness( Event , GB );
	end;
end;

Procedure ProcessShuttle( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ Call the ExpressDelivery procedure. }
begin
	{ Only open the shop if the NPC is on the current map. }
	if IsFoundAlongTrack( GB^.Meks , FindRoot( I_NPC ) ) then begin
		{ Pass all info on to the ExpressDelivery procedure. }
		ShuttleService( GB , I_PC , I_NPC );
	end else begin
		{ Call the error handler. }
		CantOpenBusiness( Event , GB );
	end;
end;

Procedure ProcessEndPlot( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ This particular plot is over- mark it for deletion. }
	{ First, though, check to see if there are any subcomponents that }
	{ need to be moved around. }
begin
	{ If we have a valid SOURCE, attempt to end the plot. }
	if ( Source <> Nil ) then begin
		{ It's possible that our SOURCE is a PERSONA rather than }
		{ a PLOT, so if SOURCE isn't a PLOT move to its parent. }
		Source := PlotMaster( GB , Source );
		if ( Source <> Nil ) and ( Source^.G = GG_Plot ) and ( NAttValue( Source^.NA , NAG_Narrative , NAS_PlotID ) > 0 ) then begin
			EndPlot( GB , Source^.Parent , Source );
			NeedGC := True;
		end;
		SetTrigger( GB , 'UPDATE' );
	end;
end;

Procedure ProcessEndPlotsByConID( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ End all plots associated with the requested mood. }
var
	CID: LongInt;
	Adv,Plot: GearPtr;
begin
	{ Determine the ControllerID. }
	CID := ScriptValue( Event , GB , Source );

	{ Locate the adventure. }
	Adv := GG_LocateAdventure( GB , Source );

	{ If we have a valid ADV, attempt to end the attached plots. }
	if ( Adv <> Nil ) then begin
		Plot := Adv^.InvCom;

		while Plot <> Nil do begin
			if ( Plot^.G = GG_Plot ) and ( NAttValue( Plot^.NA , NAG_Narrative , NAS_PlotID ) > 0 ) and ( NAttValue( Plot^.NA , NAG_Narrative , NAS_ControllerID ) = CID ) then begin
				EndPlot( GB , Adv , Plot );
			end;

			Plot := Plot^.Next;
		end;

		{ Finally, set an UPDATE trigger. }
		SetTrigger( GB , 'UPDATE' );
	end;
end;

Procedure CleanUpStoryPlots( GB: GameBoardPtr; Story: GearPtr );
	{ Give a CLEANUP trigger to all the story plots, then move the }
	{ plots which survive to the adventure invcoms. }
var
	T: String;
	Part,P2,Adv: GearPtr;
begin
	{ Send a CLEANUP trigger to the invcoms. }
	{ This should erase all the plots that want to be erased, }
	{ and leave all the plots which want to be moved. }
	T := 'CLEANUP';
	CheckTriggerAlongPath( T , GB , Story^.InvCom , False );

	{ Check whatever is left over. }
	Part := Story^.InvCom;
	Adv := GG_LocateAdventure( GB , STory );
	while Part <> Nil do begin
		P2 := Part^.Next;

		if Part^.G = GG_Plot then begin
			DelinkGear( Story^.InvCom , Part );
			if Adv <> Nil then begin
				InsertInvCom( Adv , Part );
			end else begin
				DisposeGear( Part );
			end;
		end;

		Part := P2;
	end;
end;

Procedure ProcessEndStory( GB: GameBoardPtr; Source: GearPtr );
	{ This particular story is over- mark it for deletion. }
	{ First, though, pass a CLEANUP trigger to any subcomponents that }
	{ may need to be cleaned up. }
begin
	Source := StoryMaster( GB , Source );
	if ( Source <> Nil ) and ( Source^.G = GG_Story ) then begin
		CleanupStoryPlots( GB , Source );

		{ Mark the story for deletion. }
		Source^.G := GG_AbsolutelyNothing;
		NeedGC := True;

		SetTrigger( GB , 'UPDATE' );
	end;
end;

Procedure ProcessPurgeStory( GB: GameBoardPtr; Source: GearPtr );
	{ Eliminate all plots from this story. }
begin
	{ If we have a valid SOURCE, check the invcoms. }
	if ( Source <> Nil ) and ( Source^.G = GG_Story ) then begin
		{ Send a CLEANUP trigger to the invcoms, }
		{ then move the survivors to the Adventure. }
		CleanupStoryPlots( GB , Source );

		SetTrigger( GB , 'UPDATE' );
	end;
end;

Procedure ProcessPumpNews( const Event: String; GB: GameBoardPtr; const Source: GearPtr );
	{ Once every six hours or so, check through all the QUESTs and MOODs and }
	{ see if they've got anything new to do. PumpNews will trigger the PUMPNEWS }
	{ script of the subcoms of every city in this world. }
var
	T: String;
	world,city: GearPtr;
begin
	{ Error check }
	if ( GB = Nil ) or ( GB^.Scene = Nil ) then begin
{$IFDEF DEBUG}
		ErrorMessage_fork( 'SCRIPT ERROR: Illigal PumpNews. ' + Make_ErrorMessage_ASL(Source,Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT1(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT2(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT3(Event) );
{$ENDIF DEBUG}
		DialogMsg( 'SCRIPT ERROR: Illigal PumpNews. ' + Make_ErrorMessage_ASL(Source,Event) );
		Exit;
	end;

	{ Start by locating the world. }
	world := FindWorld( GB , GB^.Scene );
	if world = Nil then begin
{$IFDEF DEBUG}
		ErrorMessage_fork( 'SCRIPT ERROR: Illigal PumpNews, World not found. ' + Make_ErrorMessage_ASL(Source,Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT1(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT2(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT3(Event) );
{$ENDIF DEBUG}
		DialogMsg( 'SCRIPT ERROR: Illigal PumpNews, World not found. ' + Make_ErrorMessage_ASL(Source,Event) );
		Exit;
	end;

	T := 'PUMPNEWS';

	{ Go through the world's cities, springing the trigger at each. }
	city := world^.SubCom;
	while city <> Nil do begin
		if City^.G = GG_Scene then begin
			CheckTriggerAlongPath( T , GB , City^.SubCom , False );
		end;
		city := city^.Next;
	end;
end;

Procedure ProcessTReputation( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ Something has happened to affect the PC's reputation. }
	{ Record the change. }
var
	T,R,V: Integer;
begin
	{ Error check - this procedure only works if GB is defined. }
	if ( GB = Nil ) then begin
{$IFDEF DEBUG}
		ErrorMessage_fork( 'SCRIPT ERROR: Illigal TReputation. ' + Make_ErrorMessage_ASL(Source,Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT1(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT2(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT3(Event) );
{$ENDIF DEBUG}
		DialogMsg( 'SCRIPT ERROR: Illigal TReputation. ' + Make_ErrorMessage_ASL(Source,Event) );
		T := ScriptValue( Event , GB , Source );
		R := ScriptValue( Event , GB , Source );
		V := ScriptValue( Event , GB , Source );
		Exit;
	end;

	T := ScriptValue( Event , GB , Source );
	R := ScriptValue( Event , GB , Source );
	V := ScriptValue( Event , GB , Source );

	SetTeamReputation( GB , T , R , V );
end;

Procedure ProcessLoseRenown( GB: GameBoardPtr );
	{ The PC has just done something to lose face. Reduce the RENOWN attribute }
	{ by 25% of its total or 5 points, whichever is more severe. }
var
	PC: GearPtr;
	Renown: Integer;
begin
	PC := LocatePilot( GG_LocatePC( GB ) );
	if PC <> Nil then begin
		Renown := NAttValue( PC^.NA , NAG_CHarDescription , NAS_Renowned );
		if Renown > 23 then begin
			AddReputation( PC , NAS_Renowned , -( Renown div 4 ) );
		end else begin
			AddReputation( PC , NAS_Renowned , -5 );
		end;
	end;
end;

Procedure ProcessMechaPrize( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ The player has just won a mecha. Cool! }
var
	Factions,FName,msg: String;
	Renown,Theme,ModPoints: Integer;
	MList,Mek,PC,Adv: GearPtr;
begin
	{ ERROR CHECK - We need the gameboard to exist!!! }
	if GB = Nil then begin
{$IFDEF DEBUG}
		ErrorMessage_fork( 'SCRIPT ERROR: Illigal MechaPrize. ' + Make_ErrorMessage_ASL(Source,Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT1(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT2(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT3(Event) );
{$ENDIF DEBUG}
		DialogMsg( 'SCRIPT ERROR: Illigal MechaPrize. ' + Make_ErrorMessage_ASL(Source,Event) );
		Factions := ExtractWord( Event );
		Renown := ScriptValue( Event , GB , Source );
		Theme := ScriptValue( Event , GB , Source );
		ModPoints := ScriptValue( Event , GB , Source );
		Exit;
	end;

	{ Find the adventure; it'll be needed later. }
	Adv := GG_LocateAdventure( GB , Source );

	{ First, find the file name of the mecha file to look for. }
	{ Because this mecha is gonna be randomly determined we'll need some information }
	{ for that. First, determine the factions whose mecha will be considered. Second, }
	{ we need the renown level the mecha will be appropriate for. }
	Factions := ExtractWord( Event );
	if Source <> Nil then begin
		Factions := ScriptMessage( Factions , GB , Source );
	end;
	Renown := ScriptValue( Event , GB , Source );
	Theme := ScriptValue( Event , GB , Source );
	ModPoints := ScriptValue( Event , GB , Source );

	{ Call the random mecha picker. }
	FName := SelectMechaByFactionAndRenown( Factions , Renown );

	{ Attempt to load the suggested mecha. }
	MList := LoadGearPattern( FName , Design_Directory );

	{ Next confirm that something was loaded. }
	if MList <> Nil then begin
		{ Something was loaded. Yay! Pick one of the gears }
		{ at random, clone it, stick it on the game board, }
		{ and get rid of the list we loaded. }
		Mek := CloneGear( SelectRandomGear( MList ) );
		DisposeGear( MList );

		{ If modifications were requested, do those now. }
		if ModPoints > 0 then begin
			MechaMakeover( Mek , 0 , Theme , ModPoints );
		end;

		SetSAttArray( Mek^.SA , TAG_SDL_COLORS , Random_Mecha_Colors );

		SetNAtt( Mek^.NA , NAG_Location , NAS_Team , NAV_DefPlayerTeam );
		DeployGear( GB , Mek , False );

		if ( Adv <> Nil ) and ( Adv^.S = GS_ArenaCampaign ) and ( GB <> Nil ) and ( GB^.Scene <> Nil ) then begin
			{ This is Arena mode. Store the mecha announcement for the }
			{ mission debriefing. }
			AddSAttArray( GB^.Scene^.SA , ARENAREPORT_MechaObtained , FullGearName( Mek ) );

		end else begin
			{ This is RPG mode. Report the mecha directly. }
			msg := ReplaceHash( MsgString( 'MechaPrize_Announce' ) , FullGearName( Mek ) );
			AToAn( msg );
			DialogMsg( msg );

			PC := GG_LocatePC( GB );
			if FindPilotsMecha( GB^.Meks , PC ) = Nil then AssociatePilotMek( GB^.Meks , PC , Mek );
		end;
	end;
end;

Procedure ProcessDeleteGG( const Event: String; GB: GameBoardPtr; var Source: GearPtr );
	{ Delete the grabbed gear. }
	{ Only physical gears can be deleted in this way. }
begin
	if ( Grabbed_Gear <> Nil ) and (( Grabbed_Gear^.G >= 0 ) or ( Grabbed_Gear^.G = GG_CityMood )) then begin
		{ Make sure we aren't currently using the grabbed gear. }
		if ( IntMenu <> Nil ) and ( I_NPC = Grabbed_Gear ) then begin
			ProcessEndChat( Event , GB , Source );
			I_NPC := Nil;
		end;
		if Source = Grabbed_Gear then begin
			Source := Nil;
		end;

		{ Delete the gear, if it can be found. }
		if IsSubCom( Grabbed_Gear ) then begin
			RemoveGear( Grabbed_Gear^.Parent^.SubCom , Grabbed_Gear );

		end else if IsInvCom( Grabbed_Gear ) then begin
			RemoveGear( Grabbed_Gear^.Parent^.InvCom , Grabbed_Gear );

		end else if ( GB <> Nil ) and IsFoundAlongTrack( GB^.Meks , Grabbed_Gear) then begin
			RemoveGear( GB^.Meks , Grabbed_Gear );

		end;
	end else begin
{$IFDEF DEBUG}
		ErrorMessage_fork( 'SCRIPT ERROR: Illigal DeleteGG. ' + Make_ErrorMessage_ASL(Source,Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT1(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT2(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT3(Event) );
{$ENDIF DEBUG}
		DialogMsg( 'SCRIPT ERROR: Illigal DeleteGG. ' + Make_ErrorMessage_ASL(Source,Event) );
	end;
end;

Procedure ProcessMoveGG( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ Move the grabbed gear to the specified scene. }
	{ Only physical gears can be moved in this way. }
	{ If the specified scene is 0, the gear will be "frozen" isntead. }
var
	SID: Integer;	{ Scene ID. }
	Scene: GearPtr;
begin
	{ Check to make sure we have a valid gear to move. }
	if ( Grabbed_Gear <> Nil ) and ( Grabbed_Gear^.G >= 0 ) then begin
		DelinkGearForMovement( GB , Grabbed_Gear );

		{ Find the new scene to stick our gear into. }
		SID := ScriptValue( Event , GB , Source );

		if SID <> 0 then begin
			Scene := FindActualScene( GB , SID );
			if Scene = Nil then Scene := GG_LocateAdventure( GB , Source );
		end else begin
			Scene := GG_LocateAdventure( GB , Source );
		end;
		InsertInvCom( Scene , Grabbed_Gear );

		{ If inserting a character, better choose a team. }
		if IsAScene( Scene ) and IsMasterGear( Grabbed_Gear ) then begin
			ChooseTeam( Grabbed_Gear , Scene );
		end;
	end else begin
		SID := ScriptValue( Event , GB , Source );
{$IFDEF DEBUG}
		ErrorMessage_fork( 'SCRIPT ERROR: Illigal MoveGG. ' + Make_ErrorMessage_ASL(Source,Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT1(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT2(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT3(Event) );
{$ENDIF DEBUG}
		DialogMsg( 'SCRIPT ERROR: Illigal MoveGG. ' + Make_ErrorMessage_ASL(Source,Event) );
	end;
end;

Procedure ProcessMoveAndPacifyGG( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ Move the grabbed gear to the specified scene, }
	{ setting its team to a nonagressive one. }
	{ Only physical gears can be moved in this way. }
	{ If the specified scene is 0, the gear will be "frozen" isntead. }
var
	SID: Integer;	{ Scene ID. }
	Scene: GearPtr;
begin
	{ Check to make sure we have a valid gear to move. }
	if ( Grabbed_Gear <> Nil ) and ( Grabbed_Gear^.G >= 0 ) then begin
		DelinkGearForMovement( GB , Grabbed_Gear );

		{ Find the new scene to stick our gear into. }
		SID := ScriptValue( Event , GB , Source );
		if SID <> 0 then begin
			Scene := FindActualScene( GB , SID );
			if Scene = Nil then Scene := GG_LocateAdventure( GB , Source );
		end else begin
			Scene := GG_LocateAdventure( GB , Source );
		end;
		InsertInvCom( Scene , Grabbed_Gear );

		{ Set the TEAMDATA here. }
		if IsACombatant( Grabbed_Gear ) then begin
			SetSAttArray( Grabbed_Gear^.SA , 'TEAMDATA' , 'SD ALLY' );
		end else begin
			SetSAttArray( Grabbed_Gear^.SA , 'TEAMDATA' , 'PASS ALLY' );
		end;

		{ If inserting a character, better choose a team. }
		if IsAScene( Scene ) and IsMasterGear( Grabbed_Gear ) then begin
			ChooseTeam( Grabbed_Gear , Scene );
		end;
	end else begin
		SID := ScriptValue( Event , GB , Source );
{$IFDEF DEBUG}
		ErrorMessage_fork( 'SCRIPT ERROR: Illigal MoveAndPacifyGG. ' + Make_ErrorMessage_ASL(Source,Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT1(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT2(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT3(Event) );
{$ENDIF DEBUG}
		DialogMsg( 'SCRIPT ERROR: Illigal MoveAndPacifyGG. ' + Make_ErrorMessage_ASL(Source,Event) );
	end;
end;

Procedure ProcessDeployGG( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ Move the grabbed gear to the current scene. }
	{ Only physical gears can be moved in this way. }
var
	TID: Integer;
begin
	{ Check to make sure we have a valid gear to move. }
	if ( Grabbed_Gear <> Nil ) and ( GB <> Nil ) and ( Grabbed_Gear^.G >= 0 ) then begin
		DelinkGearForMovement( GB , Grabbed_Gear );

		{ Find the new team for our gear. }
		TID := ScriptValue( Event , GB , Source );
		SetNAtt( Grabbed_Gear^.NA , NAG_Location , NAS_Team , TID );

		{ Stick it on the map, and maybe do a redraw. }
		EquipThenDeploy( GB , Grabbed_Gear , True );
	end else begin
		TID := ScriptValue( Event , GB , Source );
{$IFDEF DEBUG}
		ErrorMessage_fork( 'SCRIPT ERROR: Illigal DeployGG. ' + Make_ErrorMessage_ASL(Source,Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT1(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT2(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT3(Event) );
{$ENDIF DEBUG}
		DialogMsg( 'SCRIPT ERROR: Illigal DeployGG. ' + Make_ErrorMessage_ASL(Source,Event) );
	end;
end;

Procedure ProcessDynaGG( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ Move the grabbed gear to the dynamic scene. }
	{ Only physical gears can be moved in this way. }
var
	TID: Integer;	{ Team ID. }
begin
	{ Check to make sure we have a valid gear to move. }
	if ( Grabbed_Gear <> Nil ) and ( Grabbed_Gear^.G >= 0 ) and ( SCRIPT_DynamicEncounter <> Nil ) then begin
		DelinkGearForMovement( GB , Grabbed_Gear );

		{ Find out which team to stick the NPC in. }
		TID := ScriptValue( Event , GB , Source );
		SetNAtt( Grabbed_Gear^.NA , NAG_Location , NAS_Team , TID );

		{ Perform the insertion. }
		InsertInvCom( SCRIPT_DynamicEncounter , Grabbed_Gear );
	end else begin
		TID := ScriptValue( Event , GB , Source );
{$IFDEF DEBUG}
		ErrorMessage_fork( 'SCRIPT ERROR: Illigal DynaGG. ' + Make_ErrorMessage_ASL(Source,Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT1(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT2(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT3(Event) );
{$ENDIF DEBUG}
		DialogMsg( 'SCRIPT ERROR: Illigal DynaGG. ' + Make_ErrorMessage_ASL(Source,Event) );
	end;
end;

Procedure ProcessGiveGG( const Event: String; GB: GameBoardPtr; const Source: GearPtr );
	{ Give the grabbed gear to the PC. }
	{ Only physical gears can be moved in this way. }
var
	DelinkOK: Boolean;
	PC: GearPtr;
begin
	PC := GG_LocatePC( GB );

	if ( Grabbed_Gear <> Nil ) and ( Grabbed_Gear^.G >= 0 ) and (( PC = Nil ) or ( FindGearIndex( Grabbed_Gear , PC ) < 0 )) then begin

		{ Delink the gear, if it can be found. }
		if IsSubCom( Grabbed_Gear ) then begin
			DelinkGear( Grabbed_Gear^.Parent^.SubCom , Grabbed_Gear );
			DelinkOK := True;
		end else if IsInvCom( Grabbed_Gear ) then begin
			DelinkGear( Grabbed_Gear^.Parent^.InvCom , Grabbed_Gear );
			DelinkOK := True;
		end else if ( GB <> Nil ) and IsFoundAlongTrack( GB^.Meks , Grabbed_Gear) then begin
			DelinkGear( GB^.Meks , Grabbed_Gear );
			DelinkOK := True;
		end else begin
			DelinkOK := False;
		end;

		if DelinkOK then begin
			GivePartToPC( GB , Grabbed_Gear , PC );
		end;
	end else begin
{$IFDEF DEBUG}
		ErrorMessage_fork( 'SCRIPT ERROR: Illigal GiveGG. ' + Make_ErrorMessage_ASL(Source,Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT1(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT2(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT3(Event) );
{$ENDIF DEBUG}
		DialogMsg( 'SCRIPT ERROR: Illigal GiveGG. ' + Make_ErrorMessage_ASL(Source,Event) );
	end;
end;

Procedure ProcessGNewPart( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ Stick an item from the standard items list on the gameboard, }
	{ then make GRABBED_GEAR point to it. }
	{ This function will first look in the STC file, then the Monster }
	{ file, then the NPC file. }
var
	IName: String;
	Msg: String;
begin
	{ First determine the item's designation. }
	IName := ExtractWord( Event );
	if Source <> Nil then begin
		IName := AS_GetString( Source , IName );
	end;

	{ As long as we have a GB, try to stick the item there. }
	if GB <> Nil then begin
		Grabbed_Gear := LoadNewSTC_withoutErrorCheck( IName );
		if Grabbed_Gear = Nil then Grabbed_Gear := LoadNewMonster_withoutErrorCheck( IName );
		if Grabbed_Gear = Nil then Grabbed_Gear := LoadNewNPC_withoutErrorCheck( IName , True );
		if Grabbed_Gear = Nil then Grabbed_Gear := LoadNewItem_withoutErrorCheck( IName );

		{ If we found something, stick it on the map. }
		if Grabbed_Gear <> Nil then begin
			{ Clear the designation. }
			SetSAttArray( Grabbed_Gear^.SA , 'DESIG' , '' );
			SetSAttArray( Grabbed_Gear^.SA , 'DESIG_I18N' , '' );
			SetSAttArray( Grabbed_Gear^.SA , 'DESIG_REDESIG' , '' );

			{ Deploy the item. }
			EquipThenDeploy( GB , Grabbed_Gear , False );
		end else begin
			Msg := 'ERROR: ProcessGNewPart failed : ' + IName;
			ErrorMessage_fork( Msg );
			DialogMsg( Msg );
{$IFDEF DEBUG}
			ErrorMessage_fork( 'SCRIPT ERROR: Illigal GNewPart. ' + Make_ErrorMessage_ASL(Source,Event) );
			ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT1(Event) );
			ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT2(Event) );
			ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT3(Event) );
{$ENDIF DEBUG}
			DialogMsg( 'SCRIPT ERROR: Illigal GNewPart. ' + Make_ErrorMessage_ASL(Source,Event) );
		end;

		{ Any further processing must be done by other commands. }
	end;
end;

Procedure BuildGenericEncounter( GB: GameBoardPtr; Source: GearPtr; Scale: Integer );
	{ Create a SCENE gear, then do everything except stock it with }
	{ enemies. }
const
	DefaultMapSize = 50;
var
	Team,Src: GearPtr;
	T: Integer;
begin
	{ First, if for some reason there's already a dynamic encounter in }
	{ place, get rid of it. }
	if SCRIPT_DynamicEncounter <> Nil then DisposeGear( SCRIPT_DynamicEncounter );

	{ Allocate a new dynamic encounter, then fill in the blanks. }
	SCRIPT_DynamicEncounter := NewGear( Nil );
	SCRIPT_DynamicEncounter^.G := GG_Scene;
	SCRIPT_DynamicEncounter^.Stat[ STAT_MapWidth ] := DefaultMapSize;
	SCRIPT_DynamicEncounter^.Stat[ STAT_MapHeight ] := DefaultMapSize;
	if ( GB <> Nil ) and ( GB^.Scene <> Nil ) then SCRIPT_DynamicEncounter^.S := GB^.Scene^.S;
	SCRIPT_DynamicEncounter^.V := Scale;

	{ Copy over the PlotID of the source. }
	Src := PlotMaster( GB , Source );
	if Src <> Nil then begin
		SetNAtt( SCRIPT_DynamicEncounter^.NA , NAG_Narrative , NAS_PlotID , NAttValue( Src^.NA , NAG_Narrative , NAS_PlotID ) );
	end;

	{ Add a TEAM gear for each of the player and the enemy teams. }
	{ We need to do this so that we'll have some control over the placement }
	{ of the player and the enemies. }
	Team := AddGear( SCRIPT_DynamicEncounter^.SubCom , SCRIPT_DynamicEncounter );
	Team^.G := GG_Team;
	Team^.S := NAV_DefPlayerTeam;
	SetNAtt( Team^.NA , NAG_SideReaction , NAV_DefEnemyTeam , NAV_AreEnemies );
	SetNAtt( Team^.NA , NAG_ParaLocation , NAS_X , DefaultMapSize div 5 );
	SetNAtt( Team^.NA , NAG_ParaLocation , NAS_Y , DefaultMapSize div 5 );

	Team := AddGear( SCRIPT_DynamicEncounter^.SubCom , SCRIPT_DynamicEncounter );
	Team^.G := GG_Team;
	Team^.S := NAV_DefEnemyTeam;
	SetNAtt( Team^.NA , NAG_SideReaction , NAV_DefPlayerTeam , NAV_AreEnemies );
	SetNAtt( Team^.NA , NAG_ParaLocation , NAS_X , ( DefaultMapSize * 4 ) div 5 );
	SetNAtt( Team^.NA , NAG_ParaLocation , NAS_Y , ( DefaultMapSize * 4 ) div 5 );
	SetSAttArray( Team^.SA , 'Deploy' , 'SetSelfFaction CurrentSceneFac   WMecha 2 DynaRenown DynaStrength' );

	{ Set the default map generator of the Dynamic Encounter based on the position of }
	{ the PC. }
	Src := FindRoot( GG_LocatePC( GB ) );
	if ( Src <> Nil ) then begin
		SCRIPT_DynamicEncounter^.Stat[ STAT_MapGenerator ] := EncounterMapType( GB , NAttValue( Src^.NA , NAG_Location , NAS_X ) , NAttValue( Src^.NA , NAG_Location , NAS_Y ) );
		{ If this will make the encounter a space map, set the map-scroll tag. }
		if SCRIPT_DynamicEncounter^.Stat[ STAT_MapGenerator ] = TERRAIN_Space then SCRIPT_DynamicEncounter^.Stat[ STAT_SpaceMap ] := 1;
	end;

	{ If this metascene doesn't have environmental effects by default, }
	{ copy the environmental effects from the parent scene. }
	Src := FindActualScene( GB , FindGearScene( Src , GB ) );
	if Src <> Nil then begin
		{ Copy the environmental effects from the parent scene. }
		for t := 1 to Num_Environment_Variables do begin
			SetNAtt( SCRIPT_DynamicEncounter^.NA , NAG_EnvironmentData , T , NAttValue( Src^.NA , NAG_EnvironmentData , T ) );
		end;

		{ Also copy over the tileset + backdrop. }
		SetNAtt( SCRIPT_DynamicEncounter^.NA , NAG_SceneData , NAS_TileSet , NAttValue( Src^.NA , NAG_SceneData , NAS_TileSet ) );
		SetNAtt( SCRIPT_DynamicEncounter^.NA , NAG_SceneData , NAS_Backdrop , NAttValue( Src^.NA , NAG_SceneData , NAS_Backdrop ) );
	end;

	{ Set the exit values in the game board. }
	if GB <> Nil then begin
		AS_SetExit( GB , 0 );
	end;
end;

Procedure ProcessNewD( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ Create a new scene for a dynamic encounter to take place on. }
var
	Scale: Integer;
begin
 	Scale := ScriptValue( Event , GB , Source );
	BuildGenericEncounter( GB , Source , Scale );
end;

Procedure ProcessWMecha( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ Fill current scene with enemies. }
var
	TID,Renown,Strength,LMs: LongInt;
begin
	{ Find out the team, and how many enemies to add. }
	TID := ScriptValue( Event , GB , Source );
 	Renown := ScriptValue( Event , GB , Source );
	Strength := ScriptValue( Event , GB , Source );

	{ If this team is an enemy of the player team, add extra STRENGTH based on the number of }
	{ lancemates. This extra STRENGTH will not entirely cancel out the usefulness of lancemates, }
	{ but will reduce it slightly, thereby allowing solo PCs to exist. }
	if AreEnemies( GB , TID , NAV_DefPlayerTeam ) then begin
		LMs := LancematesPresent( GB );
		if LMs > 0 then Strength := Strength + ( 25 * LMs );
	end;

	AddTeamForces( GB , TID , Renown , Strength );
end; { ProcessWMecha }

Procedure ProcessWMonster( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ Fill current scene with monsters. }
var
	TID,Renown,Strength: LongInt;
	Team: GearPtr;
	MonType: String;
begin
	{ Find out the team, and how many enemies to add. }
	TID := ScriptValue( Event , GB , Source );
	Renown := ScriptValue( Event , GB , Source );
	Strength := ScriptValue( Event , GB , Source );

	{ Determine the type of monster to add. This should be indicated by }
	{ the team to which the monsters belong. }
	Team := LocateTeam( GB , TID );
	if Team <> Nil then MonType := SAttArrayValue( Team^.SA , 'TYPE' )
	else MonType := 'ROBOT NINJA';
	if MonType = '' then MonType := 'ROBOT NINJA';

	StockBoardWithMonsters( GB , Renown , Strength , TID , MonType );
end; { ProcessWMonster }

Function NumberOfPlots( Part: GearPtr ): Integer;
	{ Check the number of plots this PART has loaded. }
var
	P: GearPtr;
	N: Integer;
begin
	P := Part^.InvCom;
	N := 0;

	while P <> Nil do begin
		if P^.G = GG_Plot then Inc( N );
		P := P^.Next;
	end;

	NumberOfPlots := N;
end;

Function CurrentPCRenown( GB: GameBoardPtr ): Integer;
	{ Return the current renown score of the PC. }
var
	PC: GearPtr;
begin
	PC := LocatePilot( GG_LocatePC( GB ) );
	if PC <> Nil then begin
		CurrentPCRenown := NAttValue( PC^.NA , NAG_CharDescription , NAS_Renowned );
	end else begin
		CurrentPCRenown := 0;
	end;
end;

Function AS_ContentInsertion( GB: GameBoardPtr; Source: GearPtr; ConReq: String; Threat: Integer ): Boolean;
	{ Attempt to load and initialize the requested plot or story. }
var
	Adv,Slot,Plot: GearPtr;
	EList: ElementTable;
	Context,msg: String;
	T,ENum: Integer;
begin
	ClearElementTable( EList );
	Adv := GG_LocateAdventure( GB , Source );
	Context := ExtractWord( ConReq );

	{ Fill out the element table. }
	if Source^.G = GG_Faction then begin
		EList[1].EValue := Source^.S;
		EList[1].EType := 'F';
		Slot := Adv;
	end else if Source^.G = GG_Scene then begin
		EList[1].EValue := Source^.S;
		EList[1].EType := 'S';
		Slot := Adv;
	end else if Source^.G = GG_Story then begin
		Slot := Source;
	end else begin
		Slot := Adv;
	end;

	{ If this plot is being spawned by a plot or by a quest, we may need to }
	{ pass along some elements. }
	Plot := PlotMaster( GB , Source );
	if Plot <> Nil then begin
		T := 1;
		while ConReq <> '' do begin
			ENum := ExtractValue( ConReq );
			if ( ENum >= 0 ) and ( ENum <= Num_Plot_Elements ) then begin
				msg := SAttArrayValue( Plot^.SA , 'ELEMENT' + BSTr( ENum ) );
				if msg <> '' then begin
					EList[ T ].EType := msg[1];
					EList[ T ].EValue := ElementID( Plot , ENum );
					if EList[ T ].EType = 'C' then begin
{$IFDEF DEBUG}
						ErrorMessage_fork( 'SCRIPT ERROR: Content Insertion ' + Context + ' passed character element.' );
{$ENDIF DEBUG}
						DialogMsg( 'SCRIPT ERROR: Content Insertion ' + Context + ' passed character element.' )
					end else if EList[ T ].EType = 'M' then begin
{$IFDEF DEBUG}
						ErrorMessage_fork( 'SCRIPT ERROR: Content Insertion ' + Context + ' passed metascene element.' );
{$ENDIF DEBUG}
						DialogMsg( 'SCRIPT ERROR: Content Insertion ' + Context + ' passed metascene element.' );
					end;
				end else begin
{$IFDEF DEBUG}
					ErrorMessage_fork( 'SCRIPT ERROR: Content Insertion ' + Context + ' requested invalid element.' );
{$ENDIF DEBUG}
					DialogMsg( 'SCRIPT ERROR: Content Insertion ' + Context + ' requested invalid element.' );
				end;
			end;
			Inc( T );
		end;
	end;

	if AddRandomPlot( GB , Slot , Context , EList , Threat ) then begin
		AS_ContentInsertion := True;
	end else begin
		AS_ContentInsertion := False;
	end;
end;

Procedure ProcessStartStory( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ A new story gear is about to be loaded. }
var
	FName: String;
	Renown: Integer;
begin
	{ First, find the content request and the threat value to use. }
	FName := ExtractWord( Event );
	if Source <> Nil then begin
		FName := AS_GetString( Source , FName );
	end else begin
		FName := '';
	end;
	Renown := ScriptValue( Event , GB , Source );

	{ Call the above procedure to see if it works or not. }
	if AS_ContentInsertion( GB , Source , FName , Renown ) then begin
		SetTrigger( GB , 'UPDATE' );
		IfSuccess( Event , Source );
	end else begin
		IfFailure( Event , Source );
	end;
end;

Procedure ProcessStartPlot( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ A new plot is being loaded, probably by a story. }
	{ For now I'm going to treat such plots as global. }
var
	FName: String;
	Renown: Integer;
begin
	{ First, find the content request and the threat value to use. }
	FName := ExtractWord( Event );
	if Source <> Nil then begin
		FName := AS_GetString( Source , FName );
	end else begin
		FName := '';
	end;
	Renown := ScriptValue( Event , GB , Source );

	if ( Source <> Nil ) and ( Source^.G = GG_Story ) and ( NumberOfPlots( Source ) >= Max_Plots_Per_Story ) then begin
		{ Can't load a new plot at this time. }
		IfFailure( Event , Source );

	end else begin
		if AS_ContentInsertion( GB , Source , FName , Renown ) then begin
			SetTrigger( GB , 'UPDATE' );
			IfSuccess( Event , Source );
		end else begin
			{ File was not loaded successfully. }
			IfFailure( Event , Source );
		end;
	end;
end;

Procedure ProcessSetMood( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ A CityMood is about to be unleashed on the game world. }
var
	ID: LongInt;
	Mood,City: GearPtr;
begin
	{ First, find the mood and the city. }
	ID := ScriptValue( Event , GB , Source );
	Mood := GG_LocateItem( ID , GB , Source );

	ID := ScriptValue( Event , GB , Source );
	City := FindRootScene( FindActualScene( GB , ID ) );

	{ Call the above procedure to see if it works or not. }
	if ( Mood <> Nil ) and ( City <> Nil ) then begin
		DelinkGearForMovement( GB , Mood );
		if InsertMood( City , Mood , GB ) then begin
			SetTrigger( GB , 'UPDATE' );
			IfSuccess( Event , Source );
		end else begin
			IfFailure( Event , Source );
		end;
	end else begin
		IfFailure( Event , Source );
	end;
end;

Procedure ProcessUpdatePlots( var Trigger,Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ Check the current city and all of its moods. Attempt to load new plots as appropriate. }
var
	Threat: Integer;
begin
	{ Final error check- based on the config options, maybe exit. }
	if Load_Plots_At_Start and ( UpCase( Trigger ) <> 'START' ) then exit
	else if ( not Load_Plots_At_Start ) and ( UpCase( Trigger ) = 'START' ) then exit;

	Threat := CurrentPCRenown( GB );
	UpdatePlots( GB , Threat );

	SetTrigger( GB , 'UPDATE' );
end;

Procedure ProcessCheckComponents( GB: GameBoardPtr; Source: GearPtr );
	{ Check to see if this source needs to load a new component. If so, }
	{ then do that. }
var
	Adv: GearPtr;
begin
	{ If we have a source, and that source has its "LoadNextComponent" value set, }
	{ then we have work to do here... }
	if ( Source <> Nil ) and ( NAttValue( Source^.NA , NAG_XXRAN , NAS_LoadNextComponent ) = 0 ) then begin
		PrepareNewComponent( Source , GB );

		SetNAtt( Source^.NA , NAG_XXRAN , NAS_LoadNextComponent , 1 );

		{ Check to see whether or not the episode number has increased. }
		if NAttValue( Source^.NA , NAG_XXRAN , NAS_EpisodeNumber ) > NAttValue( Source^.NA , NAG_XXRAN , NAS_LastEpisodeNoted ) then begin
			SetNAtt( Source^.NA , NAG_XXRAN , NAS_LastEpisodeNoted , NAttValue( Source^.NA , NAG_XXRAN , NAS_EpisodeNumber ) );
			Adv := GG_LocateAdventure( GB , Source );
			AddSAttArray( Adv^.SA , 'HISTORY' , '...' );
			AddSAttArray( Adv^.SA , 'HISTORY' , ReplaceHash( MsgString( 'HISTORY_NEWEPISODE' ) , BStr( NAttValue( Source^.NA , NAG_XXRAN , NAS_EpisodeNumber ) ) ) );
		end;
	end;
end;

Procedure AlterStoryDescriptors( Story: GearPtr; var Changes: String );
	{ Alter the story descriptors based on the provided changes. If the +P Propp state is changed, }
	{ reset the story's ProppAdvancement attribute. }
	{ Note that CHANGES will be utterly destroyed by the AlterDescriptors call below. }
var
	Base: String;
begin
	Base := SAttArrayValue( Story^.SA , 'CONTEXT' );
	AlterDescriptors( Base , Changes );
	SetSAttArray( Story^.SA , 'CONTEXT' , Base );
end;

Procedure ProcessGAlterContext( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ Alter the context by the changes string provided. }
var
	Story: GearPtr;
	Changes: String;
begin
	Story := Grabbed_Gear;
	Changes := ExtractWord( Event );
	Changes := SAttArrayValue( Source^.SA , Changes );
	if Story <> Nil then begin
		AlterStoryDescriptors( Story , Changes );
	end else begin
{$IFDEF DEBUG}
		ErrorMessage_fork( 'SCRIPT ERROR: Illigal GAlterContext. ' + Make_ErrorMessage_ASL(Source,Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT1(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT2(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT3(Event) );
{$ENDIF DEBUG}
		DialogMsg( 'SCRIPT ERROR: Illigal GAlterContext. ' + Make_ErrorMessage_ASL(Source,Event) );
	end;
end;

Procedure ProcessNextComp( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ Prepare things for the next component to be loaded. }
	Procedure ContinueConversation();
		{ We want to make a choice, but there's already a conversation }
		{ going on. Best to end that conversation with a [Continue] tag, }
		{ then we can return to it later. }
	var
		RPM: RPGMenuPtr;
	begin
		RPM := CreateRPGMenu( MenuItem , MenuSelect , @ZONE_InteractMenu );
		AddRPGMenuItem( RPM , MsgString( 'Continue' ), SELECTMENU_Cancel );
		ASRD_GameBoard := GB;
		CHAT_React := ReactionScore( GB^.Scene , I_PC , I_NPC );
		SelectMenu( RPM , @InteractRedraw );
		DisposeRPGMenu( RPM );
	end;
	Function SeekDCGear( Story: GearPtr; DCID: Integer ): GearPtr;
		{ Locate the plot corresponding to the dramatic choice selected }
		{ by the PC. }
	var
		LList: GearPtr;
	begin
		LList := Story^.InvCom;
		while ( LList <> Nil ) and ( ( NAttValue( LList^.NA , NAG_XXRan , NAS_IsDramaticChoicePlot ) = 0 ) or ( LList^.V <> DCID ) ) do LList := LList^.Next;
		SeekDCGear := LList;
	end;
	Procedure MakeDramaticChoice( Story: GearPtr );
		{ In order to move on, the PC must decide what they're going }
		{ to do next. }
	var
		RPM: RPGMenuPtr;
		DC: GearPtr;
		N: Integer;
		Trigger: String;
		msg: String;
	begin
		{ Step One- create the list of dramatic choices. }
		CreateChoiceList( GB , Story );

		{ Step Two- assemble these choices into a menu. This is tougher }
		{ than it looks, since if we are currently in a conversation }
		{ we want the choices merged with the conversation. }
		if I_NPC <> Nil then begin
			ContinueConversation;
			RPM := CreateRPGMenu( MenuItem , MenuSelect , @ZONE_InteractMenu );
		end else begin
			RPM := CreateRPGMenu( MenuItem , MenuSelect , @ZONE_MemoText );
		end;
		RPM^.Mode := RPMNoCancel;

		DC := Story^.InvCom;
		while DC <> Nil do begin
			{ How do we identify the dramatic choice options? They've been marked }
			{ with a special NA tag. Do we do any error checking to make sure }
			{ they're legit? Hell no! }
			if NAttValue( DC^.NA , NAG_XXRan , NAS_IsDramaticChoicePlot ) <> 0 then begin
				{ This is a dramatic choice. Add it to the menu. }
				msg := ScriptMessage( 'PROMPT' , GB , DC );
				msg := FormatChatStringByGender( msg, I_PC );
				AddRPGMenuItem( RPM , msg , DC^.V );
			end;
			DC := DC^.Next;
		end;

		if RPM^.NumItem < 1 then begin
{$IFDEF DEBUG}
			ErrorMessage_fork( 'SCRIPT ERROR: No dramatic choices found!' );
{$ENDIF DEBUG}
			DialogMsg( 'SCRIPT ERROR: No dramatic choices found!' );
			N := SELECTMENU_Cancel;
		end else if I_NPC <> Nil then begin
			CHAT_Message := MsgString( 'Make_Dramatic_Choice' );
			N := SelectMenu( RPM , @InteractRedraw );
		end else begin
			ASRD_MemoMessage := MsgString( 'Make_Dramatic_Choice' );
			N := SelectMenu( RPM , @ChoiceRedraw );
		end;

		{ Step Three- record the choice and execute its initialization }
		{ script. }
		SetNAtt( Story^.NA , NAG_XXRan , NAS_DramaticChoice , N );

		{ Locate the choice's gear. }
		DC := SeekDCGear( Story , N );
		if DC <> Nil then begin
			Trigger := 'CHOICE';
			TriggerGearScript( GB , DC , Trigger );
			if I_NPC <> Nil then begin
				CHAT_Message := ScriptMessage( 'REPLY' , GB , DC );
				CHAT_Message := FormatChatStringByGender( CHAT_Message, I_NPC );
			end else begin
				YesNoMenu( GB , ScriptMessage( 'ALERT' , GB , DC ) , '' , '' );
			end;
		end else begin
{$IFDEF DEBUG}
			ErrorMessage_fork( 'SCRIPT ERROR: Dramatic Choice ' + BStr( N ) + ' not found.' );
{$ENDIF DEBUG}
			DialogMsg( 'SCRIPT ERROR: Dramatic Choice ' + BStr( N ) + ' not found.' );
		end;

		{ Step Four- dispose of the remaining choices, and the menu. }
		ClearChoiceList( Story );
		DisposeRPGMenu( RPM );
	end;
var
	Story: GearPtr;
begin
	Story := StoryMaster( GB , Source );
	if Story = Nil then begin
{$IFDEF DEBUG}
		ErrorMessage_fork( 'SCRIPT ERROR: ProcessNextComp called but story not found. ' + Make_ErrorMessage_ASL(Source,Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT1(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT2(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT3(Event) );
{$ENDIF DEBUG}
		DialogMsg( 'SCRIPT ERROR: ProcessNextComp called but story not found' );
		Exit;
	end;

	if ( Story <> Nil ) and ( GB <> Nil ) then begin
		MakeDramaticChoice( Story );

		SetNAtt( Story^.NA , NAG_XXRAN , NAS_LoadNextComponent , 0 );

		{ Set this component for possible deletion. }
		{ First make sure we have the plot itself. }
		Source := PlotMaster( GB , Source );
		EndPlot( GB , Source^.Parent , Source );
		SetTrigger( GB , 'UPDATE' );
	end;
end;


Procedure ProcessAttack( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ Team 1 is going to attack Team 2. }
var
	t1,T2: Integer;
	Team1,Mek: GearPtr;
begin
	{ Error check - We need that gameboard! }
	if GB = Nil then begin
{$IFDEF DEBUG}
		ErrorMessage_fork( 'SCRIPT ERROR: Illigal Attack. ' + Make_ErrorMessage_ASL(Source,Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT1(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT2(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT3(Event) );
{$ENDIF DEBUG}
		DialogMsg( 'SCRIPT ERROR: Illigal Attack. ' + Make_ErrorMessage_ASL(Source,Event) );
		T1 := ScriptValue( Event , GB , Source );
		T2 := ScriptValue( Event , GB , Source );
		Exit;
	end;

	{ Read the script values. }
	T1 := ScriptValue( Event , GB , Source );
	T2 := ScriptValue( Event , GB , Source );

	{ Find the attacking team, and set the enemy value. }
	Team1 := LocateTeam( GB , T1 );
	if Team1 <> Nil then begin
		SetNAtt( Team1^.NA , NAG_SideReaction , T2 , NAV_AreEnemies );
	end;

	{ Locate each member of the team and set AIType to SEEK AND DESTROY. }
	Mek := GB^.Meks;
	while Mek <> Nil do begin
		if NAttValue( Mek^.NA , NAG_Location , NAS_Team ) = T1 then begin
			SetNAtt( Mek^.NA , NAG_EpisodeData , NAS_Orders , NAV_SeekAndDestroy );
		end;
		Mek := Mek^.Next;
	end;
end;

Procedure ProcessSalvage( const Event: String; GB: GameBoardPtr; const Source: GearPtr );
	{ It's looting time!!! Check every mecha on the game board; if it's }
	{ not operational but not destroyed, switch its TEAM to NAV_DefPlayerTeam. }
	Function CanSalvage( M: GearPtr ): Boolean;
		{ Return TRUE if M can maybe be salvaged, or FALSE otherwise. }
	var
		Team: Integer;
	begin
		Team := NAttValue( M^.NA , NAG_Location, NAS_Team );
		CanSalvage := ( M^.G = GG_Mecha ) and ( Team <> NAV_DefPlayerTeam ) and ( Team <> NAV_LancemateTeam ) and ( not GearOperational( M ) );
	end;
var
	Mek,M2,PC: GearPtr;
	CanScavenge: Boolean;
	T: Integer;
	RPts: LongInt;
begin
	if SHOW_Pillage then begin
		DialogMsg( UTF8_MsgString('ProcessSalvage', 'You Salvaged') );
	end;

	{ ERROR CHECK - GB must be defined!!! }
	if GB = Nil then begin
{$IFDEF DEBUG}
		ErrorMessage_fork( 'SCRIPT ERROR: Illigal Salvage. ' + Make_ErrorMessage_ASL(Source,Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT1(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT2(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT3(Event) );
{$ENDIF DEBUG}
		DialogMsg( 'SCRIPT ERROR: Illigal Salvage. ' + Make_ErrorMessage_ASL(Source,Event) );
		Exit;
	end;

	{ Check to see if the PC has the Scavenger talent. }
	PC := GG_LocatePC( GB );
	CanScavenge := TeamHasTalent( GB , NAV_DefPlayerTeam , NAS_TechVulture );

	{ Loop through every mek on the board. }
	Mek := GB^.Meks;
	while Mek <> Nil do begin
		if CanSalvage( Mek ) then begin
			{ This is a salvagable part. To start with, remove its pilot(s). }
			repeat
				M2 := ExtractPilot( Mek );
				if M2 <> Nil then DeployGear( GB , M2 , False );
			until M2 = Nil;

			{ Apply emergency repair to it. }
			if CanScavenge or ( Random( 6 ) = 1 ) then begin
				for t := 0 to NumMaterial do begin
					if ( TotalRepairableDamage( Mek , T ) > 0 ) then begin
						{ Determine how many repair points it's possible }
						{ to apply. }
						RPts := RollStep( TeamSkill( GB , NAV_DefPlayerTeam , Repair_Skill_Needed[ T ] , STAT_Knowledge ) ) - 15;
						if RPts > 0 then begin
							ApplyEmergencyRepairPoints( Mek , T , RPts );
						end;
					end;
				end;	{ Checking the repair skills. }
			end;

			{ If at the end of this the mecha is NotDestroyed, it may be }
			{ added to the PC team. If it is destroyed, the PC has a chance }
			{ to use Tech Vulture. }
			if NotDestroyed( Mek ) then begin
				SetNAtt( Mek^.NA , NAG_Location , NAS_Team , NAV_DefPlayerTeam );
				SetNAtt( Mek^.NA , NAG_EpisodeData , NAS_Temporary , 0 );
				{ Also move off the map, to prevent the runaway salvage bug. }
				SetNAtt( Mek^.NA , NAG_Location , NAS_X , 0 );
				{ Record that this is a salvaged mek. }
				SetNAtt( Mek^.NA , NAG_MissionReport , NAS_WasSalvaged , 1 );

			end else if CanScavenge then begin
				M2 := SelectRandomGear( Mek^.SubCom );
				if NotDestroyed( M2 ) and CanBeExtracted( M2 ) and ( SkillRoll( GB , PC , NAS_Repair , STAT_Knowledge , 7 , 0 , True , True ) > 7 ) then begin
					ExtractMechaPart( Mek^.SubCom , M2 );
					SetNAtt( M2^.NA , NAG_Location , NAS_Team , NAV_DefPlayerTeam );
					SetSAttArray( M2^.SA , 'DESIG' , GearName_org( Mek ) );
					SetSAttArray( M2^.SA , 'DESIG_I18N' , GearName( Mek ) );
					SetSAttArray( M2^.SA , 'DESIG_REDESIG' , GearDesig( Mek ) );
					AppendGear( GB^.Meks , M2 );
					{ Record that this is a salvaged part. }
					SetNAtt( M2^.NA , NAG_MissionReport , NAS_WasSalvaged , 1 );
				end;
			end;

		end;
		Mek := Mek^.Next;
	end;

	if CHEAT_Forcibly_Pillage then begin
		SetSAttArray( Source^.SA , TAG_SALVAGED_MAP , 'done' );
	end;
end;

Procedure ProcessRetreat( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ When this command is invoked, all the functioning masters }
	{ belonging to the listed team are removed from the map. A }
	{ Number Of Units trigger is then set. }
var
	Team: Integer;
	Mek: GearPtr;
begin
	{ ERROR CHECK - GB must be defined!!! }
	if GB = Nil then begin
{$IFDEF DEBUG}
		ErrorMessage_fork( 'SCRIPT ERROR: Illigal Retreat. ' + Make_ErrorMessage_ASL(Source,Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT1(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT2(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT3(Event) );
{$ENDIF DEBUG}
		DialogMsg( 'SCRIPT ERROR: Illigal Retreat. ' + Make_ErrorMessage_ASL(Source,Event) );
		Team := ScriptValue( event , GB , Source );
		Exit;
	end;

	{ Find out which team is running away. }
	Team := ScriptValue( event , GB , Source );

	{ Loop through every mek on the board. }
	Mek := GB^.Meks;
	while Mek <> Nil do begin
		if GearOperational( Mek ) and ( NAttValue( Mek^.NA , NAG_Location , NAS_Team ) = Team ) then begin
			SetNAtt( Mek^.NA , NAG_Location , NAS_X , 0 );
			SetNAtt( Mek^.NA , NAG_Location , NAS_Y , 0 );
		end;
		Mek := Mek^.Next;
	end;

	{ Set the trigger. }
	SetTrigger( GB , TRIGGER_NumberOfUnits + BStr( Team ) );
end;

Procedure ProcessAirRaidSiren( const Event: String; GB: GameBoardPtr; const Source: GearPtr );
	{ When this command is invoked, all the functioning masters }
	{ belonging to all NPC teams are ordered to run for their lives. }
var
	Mek: GearPtr;
begin
	{ ERROR CHECK - GB must be defined!!! }
	if GB = Nil then begin
{$IFDEF DEBUG}
		ErrorMessage_fork( 'SCRIPT ERROR: Illigal AirRaidSiren. ' + Make_ErrorMessage_ASL(Source,Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT1(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT2(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT3(Event) );
{$ENDIF DEBUG}
		DialogMsg( 'SCRIPT ERROR: Illigal AirRaidSiren. ' + Make_ErrorMessage_ASL(Source,Event) );
		Exit;
	end;

	{ Loop through every mek on the board. }
	Mek := GB^.Meks;
	while Mek <> Nil do begin
		if GearOperational( Mek ) and ( NAttValue( Mek^.NA , NAG_Location , NAS_Team ) <> NAV_DefPlayerTeam ) then begin
			SetNAtt( Mek^.NA , NAG_EpisodeData , NAS_Orders , NAV_RunAway );
		end;
		Mek := Mek^.Next;
	end;
end;

Procedure ProcessGRunAway( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ When this command is invoked, the grabbed gear is removed }
	{ from the map. A Number Of Units trigger is then set. }
var
	Mek,NPC: GearPtr;
begin
	{ ERROR CHECK - GB must be defined!!! }
	if ( GB = Nil ) or ( Grabbed_Gear = Nil ) then begin
{$IFDEF DEBUG}
		ErrorMessage_fork( 'SCRIPT ERROR: Illigal GRunAway. ' + Make_ErrorMessage_ASL(Source,Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT1(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT2(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT3(Event) );
{$ENDIF DEBUG}
		DialogMsg( 'SCRIPT ERROR: Illigal GRunAway. ' + Make_ErrorMessage_ASL(Source,Event) );
		Exit;
	end;

	{ Loop through every mek on the board. }
	Mek := GB^.Meks;
	while Mek <> Nil do begin
		if Mek = Grabbed_Gear then begin
			SetNAtt( Mek^.NA , NAG_Location , NAS_X , 0 );
			SetNAtt( Mek^.NA , NAG_Location , NAS_Y , 0 );

			{ Set the trigger. }
			SetTrigger( GB , TRIGGER_NumberOfUnits + BStr( NAttValue( MEK^.NA , NAG_Location , NAS_Team ) ) );
		end else if IsMasterGear( Mek ) then begin
			NPC := LocatePilot( Mek );
			if ( NPC <> Nil ) and ( NPC = Grabbed_Gear ) then begin
				SetNAtt( Mek^.NA , NAG_Location , NAS_X , 0 );
				SetNAtt( Mek^.NA , NAG_Location , NAS_Y , 0 );

				{ Set the trigger. }
				SetTrigger( GB , TRIGGER_NumberOfUnits + BStr( NAttValue( MEK^.NA , NAG_Location , NAS_Team ) ) );
			end;
		end;
		Mek := Mek^.Next;
	end;
end;


Procedure ProcessTime( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ Advance the game clock by a specified amount. }
	{ FOr long periods of time, we don't want the PC to get hungry, as this will }
	{ result in an obscene number of "You are hungry" messages. So, break the time }
	{ into hour periods and give the PC some food between each. }
var
	N,OriginalHunger: LongInt;
	PC: GearPtr;
begin
	{ Find out how much to adjust the value by. }
	N := ScriptValue( Event , GB , Source );

	PC := LocatePilot( GG_LocatePC( GB ) );
	if PC <> Nil then begin
		OriginalHunger := NAttValue( PC^.NA , NAG_Condition , NAS_Hunger );
		if OriginalHunger > ( Hunger_Penalty_Starts - 15 ) then OriginalHunger := Hunger_Penalty_Starts - 16;
	end;

	while N > 0 do begin
		if N > 3600 then begin
			if PC <> Nil then SetNAtt( PC^.NA , NAG_Condition , NAS_Hunger , OriginalHunger );
			QuickTime( GB , 3600 );
			N := N - 3600;
		end else begin
			QuickTime( GB , N );
			N := 0;
		end;
	end;
end;

Procedure ProcessForceChat( var Event: String; GB: GameBoardPtr; Source: GearPtr; wo_SayAnything: Boolean );
	{ Force the player to talk with the specified NPC. }
var
	N: LongInt;
begin
	{ Find out which NPC to speak with. }
	N := ScriptValue( Event , GB , Source );

	if GB <> Nil then begin
		if wo_SayAnything then begin
			StoreSAttList( GB^.Trig , '!TALK_wo_SayAnything ' + BStr( N ) );
		end else begin
			StoreSAttList( GB^.Trig , '!TALK ' + BStr( N ) );
		end;
	end;
end;

Procedure ProcessAnnounce( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ The PC has done something special; show an announcement. }
var
	tag: String;
	N: LongInt;
begin
	{ Find out which message to display. }
	tag := ExtractWord( Event );
	N := ScriptValue( Event , GB , Source );

	if GB <> Nil then begin
		StoreSAttList( GB^.Trig , '!ANNOUNCE ' + tag + BStr( N ) );
	end;
end;

Procedure ProcessTrigger( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ A new trigger will be placed in the trigger queue. }
var
	BaseTrigger: String;
	N: LongInt;
begin
	{ Find out the trigger's details. }
	BaseTrigger := ExtractWord( Event );
	N := ScriptValue( Event , GB , Source );

	if GB <> Nil then begin
		StoreSAttList( GB^.Trig , BaseTrigger + BStr( N ) );
	end;
end;

Procedure ProcessTrigger0( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ A new trigger will be placed in the trigger queue. }
var
	BaseTrigger: String;
begin
	{ Find out the trigger's details. }
	BaseTrigger := ExtractWord( Event );

	if GB <> Nil then begin
		StoreSAttList( GB^.Trig , BaseTrigger );
	end;
end;

Procedure ProcessLTrigger( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ A new trigger will be placed in the local trigger queue. }
var
	BaseTrigger: String;
begin
	{ Find out the trigger's details. }
	BaseTrigger := ExtractWord( Event );
	if GB <> Nil then begin
		StoreSAttList( local_triggers , BaseTrigger );
	end;
end;


Procedure ProcessTransform( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ Alter the appearance of the SOURCE gear. }
var
	N: LongInt;
	S: String;
	Procedure SwapSAtts( tag: String );
	begin
		S := AS_GetString ( Source , tag + BStr( N ) );
		if S <> '' then SetSAttArray( Source^.SA , tag , S );
	end;
begin
	{ Find out which aspect to change to. }
	N := ScriptValue( Event , GB , Source );

	{Switch all known dispay descriptors. }
	SwapSAtts( 'ROGUECHAR' );
	SwapSAtts( 'NAME' );
	SwapSAtts( 'NAME_I18N' );
	SwapSAtts( 'SDL_SPRITE' );
	SwapSAtts( TAG_SDL_COLORS );
	SetNAtt( Source^.NA , NAG_Display , NAS_PrimaryFrame , NAttValue( Source^.NA , NAG_Display , N ) );

	if GB <> Nil then begin
		{ While we're here, redo the shadow map. }
		UpdateShadowMap( GB );
	end;
end;

Procedure ProcessMoreText( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ Load and display a text file. }
var
	FName: String;
	txt: SAttListPtr;
	L: SAttListElementPtr;
begin
	{ First, find the file name of the text file to look for. }
	FName := ExtractWord( Event );
	if Source <> Nil then begin
		FName := AS_GetString( Source , FName );
	end else begin
		FName := '';
	end;

	{ Secondly, load and display the file. }
	if FName <> '' then begin
		txt := LoadStringList( Series_Directory + FName );

		if 0 < NumSAttList( txt ) then begin
			{ Process the text. }
			L := txt^.head;
			while L <> Nil do begin
				FormatMessageString( L^.Info , GB , Source );
				L := L^.Next;
			end;
{$IFDEF ASCII}
			MoreText( txt , 1 );
{$ELSE}
			ASRD_GameBoard := GB;
			MoreText( txt , 1 , @ArenaScriptReDraw );
{$ENDIF}
		end;
		DisposeSAttList( txt );
	end;

end; { ProcessMoreText }

Procedure ProcessMoreMemo( var Event: String; GB: GameBoardPtr );
	{ View messages of a certain type - EMAIL, NEWS, or MEMO. }
var
	Key: String;
begin
	{ First, find the memo key to use. }
	Key := ExtractWord( Event );

	{ Secondly, send this to the memo browser. }
	BrowseMemoType( GB , Key );

	{ Finally, update the display. }
	CombatDisplay( GB );
end; { ProcessMoreMemo }

Procedure ProcessSeekGate( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ Aim for a specific gate when entering the next level. }
var
	N: LongInt;
begin
	{ Find out which gate we're talking about. }
	N := ScriptValue( Event , GB , Source );

	SCRIPT_Gate_To_Seek := N;
end;

Procedure ProcessUpdateProps( GB: GameBoardPtr );
	{ Just send an UPDATE trigger to all items on the gameboard. }
var
	T: String;
begin
	T := 'UPDATE';
	if GB <> Nil then begin
		CheckTriggerAlongPath( T , GB , GB^.Meks , True );
	end;
end;

Procedure ProcessBlock( var T: String );
	{ Erase the trigger, so as to prevent other narrative gears }
	{ from acting upon it. }
begin
	{ Do I really need to comment this line? }
	T := '';
end;

Procedure ProcessAccept( var T: String );
	{ Set the trigger to ACCEPT so the CONDITIONACCEPTED function }
	{ knows that it's been accepted. }
begin
	{ Do I really need to comment this line? }
	T := 'ACCEPT';
end;

Procedure ProcessBomb( GB: GameBoardPtr );
	{ Drop a bomb on the town. Yay! }
begin
	if not GB^.QuitTheGame then RandomExplosion( GB );
end;

Procedure ProcessXPV( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ Give some experience points to all PCs and lancemates. }
const
	LM_Renown_Lag = 20;	{ Lancemates will lag one renown band behind PC. }
	Procedure DoRapidLeveling( NPC,PC: GearPtr; Renown: Integer );
		{ Search through this NPC's skills. If you find one that is lower }
		{ than acceptable for the provided Renown, increase it. }
	var
		OptRank,SpecSkill,Skill,N,SkRank: Integer;
		CanGetBonus: Array [1..NumSkill] of Boolean;
	begin
		if NPC = Nil then Exit;

		{ Clamp the renown score. }
		if Renown < 1 then Renown := 1;

		{ Advance the NPC's renown. }
		if NAttValue( NPC^.NA , NAG_CharDescription , NAS_Renowned ) < Renown then AddNAtt( NPC^.NA , NAG_CharDescription , NAS_Renowned , 1 );

		{ Maybe advance the NPC's reaction score. }
		{ This will depend on the current score bonus and the renown level. }
		N := Random( Renown + 5 ) div 2;
		if N > NAttValue( PC^.NA , NAG_ReactionScore , NAttValue( NPC^.NA , NAG_Personal , NAS_CID ) ) then AddReact( GB , PC , NPC , 1 );

		{ Determine the skill level to use. }
		OptRank := SkillRankForRenown( Renown );

		{ Determine the specialist skill of this model. }
		SpecSkill := NAttValue( NPC^.NA , NAG_Personal , NAS_SpecialistSkill );

		{ Count how many skills could use a level-up. }
		N := 0;
		for Skill := 1 to NumSkill do begin
			{ If this is the specialist skill, it can go one higher. }
			SkRank := SkillRank( NPC , Skill );
			if ( Skill = SpecSkill ) and ( SkRank > 1 ) then Dec( SkRank );
			if ( SkRank > 0 ) and ( SkRank < OptRank ) then begin
				Inc( N );
				CanGetBonus[ Skill ] := True;
			end else CanGetBonus[ Skill ] := False;
		end;

		{ If any skills need boosting, select one at random and do that now. }
		if N > 0 then begin
			{ Select one skill at random }
			N := Random( N );

			{ Find it, and give a +1 bonus. }
			for Skill := 1 to NumSkill do begin
				if CanGetBonus[ Skill ] then begin
					Dec( N );
					if N = -1 then begin
						AddNAtt( NPC^.NA , NAG_Skill , Skill , 1 );
						Break;
					end;
				end;
			end;
		end;
	end;
var
	XP,T,Renown,LMs: LongInt;
	M,PC,NPC: GearPtr;
begin
	{ Find out how much to give. }
	XP := ScriptValue( Event , GB , Source );

	{ Count the lancemates, reduce XP if too many. }
	LMs := LancematesPresent( GB );
	if LMs > 4 then LMs := 4;
	if LMs > 1 then XP := ( XP * ( 5 - LMs ) ) div 4;

	{ Locate the PC, and find its Renown score. }
	PC := LocatePilot( GG_LocatePC( GB ) );
	if PC <> Nil then begin
		Renown := NAttValue( PC^.NA , NAG_CharDescription , NAS_Renowned );
	end else begin
		Renown := 1;
	end;

	{ We'll rapidly level one of the lancemates at random. }
	{ Select a model. }
	LMs := Random( LancematesPresent( GB ) ) + 1;

	{ Search for models to give XP to. }
	if GB <> Nil then begin
		M := GB^.Meks;
		while M <> Nil do begin
			T := NAttValue( M^.NA , NAG_Location , NAS_Team );
			if ( T = NAV_DefPlayerTeam ) then begin
				DoleExperience( M , XP );
			end else if ( T = NAV_LancemateTeam ) and OnTheMap( GB , M ) then begin
				DoleExperience( M , XP );

				{ Locate the pilot. }
				NPC := LocatePilot( M );
				{ Only regular lancemates get rapid leveling- Pets and temps don't. }
				if IsRegularLancemate( M ) then begin
					Dec( LMs );
					if ( LMs = 0 ) and ( NPC <> Nil ) then DoRapidLeveling( NPC , PC , Renown - LM_Renown_Lag );

					{ Lancemates are also eligible for training events. }
					if NAttValue( NPC^.NA , NAG_Narrative , NAS_LancemateTraining_Total ) < Renown then begin
						AddNAtt( NPC^.NA , NAG_Narrative , NAS_LancemateTraining_Total , 1 );
					end;
				end;
			end;
			M := M^.Next;
		end;
	end;

	DialogMsg( ReplaceHash( MsgString( 'AS_XPV' ) , Bstr( XP ) ) );
end;


Procedure ProcessGSkillXP( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ Give some skill experience points to the grabbed gear. }
var
	Sk,XP: LongInt;
begin
	{ Find out what skill to give XP for, and how much XP to give. }
	Sk := ScriptValue( Event , GB , Source );
	XP := ScriptValue( Event , GB , Source );

	{ As long as we have a grabbed gear, go for it! }
	if Grabbed_Gear <> Nil then begin
		DoleSkillExperience( Grabbed_Gear , Sk , XP );
	end else begin
{$IFDEF DEBUG}
		ErrorMessage_fork( 'SCRIPT ERROR: Illigal GSkillXP. ' + Make_ErrorMessage_ASL(Source,Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT1(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT2(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT3(Event) );
{$ENDIF DEBUG}
		DialogMsg( 'SCRIPT ERROR: Illigal GSkillXP. ' + Make_ErrorMessage_ASL(Source,Event) );
	end;
end;

Procedure ProcessGMental( const Event: String; GB: GameBoardPtr; const Source: GearPtr );
	{ The grabbed gear is doing something. Make it wait, and spend }
	{ five mental points. }
begin
	{ As long as we have a grabbed gear, go for it! }
	if Grabbed_Gear <> Nil then begin
		WaitAMinute( GB , Grabbed_Gear , ReactionTime( Grabbed_Gear ) * 3 );
		AddMentalDown( Grabbed_Gear , 5 );
	end else begin
{$IFDEF DEBUG}
		ErrorMessage_fork( 'SCRIPT ERROR: Illigal GMental. ' + Make_ErrorMessage_ASL(Source,Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT1(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT2(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT3(Event) );
{$ENDIF DEBUG}
		DialogMsg( 'SCRIPT ERROR: Illigal GMental. ' + Make_ErrorMessage_ASL(Source,Event) );
	end;
end;

Procedure ProcessGStamina( const Event: String; GB: GameBoardPtr; const Source: GearPtr );
	{ The grabbed gear is doing something. Make it wait, and spend }
	{ five stamina points. }
begin
	{ As long as we have a grabbed gear, go for it! }
	if Grabbed_Gear <> Nil then begin
		WaitAMinute( GB , Grabbed_Gear , ReactionTime( Grabbed_Gear ) * 3 );
		AddStaminaDown( Grabbed_Gear , 5 );
	end else begin
{$IFDEF DEBUG}
		ErrorMessage_fork( 'SCRIPT ERROR: Illigal GStamina. ' + Make_ErrorMessage_ASL(Source,Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT1(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT2(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT3(Event) );
{$ENDIF DEBUG}
		DialogMsg( 'SCRIPT ERROR: Illigal GStamina. ' + Make_ErrorMessage_ASL(Source,Event) );
	end;
end;

Procedure ProcessGQuitLance( const Event: String; GB: GameBoardPtr; const Source: GearPtr );
	{ The grabbed gear will quit the lance. }
begin
	if Grabbed_Gear <> Nil then begin
		RemoveLancemate( GB , Grabbed_Gear , False );
	end else begin
{$IFDEF DEBUG}
		ErrorMessage_fork( 'SCRIPT ERROR: Illigal GQuitLance. ' + Make_ErrorMessage_ASL(Source,Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT1(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT2(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT3(Event) );
{$ENDIF DEBUG}
		DialogMsg( 'SCRIPT ERROR: Illigal GQuitLance. ' + Make_ErrorMessage_ASL(Source,Event) );
	end;
end;

Procedure ProcessGJoinLance( const Event: String; GB: GameBoardPtr; const Source: GearPtr );
	{ The grabbed gear will join the lance. }
begin
	if Grabbed_Gear <> Nil then begin
		{ If the Grabbed_Gear is not in play, move it to the current scene. }
		if not IsFoundAlongTrack( GB^.Meks , FindRoot( Grabbed_Gear ) ) then begin
			DelinkGearForMovement( GB , Grabbed_Gear );
			{SetNAtt( Grabbed_Gear^.NA , NAG_Location , NAS_Team , NAV_LancemateTeam );}
			SetNAtt( Grabbed_Gear^.NA , NAG_Location , NAS_Team , NAV_DefNeutralTeam );

			{ Stick it on the map. }
			EquipThenDeploy( GB , Grabbed_Gear , True );
		end;

		AddLancemateFrontEnd( GB , GG_LocatePC( GB ) , Grabbed_Gear , False );
	end else begin
{$IFDEF DEBUG}
		ErrorMessage_fork( 'SCRIPT ERROR: Illigal GJoinLance. ' + Make_ErrorMessage_ASL(Source,Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT1(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT2(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT3(Event) );
{$ENDIF DEBUG}
		DialogMsg( 'SCRIPT ERROR: Illigal GJoinLance. ' + Make_ErrorMessage_ASL(Source,Event) );
	end;
end;

Procedure ProcessGSkillLevel( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ Set the skill points for the grabbed gear. }
var
	Renown: Integer;
begin
	{ Find out what level the NPC should be at. }
	Renown := ScriptValue( Event , GB , Source );
	{ As long as we have a grabbed gear, go for it! }
	if ( Grabbed_Gear <> Nil ) then begin
		{ Record the character's new renown score and mark as a combatant. }
		SetNAtt( Grabbed_Gear^.NA , NAG_CharDescription , NAS_Renowned , Renown );
		SetNAtt( Grabbed_Gear^.NA , NAG_CharDescription , NAS_IsCombatant , 1 );

		SetSkillsAtLevel( Grabbed_Gear , Renown );
	end else begin
{$IFDEF DEBUG}
		ErrorMessage_fork( 'SCRIPT ERROR: Illigal GSkillLevel. ' + Make_ErrorMessage_ASL(Source,Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT1(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT2(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT3(Event) );
{$ENDIF DEBUG}
		DialogMsg( 'SCRIPT ERROR: Illigal GSkillLevel. ' + Make_ErrorMessage_ASL(Source,Event) );
	end;
end;

Procedure ProcessGMoraleDmg( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ Give some morale points to the grabbed gear. }
var
	M: LongInt;
begin
	{ Find out how much morale change. }
	M := ScriptValue( Event , GB , Source );

	{ As long as we have a grabbed gear, go for it! }
	if Grabbed_Gear <> Nil then begin
		AddMoraleDMG( Grabbed_Gear , M );
	end else begin
{$IFDEF DEBUG}
		ErrorMessage_fork( 'SCRIPT ERROR: Illigal GMoraleDmg. ' + Make_ErrorMessage_ASL(Source,Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT1(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT2(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT3(Event) );
{$ENDIF DEBUG}
		DialogMsg( 'SCRIPT ERROR: Illigal GMoraleDmg. ' + Make_ErrorMessage_ASL(Source,Event) );
	end;
end;

Procedure ProcessDrawTerr( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ Alter a single gameboard tile. }
var
	X,Y,T: LongInt;
begin
	{ Find out where and what to adjust. }
	X := ScriptValue( Event , GB , Source );
	Y := ScriptValue( Event , GB , Source );
	T := ScriptValue( Event , GB , Source );

	if ( GB <> NIl ) and OnTheMap( GB , X , Y ) and ( T >= 1 ) and ( T <= NumTerr ) then begin
		SetTerrain( GB , X , Y , T );
	end;
end;

Procedure ProcessArenaRep( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ The arena unit has just won or lost a mission. Adjust its reputation }
	{ accordingly. }
var
	D: LongInt;
	Adv: GearPtr;
begin
	{ Find out how much to adjust. }
	D := ScriptValue( Event , GB , Source );

	{ Find the adventure. }
	Adv := GG_LocateAdventure( GB , Source );

	if ( Adv <> NIl ) then begin
		AddNAtt( Adv^.NA , NAG_CharDescription , -6 , D );
		if NAttValue( Adv^.NA , NAG_CharDescription , -6 ) > 100 then begin
			SetNAtt( Adv^.NA , NAG_CharDescription , -6 , 100 );
		end else if NAttValue( Adv^.NA , NAG_CharDescription , -6 ) < 0 then begin
			SetNAtt( Adv^.NA , NAG_CharDescription , -6 , 0 );
		end;
	end;
end;

Procedure ProcessAddReact( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ Adjust the reaction score of I_NPC. }
var
	DReact: integer;	{ How much do we want to change? }
	PC: GearPtr;
begin
	DReact := ScriptValue( Event , GB , Source );

	PC := LocatePilot( GG_LocatePC( GB ) );
	AddReact( GB , PC , I_NPC , DReact );
end;

Procedure ProcessMagicMap( GB: GameBoardPtr );
	{ Make every tile on the map visible, then redraw. }
var
	X,Y: Integer;
begin
	for X := 1 to GB^.MAP_Width do begin
		for Y := 1 to GB^.MAP_Height do begin
			SetVisibility( GB , X , Y , True );
		end;
	end;
	CombatDisplay( GB );
end;

Procedure ProcessGOpenInv( const Event: String; GB: GameBoardPtr; const Source: GearPtr );
	{ Attempt to access the inventory of the grabbed gear. }
begin
	if ( Grabbed_Gear <> Nil ) and ( GB <> Nil ) then begin
		PCTradeItems( GB , GG_LocatePC( GB ) , Grabbed_Gear );
	end else begin
{$IFDEF DEBUG}
		ErrorMessage_fork( 'SCRIPT ERROR: Illigal GOpenInv. ' + Make_ErrorMessage_ASL(Source,Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT1(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT2(Event) );
		ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT3(Event) );
{$ENDIF DEBUG}
		DialogMsg( 'SCRIPT ERROR: Illigal GOpenInv. ' + Make_ErrorMessage_ASL(Source,Event) );
	end;
end;

Procedure CheckMechaEquipped( GB: GameBoardPtr );
	{ A dynamic encounter is about to be entered. The PC is going to }
	{ want a mecha for it, most likely. }
var
	PC,Mek: GearPtr;
begin
	{ Error check - make sure we have a gameboard to start with. }
	if GB = Nil then Exit;

	{ Find the PC. If the PC doesn't have a mek equipped, then }
	{ prompt for one to be equipped. }
	PC := GG_LocatePC( GB );
	if ( PC <> Nil ) and ( PC^.G <> GG_Mecha ) then begin
		Mek := FindPilotsMecha( GB^.Meks , PC );
		if ( Mek = Nil ) and ( NumPCMeks( GB ) > 0 ) then begin
{$IFDEF ASCII}
			GameMSG( MsgString( 'ARENASCRIPT_CheckMechaEquipped' ) , ZONE_UsagePrompt , InfoGreen );
{$ENDIF}
			FHQ_SelectMechaForPilot( GB , PC );
		end;
	end;
end;

Procedure ProcessSetEncounter( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ Search for encounters leading to a specified scene. Set their activation value. }
var
	SID,DEID,AVal: LongInt;	{ SceneID, Dungeon Entrance ID, Activation Value }
	Adv: GearPtr;
	DestScene: GearPtr;
	Procedure CheckAlongPath( LList: GearPtr );
		{ Check for encounters along this list and throughout all children. }
	begin
		while LList <> Nil do begin
			if ( LList^.G = GG_MetaTerrain ) and ( LList^.S = GS_MetaEncounter ) and ( LList^.Stat[ STAT_Destination ] = SID ) then begin
				SetNAtt( LList^.NA , NAG_Narrative , NAS_EncounterActive , AVal );
			end;
			CheckALongPath( LList^.SubCom );
			CheckALongPath( LList^.InvCom );
			LList := LList^.Next;
		end;
	end;
begin
	{ Find out which scene and what value. }
	SID := ScriptValue( Event , GB , Source );
	AVal := ScriptValue( Event , GB , Source );

	{ Locate the scene. If this is a dungeon, we want to activate the encounter for }
	{ the entrance rather than the level provided. }
	DestScene := FindActualScene( GB , SID );
	if DestScene <> Nil then begin
		DEID := NAttValue( DestScene^.NA , NAG_Narrative , NAS_DungeonEntrance );
		if DEID <> 0 then SID := DEID;
	end;

	{ Locate the adventure. }
	Adv := GG_LocateAdventure( GB , Source );
	if Adv <> Nil then CheckAlongPath( Adv^.SubCom );

	{ Also check the currently deployed gears. }
	if GB <> Nil then CheckAlongPath( GB^.Meks );
end;

Procedure ProcessTrainNPC( var Event: String; GB: GameBoardPtr; Source: GearPtr );
	{ An NPC is going to learn a new skill or talent. Cool! }
const
	MaxTNPCChoices = 10;
var
	CID: LongInt;
	NPC: GearPtr;
	CList,msg,msg2: String;
	CanGainSkill,CanGainTalent: Boolean;
	N,T: Integer;
	Choices: Array [1..MaxTNPCChoices] of Integer;
begin
	{ Find out which NPC, and the list of choices. }
	CID := ScriptValue( Event , GB , Source );
	NPC := GG_LocateNPC( CID , GB , Source );
	CList := ExtractWord( Event );
	if Source <> Nil then begin
		CList := AS_GetString( Source , CList );
	end else begin
		CList := '';
	end;

	{ If this lancemate cannot develop at this time, exit. }
	if not LancemateCanDevelop( NPC ) then Exit;

	{ Find out if the NPC can learn a new skill or a new talent. }
	CanGainSkill := NumberOfSpecialties( NPC ) < NumberOfSkillSlots( NPC );
	CanGainTalent := NumFreeTalents( NPC ) > 0;

	{ Copy the choices list over to the array. }
	N := 0;
	while ( CList <> '' ) and ( N < MaxTNPCChoices ) do begin
		T := ScriptValue( CList , GB , Source );
		if (( T > 0 ) and CanGainSkill and ( NAttValue( NPC^.NA , NAG_Skill , T ) = 0 ) ) or (( T < 0 ) and CanGainTalent and ( NAttValue( NPC^.NA , NAG_Talent , Abs(T) ) = 0 ) and CanLearnTalent( NPC , Abs(T) )  ) then begin
			Inc( N );
			Choices[ N ] := T;
		end;
		DeleteWhiteSpace( CList );
	end;

	{ If we have some valid choices, pick one at random. Yay! }
	if N > 0 then begin
		msg := UTF8_MsgString( 'ProcessTrainNPC', 'TRAINNPC_BASIC' );
		N := Random( N ) + 1;
		if Choices[ N ] > 0 then begin
			SetNAtt( NPC^.NA , NAG_Skill , Choices[ N ] , 1 );
			msg2 := MsgString( 'SKILLNAME_' + BStr( Choices[ N ] ) );
		end else begin
			ApplyTalent( NPC , Abs( Choices[ N ] ) );
			msg2 := MsgString( 'TALENT' + BStr( Abs( Choices[ N ] ) ) );
		end;
		msg := ReplaceHash( msg , GearName( NPC ) , msg2 );
	end else begin
		{ No valid choices. Um... just increase a random stat. }
		msg := UTF8_MsgString( 'ProcessTrainNPC', 'TRAINNPC_STATS' );
		N := Random( 8 ) + 1;
		NPC^.Stat[ N ] := NPC^.Stat[ N ] + 1;
		msg := ReplaceHash( msg , GearName( NPC ) , UTF8_Name( 'STATNAME', BStr( N ) ) );
	end;

	{ Increase the training spent counter. }
	AddNAtt( NPC^.NA , NAG_Narrative , NAS_LancemateTraining_Spent , 1 );

	YesNoMenu( GB , msg , '' , '' );
end;

Procedure ProcessDestroyInvcomAtDelinkJJang( var Event: String; GB: GameBoardPtr; Source: GearPtr );
begin
	DestroyInvcomAtDelinkJJang := True;
end;


Procedure ProcessSoundEffect( var Event: String; GB: GameBoardPtr; Scene: GearPtr );
var
	name: String;
begin
	name := ExtractWord( Event );
	PlaySoundEffect_Direct( name );
end;



Procedure InvokeEvent( Event: String; GB: GameBoardPtr; Source: GearPtr; var Trigger: String; var DeleteSelf: Boolean );
	{ Do whatever is requested by game script EVENT. }
	{ SOURCE refers to the virtual gear which is currently being }
	{ used- it may be a SCENE gear, or a CONVERSATION gear, or }
	{ whatever else I might add in the future. }
var
	cmd: String;
begin
	DeleteSelf := False;
	DEBUG_cmd_org  := Event;
	DEBUG_cmd_org2 := '';
	DEBUG_cmd_org3 := '';
{$IFDEF DEBUG}
	ErrorMessage_fork('SCRIPT TRACE: INVOKEEVENT ' + ' in "' + GearName( Source ) + '" to <' + Event + '>' );
{$ENDIF DEBUG}

	{ Store the time this gear was last invoked. }
	if ( Source <> Nil ) and ( GB <> Nil ) then SetNAtt( Source^.NA , NAG_Narrative , NAS_ScriptActivatedTimer , GB^.ComTime );

	{ Process the event string. }
	while ( Event <> '' ) do begin
		cmd := UpCase( ExtractWord( Event ) );
{$IFDEF DEBUG}
		ErrorMessage_fork('SCRIPT TRACE: INVOKEEVENT "' + cmd + '" in "' + GearName( Source ) + '" to <' + Event + '>' );
{$ENDIF DEBUG}

		if SAttArrayValue( Script_Macros , cmd ) <> '' then begin
			{ Install the macro. }
			InitiateMacro( GB , Source , Event , SAttArrayValue( Script_Macros , cmd ) );
			DEBUG_cmd_org3 := Event;

		end else if ( cmd <> '' ) and ( cmd[1] = '&' ) then begin
			{ Install a local macro. }
			InitiateLocalMacro( GB , Event , cmd , Source );

		end else if Attempt_Gear_Grab( Cmd , Event , GB , Source ) then begin
{$IFDEF DEBUG}
			if ( NIL = Grabbed_Gear ) then begin
				ErrorMessage_fork( 'SCRIPT NOTICE: Gear-Grabbing command failed in InvokeEvent. ' + Make_ErrorMessage_ASL(Source,Event) );
				ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT1(Event) );
				ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT2(Event) );
				ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT3(Event) );
			end;
{$ENDIF DEBUG}
		end else begin
			{ If this is a gear-grabbing command, our work here is done. }

		if cmd = 'EXIT' then ProcessExit( Event , GB , Source )
		else if cmd = 'FORCEEXIT' then ProcessForceExit( Event , GB , Source )
		else if cmd = 'GADDNATT' then ProcessGAddNAtt( Event , GB , Source )
		else if cmd = 'GSETNATT' then ProcessGSetNAtt( Event , GB , Source )
		else if cmd = 'GADDSTAT' then ProcessGAddStat( Event , GB , Source )
		else if cmd = 'GSETSTAT' then ProcessGSetStat( Event , GB , Source )
		else if cmd = 'GSETSATT' then ProcessGSetSAttArray( Event , GB , Source )
		else if cmd = 'DELETEGG' then ProcessDeleteGG( Event , GB , Source )
		else if cmd = 'MOVEGG' then ProcessMoveGG( Event , GB , Source )
		else if cmd = 'MOVEANDPACIFYGG' then ProcessMoveAndPacifyGG( Event , GB , Source )
		else if cmd = 'DEPLOYGG' then ProcessDeployGG( Event , GB , Source )
		else if cmd = 'DYNAGG' then ProcessDynaGG( Event , GB , Source )
		else if cmd = 'GIVEGG' then ProcessGiveGG( Event , GB , Source )
		else if cmd = 'GNEWPART' then ProcessGNewPart( Event , GB , Source )
		else if cmd = 'RETURN' then ProcessReturn( GB )
		else if cmd = 'SETNPC' then ProcessSetNPC( Event , GB , Source )
		else if cmd = 'PRINT' then ProcessPrint( Event , GB , Source )
		else if cmd = 'ALERT' then ProcessAlert( Event , GB , Source )
		else if cmd = 'PRINTDESC' then ProcessPrintDesc( Event , GB , Source )
		else if cmd = 'GMONOLOGUE' then ProcessGMonologue( Event , GB , Source )
		else if cmd = 'ADDDEBRIEFING' then ProcessAddDebriefing( Event , GB , Source )
		else if cmd = 'MEMO' then ProcessMemo( Event , GB , Source )
		else if cmd = 'PMEMO' then ProcessPMemo( Event , GB , Source )
		else if cmd = 'SMEMO' then ProcessSMemo( Event , GB , Source )
		else if cmd = 'QMEMO' then ProcessQMemo( Event , GB , Source )
		else if cmd = 'NEWS' then ProcessNews( Event , GB , Source )
		else if cmd = 'EMAIL' then ProcessEMail( Event , GB , Source )
		else if cmd = 'HISTORY' then ProcessHistory( Event , GB , Source )
		else if cmd = 'VICTORY' then ProcessVictory( GB )
		else if cmd = 'VMSG' then ProcessValueMessage( Event , GB , Source )
		else if cmd = 'SAY' then ProcessSay( Event , GB , Source )
		else if cmd = 'SAYPLOTMSG' then ProcessSayPlotMsg( Event , GB , Source )
		else if cmd = 'SAYANYTHING' then ProcessSayAnything( Event , GB )
		else if cmd = 'ACTIVATEMEME' then ProcessActivateMeme( Event , GB , Source )
		else if cmd = 'IFGINPLAY' then ProcessIfGInPlay( Event , GB , Source )
		else if cmd = 'IFGOK' then ProcessIfGOK( Event , Source )
		else if cmd = 'IFGDEAD' then ProcessIfGDead( Event , Source )
		else if cmd = 'IFGSEXY' then ProcessIfGSexy( Event , GB , Source )
		else if cmd = 'IFGSEALED' then ProcessIfGSealed( Event , Source )
		else if cmd = 'IFGARCHENEMY' then ProcessIfGArchEnemy( Event , GB , Source )
		else if cmd = 'IFGARCHALLY' then ProcessIfGArchAlly( Event , GB , Source )
		else if cmd = 'IFGHASSKILL' then ProcessIfGHasSkill( Event , GB , Source )
		else if cmd = 'IFGCANJOINLANCE' then ProcessIfGCanJoinLance( Event , GB , Source )
		else if cmd = 'IFMECHACANENTERSCENE' then ProcessIfMechaCanEnterScene( Event , GB , Source )
		else if cmd = 'IFTEAMCANSEEGG' then ProcessIfTeamCanSeeGG( Event , GB , Source )
		else if cmd = 'IFFACTION' then ProcessIfFaction( Event , GB , Source )
		else if cmd = 'IFSCENE' then ProcessIfScene( Event , GB , Source )
		else if cmd = 'IFSAFEAREA' then ProcessIfSafeArea( Event , GB , Source )
		else if cmd = 'IFKEYITEM' then ProcessIfKeyItem( Event , GB , Source )
		else if cmd = 'IFGHASITEM' then ProcessIfGHasItem( Event , GB , Source )
		else if cmd = 'IF=' then ProcessIfEqual( Event , GB , Source )
		else if cmd = 'IF#' then ProcessIfNotEqual( Event , GB , Source )
		else if cmd = 'IFG' then ProcessIfGreater( Event , GB , Source )
		else if cmd = 'IFSTORYLESS' then ProcessIfStoryless( Event , Source )
		else if cmd = 'IFYESNO' then ProcessIfYesNo( Event , GB , Source )
		else if cmd = 'IFSKILLTEST' then ProcessIfSkillTest( Event , GB , Source )
		else if cmd = 'IFUSKILLTEST' then ProcessIfUSkillTest( Event , GB , Source )
		else if cmd = 'IFNOOBJECTIONS' then ProcessIfNoObjections( Event , GB , Source )
		else if cmd = 'IFMERITBADGE' then ProcessIfMeritBadge( Event , GB , Source )
		else if cmd = 'IFNODESC' then ProcessIfNoDesc( Event , GB , Source )
		else if cmd = 'TORD' then ProcessTeamOrders( Event , GB , Source )
		else if cmd = 'COMPOSE' then ProcessCompose( Event , GB , Source )
		else if cmd = 'BLOCK' then ProcessBlock( Trigger )
		else if cmd = 'ACCEPT' then ProcessAccept( Trigger )
		else if cmd = 'NEWCHAT' then ProcessNewChat( Event , GB , Source )
		else if cmd = 'ENDCHAT' then ProcessEndChat( Event , GB , Source )
		else if cmd = 'REVERTPERSONA' then RevertPersona( Event , GB , Source )
		else if cmd = 'GOTO' then ProcessGoto( Event , Source )
		else if cmd = 'ADDCHAT' then ProcessAddChat( Event , GB , Source )
		else if cmd = 'SEEKTERR' then ProcessSeekTerr( Event , GB , Source )
		else if cmd = 'SHOP' then ProcessShop( Event , GB , Source )
		else if cmd = 'SCHOOL' then ProcessSchool( Event , GB , Source )
		else if cmd = 'EXPRESSDELIVERY' then ProcessExpressDelivery( Event , GB , Source )
		else if cmd = 'SHUTTLE' then ProcessShuttle( Event , GB , Source )
		else if cmd = 'ENDPLOT' then ProcessEndPlot( Event , GB , Source )
		else if cmd = 'ENDPLOTSBYCONID' then ProcessEndPlotsByConID( Event , GB , Source )
		else if cmd = 'ENDSTORY' then ProcessEndStory( GB , Source )
		else if cmd = 'PURGESTORY' then ProcessPurgeStory( GB , Source )
		else if cmd = 'TREPUTATION' then ProcessTReputation( Event , GB , Source )
		else if cmd = 'LOSERENOWN' then ProcessLoseRenown( GB )
		else if cmd = 'XPV' then ProcessXPV( Event , GB , Source )
		else if cmd = 'MECHAPRIZE' then ProcessMechaPrize( Event , GB , Source )
		else if cmd = 'NEWD' then ProcessNewD( Event , GB , Source )
		else if cmd = 'WMECHA' then ProcessWMecha( Event , GB , Source )
		else if cmd = 'WMONSTER' then ProcessWMonster( Event , GB , Source )
		else if cmd = 'STARTPLOT' then ProcessStartPlot( Event , GB , Source )
		else if cmd = 'UPDATEPLOTS' then ProcessUpdatePlots( Trigger , Event , GB , Source )
		else if cmd = 'STARTSTORY' then ProcessStartStory( Event , GB , Source )
		else if cmd = 'SETMOOD' then ProcessSetMood( Event , GB , Source )
		else if cmd = 'CHECKCOMPONENTS' then ProcessCheckComponents( GB , Source )
		else if cmd = 'NEXTCOMP' then ProcessNextComp( Event , GB , Source )
		else if cmd = 'GALTERCONTEXT' then ProcessGAlterContext( Event , GB , Source )
		else if cmd = 'ATTACK' then ProcessAttack( Event , GB , Source )
		else if cmd = 'SALVAGE' then ProcessSalvage( Event , GB , Source )
		else if cmd = 'RETREAT' then ProcessRetreat( Event , GB , Source )
		else if cmd = 'GRUNAWAY' then ProcessGRunAway( Event , GB , Source )
		else if cmd = 'AIRRAIDSIREN' then ProcessAirRaidSiren( Event , GB , Source )
		else if cmd = 'FORCECHAT' then ProcessForceChat( Event , GB , Source , False )
		else if cmd = 'FORCECHAT_WO_SAYANYTHING' then ProcessForceChat( Event , GB , Source , True )
		else if cmd = 'ANNOUNCE' then ProcessAnnounce( Event , GB , Source )
		else if cmd = 'TIME' then ProcessTime( Event , GB , Source )
		else if cmd = 'TRANSFORM' then ProcessTransform( Event , GB , Source )
		else if cmd = 'SEEKGATE' then ProcessSeekGate( Event , GB , Source )
		else if cmd = 'TRIGGER' then ProcessTrigger( Event , GB , Source )
		else if cmd = 'TRIGGER0' then ProcessTrigger0( Event , GB , Source )
		else if cmd = 'LTRIGGER' then ProcessLTrigger( Event , GB , Source )
		else if cmd = 'UPDATEPROPS' then ProcessUpdateProps( GB )
		else if cmd = 'MORETEXT' then ProcessMoreText( Event , GB , Source )
		else if cmd = 'MOREMEMO' then ProcessMoreMemo( Event , GB )
		else if cmd = 'BOMB' then ProcessBomb( GB )
		else if cmd = 'GSKILLXP' then ProcessGSkillXP( Event , GB , Source )
		else if cmd = 'GSKILLLEVEL' then ProcessGSkillLevel( Event , GB , Source )
		else if cmd = 'GMORALEDMG' then ProcessGMoraleDmg( Event , GB , Source )
		else if cmd = 'DRAWTERR' then ProcessDrawTerr( Event , GB , Source )
		else if cmd = 'MAGICMAP' then ProcessMagicMap( GB )
		else if cmd = 'GMENTAL' then ProcessGMental( Event , GB , Source )
		else if cmd = 'GSTAMINA' then ProcessGStamina( Event , GB , Source )
		else if cmd = 'GQUITLANCE' then ProcessGQuitLance( Event , GB , Source )
		else if cmd = 'GJOINLANCE' then ProcessGJoinLance( Event , GB , Source )
		else if cmd = 'GOPENINV' then ProcessGOpenInv( Event , GB , Source )
		else if cmd = 'ARENAREP' then ProcessArenaRep( Event , GB , Source )
		else if cmd = 'ADDREACT' then ProcessAddReact( Event , GB , Source )
		else if cmd = 'PUMPNEWS' then ProcessPumpNews( Event , GB , Source )
		else if cmd = 'SETENCOUNTER' then ProcessSetEncounter( Event , GB , Source )
		else if cmd = 'TRAINNPC' then ProcessTrainNPC( Event , GB , Source )
		else if cmd = 'DESTROYINVCOMATDELINKJJANG' then ProcessDestroyInvcomAtDelinkJJang( Event , GB , Source )
		else if cmd = 'SOUNDEFFECT' then ProcessSoundEffect( Event , GB , Source )
		else if cmd = '{' then begin
			if Grabbed_Gear_Brace then begin
{$IFDEF DEBUG}
				ErrorMessage_fork( 'SCRIPT ERROR: Do not nest left brace.' );
				ErrorMessage_fork( 'CONTEXT: ' + event );
{$ENDIF DEBUG}
				DialogMsg( 'SCRIPT ERROR: Do not nest left brace.' );
				DialogMsg( 'CONTEXT: ' + event );
			end else begin
				Grabbed_Gear_Brace := True;
			end;
		end else if cmd = '}' then begin
			if Grabbed_Gear_Brace then begin
				Grabbed_Gear_Brace := False;
			end else begin
{$IFDEF DEBUG}
				ErrorMessage_fork( 'SCRIPT ERROR: Too many right brace.' );
				ErrorMessage_fork( 'CONTEXT: ' + event );
{$ENDIF DEBUG}
				DialogMsg( 'SCRIPT ERROR: Too many right brace.' );
				DialogMsg( 'CONTEXT: ' + event );
			end;
		end else if cmd <> '' then begin
{$IFDEF DEBUG}
					ErrorMessage_fork( 'SCRIPT ERROR: Unknown ASL command "' + cmd + '" . ' + Make_ErrorMessage_ASL(Source,Event) );
					ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT1(Event) );
					ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT2(Event) );
					ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT3(Event) );
{$ENDIF DEBUG}
					DialogMsg( 'SCRIPT ERROR: Unknown ASL command ' + cmd );
					DialogMsg( 'CONTEXT: ' + event );
				end
		end; { If not GrabGear }

		if ( NIL = Source ) then begin
			DeleteSelf := True;
		end;
	end;

	if Grabbed_Gear_Brace then begin
{$IFDEF DEBUG}
		ErrorMessage_fork( 'SCRIPT ERROR: No right brace.' );
		ErrorMessage_fork( 'CONTEXT: ' + DEBUG_cmd_org );
{$ENDIF DEBUG}
		DialogMsg( 'SCRIPT ERROR: No right brace.' );
		DialogMsg( 'CONTEXT: ' + DEBUG_cmd_org );
		Grabbed_Gear_Brace := False;
	end;

	{ Process rounding-up events here. }
	if ( SCRIPT_DynamicEncounter <> Nil ) and ( SCRIPT_DynamicEncounter^.V > 0 ) then CheckMechaEquipped( GB );
end;

Procedure InvokeEvent( Event: String; GB: GameBoardPtr; Source: GearPtr; var Trigger: String );
var
	DeleteSelf: Boolean;
begin
	InvokeEvent( Event , GB , Source , Trigger , DeleteSelf );
end;

Procedure HandleWhereAreYou( GB: GameBoardPtr );
	{ The PC has asked the NPC where he is. The NPC will tell the PC }
	{ his or her current location. }
var
	SID: Integer;
begin
	SID := FindSceneID( I_NPC , GB );
	if SID <> 0 then begin
		CHAT_Message := ReplaceHash( MsgString( 'WHEREAREYOU_IAMHERE' ) , SceneName( GB , SID , True ) );
	end else begin
		CHAT_Message := MsgString( 'WHEREAREYOU_Dunno' );
	end;
	Chat_Message := FormatChatStringByGender( Chat_Message, I_NPC );
end;


Procedure HandleAskAboutRumors( GB: GameBoardPtr; Source: GearPtr );
	{ The PC wants to find some rumors. Make it so. }
var
	RL,RL_Skill: GearPtr;
	SkVal,BestScore,SkRoll: Integer;
	NPCDesc,RL_Script: String;
	Rumor_List: SAttListPtr;
	R: SAttListElementPtr;
	Rumor_Error: Boolean;
	RPM: RPGMenuPtr;
begin
	{ Error check- no rumor scumming. }
	if NAttValue( I_NPC^.NA , NAG_Personal , NAS_RumorRecharge ) > GB^.ComTime then begin
		Chat_Message := MsgString( 'RUMOR_IAlreadyToldYou' + BStr( Random( 6 ) + 1 ) );
		Chat_Message := FormatChatStringByGender( Chat_Message, I_NPC );
		Exit;
	end;

	{ Step one- Locate an appropriate skill. }
	RL_Skill := Nil;
	RL := rumor_leads;
	BestScore := 0;
	NPCDesc := XNPCDesc( GB , GG_LocateAdventure( GB , Source ) , I_NPC );
	while ( RL <> Nil ) do begin
		if ( RL^.S >= 1 ) and ( RL^.S <= NumSkill ) and HasSkill( I_PC , RL^.S ) and PartMatchesCriteria( NPCDesc , SAttArrayValue( RL^.SA , 'REQUIRES' ) ) then begin
			{ This skill might be of use. Check it to see. }
			if RL_Skill = Nil then begin
				RL_Skill := RL;
				BestScore := SkillValue( I_PC , RL^.S , STAT_Charm );
			end else begin
				SkVal := SkillValue( I_PC , RL^.S , STAT_Charm );
				if SkVal > BestScore then begin
					RL_Skill := RL;
					BestScore := SkVal;
				end;
			end;
		end;
		RL := RL^.Next;
	end;

	{ Make the skill roll. }
	if ( RL_Skill <> Nil ) and ( BestScore > 3 ) then begin
		{ Make the skill roll here. }
		SkRoll := SkillRoll( GB , I_PC , RL_Skill^.S , STAT_Charm , BasicSkillTarget( NAttValue( I_PC^.NA , NAG_CharDescription , NAS_Renowned ) ) , 0 , False , True );
	end else begin
		{ Make a default crappy skill roll, and find the default lead. }
		SkRoll := RollStep( 3 );
		RL_Skill := SeekCurrentLevelGear( rumor_leads , GG_Persona , 0 );
	end;

	{ If this NPC has a relationship with the PC, the skill roll cannot fall }
	{ beneath an amount appropriate to the reaction score. }
	if NAttValue( I_NPC^.NA , NAG_Relationship , 0 ) > 0 then begin
		SkVal := BasicSkillTarget( ReactionScore( GB^.Scene , I_PC , I_NPC ) );
		if SkVal > SkRoll then SkRoll := SkVal;
	end;

	{ Harvest a list of rumors based on the skill roll. }
	{ As the rumors are added to the list, convert them to memos. }
	Rumor_List := RevealRumors( GB , I_NPC , SkRoll , Rumor_Error );

	{ If our rumor list is not empty, do the lead-in and then present those }
	{ rumors to the player. If it is empty, print a know nothing message. }
	if 0 < NumSAttList( Rumor_List ) then begin
		{ Play the RL_Skill dialog. }
		RL_Script := AS_GetString( RL_Skill , 'START' );

		{ Use NPCDesc as a dummy trigger. }
		InvokeEvent( RL_Script , GB , RL_Skill , NPCDesc );

		{ Create the menu. }
		RPM := CreateRPGMenu( MenuItem , MenuSelect , @ZONE_InteractMenu );
		RPM^.Mode := RPMNoCancel;
		AddRPGMenuItem( RPM , MsgString( 'Continue' ) , 1 );

		{ Show the rumors next. }
		R := Rumor_List^.head;
		while R <> Nil do begin
			SelectMenu( RPM , @InteractRedraw );
			Chat_Message := ReplaceHash( UTF8_MsgString('HandleAskAboutRumors_ExtractData_Series'), MadLibString( RLI_List ), R^.Info );
			Chat_Message := FormatChatStringByGender( Chat_Message, I_NPC );
			SetFlag_PhoneList( GB , Chat_Message , True , False , False , True );
			R := R^.Next;
		end;

		{ Dispose of the rumor list. }
		DisposeRPGMenu( RPM );

	end else begin
		{ Set the chat message. }
		Chat_Message := MsgString( 'RUMOR_NoRumors' + BStr( Random( 6 ) + 1 ) );
		Chat_Message := FormatChatStringByGender( Chat_Message, I_NPC );

	end;
	DisposeSAttList( Rumor_List );

	{ Set the recharge counter for this NPC. }
	if not Rumor_Error then SetNAtt( I_NPC^.NA , NAG_Personal , NAS_RumorRecharge , GB^.ComTime + 21600 + Random( 7200 ) - Random( 7200 ) )
	else SetNAtt( I_NPC^.NA , NAG_Personal , NAS_RumorRecharge , GB^.ComTime + 10 );
end;


Procedure AddLancemate( GB: GameBoardPtr; NPC: GearPtr );
	{ Add the listed NPC to the PC's lance. }
var
	Mecha,Pilot: GearPtr;
	Timer: LongInt;
begin
	{ This NPC will have to quit their current team to do this... }
	{ so, better set a trigger. }
	SetTrigger( GB , TRIGGER_NumberOfUnits + BStr( NAttValue( NPC^.NA , NAG_Location , NAS_Team ) ) );

	{ If this is the first time the lancemate has joined, add the NPC's mecha. }
	if IsACombatant( NPC ) and ( NAttValue( NPC^.NA , NAG_Narrative , NAS_GaveLMMecha ) = 0 ) then begin
		Mecha := SelectNPCMecha( GB , GB^.Scene , NPC );
		if Mecha <> Nil then begin
			SetNAtt( Mecha^.NA , NAG_Location , NAS_Team , NAV_DefPlayerTeam );
			DeployGear( GB , Mecha , False );
			AssociatePilotMek( GB^.Meks , NPC , Mecha );
			SetNAtt( NPC^.NA , NAG_Narrative , NAS_GaveLMMecha , 1 );
		end;
	end;

	{ Set the plot recharge timer to at least 12 hours from now. }
	Timer := NAttValue( NPC^.NA , NAG_Personal , NAS_PlotRecharge );
	if Timer < ( GB^.ComTime + 43200 ) then SetNAtt( NPC^.NA , NAG_Personal , NAS_PlotRecharge , GB^.ComTime + 43200 );

	Pilot := LocatePilot( NPC );
	if Pilot <> NPC then SetNAtt( Pilot^.NA , NAG_Location , NAS_Team , NAV_LancemateTeam );

	SetNAtt( NPC^.NA , NAG_Location , NAS_Team , NAV_LancemateTeam );
	SetLancemateOrders( GB );
end;

Function AddLancemateFrontEnd( GB: GameBoardPtr; PC,NPC: GearPtr; CanCancel: Boolean ): Boolean;
	{ NPC wants to join the lance. If NPC isn't a temp lancemate, and the lance is already }
	{ full, prompt to remove a member before NPC can join. }
	{ Return TRUE if the lancemate was added, or FALSE otherwise. }
var
	NoCancel: Boolean;
	RPM: RPGMenuPtr;
	LM: GearPtr;
	SI,TI: Integer;
	N: LongInt;
begin
	NoCancel := True;

	{ Step One- Check NPC's temp status, then count up the number of lancemates }
	{ to see if there's gonna be a problem. }
	if NAttValue( NPC^.NA , NAG_CharDescription , NAS_CharType ) <> NAV_TempLancemate then begin
		{ Initialize some values. }
		ASRD_GameBoard := GB;
		ASRD_MemoMessage := ReplaceHash( MsgString( 'JOINFE_Need_Space' ) , GearName( NPC ) );

		SI := 1;
		TI := 1;

		{ Count up the number of lancemates. }
		while ( LancematesPresent( GB ) >= NumLancemateSlots( GB^.Scene , PC ) ) and NoCancel do begin
			{ Remove lancemates to make room for NPC. }
			{ Create the menu. }
			RPM := CreateRPGMenu( MenuItem , MenuSelect , @ZONE_MemoMenu );

			{ Create the list. }
			LM := GB^.Meks;
			N := 1;
			while LM <> Nil do begin
				if ( NAttValue( LM^.NA , NAG_Location , NAS_Team ) = NAV_LancemateTeam ) and GearActive( LM ) and IsRegularLancemate( LM ) then begin
					AddRPGMenuItem( RPM , PilotName( LM ) , N );
				end;

				Inc( N );
				LM := LM^.Next;
			end;

			{ RPMSortAlpha( RPM ); } { In UTF-8, this sort causes a result that I cannot assume. }
			AlphaKeyMenu( RPM );
			if CanCancel then AddRPGMenuItem( RPM , MsgString( 'Cancel' ) , SELECTMENU_Cancel )
			else RPM^.Mode := RPMNoCancel;

			RPM^.SelectItem := SI;
			RPM^.TopItem    := TI;
			N := SelectMenu( RPM , @MemoPageReDraw );
			SI := RPM^.SelectItem;
			TI := RPM^.TopItem;

			DisposeRPGMenu( RPM );
			if ( SELECTMENU_Cancel = N ) then NoCancel := False
			else begin
				LM := RetrieveGearSib( GB^.Meks , N );
				RemoveLancemate( GB , LM , True );
			end;
		end;
	end;

	{ Step Two- we apparently have room now. Add the lancemate to the party. }
	if NoCancel then AddLancemate( GB , NPC );

	AddLancemateFrontEnd := NoCancel;
end;

Procedure AttemptJoin( GB: GameBoardPtr );
	{ I_NPC will attempt to join the party. Yay! }
var
	LMP: Integer;	{ Lancemates Present }
begin
	{ Make sure we've got an NPC to deal with. }
	if I_NPC = Nil then Exit;
	{ Also make sure the NPC isn't currently a member of the team. }
	if NAttValue( I_NPC^.NA , NAG_Location , NAS_Team ) = NAV_LancemateTeam then Exit;

	{ Need two more available lancemate points than are currently in use. }
	if CanJoinLance( GB , I_PC , I_NPC ) then begin
		if AddLancemateFrontEnd( GB , I_PC , I_NPC , True ) then CHAT_Message := MsgString( 'JOIN_JOIN' )
		else CHAT_Message := MsgString( 'JOIN_Cancel' );
	end else begin
		LMP := LancematesPresent( GB );
		if ReactionScore( GB^.Scene , I_PC , I_NPC ) < 25 then begin
			CHAT_Message := MsgString( 'JOIN_REFUSE' );
		end else if LMP >= NumLancemateSlots( GB^.Scene , I_PC ) then begin
			CHAT_Message := MsgString( 'JOIN_NOPOINT' );
		end else begin
			CHAT_Message := MsgString( 'JOIN_BUSY' );
		end;
	end;
	CHAT_Message := FormatChatStringByGender( CHAT_Message , I_NPC );
end;

Procedure RemoveLancemate( GB: GameBoardPtr; Mek: GearPtr; DoMessage: Boolean );
	{ Remove NPC from the party. }
	{ If this location isn't a good one for the LM, move the LM to a better place. }
var
	NPC,DestScene: GearPtr;
	msg: String;
	msg_DestScene: String;
	DoMessage_Animal: Boolean;
begin
	if Mek^.G = GG_Mecha then begin
		NPC := ExtractPilot( Mek );
		if NPC = Nil then Exit;
		DeployGear( GB , NPC , False );
	end else begin
		NPC := Mek;
	end;

	msg_DestScene := '';
	if IsInvCom( GB^.Scene ) or ( not IsSafeArea( GB ) ) then begin
		{ This isn't a good place for the NPC to stay. Better move it }
		{ to somewhere else. }

		{ Step One- Locate the destination scene. }
		DestScene := SearchForScene( FindRootScene( GB^.Scene ) , Nil , GB , '(BUILDING|MEETING) PUBLIC -ENEMY' );

		if DestScene <> Nil then begin
			{ Step Two- Announce the lancemate's destination. }
			msg_DestScene := GearName( DestScene );

			{ Step Three- Delink the lancemate and send it there. }
			{  Note that NPC might actually be the NPC's mecha. Deal with it. }
			DelinkGearForMovement( GB , NPC );
			InsertInvCom( DestScene , NPC );
			SetSAttArray( NPC^.SA , 'TEAMDATA' , 'Ally' );
			ChooseTeam( NPC , DestScene );

		end else begin
			{ Fail. Just stick the NPC here, for now. }
			SetSAttArray( NPC^.SA , 'TEAMDATA' , 'Ally' );
			ChooseTeam( NPC , GB^.Scene );
		end;
	end else begin
		{ The LM can just hang around here until the PC gets back. }
		SetSAttArray( NPC^.SA , 'TEAMDATA' , 'Ally' );
		ChooseTeam( NPC , GB^.Scene );
	end;

	DoMessage_Animal := False;
	if IsRobot( NPC ) then begin
		if IsSelfAwareRobot( NPC ) then begin
			if ( '' <> msg_DestScene ) then begin
				msg := ReplaceHash( UTF8_MsgString( 'RemoveLancemate', 'Quit_Lance_self_aware_Robot_Go' ) , msg_DestScene );
			end else begin
				msg := UTF8_MsgString( 'RemoveLancemate', 'Quit_Lance_self_aware_Robot' );
			end;
		end else begin
			if ( '' <> msg_DestScene ) then begin
				msg := ReplaceHash( UTF8_MsgString( 'RemoveLancemate', 'Quit_Lance_non_awareness_Robot_Go' ) , msg_DestScene );
			end else begin
				msg := UTF8_MsgString( 'RemoveLancemate', 'Quit_Lance_non_awareness_Robot' );
			end;
		end;
	end else if IsAnimal( NPC ) then begin
		DoMessage_Animal := True;
		if ( '' <> msg_DestScene ) then begin
			msg := ReplaceHash( UTF8_MsgString( 'RemoveLancemate', 'Quit_Lance_Animal_Go' ) , msg_DestScene );
		end else begin
			msg := UTF8_MsgString( 'RemoveLancemate', 'Quit_Lance_Animal' );
		end;
	end else begin
		if ( '' <> msg_DestScene ) then begin
			msg := ReplaceHash( UTF8_MsgString( 'RemoveLancemate', 'Quit_Lance_Go' ) , msg_DestScene );
		end else begin
			msg := UTF8_MsgString( 'RemoveLancemate', 'Quit_Lance' );
		end;
	end;
	msg := FormatChatStringByGender( msg , NPC );

	{ Set the rumor recharge to six hours from now. After all, the LM }
	{ was hanging around with the PC... they need some time to hear things. }
	SetNAtt( NPC^.NA , NAG_Personal , NAS_RumorRecharge , GB^.ComTime + 21600 );

	if DoMessage then begin
		if NPC = I_NPC then begin
			CHAT_Message := msg;
		end else begin
			if not DoMessage_Animal then begin
				Monologue( GB , NPC , msg );
			end else begin
				DialogMsg( Msg );
			end;
		end;
	end;
end;

Procedure HandleQuit( GB: GameBoardPtr );
	{ I_NPC will quit the party. }
begin
	if I_NPC = Nil then Exit;
	RemoveLancemate( GB , I_NPC , True );
end;


Procedure PruneNothings( var LList: GearPtr );
	{ Traverse the list. Anything marked as ABSOLUTELYNOTHING gets deleted, along with }
	{ all of its children gears. That's tough, but that's life... }
var
	L,L2: GearPtr;
begin
	L := LList;
	while L <> Nil do begin
		L2 := L^.Next;

		if L^.G = GG_AbsolutelyNothing then begin
			RemoveGear( LList , L );
		end else begin
			PruneNothings( L^.SubCom );
			PruneNothings( L^.InvCom );
		end;

		L := L2;
	end;
end;

Procedure HandleInteract( GB: GameBoardPtr; PC,NPC,Persona: GearPtr; with_DialogMsg: String );
	{ The player has just entered a conversation. }
	{ HOW THIS WORKS: The interaction menu is built by an ASL script. }
	{ the player selects one of the provided responses, which will }
	{ either trigger another script ( V >= 0 ) or call one of the }
	{ standard interaction routines ( V < 0 ) }
var
	IntScr: String;		{ Interaction Script }
	N,FreeRumors: Integer;
	RTT: LongInt;		{ ReTalk Time }
	T: String;
	MI: RPGMenuItemPtr;	{ For removing options once they've been used. }
	A: Char;
	skip_interact: Boolean;
begin
	{ Start by allocating the menu. }
	IntMenu := CreateRPGMenu( MenuItem , MenuSelect , @ZONE_InteractMenu );
	{IntMenu^.Mode := RPMEscCancel;}

	if CHEAT_NPC_Edit then begin
		AddRPGMenuKey( IntMenu , KeyMap[ KMC_ChatMessage_to_DialogMSG ].KCode , -7 );
		AddRPGMenuKey( IntMenu , KeyMap[ KMC_SelectPortrait ].KCode , -8 );
		AddRPGMenuKey( IntMenu , KeyMap[ KMC_RenameMecha ].KCode , -9 );
	end;

	{ If this persona has been marked as "NoEscape", make sure the PC }
	{ can't quit the conversation just by pressing ESC. }
	if ( Persona <> Nil ) and ASTringHasBString( SAttArrayValue( Persona^.SA , 'SPECIAL' ) , 'NOESCAPE' ) then IntMenu^.Mode := RPMNoCancel;

	{ Initialize interaction variables. }
	I_PC := PC;
	I_NPC := NPC;
	ASRD_GameBoard := GB;

	{ Since the conversation can be switched by REVERTPERSONA and maybe some other }
	{ effects, from this point onwards use I_PERSONA rather than PERSONA. }
	if Persona <> Nil then I_Persona := Persona
	else I_Persona := BLANK_PERSONA;

	{ Add a divider to the skill roll history. }
	SkillCommentDivider;

	{ If the NPC is fully recharged from talking with you last time, }
	{ get full endurance of 10. Otherwise, only gets partial endurance. }
	if NAttValue( NPC^.NA , NAG_Personal , NAS_ReTalk ) > GB^.ComTime then begin
		I_Endurance := 1;
	end else begin
		I_Endurance := 10;
	end;

	{ Determine the number of "Free" rumors the PC will get. }
	FreeRumors := 0;
	N := ReactionScore( GB^.Scene , PC , NPC );
	if N > 20 then FreeRumors := ( N - 13 ) div 7;
	N := CStat( PC , STAT_Charm );
	if N > 12 then FreeRumors := FreeRumors + Random( N - 11 );

	{ Invoke the greeting event. }
	if ( NAttValue( NPC^.NA , NAG_Location , NAS_Team ) = NAV_LancemateTeam ) and ( NAttValue( NPC^.NA , NAG_Chardescription , NAS_CharType ) <> NAV_TempLancemate ) and (( Persona = Nil ) or (( Persona^.Parent <> Nil ) and ( Persona^.Parent^.G = GG_Scene ))) then begin
		{ Lancemates won't use their local personas while part of the lance. }
		{ Hence the mother of all conditionals above... }
		I_PERSONA := lancemate_tactics_persona;
	end;

	if I_Persona <> Nil then begin
		IntScr := AS_GetString( I_Persona , 'GREETING' );
	end else begin
		{ If there is no standard greeting, set the event to }
		{ build the default interaction menu. }
		IntScr := 'SAYANYTHING NEWCHAT';
	end;
	T := 'Greeting';
	InvokeEvent( IntScr , GB , I_Persona , T );

	skip_interact := ForceInteract_wo_SayAnything and ( Length( CHAT_Message ) < 1 );
	if ( 0 < Length( with_DialogMsg ) ) then begin
		if not skip_interact then begin
			DialogMsg( with_DialogMsg );
		end;
	end;

	repeat
		{ Print the NPC description. }
		{ This could change depending upon what the PC does. }
		if IntMenu^.NumItem > 0 then begin
			ASRD_GameBoard := GB;
			CHAT_React := ReactionScore( GB^.Scene , PC , NPC );
			if Cheat_NPC_Edit then begin
				repeat
					N := SelectMenu( IntMenu , @InteractRedraw );
					if ( -7 = N ) then begin
						DialogMSG( CHAT_Message );
					end else if ( -8 = N ) and ( NIL <> I_NPC ) then begin
						SelectPortrait( GB , I_NPC );
					end else if ( -9 = N ) and ( NIL <> I_NPC ) then begin
						Rename_Mecha( GB , I_NPC )
					end;
				until ( -7 < N )
			end else begin
				N := SelectMenu( IntMenu , @InteractRedraw );
			end;

		end else begin
			{ If the menu is empty, we must leave this procedure. }
			{ More importantly, we better not do anything in }
			{ the conditional below... Set N to equal a "goodbye" result. }
			N := SELECTMENU_Cancel;
		end;

		if ( SELECTMENU_Enable <= N ) and ( I_Persona <> Nil ) then begin
			{ One of the placed options have been triggered. }
			{ Attempt to find the appropriate script to }
			{ invoke. }
			IntScr := AS_GetString( I_Persona , 'RESULT' + BStr( N ) );
			InvokeEvent( IntScr , GB , I_Persona , T );

		end else if N = CMD_Join then begin
			AttemptJoin( GB );
			{ After attempting to join, remove the JOIN option. }
			MI := SetItemByValue( IntMenu , CMD_Join );
			if MI <> Nil then RemoveRPGMenuItem( IntMenu , MI );
			SetItemByPosition( IntMenu , 1 );

		end else if N = CMD_Quit then begin
			HandleQuit( GB );
			ClearMenu( IntMenu );

		end else if N = CMD_WhereAreYou then begin
			HandleWhereAreYou( GB );

		end else if N = CMD_AskAboutRumors then begin
			HandleAskAboutRumors( GB , Persona );
			MI := SetItemByValue( IntMenu , CMD_AskAboutRumors );
			if MI <> Nil then RemoveRPGMenuItem( IntMenu , MI );
			SetItemByPosition( IntMenu , 1 );
		end;

	until ( SELECTMENU_Cancel = N ) or ( IntMenu^.NumItem < 1 ) or ( I_Endurance < 1 ) or ( I_NPC = Nil ) or ( I_Persona = Nil );

	{ If the menu is empty, pause for a minute. Or at least a keypress. }
	if ( IntMenu^.NumItem < 1 ) and ( NIL <> I_NPC ) and ( not skip_interact ) then begin
		InteractRedraw;
		DoFlip;
		if Cheat_NPC_Edit then begin
			repeat
				A := RPGKeyEvent;
				if (KBD_NONE = A) then begin
				end else if ( KeyMap[ KMC_ChatMessage_to_DialogMSG ].KCode = A ) then begin DialogMSG( CHAT_Message );
				end;
			until IsMoreKey( A );
		end else begin
			MoreKey;
		end;
	end;

	{ If the conversation ended because the NPC ran out of patience, }
	{ store a negative reaction modifier. }
	if I_Endurance < 1 then begin
		AddNAtt( PC^.NA , NAG_ReactionScore , NAttValue( NPC^.NA , NAG_Personal , NAS_CID ) , -5 );

		{ Overchatting is a SOCIABLE action. }
		AddReputation( PC , 3 , 1 );
	end;

	{ Set the ReTalk value. }
	{ Base retalk time is 1500 ticks; may be raised or lowered depending }
	{ upon the NPC's ENDURANCE and also how well the NPC likes the PC. }
	if ( I_NPC <> Nil ) and ( SCRIPT_DynamicEncounter = Nil ) then begin
		RTT := GB^.ComTime + 1500 - ( 15 * ReactionScore( GB^.Scene , PC , NPC ) ) - ( 50 * I_Endurance );
		if RTT < ( GB^.ComTime + AP_Minute ) then RTT := GB^.ComTime + AP_Minute;
		SetNAtt( NPC^.NA , NAG_Personal , NAS_ReTalk , RTT );
	end;

	{ Set the NumberOfConversations counter. }
	if I_NPC <> Nil then AddNAtt( NPC^.NA , NAG_Personal , NAS_NumConversation , 1 );
	I_NPC := Nil;
	I_Persona := Nil;

	{ Get rid of the menu. }
	DisposeRPGMenu( IntMenu );
end;

Procedure PCDoVerbalAttack( GB: GameBoardPtr; PC,NPC: GearPtr );
	{ The PC wants to verbally abuse this NPC. If it's possible do so. If it }
	{ isn't possible, then explain why. }
begin
	NPC := LocatePilot( NPC );
	if CurrentMental( PC ) < 1 then begin
		DialogMsg( MsgString( 'VERBALATTACK_NoMP' ) );
	end else if ( NPC <> Nil ) and CanSpeakWithTarget( GB , PC , NPC ) then begin
		if AreEnemies( GB , PC , NPC ) then begin
			DoVerbalAttack( GB , PC , FindRoot( NPC ) );
			{ When the PC taunts an enemy, it takes an action. }
			SetNAtt( PC^.NA , NAG_Action , NAS_CallTime , GB^.ComTime + ReactionTime( PC ) );
		end else begin
			DialogMsg( ReplaceHash( MsgString( 'TAUNT_OnlyEnemies' ) , GearName( NPC ) ) );
		end;
	end else begin
		DialogMsg( ReplaceHash( UTF8_MsgString( 'DoTalkingWithNPC' , 'TooFar' ) , GearName( NPC ) ) );
	end;
end;

Procedure DoTalkingWIthNPC( GB: GameBoardPtr; PC,Mek: GearPtr; ByTelephone: Boolean );
	{ Actually handle the talking with an NPC already selected. }
var
	Persona,NPC: GearPtr;
	CID: Integer;
	React: Integer;
	ReTalk: LongInt;
	msg: String;
begin
	NPC := LocatePilot( Mek );
	if ( NPC <> Nil ) and GearOperational( NPC ) and AreEnemies( GB , Mek , PC ) and ( not IsAnimal( NPC ) ) and ( not IsNoAwareRobot( NPC ) ) and IsFoundAlongTrack( GB^.Meks , FindRoot( NPC ) ) and ( NATtValue( NPC^.NA , NAG_EpisodeData , NAS_SurrenderStatus ) <> NAV_NowSurrendered ) and OnTheMap( GB , NPC ) then begin
		PCDoVerbalAttack( GB , PC , NPC );

	end else if ( NPC <> Nil ) and GearOperational( NPC ) then begin
		if ByTelephone or CanSpeakWithTarget( GB , PC , NPC ) then begin
			CID := NAttValue( NPC^.NA , NAG_Personal , NAS_CID );
			if CID <> 0 then begin
				{ Everything should be okay to talk... Now see if the NPC wants to. }
				{ Determine the NPC's RETALK and REACT values. }
				ReTalk := NAttValue( NPC^.NA , NAG_Personal , NAS_Retalk );
				React := ReactionScore( GB^.Scene , PC , NPC );

				Persona := SeekPersona( GB , CID );

				{ Surrendered NPCs never refuse to talk. }
				if NATtValue( NPC^.NA , NAG_EpisodeData , NAS_SurrenderStatus ) = NAV_NowSurrendered then begin
					msg := ReplaceHash( UTF8_MsgString( 'DoTalkingWithNPC' , 'Talking_Start' ) , GearName( NPC ) );
					msg := FormatChatStringByGender( msg, NPC );
					DialogMsg( msg );

					HandleInteract( GB , PC , NPC , Persona , '' );
					SetFlag_PhoneList( GB , GearName( NPC ) , True , True , True , True );
					CombatDisplay( gb );

				{ If the NPC really doesn't like the PC, }
				{ they'll refuse to talk on principle. }
				end else if ( ( React + RollStep( SkillValue ( PC , NAS_Conversation , STAT_Ego ) ) ) < -Random( 120 ) ) or ( AreEnemies( GB , NPC , PC ) and IsFoundAlongTrack( GB^.Meks , NPC ) ) then begin
					msg := ReplaceHash( UTF8_MsgString( 'DoTalkingWithNPC' , 'RefuseHard' ) , GearName( NPC ) );
					msg := FormatChatStringByGender( msg, NPC );
					DialogMsg( msg );
					SetFlag_PhoneList( GB , GearName( NPC ) , False , False , False , True );
					SetNAtt( NPC^.NA , NAG_Personal , NAS_Retalk , GB^.ComTime + 1500 );

				{ If the NPC is ready to talk, is friendly with the PC, or has a PERSONA gear defined, }
				{ they'll be willing to talk. }
				end else if ( ReTalk < GB^.ComTime ) or ( Random( 50 ) < ( React + 20 ) ) or ( Persona <> Nil ) then begin
					msg := ReplaceHash( UTF8_MsgString( 'DoTalkingWithNPC' , 'Talking_Start' ) , GearName( NPC ) );
					msg := FormatChatStringByGender( msg, NPC );
					DialogMsg( msg );

					HandleInteract( GB , PC , NPC , Persona , '' );
					SetFlag_PhoneList( GB , GearName( NPC ) , True , True , True , True );
					CombatDisplay( gb );

				end else begin
					msg := ReplaceHash( UTF8_MsgString( 'DoTalkingWithNPC' , 'RefuseSoft' ) , GearName( NPC ) );
					msg := FormatChatStringByGender( msg, NPC );
					DialogMsg( msg );
					SetFlag_PhoneList( GB , GearName( NPC ) , True , True , False , True );

				end;

				WaitAMinute( GB , PC , ReactionTime( PC ) );
			end else begin
				msg := ReplaceHash( UTF8_MsgString( 'DoTalkingWithNPC' , 'NoReply' ) , GearName( NPC ) );
				msg := FormatChatStringByGender( msg, NPC );
				DialogMsg( msg );
				SetFlag_PhoneList( GB , GearName( NPC ) , False , False , False , True );
			end;
		end else begin
			msg := ReplaceHash( UTF8_MsgString( 'DoTalkingWithNPC' , 'TooFar' ) , GearName( NPC ) );
			msg := FormatChatStringByGender( msg, NPC );
			DialogMsg( msg );
		end;
	end else begin
		DialogMsg( UTF8_MsgString('DoTalkingWithNPC', 'Not found') );
	end;
end;

Procedure ForceInteract( GB: GameBoardPtr; CID: LongInt; wo_SayAnything: Boolean );
	{ Attempt to force the PC to converse with the provided NPC. }
var
	PC,NPC,Interact: GearPtr;
	msg: String;
begin
	ForceInteract_wo_SayAnything := wo_SayAnything;

	{ Locate all the required elements. }
	PC := LocatePilot( GG_LocatePC( GB ) );
	NPC := SeekGearByCID( GB^.Meks , CID );
	if NPC = Nil then NPC := SeekGearByCID( FindRoot( GB^.Scene ) , CID );
	Interact := SeekPersona( GB , CID );

	if ( PC <> Nil ) and ( NPC <> Nil ) and NotDestroyed( PC ) then begin
		{ Before initiating the conversation, get rid of the }
		{ recharge timer, since the NPC initiated this chat }
		{ and won't get pissed off. }
		SetNAtt( NPC^.NA , NAG_Personal , NAS_ReTalk , 0 );

		{ Print an appropriate message. }
		msg := '';
		if NPC^.Parent = Nil then begin
			{ The NPC has no parent, so it must be on the gameboard }
			{ and not in a mecha. Use the conversation message. }
			msg := ReplaceHash( MsgString( 'FORCECHAT_SPEAK' ) , GearName( NPC ) );
		end else begin
			{ Use the contact message. }
			msg := ReplaceHash( MsgString( 'FORCECHAT_CONTACT' ) , GearName( NPC ) );
		end;

		{ Hand everything to the interaction procedure. }
		HandleInteract( GB , PC , NPC , Interact , msg );
	end;

	ForceInteract_wo_SayAnything := False;
end;

Function TriggerGearScript( GB: GameBoardPtr; Source: GearPtr; var Trigger: String; var DeleteSelf: Boolean ): Boolean;
	{ Attempt to trigger the requested script in this gear. If the }
	{ script cannot be found, then do nothing. }
var
	E: String;
	it: Boolean;
begin
	DeleteSelf := False;
	it := False;
	if Source <> Nil then begin
		E := AS_GetString( Source , Trigger );
		if E <> '' then begin
			InvokeEvent( E , GB , Source , Trigger , DeleteSelf );
			it := True;
		end;
	end;
	TriggerGearScript := it;
end;

Function TriggerGearScript( GB: GameBoardPtr; Source: GearPtr; var Trigger: String ): Boolean;
var
	DeleteSelf: Boolean;
begin
	TriggerGearScript := TriggerGearScript( GB , Source , Trigger , DeleteSelf );
end;

Function CheckTriggerAlongPath( var T: String; GB: GameBoardPtr; Plot: GearPtr; CheckAll: Boolean ): Boolean;
	{ Check all the active narrative gears in this list (plots, stories, and factions) }
	{ looking for events which match the provided trigger. }
	{ Return TRUE if an event was invoked, or FALSE if no event was encountered. }
var
	P2: GearPtr;
	DeletePlot: Boolean;
	it,I2: Boolean;
	TempList: SAttListPtr;
	LT: SAttListElementPtr;	{ Local Trigger counter }
	LT_tmp: STring;	{ The content of the local trigger currently being processed. }
begin
	it := False;
	while ( Plot <> Nil ) and ( T <> '' ) do begin
		P2 := Plot^.Next;
		if CheckAll or ( Plot^.G = GG_Plot ) or ( Plot^.G = GG_Faction ) or ( Plot^.G = GG_Story ) or ( Plot^.G = GG_Adventure ) or ( Plot^.G = GG_CityMood ) then begin
			{ FACTIONs and STORYs can hold active plots in their InvCom. }
			if ( Plot^.G = GG_Faction ) or ( Plot^.G = GG_Story ) or ( Plot^.G = GG_Adventure ) then CheckTriggerAlongPath( T , GB , Plot^.InvCom , CheckAll);

			{ Initialize the list of local triggers. }
			DisposeSAttList( local_triggers );
			local_triggers := NIL;

			{ Check the trigger against this gear's scripts. }
			I2 := TriggerGearScript( GB , Plot , T , DeletePlot );
			it := it or I2;

			{ If any local triggers were tripped, call them now. }
			while ( ( NIL <> local_triggers ) and ( 0 < NumSAttList( local_triggers ) ) ) do begin
				TempList := local_triggers;
				local_triggers := NIL;
				while 0 < NumSAttList( TempList ) do begin
					LT := TempList^.head;
					LT_Tmp := LT^.Info;
{$IFDEF DEBUG}
					ErrorMessage_fork('SCRIPT TRACE: LTRIGGER "' + LT_Tmp + '"' );
{$ENDIF DEBUG}
					RemoveSAttList( TempList , LT );
					TriggerGearScript( GB , Plot , LT_Tmp );
				end;
				DisposeSAttList( TempList );
			end;

			if not( DeletePlot ) then begin
				{ The trigger above might have changed the }
				{ structure, so reset P2. }
				P2 := Plot^.Next;

				{ Remove the plot, if it's been advanced. }
				if Plot^.G = GG_AbsolutelyNothing then begin
					if IsInvCom( Plot ) then RemoveGear( Plot^.Parent^.InvCom , Plot )
					else if IsSubCom( Plot ) then RemoveGear( Plot^.Parent^.SubCom , Plot );
				end;
			end;
		end;
		Plot := P2;
	end;
	CheckTriggerAlongPath := it;
end;

Procedure HandleTriggers( GB: GameBoardPtr );
	{ Go through the list of triggers, enacting events if any are }
	{ found. Deallocate the triggers as they are processed. }
var
	TList: SAttListPtr;	{ Trigger List }
	TP: SAttListElementPtr;	{ Trigger Pointer }
	E,msg: String;
	City: GearPtr;
begin
	IntMenu := Nil;

	{ Only try to implement triggers if this gameboard has a scenario }
	{ defined. }
	if GB^.Scene <> Nil then begin

		{ Some of the events we process might add their own }
		{ triggers to the list. So, we check all the triggers }
		{ currently set, then look at the GB^.Trig list again }
		{ to see if any more got put there. }
		while GB^.Trig <> Nil do begin
			{ Copy the list pointer to TList, clear the }
			{ list pointer from GB, and set the pointer }
			{ to the first trigger. }
			TList := GB^.Trig;
			GB^.Trig := Nil;
			TP := TList^.head;

			while TP <> Nil do begin
{$IFDEF DEBUG}
				ErrorMessage_fork('SCRIPT TRACE: HANDLETRIGGERS "' + TP^.Info + '"' );
{$ENDIF DEBUG}
				{ Commands can be embedded in the triggers list. }
				{ The actual point of this is to allow scripts }
				{ to automatically activate interactions & props. }
				if ( Length( TP^.Info ) > 0 ) and ( TP^.Info[1] = '!' ) then begin
					{ Copy the command. }
					E := UpCase( ExtractWord( TP^.Info ) );
					DeleteFirstChar( E );

					if E = 'TALK' then begin
						ForceInteract( GB , ExtractValue( TP^.Info ) , False );
					end else if E = 'TALK_WO_SAYANYTHING' then begin
						ForceInteract( GB , ExtractValue( TP^.Info ) , True );
					end else if E = 'ANNOUNCE' then begin
						msg := MsgString( TP^.Info );
						if msg <> '' then YesNoMenu( GB , msg , '' , '' );
					end else begin
{$IFDEF DEBUG}
						ErrorMessage_fork( 'SCRIPT ERROR: Unknown Trigger "' + E + '" ('+ TP^.Info + ') at HandleTriggers(). ' );
{$ENDIF DEBUG}
						DialogMsg( 'SCRIPT ERROR: Unknown Trigger "' + E + '" ('+ TP^.Info + ') at HandleTriggers(). ' );
					end;

					{ Clear this trigger. }
					TP^.Info := '';

				end else if TP^.Info <> '' then begin
					{ If there is a SAttList in the scenario description }
					{ named after this trigger description, it will }
					{ happen now. First, see if such an event exists. }

					{ Check the PLOTS, FACTIONS and STORIES in }
					{ Adventure/InvCom first. }
					if GB^.Scene^.Parent <> Nil then begin
						CheckTriggerAlongPath( TP^.Info , GB , FindRoot( GB^.Scene ) , False );
					end;

					{ Check for quests and moods in the current city next. }
					City := FindRootScene( GB^.Scene );
					if ( City <> Nil ) and ( TP^.Info <> '' ) then begin
						CheckTriggerAlongPath( TP^.Info , GB , City^.SubCom , False );
					end;

					{ Check the current scene last. }
					if TP^.Info <> '' then TriggerGearScript( GB , GB^.Scene , TP^.Info );

				end;

				TP := TP^.Next;
			end;

			{ Get rid of the trigger list. }
			DisposeSAttList( TList );

		end;

		DoScriptGC( GB );
	end;
end;

Function StartRescueScenario( GB: GameBoardPtr; PC: GearPtr; Context: String ): Boolean;
	{ Attempt to load a rescue scenario for the PC. }
	Function RescueContext: String;
		{ Generate a string telling everything that needs to be told about }
		{ the PC's current location. }
	var
		RC: String;
		C: GearPtr;
	begin
		RC := Context;
		AddTraits( RC , SAttArrayValue( GB^.Scene^.SA , 'CONTEXT' ) );
		AddTraits( RC , GearDesig_org( GB^.Scene ) );
		AddTraits( RC , SAttArrayValue( GB^.Scene^.SA , 'TYPE' ) );
		AddTraits( RC , SAttArrayValue( GB^.Scene^.SA , 'TERRAIN' ) );

		{ Next add the data for the city we're located in, its faction, and the }
		{ world that it's located in. }
		C := FindRootScene( GB^.Scene );
		if C <> Nil then begin
			AddTraits( RC , GearDesig_org( C ) );
			C := SeekFaction( FindRoot( GB^.Scene ) , NAttValue( C^.NA , NAG_Personal , NAS_FactionID ) );
			if C <> Nil then AddTraits( RC , GearDesig_org( C ) );
		end;
		RescueContext := RC;
	end;	{ RescueContext }
var
	Rescue_list,R: GearPtr;
	ItWorked: Boolean;
begin
	{ If no scene, can't be rescued. Sorry. }
	if GB^.Scene = Nil then Exit( False );

	{ Assume the rescue failed unless shown otherwise. }
	ItWorked := False;

	{ Load the rescue files. }
	Rescue_List := AggregatePattern( 'RESCUE_*.txt' , Series_Directory );

	{ Generate the complete context for the rescue. }
	Context := RescueContext;

	while Rescue_List <> Nil do begin
		R := FindNextComponent( Rescue_List , Context );

		if R <> Nil then begin
			if ( 0 = GB^.Scene^.S ) then begin
				{ BUG ? }
				GB^.Scene^.S := GB^.ReturnCode;
			end;
			DelinkGear( Rescue_List , R );
			SetNAtt( R^.NA , NAG_ElementID , 1 , GB^.Scene^.S );
			if InsertPlot( FindRootScene( GB^.Scene ) , FindRoot( GB^.Scene ) , R , GB , CurrentPCRenown( GB ) ) then begin
				{ Start by printing a message, since the time taken by the }
				{ rescue scenario is likely to cause a noticeable delay. }
				DialogMsg( MsgString( 'JustAMinute' ) );
				CombatDisplay( GB );
				DoFLip;

				DoCompleteRepair( PC );
				DisposeGear( Rescue_List );
				SetTrigger( GB , 'UPDATE' );
				HandleTriggers( GB );
				ItWorked := True;
			end;

		end else begin
			DisposeGear( Rescue_List );
		end;
	end;

	StartRescueScenario := ItWorked;
end;

Procedure DoScriptGC( GB: GameBoardPtr );
	{ Get rid of any ABSOLUTELYNOTHINGS that may be hanging about. }
var
	Adv: GearPtr;
begin
	if NeedGC then begin
		Adv := GG_LocateAdventure( GB, Nil );
		if Adv <> Nil then begin
			PruneNothings( Adv^.SubCom );
			PruneNothings( Adv^.InvCom );
			NeedGC := False;
		end;
	end;
end;


Procedure PutAwayArenascript( const Mek: GearPtr );
begin
	if ( Mek = Grabbed_Gear ) then begin
{$IF DEFINED(DEBUG) and DEFINED(TRACE)}
		ErrorMessage_fork( 'TRACE: PutAwayArenascript - Grabbed_Gear:' + GearName(Mek) + ':' + IntToHex(Int64(Mek),16) );
{$ENDIF}
		Grabbed_Gear := NIL;
	end;
	if ( Mek = I_PC ) then begin
{$IF DEFINED(DEBUG) and DEFINED(TRACE)}
		ErrorMessage_fork( 'TRACE: PutAwayArenascript - I_PC:' + GearName(Mek) + ':' + IntToHex(Int64(Mek),16) );
{$ENDIF}
		I_PC := NIL;
	end;
	if ( Mek = I_NPC ) then begin
{$IF DEFINED(DEBUG) and DEFINED(TRACE)}
		ErrorMessage_fork( 'TRACE: PutAwayArenascript - I_NPC:' + GearName(Mek) + ':' + IntToHex(Int64(Mek),16) );
{$ENDIF}
		I_NPC := NIL;
	end;
	if ( Mek = I_NPC_org ) then begin
{$IF DEFINED(DEBUG) and DEFINED(TRACE)}
		ErrorMessage_fork( 'TRACE: PutAwayArenascript - I_NPC_org:' + GearName(Mek) + ':' + IntToHex(Int64(Mek),16) );
{$ENDIF}
		I_NPC_org := NIL;
	end;
	if ( Mek = I_Persona ) then begin
{$IF DEFINED(DEBUG) and DEFINED(TRACE)}
		ErrorMessage_fork( 'TRACE: PutAwayArenascript - I_Persona:' + GearName(Mek) + ':' + IntToHex(Int64(Mek),16) );
{$ENDIF}
		I_Persona := NIL;
	end;
end;


Procedure InitArenascript;
begin
	Grabbed_Gear := NIL;
	Grabbed_Gear_Brace := False;
	I_PC := NIL;
	I_NPC := NIL;
	I_NPC_org := NIL;
	I_Persona := Nil;
	ASRD_GameBoard := NIL;
	ASRD_MemoMessage := '';
	DisposeSAttList( local_triggers );
	local_triggers := NIL;
end;



initialization
begin
{$IFDEF DEBUG}
	ErrorMessage_fork('DEBUG: arenascript.pp');
{$ENDIF DEBUG}
	DEBUG_cmd_org  := '';
	DEBUG_cmd_org2 := '';
	DEBUG_cmd_org3 := '';
	InitArenascript;

	SCRIPT_DynamicEncounter := Nil;
	Script_Macros := LoadStringArray( Script_Macro_File );
	if Script_Macros = Nil then begin
{$IFDEF DEBUG}
		ErrorMessage_fork( 'ERROR: Script Macros failed.' );
{$ENDIF DEBUG}
		DialogMsg( 'ERROR: Script Macros failed.' );
	end;
	Value_Macros := LoadStringArray( Value_Macro_File );
	if Value_Macros = Nil then begin
{$IFDEF DEBUG}
		ErrorMessage_fork( 'ERROR: Value Macros failed.' );
{$ENDIF DEBUG}
		DialogMsg( 'ERROR: Value Macros failed.' );
	end;

	Default_Scene_Scripts := LoadStringArray( Data_Directory + 'scene.txt' );

	lancemate_tactics_persona := LoadFile( 'lmtactics.txt' , Data_Directory );
	BLANK_PERSONA := LoadFile( 'blankpersona.txt' , Data_Directory );
	rumor_leads := LoadFile( 'rumor_leads.txt' , Data_Directory );

	local_triggers := CreateSAttList;

	IntMenu := Nil;

	NeedGC := False;

	DestroyInvcomAtDelinkJJang := False;

	ForceInteract_wo_SayAnything := False;
end;

finalization
begin
	if SCRIPT_DynamicEncounter <> Nil then begin
		DisposeGear( SCRIPT_DynamicEncounter );
	end;
	DisposeSAttArray( Script_Macros );
	DisposeSAttArray( Value_Macros );
	DisposeSAttArray( Default_Scene_Scripts );
	DisposeGear( lancemate_tactics_persona );
	DisposeGear( BLANK_PERSONA );
	DisposeGear( rumor_leads );
	DisposeSAttList( local_triggers );
{$IFDEF DEBUG}
	ErrorMessage_fork('DEBUG: arenascript.pp(finalization)');
{$ENDIF DEBUG}
end;

end.
