unit backpack;
	{ This unit handles both the inventory display and the }
	{ FieldHQ interface, which uses many of the same things. }
{
	GearHead2, a roguelike mecha CRPG
	Copyright (C) 2005 Joseph Hewitt

	This library is free software; you can redistribute it and/or modify it
	under the terms of the GNU Lesser General Public License as published by
	the Free Software Foundation; either version 2.1 of the License, or (at
	your option) any later version.

	The full text of the LGPL can be found in license.txt.

	This library is distributed in the hope that it will be useful, but
	WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
	General Public License for more details. 

	You should have received a copy of the GNU Lesser General Public License
	along with this library; if not, write to the Free Software Foundation,
	Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
}
{$LONGSTRINGS ON}

interface

uses gears_base,gears,locale,ghchars,
{$IFDEF ASCII}
	vidgfx;
{$ELSE}
	sdlgfx;
{$ENDIF}

const
	TRIGGER_GetItem = 'GET';

	Skill_Use_Trigger: Array [1..NumSkill] of String = (
		'USE', 'USE', 'USE', 'USE', 'USE',
		'USE', 'USE', 'USE', 'USE', 'USE',
		'USE', 'CLUE_SURVIVAL', 'CLUE_REPAIR', 'CLUE_MEDICINE', 'USE',
		'USE', 'USE', 'USE', 'USE', 'USE',
		'CLUE_SCIENCE', 'USE', 'CLUE_CODEBREAKING', 'CLUE_MYSTICISM', 'USE',
		'USE', 'CLUE_INSIGHT', 'USE'
	);


Function DefaultAtOp( Weapon: GearPtr ): Integer;

Procedure GatherFieldHQ( GB: GameBoardPtr );
Procedure GivePartToPC( var LList: GearPtr; Part, PC: GearPtr );
Procedure GivePartToPC( GB: GameBoardPtr; Part, PC: GearPtr );

{$IFNDEF ASCII}
Procedure SelectColors( M: GearPtr; Redrawer: RedrawProcedureType_with_GearPtr );
Procedure SelectSprite( M: GearPtr; Redrawer: RedrawProcedureType );
{$ENDIF}

Function DoFieldRepair( GB: GameBoardPtr; PC , Item: GearPtr; Skill: Integer ): Boolean;

Function Handless( Mek: GearPtr ): Boolean;
Function CanBeExtracted( Item: GearPtr ): Boolean;
Procedure ExtractMechaPart( var LList,Item: GearPtr );

Function ShakeDown( GB: GameBoardPtr; Part: GearPtr; X,Y: Integer ): LongInt;
Procedure PCGetItem( GB: GameBoardPtr; PC: GearPtr );
Procedure PCTradeItems( GB: GameBoardPtr; PC,Target: GearPtr );
Procedure EatItem( GB: GameBoardPtr; TruePC , Item: GearPtr );

Procedure FHQ_SelectMechaForPilot( GB: GameBoardPtr; NPC: GearPtr );
Procedure ArenaHQBackpack( Source,BPPC: GearPtr; BasicRedraw: RedrawProcedureType_with_GearPtr );
Procedure LancemateBackpack( GB: GameBoardPtr; PC,NPC: GearPtr; BasicRedraw: RedrawProcedureType_with_GearPtr );
Procedure BackpackMenu( GB: GameBoardPtr; PC: GearPtr; StartWithInv: Boolean; BasicRedraw: RedrawProcedureType_with_GearPtr );
Procedure MechaPartEditor( GB: GameBoardPtr; var LList: GearPtr; PC,Mek: GearPtr; BasicRedraw: RedrawProcedureType_with_GearPtr );

Procedure MechaPartBrowser( GB: GameBoardPtr; PC,NPC,Mek: GearPtr; RDP: RedrawProcedureType );
Procedure MysteryPartBrowser( Mek: GearPtr; RDP: RedrawProcedureType );
Procedure BrowseDesignFile( GB: GameBoardPtr; List: GearPtr; RDP: RedrawProcedureType );

Procedure SelectPortrait( GB: GameBoardPtr; M: GearPtr );
Procedure Rename_Mecha( GB: GameBoardPtr; NPC: GearPtr );
Procedure Redesig_Mecha( GB: GameBoardPtr; NPC: GearPtr );

Procedure FHQ_ThisWargearWasSelected( GB: GameBoardPtr; var LList: GearPtr; PC,M: GearPtr; RDP: RedrawProcedureType; BasicRedrawer: RedrawProcedureType_with_GearPtr );

Procedure PCDoPerformance( GB: GameBoardPtr; PC: GearPtr );
Procedure StartPerforming( GB: GameBoardPtr; PC: GearPtr );

Procedure UsableGearMenu( GB: GameBoardPtr; PC: GearPtr );


implementation

uses
{$IFDEF DEBUG}
	errmsg,
{$ENDIF DEBUG}
	utf8msg,ability,action,arenacfe,arenascript,gearutil,ghholder,
	ghmodule,ghprop,ghswag,interact,menugear,rpgdice,skilluse,texutil,
	description,ghweapon,ui4gh,narration,specialsys,ghsupport,
	ghintrinsic,effects,targetui,ghsensor,customization,
{$IFDEF ASCII}
	vidmap,vidmenus,vidinfo;
{$ELSE}
	colormenu,sdlmap,sdlmenus,sdlinfo;
{$ENDIF}

const
	WreckTarget_Cost: LongInt = 10000;

var
	ForceQuit: Boolean;
	EqpRPM,InvRPM: RPGMenuPtr;
	MenuA,MenuB: RPGMenuPtr;

	BP_Source: GearPtr;	{ Gear to appear in the INFO menu. }
	BP_SeekSibs: Boolean;	{ TRUE if the menu lists sibling gears; FALSE if it lists child gears. }
	BP_ActiveMenu: RPGMenuPtr;	{ The active menu. Used to determine the gear to show info about. }
	BP_GB: GameBoardPtr;
	BP_Redraw: RedrawProcedureType_with_GearPtr;
	MPB_Redraw: RedrawProcedureType;	{ Mecha Part Browser redraw procedure. }
						{ Since the mecha part browser may be called }
						{ from the main menu, the mecha editor, or }
						{ a dozen other places it needs to specify }
						{ a redrawer. }
	BP_TRADE_ITEM: GearPtr;
	FHQ_AR_PC: GearPtr;
	SP_GB: GameBoardPtr;
	SP_M: GearPtr;

	MenuDesc: String;

Function DefaultAtOp( Weapon: GearPtr ): Integer;
	{ Return the default Attack Options value for the weapon selected. }
var
	atop,PVal: Integer;
	Ammo: GearPtr;
begin
	AtOp := 0;
	PVal := WeaponBVSetting( Weapon );

	if ( Weapon^.G = GG_Weapon ) then begin
		if ( ( Weapon^.S = GS_Ballistic ) or ( Weapon^.S = GS_BeamGun ) ) and ( Weapon^.Stat[ STAT_BurstValue ] > 0 ) then begin
			if PVal = BV_Max then begin
				AtOp := Weapon^.Stat[ STAT_BurstValue ];
			end else if PVal = BV_Half then begin
				AtOp := Weapon^.Stat[ STAT_BurstValue ] div 2;
				if AtOp < 1 then AtOp := 1;
			end else if PVal = BV_Quarter then begin
				AtOp := Weapon^.Stat[ STAT_BurstValue ] div 4;
				if AtOp < 1 then AtOp := 1;
			end;
		end else if Weapon^.S = GS_Missile then begin
			Ammo := LocateGoodAmmo( Weapon );
			if Ammo = Nil then begin
				AtOp := 0;
			end else if PVal = BV_Max then begin
				AtOp := Ammo^.Stat[ STAT_AmmoPresent ] - 1;
				if AtOp < 0 then AtOp := 0;
			end else if PVal = BV_Half then begin
				AtOp := ( Ammo^.Stat[ STAT_AmmoPresent ] div 2 ) - 1;
				if AtOp < 0 then AtOp := 0;
			end else if PVal = BV_Quarter then begin
				AtOp := ( Ammo^.Stat[ STAT_AmmoPresent ] div 4 ) - 1;
				if AtOp < 0 then AtOp := 0;
			end;

		end;
	end;
	DefaultAtOp := atop;
end;


Procedure PlainRedraw;
	{ Miscellaneous menu redraw procedure. }
begin
	if BP_GB <> Nil then CombatDisplay( BP_GB );
end;

Procedure PlainRedraw_wM( M: GearPtr );
	{ Miscellaneous menu redraw procedure. }
begin
	if ( NIL <> BP_GB ) then begin
		if SHOW_NPCStatus_FieldHQ then begin
			CombatDisplay( M , BP_GB );
		end else begin
			CombatDisplay( BP_GB );
		end;
	end;
end;

Procedure MiscProcRedraw;
	{ Miscellaneous menu redraw procedure. The Eqp display will be shown; }
	{ the INV display won't be. }
var
	N: LongInt;
	Part: GearPtr;
begin
	BP_Redraw( BP_Source );
	DrawBPBorder;
	GameMsg( MenuDesc , ZONE_BackpackInstructions , InfoHilight );
	if ( BP_ActiveMenu <> Nil ) and ( BP_Source <> Nil ) then begin
		N := CurrentMenuItemValue( BP_ActiveMenu );
		if N > 0 then begin
			if BP_SeekSibs then Part := RetrieveGearSib( BP_Source , N )
			else Part := LocateGearByNumber( BP_Source , N );
			if Part <> Nil then begin
				BrowserInterfaceInfo( BP_GB , Part , ZONE_ItemsInfo );
			end;
		end;
	end;
	if EqpRPM <> Nil then begin
		DisplayMenu( EqpRPM , Nil );
	end;
end;

Procedure TradeProcRedraw;
var
	N: LongInt;
	Part: GearPtr;
begin
	BP_Redraw( BP_TRADE_ITEM );
	DrawBPBorder;
	GameMsg( MenuDesc , ZONE_BackpackInstructions , InfoHilight );
	if ( BP_ActiveMenu <> Nil ) and ( BP_Source <> Nil ) then begin
		N := CurrentMenuItemValue( BP_ActiveMenu );
		if N > 0 then begin
			if BP_SeekSibs then Part := RetrieveGearSib( BP_Source , N )
			else Part := LocateGearByNumber( BP_Source , N );
			if Part <> Nil then begin
				BrowserInterfaceInfo( BP_GB , Part , ZONE_ItemsInfo );
			end;
		end;
	end;
	if EqpRPM <> Nil then begin
		DisplayMenu( EqpRPM , Nil );
	end;
end;

Procedure InstallRedraw;
	{ Redrawer for installing a part into a mecha. }
begin
	BP_Redraw( BP_Source );
	DrawBPBorder;
	BrowserInterfaceInfo( BP_GB , BP_Source , ZONE_ItemsInfo );
	if EqpRPM <> Nil then begin
		DisplayMenu( EqpRPM , Nil );
	end;
end;

Procedure EqpRedraw;
	{ Show Inventory, select Equipment. }
var
	N: LongInt;
	Part: GearPtr;
begin
	BP_Redraw( BP_Source );
	DrawBPBorder;
	DisplayMenu( InvRPM , Nil );
	GameMsg( MenuDesc , ZONE_BackpackInstructions , InfoHilight );
	if ( BP_ActiveMenu <> Nil ) and ( BP_Source <> Nil ) then begin
		N := CurrentMenuItemValue( BP_ActiveMenu );
		if N > 0 then begin
			if BP_SeekSibs then Part := RetrieveGearSib( BP_Source , N )
			else Part := LocateGearByNumber( BP_Source , N );
			if Part <> Nil then begin
				BrowserInterfaceInfo( BP_GB , Part , ZONE_ItemsInfo );
			end;
		end;
	end;
end;

Procedure ThisItemRedraw;
	{ A specific item was selected, and its location stored in BP_Source. }
begin
	BP_Redraw( FindMaster( BP_Source ) );
	DrawBPBorder;
	GameMsg( MenuDesc , ZONE_BackpackInstructions , InfoHilight );
	if BP_Source <> Nil then BrowserInterfaceInfo( BP_GB , BP_Source , ZONE_ItemsInfo );

	if EqpRPM <> Nil then begin
		DisplayMenu( EqpRPM , Nil );
	end;
end;

Procedure GetItemRedraw;
begin
	CombatDisplay( BP_GB );
	DrawGetItemBorder;
end;


Procedure ThisWargearRedraw;
	{ A specific item was selected, and its location stored in BP_Source. }
begin
	BP_Redraw( BP_Source );
	SetupFHQDisplay;
	if BP_Source <> Nil then BrowserInterfaceInfo( BP_GB , BP_Source , ZONE_ItemsInfo );
end;

Procedure MechaPartEditorRedraw;
	{ Show Inventory, select Equipment. }
var
	Part: GearPtr;
begin
	BP_Redraw( BP_Source );
	SetupFHQDisplay;
	if ( BP_ActiveMenu <> Nil ) and ( BP_Source <> Nil ) then begin
		if BP_SeekSibs then Part := RetrieveGearSib( BP_Source , CurrentMenuItemValue( BP_ActiveMenu ) )
		else Part := LocateGearByNumber( BP_Source , CurrentMenuItemValue( BP_ActiveMenu ) );
		if Part <> Nil then begin
			BrowserInterfaceInfo( BP_GB , Part , ZONE_ItemsInfo );
		end;
	end;
end;

Procedure PartBrowserRedraw;
	{ Redraw the screen for the part browser. }
var
	Part: GearPtr;
begin
	if MPB_Redraw <> Nil then MPB_Redraw;
	SetupFHQDisplay;
	if ( BP_ActiveMenu <> Nil ) and ( BP_Source <> Nil ) then begin
		if BP_SeekSibs then Part := RetrieveGearSib( BP_Source , CurrentMenuItemValue( BP_ActiveMenu ) )
		else Part := LocateGearByNumber( BP_Source , CurrentMenuItemValue( BP_ActiveMenu ) );
		if Part <> Nil then begin
			BrowserInterfaceInfo( BP_GB , Part , ZONE_ItemsInfo );
		end;
	end;
end;

Procedure MysteryBrowserRedraw;
	{ Redraw the screen for the mystery display. }
	{ This is the screen that shows no real information when the PC doesn't have }
	{ the correct software to view a target. }
begin
	if MPB_Redraw <> Nil then MPB_Redraw;
	SetupFHQDisplay;
	if ( BP_Source <> Nil ) then begin
		BrowserInterfaceMystery( BP_GB , BP_Source , ZONE_ItemsInfo );
	end;
end;

Procedure TradeItemsRedraw;
	{ Trade Items menu redraw procedure. The MenuA and MenuB will both be shown. }
var
	N: LongInt;
	Part: GearPtr;
begin
	CombatDisplay( BP_GB );
	DrawBPBorder;
	GameMsg( MenuDesc , ZONE_BackpackInstructions , InfoHilight );
	DisplayMenu( MenuA , Nil );
	DisplayMenu( MenuB , Nil );

	if ( BP_ActiveMenu <> Nil ) and ( BP_Source <> Nil ) then begin
		N := CurrentMenuItemValue( BP_ActiveMenu );
		if N > 0 then begin
			Part := RetrieveGearSib( BP_Source^.InvCom , N );
			if Part <> Nil then begin
				BrowserInterfaceInfo( BP_GB , Part , ZONE_ItemsInfo );
			end;
		end;
	end;
end;


Procedure GatherFieldHQ( GB: GameBoardPtr );
	{ The PC wants to open his FieldHQ. Look through the current city and gather }
	{ up everything belonging to the PC team. Deposit these gears on the gameboard. }
	Procedure GatherFromScene( S: GearPtr );
		{ Gather any gears belonging to the PC from this scene. }
		{ Move them to the gameboard. }
	var
		M,M2: GearPtr;
	begin
		M := S^.InvCom;
		while M <> Nil do begin
			M2 := M^.Next;

			if ( M^.G >= 0 ) and ( NAttValue( M^.NA , NAG_Location , NAS_Team ) = NAV_DefPlayerTeam ) then begin
				DelinkGear( S^.InvCom , M );
				DeployGear( GB , M , False );
			end;

			M := M2;
		end;
	end;
	Procedure SearchAlongPath( SList: GearPtr );
		{ Search along this path for scenes. }
	begin
		while SList <> Nil do begin
			if SList^.G = GG_Scene then begin
				GatherFromScene( SList );
				SearchAlongPath( SList^.SubCom );
			end;
			SList := SList^.Next;
		end;
	end;
var
	City: GearPtr;
begin
	City := FindRootScene( GB^.Scene );
	if City <> Nil then begin
		GatherFromScene( City );
		SearchAlongPath( City^.SubCom );
	end;
end;

Procedure GivePartToPC( var LList: GearPtr; Part, PC: GearPtr; const Paret_Part: GearPtr; Chain_Part: GearPtr );
	{ Give the specified part to the PC. If the part cannot be }
	{ held by the PC, store it so that it can be recovered using }
	{ the FieldHQ Wargear Explorer. }
	{ The part should be delinked already. }
var
	P2,Pilot: GearPtr;
begin
	if PC^.G = GG_Mecha then Pilot := LocatePilot( PC )
	else Pilot := Nil;
	if ( Part^.G = GG_Set ) then begin
		while Part^.SubCom <> Nil do begin
			P2 := Part^.SubCom;
			DelinkGear( Part^.SubCom , P2 );
			GivePartToPC( LList , P2 , PC , NIL , NIL );
		end;
		while Part^.InvCom <> Nil do begin
			P2 := Part^.InvCom;
			DelinkGear( Part^.InvCom , P2 );
			GivePartToPC( LList , P2 , PC , NIL , NIL );
		end;
	end else if ( Pilot <> Nil ) and IsLegalInvCom( Pilot , Part ) then begin
		StripNAtt( Part , NAG_Location );
		StripNAtt( Part , NAG_EpisodeData );
		if ( Paret_Part <> Pilot ) then begin
			InsertInvCom( Pilot , Part );
		end else begin
			CancelDelinkGear( Pilot , Pilot^.InvCom , Chain_Part , Part );
		end;
	end else if ( PC <> Nil ) and IsLegalInvCom( PC , Part ) then begin
		StripNAtt( Part , NAG_Location );
		StripNAtt( Part , NAG_EpisodeData );
		if ( Paret_Part <> PC ) then begin
			InsertInvCom( PC , Part );
		end else begin
			CancelDelinkGear( PC , PC^.InvCom , Chain_Part , Part );
		end;
	end else begin
		{ If the PC can't carry this equipment, }
		{ stick it off the map. }
		SetNAtt( Part^.NA , NAG_Location , NAS_Team , 1 );
		SetNAtt( Part^.NA , NAG_Location , NAS_X , 0 );
		SetNAtt( Part^.NA , NAG_Location , NAS_Y , 0 );
		AppendGear( LList , Part );
	end;
end;

Procedure GivePartToPC( var LList: GearPtr; Part, PC: GearPtr );
begin
	GivePartToPC( LList , Part , PC , NIL , NIL );
end;

Procedure GivePartToPC( GB: GameBoardPtr; Part, PC: GearPtr );
	{ Call the above procedure, with GB^.Meks as the LList. }
begin
	GivePartToPC( GB^.Meks , Part , PC , NIL , NIL );
end;

{$IFNDEF ASCII}
Procedure SelectColors( M: GearPtr; Redrawer: RedrawProcedureType_with_GearPtr );
	{ The player wants to change the colors for this part. Make it so. }
	{ The menu will be placed in the Menu area; assume the redrawer will }
	{ show whatever changes are made here. }
var
	portraitname,oldcolor,newcolor: String;
begin
	portraitname := InfoImageName( M );
	oldcolor := SAttArrayValue( M^.SA , TAG_SDL_COLORS );

	if M^.G = GG_Character then begin
		newcolor := SelectColorPalette( colormenu_mode_character , portraitname , oldcolor , 100 , 150 , 0, Redrawer, M );
		SetSAttArray( M^.SA , TAG_SDL_COLORS_TEAM_CLOTHING , '1' );
	end else if M^.G = GG_Mecha then begin
		newcolor := SelectColorPalette( colormenu_mode_mecha , portraitname , oldcolor , 100 , 150 , 0, Redrawer, M );
	end else begin
		newcolor := SelectColorPalette( colormenu_mode_allcolors , portraitname , oldcolor , 100 , 150 , 0, Redrawer, M );
	end;

	if ( 0 < Length(newcolor) ) then begin
		SetSAttArray( M^.SA , TAG_SDL_COLORS , newcolor );
	end;
end;

Procedure SelectSprite( M: GearPtr; Redrawer: RedrawProcedureType );
	{ The player wants to change the colors for sprite for this character. }
	{ The menu will be placed in the Menu area; assume the redrawer will }
	{ show whatever changes are made here. }
var
	RPM: RPGMenuPtr;
	fname: String;
begin
	RPM := CreateRPGMenu( MenuItem , MenuSelect , @ZONE_CharViewMenu );
	if CHEAT_Unlimited_Sprite then begin
		BuildFileMenu( RPM , Graphics_Directory + 'cha_*.*' , False );
	end else begin
		if NAttValue( M^.NA , NAG_CharDescription , NAS_Gender ) = NAV_Female then begin
			BuildFileMenu( RPM , Graphics_Directory + 'cha_f_*.*' , False );
		end else if NAttValue( M^.NA , NAG_CharDescription , NAS_Gender ) = NAV_Male then begin
			BuildFileMenu( RPM , Graphics_Directory + 'cha_m_*.*' , False );
		end else begin
			BuildFileMenu( RPM , Graphics_Directory + 'cha_*_*.*' , False );
		end;
	end;
	AddRPGMenuItem( RPM , MsgString( 'EXIT' ) , SELECTMENU_Cancel );

	fname := SelectFile( RPM , Redrawer );

	if fname <> '' then begin
		SetSAttArray( M^.SA , 'SDL_SPRITE' , fname );
	end;

	DisposeRPGMenu( RPM );
end;
{$ENDIF}

Procedure AddRepairOptions( RPM: RPGMenuPtr; PC,Item: GearPtr );
	{ Check the object in question, then add options to the }
	{ provided menu if the item is in need of repairs which the }
	{ PC can provide. Repair items will be numbered 100 + RSN }
var
	N: Integer;
begin
	PC := LocatePilot( PC );
	if PC <> Nil then begin
		for N := 1 to NumSkill do begin
			{ The repair option will only be added to the menu if: }
			{ - The PC has the required skill. }
			{ - The item is in need of repair (using this skill). }
			if ( SkillMan[N].Usage = USAGE_Repair ) and ( NAttValue( PC^.NA , NAG_Skill , N ) > 0 ) and ( RepairNeededBySkill( Item , N ) > 0 ) then begin
				AddRPGMenuItem( RPM , ReplaceHash( UTF8_MsgString( 'AddRepairOptions' , 'Repair' ) , MsgString( 'SKILLNAME_' + BStr( N ) ) ) , 100 + N );
			end;
		end;
	end;
end;

Function DoFieldRepair( GB: GameBoardPtr; PC , Item: GearPtr; Skill: Integer ): Boolean;
	{ The PC is going to use one of the repair skills. Call the }
	{ standard procedure, then print output. }
	{ Return TRUE if repair fuel found, or FALSE otherwise. }
var
	msg: String;
	Dmg0,DDmg,T: LongInt;
	SFX_Check: Array [1..Num_Status_FX] of Boolean;
	RepairFuelFound,NoStatusCured: Boolean;
	Target: GearPtr;
begin
	{ Record the initial state of the repair target. }
	Dmg0 := AmountOfDamage( Item , True );
	for t := 1 to Num_Status_FX do SFX_Check[ t ] := HasStatus( Item , T );

	Target := NIL;
	RepairFuelFound := UseRepairSkill( GB , PC , Item , Skill , Target );
	if ( NIL = Target ) then begin
		Target := Item;
	end;
	if NAttValue( PC^.NA , NAG_Location , NAS_Team ) = NAV_DefPlayerTeam then begin
		msg := UTF8_MsgString( 'DoFieldRepair', 'PCREPAIR_UseSkill' );
	end else begin
		msg := UTF8_MsgString( 'DoFieldRepair', 'NPCREPAIR_UseSkill' );
	end;
	msg := ReplaceHash( msg , PilotName( PC ) , MsgString( 'SkillName_' + BStr( Skill ) ) , GearName( FindMaster( Target ) ) );

	{ Report the final state of the repair target. }
	DDmg := Dmg0 - AmountOfDamage( Item , True );

	{ Inform the user of the success. }
	if ( Item^.G = GG_Character ) and Destroyed( Item ) then begin
		msg := msg + ' ' + ReplaceHash( MsgString( 'PCREPAIR_DEAD' ) , GearName( Item ) );
	end else if not RepairFuelFound then begin
		msg := msg + ' ' + MsgString( 'PCREPAIR_NoRepairFuel' );
	end else if DDmg > 0 then begin
		msg := msg + ' ' + ReplaceHash( MsgString( 'PCREPAIR_Success' ) , BStr( DDmg ) );
	end;

	{ Assume TRUE unless proven FALSE. }
	NoStatusCured := True;
	for t := 1 to Num_Status_FX do begin
		if SFX_Check[ t ] and not HasStatus( Item , T ) then begin
			{ This status effect was cured. Add it to the list. }
			NoStatusCured := FALSE;
		end;
	end;

	{ If no damage was healed and no status was cured, this was a big waste of time. }
	if ( DDMg = 0 ) and NoStatusCured then begin
		msg := msg + ' ' + MsgString( 'PCREPAIR_Failure' )
	end else if NotDestroyed( Item ) and not NoStatusCured then begin
		if ( GG_Mecha = Item^.G ) then begin
			msg := msg + ' ' + ReplaceHash( UTF8_MsgString( 'DoFieldRepair' , 'STATUS_Remove_Mecha' ) , GearName( Item ) );
		end else begin
			msg := msg + ' ' + ReplaceHash( UTF8_MsgString( 'DoFieldRepair' , 'STATUS_Remove' ) , GearName( Item ) );
		end;
	end;

	DialogMsg( msg );

	DoFieldRepair := RepairFuelFound;
end;

Function ShakeDown( GB: GameBoardPtr; Part: GearPtr; X,Y: Integer ): LongInt;
	{ This is the workhorse for this function. It does the }
	{ dirty work of separating inventory from (former) owner. }
const
	V_MAX = 2147483647;
	V_MIN = -2147483648;
var
	cash: Int64;
	SPart: GearPtr;		{ Sub-Part }
begin
	{ Start by removing the cash from this part. }
	cash := NAttValue( Part^.NA , NAG_Experience , NAS_Credits );
	SetNAtt( Part^.NA , NAG_Experience , NAS_Credits , 0 );
	SetNAtt( Part^.NA , NAG_EpisodeData , NAS_Ransacked , 1 );

	{ Remove all InvComs, and place them on the map. }
	While Part^.InvCom <> Nil do begin
		SPart := Part^.InvCom;
		DelinkGear( Part^.InvCom , SPart );
		{ If this invcom isn't destroyed, put it on the }
		{ ground for the PC to pick up. Otherwise delete it. }
		if NotDestroyed( SPart ) then begin
			SetNAtt( SPart^.NA , NAG_Location , NAS_X , X );
			SetNAtt( SPart^.NA , NAG_Location , NAS_Y , Y );
			SPart^.Next := GB^.Meks;
			GB^.Meks := SPart;
		end else begin
			DisposeGear( SPart );
		end;
	end;

	{ Shake down this gear's subcoms. }
	SPart := Part^.SubCOm;
	while SPart <> Nil do begin
		if SPart^.G <> GG_Cockpit then cash := cash + ShakeDown( GB , SPart , X , Y );
		SPart := SPart^.Next;
	end;

	if (V_MAX < Cash) then begin
		Cash := V_MAX;
	end else if (Cash < V_MIN) then begin
		Cash := V_MIN;
	end;
	ShakeDown := Cash;
end;


Function Ransack( GB: GameBoardPtr; X,Y: Integer ): LongInt;
	{ Yay! Loot and pillage! This function has two purposes: }
	{ first, it separates all Inventory gears from any non-operational }
	{ masters standing in this tile. Secondly, it collects the }
	{ money from all those non-operational masters and returns the }
	{ total amount as the function result. }
const
	V_MAX = 2147483647;
var
	it: Int64;
	Mek: GearPtr;
begin
	it := 0;

	Mek := GB^.Meks;

	while Mek <> Nil do begin
		{ If this is a broken-down master, check to see if it's }
		{ one we want to pillage. }
		if IsMasterGear( Mek ) and not GearOperational( Mek ) then begin
			{ We will ransack this gear if it's in the correct location. }
			if ( NAttValue( Mek^.NA , NAG_Location , NAS_X ) = X ) and ( NAttValue( Mek^.NA , NAG_Location , NAS_Y ) = Y ) then begin
				it := it + ShakeDown( GB , Mek , X , Y );
			end;
		end else if ( Mek^.G = GG_MetaTerrain ) and ( ( Mek^.Stat[ STAT_Lock ] = 0 ) or Destroyed( Mek ) ) then begin
			{ Metaterrain gets ransacked if it's unlocked, }
			{ or wrecked. }
			if ( NAttValue( Mek^.NA , NAG_Location , NAS_X ) = X ) and ( NAttValue( Mek^.NA , NAG_Location , NAS_Y ) = Y ) then begin
				it := it + ShakeDown( GB , Mek , X , Y );
			end;
		end;
		Mek := Mek^.Next;
	end;

	if (V_MAX < it) then begin
		it := V_MAX;
	end;
	Ransack := it;
end;

Function Handless( Mek: GearPtr ): Boolean;
	{ Return TRUE if Mek either has no hands or can't use its hands }
	{ at the moment (say, because it's transformed into tank mode). }
	{ Return TRUE if Mek has hands and they are in perfect working order. }
var
	Hand: GearPtr;
begin
	Hand := SeekActiveIntrinsic( Mek , GG_Holder , GS_Hand , not( CHEAT_No_CockpitBarrier_Hand ) );
	if Hand = Nil then Handless := True
	else Handless := not InGoodModule( Hand );
end;

Function SelectVisibleItem( GB: GameBoardPtr; PC: GearPtr; X,Y: Integer; var FlagAll: Boolean ): GearPtr;
	{ Attempt to select a visible item from gameboard tile X,Y. }
	{ If more than one item is present, prompt the user for which one }
	{ to pick up. }
var
	N,T: Integer;
	RPM: RPGMenuPtr;
begin
	FlagAll := False;

	{ First count the number of items in this spot. }
	N := NumVisibleItemsAtSpot( GB , X , Y );

	{ If it's just 0 or 1, then our job is simple... }
	if N = 0 then begin
		SelectVisibleItem := Nil;
	end else if N = 1 then begin
		SelectVisibleItem := FindVisibleItemAtSpot( GB , X , Y );

	{ If it's more than one, better create a menu and let the user }
	{ pick one. }
	end else if N > 1 then begin
		DialogMsg( MsgString( 'GET_WHICH_ITEM?' ) );
		RPM := CreateRPGMenu( MenuItem , MenuSelect , @ZONE_GetItemMenu );
		for t := 1 to N do begin
			AddRPGMenuItem( RPM , GearName( GetVisibleItemAtSpot( GB , X , Y , T ) ) , T );
		end;
		AlphaKeyMenu( RPM );
		if ( 1 < RPM^.NumItem ) then begin
			AddRPGMenuItem_Top( RPM , UTF8_MsgString( 'SelectVisibleItem', 'PickUpAll' ) , -5 );
		end;

		BP_GB := GB;
		N := SelectMenu( RPM , @GetItemRedraw );

		DisposeRPGMenu( RPM );
		if ( SELECTMENU_Enable <= N ) then begin
			SelectVisibleItem := GetVisibleItemAtSpot( GB , X , Y , N );
		end else if ( -5 = N ) then begin
			FlagAll := True;
			SelectVisibleItem := NIL;
		end else begin
			SelectVisibleItem := Nil;
		end;
	end;
end;

Procedure PCGetItem( GB: GameBoardPtr; PC: GearPtr );
	{ The PC will attempt to pick up something lying on the ground. }

var
	IsHandless: Boolean;
	Cash: LongInt;

	Procedure GetItem( Item: GearPtr );
	var
		NID: LongInt;
	begin
		if ( NIL <> Item ) then begin
			if IsLegalInvCom( PC , Item ) then begin
				DelinkGear( GB^.Meks , Item );

				{ Clear the item's location values. }
				StripNAtt( Item , NAG_Location );

				InsertInvCom( PC , Item );
				{ Clear the home, to prevent wandering items. }
				SetSAttArray( Item^.SA , 'HOME' , '' );

				if ( PC^.G = GG_Mecha ) and IsHandless then begin
					DialogMsg( ReplaceHash( MsgString( 'YOU_STRAP_?' ) , GearName( Item ) ) );
				end else begin
					DialogMsg( ReplaceHash( MsgString( 'YOU_GET_?' ) , GearName( Item ) ) );
				end;

				NID := NAttValue( Item^.NA , NAG_Narrative , NAS_NID );
				if NID <> 0 then SetTrigger( GB , TRIGGER_GetItem + BStr( NID ) );
			end else if Cash = 0 then begin
				DialogMsg( ReplaceHash( MsgString( 'CANT_GET_?' ) , GearName( Item ) ) );
			end;
		end else if Cash = 0 then begin
			DialogMsg( UTF8_MsgString('PCGetItem', 'No item found') );
		end;

		if Cash > 0 then begin
			DialogMsg( ReplaceHash( MsgString( 'YouFind$' ) , BStr( Cash ) ) );
			AddNAtt( LocatePilot( PC )^.NA , NAG_Experience , NAS_Credits , Cash );
			Cash := 0;
		end;

		{ Picking up an item takes time. }
		{ More time if you're doing it without hands. }
		if IsHandless then begin
			WaitAMinute( GB , PC , ReactionTime( PC ) * 3 );
		end else begin
			WaitAMinute( GB , PC , ReactionTime( PC ) );
		end;
	end;
	Procedure GetAllItem( P: Point );
	var
		Match: LPattern;
		M, M_Next: GearPtr;
	begin
		Match.X := P.X;
		Match.Y := P.Y;
		Match.Z := -10;
		Match.Only_Visibles := True;
		Match.Only_Masters := LP_MustNotBeMaster;

		M := GB^.Meks;
		while ( NIL <> M ) do begin
			M_Next := M^.Next;
			if GearMatchesLPattern( GB , M , Match ) then begin
				GetItem( M );
			end;
			M := M_Next;
		end;
	end;
var
	P: Point;
	item: GearPtr;
	FlagAll: Boolean;
begin
	IsHandless := Handless( PC );
	if IsHandless and not IsSafeArea( GB ) then begin
		{ Start by checking something that other RPGs would }
		{ just assume- does the PC have any hands? }
		DialogMsg( MsgString( 'HANDLESS_PICKUP' ) );

	end else begin
		P := GearCurrentLocation( PC );

		{ Before attempting to get an item, ransack whatever }
		{ fallen enemies lie in this spot. }
		Cash := Ransack( GB , P.X , P.Y );

		{ Perform an immediate vision check- without it, items }
		{ freed by the Ransack procedure above will remain unseen. }
		VisionCheck( GB , PC );

		Item := SelectVisibleItem( GB , PC , P.X , P.Y , FlagAll );

		if FlagAll then begin
			GetAllItem( P );
		end else begin
			GetItem( Item );
		end;
	end;
end;

Procedure PCTradeItems( GB: GameBoardPtr; PC,Target: GearPtr );
	{ The PC will attempt to trade items with TARGET. }
	Procedure SetupMenu( RPM: RPGMenuPtr; M: GearPtr );
		{ The setup procedure for both menus is the same, so here }
		{ it is. }
	begin
		BuildSiblingMenu_with_TAG( RPM , M^.InvCom );
		{ RPMSortAlpha( RPM ); } { In UTF-8, this sort causes a result that I cannot assume. }

		{ If the menu is empty, add a message saying so. }
		If RPM^.NumItem < 1 then begin
			AddRPGMenuItem( RPM , UTF8_MsgString( 'PCTradeItems', 'no inventory items' ) , SELECTMENU_Cancel );
		end else begin
			AlphaKeyMenu( RPM );
{$IFDEF DEBUG}
			if ( 2 <= RPM^.NumItem ) then begin
				AddRPGMenuItem_Top( RPM , #$0 + '[edit menu order]' , SELECTMENU_MenuOrder_Edit );
			end;
{$ENDIF DEBUG}
		end;

		{ Add the menu keys. }
		AddRPGMenuKey( RPM , KeyMap[ KMC_SwitchMenu ].KCode , -2 );

		RPM^.editable_menuorder := True;

		if ( 1 < RPM^.NumItem ) then begin
			AddRPGMenuItem_Top( RPM , UTF8_MsgString( 'PCTradeItems', 'TransferAll' ) , -4 )
		end;
	end;
	Procedure TransferItem( Source , Item , Destination: GearPtr );
		{ If possible, move ITEM from SOURCE to DESTINATION. }
	var
		NID: LongInt;
	begin
		if IsLegalInvCom( Destination , Item ) then begin
			{ Clear the item's location values. }
			StripNAtt( Item , NAG_Location );

			DelinkGear( Source^.InvCom , Item );
			InsertInvCom( Destination , Item );

			{ Clear the home, to prevent wandering items. }
			SetSAttArray( Item^.SA , 'HOME' , '' );
			if Destination = PC then begin
				DialogMsg( ReplaceHash( MsgString( 'YOU_GET_?' ) , GearName( Item ) ) );
			end else begin
				DialogMsg( ReplaceHash( MsgString( 'YOU_PUT_?' ) , GearName( Item ) ) );
			end;

			NID := NAttValue( Item^.NA , NAG_Narrative , NAS_NID );
			if NID <> 0 then SetTrigger( GB , TRIGGER_GetItem + BStr( NID ) );

			if GB <> Nil then WaitAMinute( GB , PC , ReactionTime( PC ) );

		end else begin
			DialogMsg( ReplaceHash( MsgString( 'CANT_GET_?' ) , GearName( Item ) ) );
		end;
	end;
var
	item: GearPtr;
	PCMenu,TarMenu: RPGMenuPtr;
	EscMenu,UseTarInv: Boolean;
	PSI,PTI,TSI,TTI: Integer;
	N,V: LongInt;
	Part, Part_Next: GearPtr;
	Desc_MenuA: String;
	Desc_MenuB: String;
begin
	{ Error check. }
	{ PC and Target must be non-nil; they must also be separate entities, or else }
	{ weird things will result. }
	if ( PC = Nil ) or ( Target = Nil ) or ( FindRoot( PC ) = FindRoot( Target ) ) then Exit;

	{ Initialize variables. }
	BP_GB := GB;
	UseTarInv := True;

	{ Keep going back and forth between the PC and the target until }
	{ the player hits ESCape. }
	EscMenu := False;
	PSI := 1;
	PTI := 1;
	TSI := 1;
	TTI := 1;
	repeat
		{ Create the two menus. }
		PCMenu := CreateRPGMenu( MenuItem , MenuSelect , @ZONE_EqpMenu );
		MenuA := PCMenu;
		SetupMenu( PCMenu , PC );

		TarMenu := CreateRPGMenu( MenuItem , MenuSelect , @ZONE_InvMenu );
		MenuB := TarMenu;
		SetupMenu( TarMenu , Target );

		{ Determine which of the two menus is going to be our active one, }
		{ based on the UseTarInv variable. }
		if UseTarInv then begin
			BP_ActiveMenu := TarMenu;
			BP_ActiveMenu^.SelectItem := TSI;
			BP_ActiveMenu^.TopItem    := TTI;
			BP_Source := Target;
			Desc_MenuA := UTF8_MsgString( 'BACKPACK_Directions' , 'MenuB' );
			Desc_MenuB := UTF8_MsgString( 'BACKPACK_Directions' , 'MenuA' );
		end else begin
			BP_ActiveMenu := PCMenu;
			BP_ActiveMenu^.SelectItem := PSI;
			BP_ActiveMenu^.TopItem    := PTI;
			BP_Source := PC;
			Desc_MenuA := UTF8_MsgString( 'BACKPACK_Directions' , 'MenuA' );
			Desc_MenuB := UTF8_MsgString( 'BACKPACK_Directions' , 'MenuB' );
		end;

		MenuDesc := ReplaceHash( UTF8_MsgString( 'BACKPACK' , 'Directions' ) , KeyMapEncode( '' , KeyMap[ KMC_SwitchMenu ].KCode ) , Desc_MenuA , Desc_MenuB );
		N := SelectMenu( BP_ActiveMenu , @TradeItemsRedraw );
		if UseTarInv then begin
			TSI := BP_ActiveMenu^.SelectItem;
			TTI := BP_ActiveMenu^.TopItem;
			V := RPMLocateByPosition( BP_ActiveMenu , TSI )^.value;
		end else begin
			PSI := BP_ActiveMenu^.SelectItem;
			PTI := BP_ActiveMenu^.TopItem;
			V := RPMLocateByPosition( BP_ActiveMenu , PSI )^.value;
		end;

		if ( SELECTMENU_Enable < N ) then begin
			{ An item has been selected. Find it, then attempt to swap from }
			{ target to PC or vice versa. }
			if UseTarInv then begin
				Item := RetrieveGearSib( Target^.InvCom , N );
			end else begin
				Item := RetrieveGearSib( PC^.InvCom , N );
			end;
			if CheckAlongPath_DisallowTransfering( Item ) then begin
				DialogMsg( ReplaceHash( UTF8_MsgString( 'PCTradeItems' , 'YouCannotTransferIt' ) , GearName( Item ) ) );
			end else begin
				if UseTarInv then begin
					TransferItem( Target , Item , PC );
				end else begin
					TransferItem( PC , Item , Target );
				end;
			end;

		end else if ( SELECTMENU_Cancel = N ) then begin
			{ An Escape has been recieved. Quit this procedure. }
			EscMenu := True;
		end else if N = -2 then begin
			{ A menu swap has been requested. Change the active menu. }
			UseTarInv := Not UseTarInv;
		end else if N = -4 then begin
			if UseTarInv then begin
				Part := Target^.InvCom;
			end else begin
				Part := PC^.InvCom;
			end;
			while ( NIL <> Part ) do begin
				Part_Next := Part^.Next;
				if CheckAlongPath_DisallowTransfering( Part ) then begin
					DialogMsg( ReplaceHash( UTF8_MsgString( 'PCTradeItems' , 'YouCannotTransferIt' ) , GearName( Part ) ) );
				end else begin
					if UseTarInv then begin
						TransferItem( Target , Part , PC );
					end else begin
						TransferItem( PC , Part , Target );
					end;
				end;
				Part := Part_Next;
			end;
		end else if ( SELECTMENU_MenuOrder_Edit = N ) then begin
			if ( SELECTMENU_Enable <= V ) then begin
				Part := RetrieveGearSib( BP_Source^.InvCom , V );
				if UseTarInv then begin
					SwapMenu( @ZONE_InvMenu , Part );
				end else begin
					SwapMenu( @ZONE_EqpMenu , Part );
				end;
			end else begin
				if UseTarInv then begin
					SwapMenu( @ZONE_InvMenu , BP_Source^.InvCom , NIL );
				end else begin
					SwapMenu( @ZONE_EqpMenu , BP_Source^.InvCom , NIL );
				end;
			end;
		end;

		{ Dispose the two menus. }
		DisposeRPGMenu( PCMenu );
		DisposeRPGMenu( TarMenu );
	until EscMenu;
end;


Procedure Check_WeaponsNeedHand( PC, Item: GearPtr );
	Function HasFreeHand( LList: GearPtr ): Boolean;
	var
		HandFound: Boolean;
	begin
		HandFound := False;
		while ( NIL <> LList ) and ( not HandFound ) do begin
			if ( GG_Holder = LList^.G ) and ( GS_Hand = LList^.S ) and ( PC^.Scale <= LList^.Scale ) and ( NIL = LList^.InvCom ) then begin
				HandFound := True;
			end else begin
				HandFound := HasFreeHand( LList^.SubCom );
			end;
			LList := LList^.Next;
		end;
		HasFreeHand := HandFound;
	end;

	Procedure CheckAlongPath( Part: GearPtr );
	var
		Modifier_with_Hand: Integer;
		NeedHand: Boolean;
	begin
		while ( NIL <> Part ) do begin
			if ( Item <> Part ) and ( GG_Weapon = Part^.G ) then begin
				WeaponWeightModifier( Part, DefaultAtOp( Part ), PC, NeedHand, Modifier_with_Hand );
				if NeedHand then begin
					DialogMsg( ReplaceHash( UTF8_MsgString( 'Check_WeaponsNeedHand', 'KEIKOKU' ) , GearName( Part ) ) );
				end;
			end;
			CheckAlongPath( Part^.InvCom );
			CheckAlongPath( Part^.SubCom );
			Part := Part^.Next;
		end;
	end;
begin
	if HasFreeHand( PC^.SubCom ) then exit;
	CheckAlongPath( PC^.SubCom );
end;


Procedure CreateInvMenu( PC: GearPtr );
	{ Allocate the Inventory menu and fill it up with the PC's inventory. }
begin
	if InvRPM <> Nil then DisposeRPGMenu( InvRPM );
	InvRPM := CreateRPGMenu( MenuItem , MenuSelect , @ZONE_InvMenu );
	InvRPM^.Mode := RPMNoCleanup;
	InvRPM^.editable_menuorder := True;
	BuildInventoryMenu( InvRPM , PC , False , SHOW_InvMenu_SubItem );
	{ RPMSortAlpha( InvRPM ); } { In UTF-8, this sort causes a result that I cannot assume. }

	{ If the menu is empty, add a message saying so. }
	If InvRPM^.NumItem < 1 then begin
		AddRPGMenuItem( InvRPM , UTF8_MsgString( 'CreateInvMenu', 'no inventory items' ) , SELECTMENU_Cancel );
	end else begin
		AlphaKeyMenu( InvRPM );
{$IFDEF DEBUG}
		if ( 2 <= InvRPM^.NumItem ) then begin
			AddRPGMenuItem_Top( InvRPM , #$0 + '[edit menu order]' , SELECTMENU_MenuOrder_Edit );
		end;
{$ENDIF DEBUG}
	end;

	{ Add the menu keys. }
	AddRPGMenuKey( InvRPM , KeyMap[ KMC_SwitchMenu ].KCode , -2 );
	if ( 1 < InvRPM^.NumItem ) then begin
		AddRPGMenuItem_Top( InvRPM , UTF8_MsgString( 'ThisItemWasSelected', 'DropAll' ), -3 );
		AddRPGMenuItem_Top( InvRPM , UTF8_MsgString( 'ThisItemWasSelected', 'TransferAll' ), -4 );
	end;
end;

Procedure CreateEqpMenu( PC: GearPtr );
	{ Allocate the equipment menu and fill it up with the PC's gear. }

	Procedure CreateEqpMenu_MassMeter( RPM: RPGMenuPtr; PC: GearPtr);
		{ On top of the backpack seperator, display the current inventory mass, and the carrying capacity }
		{ Origin : Michael }
		{ http://gearheadrpg.com/forum/index.php?action=vthread&forum=1&topic=789 }
		{ ftp://ftp.ocis.net/pub/users/ldeutsch/ghpatches/gh-1100-massmeter1.diff }
	var
		MassString: String;
		IMass, EMass, EV, Limit, Limit2: LongInt;
	begin
		IMass := IntrinsicMass( PC );
		EMass := EquipmentMass( PC );
		EV := GearEncumberance( PC );

		Limit := EV;
		if ( GG_Character = PC^.G ) then begin
			Limit := EV * 2;	{ Maximum weight before penalty starts }
		end;
		Limit2 := EV + Limit;	{ Where penalty gets worse }

		MassString := ReplaceHash( UTF8_MsgString( 'CreateEqpMenu' , 'MassMeter' ),
					MakeMassString( IMass , PC^.Scale ),
					MakeMassString( EMass , PC^.Scale ) );
		AddRPGMenuItem_Top( RPM, #$0 + MassString, 0 );

		MassString := ReplaceHash( UTF8_MsgString( 'CreateEqpMenu' , 'MassMeter_Limit' ) , MakeMassString( Limit , PC^.Scale ) );
		if ( Limit2 < EMass ) then begin
			MassString := MassString + UTF8_MsgString( 'CreateEqpMenu' , 'MassMeter_LimitOver2' );
		end else if ( Limit < EMass ) then begin
			MassString := MassString + UTF8_MsgString( 'CreateEqpMenu' , 'MassMeter_LimitOver' );
		end else begin
		end;
		AddRPGMenuItem_Top( RPM, #$0 + MassString, 0 );
	end;
begin
	if EqpRPM <> Nil then DisposeRPGMenu( EqpRPM );
	EqpRPM := CreateRPGMenu( MenuItem , MenuSelect , @ZONE_EqpMenu );
	EqpRPM^.Mode := RPMNoCleanup;
	EqpRPM^.editable_menuorder := True;
	BuildEquipmentMenu( EqpRPM , PC , SHOW_EqpMenu_SubItem );

	{ If the menu is empty, add a message saying so. }
	If EqpRPM^.NumItem < 1 then begin
		AddRPGMenuItem( EqpRPM , UTF8_MsgString( 'CreateEqpMenu', 'no equipped items' ) , SELECTMENU_Cancel );
	end else begin
		AlphaKeyMenu( EqpRPM );
{$IFDEF DEBUG}
		if ( 2 <= EqpRPM^.NumItem ) then begin
			AddRPGMenuItem_Top( EqpRPM , #$0 + '[edit menu order]' , SELECTMENU_MenuOrder_Edit );
		end;
{$ENDIF DEBUG}
	end;

	{ Add the menu keys. }
	AddRPGMenuKey( EqpRPM , KeyMap[ KMC_SwitchMenu ].KCode , -2 );

	if ( 1 < EqpRPM^.NumItem ) then begin
		AddRPGMenuItem_Top( EqpRPM , UTF8_MsgString( 'ThisItemWasSelected', 'DropAll' ), -3 );
		AddRPGMenuItem_Top( EqpRPM , UTF8_MsgString( 'ThisItemWasSelected', 'TransferAll' ), -4 );
	end;
	If SHOW_EqpMenu_MassMeter then begin
		CreateEqpMenu_MassMeter( EqpRPM, PC );
	end;
end;

Procedure UpdateBackpack( PC: GearPtr );
	{ Redo all the menus, and display them on the screen. }
begin
	CreateInvMenu( PC );
	CreateEqpMenu( PC );
end;

Procedure UnequipItem( GB: GameBoardPtr; var LList: GearPtr; PC , Item: GearPtr );
	{ Delink ITEM from its parent, and stick it in the general inventory. }
var
	ParentOld: GearPtr;
	ChainOld: GearPtr;
begin
	ParentOld := Item^.Parent;

	{ First, delink Item from its parent. }
	ChainOld := DelinkGear( Item^.Parent^.InvCom , Item );
	{ HOW'D YA LIKE THEM CARROT DOTS, EH!?!? }

	{ Next, link ITEM into the general inventory, if possible. }
	GivePartToPC( LList , Item , PC , ParentOld , ChainOld );

	{ Unequipping takes time. }
	if GB <> Nil then WaitAMinute( GB , PC , ReactionTime( PC ) );
end;

Procedure UnequipFrontend( GB: GameBoardPtr; var LList: GearPtr; PC , Item: GearPtr );
	{ Simply unequip the provided item. }
	{ PRECOND: PC and ITEM had better be correct, dagnabbit... }
begin
	DialogMsg( ReplaceHash( MsgString( 'BACKPACK_Do_Unequip' ) , GearName( Item ) ) );
	UnequipItem( GB , LList , PC , Item );
end;

Procedure EjectAmmo( GB: GameBoardPtr; var LList: GearPtr; PC , Item: GearPtr );
	{ Remove all ammo from this item. }
	Procedure RemoveThisAmmo( Ammo: GearPtr );
		{ Remove ammo from wherever it is, then give it to the PC. }
		{ Ammo must be a subcom!!! }
	begin
		{ First, delink Ammo from its parent. }
		DelinkGear( Ammo^.Parent^.SubCom , Ammo );

		{ Next, link Ammo into the general inventory, if possible. }
		GivePartToPC( LList , Ammo , PC );

		DialogMsg( ReplaceHash( MsgString( 'BACKPACK_Do_EjectAmmo' ) , GearName( Ammo ) ) );
	end;
	Procedure CheckForAmmoToEject( IList: GearPtr );
		{ Check this list for ammo to eject, and also all subcoms. }
	var
		I2: GearPtr;
	begin
		while IList <> Nil do begin
			I2 := IList^.Next;

			if IList^.G = GG_Ammo then begin
				RemoveThisAmmo( IList );
			end else begin
				CheckForAmmoToEject( IList^.SubCom );
			end;

			IList := I2;
		end;
	end;
begin
	{ Start the search going. }
	CheckForAmmoToEject( Item^.SubCom );

	{ Unequipping takes time. }
	if GB <> Nil then WaitAMinute( GB , PC , ReactionTime( PC ) );
end;

Procedure EjectSoftware( GB: GameBoardPtr; var LList: GearPtr; PC , Item: GearPtr );
	{ Remove all software from this item. }
	Procedure RemoveThisSoftware( Soft: GearPtr );
		{ Remove Soft from wherever it is, then give it to the PC. }
		{ Soft must be a subcom!!! }
	begin
		{ First, delink Soft from its parent. }
		DelinkGear( Soft^.Parent^.SubCom , Soft );

		{ Next, link Ammo into the general inventory, if possible. }
		GivePartToPC( LList , Soft , PC );

		DialogMsg( ReplaceHash( MsgString( 'BACKPACK_Do_EjectSoftware' ) , GearName( Soft ) ) );
	end;
	Procedure CheckForSoftwareToEject( IList: GearPtr );
		{ Check this list for software to eject, and also all subcoms. }
	var
		I2: GearPtr;
	begin
		while IList <> Nil do begin
			I2 := IList^.Next;

			if IList^.G = GG_Software then begin
				RemoveThisSoftware( IList );
			end else begin
				CheckForSoftwareToEject( IList^.SubCom );
			end;

			IList := I2;
		end;
	end;
begin
	{ Start the search going. }
	CheckForSoftwareToEject( Item^.SubCom );

	{ Unequipping takes time. }
	if GB <> Nil then WaitAMinute( GB , PC , ReactionTime( PC ) );
end;


Function CanBeExtracted( Item: GearPtr ): Boolean;
	{ Return TRUE if the listed part can be extracted from a mecha, }
	{ or FALSE if it cannot normally be extracted. }
begin
	if ( GG_Support = Item^.G ) then begin
		CanBeExtracted := False;
		if CHEAT_MechaCustomize_FreeSupport then begin
			CanBeExtracted := True;
		end;
	end else if ( GG_Cockpit = Item^.G ) then begin
		CanBeExtracted := False;
		if CHEAT_MechaCustomize_FreeCockpit then begin
			CanBeExtracted := True;
		end;
	end else if IsMasterGear( Item ) then begin
		CanBeExtracted := False;
	end else if ( NIL = Item^.Parent ) then begin
		CanBeExtracted := False;
	end else if ( 0 = Item^.Parent^.Scale ) then begin
		CanBeExtracted := False;
		if CHEAT_MechaCustomize_FreeScale then begin
			CanBeExtracted := True;
		end;
	end else if ( GG_Modifier = Item^.G ) then begin
		CanBeExtracted := False;
		if CHEAT_MechaCustomize_FreeModifier then begin
			CanBeExtracted := True;
		end;
	end else if ( GG_Module = Item^.G ) and ( GS_Body = Item^.S ) then begin
		CanBeExtracted := False;
		if CHEAT_MechaCustomize_FreeBodyModule then begin
			CanBeExtracted := True;
		end;
	end else if ( GG_Software = Item^.G ) then begin
		{ Software can't be extracted; it must be uninstalled. }
		CanBeExtracted := False;
	end else if ( NIL <> SeekGear( Item , GG_Cockpit , 0 , False ) ) then begin
		{ If the item contains the cockpit, it can't be extracted. }
		CanBeExtracted := False;
	end else begin
		CanBeExtracted := Not PartHasIntrinsic( Item , NAS_Integral );
	end;
end;

Function CanBeInstalled( Item: GearPtr ): Boolean;
	{ Return TRUE if the listed part can maybe be installed in a mecha }
	{ with the Mecha Engineering skill, or FALSE if it definitely can't. }
begin
	if ( Item = Nil ) or ( Item^.G = GG_Ammo ) or ( Item^.G = GG_Shield ) or ( Item^.G = GG_ExArmor ) or ( Item^.G = GG_Tool ) or ( Item^.G = GG_RepairFuel ) or ( Item^.G = GG_Consumable )
	or ( Item^.G = GG_WeaponAddOn ) or ( Item^.G = GG_Software ) then begin
		CanBeInstalled := False;
	end else begin
		CanBeInstalled := True;
	end;
end;

Procedure ExtractMechaPart( var LList,Item: GearPtr );
	{ Remove this part from the mecha. }
begin
	DelinkGear( LList , Item );

	{ If this is a variable form module, assume the primary form. }
	if ( Item^.G = GG_Module ) and ( Item^.Stat[ STAT_VariableModuleForm ] <> 0 ) then begin
		Item^.S := Item^.Stat[ STAT_PrimaryModuleForm ];
	end;
end;

Function ExtractItem( GB: GameBoardPtr; TruePC , NPC , PC: GearPtr; var Item: GearPtr ): Boolean;
	{ As of GH2 all attempts to extract an item will be successful. }
	{ The only question is whether the part will be destroyed in the process, }
	{ or whether some other bad effect will take place. }
	{ Note that pulling a gear out of its mecha may well wreck it }
	{ beyond any repair! Therefore, after this call, ITEM might no }
	{ longer exist... i.e. it may equal NIL. }
	{ This function returns TRUE if the item is okay, or FALSE if it was destroyed. }
var
	it: Boolean;
	SkRoll,WreckTarget: Integer;
	Slot: GearPtr;
	name: String;
	WreckTarget_BaseCost, Cost: LongInt;
	R: Integer;
	msg: String;
begin
	Slot := Item^.Parent;

	{ First, calculate the skill target. }
	if Item^.G = GG_Module then begin
		WreckTarget := 6;
		if Slot^.G = GG_Mecha then WreckTarget := WreckTarget + Slot^.V - Item^.V;
	end else begin
		WreckTarget := ComponentComplexity( Item , NIL ) + 10 - UnscaledMaxDamage( Item );
		WreckTarget := WreckTarget + SubComComplexity( Slot ) - ComponentComplexity( Slot , NIL );
	end;
	if WreckTarget < 5 then WreckTarget := 5;

	if ( NIL = NPC ) then begin
		SkRoll := SkillRoll( GB , TruePC , NAS_MechaEngineering , STAT_Knowledge , WreckTarget , 0 , True , True );

		if GB <> Nil then begin
			AddMentalDown( TruePC , 1 );
			WaitAMinute( GB , TruePC , ReactionTime( TruePC ) * 5 );
		end;
	end else begin
		R := ReactionScore( NIL , TruePC , NPC );
		WreckTarget_BaseCost := WreckTarget_Cost * ( 200 - R ) div 200;
		Cost := WreckTarget * WreckTarget_BaseCost;
		msg := ReplaceHash( UTF8_MsgString( 'MechaPartBrowser','Cost' ) , BStr( Cost ) , BStr( NAttValue( TruePC^.NA , NAG_Experience , NAS_Credits ) ) );
		DialogMsg( msg );
		if ( NAttValue( TruePC^.NA , NAG_Experience , NAS_Credits ) < Cost ) then begin
			DialogMsg( MsgString( 'ARENA_APTU_TooExpensive' ) );
			exit( False );
		end;
		if not YesNoMenu( GB , msg , UTF8_MsgString( 'MechaPartBrowser','Yes' ) , UTF8_MsgString( 'MechaPartBrowser','No' ) , False ) then begin
			DialogMsg( UTF8_MsgString( 'MechaPartBrowser','Canceled' ) );
			exit( False );
		end;
		AddNAtt( TruePC^.NA , NAG_Experience , NAS_Credits , -Cost );

		SkRoll := SkillRoll( GB , NPC , NAS_MechaEngineering , STAT_Knowledge , WreckTarget , 0 , True , True );
		if ( NIL <> GB ) then begin
			WaitAMinute( GB , TruePC , ReactionTime( TruePC ) * 5 );
		end;
	end;

	{ Decide whether to extract the part and keep it, or just wreck it trying }
	{ to get it out. }
	name := GearName( Item );
	if SkRoll > WreckTarget then begin
		{ First, delink Item from its parent. }
		ExtractMechaPart( Item^.Parent^.SubCom , Item );

		{ Try to stick as invcom of parent. }
		GivePartToPC( GB , Item , PC );

		it := True;
		DialogMsg( ReplaceHash( MsgString( 'EXTRACT_OK' ) , name ) );
	end else begin
		RemoveGear( Item^.Parent^.SubCom , Item );
		Item := Nil;
		it := False;
		DialogMsg( ReplaceHash( MsgString( 'EXTRACT_WRECK' ) , name ) );
	end;

	ExtractItem := it;
end;

Procedure ExtractFrontend( GB: GameBoardPtr; TruePC , NPC , PC , Item: GearPtr );
	{ Simply remove the provided item. }
	{ PRECOND: PC and ITEM had better be correct, dagnabbit... }

	Function IncludeHand( LList: GearPtr ): Boolean;
	var
		Found: Boolean;
	begin
		Found := False;
		while ( NIL <> LList ) and ( not Found ) do begin
			if ( GG_Holder = LList^.G ) and ( GS_Hand = LList^.S ) and ( PC^.Scale <= LList^.Scale ) then begin
				Found := True;
			end else begin
				Found := IncludeHand( LList^.SubCom );
			end;
			LList := LList^.Next;
		end;
		IncludeHand := Found;
	end;

var
	Require_CheckWeaponsNeedHand: Boolean;
begin
	if GearActive( PC ) then begin
		DialogMsg( MsgString( 'EXTRACT_NOTACTIVE' ) );
	end else begin 
		Require_CheckWeaponsNeedHand := IncludeHand( Item );
		ExtractItem( GB , TruePC , NPC , PC , Item );
		if Require_CheckWeaponsNeedHand then begin
			Check_WeaponsNeedHand( PC , NIL );
		end;
	end;
end;


Procedure EquipItem( GB: GameBoardPtr; var LList: GearPtr; PC , Slot , Item: GearPtr );
	{ This is the real equipping procedure. Stuff ITEM into SLOT. }
	{ As noted in TheRules.txt, any nonmaster gear can only have one }
	{ item of any particular "G" type equipped at a time. So, if }
	{ SLOT already has equipment of type ITEM^.G, unequip that and }
	{ stuff it into PC's general inventory. }
var
	I2,I3: GearPtr;
begin
	{ First, check for already equipped items. }
	I2 := Slot^.InvCom;
	while I2 <> Nil do begin
		I3 := I2^.Next;		{ This next step might delink I2, so... }
		if ( I2^.G = Item^.G ) or ( Slot^.G = GG_Holder ) then begin
			UnequipItem( GB , LList , PC , I2 );
		end;
		I2 := I3;
	end;

	{ Next, delink Item from PC. }
	DelinkGear( PC^.InvCom , Item );

	{ Next, link ITEM into SLOT. }
	InsertInvCom( Slot , Item );

	{ Equipping an item takes time. }
	if GB <> Nil then WaitAMinute( GB , PC , ReactionTime( PC ) );
end;


Procedure EquipItemFrontend( GB: GameBoardPtr; var LList: GearPtr; PC , Item: GearPtr );
	{ Assign ITEM to a legal equipment slot. Move it from the }
	{ general inventory into its new home. }
var
	EI_Menu: RPGMenuPtr;
	N: LongInt;
	Slot: GearPtr;
	Modifier: Integer;
	Modifier_with_Hand: Integer;
	NeedHand: Boolean;
begin
	{ Build the slot selection menu. }
	EI_Menu := CreateRPGMenu( MenuItem , MenuSelect , @ZONE_InvMenu );
	BuildSlotMenu( EI_Menu , PC , Item , SHOW_EquipItem_SubItem );
	AlphaKeyMenu( EI_Menu );
	if EI_Menu^.NumItem < 1 then AddRPGMenuItem( EI_Menu , ReplaceHash( MsgString( 'BACKPACK_CantEquip' ) , GearName( Item ) ) , SELECTMENU_Cancel );

	{ Select a slot for the item to go into. }
	BP_Source := Item;
	MenuDesc := UTF8_MsgString( 'BACKPACK_Directions' , 'EquipItemTo' );
	N := SelectMenu( EI_Menu , @ThisItemRedraw);

	DisposeRPGMenu( EI_Menu );

	{ If a slot was selected, pass that info on to the workhorse. }
	if ( SELECTMENU_Enable <= N ) then begin
		Slot := LocateGearByNumber( PC , N );
		DialogMsg( ReplaceHash( MsgString( 'BACKPACK_Do_Equip' ) , GearName( Item ) ) );
		EquipItem( GB , LList , PC , Slot , Item );
		if Item^.G = GG_Weapon then begin
			Modifier := WeaponWeightModifier( Item, DefaultAtOp( Item ), PC, NeedHand, Modifier_with_Hand );
			if NeedHand then begin
				if ( Modifier < 0 ) then begin
					if ( Modifier_with_Hand < 0 ) then begin
						DialogMsg( UTF8_MsgString( 'EquipItemFrontend', 'TooHeavy' ) );
					end else begin
						DialogMsg( UTF8_MsgString( 'EquipItemFrontend', 'NoFreeHand_TooHeavy' ) );
					end;
				end else begin
					DialogMsg( UTF8_MsgString( 'EquipItemFrontend', 'NeedHand' ) );
				end;
			end else begin
				if ( Modifier < 0 ) then begin
					DialogMsg( UTF8_MsgString( 'EquipItemFrontend', '***BUG***' ) );
				end else begin
					{ No Penalty }
				end;
			end;
		end;
		if ( GG_Holder = Slot^.G ) and ( GS_Hand = Slot^.S ) then begin
			Check_WeaponsNeedHand( PC, Item );
		end;
	end;
end;

Function InstallItem( GB: GameBoardPtr; TruePC , NPC , Slot: GearPtr; var Item: GearPtr ): Boolean;
	{ Attempt the skill rolls needed to install ITEM into the }
	{ requested slot. }
	{ ITEM should be linked as an inventory gear, and may be deleted by this function. }
	{ Return TRUE if the install was successful, or FALSE otherwise. }
	Procedure ResortInventory( Part: GearPtr );
		{ Take all the inventory from PART and move it to FindMaster(Slot) }
	var
		M,I: GearPtr;
	begin
		M := FindMaster( Slot );
		if M <> Nil then begin
			while Part^.InvCom <> Nil do begin
				I := Part^.InvCom;
				DelinkGear( Part^.InvCom , I );
				InsertInvCom( M , I );
			end;
		end else begin
			DisposeGear( Part^.InvCom );
		end;
	end;
	Procedure ShakeDownItem( LList: GearPtr );
		{ Remove all InvComs from this list, and for all children as well. }
	begin
		while LList <> Nil do begin
			if LList^.InvCom <> Nil then ResortInventory( LList );
			ShakeDownItem( LList^.SubCom );
			LList := LList^.Next;
		end;
	end;
var
	SlotCom,ItemCom,UsedCom,HardLimit: Integer;
	WreckTarget,SkRoll: Integer;
	WreckTarget_BaseCost, Cost: LongInt;
	R: Integer;
	msg: String;
begin
	{ Error Check - no circular references! }
	if ( FindGearIndex( Item , Slot ) <> -1 ) then begin
		DialogMsg( ReplaceHash( MsgString( 'INSTALL_FAIL' ) , GearName( Item ) ) );
		Exit( False );
	end;

	{ Also, can't engineer things when you're exhausted. }
	if ( NIL = NPC ) and IsMasterGear( TruePC ) and ( CurrentMental( TruePC ) < 1 ) then begin
		DialogMsg( ReplaceHash( MsgString( 'INSTALL_FAIL' ) , GearName( Item ) ) );
		Exit( False );
	end;

	{ Can't install into a personal-scale slot. }
	if ( 0 = Slot^.Scale ) and not( CHEAT_MechaCustomize_FreeScale ) then begin
		DialogMsg( ReplaceHash( MsgString( 'INSTALL_FAIL' ) , GearName( Item ) ) );
		Exit( False );
	end;

	SlotCom := ComponentComplexity( Slot , NIL );
	HardLimit := SlotCom;
	ItemCom := ComponentComplexity( Item , Slot );
	UsedCom := SubComComplexity( Slot );
	{ If the INNOVATION talent is known, increase the HardLimit. }
	if GB <> Nil then begin
		if TeamHasTalent( GB , NAV_DefPlayerTeam , NAS_Innovation ) then HardLimit := HardLimit + 5;
	end else begin
		if ( NIL = NPC ) then begin
			if HasTalent( TruePC , NAS_Innovation ) then HardLimit := HardLimit + 5;
		end else begin
			if HasTalent( NPC , NAS_Innovation ) then HardLimit := HardLimit + 5;
		end;
	end;

	if (( UsedCom + ItemCom ) > HardLimit ) and not IsMasterGear( Slot ) then begin
		DialogMsg( MsgString( 'INSTALL_NoRoom' ) );
		Exit( False );
	end;

	{ The WreckTarget is the target number that must be beat }
	{ in order to avoid accidentally destroying the part... }
	if ( Item^.G = GG_Module ) then begin
		WreckTarget := 3 + NumSiblingGears( Slot^.SubCom ) - Item^.V;
	end else if Item^.G = GG_MoveSys then begin
		WreckTarget := 5 + ItemCom;
	end else if ( Item^.G = GG_Modifier ) then begin
		WreckTarget := 10 + ItemCom;
	end else if ( UnscaledMaxDamage( Item ) < 1 ) or ( Item^.Scale < Slot^.Scale ) then begin
		WreckTarget := 10;
	end else begin
		WreckTarget := 10 - UnscaledMaxDamage( Item );
	end;
	{ The insertion target is easier for heavy items, harder for lightened ones. }
	WreckTarget := WreckTarget - NAttValue( Item^.NA , NAG_GearOps , NAS_MassAdjust );
	if WreckTarget < 3 then WreckTarget := 3;

	{ If the SLOT is going to be overstuffed, better raise the }
	{ number of successes and the target number drastically. }
	if ( ( ItemCom + UsedCom ) > SlotCom ) and ( Not IsMasterGear( Slot ) ) then begin
		WreckTarget := WreckTarget + 2 + UsedCom + ItemCom - SlotCom + ItemCom * 5 div SlotCom;
	end;

	if ( NIL = NPC ) then begin
		if GB <> Nil then WaitAMinute( GB , TruePC , ReactionTime( TruePC ) * 5 );

		SkRoll := SkillRoll( GB , TruePC , NAS_MechaEngineering , STAT_Knowledge , WreckTarget , 0 , True , True );
	end else begin
		R := ReactionScore( NIL , TruePC , NPC );
		WreckTarget_BaseCost := WreckTarget_Cost * ( 200 - R ) div 200;
		Cost := WreckTarget * WreckTarget_BaseCost;
		msg := ReplaceHash( UTF8_MsgString( 'MechaPartBrowser','Cost' ) , BStr( Cost ) , BStr( NAttValue( TruePC^.NA , NAG_Experience , NAS_Credits ) ) );
		DialogMsg( msg );
		if ( NAttValue( TruePC^.NA , NAG_Experience , NAS_Credits ) < Cost ) then begin
			DialogMsg( MsgString( 'ARENA_APTU_TooExpensive' ) );
			exit( False );
		end;
		if not YesNoMenu( GB , msg , UTF8_MsgString( 'MechaPartBrowser','Yes' ) , UTF8_MsgString( 'MechaPartBrowser','No' ) , False ) then begin
			DialogMsg( UTF8_MsgString( 'MechaPartBrowser','Canceled' ) );
			exit( False );
		end;
		AddNAtt( TruePC^.NA , NAG_Experience , NAS_Credits , -Cost );

		if ( NIl <> GB ) then WaitAMinute( GB , TruePC , ReactionTime( TruePC ) * 5 );
		SkRoll := SkillRoll( GB , NPC , NAS_MechaEngineering , STAT_Knowledge , WreckTarget , 0 , True , True );
	end;
	if SkRoll >= WreckTarget then begin
		DialogMsg( ReplaceHash( MsgString( 'INSTALL_OK' ) , GearName( Item ) ) );
		{ Install the item. }
		ResortInventory( Item );
		ShakeDownItem( Item^.SubCom );
		DelinkGear( Item^.Parent^.InvCom , Item );
		InsertSubCom( Slot , Item );
	end else begin
		DialogMsg( ReplaceHash( MsgString( 'INSTALL_WRECK' ) , GearName( Item ) ) );
		RemoveGear( Item^.Parent^.InvCom , Item );
		Item := Nil;
	end;

	if ( GB <> Nil ) and ( NIL = NPC ) and IsMasterGear( TruePC ) then begin
		AddMentalDown( TruePC , 1 );
	end;

	InstallItem := SkRoll >= WreckTarget;
end;

Procedure InstallFrontend( GB: GameBoardPtr; TruePC , NPC , PC , Item: GearPtr );
	{ Assign ITEM to a legal equipment slot. Move it from the }
	{ general inventory into its new home. }
var
	EI_Menu: RPGMenuPtr;
	N: LongInt;
begin
	{ Error check- can't install into an active master. }
	if GearActive( PC ) then begin
		DialogMsg( MsgString( 'INSTALL_NOTACTIVE' ) );
		Exit;
	end;

	{ Build the slot selection menu. }
	EI_Menu := CreateRPGMenu( MenuItem , MenuSelect , @ZONE_InvMenu );
	BuildSubMenu( EI_Menu , PC , Item , True , SHOW_Install_SubItem );
	AlphaKeyMenu( EI_Menu );
	if EI_Menu^.NumItem < 1 then AddRPGMenuItem( EI_Menu , ReplaceHash( UTF8_MsgString('InstallFrontend','cannot install'), GearName(Item) ) , SELECTMENU_Cancel );

	{ Select a slot for the item to go into. }
	DialogMsg( ReplaceHash( MsgString( 'BACKPACK_InstallInfo' ) , GearName( Item ) ) );
	BP_Source := Item;
	N := SelectMenu( EI_Menu , @InstallRedraw);

	DisposeRPGMenu( EI_Menu );

	{ If a slot was selected, pass that info on to the workhorse. }
	if ( SELECTMENU_Enable <= N ) then begin
		{ Store the name here, since the item might get destroyed }
		{ during the installation process. }
		InstallItem( GB , TruePC , NPC , LocateGearByNumber( PC , N ) , Item );
	end;
end;

Procedure InstallAmmoFrontend( GB: GameBoardPtr; PC , Item: GearPtr );
	{ Assign ITEM to a legal projectile weapon. Move it from the }
	{ general inventory into its new home. }
var
	IA_Menu: RPGMenuPtr;
	Gun: GearPtr;
	N: LongInt;
begin
	{ Build the slot selection menu. }
	IA_Menu := CreateRPGMenu( MenuItem , MenuSelect , @ZONE_InvMenu );
	BuildSubMenu( IA_Menu , PC , Item , False , SHOW_InstallAmmo_SubItem );
	AlphaKeyMenu( IA_Menu );
	if IA_Menu^.NumItem < 1 then AddRPGMenuItem( IA_Menu , ReplaceHash( UTF8_MsgString('InstallAmmoFrontend','no weapon'), GearName(Item) ) , SELECTMENU_Cancel );

	{ Select a slot for the item to go into. }
	MenuDesc := UTF8_MsgString( 'BACKPACK_Directions' , 'InstallAmmoTo' );
	N := SelectMenu( IA_Menu , @MiscProcRedraw);

	DisposeRPGMenu( IA_Menu );

	{ If a slot was selected, pass that info on to the workhorse. }
	if ( SELECTMENU_Enable <= N ) then begin
		Gun := LocateGearByNumber( PC , N );
		DialogMsg( ReplaceHash( UTF8_MsgString('InstallAmmoFrontend','You load'), GearName(Gun), GearName(Item) ) );
		InstallAmmo( GB , PC , Gun , Item );
	end;
end;


Procedure DropFrontEnd( GB: GameBoardPtr; var LList: GearPtr; PC , Item: GearPtr );
	{ How to drop an item: Make sure PC is a root-level gear. }
	{ Delink ITEM from its current location. }
	{ Copy PC's location variables to ITEM. }
	{ Install ITEM as the next sibling of PC. }
begin
	{ Make sure PC is at root level... }
	PC := FindRoot( PC );

	{ Delink ITEM from its parent... }
	DelinkGear( Item^.Parent^.InvCom , Item );

	{ Copy the location variables to ITEM... }
	SetNAtt( Item^.NA , NAG_Location , NAS_X , NAttValue( PC^.NA , NAG_Location , NAS_X ) );
	SetNAtt( Item^.NA , NAG_Location , NAS_Y , NAttValue( PC^.NA , NAG_Location , NAS_Y ) );
	if ( GB <> Nil ) and not OnTheMap( GB , PC ) then SetNAtt( Item^.NA , NAG_Location , NAS_Team , NAV_DefPlayerTeam );

	{ Install ITEM into LList... }
	AppendGear( LList , Item );

	{ Do display stuff. }
	DialogMsg( ReplaceHash( MsgString( 'BACKPACK_Do_Drop' ) , GearName( Item ) ) );
end;

Procedure DropAllFrontEnd( GB: GameBoardPtr; var LList: GearPtr; PC , Part: GearPtr );
var
	RPM: RPGMenuPtr;
	N: Integer;
	Part_Next: GearPtr;
begin
	RPM := CreateRPGMenu( MenuItem , MenuSelect , @ZONE_Menu );
	AddRPGMenuItem( RPM , UTF8_MsgString('DropAllFrontEnd','Cancel') , -1 );
	AddRPGMenuItem( RPM , UTF8_MsgString('DropAllFrontEnd','OK') , 1 );
	RPM^.Mode := RPMNormal;
	InfoBox( ZONE_Menu );
	N := SelectMenu( RPM , NIL );
	DisposeRPGMenu( RPM );
	if ( -1 = N ) then exit;

	while ( NIL <> Part ) do begin
		Part_Next := Part^.Next;
		if CheckAlongPath_DisallowDropping( Part ) then begin
		end else begin
			DropFrontEnd( GB , LList , PC , Part );
		end;
		Part := Part_Next;
	end;
end;

Procedure DropEqpAllFrontEnd( GB: GameBoardPtr; var LList: GearPtr; PC , Master: GearPtr );
	Procedure CheckAlongPath( Part: GearPtr; IsInv: Boolean );
	var
		Part_Next: GearPtr;
	begin
		while ( NIL <> Part ) do begin
			Part_Next := Part^.Next;
			if IsInv and ( not CheckAlongPath_DisallowDropping( Part ) ) then begin
				DropFrontEnd( GB , LList , PC , Part );
			end;
			CheckAlongPath( Part^.InvCom , True );
			CheckAlongPath( Part^.SubCom , False );
			Part := Part_Next;
		end;
	end;
var
	RPM: RPGMenuPtr;
	N: Integer;
begin
	RPM := CreateRPGMenu( MenuItem , MenuSelect , @ZONE_Menu );
	AddRPGMenuItem( RPM , UTF8_MsgString('DropAllFrontEnd','Cancel') , -1 );
	AddRPGMenuItem( RPM , UTF8_MsgString('DropAllFrontEnd','OK') , 1 );
	RPM^.Mode := RPMNormal;
	InfoBox( ZONE_Menu );
	N := SelectMenu( RPM , NIL );
	DisposeRPGMenu( RPM );
	if ( -1 = N ) then exit;

	CheckAlongPath( Master^.InvCom , False );
	CheckAlongPath( Master^.SubCom , False );
end;

Function TradeFrontEnd_GetDest( GB: GameBoardPtr; PC , Item , LList: GearPtr ): GearPtr;
const
	Unsafe_Transfer_Range = 10;
var
	TI_Menu: RPGMenuPtr;
	M: GearPtr;
	Team: Integer;
	N: LongInt;
	X,Y: Integer;

	Function Transferable_To ( Dest: GearPtr ) : Boolean;
	var
		DTeam, DX, DY : Integer;
	begin
		If Dest = PC then Exit(False);

		{ Team check.  This could probably be simplified --
		  How could the source Master's team be other than
		  DefPlayer or Lancemate anyway? }
		DTeam := NAttValue( Dest^.NA , NAG_Location , NAS_Team );
		if DTeam = NAV_LancemateTeam then DTeam := NAV_DefPlayerTeam;
		if DTeam <> Team then Exit(False);

		if X = 0 then Exit(True); {safe area case}

		{we're now in the unsafe area case, check for adjacency}
		DX := NAttValue( Dest^.NA, NAG_Location, NAS_X);
		DY := NAttValue( Dest^.NA, NAG_Location, NAS_Y);
		if not OnTheMap(GB,DX,DY) then Exit(False); {cannot transfer to offmap stuff}

		Transferable_To := ( Range(X,Y,DX,DY) <= Unsafe_Transfer_Range );
	end;

begin
	if ( GB = Nil ) or IsSafeArea( GB ) then begin
		X := 0;
		Y := 0;
	end else begin
		DialogMsg( MsgString( 'TRANSFER_UNSAFE' ) );
		X := NAttValue(PC^.NA, NAG_Location, NAS_X);
		Y := NAttValue(PC^.NA, NAG_Location, NAS_Y);
	end;

	{ Build the slot selection menu. }
	TI_Menu := CreateRPGMenu( MenuItem , MenuSelect , @ZONE_InvMenu );
	N := 1;
	M := LList;
	Team := NAttValue( PC^.NA , NAG_Location , NAS_Team );
	if Team = NAV_LancemateTeam then Team := NAV_DefPlayerTeam;

	{ This menu should contain all the masters from LList which }
	{ belong to Team 1. }
	while ( NIL <> M ) do begin
		if ( NIL = GB ) and IsMasterGear( M ) and ( M <> PC ) then begin
			AddRPGMenuItem( TI_Menu , TeamMateName( GB , M ) , N );
		end else if IsMasterGear( M ) and Transferable_To(M) then begin
			AddRPGMenuItem( TI_Menu , GearName( M ) , N );
		end;
		M := M^.Next;
		Inc( N );
	end;
	{ RPMSortAlpha( TI_Menu ); } { In UTF-8, this sort causes a result that I cannot assume. }
	AlphaKeyMenu( TI_Menu );

	if TI_Menu^.NumItem < 1 then AddRPGMenuItem( TI_Menu , MsgString( 'CANCEL' ) , SELECTMENU_Cancel );

	{ Select a slot for the item to go into. }
	BP_Source := LList;
	BP_SeekSibs := True;
	BP_ActiveMenu := TI_Menu;
	BP_TRADE_ITEM := Item;

	DialogMSG( ReplaceHash( MsgString( 'FHQ_SelectDestination' ), GearName( Item ) ) );
	MenuDesc := UTF8_MsgString( 'BACKPACK_Directions' , 'TradeItem' );
	N := SelectMenu( TI_Menu , @TradeProcRedraw );

	DisposeRPGMenu( TI_Menu );

	if ( N < SELECTMENU_Enable ) then begin
		Exit( NIL );
	end;
	TradeFrontEnd_GetDest := RetrieveGearSib( LList , N );
end;

Procedure TradeFrontend( GB: GameBoardPtr; PC , Item, LList: GearPtr );
	{ Assign ITEM to a different master. Move it from the }
	{ general inventory of PC into its new home. }
var
	M: GearPtr;
begin
	M := TradeFrontEnd_GetDest( GB , PC , Item , LList );

	if ( NIL <> M ) then begin
		if IsLegalInvCom( M , Item ) then begin
			DelinkGear( Item^.Parent^.InvCom , Item );
			InsertInvCom( M , Item );
			DialogMsg( MsgString( 'BACKPACK_ItemTraded' ) );
		end else begin
			DialogMsg( MsgString( 'BACKPACK_NotTraded' ) );
		end;
	end;
end;

Procedure TradeAllFrontend( GB: GameBoardPtr; PC , Item, LList: GearPtr );
	{ Assign ITEM to a different master. Move it from the }
	{ general inventory of PC into its new home. }
var
	M: GearPtr;
	Item_Next: GearPtr;
begin
	M := TradeFrontEnd_GetDest( GB , PC , Item , LList );

	if ( NIL <> M ) then begin
		while ( NIL <> Item ) do begin
			Item_Next := Item^.Next;
			if CheckAlongPath_DisallowTransfering( Item ) then begin
			end else begin
				if IsLegalInvCom( M , Item ) then begin
					DelinkGear( Item^.Parent^.InvCom , Item );
					InsertInvCom( M , Item );
					DialogMsg( ReplaceHash( UTF8_MsgString( 'TradeAllFrontend' , 'ItemTraded' ) , GearName( Item ) ) );
				end else begin
					DialogMsg( ReplaceHash( UTF8_MsgString( 'TradeAllFrontend' , 'NotTraded' ) , GearName( Item ) ) );
				end;
			end;
			Item := Item_Next;
		end;
	end;
end;

Procedure TradeEqpAllFrontEnd( GB: GameBoardPtr; PC , Item, LList: GearPtr );
	{ Assign ITEM to a different master. Move it from the }
	{ general inventory of PC into its new home. }
var
	M: GearPtr;

	Procedure CheckAlongPath( Part: GearPtr; IsInv: Boolean );
	var
		Part_Next: GearPtr;
	begin
		while ( NIL <> Part ) do begin
			Part_Next := Part^.Next;
			if IsInv and ( not CheckAlongPath_DisallowTransfering( Part ) ) then begin
				if IsLegalInvCom( M , Part ) then begin
					DelinkGear( Part^.Parent^.InvCom , Part );
					InsertInvCom( M , Part );
					DialogMsg( ReplaceHash( UTF8_MsgString( 'TradeAllFrontend' , 'ItemTraded' ) , GearName( Part ) ) );
				end else begin
					DialogMsg( ReplaceHash( UTF8_MsgString( 'TradeAllFrontend' , 'NotTraded' ) , GearName( Part ) ) );
				end;
			end;
			CheckAlongPath( Part^.InvCom , True );
			CheckAlongPath( Part^.SubCom , False );
			Part := Part_Next;
		end;
	end;
begin
	M := TradeFrontEnd_GetDest( GB , PC , Item , LList );

	if ( NIL <> M ) then begin
		CheckAlongPath( Item^.InvCom , False );
		CheckAlongPath( Item^.SubCom , False );
	end;
end;

Procedure FHQ_AssociatePilotMek( PC , M , LList: GearPtr );
	{ Associate the mecha with the pilot. }
begin
	AssociatePilotMek( LList , PC , M );
	DialogMsg( ReplaceHash( MsgString( 'FHQ_AssociatePM' ) , GearName( PC ) ) );
end;

Procedure FHQ_AssociateRedraw;
	{ Do a redraw for the Field HQ. }
var
	Part: GearPtr;
begin
	if ( NIL <> BP_ActiveMenu ) and ( NIL <> BP_Source ) then begin
		Part := RetrieveGearSib( BP_Source , CurrentMenuItemValue( BP_ActiveMenu ) );
	end;
	if SHOW_NPCStatus_FieldHQ then begin
		if ( NIL <> Part ) then begin
			CombatDisplay( Part , BP_GB );
		end else begin
			CombatDisplay( FHQ_AR_PC , BP_GB );
		end;
	end else begin
		CombatDisplay( BP_GB );
	end;
	SetupFHQDisplay;
	if ( NIL <> Part ) then begin
		BrowserInterfaceInfo( BP_GB , Part , ZONE_ItemsInfo );
	end;
end;

Procedure FHQ_SelectPilotForMecha( GB: GameBoardPtr; Mek: GearPtr );
	{ Select a pilot for the mecha in question. }
	{ Pilots must be characters- they must either belong to the default }
	{ player team or, if they're lancemates, they must have a CID. }
	{ This is to prevent the PC from dominating some sewer rats and }
	{ training them to be pilots. }
var
	RPM: RPGMenuPtr;
	N: LongInt;
	M: GearPtr;
begin
	BP_GB := GB;
	BP_Source := GB^.Meks;
	DialogMsg( ReplaceHash( MsgString( 'FHQ_SP4M_Prompt' ) , FullGearName( Mek ) ) );

	{ Create the menu. }
	RPM := CreateRPGMenu( MenuItem , MenuSelect , @ZONE_FieldHQMenu );
	BP_ActiveMenu := RPM;
	M := GB^.Meks;
	N := 1;
	while M <> Nil do begin
		if M^.G = GG_Character then begin
			if ( NAttValue( M^.NA , NAG_Location , NAS_Team ) = NAV_LancemateTeam ) and ( NAttValue( M^.NA , NAG_Personal , NAS_CID ) <> 0 ) then begin
				AddRPGMenuItem( RPM , GearName( M ) , N );
			end else if NAttValue( M^.NA , NAG_Location , NAS_Team ) = NAV_DefPlayerTeam then begin
				AddRPGMenuItem( RPM , GearName( M ) , N );
			end;
		end;
		M := M^.Next;
		Inc( N );
	end;
	{ RPMSortAlpha( RPM ); } { In UTF-8, this sort causes a result that I cannot assume. }
	AlphaKeyMenu( RPM );
	AddRPGMenuItem( RPM , MsgString( 'EXIT' ) , SELECTMENU_Cancel );

	{ Get a selection from the menu. }
	FHQ_AR_PC := Mek;
	n := SelectMenu( RPM , @FHQ_AssociateRedraw );

	DisposeRPGMenu( RPM );

	if ( SELECTMENU_Enable < N ) then begin
		M := RetrieveGearSib( GB^.Meks , N );
		FHQ_AssociatePilotMek( M , Mek , GB^.Meks );
	end;
end;

Procedure FHQ_SelectMechaForPilot( GB: GameBoardPtr; NPC: GearPtr );
	{ Select a pilot for the mecha in question. }
	{ Pilots must be characters- they must either belong to the default }
	{ player team or, if they're lancemates, they must have a CID. }
	{ This is to prevent the PC from dominating some sewer rats and }
	{ training them to be pilots. }
var
	RPM: RPGMenuPtr;
	N: LongInt;
	M: GearPtr;
begin
	BP_GB := GB;
	BP_Source := GB^.Meks;
	DialogMsg( ReplaceHash( MsgString( 'FHQ_SM4P_Prompt' ) , GearName( NPC ) ) );

	{ Error check- only characters can pilot mecha! Pets can't. }
	if ( NAttValue( NPC^.NA , NAG_Personal , NAS_CID ) = 0 ) and ( NAttValue( NPC^.NA , NAG_Location , NAS_Team ) <> NAV_DefPlayerTeam ) then begin
		DialogMsg( ReplaceHash( MsgString( 'FHQ_SMFP_NoPets' ) , GearName( NPC ) ) );
		Exit;
	end;

	{ Create the menu. }
	RPM := CreateRPGMenu( MenuItem , MenuSelect , @ZONE_FieldHQMenu );
	BP_ActiveMenu := RPM;
	M := GB^.Meks;
	N := 1;
	while M <> Nil do begin
		if ( M^.G = GG_Mecha ) and ( NAttValue( M^.NA , NAG_Location , NAS_Team ) = NAV_DefPlayerTeam ) then begin
			AddRPGMenuItem( RPM , GearName( M ) , N );
		end;
		M := M^.Next;
		Inc( N );
	end;
	{ RPMSortAlpha( RPM ); } { In UTF-8, this sort causes a result that I cannot assume. }
	AlphaKeyMenu( RPM );
	AddRPGMenuItem( RPM , MsgString( 'EXIT' ) , SELECTMENU_Cancel );

	{ Get a selection from the menu. }
	FHQ_AR_PC := NPC;
	n := SelectMenu( RPM , @FHQ_AssociateRedraw );

	DisposeRPGMenu( RPM );

	if ( SELECTMENU_Enable < N ) then begin
		M := RetrieveGearSib( GB^.Meks , N );
		FHQ_AssociatePilotMek( NPC , M , GB^.Meks );
	end;
end;

Procedure UseScriptItem( GB: GameBoardPtr; TruePC, Item: GearPtr; T: String );
	{ This item has a script effect. Exit the backpack and use it. }
begin
	if SAttArrayValue( Item^.SA , T ) <> '' then begin
		{ Announce the intention. }
		DialogMsg( ReplaceHash( MsgString( 'BACKPACK_Script_' + T ) , GearName( Item ) ) );

		{ Using items takes time... }
		WaitAMinute( GB , TruePC , ReactionTime( TruePC ) );

		{ ...and also exits the backpack. }
		ForceQuit := True;
		CombatDisplay( GB );

		{ Finally, trigger the script. }
		TriggerGearScript( GB , Item , T );
	end else begin
		{ Announce the lack of a valid script. }
		DialogMsg( ReplaceHash( MsgString( 'BACKPACK_CannotUseScript' ) , GearName( Item ) ) );
	end;
end;

Procedure UseSkillOnItem( GB: GameBoardPtr; TruePC, Item: GearPtr );
	{ The PC will have the option to use a CLUE-type skill on this }
	{ item, maybe to gain some new information, activate an effect, }
	{ or whatever else. }
var
	SkMenu: RPGMenuPtr;
	T: Integer;
	msg: String;
begin
	SkMenu := CreateRPGMenu( MenuItem , MenuSelect , @ZONE_InvMenu );

	{ Add the usable skills. }
	for t := 1 to NumSkill do begin
		{ In order to be usable, it must be a CLUE type skill, }
		{ and the PC must have ranks in it. }
		if ( SkillMan[ T ].Usage = USAGE_Clue ) and ( TeamHasSkill( GB , NAV_DefPlayerTeam , T ) or TeamHasTalent( GB , NAV_DefPlayerTeam , NAS_JackOfAll ) ) then begin
			msg := UTF8_MsgString( 'UseSkillOnItem', 'BACKPACK_ClueSkillPrompt' );
			msg := ReplaceHash( msg , MsgString( 'SKILLNAME_' + BStr( T ) ) , GearName( Item ) );
			AddRPGMenuItem( SkMenu , msg , T );
		end;
	end;
	RPMSortValue( SkMenu );
	AlphaKeyMenu( SkMenu );
	AddRPGMenuItem( SkMenu , MsgString( 'BACKPACK_CancelSkillUse' ) , SELECTMENU_Cancel );

	BP_GB := GB;
	BP_Source := Item;
	MenuDesc := UTF8_MsgString( 'BACKPACK_Directions' , 'SelectSkill' );
	T := SelectMenu( SkMenu , @ThisItemRedraw);

	DisposeRPGMenu( SkMenu );

	if ( SELECTMENU_Enable <= T ) then begin
		UseScriptItem( GB , TruePC , Item , Skill_Use_Trigger[ T ] );
	end;
end;

Procedure EatItem( GB: GameBoardPtr; TruePC , Item: GearPtr );
	{ The PC wants to eat this item. Give it a try. }
const
	RT_MAX = 2147483647;
var
	rt: Int64;
	effect: String;
begin
	TruePC := LocatePilot( TruePC );

	if TruePC = Nil then begin
		DialogMsg( ReplaceHash( MsgString( 'BACKPACK_CantBeEaten' ) , GearName( Item ) ) );

	end else if ( NAttValue( TruePC^.NA , NAG_Condition , NAS_Hunger ) > ( Item^.V div 2 ) ) or ( Item^.V = 0 ) then begin
		{ Show a message. }
		DialogMsg( ReplaceHash( UTF8_MsgString( 'EatItem', 'BACKPACK_YouAreEating' ) , GearName( TruePC ) , GearName( Item ) ) );

		{ Eating takes time... }
		rt := Int64(ReactionTime( TruePC )) * Int64(GearMass( Item )) + 1;
		if ( RT_MAX < rt ) then begin
			rt := RT_MAX;
		end;
		WaitAMinute( GB , TruePC , rt );

		{ ...and also exits the backpack. }
		ForceQuit := True;

		{ Locate the PC's Character record, then adjust hunger values. }
		AddNAtt( TruePC^.NA , NAG_Condition , NAS_Hunger , -Item^.V );
		AddMoraleDmg( TruePC , -( Item^.Stat[ STAT_MoraleBoost ] * FOOD_MORALE_FACTOR ) );

		{ Invoke the item's effect, if any. }
		if Item^.Stat[ STAT_FoodEffectType ] <> 0 then begin
			effect := ReplaceHash( Food_Effect_String[ Item^.Stat[ STAT_FoodEffectType ] ] , BStr( Item^.Stat[ STAT_FoodEffectMod ] ) );
			EffectFrontEnd( GB , Item , TruePC , effect , '' );
		end;

		{ Apply the item's SkillXP, if any. }
		if Item^.Stat[ STAT_FoodSkillXP ] <> 0 then begin
			if DoleSkillExperience( TruePC , Item^.Stat[ STAT_FoodSkillXP ] , Item^.Stat[ STAT_FoodSkillXPAmount ] ) then begin
				DialogMsg( ReplaceHash( MsgString( 'BACKPACK_FoodSkillBoost' ) , GearName( Item ) ) );
			end;
		end;

		{ Destroy the item, if appropriate. }
		if IsInvCom( Item ) then begin
			RemoveGEar( Item^.Parent^.InvCom , Item );
		end else if IsSubCom( Item ) then begin
			RemoveGEar( Item^.Parent^.SubCom , Item );
		end;
	end else begin
		DialogMsg( MsgString( 'BACKPACK_NotHungry' ) );
	end;
end;

Procedure SwitchQuickFire( Item: GearPtr );
	{ Swap the quickfire prefs for this item. }
var
	QFP: Integer;	{ Quick Fire Priority }
begin
	QFP := ( NAttValue( Item^.NA, NAG_WeaponModifier, NAS_QuickFire ) + 3 ) mod 4;
	SetNAtt( Item^.NA, NAG_WeaponModifier, NAS_QuickFire , QFP );
end;

Procedure InstallSoftware( GB: GameBoardPtr; PC , SW: GearPtr );
	{ Attempt to install some software into a computer. This is how it's }
	{ going to work: First pick a computer to install into. If a computer }
	{ was selected, see if it has enough free space for the requested }
	{ software. If it doesn't, prompt to uninstall some programs. If after }
	{ the purge there's enough room for the requested program install it. }
	Function SelectComputer: GearPtr;
		{ Attempt to select a computer. }
	var
		RPM: RPGMenuPtr;
		N: LongInt;
	begin
		RPM := CreateRPGMenu( MenuItem , MenuSelect , @ZONE_InvMenu );
		BuildGearMenu( RPM , PC , GG_Computer );
		AlphaKeyMenu( RPM );
		BP_Source := PC;
		BP_SeekSibs := False;
		BP_ActiveMenu := RPM;

		if RPM^.NumItem < 1 then AddRPGMenuItem( RPM , ReplaceHash( UTF8_MsgString('InstallSoftware','no computer'), GearName(SW) ) , SELECTMENU_Cancel );

		{ Select a computer, and get rid of the menu. }
		MenuDesc := UTF8_MsgString( 'BACKPACK_Directions' , 'InstallSoftware' );
		N := SelectMenu( RPM , @MiscProcRedraw);
		DisposeRPGMenu( RPM );

		if ( SELECTMENU_Enable < N ) then begin
			SelectComputer := LocateGearByNumber( PC , N );
		end else begin
			SelectComputer := Nil;
		end;
	end;
	Procedure FreeSoftwareSpace( Compy: GearPtr );
		{ Unload programs from this computer until SW can be installed, }
		{ or the user cancels. }
	var
		RPM: RPGMenuPtr;
		S: GearPtr;
		SI,TI: Integer;
		N,V: LongInt;
		Part : GearPtr;
	begin
		DialogMsg( ReplaceHash( MsgString( 'BACKPACK_FreeSoftwareSpace' ) , GearName( SW ) ) );
		SI := 1;
		TI := 1;
		repeat
			RPM := CreateRPGMenu( MenuItem , MenuSelect , @ZONE_InvMenu );
			RPM^.editable_menuorder := True;
			BP_Source := Compy^.SubCom;
			BP_SeekSibs := True;
			BP_ActiveMenu := RPM;

			{ Add the software to the menu. }
			N := 1;
			S := Compy^.SubCom;
			while S <> nil do begin
				AddRPGMenuItem( RPM , GearName( S ) , N );
				Inc( N );
				S := S^.Next;
			end;
			AlphaKeyMenu( RPM );
{$IFDEF DEBUG}
			if ( 2 <= RPM^.NumItem ) then begin
				AddRPGMenuItem_Top( RPM , #$0 + '[edit menu order]' , SELECTMENU_MenuOrder_Edit );
			end;
{$ENDIF DEBUG}

			{ Select from the menu. }
			RPM^.SelectItem := SI;
			RPM^.TopItem    := TI;
			MenuDesc := UTF8_MsgString( 'BACKPACK_Directions' , 'UninstallSoftware' );
			N := SelectMenu( RPM , @MiscProcRedraw);
			SI := RPM^.SelectItem;
			TI := RPM^.TopItem;
			V := RPMLocateByPosition( RPM , SI )^.value;
			DisposeRPGMenu( RPM );

			{ If a software program was selected, uninstall it and place }
			{ it in the inventory. }
			if ( SELECTMENU_Enable <= N ) then begin
				S := RetrieveGearSib( Compy^.SubCom , N );
				DelinkGear( Compy^.SubCom , S );
				InsertInvCom( PC , S );
			end else if ( SELECTMENU_MenuOrder_Edit = N ) then begin
				if ( SELECTMENU_Enable <= V ) then begin
					Part := RetrieveGearSib( Compy^.SubCom , V );
					SwapMenu( @ZONE_InvMenu , Part );
				end else begin
					SwapMenu( @ZONE_InvMenu , Compy^.SubCom , NIL );
				end;
			end;
		until ( SELECTMENU_Cancel = N ) or IsLegalSubCom( Compy , SW );
	end;
var
	Compy: GearPtr;
begin
	BP_GB := GB;
	Compy := SelectComputer;
	if ( Compy <> Nil ) then begin
		if ( ZetaGigs( Compy ) >= ZetaGigs( SW ) ) then begin
			if not IsLegalSubCom( Compy , SW ) then FreeSoftwareSpace( Compy );
			if IsLegalSubCom( Compy , SW ) then begin
				DelinkGear( SW^.Parent^.InvCom , SW );
				InsertSubCom( Compy , SW );
				DialogMsg( ReplaceHash( MsgString( 'BACKPACK_InstallSoftware' ) , GearName( SW ) ) );
				if GB <> Nil then WaitAMinute( GB , PC , ReactionTime( PC ) );
			end else begin
				DialogMsg( ReplaceHash( MsgString( 'BACKPACK_ComputerTooFull' ) , GearName( SW ) ) );
			end;
		end else begin
			{ This computer is too small to install this program. }
			DialogMsg( ReplaceHash( MsgString( 'BACKPACK_ComputerTooSmall' ) , GearName( SW ) ) );
		end;
	end;
end;

Procedure PCDoPerformance( GB: GameBoardPtr; PC: GearPtr );
	{ The PC is playing an instrument. Whee! Check how many positive }
	{ reactions the PC scores. The PC might also earn money, if the }
	{ public response is positive enough. }
var
	Target: GearPtr;
	Success: LongInt;
	msg: String;
begin
	{ Select a target for this performance. }
	Target := SelectPerformanceTarget( GB , PC );
	msg := ReplaceHash( MsgString( 'PERFORMANCE_Base' ) , GearName( PC ) );

	{ If we have a target, then perform. }
	if Target <> Nil then begin
		{ Call the performance procedure to find out how well the }
		{ player has done. }
		Success := UsePerformance( GB , PC , Target );

		{ Print an appropriate message. }
		if Success > 0 then begin
			{ Good show! The PC made some money as a busker. }
			msg := msg + ' ' + ReplaceHash( MsgString( 'PERFORMANCE_DidWell' + BStr( Random( 3 ) ) ) , BStr( Success ) );
		end else if Success < 0 then begin
			{ The PC flopped. No money made, and possibly damage }
			{ to his reputation. }
			msg := msg + ' ' + MsgString( 'PERFORMANCE_Horrible' + BStr( Random( 3 ) ) );
		end;
	end else begin
		msg := msg + ' ' + MsgString( 'PERFORMANCE_NoAudience' );
		SetNAtt( PC^.NA , NAG_Location , NAS_SmartAction , 0 );
	end;

	DialogMsg( msg );
end;

Procedure StartPerforming( GB: GameBoardPtr; PC: GearPtr );
	{ Start performing on a musical instrument. This procedure will set }
	{ up the continuous action. }
begin
	if ( PC = Nil ) or ( PC^.G <> GG_Character ) then Exit;


	SetNAtt( PC^.NA , NAG_Location , NAS_SmartCount , 4 );
	SetNAtt( PC^.NA , NAG_Location , NAS_SmartAction , NAV_UseSkill );
	SetNAtt( PC^.NA , NAG_Location , NAS_SmartSkill , NAS_Performance );

	{ ...and also exit the backpack. }
	ForceQuit := True;

	PCDoPerformance( GB , PC );
	WaitAMinute( GB , PC , ReactionTime( PC ) );
end;

Procedure ThisItemWasSelected( GB: GameBoardPtr; var LList: GearPtr; TruePC , PC , Item: GearPtr );
	{ TruePC is the primary character, who may be doing repairs }
	{  and stuff. }
	{ PC is the current master being examined, which may well be }
	{  a mecha belonging to the TruePC rather than the TruePC itself. }
	{ LList is a list of mecha and other things which may or may not }
	{  belong to the same team as TruePC et al. }
	{ Item is the piece of wargear currently being examined. }
	{ BP_Redraw must have been set some time before this procedure was called. }

	Procedure DoSwitchBV( Weapon: GearPtr );
		{ Switch the burst value used for Weapon, then store the }
		{ new burst value as the weapon's default. }
	var
		BV: Integer;
	begin
		if Weapon = Nil then Exit;

		{ Determine the current BV; this will tell us what to do next. }
		BV := WeaponBVSetting( Weapon );
		if ( GG_Weapon = Weapon^.G ) then begin
			if ( GS_Missile = Weapon^.S ) then begin
				BV := BV + 1;
				if ( 4 < BV ) then BV := 1;
				SetNAtt( Weapon^.NA , NAG_Prefrences , NAS_DefAtOp , BV );
			end else if ( GS_Ballistic = Weapon^.S ) or ( GS_BeamGun = Weapon^.S ) then begin
				BV := 5 - BV;
				SetNAtt( Weapon^.NA , NAG_Prefrences , NAS_DefAtOp , BV );
			end;
		end;
	end;
var
	TIWS_Menu: RPGMenuPtr;
	SI,TI: Integer;
	N: LongInt;
begin
	N := 0;
	SI := 1;
	TI := 1;
	repeat
		TIWS_Menu := CreateRPGMenu( MenuItem , MenuSelect , @ZONE_InvMenu );

		if ( Item^.G = GG_Tool ) and ( Item^.S = NAS_Performance ) and ( GB <> Nil ) then AddRPGMenuItem( TIWS_Menu , ReplaceHash( MsgString( 'BACKPACK_UseInstrument' ) , GearName( Item ) ) , -9 );
		if ( Item^.G = GG_Consumable ) and ( GB <> Nil ) then AddRPGMenuItem( TIWS_Menu , ReplaceHash( MsgString( 'BACKPACK_EatItem' ) , GearName( Item ) ) , -10 );
		if ( Item^.G = GG_Software ) and IsInvCom( Item ) then AddRPGMenuItem( TIWS_Menu , ReplaceHash( MsgString( 'BACKPACK_TIWS_InstallSoftware' ) , GearName( Item ) ) , -12 );

		if ( GB <> Nil ) and ( SAttArrayValue( Item^.SA , 'USE' ) <> '' ) then AddRPGMenuItem( TIWS_Menu , ReplaceHash( MsgString( 'BACKPACK_UseItemScript' ) , GearName( Item ) ) , -11 );

		if ( Item^.G = GG_Ammo ) and IsInvCom( Item ) then AddRPGMenuItem( TIWS_Menu , MsgString( 'BACKPACK_LoadAmmo' ) , -5 );

		if IsInvCom( Item ) then begin
			if Item^.Parent = PC then begin
				AddRPGMenuItem( TIWS_Menu , ReplaceHash( MsgString( 'BACKPACK_EquipItem' ) , GearName( Item ) ) , -2 );
				if ( FindMaster( Item ) <> Nil ) and ( FindMaster( Item )^.G = GG_Mecha ) and CanBeInstalled( Item ) then begin
					AddRPGMenuItem( TIWS_Menu , ReplaceHash( UTF8_MsgString( 'BACKPACK_Install' ) , GearName( Item ) ) , -8 );
				end;
			end else begin
				AddRPGMenuItem( TIWS_Menu , ReplaceHash( MsgString( 'BACKPACK_UnequipItem' ) , GearName( Item ) ) , -3 );
			end;
			if ( LList <> Nil ) then begin
				if CheckAlongPath_DisallowTransfering( Item ) then begin
					AddRPGMenuItem( TIWS_Menu , UTF8_MsgString( 'ThisItemWasSelected' , 'YouCannotTransferIt' ) , 0 );
				end else begin
					AddRPGMenuItem ( TIWS_Menu , MsgString( 'BACKPACK_TradeItem' ) , -6 );
				end;
			end;
			if CheckAlongPath_DisallowDropping( Item ) then begin
				AddRPGMenuItem( TIWS_Menu , UTF8_MsgString( 'ThisItemWasSelected' , 'YouCannotDropIt' ) , 0 );
			end else begin
				AddRPGMenuItem( TIWS_Menu , MsgString( 'BACKPACK_DropItem' ) , -4 );
			end;
		end else if ( FindMaster( Item ) <> Nil ) and ( FindMaster( Item )^.G = GG_Mecha ) and CanBeExtracted( Item ) then begin
			AddRPGMenuItem( TIWS_Menu , ReplaceHash( UTF8_MsgString( 'BACKPACK_Remove' ) , GearName( Item ) ) , -7 );
		end;

		if ( SeekSubsByG( Item^.SubCom , GG_Ammo ) <> Nil ) and not IsMasterGear( Item ) then AddRPGMenuItem( TIWS_Menu , MsgString( 'BACKPACK_EjectAmmo' ) , 4 );
		if ( SeekSubsByG( Item^.SubCom , GG_Software ) <> Nil ) and not IsMasterGear( Item ) then AddRPGMenuItem( TIWS_Menu , MsgString( 'BACKPACK_EjectSoftware' ) , 5 );

		if GB <> Nil then AddRepairOptions( TIWS_Menu , TruePC , Item );
		if ( Item^.G = GG_Weapon ) or ( ( Item^.G = GG_Ammo ) and ( Item^.S = GS_Grenade ) ) then begin
			AddRPGMenuItem( TIWS_Menu, ReplaceHash( MsgString( 'BACKPACK_QF' + BStr( NAttValue( Item^.NA, NAG_WeaponModifier, NAS_QuickFire ) ) ), GearName( Item ) ), 2 );

			if NAttValue( Item^.NA , NAG_WeaponModifier , NAS_SafetySwitch ) = 0 then begin
				AddRPGMenuItem( TIWS_Menu , MsgString( 'BACKPACK_EngageSafety' ) , 3 );
			end else begin
				AddRPGMenuItem( TIWS_Menu , MsgString( 'BACKPACK_DisengageSafety' ) , 3 );
			end;
		end;
		if ( GG_Weapon = Item^.G ) then begin
			AddRPGMenuItem( TIWS_Menu , ReplaceHash( UTF8_MsgString( 'ThisItemWasSelected' , 'BV' ) , BVTypeName[ WeaponBVSetting( Item ) ] ) , 12 );
		end;
		if GB <> Nil then AddRPGMenuItem( TIWS_Menu , MsgString( 'BACKPACK_UseSkillOnItem' ) , 1 );
		if DisallowSelling( Item ) then begin
			AddRPGMenuItem( TIWS_Menu , UTF8_MsgString( 'ThisItemWasSelected' , 'AllowSelling' ) , 7 );
		end else begin
			AddRPGMenuItem( TIWS_Menu , UTF8_MsgString( 'ThisItemWasSelected' , 'DisallowSelling' ) , 6 );
		end;
		if DisallowDropping( Item ) then begin
			AddRPGMenuItem( TIWS_Menu , UTF8_MsgString( 'ThisItemWasSelected' , 'AllowDropping' ) , 9 );
		end else begin
			AddRPGMenuItem( TIWS_Menu , UTF8_MsgString( 'ThisItemWasSelected' , 'DisallowDropping' ) , 8 );
		end;
		if DisallowTransfering( Item ) then begin
			AddRPGMenuItem( TIWS_Menu , UTF8_MsgString( 'ThisItemWasSelected' , 'AllowTransfering' ) , 11 );
		end else begin
			AddRPGMenuItem( TIWS_Menu , UTF8_MsgString( 'ThisItemWasSelected' , 'DisallowTransfering' ) , 10 );
		end;
		AddRPGMenuItem( TIWS_Menu , MsgString( 'BACKPACK_ExitTIWS' ) , SELECTMENU_Cancel );

		NumKeyMenu_minus( TIWS_Menu );
		AlphaKeyMenu( TIWS_Menu );

		{ Restore the menu item in case this isn't the first iteration. }
		SetItemByValue( TIWS_Menu , N );

		BP_GB := GB;
		BP_Source := Item;
		TIWS_Menu^.SelectItem := SI;
		TIWS_Menu^.TopItem    := TI;
		MenuDesc := UTF8_MsgString( 'BACKPACK_Directions' , 'SelectOperation' );
		N := SelectMenu( TIWS_Menu , @ThisItemRedraw );
		SI := TIWS_Menu^.SelectItem;
		TI := TIWS_Menu^.TopItem;
		DisposeRPGMenu( TIWS_Menu );

		if N > 100 then begin
			DoFieldRepair( GB , TruePC , Item , N-100 );
		end else begin
			case N of
				5: EjectSoftware( GB , LList , PC , Item );
				4: EjectAmmo( GB , LList , PC , Item );
				3: SetNAtt( Item^.NA , NAG_WeaponModifier , NAS_SafetySwitch , 1 - NAttValue( Item^.NA , NAG_WeaponModifier , NAS_SafetySwitch ) );
				2: SwitchQuickFire( Item );
				1: UseSkillOnItem( GB , TruePC , Item );
				-2: EquipItemFrontend( GB , LList , PC , Item );
				-3: UnequipFrontEnd( GB , LList , PC , Item );
				-4: DropFrontEnd( GB , LList , PC , Item );
				-5: InstallAmmoFrontEnd( GB , PC , Item );
				-6: TradeFrontEnd( GB , PC, Item , LList );
				-7: ExtractFrontEnd( GB , TruePC , NIL , PC , Item );
				-8: InstallFrontEnd( GB , TruePC , NIL , PC , Item );
				-9: StartPerforming( GB , PC );
				-10: EatItem( GB , PC , Item );
				-11: UseScriptItem( GB , TruePC , Item , 'USE' );
				-12: InstallSoftware( GB , PC , Item );	{ Install Software }
				6: SetSAttArray( Item^.SA , TAG_DisallowSelling , 'true' );
				7: SetSAttArray( Item^.SA , TAG_DisallowSelling , '' );
				8: SetSAttArray( Item^.SA , TAG_DisallowDropping , 'true' );
				9: SetSAttArray( Item^.SA , TAG_DisallowDropping , '' );
				10: SetSAttArray( Item^.SA , TAG_DisallowTransfering , 'true' );
				11: SetSAttArray( Item^.SA , TAG_DisallowTransfering , '' );
				12: DoSwitchBV( Item );
			end;
		end;
	until ( N < SELECTMENU_Enable ) or ForceQuit;
end;

Function DoInvMenu( GB: GameBoardPtr; var LList: GearPtr; PC,M: GearPtr ): Boolean;
	{ Return TRUE if the user selected Quit. }
	{ M is the MASTER whose inventory we're examining. In a normal case, when }
	{ the PC is examining his own stuff, then M = PC. }
var
	SI,TI: Integer;
	N,V: LongInt;
	Part: GearPtr;
	Desc_MenuA: String;
	Desc_MenuB: String;
begin
	SI := 1;
	TI := 1;
	Repeat
		BP_GB := GB;
		BP_Source := M;
		BP_SeekSibs := False;
		BP_ActiveMenu:= InvRPM;

		InvRPM^.SelectItem := SI;
		InvRPM^.TopItem    := TI;
		Desc_MenuA := UTF8_MsgString( 'BACKPACK_Directions' , 'Inventory' );
		Desc_MenuB := UTF8_MsgString( 'BACKPACK_Directions' , 'Equipment' );
		MenuDesc := ReplaceHash( UTF8_MsgString( 'BACKPACK' , 'Directions' ) , KeyMapEncode( '' , KeyMap[ KMC_SwitchMenu ].KCode ) , Desc_MenuA , Desc_MenuB );
		N := SelectMenu( InvRPM , @MiscProcRedraw );
		SI := InvRPM^.SelectItem;
		TI := InvRPM^.TopItem;

		V := RPMLocateByPosition( InvRPM , SI )^.value;

		{ If an item was selected, pass it along to the appropriate }
		{ procedure. }
		if ( SELECTMENU_Enable < N ) then begin
			ThisItemWasSelected( GB , LList , PC , M , LocateGearByNumber( M , N ) );
			{ Restore the display. }
			UpdateBackpack( M );
		end else if ( SELECTMENU_MenuOrder_Edit = N ) then begin
			if ( SELECTMENU_Enable <= V ) then begin
				Part := LocateGearByNumber( M , V );
				SwapMenu( @ZONE_InvMenu , Part );
			end else begin
				SwapMenu( @ZONE_InvMenu , M^.InvCom , NIL );
			end;
			UpdateBackpack( M );
			N := SELECTMENU_Enable;
		end else if ( -3 = N ) then begin
			DropAllFrontEnd( GB , LList , M , M^.InvCom );
			UpdateBackpack( M );
			N := SELECTMENU_Enable;
		end else if ( -4 = N ) then begin
			TradeAllFrontEnd( GB , M , M^.InvCom , LList );
			UpdateBackpack( M );
			N := SELECTMENU_Enable;
		end;
	until ( N < SELECTMENU_Enable ) or ForceQuit;

	DoInvMenu := ( SELECTMENU_Cancel = N );
end;

Function DoEqpMenu( GB: GameBoardPtr; var LList: GearPtr; PC,M: GearPtr ): Boolean;
	{ Return TRUE if the user selected Quit. }
	{ M is the MASTER whose inventory we're examining. In a normal case, when }
	{ the PC is examining his own stuff, then M = PC. }
var
	SI,TI: Integer;
	N,V: LongInt;
	Part: GearPtr;
	Desc_MenuA: String;
	Desc_MenuB: String;
begin
	SI := 1;
	TI := 1;
	Repeat
		BP_GB := GB;

		BP_Source := M;
		BP_SeekSibs := False;
		BP_ActiveMenu:= EqpRPM;

		EqpRPM^.SelectItem := SI;
		EqpRPM^.TopItem    := TI;
		Desc_MenuA := UTF8_MsgString( 'BACKPACK_Directions' , 'Equipment' );
		Desc_MenuB := UTF8_MsgString( 'BACKPACK_Directions' , 'Inventory' );
		MenuDesc := ReplaceHash( UTF8_MsgString( 'BACKPACK' , 'Directions' ) , KeyMapEncode( '' , KeyMap[ KMC_SwitchMenu ].KCode ) , Desc_MenuA , Desc_MenuB );
		N := SelectMenu( EqpRPM , @EqpRedraw);
		SI := EqpRPM^.SelectItem;
		TI := EqpRPM^.TopItem;

		V := RPMLocateByPosition( EqpRPM , SI )^.value;

		{ If an item was selected, pass it along to the appropriate }
		{ procedure. }
		if ( SELECTMENU_Enable < N ) then begin
			ThisItemWasSelected( GB , LList , PC , M , LocateGearByNumber( M , N ) );
			{ Restore the display. }
			UpdateBackpack( M );
		end else if ( SELECTMENU_MenuOrder_Edit = N ) then begin
			if ( SELECTMENU_Enable <= V ) then begin
				Part := LocateGearByNumber( M , V );
				SwapMenu( @ZONE_EqpMenu , Part );
			end else begin
				SwapMenu( @ZONE_EqpMenu , M^.SubCom , NIL );
			end;
			UpdateBackpack( M );
			N := SELECTMENU_Enable;
		end else if ( -3 = N ) then begin
			DropEqpAllFrontEnd( GB , LList , M , M );
			UpdateBackpack( M );
			N := SELECTMENU_Enable;
		end else if ( -4 = N ) then begin
			TradeEqpAllFrontEnd( GB , M , M , LList );
			UpdateBackpack( M );
			N := SELECTMENU_Enable;
		end;
	until ( N < SELECTMENU_Enable ) or ForceQuit;

	DoEqpMenu := ( SELECTMENU_Cancel = N );
end;


Procedure RealBackpack( GB: GameBoardPtr; var LList: GearPtr; PC,M: GearPtr; StartWithInv: Boolean; BasicRedraw: RedrawProcedureType_with_GearPtr );
	{ This is the backpack routine which should allow the player to go }
	{ through all the stuff in his/her inventory, equip items, drop them, }
	{ reload weapons, and whatnot. It is based roughly upon the procedures }
	{ from DeadCold. }
	{ GB = The gameboard; may be nil. }
	{ LList = The list of stuff surrounding M; where things go when dropped. }
	{ PC = The controller of the party; the primary PC. }
	{ M = The model whose backpack we're dealing with. }
var
	QuitBP: Boolean;
begin
	{ Set up the display. }
	ForceQuit := False;

	BP_Redraw := BasicRedraw;

	{ Initialize menus to NIL, then create them. }
	InvRPM := Nil;
	EqpRPM := Nil;
	UpdateBackpack( M );

	repeat
		if StartWithInv then begin
			QuitBP := DoInvMenu( GB , LList , PC , M );
		end else begin
			QuitBP := DoEqpMenu( GB , LList , PC , M );
		end;

		{ If we have not been ordered to exit the loop, we must }
		{ have been ordered to switch menus. }
		StartWithInv := Not StartWithInv;
	until QuitBP or ForceQuit;

	DisposeRPGMenu( InvRPM );
	DisposeRPGMenu( EqpRPM );
end;

Procedure ArenaHQBackpack( Source,BPPC: GearPtr; BasicRedraw: RedrawProcedureType_with_GearPtr );
	{ Open the backpack menu for a member of this arena unit. }
begin
	RealBackpack( Nil , Source^.SubCom , Source , BPPC , True , BasicRedraw );
end;

Procedure LancemateBackpack( GB: GameBoardPtr; PC,NPC: GearPtr; BasicRedraw: RedrawProcedureType_with_GearPtr );
	{ This is a header for the REALBACKPACK function. }
begin
	RealBackPack( GB , GB^.Meks , PC , NPC , True , BasicRedraw );
end;

Procedure BackpackMenu( GB: GameBoardPtr; PC: GearPtr; StartWithInv: Boolean; BasicRedraw: RedrawProcedureType_with_GearPtr );
	{ This is a header for the REALBACKPACK function. }
begin
	RealBackPack( GB , GB^.Meks , PC , PC , StartWithInv , BasicRedraw );
end;

Procedure MechaPartEditor( GB: GameBoardPtr; var LList: GearPtr; PC,Mek: GearPtr; BasicRedraw: RedrawProcedureType_with_GearPtr );
	{ This procedure may be used to browse through all the various }
	{ bits of a mecha and examine each one individually. }
	{ LList is the list of mecha of which MEK is a sibling. If any item gets removed }
	{ from Mek but can't be placed in the general inventory, it will be put there. }
var
	RPM: RPGMenuPtr;
	SI,TI: Integer;
	N,V: LongInt;
	Part : GearPtr;
begin
	{ Set up the display. }
	DrawBPBorder;
	SI := 1;
	TI := 1;

	Repeat
		RPM := CreateRPGMenu( MenuItem , MenuSelect , @ZONE_FieldHQMenu );
		RPM^.editable_menuorder := True;
		BP_ActiveMenu := RPM;
		BuildGearMenu( RPM , Mek );
		AlphaKeyMenu( RPM );
		AddRPGMenuItem( RPM , UTF8_MsgString('MechaPartBrowser','Exit Browser') , SELECTMENU_Cancel );
		if ( 1 < RPM^.NumItem ) then begin
			{ AddRPGMenuItem_Top( RPM , UTF8_MsgString( 'ThisItemWasSelected', 'DropAll' ), -3 ); }
			{ AddRPGMenuItem_Top( RPM , UTF8_MsgString( 'ThisItemWasSelected', 'TransferAll' ), -4 ); }
		end;
{$IFDEF DEBUG}
		if ( 2 <= RPM^.NumItem ) then begin
			AddRPGMenuItem_Top( RPM , #$0 + '[edit menu order]' , SELECTMENU_MenuOrder_Edit );
		end;
{$ENDIF DEBUG}

		BP_Redraw := BasicRedraw;
		BP_GB := GB;
		BP_Source := Mek;
		BP_SeekSibs := False;

		RPM^.SelectItem := SI;
		RPM^.TopItem    := TI;
		N := SelectMenu( RPM , @MechaPartEditorRedraw);
		SI := RPM^.SelectItem;
		TI := RPM^.TopItem;

		V := RPMLocateByPosition( RPM , SI )^.value;

		DisposeRPGMenu( RPM );

		if ( SELECTMENU_Enable <= N ) then begin
			ThisItemWasSelected( GB , LList , PC , Mek , LocateGearByNumber( Mek , N ) );
		end else if ( SELECTMENU_MenuOrder_Edit = N ) then begin
			if ( SELECTMENU_Enable <= V ) then begin
				Part := LocateGearByNumber( Mek , V );
				SwapMenu( @ZONE_FieldHQMenu , Part );
			end else begin
				SwapMenu( @ZONE_FieldHQMenu , Mek^.InvCom , NIL );
			end;
		end else if ( -3 = N ) then begin
			DropAllFrontEnd( GB , LList , Mek , Mek^.InvCom );
		end else if ( -4 = N ) then begin
			TradeAllFrontEnd( GB , Mek , Mek^.InvCom , LList );
		end;
	until ( SELECTMENU_Cancel = N );
end;


Procedure MechaPartBrowser( GB: GameBoardPtr; PC,NPC,Mek: GearPtr; RDP: RedrawProcedureType );
	{ This procedure may be used to browse through all the various }
	{ bits of a mecha and examine each one individually. }

	Function MechaPartBrowser_PaidServices( Part: GearPtr ): Boolean;
	var
		RPM: RPGMenuPtr;
		Part_M: GearPtr;
		N: LongInt;
	begin
		RPM := CreateRPGMenu( MenuItem , MenuSelect , @ZONE_FieldHQMenu );
		Part_M := FindMaster( Part );
		if IsInvCom( Part ) and ( NIL <> Part_M ) and ( GG_Mecha = Part_M^.G ) and CanBeInstalled( Part ) then begin
			AddRPGMenuItem( RPM , UTF8_MsgString( 'MechaPartBrowser','Install' ) , -8 );
		end;
		if ( not IsInvCom( Part ) ) and ( NIL <> Part_M ) and ( GG_Mecha = Part_M^.G ) and CanBeExtracted( Part ) then begin
			AddRPGMenuItem( RPM , UTF8_MsgString( 'MechaPartBrowser','Remove' ) , -7 );
		end;
		AlphaKeyMenu( RPM );
		AddRPGMenuItem( RPM , UTF8_MsgString('MechaPartBrowser','Exit') , SELECTMENU_Cancel );

		N := SelectMenu( RPM , @PartBrowserRedraw );
		BP_Redraw := @PlainRedraw_wM;
		MechaPartBrowser_PaidServices := False;
		case N of
			-7: begin ExtractFrontend( GB , PC , NPC , Part_M , Part ); MechaPartBrowser_PaidServices := True; end;
			-8: begin InstallFrontend( GB , PC , NPC , Part_M , Part ); MechaPartBrowser_PaidServices := True; end;
		end;

		DisposeRPGMenu( RPM );
	end;
var
	RPM: RPGMenuPtr;
	N,V: LongInt;
	Part: GearPtr;
	Modified: Boolean;
	SI,TI: Integer;
begin
	BP_GB := GB;
	MPB_Redraw := RDP;

	RPM := CreateRPGMenu( MenuItem , MenuSelect , @ZONE_FieldHQMenu );
	{ RPM^.editable_menuorder := True; }
	BuildGearMenu( RPM , Mek );
	AlphaKeyMenu( RPM );
	AddRPGMenuItem( RPM , UTF8_MsgString('MechaPartBrowser','Exit Browser') , SELECTMENU_Cancel );
{$IFDEF DEBUG}
	if ( 2 <= RPM^.NumItem ) then begin
		{ AddRPGMenuItem_Top( RPM , #$0 + '[edit menu order]' , SELECTMENU_MenuOrder_Edit ); }
	end;
{$ENDIF DEBUG}

	Repeat
		BP_Source := Mek;
		BP_SeekSibs := False;
		BP_ActiveMenu := RPM;
		N := SelectMenu( RPM , @PartBrowserRedraw );
		V := RPMLocateByPosition( RPM , RPM^.SelectItem )^.value;
		if ( NIL <> PC ) and ( NIL <> NPC ) and ( 0 < N ) and ( 0 < V ) and CHEAT_Shops_Customize_Mecha then begin
			Part := LocateGearByNumber( Mek , V );
			Modified := MechaPartBrowser_PaidServices( Part );
			if Modified then begin
				SI := RPM^.SelectItem;
				TI := RPM^.TopItem;
				DisposeRPGMenu( RPM );
				RPM := CreateRPGMenu( MenuItem , MenuSelect , @ZONE_FieldHQMenu );
				{ RPM^.editable_menuorder := True; }
				BuildGearMenu( RPM , Mek );
				AlphaKeyMenu( RPM );
				AddRPGMenuItem( RPM , UTF8_MsgString('MechaPartBrowser','Exit Browser') , SELECTMENU_Cancel );
				RPM^.SelectItem := SI;
				RPM^.TopItem    := TI;
			end;
		end else if ( SELECTMENU_MenuOrder_Edit = N ) then begin
			if ( SELECTMENU_Enable <= V ) then begin
				Part := LocateGearByNumber( Mek , V );
				SwapMenu( @ZONE_FieldHQMenu , Part );
			end else begin
				SwapMenu( @ZONE_FieldHQMenu , Mek^.InvCom , NIL );
			end;
		end;
	until ( SELECTMENU_Cancel = N );
	DisposeRPGMenu( RPM );
end;

Procedure MysteryPartBrowser( Mek: GearPtr; RDP: RedrawProcedureType );
	{ Like the above procedure, but provide no actual info. }
	{ This procedure is used when the PC attempts to inspect a target, }
	{ but lacks the proper identification software. }
var
	RPM: RPGMenuPtr;
begin
	BP_GB := NIL;
	MPB_Redraw := RDP;
	BP_Source := Mek;
	RPM := CreateRPGMenu( MenuItem , MenuSelect , @ZONE_FieldHQMenu );
	AddRPGMenuItem( RPM , '????' , SELECTMENU_Cancel );
	SelectMenu( RPM , @MysteryBrowserRedraw );
	DisposeRPGMenu( RPM );
end;

Procedure BrowseDesignFile( GB: GameBoardPtr; List: GearPtr; RDP: RedrawProcedureType );
	{ Choose one of the sibling gears from LIST and display its properties. }
var
	BrowseMenu: RPGMenuPtr;
	Part: GearPtr;
	N,V: LongInt;
begin
	BP_GB := GB;

	{ Create the menu. }
	BrowseMenu := CreateRPGMenu( MenuItem , MenuSelect , @ZONE_FieldHQMenu );
	BrowseMenu^.editable_menuorder := True;

	{ Add each of the gears to the menu. }
	BuildSiblingMenu( BrowseMenu , List );
	RPMSortAlpha( BrowseMenu );
	AlphaKeyMenu( BrowseMenu );
	AddRPGMenuItem( BrowseMenu , UTF8_MsgString( 'BrowseDesignFile', 'Cancel') , SELECTMENU_Cancel );
{$IFDEF DEBUG}
	if ( 2 <= BrowseMenu^.NumItem ) then begin
		AddRPGMenuItem_Top( BrowseMenu , #$0 + '[edit menu order]' , SELECTMENU_MenuOrder_Edit );
	end;
{$ENDIF DEBUG}

	repeat
		MPB_Redraw := RDP;
		BP_Source := List;
		BP_SeekSibs := True;
		BP_ActiveMenu := BrowseMenu;

		{ Select a gear. }
		N := SelectMenu( BrowseMenu, @PartBrowserRedraw );
		V := RPMLocateByPosition( BrowseMenu , BrowseMenu^.SelectItem )^.value;

		if ( SELECTMENU_Enable <= N ) then begin
			Part := RetrieveGearSib( List , N );
			MechaPartBrowser( GB , NIL , NIL , Part , RDP );
			if Part^.G = GG_Theme then CheckTheme( Part );
		end else if ( SELECTMENU_MenuOrder_Edit = N ) then begin
			if ( SELECTMENU_Enable <= V ) then begin
				Part := RetrieveGearSib( List , V );
				if ( NIL = Part^.Parent ) then begin
					SwapMenu( @ZONE_FieldHQMenu , List , NIL );
				end else begin
					SwapMenu( @ZONE_FieldHQMenu , Part );
				end;
			end else begin
				SwapMenu( @ZONE_FieldHQMenu , List , NIL );
			end;
		end;
	until ( SELECTMENU_Cancel = N );

	DisposeRPGMenu( BrowseMenu );
end;


Procedure FHQ_Transfer( GB: GameBoardPtr; var LList: GearPtr; PC,Item: GearPtr );
	{ An item has been selected. Allow it to be transferred to }
	{ one of the team's master gears. }
var
	RPM: RPGMenuPtr;
	M: GearPtr;
	N: LongInt;
	Team: Integer;
begin
	{ Create the menu. }
	RPM := CreateRPGMenu( MenuItem, MenuSelect, @ZONE_FieldHQMenu );
	M := LList;
	N := 1;
	Team := NAttValue( PC^.NA , NAG_LOcation , NAS_Team );
	while M <> Nil do begin
		if ( ( NAttValue( M^.NA , NAG_LOcation , NAS_Team ) = Team ) or ( NAttValue( M^.NA , NAG_LOcation , NAS_Team ) = NAV_LancemateTeam ) ) and IsMasterGear( M ) and IsLegalInvcom( M , Item ) then begin
			AddRPGMenuItem( RPM , TeamMateName( GB , M ) , N );
		end;

		M := M^.Next;
		Inc( N );
	end;

	{ Sort the menu, then add an exit option. }
	{ RPMSortAlpha( RPM ); } { In UTF-8, this sort causes a result that I cannot assume. }
	AlphaKeyMenu( RPM );
	AddRPGMenuItem( RPM , MsgString( 'FHQ_ReturnToMain' ) , SELECTMENU_Cancel );

	{ Get a menu selection, then exit the menu. }
	BP_Source := LList;
	BP_SeekSibs := True;
	BP_ActiveMenu := RPM;
	DialogMSG( ReplaceHash( MsgString( 'FHQ_SelectDestination' ), GearName( Item ) ) );
	N := SelectMenu( RPM , @PartBrowserRedraw );
	{ N := SelectMenu( RPM , @MiscProcRedraw ); }

	DisposeRPGMenu( RPM );

	if ( SELECTMENU_Enable <= N ) then begin
		M := RetrieveGearSib( LList , N );
		DelinkGear( LList , Item );
		InsertInvCom( M , Item );
		DialogMsg( MsgString( 'FHQ_ItemMoved' ) );
	end else begin
		DialogMsg( MsgString( 'Cancelled' ) );
	end;
end;

Procedure SelectPortraitRedraw;
var
	msg: String;
begin
	InfoBox( ZONE_Menu );
{$IFDEF ASCII}
{$ELSE}
	if ( NIL <> SP_GB ) and ( NIL <> SP_M ) then begin
		InfoBox( ZONE_Info.GetRect() );
		msg := Temporary_SelectItem^.desc;
		DrawSprite( LocateSprite( msg , SpriteColor( SP_GB , SP_M ) , 100 , 150 ) , ZONE_Info.GetRect() , 0 );
	end;
{$ENDIF}
end;

Procedure SelectPortrait( GB: GameBoardPtr; M: GearPtr );
	{ The player wants to change the colors for sprite for this character. }
	{ The menu will be placed in the Menu area; assume the redrawer will }
	{ show whatever changes are made here. }
var
	RPM: RPGMenuPtr;
	fname: String;
begin
	SP_GB := GB;
	SP_M := M;

	RPM := CreateRPGMenu( MenuItem , MenuSelect , @ZONE_Menu );
	if CHEAT_Unlimited_Portrait then begin
		BuildFileMenu( RPM , Graphics_Directory + 'por_*.*' , False );
	end else begin
		if ( NAV_Male = NAttValue( M^.NA , NAG_CharDescription , NAS_Gender ) ) then begin
			BuildFileMenu( RPM , Graphics_Directory + 'por_m_*.*' , False );
		end else if ( NAV_Female = NAttValue( M^.NA , NAG_CharDescription , NAS_Gender ) ) then begin
			BuildFileMenu( RPM , Graphics_Directory + 'por_f_*.*' , False );
		end else begin
			BuildFileMenu( RPM , Graphics_Directory + 'por_m_*.*' , False );
			BuildFileMenu( RPM , Graphics_Directory + 'por_f_*.*' , False );
			BuildFileMenu( RPM , Graphics_Directory + 'por_n_*.*' , False );
		end;
	end;
	AddRPGMenuItem( RPM , MsgString( 'EXIT' ) , SELECTMENU_Cancel );

	fname := SAttArrayValue( M^.SA , 'SDL_PORTRAIT' );
	while ( RPM^.SelectItem < RPM^.NumItem ) do begin
		if ( RPMLocateByPosition( RPM , RPM^.SelectItem )^.msg = fname ) then begin
			break;
		end;
		Inc( RPM^.SelectItem );
	end;
	RPM^.TopItem := -1;

	fname := SelectFile( RPM , @SelectPortraitRedraw );

	if ( '' <> fname ) then begin
		SetSAttArray( M^.SA , 'SDL_PORTRAIT' , fname );
	end;

	DisposeRPGMenu( RPM );

	SP_GB := NIL;
	SP_M := NIL;
end;

Procedure RenameRedraw_wM( M: GearPtr );
begin
	if SHOW_NPCStatus_FieldHQ then begin
		CombatDisplay( M , BP_GB );
	end else begin
		CombatDisplay( BP_GB );
	end;
	BP_Redraw( BP_Source );
	SetupFHQDisplay;
	BrowserInterfaceInfo( BP_GB , BP_Source , ZONE_ItemsInfo );
end;

Procedure Rename_Mecha( GB: GameBoardPtr; NPC: GearPtr );
	{ Enter a new name for NPC. }
var
	name: String;
	ask_msg: String;
	old_name: String;
	q: Boolean;
	canceled: Boolean;
begin
	BP_GB := GB;
	BP_Source := NPC;
	BP_Redraw := @PlainRedraw_wM;

	q := False;
	old_name := SAttArrayValue( NPC^.SA , 'name_rename' );
	name := GearName( NPC );
	ask_msg := ReplaceHash( MsgString( 'FHQ_Rename_Prompt' ) , GearName( NPC ) );
	repeat
		name := GetStringFromUser( ask_msg , name , @RenameRedraw_wM , NPC , canceled );
		if canceled then begin
			q := True;
		end else if ( '' = name ) then begin
			q := True;
			Remove_Check_DuplicateNames( old_name );
			SetSAttArray( NPC^.SA , 'name_rename' , '' );
		end else if ( UpCase( old_name ) = UpCase( name ) ) then begin
			q := True;
		end else if ( UpCase( SAttArrayValue( NPC^.SA , 'name' ) ) = UpCase( name ) ) then begin
			q := True;
			Remove_Check_DuplicateNames( old_name );
			SetSAttArray( NPC^.SA , 'name_rename' , '' );
		end else if not Check_DuplicateNames( name ) then begin
			q := True;
			Remove_Check_DuplicateNames( old_name );
			Set_Check_DuplicateNames( name );
			SetSAttArray( NPC^.SA , 'name_rename' , name );
		end else begin
			ask_msg := ReplaceHash( MsgString( 'FHQ_Rename_Prompt_Retry' ) , GearName( NPC ) );
		end;
	until q;
end;

Procedure Redesig_Mecha( GB: GameBoardPtr; NPC: GearPtr );
	{ Enter a new desig for NPC. }
var
	desig: String;
begin
	BP_GB := GB;
	BP_Source := NPC;
	BP_Redraw := @PlainRedraw_wM;
	desig := GetStringFromUser( ReplaceHash( MsgString( 'FHQ_Redesig_Prompt' ) , Geardesig( NPC ) ) , Geardesig( NPC ) , @RenameRedraw_wM , NPC );

	SetSAttArray( NPC^.SA , 'desig_redesig' , desig );
end;

Procedure SelectEmblemRedraw;
var
	N: LongInt;
	emblem: String;
	fname: String;
begin
	N := CurrentMenuItemValue( BP_ActiveMenu );
	emblem := '';
	if ( 0 < N ) then begin
		fname := RPMLocateByPosition( BP_ActiveMenu , BP_ActiveMenu^.SelectItem )^.msg;
		if ( 0 < Length( fname ) ) then begin
			emblem := Copy( fname , ( Length( EMBLEM_PREFIX ) + 1 ) , ( Length( fname ) - Length( EMBLEM_PREFIX ) - Length( EMBLEM_SUFFIX ) ) );
		end;
	end;
	SetSAttArray( BP_Source^.SA , TAG_CUSTOMIZATION_EMBLEM , emblem );

	BP_Redraw( BP_Source );
	SetupFHQDisplay;
	BrowserInterfaceInfo( BP_GB , BP_Source , ZONE_ItemsInfo );
end;

Procedure Reemblem_Mecha( GB: GameBoardPtr; NPC: GearPtr );
	{ Enter a new emblem for NPC. }
var
	emblem: String;
	RPM: RPGMenuPtr;
	fname: String;
begin
	emblem := SAttArrayValue( NPC^.SA , TAG_CUSTOMIZATION_EMBLEM );

	RPM := CreateRPGMenu( MenuItem , MenuSelect , @ZONE_FieldHQMenu );
	BuildFileMenu( RPM , Graphics_Directory + EMBLEM_PREFIX + '*' + EMBLEM_SUFFIX , False );
	AddRPGMenuItem( RPM , MsgString( 'DELETE' ) , -2 );
	AddRPGMenuItem( RPM , MsgString( 'EXIT' ) , SELECTMENU_Cancel );

	fname := EMBLEM_PREFIX + emblem + EMBLEM_SUFFIX;
	while ( RPM^.SelectItem < RPM^.NumItem ) do begin
		if ( RPMLocateByPosition( RPM , RPM^.SelectItem )^.msg = fname ) then begin
			break;
		end;
		Inc( RPM^.SelectItem );
	end;
	RPM^.TopItem := -1;

	BP_ActiveMenu := RPM;
	fname := SelectFile( RPM , @SelectEmblemRedraw );

	if ( 0 < Length( fname ) ) then begin
		emblem := Copy( fname , ( Length( EMBLEM_PREFIX ) + 1 ) , ( Length( fname ) - Length( EMBLEM_PREFIX ) - Length( EMBLEM_SUFFIX ) ) );
	end else if ( -2 = RPM^.SelectItem ) then begin
		emblem := '';
	end;
	SetSAttArray( NPC^.SA , TAG_CUSTOMIZATION_EMBLEM , emblem );

	DisposeRPGMenu( RPM );
end;

Procedure FHQ_ThisWargearWasSelected( GB: GameBoardPtr; var LList: GearPtr; PC,M: GearPtr; RDP: RedrawProcedureType; BasicRedrawer: RedrawProcedureType_with_GearPtr );
	{ A mecha has been selected by the PC from the FHQ main menu. }
	{ Offer up all the different choices of things the PC can }
	{ do with mecha - select pilot, repair, check inventory, etc. }
var
	RPM: RPGMenuPtr;
	SI,TI: Integer;
	N: LongInt;
begin
	SI := 1;
	TI := 1;
	repeat
		{ Create the FHQ menu. }
		RPM := CreateRPGMenu( MenuItem, MenuSelect, @ZONE_FieldHQMenu );
		RPM^.Mode := RPMNoCleanup;

		if IsMasterGear( M ) then begin
			if IsSafeArea( GB ) or OnTheMap( GB , M ) then AddRPGMenuItem( RPM , MsgString( 'FHQ_GoBackpack' ) , 1 );
		end else if IsSafeArea( GB ) then begin
			AddRPGMenuItem( RPM , MsgString( 'FHQ_Transfer' ) , -3 );
		end;

		if IsSafeArea( GB ) then AddRepairOptions( RPM , PC , M );

		if M^.G = GG_Mecha then begin
			AddRPGMenuItem( RPM , MsgString( 'FHQ_SelectPilot' ) , 2 );
			AddRPGMenuItem( RPM , MsgString( 'FHQ_Rename' ) , 6 );
			AddRPGMenuItem( RPM , MsgString( 'FHQ_Redesig' ) , 7 );
			AddRPGMenuItem( RPM , MsgString( 'FHQ_Reemblem' ) , 8 );
		end;
		if IsSafeArea( GB ) then begin
			if ( GG_Character = M^.G ) then begin
				AddRPGMenuItem( RPM , MsgString( 'FHQ_CharPartEditor' ) , 4 );
			end else begin
				AddRPGMenuItem( RPM , MsgString( 'FHQ_PartEditor' ) , 4 );
			end;
		end;


{$IFNDEF ASCII}
		if M^.G = GG_Mecha then AddRPGMenuItem( RPM , MsgString( 'FHQ_EditColor' ) , 5 );
{$ENDIF}

		NumKeyMenu( RPM );
		AlphaKeyMenu( RPM );

		AddRPGMenuItem( RPM , MsgString( 'FHQ_ReturnToMain' ) , SELECTMENU_Cancel );

		{ Get a selection from the menu, then dispose of it. }
		BP_GB := GB;
		BP_Source := M;
		BP_Redraw := BasicRedrawer;
		MPB_Redraw := RDP;
		RPM^.SelectItem := SI;
		RPM^.TopItem    := TI;
		N := SelectMenu( RPM , @ThisWargearRedraw );
		SI := RPM^.SelectItem;
		TI := RPM^.TopItem;

		DisposeRPGMenu( RPM );

		if N > 100 then begin
			{ A repair option must have been selected. }
			DoFieldRepair( GB , PC , M , N-100 );

		end else begin
			case N of
				1: RealBackpack( GB , LList , PC , M , False , BasicRedrawer );
				2: FHQ_SelectPilotForMecha( GB , M );
				-3: FHQ_Transfer( GB , LList , PC , M );
				4: MechaPartEditor( GB , LList , PC , M , @PlainRedraw_wM );
{$IFNDEF ASCII}
				5: SelectColors( M , BasicRedrawer );
{$ENDIF}
				6: Rename_Mecha( GB , M );
				7: Redesig_Mecha( GB , M );
				8: Reemblem_Mecha( GB , M );
			end;

		end;

	until N < SELECTMENU_Enable;
{$IFNDEF ASCII}
	CleanSpriteList;
{$ENDIF}
end;

Procedure UsableGearMenu( GB: GameBoardPtr; PC: GearPtr );
	{ The PC is about to invoke a usable gear. Take a look and see }
	{ which effects are available, then invoke one of them. }
var
	RPM: RPGMenuPtr;
	N: LongInt;
	Part: GearPtr;
begin
	BP_GB := GB;
	BP_Source := PC;
	BP_Redraw := @PlainRedraw_wM;
	BP_SeekSibs := False;

	RPM := CreateRPGMenu( MenuItem, MenuSelect, @ZONE_FieldHQMenu );
	BuildGearMenu( RPM , PC , GG_Usable );

	{ RPMSortAlpha( RPM ); } { In UTF-8, this sort causes a result that I cannot assume. }
	AlphaKeyMenu( RPM );

	AddRPGMenuItem( RPM , MsgString( 'Cancel' ) , SELECTMENU_Cancel );

	BP_ActiveMenu := RPM;

	N := SelectMenu( RPM , @MechaPartEditorRedraw );
	DisposeRPGMenu( RPM );

	if ( SELECTMENU_Cancel = N ) then exit;

	Part := LocateGearByNumber( PC , N );
	if Part <> Nil then begin
		if Part^.S = GS_Transformation then begin
			if CanDoTransformation( GB , PC , Part ) then begin
				DoTransformation( GB , PC , Part , True );
			end else begin
				DialogMsg( MsgString( 'TRANSFORM_NotNow' ) );
			end;
		end else if Part^.S = GS_LongRangeScanner then begin
			if LongRangeScanEPCost( GB, Part ) > EnergyPoints( FindRoot( Part ) ) then begin
				DialogMsg( MsgString( 'LONGRANGESCAN_NoPower' ) );
			end else if CanLRScanHere( GB, Part ) then begin
				DoLongRangeScan( GB , PC , Part );
			end else begin
				DialogMsg( MsgString( 'LONGRANGESCAN_NotNow' ) );
			end;
		end;
	end;
end;



initialization
begin
{$IFDEF DEBUG}
	ErrorMessage_fork('DEBUG: backpack.pp');
{$ENDIF DEBUG}
end;

finalization
begin
{$IFDEF DEBUG}
	ErrorMessage_fork('DEBUG: backpack.pp(finalization)');
{$ENDIF DEBUG}
end;

end.
