unit gears_base;
	{The building block from which everything in this game}
	{is constructed is called a GEAR. Just seemed a good}
	{thing to name the record, given the name of the game.}
{
	GearHead2, a roguelike mecha CRPG
	Copyright (C) 2005 Joseph Hewitt

	This library is free software; you can redistribute it and/or modify it
	under the terms of the GNU Lesser General Public License as published by
	the Free Software Foundation; either version 2.1 of the License, or (at
	your option) any later version.

	The full text of the LGPL can be found in license.txt.

	This library is distributed in the hope that it will be useful, but
	WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
	General Public License for more details. 

	You should have received a copy of the GNU Lesser General Public License
	along with this library; if not, write to the Free Software Foundation,
	Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
}
{$LONGSTRINGS ON}

interface

uses sysutils,dos,texutil;

Const
	{ ******************************* }
	{ ***  FILE  NAME  CONSTANTS  *** }
	{ ******************************* }

	OS_Dir_Separator = DirectorySeparator;
	OS_Search_Separator = PathSeparator;
	OS_Current_Directory = '.';

	GZ_Archive_BufLen = 16384;
	GZ_Archive_Suffix = '.gz';
	Design_DirName = 'design';
	Design_Directory = Design_DirName + OS_Dir_Separator;
	Series_DirName = 'series';
	Series_Directory = Series_DirName + OS_Dir_Separator;

	Data_DirName = 'gamedata';

	Data_Directory = Data_DirName + OS_Dir_Separator;
	UTF8_Settings_File         = Data_Directory + 'utf8_settings.txt';
	UTF8_Name_File             = Data_Directory + 'utf8_name.txt';
	UTF8_Messages_File         = Data_Directory + 'utf8_messages.txt';
	UTF8_Help_Keymap_Name_File = Data_Directory + 'utf8_keymap_name.txt';
	UTF8_Help_Keymap_Desc_File = Data_Directory + 'utf8_keymap_desc.txt';

	UTF8_NPC_PGT_File           = Data_Directory + 'utf8_pgt.txt';
	UTF8_Standard_Modifier_File = Data_Directory + 'utf8_modifier.txt';

	Graphics_DirName = 'image';
	Graphics_Directory = Graphics_Dirname + OS_Dir_Separator;

	Sound_DirName = 'sound';
	Sound_Directory = Sound_DirName + OS_Dir_Separator;

	Sound_Settings_File = Sound_Directory + 'sound.txt';

	Startup_OK: Boolean = True;

Type
	{*** STRING ATTRIBUTE ***}

	SAttListElementPtr = ^SAttListElement;
	SAttListElement = Record
		prev: SAttListElementPtr;
		next: SAttListElementPtr;
		code: String;
		info: String;
	end;
	SAttList = Record
		size: Longint;
		head: SAttListElementPtr;
		tail: SAttListElementPtr;
	end;
	SAttListPtr = ^SAttList;

	SAttArrayElement = Record
		act: Boolean;
		hash: Longword;
		code: String;
		info: String;
	end;
	SAttArrayElementPtr = ^SAttArrayElement;
	SAttArray = Record
		size: Longint;
		num: Longint;
		AoH: Array of SAttArrayElementPtr;
	end;
	SAttArrayPtr = ^SAttArray;

var
	Save_Game_DirName,Save_Game_Directory,Save_Character_Base,Save_Egg_Base,Save_Unit_Base,Save_Campaign_Base: String;
	Config_Directory,Config_File: String;



Function CreateSAttList: SAttListPtr;
Function NewSAttList: SAttListElementPtr;
Procedure DisposeSAttList( var SList: SAttListPtr );
Procedure RemoveSAttList( SList: SAttListPtr; LMember: SAttListElementPtr );
Function FindSAttList( SList: SAttListPtr; const Code_In: String ): SAttListElementPtr;
Function SetSAttList( var SList: SAttListPtr; Code:String; const Info: String ): SAttListElementPtr;
Function StoreSAttList( var SList: SAttListPtr; const Info: String ): SAttListElementPtr;
Function AddSAttList( var SList: SAttListPtr; const S_Label_in,S_Data: String ): SAttListElementPtr;
Function SAttListValue( SList: SAttListPtr; const Code: String ): String;
Function LoadStringList( const FName_In: String ): SAttListPtr;

Procedure DumpSAttArray( SArray: SAttArrayPtr; flag: Boolean );
Function CreateSAttArray: SAttArrayPtr;
Function NewSAttArray: SAttArrayElementPtr;
Procedure DisposeSAttArray( var SArray: SAttArrayPtr );
Procedure RemoveSAttArray( SArray: SAttArrayPtr; const AMember: SAttArrayElementPtr );
Function FindSAttArray( SArray: SAttArrayPtr; const code_in: String ): SAttArrayElementPtr;
Function SetSAttArray( var SArray: SAttArrayPtr; const p: SAttArrayElementPtr ): SAttArrayElementPtr;
Function SetSAttArray( var SArray: SAttArrayPtr; Code: String; const Info: String ): SAttArrayElementPtr;
Function AddSAttArray( var SArray: SAttArrayPtr; const S_Label_in,S_Data: String ): SAttArrayElementPtr;
Function CloneSAttArray( SArray: SAttArrayPtr ): SAttArrayPtr;
Function SAttArrayValue( SArray: SAttArrayPtr; const Code: String ): String;
Function SAttArrayValueToInt( SArray: SAttArrayPtr; const Code: String ): Integer;
Function LoadStringArray( const FName_In: String ): SAttArrayPtr;

implementation

uses	errmsg,hash;

Function CreateSAttList: SAttListPtr;
var
	it: SAttListPtr;
begin
	New(it);
	if ( NIL = it ) then exit( NIL );

	it^.size := 0;
	it^.head := NIL;
	it^.tail := NIL;

	CreateSAttList := it;
end;

Function NewSAttList: SAttListElementPtr;
var
	it: SAttListElementPtr;
begin
	New(it);
	if ( NIL = it ) then exit( NIL );
{$IFDEF PATCH_GH_PARANOID_SAFER}
	it^.code := '';
	it^.info := '';
{$ENDIF PATCH_GH_PARANOID_SAFER}
	it^.prev := NIL;
	it^.next := NIL;

	NewSAttList := it;
end;

Procedure DisposeSAttList( var SList: SAttListPtr );
	{Dispose of the list, freeing all associated system resources.}
var
	tmp: SAttListElementPtr;
	next: SAttListElementPtr;
begin
	if ( NIL = SList ) then exit;

	tmp := SList^.head;
	while ( NIL <> tmp ) do begin
		next := tmp^.next;
{$IFDEF PATCH_GH_PARANOID_SAFER}
		tmp^.code := '@';
		tmp^.info := '@';
		tmp^.prev := SAttListElementPtr(-1);
		tmp^.next := SAttListElementPtr(-1);
{$ENDIF PATCH_GH_PARANOID_SAFER}
{$IFNDEF PATCH_GH_PARANOID_CHECKER}
		Dispose(tmp);
{$ENDIF PATCH_GH_PARANOID_CHECKER}
		tmp := next;
	end;
{$IFDEF PATCH_GH_PARANOID_SAFER}
	SList^.size := $FFFF;
	SList^.head := SAttListElementPtr(-1);
	SList^.tail := SAttListElementPtr(-1);
{$ENDIF PATCH_GH_PARANOID_SAFER}
{$IFNDEF PATCH_GH_PARANOID_CHECKER}
	Dispose(SList);
{$ENDIF PATCH_GH_PARANOID_CHECKER}
{$IFDEF PATCH_GH_PARANOID_SAFER}
	SList := SAttListPtr(-1);
{$ENDIF PATCH_GH_PARANOID_SAFER}
end;

Procedure RemoveSAttList( SList: SAttListPtr; LMember: SAttListElementPtr );
	{Locate and extract member LMember from list SList.}
	{Then, dispose of LMember.}
begin
	if ( NIL <> LMember^.prev ) then begin
		LMember^.prev^.next := LMember^.next;
	end else begin
		SList^.head := LMember^.next;
	end;
	if ( NIL <> LMember^.next ) then begin
		LMember^.next^.prev := LMember^.prev;
	end else begin
		SList^.tail := LMember^.prev;
	end;
	dec( SList^.size );
{$IFDEF PATCH_GH_PARANOID_SAFER}
	LMember^.code := '@';
	LMember^.info := '@';
	LMember^.prev := SAttListElementPtr(-1);
	LMember^.next := SAttListElementPtr(-1);
{$ENDIF PATCH_GH_PARANOID_SAFER}
{$IFNDEF PATCH_GH_PARANOID_CHECKER}
	Dispose(LMember);
{$ENDIF PATCH_GH_PARANOID_CHECKER}
end;

Function FindSAttList( SList: SAttListPtr; const Code_In: String ): SAttListElementPtr;
	{Search through the list looking for a String Attribute}
	{whose code matches CODE and return its address.}
	{Return Nil if no such SAttList can be found.}
var
	code: String;
	it: SAttListElementPtr;
	it_ptr: Longint;

	Function BinarySearch( lmin, lmax: Longint ): SAttListElementPtr;
	var
		lmid: Longint;
	begin
		while ( lmin <= lmax ) do begin
			lmid := lmin + ( lmax - lmin ) div 2;
			if ( lmid < it_ptr ) then begin
				while ( lmid < it_ptr ) do begin
					it := it^.prev;
					dec( it_ptr );
				end;
			end else begin
				while ( it_ptr < lmid ) do begin
					it := it^.next;
					inc( it_ptr );
				end;
			end;
			if ( code < it^.code ) then begin
				lmax := ( lmid - 1 );
			end else if ( it^.code < code ) then begin
				lmin := ( lmid + 1 );
			end else begin
				exit( it );
			end;
		end;
		exit( NIL );
	end;
begin
	if ( 0 = SList^.size ) then Exit( NIL );

	code := UpCase(Code_In);

	it := SList^.head;
	it_ptr := 0;

	FindSAttList := BinarySearch( 0 , ( SList^.size - 1 ) );
end;

Procedure SortStoreSAttList( SList: SAttListPtr; LMember: SAttListElementPtr );
	Procedure StorePrev( it: SAttListElementPtr );
	var
		tmp: SAttListElementPtr;
	begin
		LMember^.next := it;
		tmp := it^.prev;
		it^.prev := LMember;
		LMember^.prev := tmp;
		if ( NIL = tmp ) then begin
			SList^.head := LMember;
		end else begin
			tmp^.next := LMember;
		end;
		inc( SList^.size );
	end;
	Procedure StoreNext( it: SAttListElementPtr );
	var
		tmp: SAttListElementPtr;
	begin
		LMember^.prev := it;
		tmp := it^.next;
		it^.next := LMember;
		LMember^.next := tmp;
		if ( NIL = tmp ) then begin
			SList^.tail := LMember;
		end else begin
			tmp^.prev := LMember;
		end;
		inc( SList^.size );
	end;
var
	it: SAttListElementPtr;
	it_ptr: Longint;

	Procedure BinarySearch( lmin, lmax: Longint );
	var
		lmid: Longint;
		tmp: SAttListElementPtr;
	begin
		while ( lmin <= lmax ) do begin
			lmid := lmin + ( lmax - lmin ) div 2;
			if ( lmid < it_ptr ) then begin
				while ( lmid < it_ptr ) do begin
					it := it^.prev;
					dec( it_ptr );
				end;
			end else begin
				while ( it_ptr < lmid ) do begin
					it := it^.next;
					inc( it_ptr );
				end;
			end;
			if ( LMember^.code < it^.code ) then begin
				lmax := ( lmid - 1 );
			end else if ( it^.code < LMember^.code ) then begin
				lmin := ( lmid + 1 );
			end else begin
				while ( ( NIL <> it ) and ( it^.code = LMember^.code ) ) do begin
					tmp := it;
					it := it^.next;
				end;
				StoreNext( tmp );
				exit;
			end;
		end;
		if ( it_ptr < lmin ) then begin
			StoreNext( it );
		end else begin
			StorePrev( it );
		end;
	end;
begin
	LMember^.prev := NIL;
	LMember^.next := NIL;

	if ( 0 = SList^.size ) then begin
		SList^.head := LMember;
		SList^.tail := LMember;
		inc( SList^.size );
	end else begin
		it := SList^.head;
		it_ptr := 0;

		BinarySearch( 0 , ( SList^.size - 1 ) );
	end;
end;

Function SetSAttList( var SList: SAttListPtr; Code: String; const Info: String ): SAttListElementPtr;
	{Add string attribute Info to the list. However, a gear}
	{may not have two string attributes with the same name.}
	{So, check to see whether or not the list already contains}
	{a string attribute of this type; if so, just replace the}
	{INFO field. If not, create a new SAttList and fill it in.}
var
	it: SAttListElementPtr;
begin
	Code := UpCase( Code );

	if ( NIL = SList ) then begin
		SList := CreateSAttList;
	end;

	{See if that code already exists in the list,}
	{if not create a new entry for it.}
	it := FindSAttList( SList , Code );

	{Plug in the value.}
	if ( '' = Info ) then begin
		if ( NIL <> it ) then begin
			RemoveSAttList( SList , it );
			dec( SList^.size );
		end;
	end else begin
		if ( NIL = it ) then begin
			it := NewSAttList;
			it^.code := Code;
			it^.info := Info;
			SortStoreSAttList( SList , it );
		end else begin
			it^.code := Code;
			it^.info := Info;
		end;
	end;

	{Return a pointer to the new attribute.}
	SetSAttList := it;
end;

Function StoreSAttList( var SList: SAttListPtr; const Info: String ): SAttListElementPtr;
	{ *** This function destroys the order of data. *** }
	{ Add string attribute Info to the list. This procedure }
	{ doesn't check to make sure this attribute isn't duplicated. }
var
	it: SAttListElementPtr;
	tmp: SAttListElementPtr;
begin
	if ( NIL = SList ) then begin
		SList := CreateSAttList;
	end;

	it := NewSAttList;
	it^.code := #255;
	it^.info := Info;
	it^.prev := NIL;
	it^.next := NIL;

	if ( 0 < SList^.size ) then begin
		tmp := SList^.tail;
		SList^.tail := it;
		if ( NIL <> tmp ) then begin
			tmp^.next := it;
		end;
		it^.prev := tmp;
	end else begin
		SList^.head := it;
		SList^.tail := it;
	end;
	inc( SList^.size );

	{Return a pointer to the new attribute.}
	StoreSAttList := it;
end;

Function AddSAttList( var SList: SAttListPtr; const S_Label_In,S_Data: String ): SAttListElementPtr;
	{ Store this data in the string attributes list with kind-of the }
	{ same label. If the label already exists, store under Label1, }
	{ then the next data under Label2, and so on. }
var
	T: SAttListElementPtr;
	S_Label,code: String;
	Max,N: Longint;
begin
	if ( NIL = SList ) then begin
		SList := CreateSAttList;
	end;

	{ ERROR CHECK- if no line exists currently, just add the basic label. }
	if FindSAttList( SList , S_Label_In ) = Nil then begin
		Exit( SetSAttList( SList , S_Label_In , S_Data ) );
	end;

	{ Find the maximum value of this label currently stored in }
	{ the list. }
	Max := 1;
	S_Label := UpCase( S_Label_In ) + '_';

	{ Scan the list for examples of this label. }
	T := SList^.head;
	while T <> Nil do begin
		code := T^.code;

		{ If the first characters are the same as S_label, this }
		{ is another copy of the list. }
		if Copy( code , 1 , Length( S_Label ) ) = S_Label then begin
			code := Copy( code , Length( S_Label ) + 1 , Length( code ) );
			N := ExtractValue( code );
			if N >= Max then Max := N + 1;
		end;

		T := T^.next;
	end;

	AddSAttList := SetSAttList( SList , S_Label + BStr( Max ) , S_Data );
end;

Function SAttListValue( SList: SAttListPtr; const Code: String ): String;
	{Find a String Attribute which corresponds to Code, then}
	{return its embedded alligator string.}
var
	it: SAttListElementPtr;
begin
	it := FindSAttList( SList , Code );

	if it = Nil then Exit('');

	SAttListValue := it^.info;
end;

Function LoadStringList( const FName_In: String ): SAttListPtr;
	{ Load a list of string attributes from the listed file, }
	{ if it can be found. }
var
	SList: SAttListPtr;
	F: Text;
	S: String;
        FName: String;
begin
	SList := CreateSAttList;
	FName := FSearch( FName_In , '.' );
	if FName <> '' then begin
		Assign( F , FName );
		Reset( F );

		{ Get rid of the opening comment }
		ReadLn( F , S );

		while not EOF( F ) do begin
			ReadLn( F , S );
			if S <> '' then StoreSAttList( SList , S );
		end;

		Close( F );
	end;
	LoadStringList := SList;
end;



Procedure DumpSAttArray( SArray: SAttArrayPtr; flag: Boolean );
var
	i: Longint;
	tmp: SAttArrayElementPtr;
begin
	if ( NIL = SArray ) then exit;
	for i := 0 to ( SArray^.size - 1 ) do begin
		tmp := SArray^.AoH[i];
		if ( NIL <> tmp ) then begin
			if flag then begin
				if ( tmp^.act ) then begin
					writeln(BStr(i) + ':' + 'ACT_ON:' + tmp^.code + ' <' + tmp^.info + '>');
				end else begin
					writeln(BStr(i) + ':' + 'ACT_OFF:' + tmp^.code + ' <' + tmp^.info + '>');
				end;
			end;
		end;
	end;
end;

Function CreateSAttArray( size: Longint ): SAttArrayPtr;
var
	it: SAttArrayPtr;
	i: Longint;
begin
	New(it);
	if ( NIL = it ) then exit( NIL );

	it^.size := size;
	it^.num := 0;
	SetLength( it^.AoH , it^.size );
	for i := 0 to ( size - 1 ) do begin
		it^.AoH[i] := NIL;
	end;

	CreateSAttArray := it;
end;

Function CreateSAttArray: SAttArrayPtr;
begin
	CreateSAttArray := CreateSAttArray( 256 );
end;

Function NewSAttArray( src: SAttArrayElementPtr ): SAttArrayElementPtr;
var
	it: SAttArrayElementPtr;
begin
	if ( NIL = src ) then exit( NIL );

	New(it);
	if ( NIL = it ) then exit( NIL );

	it^.act  := True;
	it^.hash := src^.hash;
	it^.code := src^.code;
	it^.info := src^.info;

	NewSAttArray := it;
end;

Function NewSAttArray: SAttArrayElementPtr;
var
	it: SAttArrayElementPtr;
begin
	New(it);
	if ( NIL = it ) then exit( NIL );
	it^.act  := False;
{$IFDEF PATCH_GH_PARANOID_SAFER}
	it^.hash := $FFFFFFFF;
	it^.code := '';
	it^.info := '';
{$ENDIF PATCH_GH_PARANOID_SAFER}

	NewSAttArray := it;
end;

Function ReallocateHashTable( src: SAttArrayPtr ): SAttArrayPtr;
var
	dst: SAttArrayPtr;
	old_size: Longint;
	new_size: Longint;
	i: Longint;
	p: SAttArrayElementPtr;
begin
	old_size := src^.size;
	new_size := old_size * 2;
	dst := CreateSAttArray( new_size );

	for i := 0 to ( old_size - 1 ) do begin
		p := src^.AoH[i];
		if ( ( NIL <> p ) and ( p^.act ) ) then begin
			SetSAttArray( dst , p );
		end;
	end;
	DisposeSAttArray( src );
	ReallocateHashTable := dst;
end;

Procedure DisposeSAttArray( var SArray: SAttArrayPtr );
var
	tmp: SAttArrayElementPtr;
	i: Longint;
begin
	for i := 0 to ( SArray^.size - 1 ) do begin
		tmp := SArray^.AoH[i];
		if ( NIL <> tmp ) then begin
{$IFDEF PATCH_GH_PARANOID_SAFER}
			SArray^.AoH[i] := SAttArrayElementPtr(-1);
			tmp^.act  := False;
			tmp^.hash := $FFFFFFFF;
			tmp^.code := '@';
			tmp^.info := '@';
{$ENDIF PATCH_GH_PARANOID_SAFER}
{$IFNDEF PATCH_GH_PARANOID_CHECKER}
			Dispose(tmp);
{$ENDIF PATCH_GH_PARANOID_CHECKER}
		end;
	end;
{$IFDEF PATCH_GH_PARANOID_SAFER}
	SArray^.size := $FFFF;
	SArray^.num  := $FFFF;
{$ENDIF PATCH_GH_PARANOID_SAFER}
{$IFNDEF PATCH_GH_PARANOID_CHECKER}
	SetLength( SArray^.AoH , 0 );
	Dispose(SArray);
{$ENDIF PATCH_GH_PARANOID_CHECKER}
{$IFDEF PATCH_GH_PARANOID_SAFER}
	SArray := SAttArrayPtr(-1);
{$ENDIF PATCH_GH_PARANOID_SAFER}
end;

Procedure RemoveSAttArray( SArray: SAttArrayPtr; const AMember: SAttArrayElementPtr );
var
	tmp: SAttArrayElementPtr;
	i: Longint;
	h: Longint;
	h_limit: Longint;
begin
	h := ( AMember^.hash mod SArray^.size );
	if ( 0 = h ) then h_limit := ( SArray^.size - 1 ) else h_limit := ( h - 1 );
	i := h;
	while ( i <> h_limit ) do begin
		tmp := SArray^.AoH[i];
		if ( NIL = tmp ) then break;
		if ( tmp^.act ) then begin
			if ( AMember = tmp ) then begin
				tmp^.act := False;
				exit;
			end;
		end;
		inc( i );
		if ( SArray^.size <= i ) then begin
			i := 0;
		end;
	end;
	ErrorMessage('ERROR- RemoveSAttArray asked to remove a link that doesnt exist.');
end;

Function FindSAttArray( SArray: SAttArrayPtr; const code_in: String ): SAttArrayElementPtr;
var
	code: String;
	tmp: SAttArrayElementPtr;
	i: Longint;
	h: Longint;
	h_limit: Longint;
begin
	if ( NIL = SArray ) then exit( NIL );
	if ( 0 = SArray^.size ) then exit( NIL );
	code := UpCase( code_in );

	h := ( hash_string( code ) mod SArray^.size );
	if ( 0 = h ) then h_limit := ( SArray^.size - 1 ) else h_limit := ( h - 1 );
	i := h;
	while ( i <> h_limit ) do begin
		tmp := SArray^.AoH[i];
		if ( NIL = tmp ) then exit( NIL );
		if ( tmp^.act ) then begin
			if ( code = tmp^.code ) then begin
				exit( tmp );
			end;
		end;
		inc( i );
		if ( SArray^.size <= i ) then begin
			i := 0;
		end;
	end;
	exit( NIL );
end;

Function SetSAttArray( var SArray: SAttArrayPtr; const p: SAttArrayElementPtr ): SAttArrayElementPtr;
var
	i: Longint;
	h: Longint;
	h_limit: Longint;
	tmp: SAttArrayElementPtr;
begin
	if ( NIL = SArray ) then begin
		SArray := CreateSAttArray;
	end;

	if ( ( SArray^.size div 3 ) < SArray^.num ) then begin
		SArray := ReallocateHashTable( SArray );
	end;

	h := ( p^.hash mod SArray^.size );
	if ( 0 = h ) then h_limit := ( SArray^.size - 1 ) else h_limit := ( h - 1 );
	i := h;
	while ( i <> h_limit ) do begin
		tmp := SArray^.AoH[i];
		if ( NIL = tmp ) then begin
			tmp := NewSAttArray;
			tmp^.act  := True;
			tmp^.hash := p^.hash;
			tmp^.code := p^.code;
			tmp^.info := p^.info;
			SArray^.AoH[i] := tmp;
			inc( SArray^.num );
			exit( tmp );
		end;
		if ( p^.code = tmp^.code ) then begin
			if ( not( tmp^.act ) ) then begin
				tmp^.act  := True;
				tmp^.hash := p^.hash;
				tmp^.code := p^.code;
				tmp^.info := p^.info;
				exit( tmp );
			end;
			tmp^.hash := p^.hash;
			tmp^.info := p^.info;
			exit( tmp );
		end;
		inc( i );
		if ( SArray^.size <= i ) then begin
			i := 0;
		end;
	end;
	exit( NIL );
end;

Function SetSAttArray( var SArray: SAttArrayPtr; Code: String; const Info: String ): SAttArrayElementPtr;
var
	i: Longint;
	h_org: Longword;
	h: Longint;
	h_limit: Longint;
	tmp: SAttArrayElementPtr;
{$IF DEFINED(DEBUG) and DEFINED(TRACE)}
	Len, P: Integer;
	C: Char;
{$ENDIF}
begin
	Code := UpCase( Code );
	if ( NIL = SArray ) then begin
		SArray := CreateSAttArray;
	end;
{$IF DEFINED(DEBUG) and DEFINED(TRACE)}
	if ( 'NAME' = Code ) then begin
		Len := Length( Info );
		P := 1;
		while ( P <= Len ) do begin
			C := Info[P];
			if ( #$80 <= C ) then begin
				ErrorMessage_fork( 'WARNING:SetSAttArray:' + Info );
				break;
			end;
			Inc(P);
		end;
	end;
{$ENDIF}

	if ( ( SArray^.size div 3 ) < SArray^.num ) then begin
		SArray := ReallocateHashTable( SArray );
	end;

	h_org := hash_string( Code );
	h := ( h_org mod SArray^.size );
	if ( 0 = h ) then h_limit := ( SArray^.size - 1 ) else h_limit := ( h - 1 );
	i := h;
	while ( i <> h_limit ) do begin
		tmp := SArray^.AoH[i];
		if ( NIL = tmp ) then begin
			tmp := NewSAttArray;
			tmp^.act  := True;
			tmp^.hash := h_org;
			tmp^.code := Code;
			tmp^.info := Info;
			SArray^.AoH[i] := tmp;
			inc( SArray^.num );
			exit( tmp );
		end;
		if ( Code = tmp^.code ) then begin
			if ( not( tmp^.act ) ) then begin
				tmp^.act  := True;
				tmp^.hash := h_org;
				tmp^.code := Code;
				tmp^.info := Info;
				exit( tmp );
			end;
			tmp^.hash := h_org;
			tmp^.info := Info;
			exit( tmp );
		end;
		inc( i );
		if ( SArray^.size <= i ) then begin
			i := 0;
		end;
	end;
	exit( NIL );
end;

Function AddSAttArray( var SArray: SAttArrayPtr; const S_Label_in,S_Data: String ): SAttArrayElementPtr;
var
	T: SAttArrayElementPtr;
	S_Label,code: String;
	Max,N: Longint;
	i: Longint;
begin
	if ( NIL = FindSAttArray( SArray , S_Label_In ) ) then begin
		Exit( SetSAttArray( SArray , S_Label_In , S_Data ) );
	end;

	Max := 1;
	S_Label := UpCase( S_Label_In ) + '_';

	for i := 0 to ( SArray^.size - 1 ) do begin
		T := SArray^.AoH[i];
		if ( ( NIL <> T ) and ( T^.act ) ) then begin
			code := T^.code;
			if Copy( code , 1 , Length( S_Label ) ) = S_Label then begin
				code := Copy( code , Length( S_Label ) + 1 , Length( code ) );
				N := ExtractValue( code );
				if ( Max <= N ) then begin
					Max := N + 1;
				end;
			end;
		end;
	end;

	AddSAttArray := SetSAttArray( SArray , S_Label + BStr( Max ) , S_Data );
end;

Function CloneSAttArray( SArray: SAttArrayPtr ): SAttArrayPtr;
var
	i: Longint;
begin
	if ( NIL = SArray ) then exit( NIL );
	CloneSAttArray := CreateSAttArray( SArray^.size );
	for i := 0 to ( SArray^.size - 1 ) do begin
		if ( ( NIL <> SArray^.AoH[i] ) and ( SArray^.AoH[i]^.act ) ) then begin
			SetSAttArray( CloneSAttArray , SArray^.AoH[i] );
		end;
	end;
	CloneSAttArray^.num := SArray^.num;
end;

Function SAttArrayValue( SArray: SAttArrayPtr; const Code: String ): String;
var
	it: SAttArrayElementPtr;
begin
	it := FindSAttArray( SArray , Code );
	if ( NIL = it ) then exit('');
	SAttArrayValue := it^.info;
end;

Function SAttArrayValueToInt( SArray: SAttArrayPtr; const Code: String ): Integer;
var
	it: SAttArrayElementPtr;
	S: String;
begin
	it := FindSAttArray( SArray , Code );
	if ( NIL = it ) then exit(0);

	S := it^.info;
	if ( '' = S ) then exit(0);

	SAttArrayValueToInt := StrToInt(S);
end;

Function LoadStringArray( const FName_In: String ): SAttArrayPtr;
var
	SArray: SAttArrayPtr;
	F: Text;
	S: String;
        FName: String;
begin
	SArray := CreateSAttArray;
	FName := FSearch( FName_In , '.' );
	if FName <> '' then begin
		Assign( F , FName );
		Reset( F );

		ReadLn( F , S );

		while not EOF( F ) do begin
			ReadLn( F , S );
			if S <> '' then SetSAttArray( SArray , RetrieveACode( S ) , RetrieveAString( S ) );
		end;

		Close( F );
	end;
	LoadStringArray := SArray;
end;



Procedure CheckDirectoryPresent;
	{ Make sure that the default save directory exists. If not, }
	{ create it. }
	Function DirExists( dir: String ): Boolean;
{$IF DEFINED(FPC_FULLVERSION) and (30200 <= FPC_FULLVERSION)}
	const
  {$IF DEFINED(WINDOWS)}
		FollowLink: Boolean = False;
  {$ELSE}
		FollowLink: Boolean = True;
  {$ENDIF}
	begin
		DirExists := DirectoryExists( dir , FollowLink );
{$ELSE}
	begin
		DirExists := DirectoryExists( dir );
{$ENDIF}
	end;
begin
	if not DirExists( Config_Directory ) then begin
		MkDir( Config_Directory );
	end;
	if not DirExists( Save_Game_Directory ) then begin
		MkDir( Save_Game_Directory );
	end;


	{ Check to make sure all the other directories can be found. }
	Startup_OK := DirExists( Design_DirName );
	Startup_OK := Startup_OK and DirExists( Series_DirName );
	Startup_OK := Startup_OK and DirExists( Data_DirName );
{$IFNDEF ASCII}
	Startup_OK := Startup_OK and DirExists( Graphics_DirName );
{$ENDIF}
	Startup_OK := Startup_OK and DirExists( Sound_DirName );
end;



initialization
begin
{$IFDEF DEBUG}
	ErrorMessage_fork('DEBUG: gears_base.pp');
{$ENDIF DEBUG}
	{ Make sure we have the required data directories. }
	if paramcount() > 0 then begin
		Config_Directory := IncludeTrailingPathDelimiter( paramstr(1) );
	end else begin
{$IFDEF WINDOWS}
		Config_Directory := GetUserDir() + OS_Dir_Separator + 'gearhead2' + OS_Dir_Separator;
{$ELSE}
		Config_Directory := GetAppConfigDir(False);
{$ENDIF}
	end;
	Config_File := Config_Directory + 'gearhead2.cfg';

	Save_Game_DirName := 'savegame';
	Save_Game_Directory := Config_Directory + Save_Game_Dirname + OS_Dir_Separator;

	Save_Character_Base := Save_Game_Directory + 'CHA';
	Save_Egg_Base := Save_Game_Directory + 'EGG';
	Save_Unit_Base := Save_Game_Directory + 'GHU';
	Save_Campaign_Base := Save_Game_Directory + 'RPG';

	{ Make sure we have the required data directories. }
	CheckDirectoryPresent;
end;

finalization
begin
{$IFDEF DEBUG}
	ErrorMessage_fork('DEBUG: gears_base.pp(finalization)');
{$ENDIF DEBUG}
end;

end.
