unit menugear;
	{ This is the RPGMenus / Gear Tree utilities unit. }
{
	GearHead2, a roguelike mecha CRPG
	Copyright (C) 2005 Joseph Hewitt

	This library is free software; you can redistribute it and/or modify it
	under the terms of the GNU Lesser General Public License as published by
	the Free Software Foundation; either version 2.1 of the License, or (at
	your option) any later version.

	The full text of the LGPL can be found in license.txt.

	This library is distributed in the hope that it will be useful, but
	WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
	General Public License for more details. 

	You should have received a copy of the GNU Lesser General Public License
	along with this library; if not, write to the Free Software Foundation,
	Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
}
{$LONGSTRINGS ON}

interface

uses gears_base,gears,locale,
{$IFDEF ASCII}
	vidgfx,vidmenus;
{$ELSE}
	sdl,sdlgfx,sdlmenus;
{$ENDIF}

Procedure BuildGearMenu( RPM: RPGMenuPtr; Master: GearPtr; G: Integer; IncludeDestroyed: Boolean );
Procedure BuildGearMenu( RPM: RPGMenuPtr; Master: GearPtr; G: Integer );
Procedure BuildGearMenu( RPM: RPGMenuPtr; Master: GearPtr );

Procedure BuildEquipmentMenu( RPM: RPGMenuPtr; Master: GearPtr; Show_SubItem: Boolean );
Procedure BuildInventoryMenu( RPM: RPGMenuPtr; Master: GearPtr; UseSaleTag: Boolean; Show_SubItem: Boolean );
Procedure BuildSlotMenu( RPM: RPGMenuPtr; Master,Item: GearPtr; ShowSub: Boolean );
Function BuildSubMenu( RPM: RPGMenuPtr; Master,Item: GearPtr; DoMultiplicityCheck: Boolean; const ShowSub: Boolean ): Boolean;

Function LocateGearByNumber( Master: GearPtr; Num: LongInt ): GearPtr;
Function FindNextWeapon( GB: GameBoardPtr; Master,Weapon: GearPtr; MinRange: Integer ): GearPtr;
Function FindGearIndex( Master , FindThis: GearPtr ): LongInt;

Procedure BuildSiblingMenu( RPM: RPGMenuPtr; LList: GearPtr );
Procedure BuildSiblingMenu_with_TAG( RPM: RPGMenuPtr; LList: GearPtr );
Procedure BuildSiblingMenu_with_Mark( RPM: RPGMenuPtr; LList: GearPtr; const MarkedItem: GearPtr );

Procedure AlphaKeyMenu( RPM: RPGMenuPtr );
Procedure NumKeyMenu( RPM: RPGMenuPtr );
Procedure NumKeyMenu_minus( RPM: RPGMenuPtr );

{$IFDEF ASCII}
Function SwapMenu( Z: vgfx_zoneptr; var FirstPart: GearPtr; A: GearPtr ):Boolean;
Function SwapMenu( Z: vgfx_zoneptr; Part: GearPtr ):Boolean;
{$ELSE ASCII}
Function SwapMenu( Z: DynamicRectPtr; var FirstPart: GearPtr; A: GearPtr ):Boolean;
Function SwapMenu( Z: DynamicRectPtr; Part: GearPtr ):Boolean;
{$ENDIF ASSCII}

implementation

uses
{$IFDEF DEBUG}
	errmsg,
{$ENDIF DEBUG}
	utf8msg,ui4gh,effects,gearutil,ghswag,ghweapon,texutil;

Procedure BuildGearMenu( RPM: RPGMenuPtr; Master: GearPtr; G: Integer; IncludeDestroyed: Boolean );
	{ Search through MASTER, adding to menu RPM any part which }
	{ corresponds to descriptor G. Add each matching part to the }
	{ menu, along with its locator number. }
var
	N: LongInt;
{ PROCEDURES BLOCK }
	Procedure CheckAlongPath( Part: GearPtr; AddToMenu: Boolean );
		{ CHeck along the path specified. }
	var
		PartOK: Boolean;
	begin
		while Part <> Nil do begin
			Inc(N);
			PartOK := NotDestroyed( Part );
			if ( Part^.G = G ) and AddToMenu and ( PartOK or IncludeDestroyed ) then AddRPGMenuItem( RPM , GearName( Part ) , N );
			if Part^.G = GG_Cockpit then begin
				{ Don't add parts beyond the cockpit barrier. }
				CheckAlongPath( Part^.InvCom , False );
				CheckAlongPath( Part^.SubCom , False );
			end else begin
				CheckAlongPath( Part^.InvCom , AddToMenu and ( PartOK or IncludeDestroyed ) );
				CheckAlongPath( Part^.SubCom , AddToMenu and ( PartOK or IncludeDestroyed ) );
			end;
			Part := Part^.Next;
		end;
	end;
begin
	N := 0;
	if Master^.G = G then AddRPGMenuItem( RPM , GearName( Master ) , 0 );
	CheckAlongPath( Master^.InvCom , True );
	CheckAlongPath( Master^.SubCom , True );
end; { BuildGearMenu }

Procedure BuildGearMenu( RPM: RPGMenuPtr; Master: GearPtr; G: Integer );
	{ Call the above procedure, counting even destroyed gears. }
begin
	BuildGearMenu( RPM , Master , G , True );
end;

Procedure BuildGearMenu( RPM: RPGMenuPtr; Master: GearPtr );
	{ Search through MASTER, adding to menu all parts. }
const
	InvStr = '+';
	SubStr = '>';
var
	N: LongInt;
{ PROCEDURES BLOCK }
	Procedure CheckAlongPath( Part: GearPtr; TabPos,Prefix: String );
		{ CHeck along the path specified. }
	begin
		while Part <> Nil do begin
			Inc(N);
			if Part^.G <> GG_AbsolutelyNothing then AddRPGMenuItem( RPM , TabPos + Prefix + GearName( Part ) , N );
			CheckAlongPath( Part^.InvCom , TabPos + '   ' , InvStr );
			CheckAlongPath( Part^.SubCom , TabPos + '   ' , SubStr );
			Part := Part^.Next;
		end;
	end;{CheckAlongPath}
begin
	N := 0;
	AddRPGMenuItem( RPM , GearName( Master ) , 0 );
	CheckAlongPath( Master^.InvCom , '   ' , '+' );
	CheckAlongPath( Master^.SubCom , '   ' , '>' );
end; { BuildGearMenu }

Procedure BuildEquipmentMenu( RPM: RPGMenuPtr; Master: GearPtr; Show_SubItem: Boolean );
	{ Create a menu for this master's equipment. Equipment is defined as }
	{ an InvCom of any part other than the master itself. }
var
	N: LongInt;
	Procedure CheckAlongPath( Part: GearPtr; IsInv: Boolean );
		{ CHeck along the path specified. }
	var
		msg: String;
	begin
		while Part <> Nil do begin
			Inc(N);
			if ( Part^.G <> GG_AbsolutelyNothing ) and IsInv then begin
				{ Creating a message line for this equipment is made tricky by the }
				{ fact that a pilot riding in a mecha has a separate inventory from }
				{ the mecha itself. }
				if IsMasterGear( Part^.Parent ) then begin
					if SHOW_EqpMenu_SubItem_FullGearName then begin
						msg := '[' + GearName( Part^.Parent ) + '] ' + FullGearName( Part );
					end else begin
						msg := '[' + GearName( Part^.Parent ) + '] ' + GearName( Part );
					end;
				end else begin
					if SHOW_EqpMenu_SubItem_FullGearName then begin
						msg := FullGearName( Part ) + ' [';
					end else begin
						msg := GearName( Part ) + ' [';
					end;
					if FindMaster(Part)^.Parent <> Nil then msg := msg + GearName( FindMaster( Part ) ) + ':';
					msg := msg + GearName( Part^.Parent ) + ']';
				end;
				AddRPGMenuItem( RPM , msg , N , FormatDescString( Part ) );
			end;
			CheckAlongPath( Part^.InvCom , True );
			CheckAlongPath( Part^.SubCom , False );
			Part := Part^.Next;
		end;
	end;{CheckAlongPath}
const
	InvStr = '+';
	SubStr = '>';
	Procedure CheckAlongPath( Part: GearPtr; const IsInv,IsInvSub: Boolean; const TabPos,Prefix: String; Show: Boolean );
	var
		msg: String;
	begin
		while ( NIL <> Part ) do begin
			Inc(N);
			if ( Part^.G <> GG_AbsolutelyNothing ) and Show then begin
				if IsInv then begin
					if IsMasterGear( Part^.Parent ) then begin
						if SHOW_EqpMenu_SubItem_FullGearName then begin
							msg := '[' + GearName( Part^.Parent ) + '] ' + FullGearName( Part );
						end else begin
							msg := '[' + GearName( Part^.Parent ) + '] ' + GearName( Part );
						end;
					end else begin
						if SHOW_EqpMenu_SubItem_FullGearName then begin
							msg := FullGearName( Part ) + ' [';
						end else begin
							msg := GearName( Part ) + ' [';
						end;
						if ( NIL <> FindMaster(Part)^.Parent ) then begin
							msg := msg + GearName( FindMaster( Part ) ) + ':';
						end;
						msg := msg + GearName( Part^.Parent ) + ']';
					end;
					AddRPGMenuItem( RPM, msg, N, FormatDescString(Part) );
				end else if IsInvSub then begin
					AddRPGMenuItem( RPM, #$0 + TabPos + Prefix + GearName(Part), N, FormatDescString(Part) );
				end;
			end;
			if ( ( not ( Cheat_No_CockpitBarrier ) ) and ( GG_Cockpit = Part^.G ) ) then begin
				{ Don't add parts beyond the cockpit barrier. }
				CheckAlongPath( Part^.InvCom, True, True,     '',             InvStr, Show );
				CheckAlongPath( Part^.SubCom, False,IsInvSub, TabPos + '   ', SubStr, False );
			end else begin
				CheckAlongPath( Part^.InvCom, True, True,     '',             InvStr, Show );
				CheckAlongPath( Part^.SubCom, False,IsInvSub, TabPos + '   ', SubStr, Show );
			end;
			Part := Part^.Next;
		end;
	end;
begin
	N := 0;
	if Show_SubItem then begin
		CheckAlongPath( Master^.InvCom , False,False, '', InvStr, True );
		CheckAlongPath( Master^.SubCom , False,False, '', SubStr, True );
	end else begin
		CheckAlongPath( Master^.InvCom , False );
		CheckAlongPath( Master^.SubCom , False );
	end;
end; {BuildEquipmentMenu}

Procedure BuildInventoryMenu( RPM: RPGMenuPtr; Master: GearPtr; UseSaleTag: Boolean; Show_SubItem: Boolean );
	{ Create a menu for this master's inventory. Inventory is defined as }
	{ any InvCom of the master. }
var
	N: LongInt;
	T: Integer;
	Part: GearPtr;
	num_items,num_extras: Integer;
	was_already_added: Array of Boolean;
	Procedure CountTheKids( P: GearPtr );
		{ This procedure ignores the sub/inv components of things }
		{ in the general inventory, but they have to be counted so }
		{ the locator numbers will work properly. }
	begin
		While P <> Nil do begin
			Inc( N );
			if P^.InvCom <> Nil then CountTheKids( P^.InvCom );
			if P^.SubCom <> Nil then CountTheKids( P^.SubCom );
			P := P^.Next;
		end;
	end;
	Function IMString( P: GearPtr; num_copies: Integer ): String;
		{ Given part P, return a string to use in the menu. }
	var
		msg,msg2: String;
		AmmoMax,AmmoUsed: Integer;
	begin
		msg := FullGearName( P );

		{ Add extra information, depending upon item type. }
		if UseSaleTag then begin
			msg2 := SAttArrayValue( P^.SA , 'SALETAG' );
			if msg2 <> '' then msg := msg + ' (' + msg2 + ')';
		end else begin
			if P^.G = GG_Weapon then begin
				msg := msg + '  (DC:' + BStr( ScaleDC( P^.V , P^.Scale ) ) + ')';
			end else if ( P^.G = GG_ExArmor ) or ( P^.G = GG_Shield ) then begin
				msg := msg + '  [AC:' + BStr( GearMaxArmor( P ) ) + ']';
			end else if P^.G = GG_Ammo then begin
				AmmoMax  := P^.Stat[ STAT_AmmoPresent ];
				AmmoUsed := NAttValue( P^.NA , NAG_WeaponModifier , NAS_AmmoSpent );
				msg := msg + '  (' + BStr( AmmoMax - AmmoUsed ) + '/' + BStr( AmmoMax ) + 'a)';
			end;
		end;

		if num_copies > 0 then msg := msg + ' x' + Bstr( num_copies + 1 );
		if DisallowSelling( P ) then begin
			msg := msg + UTF8_MsgString( 'BuildInventoryMenu' , 'CannotBeSold' );
		end;
		if DisallowDropping( P ) then begin
			msg := msg + UTF8_MsgString( 'BuildInventoryMenu' , 'CannotBeDropped' );
		end;
		if DisallowTransfering( P ) then begin
			msg := msg + UTF8_MsgString( 'BuildInventoryMenu' , 'CannotBeTransfered' );
		end;

		IMString := Msg;
	end;
	Function CountIdenticalSibs( P1: GearPtr; T1: Integer ): Integer;
		{ Return how many identical siblings this gear has. }
	var
		P2: GearPtr;
		it: Integer;
	begin
		it := 0;
		P2 := P1^.Next;
		while P2 <> Nil do begin
			inc( T1 );
			if GearsAreIdentical( P1 , P2 ) then begin
				Inc( it );
				was_already_added[ t1 ] := True;
			end;
			P2 := P2^.Next;
		end;
		CountIdenticalSibs := it;
	end;
const
	InvStr = '+';
	SubStr = '>';
	Procedure CheckAlongPath( Part: GearPtr; const TabPos,Prefix: String );
	begin
		while Part <> Nil do begin
			Inc(N);
			if Part^.G <> GG_AbsolutelyNothing then begin
				AddRPGMenuItem( RPM, #$0 + TabPos + Prefix + IMString(Part,0), N, FormatDescString(Part) );
			end;
			CheckAlongPath( Part^.InvCom , TabPos + '   ' , InvStr );
			CheckAlongPath( Part^.SubCom , TabPos + '   ' , SubStr );
			Part := Part^.Next;
		end;
	end;
begin
	N := 0;
	Part := Master^.InvCom;
	if Part = Nil then Exit;

	{ Determine the number of items in the inventory, and initialize the Was_Already_Added array. }
	num_items := NumSiblingGears( Part );
	SetLength( was_already_added , num_items + 1 );
	for t := 1 to num_items do was_already_added[ t ] := False;
	T := 0;

	while Part <> Nil do begin
		{ N is the tree index for the current part. T is its sibling index. }
		Inc( N );
		Inc( T );

		if not was_already_added[ t ] then begin
			num_extras := CountIdenticalSibs( Part , T );
			AddRPGMenuItem( RPM , IMString( Part , num_extras ) , N , FormatDescString( Part ) );
			if Show_SubItem then begin
				CheckAlongPath( Part^.InvCom, '   ', InvStr );
				CheckAlongPath( Part^.SubCom, '   ', SubStr );
			end else begin
				CountTheKids( Part^.InvCom );
				CountTheKids( Part^.SubCom );
			end;
		end else begin
			CountTheKids( Part^.InvCom );
			CountTheKids( Part^.SubCom );
		end;
		Part := Part^.Next;
	end;

end;

Procedure BuildSlotMenu( RPM: RPGMenuPtr; Master,Item: GearPtr; ShowSub: Boolean );
	{ Search through MASTER, adding to menu all parts which can }
	{ equip ITEM. }
var
	N: LongInt;
{ PROCEDURES BLOCK }
	Function IMString_addParent( const Part: GearPtr; const basemsg: String ): String;
	var
		msg: String;
	begin
		if IsMasterGear( Part^.Parent ) then begin
			msg := '[' + GearName( Part^.Parent ) + '] ' + basemsg;
		end else begin
			msg := basemsg + ' [';
			if ( NIL <> FindMaster(Part)^.Parent ) then begin
				msg := msg + GearName( FindMaster( Part ) ) + ':';
			end;
			msg := msg + GearName( Part^.Parent ) + ']';
		end;
		IMString_addParent := msg;
	end;
	Function IMString( const P: GearPtr ): String;
	var
		msg: String;
		AmmoMax,AmmoUsed: Integer;
	begin
		msg := FullGearName( P );
		if ( P^.G = GG_Weapon ) then begin
			msg := msg + '  (DC:' + BStr( ScaleDC( P^.V , P^.Scale ) ) + ')';
		end else if ( P^.G = GG_ExArmor ) or ( P^.G = GG_Shield ) then begin
			msg := msg + '  [AC:' + BStr( GearMaxArmor( P ) ) + ']';
		end else if ( P^.G = GG_Ammo ) then begin
			AmmoMax  := P^.STat[ STAT_AmmoPresent ];
			AmmoUsed := NAttValue( P^.NA , NAG_WeaponModifier , NAS_AmmoSpent );
			msg := msg + '  (' + BStr( AmmoMax - AmmoUsed ) + '/' + BStr( AmmoMax ) + 'a)';
		end else if ( P^.G = GG_Consumable ) then begin
			msg := msg + '  (' + BStr( P^.V ) + ')';
		end;
		IMString := Msg;
	end;
const
	InvStr = '+';
	SubStr = '>';
	Procedure CheckAlongPath( Part: GearPtr; const LastN: LongInt; const IsHitSub: Boolean; const TabPos,Prefix: String );
		{ CHeck along the path specified. }
	var
		Hit: Boolean;
		NewN: LongInt;
		NewTabPos: String;
		msg: String;
	begin
		while Part <> Nil do begin
			Hit := False;
			NewN := -2;
			NewTabPos := '';
			Inc(N);
			if IsLegalInvCom( Part , Item ) and PartActive( Part ) then begin
				msg := GearName( Part );
				if SHOW_EquipItem_ParentItem then begin
					msg := IMString_addParent( Part , msg );
				end;
				if SHOW_EquipItem_InvStr then begin
					msg := Prefix + msg;
				end;
				AddRPGMenuItem( RPM , msg , N );
				Hit := True;
				NewN := N;
				NewTabPos := '   ';
			end else if SHOW_EquipItem_SubItem and IsHitSub then begin
				AddRPGMenuItem( RPM , #$0 + TabPos + Prefix + IMString(Part) , LastN );
				Hit := True;
				NewTabPos := TabPos + '   ';
			end;
			CheckAlongPath( Part^.InvCom , NewN , Hit , NewTabPos , InvStr );
			CheckAlongPath( Part^.SubCom , NewN , Hit , NewTabPos , SubStr );
			Part := Part^.Next;
		end;
	end;{CheckAlongPath}
begin
	N := 0;
	CheckAlongPath( Master^.InvCom , -2 , False , '' , InvStr );
	CheckAlongPath( Master^.SubCom , -2 , False , '' , SubStr );
end; { BuildSlotMenu }

Function BuildSubMenu( RPM: RPGMenuPtr; Master,Item: GearPtr; DoMultiplicityCheck: Boolean; const ShowSub: Boolean ): Boolean;
	{ Search through MASTER, adding to menu all parts which can }
	{ take ITEM as a subcomponent. }
var
	N: LongInt;
{ PROCEDURES BLOCK }
	Function MenuMsg( Part: GearPtr ): String;
	begin
		MenuMsg := GearName( Part ) + ' (' + BStr( SubComComplexity( Part ) ) + '/' + BStr( ComponentComplexity( Part , NIL ) ) + ')';
	end;
	Function IMString_addParent( const Part: GearPtr; const basemsg: String ): String;
	var
		msg: String;
	begin
		if IsMasterGear( Part^.Parent ) then begin
			msg := '[' + GearName( Part^.Parent ) + '] ' + basemsg;
		end else begin
			msg := basemsg + ' [';
			if ( NIL <> FindMaster(Part)^.Parent ) then begin
				msg := msg + GearName( FindMaster( Part ) ) + ':';
			end;
			msg := msg + GearName( Part^.Parent ) + ']';
		end;
		IMString_addParent := msg;
	end;
	Function IMString( const P: GearPtr ): String;
	var
		msg: String;
		AmmoMax,AmmoUsed: Integer;
	begin
		msg := FullGearName( P );
		if ( P^.G = GG_Weapon ) then begin
			msg := msg + '  (DC:' + BStr( ScaleDC( P^.V , P^.Scale ) ) + ')';
		end else if ( P^.G = GG_ExArmor ) or ( P^.G = GG_Shield ) then begin
			msg := msg + '  [AC:' + BStr( GearMaxArmor( P ) ) + ']';
		end else if ( P^.G = GG_Ammo ) then begin
			AmmoMax  := P^.STat[ STAT_AmmoPresent ];
			AmmoUsed := NAttValue( P^.NA , NAG_WeaponModifier , NAS_AmmoSpent );
			msg := msg + '  (' + BStr( AmmoMax - AmmoUsed ) + '/' + BStr( AmmoMax ) + 'a)';
		end else if ( P^.G = GG_Consumable ) then begin
			msg := msg + '  (' + BStr( P^.V ) + ')';
		end;
		IMString := Msg;
	end;
	Function CheckThisBit( const Part: GearPtr; const Prefix: String ): Boolean;
		{ Check this bit, and maybe add it to the menu. }
	var
		Hit: Boolean;
		msg: String;
	begin
		msg := '';
		Hit := False;

		if DoMultiplicityCheck then begin
			if CanBeInstalled( Part , Item ) then begin
				msg := MenuMsg( Part );
				Hit := True;
			end;
		end else begin
			if IsLegalSubCom( Part , Item ) then begin
				msg := GearName( Part );
				Hit := True;
			end;
		end;

		if ( 1 <= Length(msg) ) then begin
			if SHOW_InstallMisc_InvStr then begin
				msg := Prefix + msg;
			end;
			if SHOW_InstallMisc_ParentItem then begin
				msg := IMString_addParent( Part , msg );
			end;
			if SHOW_InstallMisc_Hit then begin
				msg := '* ' + msg;
			end;
			AddRPGMenuItem( RPM , msg , N );
		end;
		CheckThisBit := Hit;
	end;
const
	InvStr = '+';
	SubStr = '>';
	Procedure CheckAlongPath( Part: GearPtr; const LastN: LongInt; const IsHitSub: Boolean; const TabPos,Prefix: String );
		{ CHeck along the path specified. }
	var
		Hit: Boolean;
		NewN: LongInt;
		NewTabPos: String;
	begin
		while ( NIL <> Part ) do begin
			Inc(N);
			Hit := CheckThisBit( Part, Prefix );
			NewN := -2;
			NewTabPos := '';
			If Hit then begin
				NewN := N;
				NewTabPos := '   ';
			end else begin
				if ShowSub and IsHitSub then begin
					AddRPGMenuItem( RPM, #$0 + TabPos + Prefix + IMString(Part), LastN );
					Hit := True;
					NewTabPos := TabPos + '   ';
				end;
			end;
			CheckAlongPath( Part^.InvCom, NewN, Hit, NewTabPos, InvStr );
			CheckAlongPath( Part^.SubCom, NewN, Hit, NewTabPos, SubStr );
			Part := Part^.Next;
		end;
	end;{CheckAlongPath}
var
	Hit: Boolean;
	NewN: LongInt;
	NewTabPos: String;
begin
	N := 0;
	Hit := CheckThisBit( Master , '' );
	NewN := -2;
	NewTabPos := '';
	if Hit then begin
		{ NewN := N; }
		NewTabPos := '   ';
	end;
	CheckAlongPath( Master^.InvCom , NewN , Hit , NewTabPos , InvStr );
	CheckAlongPath( Master^.SubCom , NewN , Hit , NewTabPos , SubStr );
	BuildSubMenu := Hit;
end; { BuildSubMenu }


Procedure BuildSiblingMenu( RPM: RPGMenuPtr; LList: GearPtr );
	{ Build a menu of these sibling gears. The index numbers will be }
	{ sibling indicies, not the universal gear numbers used in other }
	{ procedures here. }
var
	N: LongInt;
begin
	N := 1;
	while LList <> Nil do begin
		AddRPGMenuItem( RPM , FullGearName( LList ) , N );
		Inc( N );
		LList := LList^.Next;
	end;
end;

Procedure BuildSiblingMenu_with_TAG( RPM: RPGMenuPtr; LList: GearPtr );
	{ Build a menu of these sibling gears. The index numbers will be }
	{ sibling indicies, not the universal gear numbers used in other }
	{ procedures here. }
var
	N: LongInt;
	msg: String;
begin
	N := 1;
	while LList <> Nil do begin
		msg := FullGearName( LList );
		if CheckAlongPath_DisallowTransfering( LList ) then begin
			msg := msg + UTF8_MsgString( 'BuildSiblingMenu' , 'CannotBeTransfered' );
		end;
		AddRPGMenuItem( RPM , msg , N );
		Inc( N );
		LList := LList^.Next;
	end;
end;

Procedure BuildSiblingMenu_with_Mark( RPM: RPGMenuPtr; LList: GearPtr; const MarkedItem: GearPtr );
	{ Build a menu of these sibling gears with mark. The index numbers will be }
	{ sibling indicies, not the universal gear numbers used in other }
	{ procedures here. }
var
	N: Integer;
begin
	N := 1;
	while ( NIL <> LList ) do begin
		if ( MarkedItem = LList ) then begin
			AddRPGMenuItem( RPM , '* ' + FullGearName( LList ) , N );
		end else begin
			AddRPGMenuItem( RPM , '   ' + FullGearName( LList ) , N )
		end;
		Inc( N );
		LList := LList^.Next;
	end;
end;

Function LocateGearByNumber( Master: GearPtr; Num: LongInt ): GearPtr;
	{ Locate the Nth part in the tree. }
var
	N: LongInt;
	TheGearWeWant: GearPtr;
{ PROCEDURES BLOCK. }
	Procedure CheckAlongPath( Part: GearPtr );
		{ CHeck along the path specified. }
	begin
		while ( Part <> Nil ) and ( TheGearWeWant = Nil ) do begin
			Inc(N);
			if N = Num then TheGearWeWant := Part;
			if TheGearWeWant = Nil then CheckAlongPath( Part^.InvCom );
			if TheGearWeWant = Nil then CheckAlongPath( Part^.SubCom );
			Part := Part^.Next;
		end;
	end;
begin
	TheGearWeWant := Nil;
	N := 0;

	{ Part 0 is the master gear itself. }
	if Num < 1 then Exit( Master );

	CheckAlongPath( Master^.InvCom );
	if TheGearWeWant = Nil then CheckAlongPath( Master^.SubCom );

	LocateGearByNumber := TheGearWeWant;
end; { LocateGearByNumber }

Function FindNextWeapon( GB: GameBoardPtr; Master,Weapon: GearPtr; MinRange: Integer ): GearPtr;
	{ This procedure will check recursively through MASTER looking }
	{ for the first weapon (ready to fire) in standard order following PART. }
	{ If MinRange > 0, the weapon's range or throwing range must exceed MinRange. }
	{ If no further weapons are found, it will return the first }
	{ weapon. }
var
	FirstWep,NextWep: GearPtr;
	FoundStart: Boolean;
{ PROCEDURES BLOCK }
	Function WeaponIsOkay( W: GearPtr ): Boolean;
		{ Return TRUE if W is ready to fire and meets our other criteria, or }
		{ FALSE otherwise. }
	begin
		if MinRange = 0 then begin
			WeaponIsOkay := ReadyToFire( GB , Master , W );
		end else begin
			WeaponIsOkay := ReadyToFire( GB , Master , W ) and ( ( WeaponRange( GB , W , RANGE_Long ) >= MinRange ) or ( ThrowingRange( GB , Master , W ) >= MinRange ) );
		end;
	end;
	Procedure CheckAlongPath( Part: GearPtr );
		{ CHeck along the path specified. }
	begin
		while Part <> Nil do begin
			if WeaponIsOkay( Part ) then begin
				if FirstWep = Nil then FirstWep := Part;
				if FoundStart and ( NextWep = Nil ) then NextWep := Part;
			end;

			if Part = Weapon then FoundStart := True;

			CheckAlongPath( Part^.InvCom );
			CheckAlongPath( Part^.SubCom );
			Part := Part^.Next;
		end;
	end;
begin
	FirstWep := Nil;
	NextWep := Nil;

	if Weapon = Nil then FoundStart := True
	else FoundStart := False;

	CheckAlongPath( Master^.InvCom );
	CheckAlongPath( Master^.SubCom );

	{ Return either the next weapon or the first weapon, }
	{ depending upon what we found. }
	if NextWep = Nil then begin
		if FirstWep = Nil then FindNextWeapon := Weapon
		else FindNextWeapon := FirstWep;
	end else FindNextWeapon := NextWep;
end; { FindNextWeapon }

Function FindGearIndex( Master , FindThis: GearPtr ): LongInt;
	{ Search through master looking for FINDTHIS. }
	{ Once found, return its index number. Return -1 if it }
	{ cannot be found. }
var
	N,it: LongInt;
{ PROCEDURES BLOCK }
	Procedure CheckAlongPath( Part: GearPtr );
		{ CHeck along the path specified. }
	begin
		while ( Part <> Nil ) and ( it = -1 ) do begin
			Inc(N);
			if ( Part = FindThis ) then it := N;
			CheckAlongPath( Part^.InvCom );
			CheckAlongPath( Part^.SubCom );
			Part := Part^.Next;
		end;
	end;
begin
	N := 0;
	it := -1;
	if Master = FindThis then it := 0;
	CheckAlongPath( Master^.InvCom );
	CheckAlongPath( Master^.SubCom );
	FindGearIndex := it;
end; { FindGearIndex }

Procedure AlphaKeyMenu( RPM: RPGMenuPtr );
	{ Alter this menu so that each item in it has a letter key }
	{ hotlinked. }
	{ This procedure has nothing to do with gears, but it's easier }
	{ to stick it here than keep two copies in the conmenus and }
	{ sdlmenus units. What I really need is a separate menu-utility }
	{ unit, I guess. }
const
	MENUKEY_TABLE_MAX = 20;
	MenuKey_Table: Array [1..MENUKEY_TABLE_MAX] of Integer = (
			KMC_Up,
			KMC_Down,
			KMC_UpRight,
			KMC_DownRight,
			KMC_UpLeft,
			KMC_DownLeft,
			KMC_Left,
			KMC_Right,
			KMC_MenuUp,
			KMC_MenuDown,
			KMC_MenuLeft,
			KMC_MenuRight,
			KMC_PageUp,
			KMC_PageDown,
			KMC_ScrollUp,
			KMC_ScrollDown,
			KMC_ButtonWUp,
			KMC_ButtonWDown,
			KMC_ButtonWLeft,
			KMC_ButtonWRight
		);
var
	Key: Char;
	MI: RPGMenuItemPtr;
	i: Integer;
label
	NextKey;
begin
	{ The hotkeys start with 'a'. }
	Key := 'a';

	MI := RPM^.firstitem;
	while MI <> Nil do begin
		if ( 1 <= Length(MI^.key) ) and not ( ' ' = MI^.key[1] ) then begin
			{ A some key has been already assigned. }
		end else if ( 1 <= Length(MI^.msg) ) and ( #$0 = MI^.msg[1] ) then begin
			MI^.key := '   ';
		end else begin
			{ Alter the message. }
			MI^.key := Key + ') ';

			{ Add the key. }
			if ( '*' <> Key ) then begin
				AddRPGMenuKey( RPM , Key , MI^.value );
			end;

			{ Move to the next letter in the series. }
			{ note that only 52 letters can be assigned. }
			while ('*' <> Key) do begin
			NextKey:
				if ( 'z' = Key ) then begin Key := 'A'
				end else if ( 'Z' = Key ) or ( '*' = Key ) then begin Key := '*'
				end else begin inc( Key );
				end;
				if ( '*' <> Key ) then begin
					for i := 1 to MENUKEY_TABLE_MAX do begin
						if ( KeyMap[ MenuKey_Table[i] ].KCode = Key ) then begin
							goto NextKey;
						end;
					end;
				end;
				break;
			end;
		end;
		MI := MI^.Next;
	end;
end;

Procedure NumKeyMenu( RPM: RPGMenuPtr );
const
	MENUKEY_TABLE_MAX = 12;
	MenuKey_Table: Array [1..MENUKEY_TABLE_MAX] of Integer = (
			KMC_Up,
			KMC_Down,
			KMC_UpRight,
			KMC_DownRight,
			KMC_UpLeft,
			KMC_DownLeft,
			KMC_Left,
			KMC_Right,
			KMC_ButtonWUp,
			KMC_ButtonWDown,
			KMC_ButtonWLeft,
			KMC_ButtonWRight
		);
var
	i: LongInt;
	MI: RPGMenuItemPtr;
	Key: Char;
	j: Integer;
	Brk: Boolean;
begin
	if not ( ENABLE_NumKeyMenu ) then exit;

	MI := RPM^.firstitem;
	while ( NIL <> MI ) do begin
		MI^.key := '   ';
		MI := MI^.Next;
	end;
	for i := 0 to 9 do begin
		MI := RPMLocateByValue( RPM , i );
		if ( NIL <> MI ) then begin
			Key := BStr(i)[1];
			Brk := False;
			for j := 1 to MENUKEY_TABLE_MAX do begin
				if ( KeyMap[ MenuKey_Table[j] ].KCode = Key ) then begin
					Brk := True;
				end;
			end;
			if not ( Brk ) then begin
				MI^.key := Key + ') ';
				AddRPGMenuKey( RPM , Key , MI^.value );
			end;
		end;
	end;
end;

Procedure NumKeyMenu_minus( RPM: RPGMenuPtr );
const
	MENUKEY_TABLE_MAX = 12;
	MenuKey_Table: Array [1..MENUKEY_TABLE_MAX] of Integer = (
			KMC_Up,
			KMC_Down,
			KMC_UpRight,
			KMC_DownRight,
			KMC_UpLeft,
			KMC_DownLeft,
			KMC_Left,
			KMC_Right,
			KMC_ButtonWUp,
			KMC_ButtonWDown,
			KMC_ButtonWLeft,
			KMC_ButtonWRight
		);
var
	i: LongInt;
	MI: RPGMenuItemPtr;
	Key: Char;
	j: Integer;
	Brk: Boolean;
begin
	if not ( ENABLE_NumKeyMenu ) then exit;

	MI := RPM^.firstitem;
	while ( NIL <> MI ) do begin
		MI^.key := '   ';
		MI := MI^.Next;
	end;
	for i := 0 to 9 do begin
		MI := RPMLocateByValue( RPM , - i );
		if ( NIL <> MI ) then begin
			Key := BStr(i)[1];
			Brk := False;
			for j := 1 to MENUKEY_TABLE_MAX do begin
				if ( KeyMap[ MenuKey_Table[j] ].KCode = Key ) then begin
					Brk := True;
				end;
			end;
			if not ( Brk ) then begin
				MI^.key := Key + ') ';
				AddRPGMenuKey( RPM , Key , MI^.value );
			end;
		end;
	end;
end;


var
{$IFDEF ASCII}
	Z_SwapMenu: vgfx_zoneptr;
{$ELSE ASCII}
	Z_SwapMenu: DynamicRectPtr;
{$ENDIF ASCII}

Procedure SwapMenuRedraw;
begin
	InfoBox( Z_SwapMenu^ );
end;

{$IFDEF ASCII}
Function SwapMenu( Z: vgfx_zoneptr; var FirstPart: GearPtr; A: GearPtr ):Boolean;
{$ELSE ASCII}
Function SwapMenu( Z: DynamicRectPtr; var FirstPart: GearPtr; A: GearPtr ):Boolean;
{$ENDIF ASCII}
var
	done: Boolean;
	RPM: RPGMenuPtr;
	B: GearPtr;
	SI,TI: Integer;
	N: LongInt;
begin
	if ( NIL = FirstPart ) then exit( False );
	if NumSiblingGears( FirstPart ) <= 1 then exit( False );

	Z_SwapMenu := Z;

	done := False;
	RPM := CreateRPGMenu( MenuItem , MenuSelect , Z );

	SI := 1;
	TI := 1;
	repeat
		BuildSiblingMenu_with_Mark( RPM, FirstPart, A );
		AlphaKeyMenu( RPM );
		if done then begin
			AddRPGMenuItem( RPM , UTF8_MsgString('SwapMenu','Exit') , SELECTMENU_Cancel )
		end else begin
			AddRPGMenuItem( RPM , UTF8_MsgString('SwapMenu','Cancel') , SELECTMENU_Cancel );
		end;
		RPM^.SelectItem := SI;
		RPM^.TopItem    := TI;
		N := SelectMenu( RPM , @SwapMenuRedraw );
		SI := RPM^.SelectItem;
		TI := RPM^.TopItem;

		if ( SELECTMENU_Enable <= N ) then begin
			B := RetrieveGearSib( FirstPart, N );
			if ( NIL = A ) then begin
				A := B;
			end else if ( A = B ) then begin
				A := NIL;
			end else begin
				SwapGears( FirstPart, A, B );
				done := True;
				A := NIL;
			end;

			ClearMenu( RPM );
		end;
	until ( SELECTMENU_Cancel = N );

	DisposeRPGMenu( RPM );
	SwapMenu := done;
end;

{$IFDEF ASCII}
Function SwapMenu( Z: vgfx_zoneptr; Part: GearPtr ):Boolean;
{$ELSE ASCII}
Function SwapMenu( Z: DynamicRectPtr; Part: GearPtr ):Boolean;
{$ENDIF ASCII}
var
	done: Boolean;
begin
	if ( NIL = Part ) or ( NIL = Part^.Parent ) then exit( False );

	if IsInvCom( Part ) then begin
		done := SwapMenu( Z , Part^.Parent^.InvCom , Part );
	end else begin
		done := SwapMenu( Z , Part^.Parent^.SubCom , Part );
	end;
	SwapMenu := done;
end;



initialization
begin
{$IFDEF DEBUG}
	ErrorMessage_fork('DEBUG: menugear.pp');
{$ENDIF DEBUG}
end;

finalization
begin
{$IFDEF DEBUG}
	ErrorMessage_fork('DEBUG: menugear.pp(finalization)');
{$ENDIF DEBUG}
end;

end.
