unit sdlmenus;
{$MODE FPC}

	{ BUGS - If SelectMenu is handed an empty menu, all heck will }
	{  break loose. This can be a particular problem for SelectFile. }
{
	GearHead2, a roguelike mecha CRPG
	Copyright (C) 2005 Joseph Hewitt

	This library is free software; you can redistribute it and/or modify it
	under the terms of the GNU Lesser General Public License as published by
	the Free Software Foundation; either version 2.1 of the License, or (at
	your option) any later version.

	The full text of the LGPL can be found in license.txt.

	This library is distributed in the hope that it will be useful, but
	WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
	General Public License for more details. 

	You should have received a copy of the GNU Lesser General Public License
	along with this library; if not, write to the Free Software Foundation,
	Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
}
{$LONGSTRINGS ON}

interface

{$IFDEF PLUSGL}
uses SDL,SDL_TTF,dos,glgfx,texutil,ui4gh;
{$ELSE}
uses SDL,SDL_TTF,dos,sdlgfx,texutil,ui4gh;
{$ENDIF}

const
	{These two constants are used to tell the SELECT procedure whether or not}
	{the user is allowed to cancel.}
	RPMNormal = 0;
	RPMNoCancel = 1;
	RPMEscCancel = 2;	{ Menu can be cancelled, but not by mouse click. }

	RPMNoCleanup = 2; {If you want the menu left on the screen after we've finished, use this.}

	SELECTMENU_Enable = 0;
	SELECTMENU_Cancel = -1;
	SELECTMENU_MenuOrder_Edit = -128;
	SELECTMENU_MenuOrder_SortAlpha = -127;
	SELECTMENU_MenuOrder_SortValue = -126;
	IsMenuActive: Boolean = False;

type
	RPGMenuKeyPtr = ^RPGMenuKey;
	RPGMenuKey = Record
		k: Char;
		value: integer;		{The value returned when this key is pressed.}
		next: RPGMenuKeyPtr;
	end;

	RPGMenuItemPtr = ^RPGMenuItem;
	RPGMenuItem = Record
		key: string;		{The text which appears in the menu}
		msg: string;		{The text which appears in the menu}
		value: integer;		{A value, returned by SelectMenu. -1 is reserved for Cancel}
		desc: string;		{Pointer to the item description. If Nil, no desc.}
		next: RPGMenuItemPtr;
	end;
	RPGMenu = Record
		active: boolean;
		itemcolor,selcolor,dtexcolor: TSDL_Color;
		Menu_Zone,Desc_Zone: DynamicRectPtr;
		mode: Byte;
		editable_menuorder: Boolean;
		topitem,selectitem,numitem: integer; {fields holding info about the status of the menu.}
		FirstItem: RPGMenuItemPtr;
		FirstKey: RPGMenuKeyPtr;
	end;
	RPGMenuPtr = ^RPGMenu;

const
	Temporary_SelectItem: RPGMenuItemPtr = NIL;


Function UpdateRPGMenuItem( RPM: RPGMenuPtr; var it: RPGMenuItemPtr; const msg: string ): RPGMenuItemPtr;
Function UpdateRPGMenuItem( RPM: RPGMenuPtr; const i: Integer; const msg: string ): RPGMenuItemPtr;
Function AddRPGMenuItem(var RPM: RPGMenuPtr; const msg: string; value: integer; const desc: string): RPGMenuItemPtr;
Function AddRPGMenuItem(var RPM: RPGMenuPtr; const msg: string; value: integer): RPGMenuItemPtr;
Function AddRPGMenuItem_Top(var RPM: RPGMenuPtr; const msg: string; value: integer; const desc: string): RPGMenuItemPtr;
Function AddRPGMenuItem_Top(var RPM: RPGMenuPtr; const msg: string; value: integer): RPGMenuItemPtr;
Procedure DisposeRPGMenuItem( var LList: RPGMenuItemPtr );
Procedure ClearMenu( RPM: RPGMenuPtr );
Procedure RemoveRPGMenuItem(RPM: RPGMenuPtr; var LMember: RPGMenuItemPtr);

Procedure AddRPGMenuKey(RPM: RPGMenuPtr; k: Char; value: Integer);
Function CreateRPGMenu(icolor,scolor: TSDL_Color; Z: DynamicRectPtr): RPGMenuPtr;
Procedure AttachMenuDesc( RPM: RPGMenuPtr; Z: DynamicRectPtr );

Procedure DisposeRPGMenu(var RPM: RPGMenuPtr);
Procedure DisplayMenu( RPM: RPGMenuPtr; ReDrawer: RedrawProcedureType );
Function RPMLocateByPosition(RPM: RPGMenuPtr; i: integer): RPGMenuItemPtr;
Function RPMLocateByValue(RPM: RPGMenuPtr; i: integer): RPGMenuItemPtr;
Function SelectMenu( RPM: RPGMenuPtr; ReDrawer: RedrawProcedureType ): integer;
Procedure RPMSortAlpha(RPM: RPGMenuPtr);
Procedure RPMSortValue(RPM: RPGMenuPtr);

Function CurrentMenuItemValue( RPM: RPGMenuPtr ): integer;
Function SetItemByValue( RPM: RPGMenuPtr ; V: Integer ): RPGMenuItemPtr;
Procedure SetItemByPosition( RPM: RPGMenuPtr ; N: Integer );

Function BuildFileMenu( RPM: RPGMenuPtr; const SearchPattern: String; EncodedFileName: Boolean; N: Integer ): Integer;
Function BuildFileMenu( RPM: RPGMenuPtr; const SearchPattern: String; EncodedFileName: Boolean ): Integer;
Function SelectFile( RPM: RPGMenuPtr; ReDrawer: RedrawProcedureType ): String;


implementation

uses errmsg,utf8msg;

var
	SearchPattern: String = '';


Function LastMenuItem(MIList: RPGMenuItemPtr): RPGMenuItemPtr;
	{This procedure will find the last item in the linked list.}
begin
	{While the menu item is pointing to a next menu item, it's not the last. duh.}
	{So, move through the list until we hit a Nil pointer.}
	while MIList^.next <> Nil do begin
		{Check the next one.}
		MIList := MIList^.next;
	end;
	{We've found a MI.next = Nil. Yay! Return its address.}
	LastMenuItem := MIList;
end;


Function UpdateRPGMenuItem( RPM: RPGMenuPtr; var it: RPGMenuItemPtr; const msg: string ): RPGMenuItemPtr;
begin
	it^.msg := msg;
	UpdateRPGMenuItem := it;
end;

Function UpdateRPGMenuItem( RPM: RPGMenuPtr; const i: Integer; const msg: string ): RPGMenuItemPtr;
var
	it: RPGMenuItemPtr;
begin
	it := RPMLocateByPosition( RPM , i );
	UpdateRPGMenuItem := UpdateRPGMenuItem( RPM , it , msg );
end;

Function AddRPGMenuItem(var RPM: RPGMenuPtr; const msg: string; value: integer; const desc: string): RPGMenuItemPtr;
	{This procedure will add an item to the RPGToolMenu.}
	{The new item will be added as the last item in the list.}
var
	it: ^RPGMenuItem;		{Here's a pointer for the item we're creating.}
	temp: RPGMenuItemPtr;
begin
	{Allocate memory for it.}
	New(it);

	{Check to make sure that the allocation succeeded.}
	if it = Nil then begin
		exit( Nil );
	end;

	{Initialize it to the correct values.}
	UpdateRPGMenuItem( RPM , it , msg );
	it^.key := '';
	it^.value := value;
	it^.next := Nil;
	it^.desc := desc; 	{The desc field is assigned the value of PChar since it}
				{is assumed that we arent responsible for the allocation,}
				{disposal, or contents of this string.}

	{Locate the last item in the list, then assign "it" to it.}
	{If the list is currently empty, stick "it" in as the first item.}
	if RPM^.firstitem = Nil then begin
		RPM^.firstitem := it;
	end else begin
		temp := LastMenuItem(RPM^.FirstItem);
		temp^.next := it;		
	end;

	{Increment the NumItem field.}
	Inc(RPM^.numitem);

	AddRPGMenuItem := it;
end;

Function AddRPGMenuItem(var RPM: RPGMenuPtr; const msg: string; value: integer): RPGMenuItemPtr;
	{ Just like the above, but no desc. }
begin
	AddRPGMenuItem := AddRPGMenuItem( RPM , msg , value , '' );
end;

Function AddRPGMenuItem_Top(var RPM: RPGMenuPtr; const msg: string; value: integer; const desc: string): RPGMenuItemPtr;
	{This procedure will add an item to the RPGToolMenu.}
	{The new item will be added as the top item in the list.}
var
	it: ^RPGMenuItem;		{Here's a pointer for the item we're creating.}
	temp: RPGMenuItemPtr;
begin
	{Allocate memory for it.}
	New(it);

	{Check to make sure that the allocation succeeded.}
	if it = Nil then begin
		exit( Nil );
	end;

	{Initialize it to the correct values.}
	UpdateRPGMenuItem( RPM , it , msg );
	it^.key := '';
	it^.value := value;
	it^.next := Nil;
	it^.desc := desc; 	{The desc field is assigned the value of PChar since it}
				{is assumed that we arent responsible for the allocation,}
				{disposal, or contents of this string.}

	it^.next := RPM^.firstitem;
	RPM^.firstitem := it;

	{Increment the NumItem field.}
	Inc(RPM^.numitem);

	AddRPGMenuItem_Top := it;
end;

Function AddRPGMenuItem_Top(var RPM: RPGMenuPtr; const msg: string; value: integer): RPGMenuItemPtr;
	{ Just like the above, but no desc. }
begin
	AddRPGMenuItem_Top := AddRPGMenuItem_Top( RPM , msg , value , '' );
end;

Procedure DisposeRPGMenuItem( var LList: RPGMenuItemPtr );
	{ Get rid of this list of items. }
	{ WARNING - If you call this procedure for a menu, it will not }
	{ change the value of the NumItems field!!! This will cause }
	{ problems when trying to use the menu. Unless you know exactly }
	{ what you're doing, use the ClearMenu procedure instead. }
var
	NextItem: RPGMenuItemPtr;
begin
	while LList <> Nil do begin
		NextItem := LList^.Next;
{$IFDEF PATCH_GH_PARANOID_SAFER}
		LList^.key     := '@';
		LList^.msg     := '@';
		LList^.value   := -32767;
		LList^.desc    := '@';
		LList^.Next    := RPGMenuItemPtr(-1);
{$ENDIF PATCH_GH_PARANOID_SAFER}
{$IFNDEF PATCH_GH_PARANOID_CHECKER}
		Dispose( LList );
{$ENDIF PATCH_GH_PARANOID_CHECKER}
		LList := NextItem;
	end;
end;

Procedure ClearMenu( RPM: RPGMenuPtr );
	{ Deallocate all the items in this menu, and set the number of }
	{ items to 0. }
begin
	DisposeRPGMenuItem( RPM^.FirstItem );
	RPM^.NumItem := 0;
	RPM^.SelectItem := 1;
	RPM^.topitem := 1;
end;

Procedure RemoveRPGMenuItem(RPM: RPGMenuPtr; var LMember: RPGMenuItemPtr);
	{Locate and extract member LMember from list LList.}
	{Then, dispose of LMember.}
var
	a,b: RPGMenuItemPtr;
begin
	{Initialize A and B}
	B := RPM^.FirstItem;
	A := Nil;

	{Locate LMember in the list. A will thereafter be either Nil,}
	{if LMember if first in the list, or it will be equal to the}
	{element directly preceding LMember.}
	while (B <> LMember) and (B <> Nil) do begin
		A := B;
		B := B^.next;
	end;

	if B = Nil then begin
		{Major FUBAR. The member we were trying to remove can't}
		{be found in the list.}
		ErrorMessage('ERROR- RemoveLink asked to remove a link that doesnt exist.');
		end
	else if A = Nil then begin
		{There's no element before the one we want to remove,}
		{i.e. it's the first one in the list.}
		RPM^.FirstItem := B^.Next;
{$IFDEF PATCH_GH_PARANOID_SAFER}
		B^.key     := '@';
		B^.msg     := '@';
		B^.value   := -32767;
		B^.desc    := '@';
		B^.Next    := RPGMenuItemPtr(-1);
{$ENDIF PATCH_GH_PARANOID_SAFER}
{$IFNDEF PATCH_GH_PARANOID_CHECKER}
		Dispose(B);
{$ENDIF PATCH_GH_PARANOID_CHECKER}
		end
	else begin
		{We found the attribute we want to delete and have another}
		{one standing before it in line. Go to work.}
		A^.next := B^.next;
{$IFDEF PATCH_GH_PARANOID_SAFER}
		B^.key     := '@';
		B^.msg     := '@';
		B^.value   := -32767;
		B^.desc    := '@';
		B^.Next    := RPGMenuItemPtr(-1);
{$ENDIF PATCH_GH_PARANOID_SAFER}
{$IFNDEF PATCH_GH_PARANOID_CHECKER}
		Dispose(B);
{$ENDIF PATCH_GH_PARANOID_CHECKER}
	end;

	{ Reduce the number of items in the menu. }
	Dec(RPM^.NumItem);
end;

Procedure AddRPGMenuKey(RPM: RPGMenuPtr; k: Char; value: Integer);
	{Add a dynamically defined RPGMenuKey to the menu.}
var
	it: RPGMenuKeyPtr;
begin
	New(it);
	if it = Nil then begin
		exit;
	end;

	{Initialize the values.}
	it^.k := k;
	it^.value := value;
	it^.next := RPM^.FirstKey;
	RPM^.FirstKey := it;
end;

Function CreateRPGMenu(icolor,scolor: TSDL_Color; Z: DynamicRectPtr): RPGMenuPtr;
	{This function creates a new RPGMenu record, and returns the address.}
var
	it: ^RPGMenu;			{Here's a pointer for the menu we're making.}
begin
	{Allocate memory for it.}
	New(it);

	{Check to make sure that we've actually initialized something.}
	if it = Nil then exit( Nil );

	{Initialize the elements of the record.}
	it^.itemcolor := icolor;
	it^.selcolor := scolor;
	it^.Menu_Zone := Z;
	it^.Desc_Zone := NIL; {A width value of 0 means there is no desc window.}
	it^.Mode := RPMNormal;
	it^.editable_menuorder := False;
	it^.FirstItem := Nil;
	it^.FirstKey := Nil;
	it^.dtexcolor := icolor;
	it^.active := False;

	{TopItem refers to the highest item on the screen display.}
	it^.topitem := 1;

	{SelectItem refers to the item currently being pointed at by the selector.}
	it^.selectitem := 1;

	{NumItem refers to the total number of items currently in the linked list.}
	it^.numitem := 0;

	{Return the address.}
	CreateRPGMenu := it;
end;

Procedure AttachMenuDesc( RPM: RPGMenuPtr; Z: DynamicRectPtr );
	{ Set the area for description items to zone Z. }
begin
	RPM^.Desc_Zone := Z;
end;

Procedure DisposeRPGMenu(var RPM: RPGMenuPtr);
	{This procedure is called when you want to get rid of the menu. It will deallocate}
	{the memory for the RPGMenu record and also for all of the linked RPGMenuItems.}
var
	c,d: RPGMenuKeyPtr;
begin
	{Check to make sure that we've got a valid pointer here.}
	if RPM = Nil then begin
		exit;
	end;

	{Save the location of the first menu item...}
	DisposeRPGMenuItem( RPM^.FirstItem );
	c := RPM^.FirstKey;
	{... then get rid of the menu record.}
{$IFDEF PATCH_GH_PARANOID_SAFER}
	RPM^.ItemColor.r := 126;
	RPM^.ItemColor.g := 126;
	RPM^.ItemColor.b := 126;
	RPM^.SelColor.r  := 126;
	RPM^.SelColor.g  := 126;
	RPM^.SelColor.b  := 126;
	RPM^.DtexColor.r := 126;
	RPM^.DtexColor.g := 126;
	RPM^.DtexColor.b := 126;
	RPM^.Menu_Zone := DynamicRectPtr(-1);
	RPM^.Desc_Zone := DynamicRectPtr(-1);
	RPM^.mode        := 126;
	RPM^.editable_menuorder := False;
	RPM^.TopItem     := -32767;
	RPM^.SelectItem  := -32767;
	RPM^.NumItem     := -32767;
	RPM^.FirstItem   := RPGMenuItemPtr(-1);
	RPM^.FirstKey    := RPGMenuKeyPtr(-1);
{$ENDIF PATCH_GH_PARANOID_SAFER}
{$IFNDEF PATCH_GH_PARANOID_CHECKER}
	Dispose(RPM);
{$ENDIF PATCH_GH_PARANOID_CHECKER}
	RPM := Nil;

	{Keep processing the menu items until we hit a Nil nextitem.}
	while c <> Nil do begin
		d := c^.next;
{$IFDEF PATCH_GH_PARANOID_SAFER}
		c^.k     := #$FF;
		c^.value := -32767;
		c^.Next  := RPGMenuKeyPtr(-1);
{$ENDIF PATCH_GH_PARANOID_SAFER}
{$IFNDEF PATCH_GH_PARANOID_CHECKER}
		Dispose(c);
{$ENDIF PATCH_GH_PARANOID_CHECKER}
		c := d;
	end;
end;

Function RPMLocateByPosition(RPM: RPGMenuPtr; i: integer): RPGMenuItemPtr;
	{Locate the i'th element of the item list, then return its address.}
var
	a: RPGMenuItemPtr;	{Our pointer}
	t: integer;		{Our counter}
begin
	{Error check, first off.}
	if i > RPM^.numitem then begin
		exit( Nil );
	end;

	a := RPM^.FirstItem;
	t := 1;

	if i > 1 then begin
		for t := 2 to i do
			a := a^.next;
	end;

	RPMLocateByPosition := a;
end;

Function RPMLocateByValue(RPM: RPGMenuPtr; i: integer): RPGMenuItemPtr;
	{Locate the i'th element of the item list, then return its address.}
var
	t,a: RPGMenuItemPtr;	{Our counter and a pointer}
begin
	a := Nil;
	t := RPM^.FirstItem;

	while ( a = Nil ) and ( t <> Nil ) do begin
		if t^.value = i then a := t;
		t := t^.Next;
	end;

	RPMLocateByValue := a;
end;

Function MenuHeight( RPM: RPGMenuPtr ): Integer;
	{ Return the height of the menu, in text rows. }
var
	MH: Integer;
begin
	MH := ( RPM^.Menu_Zone^.h div TTF_FontLineSkip( game_font ) );
	if MH < 1 then MH := 1;
	MenuHeight := MH;
end;

Procedure RPMRefreshDesc(RPM: RPGMenuPtr);
	{Refresh the menu description box, if appropriate.}
begin
	{Check to make sure that this menu has a description box, first off.}
	if ( NIL <> RPM^.Desc_Zone ) then begin
		CMessage( RPMLocateByPosition(RPM,RPM^.selectitem)^.desc , RPM^.Desc_Zone^.GetRect() , RPM^.dtexcolor );
	end;
end;

Procedure DisplayMenu( RPM: RPGMenuPtr; ReDrawer: RedrawProcedureType );
	{Display the menu on the screen.}
var
	topitem: RPGMenuItemPtr;
	a: RPGMenuItemPtr;		{A pointer to be used while printing.}
	t: integer;
	height: integer;		{The width of the menu display.}
	NextColor: TSDL_Color;
	MyRect,MyDest: TSDL_Rect;
	Y,DY: Integer;
	MyDestBase: TSDL_Rect;
begin
	{Error check- make sure the menu has items in it.}
	if RPM^.FirstItem = Nil then Exit;

	{ If a redraw procedure has been specified, call it. }
	Temporary_SelectItem := RPMLocateByPosition(RPM,RPM^.selectitem);
	if ReDrawer <> Nil then ReDrawer;
	Temporary_SelectItem := NIL;

	MyRect := RPM^.Menu_Zone^.GetRect();
	SDL_SetClipRect( Game_Screen , @MyRect );

	{Calculate the height of the menu.}
	height := MenuHeight( rpm );

	{Locate the top of the menu.}
	topitem := RPMLocateByPosition(RPM,RPM^.topitem);

	MyDest.X := MyRect.X;
	Y := MyRect.Y;
	DY := TTF_FontLineSkip( game_font );
	MyDest.W := RPM^.Menu_Zone^.W;
	MyDestBase := MyDest;

	a := topitem;
	for t := 1 to Height do begin
		MyDest.Y := Y;
		Y := Y + DY;

		{If we're at the currently selected item, highlight it.}
		if ((t + RPM^.topitem - 1) = RPM^.selectitem) and RPM^.Active then
			NextColor := RPM^.selcolor
		else
			NextColor := RPM^.itemcolor;

		if (1 < Length(a^.msg)) and (#$0 = a^.msg[1]) then begin
			QuickText( a^.key + Copy( a^.msg , 2 , Length( a^.msg ) - 1 ) , MyDest , NextColor , game_font );
		end else begin
			QuickText( a^.key + a^.msg , MyDest , NextColor , game_font );
		end;

		a := a^.next;

		{Check to see if we've prematurely encountered the end of the list.}
		if a = Nil then break;
	end;

	if SDL_Show_MenuScrollbar then begin
		MyDestBase := RPM^.Menu_Zone^.GetRect();
		MyDestBase.Y := MyDestBase.Y + SDL_Scrollbar_Size;
		MyDestBase.H := MyDestBase.H - SDL_Scrollbar_Size - SDL_Scrollbar_Size;

		{ Button: Up }
		MyDest.W := SDL_Scrollbar_Size - 2;
		MyDest.X := MyDestBase.X + MyDestBase.W - SDL_Scrollbar_Size + 1;
		MyDest.H := SDL_Scrollbar_Size - 2;
		MyDest.Y := RPM^.Menu_Zone^.GetRect().Y;
		if (1 < RPM^.topitem) then begin
			SDL_FillRect( Game_Screen, @MyDest, SDL_MapRGB( Game_Screen^.Format , MenuSelect.R , MenuSelect.G , MenuSelect.B ) );
		end else begin
			SDL_FillRect( Game_Screen, @MyDest, SDL_MapRGB( Game_Screen^.Format , MenuItem.R , MenuItem.G , MenuItem.B ) );
		end;

		{ Button: Down }
		MyDest.H := SDL_Scrollbar_Size - 2;
		MyDest.Y := RPM^.Menu_Zone^.GetRect().Y + RPM^.Menu_Zone^.H - SDL_Scrollbar_Size + 2;
		if (Height <= (RPM^.numitem - RPM^.topitem)) then begin
			SDL_FillRect( Game_Screen, @MyDest, SDL_MapRGB( Game_Screen^.Format , MenuSelect.R , MenuSelect.G , MenuSelect.B ) );
		end else begin
			SDL_FillRect( Game_Screen, @MyDest, SDL_MapRGB( Game_Screen^.Format , MenuItem.R , MenuItem.G , MenuItem.B ) );
		end;

		{ Bar: Silhouette }
		MyDest.W := SDL_Scrollbar_Size;
		MyDest.X := MyDestBase.X + MyDestBase.W - MyDest.W;
		MyDest.H := MyDestBase.H;
		MyDest.Y := MyDestBase.Y;
		SDL_FillRect( Game_Screen, @MyDest, SDL_MapRGB( Game_Screen^.Format , MenuItem.R , MenuItem.G , MenuItem.B ) );

		{ Bar: Light }
		MyDest.Y := MyDestBase.Y + (((RPM^.topitem - 1) * MyDestBase.H) div RPM^.numitem);
		MyDest.H := ((Height * MyDestBase.H) div RPM^.numitem);
		if (MyDest.H < 1) then begin
			MyDest.H := 1;
		end;
		if (MyDestBase.H < MyDest.H) then begin
			MyDest.H := MyDestBase.H;
		end;
		SDL_FillRect( Game_Screen, @MyDest, SDL_MapRGB( Game_Screen^.Format , MenuSelect.R , MenuSelect.G , MenuSelect.B ) );

		{ Bar: Highlight }
		MyDest.X := MyDestBase.X + MyDestBase.W - (SDL_Scrollbar_Size div 2);
		MyDest.W := 1;
		MyDest.Y := MyDestBase.Y + (((RPM^.selectitem - 1) * MyDestBase.H) div RPM^.numitem);
		MyDest.H := (MyDestBase.H div RPM^.numitem);
		if (MyDest.H < 1) then begin
			MyDest.H := 1;
		end;
		SDL_FillRect( Game_Screen, @MyDest, SDL_MapRGB( Game_Screen^.Format , SDLBorderBlue.R , SDLBorderBlue.G , SDLBorderBlue.B ) );
	end;
	{Restore the window to its regular size.}
	SDL_SetClipRect( Game_Screen , Nil );

	{If there's an associated Desc field, display it now.}
	RPMRefreshDesc(RPM);
end;

Procedure RPMReposition( RPM: RPGMenuPtr; FullScroll: Boolean );
	{The selected item has just changed, and is no longer visible on screen.}
	{Adjust the RPGMenu's topitem field to an appropriate value.}
begin
	{When this function is called, there are two possibilities: either the}
	{selector has moved off the bottom of the page or the top.}

	if RPM^.selectitem < RPM^.topitem then begin
		{The selector has moved off the bottom of the list. The new page}
		{display should start with SelectItem on the bottom.}
		if FullScroll then begin
			RPM^.topitem := RPM^.selectitem - MenuHeight( RPM ) + 1;
		end else begin
			RPM^.topitem := RPM^.selectitem;
		end;

		{Error check- if this moves topitem below 1, that's bad.}
		if RPM^.topitem < 1 then begin
			RPM^.topitem := 1;
		end;
	end else begin
		{The selector has moved off the top of the list. The new page should}
		{start with SelectItem at the top, unless this would make things look}
		{funny.}
		if FullScroll then begin
			if ((RPM^.selectitem + MenuHeight( RPM ) - 1) > RPM^.numitem) then begin
				{There will be whitespace at the bottom of the menu if we assign}
				{SelectItem to TopItem. Make TopItem equal to the effective last}
				{page.}
				RPM^.topitem := RPM^.numitem - MenuHeight( RPM ) + 1;
				if RPM^.topitem < 1 then begin
					RPM^.topitem := 1;
				end;
			end else begin
				RPM^.topitem := RPM^.selectitem;
			end;
		end else if ((RPM^.topitem + MenuHeight( RPM ) - 1) < RPM^.numitem) then begin
			Inc( RPM^.TopItem );
		end;
	end;

end;

Procedure RPMReposition( RPM: RPGMenuPtr );
begin
	{ Check a limit. }
	if (RPM^.selectitem < 1) then begin RPM^.selectitem := 1
	end else if (RPM^.numitem < RPM^.selectitem) then begin RPM^.selectitem := RPM^.numitem
	end;

	if (RPM^.selectitem < RPM^.topitem) then begin
		{ Page Up }
		RPM^.topitem := RPM^.selectitem;

		{ Check a limit. }
		if RPM^.topitem < 1 then begin
			RPM^.topitem := 1;
		end;
	end else if (RPM^.topitem + MenuHeight(RPM) <= RPM^.selectitem) then begin
		{ Page Down }
		RPM^.topitem := RPM^.selectitem - MenuHeight(RPM) + 1;

		{ Check a limit. }
		if RPM^.numitem < RPM^.topitem then begin
			RPM^.topitem := RPM^.numitem;
		end;
	end;

end;

Procedure RPMReposition_center( RPM: RPGMenuPtr );
begin
	{ Check a limit. }
	if (RPM^.selectitem < 1) then begin
		RPM^.selectitem := 1
	end else if (RPM^.numitem < RPM^.selectitem) then begin
		RPM^.selectitem := RPM^.numitem
	end;

	{ Auto reposition mode. }
	if (RPM^.topitem < 1) then begin
		RPM^.topitem := RPM^.selectitem - ( MenuHeight(RPM) div 2 );
		if (RPM^.topitem < 1) then begin
			RPM^.topitem := 1;
		end;
	end else if (RPM^.numitem < RPM^.topitem) then begin
		RPM^.topitem := RPM^.selectitem - MenuHeight(RPM) + 1;
		if (RPM^.topitem < 1) then begin
			RPM^.topitem := 1;
		end;
	end else if (RPM^.selectitem < RPM^.topitem) then begin
		RPM^.topitem := RPM^.selectitem - MenuHeight(RPM) + 1;
		if RPM^.topitem < 1 then begin
			RPM^.topitem := 1;
		end;
	end else if (RPM^.topitem + MenuHeight(RPM) <= RPM^.selectitem) then begin
		RPM^.topitem := RPM^.selectitem;
		if RPM^.numitem < RPM^.topitem then begin
			RPM^.topitem := RPM^.numitem;
		end;
	end;

end;

Procedure RPMUpKey( RPM: RPGMenuPtr; FullScroll: Boolean );
	{Someone just pressed the UP key, and we're gonna process that input.}
	{PRECONDITIONS: RPM has been initialized properly, and is currently being}
	{  displayed on the screen.}
begin
	{Decrement the selected item by one.}
	Dec(RPM^.selectitem);

	{If this causes it to go beneath one, wrap around to the last item.}
	if RPM^.selectitem = 0 then
		RPM^.selectitem := RPM^.numitem;

	{If the movement takes the selected item off the screen, do a redisplay.}
	{Otherwise, indicate the newly selected item.}
	if (RPM^.selectitem < RPM^.topitem) or ((RPM^.selectitem - RPM^.topitem) >= MenuHeight( RPM )) then begin
		{Determine an appropriate new value for topitem.}
		RPMReposition(RPM,FullScroll);
	end;
end;

Procedure RPMDownKey( RPM: RPGMenuPtr; FullScroll: Boolean );
	{Someone just pressed the DOWN key, and we're gonna process that input.}
	{PRECONDITIONS: RPM has been initialized properly, and is currently being}
	{  displayed on the screen.}
begin
	{Increment the selected item.}
	Inc(RPM^.selectitem);
	{If this takes the selection out of bounds, restart at the first item.}
	if RPM^.selectitem = RPM^.numitem + 1 then
		RPM^.selectitem := 1;

	{If the movement takes the selected item off the screen, do a redisplay.}
	{Otherwise, indicate the newly selected item.}
	if (RPM^.selectitem < RPM^.topitem) or ((RPM^.selectitem - RPM^.topitem) >= MenuHeight( RPM ) ) then begin
		{Determine an appropriate new value for topitem.}
		RPMReposition(RPM,FullScroll);
	end;
end;


Procedure RPMPgUpKey( RPM: RPGMenuPtr; FullScroll: Boolean );
var
	p: Integer;
begin
	p := RPM^.selectitem - RPM^.topitem;
	RPM^.selectitem := RPM^.selectitem - MenuHeight( RPM );
	RPM^.topitem := RPM^.selectitem - p;

	if RPM^.selectitem < 1 then RPM^.selectitem := 1;
	if RPM^.topitem < 1 then RPM^.topitem := 1;
end;

Procedure RPMPgDownKey( RPM: RPGMenuPtr; FullScroll: Boolean );
var
	p: Integer;
begin
	p := RPM^.selectitem - RPM^.topitem;
	RPM^.selectitem := RPM^.selectitem + MenuHeight( RPM );
	RPM^.topitem := RPM^.selectitem - p;
	
	if RPM^.selectitem > RPM^.numitem then  RPM^.selectitem := RPM^.numitem;
	if (RPM^.numitem > MenuHeight( RPM )) then begin
		if RPM^.topitem > (RPM^.numitem - MenuHeight( RPM ) + 1) then RPM^.topitem := RPM^.numitem - MenuHeight( RPM ) + 1;
	end else RPM^.topitem := 1;
end;
Procedure RPMScrollUpKey( RPM: RPGMenuPtr; FullScroll: Boolean );
begin
	Dec(RPM^.selectitem);
	if RPM^.selectitem < 1 then
		RPM^.selectitem := 1;
	Dec(RPM^.topitem);
	if RPM^.topitem < 1 then
		RPM^.topitem := 1;
end;

Procedure RPMScrollDownKey( RPM: RPGMenuPtr; FullScroll: Boolean );
begin
	Inc(RPM^.selectitem);
	if RPM^.numitem < RPM^.selectitem then
		RPM^.selectitem := RPM^.numitem;
	Inc(RPM^.topitem);
	if (RPM^.numitem > MenuHeight( RPM )) then begin
		if RPM^.topitem > (RPM^.numitem - MenuHeight( RPM ) + 1) then RPM^.topitem := RPM^.numitem - MenuHeight( RPM ) + 1;
	end else RPM^.topitem := 1;
end;

Procedure RPMSearchForwardKey( RPM: RPGMenuPtr );
var
	key: String;
	sel: Integer;
	RPM_Item: RPGMenuItemPtr;
begin
	key := GetStringFromUser( UTF8_MsgString( 'RPMSearchForwardKey' , 'SearchForward' ) , SearchPattern , NIL , NIL );

	sel := RPM^.SelectItem;
	RPM_Item := RPMLocateByPosition( RPM , sel );
	repeat
		Inc( sel );
		RPM_Item := RPM_Item^.Next;
		if ( RPM^.NumItem < sel ) then begin
			sel := 1;
			RPM_Item := RPM^.FirstItem;
		end;
		if 0 < Pos( UpCase( Key ) , UpCase( RPM_Item^.msg ) ) then begin
			RPM^.SelectItem := sel;
			SearchPattern := key;
			break;
		end;
	until ( sel = RPM^.SelectItem );

	RPMReposition_center( RPM );
end;

Procedure RPMSearchBackwardKey( RPM: RPGMenuPtr );
var
	key: String;
	sel: Integer;
	RPM_Item: RPGMenuItemPtr;
begin
	key := GetStringFromUser( UTF8_MsgString( 'RPMSearchBackwardKey' , 'SearchBackward' ) , SearchPattern , NIL , NIL );

	sel := RPM^.SelectItem;
	repeat
		Dec( sel );
		if (sel < 1) then begin
			sel := RPM^.NumItem;
		end;
		RPM_Item := RPMLocateByPosition( RPM, sel );
		if 0 < Pos( UpCase( Key ) , UpCase( RPM_Item^.msg ) ) then begin
			RPM^.SelectItem := sel;
			SearchPattern := key;
			break;
		end;
	until ( sel = RPM^.SelectItem );

	RPMReposition_center( RPM );
end;

Procedure RPMSearchParentKey( RPM: RPGMenuPtr );
var
	len_currentindent: Integer;
	sel: Integer;
	RPM_Item: RPGMenuItemPtr;
	key: String;
	len_indent: Integer;
begin
	sel := RPM^.SelectItem;
	RPM_Item := RPMLocateByPosition( RPM, sel );
	key := RPM_Item^.msg;
	len_currentindent := DeleteWhiteSpace( key );
	repeat
		Dec( sel );
		if (sel < 1) then begin
			sel := 1;
			break;
		end;
		RPM_Item := RPMLocateByPosition( RPM, sel );
		key := RPM_Item^.msg;
		len_indent := DeleteWhiteSpace( key );
		if len_indent < len_currentindent then begin
			RPM^.SelectItem := sel;
			break;
		end;
	until ( sel = RPM^.SelectItem );

	RPMReposition_center( RPM );
end;

Function SelectMenu( RPM: RPGMenuPtr; ReDrawer: RedrawProcedureType ): integer;
	{This function will allow the user to browse through the menu and will}
	{return a value based upon the user's selection.}
var
	getit: char;		{Character used to store user input}
	r,I: integer;		{The value we'll be sending back.}
	m: RPGMenuKeyPtr;
	UK: Boolean;		{Has a special MenuKey been pressed?}
	OldMouseX, OldMouseY: Integer;	{ TUNGINOBI: I got sick of the mouse cursor getting }
					{ in the way of the keyboard, so this will only }
					{ change the menu item if the mouse has moved. }
	Scrollbar_DragMode: Boolean = False;
	MyRect: TSDL_Rect;
begin
	RPMReposition_center( RPM );

	{The menu is now active!}
	RPM^.Active := True;
	{Show the menu to the user.}
	DisplayMenu( RPM , ReDrawer );
	DoFlip;

	{TUNGINOBI: Initialise the mouse movement variables}
	OldMouseX := Mouse_X;
	OldMouseY := Mouse_Y;

	{Initialize UK}
	UK := False;

	IsMenuActive := True;

	{Start the loop. Remain in this loop until either the player makes a selection}
	{or cancels the menu using the ESC key.}
	repeat
		DisplayMenu(RPM,ReDrawer);
		MyRect := RPM^.Menu_Zone^.GetRect();
		DoFlip;

		{Read the input from the keyboard.}
		getit := RPGKeyEvent;


		{Check to see if a special MENU KEY has been pressed.}
		if RPM^.FirstKey <> Nil then begin
			m := RPM^.FirstKey;
			while m <> Nil do begin
				if getit = m^.k then begin
					UK := True;
					r := m^.value;
				end;
				m := m^.next;
			end;
		end;
		if UK then begin
			break;
		end;


		{Certain keys need processing- if so, process them.}

		{Selection Movement Keys}
		if (KBD_NONE = getit) then begin
		end else if (KeyMap[ KMC_MenuUp    ].KCode = getit) then begin RPMUpKey( RPM , True );
		end else if (KeyMap[ KMC_MenuDown  ].KCode = getit) then begin RPMDownKey( RPM , True );
		end else if (KeyMap[ KMC_PageUp    ].KCode = getit) or (KeyMap[ KMC_ButtonWLeft  ].KCode = getit) or (KeyMap[ KMC_ButtonWBack    ].KCode = getit) then begin RPMPgUpKey( RPM , True );
		end else if (KeyMap[ KMC_PageDown  ].KCode = getit) or (KeyMap[ KMC_ButtonWRight ].KCode = getit) or (KeyMap[ KMC_ButtonWForward ].KCode = getit) then begin RPMPgDownKey( RPM , True );
		end else if (KeyMap[ KMC_ScrollUp  ].KCode = getit) or (KeyMap[ KMC_ButtonWUp   ].KCode = getit) then begin RPMScrollUpKey( RPM , False );
		end else if (KeyMap[ KMC_ScrollDown].KCode = getit) or (KeyMap[ KMC_ButtonWDown ].KCode = getit) then begin RPMScrollDownKey( RPM , False );
		end else if (KeyMap[ KMC_SearchParent   ].KCode = getit) then begin RPMSearchParentKey( RPM );
		end else if (KeyMap[ KMC_SearchBackward ].KCode = getit) then begin RPMSearchBackwardKey( RPM );
		end else if (KeyMap[ KMC_SearchForward  ].KCode = getit) then begin RPMSearchForwardKey( RPM );

		{If we receive an ESC, better check to make sure we're in a}
		{cancelable menu. If not, convert the ESC to an unused key.}
		end else if (KBD_ESC = getit) then begin if RPM^.Mode = RPMNoCancel then getit := 'Q';
		{ If we get a backspace, conver that to ESC. }
		end else if (KBD_BS = getit) then begin if RPM^.Mode <> RPMNoCancel then getit := KBD_ESC;
		{ Convert enter to space. }
		end else if (KBD_RET = getit) then begin getit := ' ';
		end else if (KBD_LF = getit) then begin getit := ' ';

		{Certain keys need processing- if so, process them.}
		end else if (RPK_MouseMotion = getit) then begin
				if Scrollbar_DragMode and SDL_Show_MenuScrollbar then begin
					if (MyRect.Y <= Mouse_Y) and (Mouse_Y <= (MyRect.Y + MyRect.H)) then begin
						RPM^.selectitem := ((Mouse_Y - (MyRect.Y + SDL_Scrollbar_Size)) * RPM^.numitem) div (MyRect.H - SDL_Scrollbar_Size - SDL_Scrollbar_Size) + 1;
						RPMReposition( RPM );
					end;
				end else
				{ If the mouse pointer is around }
				{ the menu, we may have to do something. }
				if ( Mouse_X >= MyRect.X ) and ( Mouse_X <= ( MyRect.X + MyRect.W ) ) and (( Mouse_X <> OldMouseX ) or ( Mouse_Y <> OldMouseY )) then begin
					if SDL_Show_MenuScrollbar and ((MyRect.X + MyRect.W - SDL_Scrollbar_Size) <= Mouse_X) then begin
						{ Region of ScrollBar }
					end else if ( Mouse_Y < ( MyRect.Y ) ) then begin
					end else if ( Mouse_Y > ( MyRect.Y + MyRect.H ) ) then begin
					end else begin
						I := ( Mouse_Y - MyRect.Y ) div TTF_FontLineSkip( game_font );
						if (0 <= I) and (I < MenuHeight( RPM )) then begin
							SetItemByPosition( RPM , RPM^.TopItem + I );
						end;
						{ Upon setting an item directly, freeze the mouse. }
						OldMouseX := Mouse_X;
						OldMouseY := Mouse_Y;
					end;
				end;
		end else if (RPK_MouseButton = getit) then begin
				{ If the mouse hit happened inside }
				{ the menu area, it was a selection. }
				{ Otherwise it was a cancel. }
				if ( Mouse_X >= MyRect.X ) and ( Mouse_X <= ( MyRect.X + MyRect.W )) and ( Mouse_Y >= MyRect.Y ) and ( Mouse_Y <= ( MyRect.Y + MyRect.H )) then begin
					if SDL_Show_MenuScrollbar and ((MyRect.X + MyRect.W - SDL_Scrollbar_Size) <= Mouse_X) then begin
						{ Region of ScrollBar }
						if (Mouse_Y <= (MyRect.Y + SDL_Scrollbar_Size)) then begin
							RPMScrollUpKey( RPM , False );
						end else if ((MyRect.Y + MyRect.H - SDL_Scrollbar_Size) <= Mouse_Y) then begin
							RPMScrollDownKey( RPM , False );
						end else if (Mouse_Y < (MyRect.Y + SDL_Scrollbar_Size + (((RPM^.topitem - 1) * (MyRect.H - SDL_Scrollbar_Size - SDL_Scrollbar_Size)) div RPM^.numitem))) then begin
							RPMPgUpKey( RPM , True );
						end else if ((MyRect.Y + SDL_Scrollbar_Size + (((RPM^.topitem - 1 + MenuHeight(RPM)) * (MyRect.H - SDL_Scrollbar_Size - SDL_Scrollbar_Size)) div RPM^.numitem)) < Mouse_Y) then begin
							RPMPgDownKey( RPM , True );
						end else begin
							Scrollbar_DragMode := True;
						end;
						getit := KBD_NONE;
					end else begin
						getit := ' ';
					end;
				end else if ENABLE_MouseClick_OutsideTheMenuArea then begin
					if RPM^.Mode <> RPMNoCancel then getit := KBD_ESC
					else getit := ' ';
				end;
		end else if (RPK_MouseButtonRelease = getit) then begin
				begin
					if Scrollbar_DragMode and SDL_Show_MenuScrollbar then begin
						if (MyRect.Y <= Mouse_Y) and (Mouse_Y <= (MyRect.Y + MyRect.H)) then begin
							RPM^.selectitem := ((Mouse_Y - (MyRect.Y + SDL_Scrollbar_Size)) * RPM^.numitem) div (MyRect.H - SDL_Scrollbar_Size - SDL_Scrollbar_Size) + 1;
							RPMReposition( RPM );
						end;
					end;
					Scrollbar_DragMode := False;
					getit := KBD_NONE;
				end;
		end else if (RPK_RightButton = getit) then begin
			if ( RPM^.Mode <> RPMNoCancel ) and ( RPM^.Mode <> RPMEscCancel ) then getit := KBD_ESC;

		end;
		if EDIT_MenuOrder and ( RPMNoCancel <> RPM^.Mode ) then begin
			if RPM^.editable_menuorder and ( getit = KeyMap[ KMC_MenuOrder_Edit ].KCode ) then begin
				getit := KBD_NONE; r := SELECTMENU_MenuOrder_Edit; UK := True;
			end else if ( getit = KeyMap[ KMC_MenuOrder_SortAlpha ].KCode ) then begin
				getit := KBD_NONE; r := SELECTMENU_MenuOrder_SortAlpha; RPMSortAlpha( RPM );
			end else if ( getit = KeyMap[ KMC_MenuOrder_SortValue ].KCode ) then begin
				getit := KBD_NONE; r := SELECTMENU_MenuOrder_SortValue; RPMSortValue( RPM );
			end;
		end;

	{Check for a SPACE or ESC.}
	until (' ' = getit) or (KBD_ESC = getit) or UK;

	IsMenuActive := False;

	{The menu is no longer active.}
	RPM^.Active := False;

	{We have to send back a different value depending upon whether a selection}
	{was made or the menu was cancelled. If an item was selected, return its}
	{value field. The value always returned by a cancel will be -1.}
	{If a MenuKey was pressed, r already contains the right value.}
	if ( ' ' = getit ) then begin
		r := RPMLocateByPosition(RPM,RPM^.selectitem)^.value;
	end else if not UK then begin
		r := SELECTMENU_Cancel;
	end;

	SelectMenu := r;
end;

Procedure RPMSortAlpha(RPM: RPGMenuPtr);
	{Given a menu, RPM, sort its items based on the alphabetical}
	{order of their msg fields.}
	{I should mention here that I haven't written a sorting}
	{algorithm in years, and only once on a linked list (CS assignment).}
	{I think this is an insertion sort... I checked on internet for}
	{examples of sorting techniques, found a bunch of contradictory}
	{information, and decided to just write the easiest thing that}
	{would work. Since we're dealing with a relatively small number}
	{of items here, speed shouldn't be that big a concern.}
var
	Sorted: RPGMenuItemPtr;	{The sorted list}
	NextBranches: RPGMenuItemPtr;
	CurrentBranch_Top,CurrentBranch_Bottom: RPGMenuItemPtr;
	TmpBranch_Top,TmpBranch_Bottom: RPGMenuItemPtr;
	TmpNextBranches: RPGMenuItemPtr;
	youshouldstop: Boolean;	{Can you think of a better name?}
begin
	{Initialize NextBranches and Sorted.}
	NextBranches := RPM^.firstitem;
	Sorted := NIL;

	while ( NIL <> NextBranches ) do begin
		CurrentBranch_Top    := NextBranches;	{CurrentBranch_Top is to be added to Sorted}
		CurrentBranch_Bottom := NextBranches;
		while ( NIL <> CurrentBranch_Bottom^.next ) and ( 1 <= Length(CurrentBranch_Bottom^.next^.msg) ) and ( #$0 = CurrentBranch_Bottom^.next^.msg[1] ) do begin
			CurrentBranch_Bottom := CurrentBranch_Bottom^.next;
		end;
		NextBranches := CurrentBranch_Bottom^.next;	{increase a to the next item in the menu}

		{Give CurrentBranch_Top's Next field a value of Nil.}
		CurrentBranch_Bottom^.next := NIL;

		{Locate the correct position in Sorted to store CurrentBranch_Top}
		if ( NIL = Sorted ) then begin
			{This is the trivial case- Sorted is empty.}
			Sorted := CurrentBranch_Top
		end else if ( CurrentBranch_Top^.msg < Sorted^.msg ) then begin
			{CurrentBranch_Top-CurrentBranch_Bottom should be the first element in the list.}
			TmpNextBranches := Sorted;
			Sorted := CurrentBranch_Top;
			CurrentBranch_Bottom^.next := TmpNextBranches;
		end else begin
			{TmpNextBranches and TmpBranch_Top will be used to move through Sorted.}
			TmpNextBranches := Sorted;

			{Locate the last item lower than CurrentBranch_Top-CurrentBranch_Bottom}
			youshouldstop := false;
			repeat
				TmpBranch_Top    := TmpNextBranches;
				TmpBranch_Bottom := TmpNextBranches;
				while ( NIL <> TmpBranch_Bottom^.next ) and ( 1 <= Length(TmpBranch_Bottom^.next^.msg) ) and (#$0 = TmpBranch_Bottom^.next^.msg[1]) do begin
					TmpBranch_Bottom := TmpBranch_Bottom^.next;
				end;
				TmpNextBranches := TmpBranch_Bottom^.next;

				if ( NIL = TmpNextBranches ) then begin
					youshouldstop := true
				end else if ( CurrentBranch_Top^.msg < TmpNextBranches^.msg ) then begin
					youshouldstop := true;
				end;
			until youshouldstop;
			CurrentBranch_Bottom^.next := TmpNextBranches;
			TmpBranch_Bottom^.next := CurrentBranch_Top;
		end;
	end;
	RPM^.firstitem := Sorted;
end;

Procedure RPMSortValue(RPM: RPGMenuPtr);
	{Given a menu, RPM, sort its items based on the value}
	{order of their value fields.}
var
	Sorted: RPGMenuItemPtr;	{The sorted list}
	NextBranches: RPGMenuItemPtr;
	CurrentBranch_Top,CurrentBranch_Bottom: RPGMenuItemPtr;
	TmpBranch_Top,TmpBranch_Bottom: RPGMenuItemPtr;
	TmpNextBranches: RPGMenuItemPtr;
	youshouldstop: Boolean;	{Can you think of a better name?}
begin
	{Initialize NextBranches and Sorted.}
	NextBranches := RPM^.firstitem;
	Sorted := NIL;

	while ( NIL <> NextBranches ) do begin
		CurrentBranch_Top    := NextBranches;	{CurrentBranch_Top is to be added to Sorted}
		CurrentBranch_Bottom := NextBranches;
		while ( NIL <> CurrentBranch_Bottom^.next ) and ( 1 <= Length(CurrentBranch_Bottom^.next^.msg) ) and ( #$0 = CurrentBranch_Bottom^.next^.msg[1] ) do begin
			CurrentBranch_Bottom := CurrentBranch_Bottom^.next;
		end;
		NextBranches := CurrentBranch_Bottom^.next;	{increase a to the next item in the menu}

		{Give CurrentBranch_Top's Next field a value of Nil.}
		CurrentBranch_Bottom^.next := NIL;

		{Locate the correct position in Sorted to store CurrentBranch_Top}
		if ( NIL = Sorted ) then begin
			{This is the trivial case- Sorted is empty.}
			Sorted := CurrentBranch_Top
		end else if ( CurrentBranch_Top^.value < Sorted^.value ) then begin
			{CurrentBranch_Top-CurrentBranch_Bottom should be the first element in the list.}
			TmpNextBranches := Sorted;
			Sorted := CurrentBranch_Top;
			CurrentBranch_Bottom^.next := TmpNextBranches;
		end else begin
			{TmpNextBranches and TmpBranch_Top will be used to move through Sorted.}
			TmpNextBranches := Sorted;

			{Locate the last item lower than CurrentBranch_Top-CurrentBranch_Bottom}
			youshouldstop := false;
			repeat
				TmpBranch_Top    := TmpNextBranches;
				TmpBranch_Bottom := TmpNextBranches;
				while ( NIL <> TmpBranch_Bottom^.next ) and ( 1 <= Length(TmpBranch_Bottom^.next^.msg) ) and (#$0 = TmpBranch_Bottom^.next^.msg[1]) do begin
					TmpBranch_Bottom := TmpBranch_Bottom^.next;
				end;
				TmpNextBranches := TmpBranch_Bottom^.next;

				if ( NIL = TmpNextBranches ) then begin
					youshouldstop := true
				end else if ( CurrentBranch_Top^.value < TmpNextBranches^.value ) then begin
					youshouldstop := true;
				end;
			until youshouldstop;
			CurrentBranch_Bottom^.next := TmpNextBranches;
			TmpBranch_Bottom^.next := CurrentBranch_Top;
		end;
	end;
	RPM^.firstitem := Sorted;
end;

Function CurrentMenuItemValue( RPM: RPGMenuPtr ): integer;
	{ Determine the value of the current menu item, and return it. }
	{ Return 0 if the item is not found. }
var
	Item: RPGMenuItemPtr;
begin
	item := RPMLocateByPosition( RPM , RPM^.SelectItem );
	if item = Nil then begin
		CurrentMenuItemValue := 0;
	end else begin
		CurrentMenuItemValue := item^.value;
	end;
end;

Function SetItemByValue( RPM: RPGMenuPtr ; V: Integer ): RPGMenuItemPtr;
	{ Search through the list, and set the SelectItem }
	{ field to the first menu item which matches V. }
var
	T: Integer;
	MI: RPGMenuItemPtr;
begin
	if RPM = Nil then exit;

	MI := RPM^.FirstItem;
	T := 1;

	while (MI <> Nil) and (MI^.Value <> V) do begin
		MI := MI^.Next;
		Inc( T );
	end;

	if MI <> Nil then begin
		RPM^.SelectItem := T;

		if (RPM^.selectitem < RPM^.topitem) or ((RPM^.selectitem - RPM^.topitem) > MenuHeight( RPM ) ) then begin
			{Determine an appropriate new value for topitem.}
			RPMReposition(RPM,True);
		end;
	end;

	SetItemByValue := MI;
end;

Procedure SetItemByPosition( RPM: RPGMenuPtr ; N: Integer );
	{ Search through the list, and set the SelectItem }
	{ field to the Nth menu item. }
begin
	if RPM = Nil then exit;

	if N <= RPM^.NumItem then begin
		RPM^.SelectItem := N;

		if (RPM^.selectitem < RPM^.topitem) or ((RPM^.selectitem - RPM^.topitem + 1) > MenuHeight( RPM ) ) then begin
			{Determine an appropriate new value for topitem.}
			RPMReposition(RPM,True);
		end;
	end;
end;


Function BuildFileMenu( RPM: RPGMenuPtr; const SearchPattern: String; EncodedFileName: Boolean; N: Integer ): Integer;
	{ Do a DosSearch for files matching SearchPattern, then add }	
	{ each of the files found to the menu. }
var
	F: SearchRec;
begin
	FindFirst( SearchPattern , AnyFile , F );

	While DosError = 0 do begin
		if EncodedFileName then begin
			AddRPGMenuItem( RPM , TextDecode(F.Name) , N , F.Name );
		end else begin
			AddRPGMenuItem( RPM , F.Name , N , F.Name );
		end;
		Inc(N);
		FindNext( F );
	end;

	FindClose( F );
	BuildFileMenu := N;
end;

Function BuildFileMenu( RPM: RPGMenuPtr; const SearchPattern: String; EncodedFileName: Boolean ): Integer;
begin
	BuildFileMenu := BuildFileMenu( RPM , SearchPattern , EncodedFileName , 1 );
end;

Function SelectFile( RPM: RPGMenuPtr; ReDrawer: RedrawProcedureType ): String;
	{ RPM is a menu created by the BuildFileMenu procedure. }
	{ So, select one of the items and return the item name, which }
	{ should be a filename. }
var
	N: Integer;	{ The number of the file selected. }
	Name: String;	{ The name of the filename selected. }
begin
	{ Do the menu selection first. }
	N := SelectMenu( RPM , ReDrawer );

	if ( SELECTMENU_Cancel = N ) then begin
		{ Selection was canceled. So, return an empty string. }
		Name := '';
	end else begin
		{ Locate the selected element of the menu. }
		Name := RPMLocateByPosition(RPM,RPM^.SelectItem)^.msg;
	end;

	SelectFile := Name;
end;



initialization
begin
{$IFDEF DEBUG}
	ErrorMessage_fork('DEBUG: sdlmenus.pp');
{$ENDIF DEBUG}
	IsMenuActive := False;
end;

finalization
begin
{$IFDEF DEBUG}
	ErrorMessage_fork('DEBUG: sdlmenus.pp(finalization)');
{$ENDIF DEBUG}
end;

end.
