unit texutil;
	{This unit contains various useful functions for dealing}
	{with strings.}
{
	GearHead2, a roguelike mecha CRPG
	Copyright (C) 2005 Joseph Hewitt

	This library is free software; you can redistribute it and/or modify it
	under the terms of the GNU Lesser General Public License as published by
	the Free Software Foundation; either version 2.1 of the License, or (at
	your option) any later version.

	The full text of the LGPL can be found in license.txt.

	This library is distributed in the hope that it will be useful, but
	WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
	General Public License for more details. 

	You should have received a copy of the GNU Lesser General Public License
	along with this library; if not, write to the Free Software Foundation,
	Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
}
{$LONGSTRINGS ON}


interface

{ 'sysutils' must be the top of uses. }
uses sysutils,strings;

Function TextISO646_AllowableCheck( const c: Char ): Boolean;

Function TextEncode( const src: String ): String;
Function TextEncode_( const src: String ): String;
Function TextDecode( const src: String ): String;

Function IsUTF8CharLeadByte( c: Char ): Boolean;
Function IsUTF8CharTrailByte( c: Char ): Boolean;
Function LengthUTF8Char( c: Char ): Integer;

Function HeadUTF8Char( const msg: String ): String;
Function TailUTF8Char( const msg: String ): String;

Function EditUTF8CharStr( var basestr: String; const MaxLen: Integer; const MaxWidth: Integer; const key: Char; const addstr: PChar; var state: ShortInt; var mbchar_work: String ): Char;

Function DeleteWhiteSpace(var S: String): Integer;
Procedure DeleteFirstChar(var S: String);
Function UTF8StrWidth( const S: String ): Integer;
Function MBCharTrimedLength( const S: String; MaxWidth: Integer ): Integer;
Function ExtractWordForParse(var S: String; var SpaceDeleted,WordIsUTF8: Boolean ): String;
Function ExtractWordForPrint(var S: String; var SpaceDeleted,WordIsUTF8: Boolean ): String;
Function ExtractWord(var S: String): String;
Function ExtractValue(var S: String): LongInt;
Function ExtractTF(var S: String): Boolean;
Function EvaluateTF(S: String): Boolean;
Function ExtractReal(var S: String): Real;
Function RetrieveAString(const S: String): String;
Function RetrieveBracketString(const S: String): String;
Function RetrieveAPreamble(const S: String ): String;
Function RetrieveACode(const S: String ): String;

Function SanitizeFilename( S: String ): String;

Function BStr( N: Int64 ): String;
Function SgnStr( N: Integer ): String;
Function WideStr( N,Width: LongInt ): String;

Function Acronym( phrase: String ): String; {can't const}
Function Acronym( const phrase: String ; NumPlaces: Byte ): String;
Function Concentrate( const S: String ): String;

Function Sgn( N: LongInt ): Integer;
Function PartMatchesCriteria( const Part_In,Desc_In: String ): Boolean;
Function PartAtLeastOneMatch( const Part_In,Desc_In: String ): Boolean;

Function AStringHasBString( const A,B: String ): Boolean;
Function AStringHasBStringNum( const A,B: String ): Integer;
Function HeadMatchesString( const H,S: String ): Boolean;

Function QuickPCopy( const msg: String ): PChar;

Function IsPunctuation( C: Char ): Boolean;

Procedure ReplacePat( var msg: String; const pat_in,s: String );
Function ReplaceHash( const msg, S1, S2, S3, S4: String ): String;
Function ReplaceHash( const msg, S1, S2, S3: String ): String;
Function ReplaceHash( const msg, S1, S2: String ): String;
Function ReplaceHash( const msg, s: String ): String;
Function StringMatchWeight( Part,Desc: String ): Integer;

Procedure AlterDescriptors( var Original,Change: String );

Procedure AtoAn( var msg: String );
Procedure AddTraits( var Trait_List: String; Traits_To_Add: String );

Function QuoteString( A: String ): String;
Procedure AddToQuoteString( var QList,QToAdd: String );
Function NoQItemsMatch( const QList: String; var QToCheck: String ): Boolean;

implementation

uses
{$IFDEF DEBUG}
	errmsg,
{$ENDIF DEBUG}
	utf8msg;

var
	End_of_Word_Contains_WhiteSpace: Boolean = True;


Function TextISO646_AllowableCheck( const C: Char ): Boolean;
	{ Check, is a character C in ISO/IEC 646 (in other words, ISO 8859-1 G0 GL page), printable and allowable by SAtt in gears.pp. }
	{ return-code: True is OK, False is BAD. }
begin
	if c < #$20 then Exit( False ); { Not Printable }
	if #$7E < c then Exit( False ); { Not ISO/IEC 646 }
	if '<' = c then Exit( False ); { Used by *SAtt() }
	if '>' = c then Exit( False ); { Used by *SAtt() }
	{ if '#' = c then Exit( False ); } { Used by ReplaceHash() }
	if '%' = c then Exit( False ); { Used by FormatChatStringByGender() }
	{ if '\\' = c then Exit( False ); } { Used by script }
	Exit( True );
end;

Function TextEncode( const src: String ): String;
const
	AllowableCharacters = 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ1234567890()-=_.';
var
	Len, P: Integer;
begin
	TextEncode := '';
	Len := Length(src);
	P := 1;
	while (P <= Len) do begin
		if 0 < Pos(src[P], AllowableCharacters) then begin
			TextEncode := TextEncode + src[P];
			Inc(P);
		end else begin
			TextEncode := TextEncode + '%' + IntToHex(Ord(src[P]),2);
			Inc(P);
		end;
	end;
end;

Function TextEncode_( const src: String ): String;
const
	AllowableCharacters = 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ1234567890()-=_.';
var
	Len, P: Integer;
begin
	TextEncode_ := '';
	Len := Length(src);
	P := 1;
	while (P <= Len) do begin
		if 0 < Pos(src[P], AllowableCharacters) then begin
			TextEncode_ := TextEncode_ + src[P];
			Inc(P);
		end else begin
			TextEncode_ := TextEncode_ + '_';
			Inc(P);
		end;
	end;
end;

Function TextDecode( const src: String ): String;
var
	Len, P: Integer;
	tmp: String;
begin
	TextDecode := '';
	Len := Length(src);
	P := 1;
	while (P <= Len) do begin
		if '%' = src[P] then begin
			Inc(P);
			tmp := '';
			if (P +1) <= Len then begin
				tmp := '$' + src[P] + src[P+1];
				Inc(P);
				Inc(P);
			end else if P <= Len then begin
				tmp := '$' + src[P];
				Inc(P);
			end;
			TextDecode := TextDecode + Chr(StrToInt(tmp));
		end else begin
			TextDecode := TextDecode + src[P];
			Inc(P);
		end;
	end;
end;


Function IsUTF8CharLeadByte( c: Char ): Boolean;
begin
	exit (((#$C0 <= c) and (c <= #$DF)) or ((#$E0 <= c) and (c <= #$EF)) or ((#$F0 <= c) and (c <= #$F7)));
end;

Function IsUTF8CharTrailByte( c: Char ): Boolean;
begin
	exit ((#$80 <= c) and (c <= #$BF));
end;

Function LengthUTF8Char( c: Char ): Integer;
begin
	if IsUTF8CharLeadByte(c) then begin
		if (#$C0 <= c) and (c <= #$DF) then LengthUTF8Char := 2
		else if (#$E0 <= c) and (c <= #$EF) then LengthUTF8Char := 3
		else if (#$F0 <= c) and (c <= #$F7) then LengthUTF8Char := 4;
	end else if IsUTF8CharTrailByte(c) then LengthUTF8Char := 1
	else LengthUTF8Char := 0;
end;

Function HeadUTF8Char( const msg: String ): String;
var
	len: Integer;
begin
	len := LengthUTF8Char( msg[1] );
	if 0 = len then HeadUTF8Char := msg[1]
	else HeadUTF8Char := Copy( msg, 1, len );
end;

Function TailUTF8Char( const msg: String ): String;
var
	MaxLen: Integer;
	P, LastP: Integer;
	len: Integer;
begin
	MaxLen := Length( msg );
	P := 1;	LastP := 1;
	while (P <= MaxLen) do begin
		LastP := P;
		len := LengthUTF8Char( msg[P] );
		if 0 = len then Inc(P)
		else P := P + len;
	end;

	if 1 = P then TailUTF8Char := ''
	else if 0 = len then TailUTF8Char := msg[LastP]
	else TailUTF8Char := Copy( msg, LastP, len );
end;

Function EditUTF8CharStr( var basestr: String; const MaxLen: Integer; const MaxWidth: Integer; const key: Char; const addstr: PChar; var state: ShortInt; var mbchar_work: String ): Char;
	{ input: 'basestr': To edit string. }
	{ input: 'MaxLen': Maximum size (byte length) of basestr. }
	{ input: 'MaxWidth': Maximum width (drawable length) of basestr. }
	{ input: 'key': A charactor to add basestr. It is exclusive using with 'addstr'. }
	{ input: 'addstr': Charactors to add basestr. It is exclusive using with 'key'. }
	{ input: 'state': In initially, set 0 by calling side. If not 0, now in processing multi-byte charactors. }
	{ return: 'basestr': Edited string. }
	{ return: 'state': Length of remnant multi-byte chractors. }
	{ return code is #255: 'key' is continued multi-byte charctor. 'state' is length of the continuing. }
	{ return code is #0: Finished successfully. }
	{ return code is others: 'key' is unknown charactor. Do nothing. }
	{ ATTENTION: Input value 'MaxLen' is a length of bytes. }
	{ ATTENTION: Input value 'MaxWidth' is a width for display. }
	{ ATTENTION: In CJK, there are some charctors, one charactor have double size for one ANK charactor and data length is 3 or 4 bytes. }
var
	BW: String;
begin
	EditUTF8CharStr := #0;
	if (0 = Length(basestr)) and (0 = state) then begin
		mbchar_work := '';
	end;

	if (#0 = key) and (NIL = addstr) then begin
		// May be bug.
	end else if (#0 = key) then begin
		// if not(0 = state) then  May be bug.
		if ((Length(basestr) + Length(addstr)) <= MaxLen) and ((UTF8StrWidth(basestr) + UTF8StrWidth(addstr)) <= MaxWidth) then begin
			basestr := basestr + addstr;
		end;
		state := 0;
		mbchar_work := '';
	end else begin
		if 0 = state then begin
			if (#$8 = key) then begin
				BW := TailUTF8Char( basestr );
				basestr := Copy( basestr, 1, Length(basestr) - Length(BW) );
				state := 0;
				mbchar_work := '';
			end else if (#$15 = key) then begin
				basestr := '';
				state := 0;
				mbchar_work := '';
			end else if TextISO646_AllowableCheck(key) then begin
				if (Length( basestr ) < MaxLen) and (UTF8StrWidth(basestr) < MaxWidth) then begin
					basestr := basestr + key;
				end;
				state := 0;
				mbchar_work := '';
			end else if IsUTF8CharLeadByte(key) then begin
				state := LengthUTF8Char( key );
				if ((Length(basestr) + state) <= MaxLen) and ((UTF8StrWidth(basestr) +2) <= MaxWidth) then begin
					mbchar_work := key;
				end else begin
					mbchar_work := '';
				end;
				Dec( state );
				EditUTF8CharStr := #$FF;
			end else begin
				state := 0;
				mbchar_work := '';
				EditUTF8CharStr := key;
			end;
		end else begin
			if IsUTF8CharTrailByte(key) then begin
				if '' <> mbchar_work then begin
					mbchar_work := mbchar_work + key;
				end;
				Dec( state );
				if 0 < state then EditUTF8CharStr := #$FF
				else begin
					EditUTF8CharStr := #0;
					basestr := basestr + mbchar_work;
					state := 0;
					mbchar_work := '';
				end;
			end else begin
				// May be bug.
				EditUTF8CharStr := #0;
				state := 0;
				mbchar_work := '';
			end;
		end;
	end;
end;

Function DeleteWhiteSpace(var S: String): Integer;
	{Delete any whitespace which is at the beginning of}
	{string S. If S is nothing but whitespace, or if it}
	{contains nothing, return an empty string.}
	{ BUGS - None detected. Test harnessed and everything.}
var
	len: Integer;
	P: Integer;
begin
	DeleteWhiteSpace := 0;

	{ Error check }
	if S = '' then Exit;

	{Locate the first relevant char.}
	len := Length(S);
	P := 1;
	while (P < len) and ((S[P] = ' ') or (S[P] = #9) or (S[P] = #$0)) do begin
		Inc(P);
	end;

	{Copy the string from the first nonspace to the end.}
	if (S[P] = ' ') or (S[P] = #9) then begin
		S := '';
		DeleteWhiteSpace := ( P - 1 );
	end else if (1 < P) then begin
		S := Copy(S,P,Length(S)-P+1);
		DeleteWhiteSpace := ( P - 1 );
	end;
end;

Procedure DeleteFirstChar(var S: String);
	{ Remove the first character from string S. }
begin
	{Copy the string from the first nonspace to the end.}
	if Length( S ) < 2 then S := ''
	else S := Copy(S,2,Length(S));
end;

Function UTF8StrWidth( const S: String ): Integer;
	{ BUG: Half-width character is not supported. }
var
	MaxLen: Integer;
	len: Integer;
	UTF8Length: Integer;
begin
	MaxLen := Length(S);
	UTF8StrWidth := 0;
	UTF8Length := 0;

	while (UTF8Length < MaxLen) do begin
		len := LengthUTF8Char( S[UTF8Length +1] );
		if 0 < len then begin
			UTF8StrWidth := UTF8StrWidth + 2;
			UTF8Length := UTF8Length + len;
		end else begin
			Inc(UTF8StrWidth);
			Inc(UTF8Length);
		end;
	end;
end;

Function MBCharTrimedLength( const S: String; MaxWidth: Integer ): Integer;
	{ Calclate a length, a string to be in a MaxWidth. }
	{ ATTENTION: Input value 'MaxWidth' is a width for display. }
	{ ATTENTION: Returned value is a length of bytes. }
	{ ATTENTION: In CJK, there are many charctors, one charactor have double size for one ANK charactor and data length is 3 or 4 bytes. }
var
	MaxLen: Integer;
	len: Integer;
	TrimedWidth: Integer;
begin
	MaxLen := Length(S);
	TrimedWidth := 0;
	MBCharTrimedLength := 0;

	while (MBCharTrimedLength < MaxLen) and (TrimedWidth < MaxWidth) do begin
		len := LengthUTF8Char( S[MBCharTrimedLength +1] );
		if 0 < len then begin
			if ((TrimedWidth + 2) <= MaxWidth) then begin
				TrimedWidth := TrimedWidth + 2;
				MBCharTrimedLength := MBCharTrimedLength + len;
			end else begin
				break;
			end;
		end else begin
			Inc(TrimedWidth);
			Inc(MBCharTrimedLength);
		end;
	end;
end;


Function ExtractWord_Internal(var S: String; var SpaceDeleted,WordIsUTF8: Boolean; EOW_Contains_SP: Boolean ): String;
	{Extract the next word from string S.}
	{Return this substring as the function's result;}
	{truncate S so that it is now the remainder of the string.}
	{If there is no word to extract, both S and the function}
	{result will be set to empty strings.}
	{ BUGS - None found.}
var
	Len: Integer;
	P, P2: Integer;
	I: Integer;
	it: String;
begin
	{To start the process, strip all whitespace from the}
	{beginning of the string.}
	SpaceDeleted := ( 1 <= DeleteWhiteSpace(S) );
	WordIsUTF8 := False;

	{Error check- make sure that we have something left to}
	{extract! The string could have been nothing but white space.}
	if S <> '' then begin

		{Determine the position of the next whitespace.}
		if not(EOW_Contains_SP) and IsUTF8CharLeadByte(S[1]) then begin
			{  When a function parse UTF-8 string, }
			{ it is difficult to determine pauses of words. }
			{  Therefore, this function return }
			{ only one UTF-8 character. }
			len := LengthUTF8Char( S[1] );
			if (len <= Length(S)) then begin
				it := Copy(S,1,len);
				S := Copy(S,1+len,Length(S)-len);
			end else begin
				// May be bug.
				it := Copy(S,1,Length(S));
				S := '';
			end;
			WordIsUTF8 := true;
		end else begin
			P := Pos(' ',S);
			P2 := Pos(#9,S);
			if (0 = P) or ((0 < P2) and (P2 < P)) then P := P2;

			if 0 < P then Len := P - 1
			else Len := Length(S);

			if not(EOW_Contains_SP) then begin
				for I := 1 to Len do begin
					if IsUTF8CharLeadByte(S[I]) then begin
						P := I;
						break;
					end;
				end;
			end;

			{Extract the command.}
			if P <> 0 then begin
				it := Copy(S,1,P-1);
				S := Copy(S,P,Length(S)-P+1);
			end else begin
				it := Copy(S,1,Length(S));
				S := '';
			end;
		end;

	end else begin
		it := '';
	end;

	ExtractWord_Internal := it;
end;

Function ExtractWordForParse(var S: String; var SpaceDeleted,WordIsUTF8: Boolean ): String;
begin
	ExtractWordForParse := ExtractWord_Internal( S , SpaceDeleted , WordIsUTF8 , False );
end;

Function ExtractWordForPrint(var S: String; var SpaceDeleted,WordIsUTF8: Boolean ): String;
begin
	ExtractWordForPrint := ExtractWord_Internal( S , SpaceDeleted , WordIsUTF8 , End_of_Word_Contains_WhiteSpace );
end;

Function ExtractWord(var S: String): String;
	{Extract the next word from string S.}
	{Return this substring as the function's result;}
	{truncate S so that it is now the remainder of the string.}
	{If there is no word to extract, both S and the function}
	{result will be set to empty strings.}
	{ BUGS - None found.}
var
	Len: Integer;
	P, P2: Integer;
	I: Integer;
	it: String;
begin
	{To start the process, strip all whitespace from the}
	{beginning of the string.}
	DeleteWhiteSpace(S);

	{Error check- make sure that we have something left to}
	{extract! The string could have been nothing but white space.}
	if S <> '' then begin

		{Determine the position of the next whitespace.}
		if IsUTF8CharLeadByte(S[1]) then begin
			{  When a function parse UTF-8 string, }
			{ it is difficult to determine pauses of words. }
			{  Therefore, this function return }
			{ from start of UTF-8 string to end of UTF-8 string. }
			P := 1;
			repeat
				len := LengthUTF8Char( S[P] );
				P := P + len;
			until (P > Length(S)) or not(IsUTF8CharLeadByte(S[P]));
			if P <= Length(S) then begin
				it := Copy(S,1,P-1);
				S := Copy(S,P,Length(S)-P+1);
			end else begin
				it := Copy(S,1,Length(S));
				S := '';
			end;
		end else begin
			P := Pos(' ',S);
			P2 := Pos(#9,S);
			if (0 = P) or ((0 < P2) and (P2 < P)) then P := P2;

			if 0 < P then Len := P - 1
			else Len := Length(S);

			for I := 1 to Len do begin
				if IsUTF8CharLeadByte(S[I]) then begin
					P := I;
					break;
				end;
			end;

			{Extract the command.}
			if P <> 0 then begin
				it := Copy(S,1,P-1);
				S := Copy(S,P,Length(S)-P+1);
			end else begin
				it := Copy(S,1,Length(S));
				S := '';
			end;
		end;

	end else begin
		it := '';
	end;

	ExtractWord := it;
end;

Function ExtractValue(var S: String): LongInt;
	{This is similar to the above procedure, but}
	{instead of a word it extracts a numeric value.}
	{Return 0 if the extraction should fail for any reason.}
var
	S2: String;
	it,C: LongInt;
begin
	S2 := ExtractWord(S);
	Val(S2,it,C);
	if C <> 0 then it := 0;
	ExtractValue := it;
end;

Function ExtractTF(var S: String): Boolean;
var
	S2: String;
begin
	S2 := UpCase(ExtractWord(S));

	if '' = S2 then Exit(True);

	if 'TRUE'  = S2 then Exit(True);
	if 'FALSE' = S2 then Exit(False);
	if 'T' = S2[1] then Exit(True);
	if 'F' = S2[1] then Exit(False);

	if 'YES' = S2 then Exit(True);
	if 'NO'  = S2 then Exit(False);
	if 'Y' = S2[1] then Exit(True);
	if 'N' = S2[1] then Exit(False);

	if '-1' = S2 then Exit(True);
	if '1'  = S2 then Exit(True);
	if '0'  = S2 then Exit(False);

	ExtractTF := False;
end;

Function EvaluateTF(S: String): Boolean;
var
	S2: String;
begin
	S2 := ExtractWord(S);
	EvaluateTF := ExtractTF(S2);
end;

Function ExtractReal(var S: String): Real;
	{This is similar to the above procedure, but}
	{instead of a word it extracts a numeric value.}
	{Return 0 if the extraction should fail for any reason.}
var
	S2: String;
	it: Real;
	C: Byte;
begin
	S2 := ExtractWord(S);
	Val(S2,it,C);
	if C <> 0 then it := 0;
	ExtractReal := it;
end;

Function RetrieveAString(const S: String): String;
	{Retrieve an Alligator String from S.}
	{Alligator Strings are defined as the part of the string}
	{that both alligarors want to eat, i.e. between < and >.}
var
	A1,A2: Integer;
begin
	{Locate the position of the two alligators.}
	A1 := Pos('<',S);
	A2 := Pos('>',S);

	{If the string has not been declared with <, return}
	{an empty string.}
	if A1 = 0 then Exit('');

	{If the string has not been closed with >, return the}
	{entire remaining length of the string.}
	if A2 = 0 then A2 := Length(S)+1;

	RetrieveAString := Copy(S,A1+1,A2-A1-1);
end;

Function RetrieveBracketString(const S: String): String;
	{ Like the above, but the string is surrounded by ( and ) . }
var
	A1,A2: Integer;
begin
	{Locate the position of the two alligators.}
	A1 := Pos('(',S);
	A2 := Pos(')',S);

	{If the string has not been declared with <, return}
	{an empty string.}
	if A1 = 0 then Exit('');

	{If the string has not been closed with >, return the}
	{entire remaining length of the string.}
	if A2 = 0 then A2 := Length(S)+1;

	RetrieveBracketString := Copy(S,A1+1,A2-A1-1);
end;


Function RetrieveAPreamble( const S: String ): String;
	{ Usually an alligator string will have some kind of label in }
	{ front of it. This function will retrieve the label in its }
	{ entirety. }
	{ LIMITATION: Doesn't return the character immediately before }
	{ the AString, which should be a space. }
var
	A1: Integer;
	msg: String;
begin
	A1 := Pos('<',S);

	if A1 <> 0 then begin
		msg := Copy(S, 1 , A1-2);
	end else begin
		msg := '';
	end;

	RetrieveAPreamble := msg;
end;

Function RetrieveACode( const S: String ): String;
	{ Usually an alligator string will have some kind of label in }
	{ front of it. This function will retrieve the label in its }
	{ entirety. }
var
	msg: String;
	code_len: Integer;
	p: Integer;
	c: Char;
begin
	msg := '';
	code_len := Pos('<',S);
	if ( 2 < code_len ) then begin
		code_len := code_len - 2;
		p := 1;
		while ( p <= code_len ) do begin
			c := s[p];
			if ( #$0 = c ) then break;
			if ( ' ' = c ) then break;
			if ( #9 = c ) then break;
			msg := msg + UpCase( c );
			inc( p );
		end;
	end;

	RetrieveACode := msg;
end;

Function BStr( N: Int64 ): String;
	{ This function functions as the BASIC Str function. }
var
	it: String;
begin
	Str(N, it);
	BStr := it;
end;

Function SgnStr( N: Integer ): String;
	{ Convert the string to a number, including either a '+' or '-'. }
var
	it: String;
begin
	it := BStr( N );
	if N>= 0 then it := '+' + it;
	SgnStr := it;
end;

Function WideStr( N,Width: LongInt ): String;
	{ Pack the string with zeroes until it's the specified width. }
	{ This command is being used for my clock. }
var
	msg: String;
begin
	msg := BStr( Abs( N ) );
	while Length( msg ) < Width do msg := '0' + msg;
	if N < 0 then msg := '-' + msg;
	WideStr := msg;
end;

function IsAlpha( C: Char ): Boolean;
	{ Return TRUE if C is a letter, FALSE otherwise. }
begin
	if ( UpCase( C ) >= 'A' ) and ( UpCase( C ) <= 'Z' ) then IsAlpha := True
	else IsAlpha := False;
end;

Function Acronym( phrase: String ): String; {can't const}
	{ Copy all the capital letters from the PHRASE, and construct an acronym. }
var
	A: String;	{ A String. In honor of the C64. }
	T: Integer;	{ A loop counter. In honor of the C64. }
begin
	A := '';

	for t := 1 to Length( phrase ) do begin
		if ( phrase[T] = UpCase( phrase[T] ) ) and IsAlpha( phrase[T] ) then A := A + phrase[T];
	end;

	Acronym := A;
end;

Function Acronym( const phrase: String ; NumPlaces: Byte ): String; 
	{ This function works like the above one, but pad out the acronym to }
	{ NumPlaces characters. }
var
	A: String;
begin
	A := Acronym( phrase );

	if Length( A ) > NumPlaces then begin
		A := Copy( A , 1 , NumPlaces );
	end else if Length( A ) < NumPlaces then begin
		while Length( A ) < NumPlaces do A := A + ' ';
	end;

	Acronym := A;
end;

Function Concentrate( const S: String ): String;
	{ Remove all white space from this string, leaving nothing }
	{ but concentrated alphanumeric goodness. }
var
	T: Integer;
	CS: String;
begin
	CS := '';
	for T := 1 to Length( S ) do begin
		{ If this character is neither a space nor a tab, }
		{ add it to our concentrated string. }
		if (S[T] <> ' ') and (S[T] = #9) then CS := CS + S[T];
	end;
	Concentrate := CS;
end;

Function SanitizeFilename( S: String ): String;
	{ Replace all proscribed characters with an underscore. }
const
	ProscribedCharacters = ',?"*~#%&{}:<>+|';
var
	T: Integer;
begin
	for T := 1 to Length( S ) do begin
		if Pos( S[T] , ProscribedCharacters ) > 0 then begin
			S[T] := '_';
		end;
	end;
	SanitizeFilename := S;
end;


Function Sgn( N: LongInt ): Integer;
	{ Return the sign of this number, just like in BASIC. }
begin
	if N > 0 then Sgn := 1
	else if N < 0 then Sgn := -1
	else Sgn := 0;
end;

Function StringMatchWeight( Part,Desc: String ): Integer;
	{ Return the match weight of PART to DESC. If the weight is 0, then }
	{ PART doesn't match DESC. All important traits listed in DESC must }
	{ be found in PART. Optional traits are preceded by a ~; these don't have }
	{ to be included in PART but increase the weight if they are. }
	{ Negative traits must be preceded by a -; these must _not_ be present }
	{ in Part. }
	Function ExtractOrClause(var S: String): String;
		{Extract the next trait from the or-list.}
	var
		P: Integer;
		it: String;
	begin
		{Error check- make sure that we have something left to}
		{extract! The string could have been nothing but white space.}
		if S <> '' then begin

			{Determine the position of the next whitespace.}
			P := Pos('|',S);
			if P = 0 then P := Pos(')',S);

			{Extract the command.}
			if P <> 0 then begin
				it := Copy(S,1,P-1);
				S := Copy(S,P+1,Length(S));
			end else begin
				it := Copy(S,1,Length(S));
				S := '';
			end;

		end else begin
			it := '';
		end;

		ExtractOrClause := it;
	end;
var
	Trait,T2: String;
	it,MatchFound: Boolean;
	N: Integer;
begin
	Part := UpCase( Part );
	Desc := UpCase( Desc );

	{ Assume TRUE unless a trait is found that isn't in NDesc. }
	it := True;
	N := -1;

	DeleteWhiteSpace( Desc );

	while Desc <> '' do begin
		Trait := ExtractWord( Desc );
		if Trait <> '' then begin
			if Trait[1] = '~' then begin
				DeleteFirstChar( Trait );
				if Pos( Trait , Part ) > 0 then Inc( N );

			end else if Trait[1] = '-' then begin
				{ A trait beginning with a "-" must NOT be present. }
				DeleteFirstChar( Trait );
				if Pos( Trait , Part ) <> 0 then begin
					it := False;
				end;
			end else if Trait[1] = '(' then begin
				{ A set of traits surrounded by parenthesis and separated by |s }
				{ is an or-list. One of the traits must be present. }
				DeleteFirstChar( Trait );
				MatChFound := False;
				repeat
					T2 := ExtractOrClause( Trait );
					if Pos( T2 , Part ) <> 0 then MatchFound := True;
				until ( Trait = '' ) or MatchFound;

				if MatchFound then Inc( N )
				else it := False;

			end else if UpCase( Trait ) = 'COMMON' then begin
				{ A trait marked as COMMON will appear more often, despite number }
				{ of matches. Use sparingly. }
				N := N + 5;

			end else begin
				if Pos( Trait , Part ) = 0 then begin
					it := False;
				end else begin
					Inc( N );
				end;
			end;
		end; { if Trait <> '' }
	end;

	if IT and ( N >= 0 ) then begin
		if N < 4 then begin
			StringMatchWeight := N + 1;
		end else begin
			StringMatchWeight := ( N * N div 2 ) - N + 2;
		end;
	end else begin
		StringMatchWeight := 0;
	end;
end;

Function PartMatchesCriteria( const Part_In,Desc_In: String ): Boolean;
	{ Return TRUE if the provided part description matches the provided }
	{ search criteria. Return FALSE otherwise. }
	{ A match is had if all the words in DESC are found in PART. }
begin
	PartMatchesCriteria := StringMatchWeight( Part_In + ' ONE_MATCH' , Desc_In + ' ONE_MATCH' ) > 0;
end;

Function PartAtLeastOneMatch( const Part_In,Desc_In: String ): Boolean;
	{ Return TRUE if the provided part description partially matches the provided }
	{ search criteria. Return FALSE otherwise. }
	{ A match is had if at least one word in DESC is found in PART. }
var
	Trait: String;
        Part, Desc: String;
	N: Integer;
begin
	Part := UpCase( Part_In );
	Desc := UpCase( Desc_In );

	N := 0;

	DeleteWhiteSpace( Desc );

	while Desc <> '' do begin
		Trait := ExtractWord( Desc );
		if Pos( Trait , Part ) <> 0 then Inc( N );
	end;

	PartAtLeastOneMatch := N > 0;
end;


Function AStringHasBString( const A,B: String ): Boolean;
	{ Return TRUE if B is contained in A, FALSE otherwise. }
begin
	AStringHasBString := Pos( UpCase( B ) , UpCase( A ) ) > 0;
end;

Function AStringHasBStringNum( const A,B: String ): Integer;
var
	p: Integer;
	l: Integer;
	s: String;
	n: Integer;
begin
	p := Pos( UpCase( B ) , UpCase( A ) );
	if ( p <= 0 ) then exit( 0 );
	l := p + length( B );
	if ( length( A ) <= l ) then exit( 0 );
	s := Copy( A , ( l + 1 ), length( A ) - l );
	s := ExtractWord( s );
	if ( Pos( Copy( S , 1 , 1 ) , '-0123456789' ) <= 0 ) then exit( 0 );
	n := StrToInt( s );
	AStringHasBStringNum := n;
end;

Function HeadMatchesString( const H,S: String ): Boolean;
	{ Return TRUE if the beginning Len(H) characters of S are H. }
var
	T : String;
begin
	T := Copy( S , 1 , Length( H ) );
	HeadMatchesString := UpCase( T ) = UpCase( H );
end;

Function QuickPCopy( const msg: String ): PChar;
	{ Life is short. Copy msg to a pchar without giving me any attitude about it. }
	{ Remember to deallocate that sucker when you're done playing with it. }
var
	pmsg: PChar;
begin
	pmsg := sysutils.StrAlloc( length(msg ) + 1 );
	StrPCopy( pmsg , msg );
	QuickPCopy := pmsg;
end;

Function IsPunctuation( C: Char ): Boolean;
	{ Return TRUE if C is some kind of punctuation, or FALSE otherwise. }
	{ This is used for the message scripting commands so please }
	{ forgive me if my definition of punctuation in this function }
	{ is not the same as my own. }
begin
	case C of
		'.',',',':',';','@','!','/','?','''': IsPunctuation := True;
		else IsPunctuation := False;
	end;
end;

Procedure ReplacePat( var msg: String; const pat_in,s: String );
	{ Replace all instances of PAT in MSG with S. }
var
	N: Integer;
	pat: String;
begin
	pat := UpCase( pat_in);
	{ Error check- if S contains the pattern there could be an infinite loop. }
	if AStringHasBString( S , pat ) then Exit;
	repeat
		N := Pos( pat , UpCase( msg ) );
		if N <> 0 then begin
			msg := Copy( msg , 1 , N - 1 ) + S + Copy( msg , N + Length( pat ) , Length( msg ) );
		end;
	until N = 0;
end;

Function ReplaceHash( const msg, S1, S2, S3, S4: String ): String;
var
	MaxLen: Integer;
	P: Integer;
	len: Integer;
	msg_out: String;
	tmp: String;
begin
	MaxLen := Length( msg );
	P := 1;
	msg_out := '';
	while (P <= MaxLen) do begin
		len := LengthUTF8Char( msg[P] );
		if 0 < len then begin
			msg_out := msg_out + Copy( msg, P, len );
			P := P + len;
		end else if ('#' = msg[P]) and (P+1 <= MaxLen) then begin
			case msg[P+1] of
			'#':	tmp := '#';
			'1':	tmp := S1;
			'2':	tmp := S2;
			'3':	tmp := S3;
			'4':	tmp := S4;
			else	tmp := '';
			end;
			msg_out := msg_out + tmp;
			P := P + 2;
		end else begin
			msg_out := msg_out + msg[P];
			Inc(P);
		end;
	end;
	ReplaceHash := msg_out;
end;

Function ReplaceHash( const msg, S1, S2, S3: String ): String;
begin
	ReplaceHash := ReplaceHash( msg, S1, S2, S3, '' );
end;

Function ReplaceHash( const msg, S1, S2: String ): String;
begin
	ReplaceHash := ReplaceHash( msg, S1, S2, '', '' );
end;

Function ReplaceHash( const msg,s: String ): String;
	{ Look for a hash sign in MSG. Replace it with S. }
var
	N: Integer;
        msg_out: String;
begin
	N := Pos( '#' , msg );
	if N <> 0 then begin
		msg_out := Copy( msg , 1 , N - 1 ) + S + Copy( msg , N + 1 , Length( msg ) );
	end else begin
                msg_out := msg;
	end;
	ReplaceHash := msg_out;
end;

Procedure AlterDescriptors( var Original,Change: String );
	{ Alter the XRan descriptors held in ORIGINAL, based on the changes }
	{ requested by CHANGE. Note that the contents of CHANGE will be utterly destrouyed }
	{ by this process. }
var
	cmd: String;
	N: Integer;
begin
	while change <> '' do begin
		cmd := extractword( Change );
		if cmd <> '' then begin
			N := Pos( Copy( cmd , 1 , 2 ) , Original );
			if N > 0 then begin
				Original := Copy( Original , 1 , N - 1 ) + cmd + Copy( Original , N + Length( cmd ) , Length( Original ) );
			end else begin
				Original := Original + ' ' + cmd;
			end;
		end;
	end;
end;

Procedure AtoAn( var msg: String );
	{ Go through the message, replacing "A" with "An" where appropriate. }
const
	vowels = 'aeiouAEIOU';
var
	w,msg_out: String;
begin
	msg_out := '';
	while msg <> '' do begin
		w := ExtractWord( msg );
		DeleteWhiteSpace( msg );

		if UpCase( w ) = 'A' then begin
			if Pos( msg[1] , vowels ) <> 0 then w := w + 'n';
		end;
		msg_out := msg_out + ' ' + w;
	end;
	msg := msg_out;
end;

Procedure AddTraits( var Trait_List: String; Traits_To_Add: String );
	{ TRAIT_LIST is a list of traits, such as that which may occur in a TYPE string. }
	{ Add the traits in TRAITS_TO_ADD to the list, but don't add any that already }
	{ occur in the list. }
var
	T: String;
begin
	while Traits_To_Add <> '' do begin
		T := ExtractWord( Traits_To_Add );
		if ( T <> '' ) and not AStringHasBString( Trait_List , T ) then Trait_List := Trait_List + ' ' + T;
	end;
end;

Function QuoteString( A: String ): String;
	{ Replace all spaces with quotes, so any single word from A can be }
	{ searched for like "this". }
var
	B,W: String;
begin
	B := '"';
	while A <> '' do begin
		W := ExtractWord( A );
		if W <> '' then B := B + W + '"';
	end;
	QuoteString := B;
end;

Procedure AddToQuoteString( var QList,QToAdd: String );
	{ Add some traits to a quote string. Make sure that everything is separated by quotes. }
var
	A: String;
begin
	while QToAdd <> '' do begin
		A := ExtractWord( QToAdd );
		if A <> '' then begin
			if QList = '' then QList := '"';
			QList := QList + A + '"';
		end;
	end;
end;

Function NoQItemsMatch( const QList: String; var QToCheck: String ): Boolean;
	{ Check to make sure that none of the words in QToCheck are present in QList. }
	{ This procedure will completely destroy QToCheck, by the way. }
var
	AllOK: Boolean;
	A: String;
begin
	{ Assume TRUE unless a match is found. }
	AllOK := True;
	while ( QToCheck <> '' ) and AllOK do begin
		A := ExtractWord( QToCheck );
		if ( A <> '' ) and AStringHasBString( QList , '"' + A + '"' ) then AllOK := False;
	end;
	NoQItemsMatch := AllOK;
end;



initialization
begin
{$IFDEF DEBUG}
	ErrorMessage_fork('DEBUG: texutil.pp');
{$ENDIF DEBUG}
	End_of_Word_Contains_WhiteSpace := EvaluateTF(UTF8_Settings('END_OF_WORD_CONTAINS_WHITESPACE',''));
end;

finalization
begin
{$IFDEF DEBUG}
	ErrorMessage_fork('DEBUG: texutil.pp(finalization)');
{$ENDIF DEBUG}
end;

end.
