unit utf8video;
{ GearHead UTF8 ASCII port }
{$LONGSTRINGS ON}


interface

const
{ Foreground and background color constants }
	Black         = 0;
	Blue          = 1;
	Green         = 2;
	Cyan          = 3;
	Red           = 4;
	Magenta       = 5;
	Brown         = 6;
	LightGray     = 7;

{ Foreground color constants }
	DarkGray      = 8;
	LightBlue     = 9;
	LightGreen    = 10;
	LightCyan     = 11;
	LightRed      = 12;
	LightMagenta  = 13;
	Yellow        = 14;
	White         = 15;

Procedure RedrawScreen;
Procedure ClrEol;
Procedure ClrScr;
Procedure GotoXY( X: Byte; Y: Byte );
Procedure TextBackground( CL: Byte );
Procedure TextColor( CL: Byte );
Function WhereX: Byte;
Function WhereY: Byte;
Procedure Window( AX1, AY1, AX2, AY2: Byte );
Procedure WriteLn( const Str: String );
Procedure Write( const Str: String );
Procedure WriteChar( c: Char );
Procedure CrtInit( W,H: Integer );
Procedure NormVideo;
Procedure DisposeWindow;



implementation

uses crt, dos, texutil, errmsg, ui4gh;

{ -------- Virtual Console-VRAM -------- }
const
	MaxMultibyteLen = 8;
Type
	rect = Record
		X1, Y1, X2, Y2: Integer;
	end;
var
	MaxCol: Integer = 80;
	MaxRow: Integer = 25;
	virtualscreen: PChar = NIL;	{ Text Image }
	fgcol: PByte = NIL;	{ foreground color }
	bgcol: PByte = NIL;	{ background color }
	currfgc: Integer = White;
	currbgc: Integer = 0;

	{ cursorstate: Integer; }
	cursorx: Integer = 0;
	cursory: Integer = 0;

	clip: rect;	{ clipping }

	bWindowClosing: Boolean = True;

{ -------- window proc -------- }
Procedure RedrawArea( ax1, ay1, ax2, ay2: Byte );
var
	x, y: Integer;
	bufptr: Integer;
	MBCharLen: Integer;
	m: Integer;
	S: String;
	need_goto: Boolean;
	fgc, bgc: Byte;
	S_clrline: String;
begin
	if bWindowClosing then exit;

	crt.TextColor( currfgc );
	crt.TextBackground( currbgc );
	S_clrline := '';
	for x := ax1 to ax2 do begin
		S_clrline := S_clrline + ' ';
	end;

	fgc := 255;
	bgc := 255;

	for y := ay1 to ay2 do begin
		bufptr := y * MaxCol + ax1;
		x := ax1;

		crt.GotoXY( x +1, y +1 );
		system.Write( S_clrline );
		need_goto := True;

		while x <= ax2 do begin
			if (#0 = virtualscreen[bufptr*MaxMultibyteLen]) then begin
				need_goto := True;
			end else begin
				if need_goto then begin
					crt.GotoXY( 1, y +1 );
					crt.GotoXY( x +1, y +1 );
					need_goto := False;
				end;
				if (fgc <> fgcol[bufptr]) then begin
					fgc := fgcol[bufptr];
					crt.TextColor( fgc );
				end;
				if (bgc <> bgcol[bufptr]) then begin
					bgc := bgcol[bufptr];
					crt.TextBackground( bgc );
				end;
				if (' ' = virtualscreen[bufptr*MaxMultibyteLen]) then begin
					system.Write( ' ' );
				end else begin
					MBCharLen := LengthUTF8Char( virtualscreen[bufptr*MaxMultibyteLen] );
					if 1 < MBCharLen then begin
						S := '';
						for m := 0 to (MBCharLen - 1) do begin
							S := S + virtualscreen[bufptr*MaxMultibyteLen+m];
						end;
						system.Write( S );
						need_goto := True;
						bufptr := bufptr + 1;
						x := x + 1;
					end else begin
						system.Write( virtualscreen[bufptr*MaxMultibyteLen] );
					end;
				end;
			end;
			Inc( bufptr );
			Inc( x );
		end;
	end;
end;


{----------------------------------------------------------------------------------}

Procedure RedrawScreen;
begin
	utf8video.RedrawArea( 0,0, (MaxCol - 1),(MaxRow - 1) );
end;

Procedure ClrEol;
var
	x: Integer;
	bufptr: Integer;
	m: Integer;
begin
	if bWindowClosing then exit;

	bufptr := cursory * MaxCol + cursorx;
	for x := cursorx to clip.X2 do begin
		for m := 0 to (MaxMultibyteLen - 1) do begin
			virtualscreen[bufptr*MaxMultibyteLen+m] := #$0;
		end;
		fgcol[bufptr] := currfgc;
		bgcol[bufptr] := currbgc;
		bufptr := bufptr+1;
	end;
end;


Procedure ClrScr;
var
	x, y: Integer;
	bufptr: Integer;
	m: Integer;
begin
	if bWindowClosing then exit;

	for y := clip.Y1 to clip.Y2 do begin
		bufptr := y * MaxCol + clip.X1;
		for x := clip.X1 to clip.X2 do begin
			for m := 0 to (MaxMultibyteLen - 1) do begin
				virtualscreen[bufptr*MaxMultibyteLen+m] := #$0;
			end;
			fgcol[bufptr] := currfgc;
			bgcol[bufptr] := currbgc;
			bufptr := bufptr + 1;
		end;
	end;
end;

Procedure GotoXY( X: Byte; Y: Byte );
begin
	cursorx := X - 1 + clip.X1;
	cursory := Y - 1 + clip.Y1;
	if (cursorx < clip.X1) then cursorx := clip.X1
	else if (cursorx > clip.X2) then cursorx := clip.X2;
	if (cursory < clip.Y1) then cursory := clip.Y1
	else if (cursory > clip.Y2) then cursory := clip.Y2;
end;

Procedure TextBackground( CL: Byte );
begin
	currbgc := CL;
end;

Procedure TextColor( CL: Byte );
begin
	currfgc := CL;
end;

Function WhereX: Byte;
begin
	WhereX := cursorx - clip.X1 + 1;
end;

Function WhereY: Byte;
begin
	WhereY := cursory - clip.Y1 + 1;
end;

Procedure Window( AX1, AY1, AX2, AY2: Byte );
var
	tmp: Integer;
begin
	clip.X1 := AX1 - 1;
	clip.Y1 := AY1 - 1;
	clip.X2 := AX2 - 1;
	clip.Y2 := AY2 - 1;
	if (clip.X1 > clip.X2) then begin tmp := clip.X1; clip.X1 := clip.X2; clip.X2 := tmp; end;
	if (clip.Y1 > clip.Y2) then begin tmp := clip.Y1; clip.Y1 := clip.Y2; clip.Y2 := tmp; end;
	if (clip.X1 < 0) then clip.X1 := 0;
	if (clip.Y1 < 0) then clip.Y1 := 0;
	if (clip.X2 > (MaxCol - 1)) then clip.X2 := (MaxCol - 1);
	if (clip.Y2 > (MaxRow - 1)) then clip.Y2 := (MaxRow - 1);
	cursorx := clip.X1;
	cursory := clip.Y1;
end;

Procedure NormVideo;
begin
	currfgc := LightGray;
	currbgc := Black;
	{ crt.Window( 0,0, (MaxCol - 1),(MaxRow - 1) ); }
	crt.ClrScr;
end;

Procedure ScrollOneLine;
var
	x, y: Integer;
	bufptr, bufptr2: Integer;
	m: Integer;
begin
	if bWindowClosing then exit;

	if clip.Y1 < clip.Y2 then begin
		for y := clip.Y1 + 1 to clip.Y2 do begin
			bufptr := (y - 1) * MaxCol + clip.X1;
			bufptr2 := y * MaxCol + clip.X1;
			for x := clip.X1 to clip.X2 do begin
				for m := 0 to (MaxMultibyteLen - 1) do begin
					virtualscreen[bufptr*MaxMultibyteLen+m] := virtualscreen[bufptr2*MaxMultibyteLen+m];
				end;
				fgcol[bufptr] := fgcol[bufptr2];
				bgcol[bufptr] := bgcol[bufptr2];
				Inc( bufptr );
				Inc( bufptr2 );
			end;
		end;
		{ utf8video.RedrawArea( clip.X1,clip.Y1, clip.X2,clip.Y2-1 ); }
		utf8video.ClrEol;
	end;
end;

Procedure Write( const Str: String );
var
	len, strptr, xlen, xptr: Integer;
	bufptr: Integer;
	m: Integer;
	u: rect;
	MBCharLen: Integer;
begin
	if bWindowClosing then exit;

	len := Length(Str);
	if (len = 0) then exit;

	u.X1 := cursorx;
	u.Y1 := cursory;
	u.X2 := cursorx;
	u.Y2 := cursory;
	strptr := 1;
	repeat
		xlen := clip.X2 - cursorx + 1;
		if (xlen > len) then xlen := len;

		bufptr := cursory * MaxCol + cursorx;
		xptr := 1;
		while xptr <= xlen do begin
			if (Length(Str) < strptr) then begin
				break;
			end;
			MBCharLen := LengthUTF8Char( Str[strptr] );
			if len < MBCharLen then begin
				{ If the last char is incompleted multibyte virtualscreencter, exit loop. }
				len := -1;
				break;
			end;
			if 1 < MBCharLen then begin
				if (xptr + 1) <= xlen then begin
					{ multibyte virtualscreencter }
					len := len - MBCharLen;
					for m := 0 to (MBCharLen - 1) do begin
						virtualscreen[bufptr*MaxMultibyteLen+m] := Str[strptr];
						Inc( strptr );
					end;
					fgcol[bufptr] := currfgc;
					bgcol[bufptr] := currbgc;
					Inc( cursorx );
					Inc( bufptr );
					Inc( xptr );
					virtualscreen[bufptr*MaxMultibyteLen] := #$0;
					fgcol[bufptr] := currfgc;
					bgcol[bufptr] := currbgc;
					Inc( cursorx );
					Inc( bufptr );
					Inc( xptr );
				end else begin
					{ first byte of multibyte char is placed at last of line }
					while cursorx <= clip.X2 do begin
						for m := 0 to (MBCharLen - 1) do begin
							virtualscreen[bufptr*MaxMultibyteLen+m] := #$0;
						end;
						fgcol[bufptr] := currfgc;
						bgcol[bufptr] := currbgc;
						Inc( cursorx );
						Inc( bufptr );
					end;
					cursorx := 9999;
					break;
				end;
			end else begin
				{ single byte virtualscreencter }
				Dec( len );
				for m := 1 to (MBCharLen - 1) do begin
					virtualscreen[bufptr*MaxMultibyteLen+m] := #$0;
				end;
				virtualscreen[bufptr*MaxMultibyteLen] := Str[strptr];
				Inc( strptr );
				fgcol[bufptr] := currfgc;
				bgcol[bufptr] := currbgc;
				Inc( cursorx );
				Inc( bufptr );
				Inc( xptr );
			end;
		end;
		if ( u.X2 < cursorx ) then u.X2 := cursorx;

		if (cursorx > clip.X2) then begin
			cursorx := clip.X1;
			cursory := cursory + 1;
			u.X2 := clip.X2;
			if (cursory > clip.Y2) then begin
				cursory := clip.Y2;
				{ utf8video.ScrollOneLine; }
				{ utf8video.ClrEol; }
				u.Y1 := clip.Y1;
				u.Y2 := clip.Y2;
			end;
		end;
		if ( cursorx < u.X1 ) then u.X1 := cursorx;
		u.Y2 := cursory;

	until len <= 0;

	{ utf8video.RedrawArea( u.X1, u.Y1, u.X2, u.Y2 ); }
end;

Procedure WriteChar( c: Char );
var
	m: Integer;
	bufptr: Integer;
begin
	if bWindowClosing then exit;

	bufptr := cursory * MaxCol + cursorx;
	{ if ((virtualscreen[bufptr*MaxMultibyteLen] <> c) or (fgcol[bufptr] <> currfgc) or (bgcol[bufptr] <> currbgc)) then begin }
		for m := 1 to (MaxMultibyteLen - 1) do begin
			virtualscreen[bufptr*MaxMultibyteLen+m] := #$0;
		end;
		virtualscreen[bufptr*MaxMultibyteLen] := c;
		fgcol[bufptr] := currfgc;
		bgcol[bufptr] := currbgc;
		{ utf8video.RedrawArea( cursorx, cursory, cursorx, cursory ); }
	{ end; }
	cursorx := cursorx + 1;
	if (cursorx > clip.X2) then begin
		cursorx := clip.X1;
		cursory := cursory + 1;
		if (cursory > clip.Y2) then begin
			cursory := clip.Y2;
			{ utf8video.ScrollOneLine; }
			{ utf8video.ClrEol; }
		end;
	end;
end;


Procedure WriteLn( const Str: String );
begin
	if (Str <> '') then utf8video.Write( Str );
	cursorx := clip.X1;
	Inc( cursory );
	if cursory > clip.Y2 then begin
		cursory := clip.Y2;
		utf8video.ScrollOneLine;
		utf8video.ClrEol;
	end;
end;

Procedure CrtInit( W,H: Integer );
begin
	if bWindowClosing then begin
		MaxCol := W;
		MaxRow := H;
		virtualscreen := AllocMem( MaxCol * MaxRow * MaxMultibyteLen );
		fgcol := AllocMem( MaxCol * MaxRow );
		bgcol := AllocMem( MaxCol * MaxRow );
		if (NIL = virtualscreen) or (NIL = fgcol) or (NIL = bgcol) then begin
			ErrorMessage('ERROR- utf8video. Out of memory.');
			halt(255);
		end;
		bWindowClosing := False;
	end;
end;

Procedure DisposeWindow;
begin
	if not bWindowClosing then begin
		bWindowClosing := true;
		FreeMem( bgcol );
		FreeMem( fgcol );
		FreeMem( virtualscreen );
		bgcol := NIL;
		fgcol := NIL;
		virtualscreen := NIL;
	end;
end;

{----------------------------------------------------------------------------------}
initialization
begin
{$IFDEF DEBUG}
	ErrorMessage_fork('DEBUG: utf8video.pp');
{$ENDIF DEBUG}
end;

finalization
begin
{$IFDEF DEBUG}
	ErrorMessage_fork('DEBUG: utf8video.pp(finalization)');
{$ENDIF DEBUG}
end;

end.
