unit w32crt;
{ Proposed and made by l0ugh, Internationalizationed by G-HAL. }
{ GearHead W32 non-console ASCII port }
{$LONGSTRINGS ON}

interface

const
{ Foreground and background color constants }
	Black         = 0;
	Blue          = 1;
	Green         = 2;
	Cyan          = 3;
	Red           = 4;
	Magenta       = 5;
	Brown         = 6;
	LightGray     = 7;

{ Foreground color constants }
	DarkGray      = 8;
	LightBlue     = 9;
	LightGreen    = 10;
	LightCyan     = 11;
	LightRed      = 12;
	LightMagenta  = 13;
	Yellow        = 14;
	White         = 15;

{ configuration file }
{ to avoid circular unit reference this defined here instead of gears.pp }
	WIN_CONFIG_FILE = 'win.cfg';

	keybufsize = 63;

Procedure RedrawScreen;
Procedure ClrEol;
Procedure ClrScr;
Procedure CursorOn;
Procedure CursorOff;
Procedure Delay( MS: Word );
Procedure GotoXY( X: Byte; Y: Byte );
Function KeyPressed: Boolean;
Function ReadKey: Char;
Function GetKeyEvent: Char;
Procedure TextBackground( CL: Byte );
Procedure TextColor( CL: Byte );
Function WhereX: Byte;
Function WhereY: Byte;
Procedure Window( AX1, AY1, AX2, AY2: Byte );
Function GetLine( it: String; var canceled: Boolean ): String;
Function GetLine( it: String ): String;
Function GetLine: String;
Procedure WriteLn( const Str: String );
Procedure Write( const Str: String );
Procedure WriteChar( c: Char );
Procedure CrtInit( W,H: Integer );
Procedure NormVideo;
Function DisposeWindow: LongInt;



implementation

uses windows, dos, strutils, texutil, errmsg, w32, ui4gh;

{ -------- color table -------- }
const
	MaxColorTable = 15;
var
	colortable: array[0..MaxColorTable] of Windows.COLORREF = (
		$00000000,	{RGB(   0,    0,    0),}	{ black }
		$00c00000,	{RGB(   0,    0, 0xc0),}	{ blue }
		$00008000,	{RGB(   0, 0x80,    0),}	{ green }
		$00808000,	{RGB(   0, 0x80, 0x80),}	{ cyan }
		$000000c0,	{RGB(0xc0,    0,    0),}	{ red }
		$00800080,	{RGB(0x80,    0, 0x80),}	{ magenta }
		$00006080,	{RGB(0x80, 0x60,    0),}	{ brown }
		$00c0c0c0,	{RGB(0xc0, 0xc0, 0xc0),}	{ lightgray }
		$00808080,	{RGB(0x80, 0x80, 0x80),}	{ darkgray }
		$00ff0000,	{RGB(   0,    0, 0xff),}	{ light blue }
		$0000ff00,	{RGB(   0, 0xff,    0),}	{ light green }
		$00ffff00,	{RGB(   0, 0xff, 0xff),}	{ light cyan }
		$000000ff,	{RGB(0xff,    0,    0),}	{ light red }
		$00ff00ff,	{RGB(0xff,    0, 0xff),}	{ light magenta }
		$0000ffff,	{RGB(0xff, 0xff,    0),}	{ yellow }
		$00ffffff	{RGB(0xff, 0xff, 0xff)}		{ white }
	);

{ -------- Virtual Console-VRAM -------- }
const
	MaxMultibyteLen = 8;
Type
	rect = Record
		X1, Y1, X2, Y2: Integer;
	end;
var
	MaxCol: Integer = 80;
	MaxRow: Integer = 25;
	virtualscreen: PChar = NIL;	{ Text Image }
	fgcol: PByte = NIL;	{ foreground color }
	bgcol: PByte = NIL;	{ background color }
	currfgc: Integer = MaxColorTable;
	currbgc: Integer = 0;

	{ cursorstate: Integer; }
	cursorx: Integer = 0;
	cursory: Integer = 0;

	clip: rect;	{ clipping }

	FONTH, FONTW: Integer;
	WIN_X, WIN_Y: LongInt;
	FontName: String;
	FontWeight: Integer;
	bWindowClosing: Boolean = True;

	myhwnd: Windows.HWND;
	backhdc: Windows.HDC;
	backbmp: Windows.HBITMAP;

{ -------- font -------- }
var
	myfont: Windows.HFONT;

Procedure PrepareFont;
var
	lf: Windows.LOGFONT;
begin
	Windows.ZeroMemory( @lf, sizeof(Windows.LOGFONT) );
	With lf do begin
		lfHeight := FONTH;
		lfOutPrecision := Windows.OUT_DEFAULT_PRECIS;
		lfClipPrecision := Windows.CLIP_DEFAULT_PRECIS;
		lfQuality := Windows.DEFAULT_QUALITY;
		lfWeight := FontWeight;
		lfPitchAndFamily := (Windows.FIXED_PITCH or Windows.FF_DONTCARE);
		lfFaceName := (FontName + #0);
		lfCharSet := Windows.DEFAULT_CHARSET;
	end;
	myfont := Windows.CreateFontIndirect( @lf );

	{if creating font failed, then create default font.}
	if (0 = myfont) then begin
		FONTH := 16;
		FONTW := 8;
		FontName := 'Terminal';
		With lf do begin
			lfHeight := FONTH;
			lfFaceName := (FontName + #0);
			lfWeight := 0;
		end;
		myfont := Windows.CreateFontIndirect( @lf );
		if (0 = myfont) then begin
			Windows.MessageBox(
				myhwnd,
				'failed to create font. program will be aborted.'#0,
				'Sorry'#0,
				Windows.MB_OK or Windows.MB_ICONERROR );
			w32crt.DisposeWindow;
		end;
	end;
end;

Procedure DestroyFont;
begin
	if not(Windows.DeleteObject(myfont)) then begin
		ErrorMessage( 'failed to delete font. ' );
	end;
	myfont := 0;
end;

{ -------- Key Buffering -------- }
var
	keybuf: array[0..keybufsize] of Char;
	keychar: array[0..keybufsize] of Boolean;
	keybufrp: Integer;
	keybufwp: Integer;

Procedure KeybufPut( c: Char; b: Boolean );
begin
	if (((keybufwp + 1) and keybufsize) <> keybufrp) then begin
		keybuf[keybufwp] := c;
		keychar[keybufwp] := b;
		keybufwp := ((keybufwp + 1) and keybufsize);
	end;
end;

Function KeybufEmpty: Boolean;
begin
	KeybufEmpty := (keybufrp = keybufwp);
end;

Function KeybufGet( var b: Boolean ): Char;
var
	c: Char;
begin
	if w32crt.KeybufEmpty then begin
		b := false;
		exit(KBD_NONE);
	end;
	c := keybuf[keybufrp];
	b := keychar[keybufrp];
	keybufrp := ((keybufrp + 1) and keybufsize);
	KeybufGet := c;
end;

{ -------- window proc -------- }
Procedure RedrawArea( ax1, ay1, ax2, ay2: Byte );
var
	mybrush: Windows.HBRUSH;
	r: Windows.RECT;
	x, y: Integer;
	bufptr: Integer;
	MBCharLen: Integer;
	WCharBuf: array[0..MaxMultibyteLen] of Byte;
	WCharBufLen: Integer;
begin
	if bWindowClosing then exit;
	if (0 = myfont) then exit;

{$IFDEF DEBUG}
	mybrush := Windows.CreateSolidBrush( colortable[currbgc] );
{$ELSE DEBUG}
	mybrush := Windows.CreateSolidBrush( colortable[0] );
{$ENDIF DEBUG}
	Windows.SetRect( @r, (ax1 * FONTW), (ay1 * FONTH), ((ax2 + 1) * FONTW), ((ay2 + 1) * FONTH) );
	Windows.FillRect( backhdc, r, mybrush );
	Windows.DeleteObject( mybrush );

	for y := ay1 to ay2 do begin
		bufptr := (y * MaxCol + ax1);
		x := ax1;
		while (x <= ax2) do begin
			if (#0 = virtualscreen[bufptr * MaxMultibyteLen]) then begin
			end else if (' ' = virtualscreen[bufptr * MaxMultibyteLen]) then begin
				mybrush := Windows.CreateSolidBrush( colortable[bgcol[bufptr]] );
				Windows.SetRect( @r, (x * FONTW), (y * FONTH), ((x + 1) * FONTW), ((y + 1) * FONTH) );
				Windows.FillRect( backhdc, r, mybrush );
				Windows.DeleteObject( mybrush );
			end else begin
				Windows.SetTextColor( backhdc, colortable[fgcol[bufptr]] );
				Windows.SetBkColor( backhdc, colortable[bgcol[bufptr]] );
				MBCharLen := LengthUTF8Char( virtualscreen[bufptr * MaxMultibyteLen] );
				if (1 < MBCharLen) then begin
					{ WCharBufLen := Windows.MultiByteToWideChar( Windows.CP_UTF8, Windows.MB_PRECOMPOSED, @virtualscreen[bufptr * MaxMultibyteLen], MBCharLen, @WCharBuf, MaxMultibyteLen ); }
					WCharBufLen := Windows.MultiByteToWideChar( Windows.CP_UTF8, 0, @virtualscreen[bufptr * MaxMultibyteLen], MBCharLen, @WCharBuf, MaxMultibyteLen );
					Windows.SetTextAlign( backhdc, Windows.VTA_CENTER or Windows.TA_TOP );
					Windows.TextOutW( backhdc, (x * FONTW + FONTW), (y * FONTH), @WCharBuf, WCharBufLen );
					bufptr := (bufptr + 1);
					x := (x + 1);
				end else begin
					Windows.SetTextAlign( backhdc, Windows.TA_CENTER or Windows.TA_TOP );
					Windows.TextOut( backhdc, (x * FONTW + FONTW div 2), (y * FONTH), @virtualscreen[bufptr * MaxMultibyteLen], 1 );
				end;
			end;
			inc( bufptr );
			inc( x );
		end;
	end;
end;


Function MyWndProc( myhwnd: Windows.HWND; msg: Windows.UINT; wp: Windows.WPARAM; lp: Windows.LPARAM ): Windows.LRESULT; StdCall;
const
	KB_STAT_LEN = 256;
var
	myhdc: Windows.HDC;
	ps: Windows.PAINTSTRUCT;
	c: Char;
	kb_stat: array[0..(KB_STAT_LEN - 1)] of Byte;
begin
	Case msg of

	    Windows.WM_CREATE:
		begin
			myhdc := Windows.GetDC( myhwnd );
			backhdc := Windows.CreateCompatibleDC( myhdc );
			backbmp := Windows.CreateCompatibleBitmap( myhdc, (FONTW * MaxCol), (FONTH * MaxRow) );
			Windows.SelectObject( backhdc, backbmp );
			Windows.ReleaseDC( myhwnd, myhdc );
			w32crt.PrepareFont;
			Windows.SelectObject( backhdc, myfont );
		end;

	    Windows.WM_PAINT:
		begin
			myhdc := Windows.BeginPaint( myhwnd, @ps );
			Windows.BitBlt( myhdc, 0,0, (MaxCol * FONTW), (MaxRow * FONTH), backhdc, 0, 0, Windows.SRCCOPY );
			Windows.EndPaint( myhwnd, @ps );
			exit(0);
		end;

	    Windows.WM_DESTROY:
		begin
			w32crt.DestroyFont;
			Windows.DeleteDC( backhdc );
			Windows.DeleteObject( backbmp );
			Windows.PostQuitMessage( 0 );
			exit(0);
		end;

	    Windows.WM_CLOSE:
		begin
			if not(bWindowClosing) then begin
				Windows.MessageBox( myhwnd, 'Please Quit from a game menu.'#0, 'Sorry'#0, Windows.MB_OK );
				exit(0);
			end;
		end;

	    Windows.WM_CHAR:
		begin
			c := Chr( wp );
			w32crt.KeybufPut( c, true );
			exit(0);
		end;

	    Windows.WM_KEYDOWN: {, Windows.WM_KEYUP:}
		begin
			{ get current keyboard status }
			Windows.ZeroMemory( @kb_stat[0], KB_STAT_LEN );
			Windows.GetKeyboardState( @kb_stat[0] );
			c := KBD_NONE;

			Case wp of
			    {Windows.VK_BACK:    c := KBD_BS;}
			    {Windows.VK_TAB:     c := KBD_TAB;}
			    Windows.VK_CLEAR:   c := KBD_KP5;
			    {Windows.VK_RETURN:  c := KBD_RET;}
			    {Windows.VK_ESCAPE:  c := KBD_ESC;}
			    Windows.VK_PRIOR:   c := KBD_PgUp;
			    Windows.VK_NEXT:    c := KBD_PgDn;
			    Windows.VK_END:     c := KBD_End;
			    Windows.VK_HOME:    c := KBD_Home;
			    Windows.VK_LEFT:    c := KBD_Left;
			    Windows.VK_UP:      c := KBD_Up;
			    Windows.VK_RIGHT:   c := KBD_Right;
			    Windows.VK_DOWN:    c := KBD_Down;
			    Windows.VK_INSERT:  c := KBD_Insert;
			    Windows.VK_DELETE:  c := KBD_Delete;
			    {Windows.VK_NUMPAD0:     c := KBD_Insert;}
			    {Windows.VK_NUMPAD1:     c := KBD_End;}
			    {Windows.VK_NUMPAD2:     c := KBD_Down;}
			    {Windows.VK_NUMPAD3:     c := KBD_PgDn;}
			    {Windows.VK_NUMPAD4:     c := KBD_Left;}
			    {Windows.VK_NUMPAD5:     c := KBD_KP5;}
			    {Windows.VK_NUMPAD6:     c := KBD_Right;}
			    {Windows.VK_NUMPAD7:     c := KBD_Home;}
			    {Windows.VK_NUMPAD8:     c := KBD_Up;}
			    {Windows.VK_NUMPAD9:     c := KBD_PgUp;}
			    {Windows.VK_MULTIPLY:    c := '*';}
			    {Windows.VK_ADD:         c := '+';}
			    Windows.VK_SEPARATOR:   c := ',';
			    {Windows.VK_SUBTRACT:    c := '-';}
			    Windows.VK_DECIMAL:     c := KBD_Delete;
			    {Windows.VK_DIVIDE:      c := '/';}
			    Windows.VK_F1:      c := KBD_F1;
			    Windows.VK_F2:      c := KBD_F2;
			    Windows.VK_F3:      c := KBD_F3;
			    Windows.VK_F4:      c := KBD_F4;
			    Windows.VK_F5:      c := KBD_F5;
			    Windows.VK_F6:      c := KBD_F6;
			    Windows.VK_F7:      c := KBD_F7;
			    Windows.VK_F8:      c := KBD_F8;
			    Windows.VK_F9:      c := KBD_F9;
			    Windows.VK_F10:     c := KBD_F10;
			    Windows.VK_F11:     c := KBD_F11;
			    Windows.VK_F12:     c := KBD_F12;
			    Windows.VK_F13:     c := KBD_F13;
			    Windows.VK_F14:     c := KBD_F14;
			    Windows.VK_F15:     c := KBD_F15;
			    Windows.VK_F16:     c := KBD_F16;
			    Windows.VK_F17:     c := KBD_F17;
			    Windows.VK_F18:     c := KBD_F18;
			    Windows.VK_F19:     c := KBD_F19;
			    Windows.VK_F20:     c := KBD_F20;
			    Windows.VK_F21:     c := KBD_NONE;
			    Windows.VK_F22:     c := KBD_NONE;
			    Windows.VK_F23:     c := KBD_NONE;
			    Windows.VK_F24:     c := KBD_NONE;
			end;

			if ((KBD_NONE <> c) and (0 <> (kb_stat[wp] and $80{KEY_PRESSED}))) then begin
				w32crt.KeybufPut( c, false );
				exit(0);
			end;

		end;{KEYDOWN,KEYUP}

	end;

	MyWndProc := Windows.DefWindowProc( myhwnd, msg, wp, lp );
end;

Procedure LoadWinCfg;
var
	F: Text;
	S, CMD, C: String;
	T: Integer;
	ColorNum: Integer;
begin

	WIN_X := -1;
	WIN_Y := -1;
	FONTH := MSWINGUI_FontSize;
	FONTW := (MSWINGUI_FontSize div 2 + 1);
	FontName   := MSWINGUI_FontName;
	FontWeight := MSWINGUI_FontWeight;

	S := FSearch( WIN_CONFIG_FILE, '.' );
	if ('' <> S) then begin
		Assign( F, S );
		Reset( F );

		while not(eof(F)) do begin
			ReadLn( F, S );
			if (Length(S) < 1) then continue;
			if ('%' = S[1]) then continue;
			cmd := UpCase( ExtractWord( S ) );
			if ('FONTHEIGHT' = cmd) then begin
				T := ExtractValue( S );
				if (0 < T) then begin
					FONTH := T;
					FONTW := (T div 2 + 1);
				end;
			end else if ('FONTNAME' = cmd) then begin
				C := '';
				while ('' <> S) do begin
					if ('' <> C) then C := (C + ' ');
					C := (C + ExtractWord( S ));
				end;
				FontName := C;
			end else if ('WINPOS_X' = cmd) then begin
				T := ExtractValue( S );
				if (0 < T) then begin
					WIN_X := T;
				end;
			end else if ('WINPOS_Y' = cmd) then begin
				T := ExtractValue( S );
				if (0 < T) then begin
					WIN_Y := T;
				end;
			end else if ('FONTWEIGHT' = cmd) then begin
				T := ExtractValue( S );
				if ((0 < T) and (T <= 900)) then begin
					FontWeight := T;
				end;
			end else if ('COLOR' = cmd) then begin
				ColorNum := ExtractValue( S );
				if (0 <= ColorNum) and (ColorNum <= MaxColorTable) then begin
					colortable[ColorNum] := (ExtractValue( S ) * $10000 + ExtractValue( S ) * $100 + ExtractValue( S ) * $1);
				end;
			end;
		end;
		Close( F );
	end;
end;

{$IFDEF 0}
Procedure SaveWinCfg;
var
	F: Text;
	r: Windows.RECT;
begin
	Windows.GetWindowRect( myhwnd, @r );
	WIN_X := r.left;
	WIN_Y := r.top;

	Assign( F, WIN_CONFIG_FILE );
	Rewrite( F );
	system.WriteLn( F, '% window configuration file' );
	system.WriteLn( F, '' );
	system.WriteLn( F, '% font name' );
	system.WriteLn( F, 'FontName ' + FontName );
	system.WriteLn( F, '% height of the font' );
	system.WriteLn( F, 'FontHeight ' + BSTR(FONTH) );
	system.WriteLn( F, '% weight of the font (default 0, bold = 700)' );
	system.WriteLn( F, 'FontWeight ' + BSTR(FontWeight) );
	system.WriteLn( F, '% x position of the window' );
	system.WriteLn( F, 'WinPos_X ' + BSTR(WIN_X) );
	system.WriteLn( F, '% y position of the window' );
	system.WriteLn( F, 'WinPos_Y ' + BSTR(WIN_Y) );
	Close( F );
end;
{$ENDIF 0}

{ -------- main -------- }
Procedure MyMain;
const
	mywndclassname = 'GHWindow'#0;
	mywndtitle = 'GearHead2'#0;
var
	wndclass: Windows.WNDCLASSEX;
	myhinst: Windows.HINST;
	r:  Windows.RECT;
	hDeskWnd: Windows.HWND;
	deskRect: Windows.RECT;
begin
	myhinst := Windows.GetModuleHandle( NIL );
	w32crt.LoadWinCfg;

	With wndclass do begin
		cbSize        := sizeof(wndclass);
		style         := (Windows.CS_HREDRAW or Windows.CS_VREDRAW);
		lpfnWndProc   := @w32crt.MyWndProc;
		cbClsExtra    := 0;
		cbWndExtra    := 0;
		hInstance     := myhinst;
		hIcon         := Windows.LoadIcon( 0, Windows.IDI_APPLICATION );
		hCursor       := Windows.LoadCursor( 0, Windows.IDC_ARROW );
		hbrBackground := Windows.GetStockObject( Windows.BLACK_BRUSH );
		lpszMenuName  := NIL;
		lpszClassName := mywndclassname;
		hIconSm       := Windows.LoadIcon( 0, Windows.IDI_APPLICATION );
	end;

	Windows.RegisterClassEx( @wndclass );

	myhwnd := Windows.CreateWindow(
			mywndclassname,			{ name of window class }
			mywndtitle,				{ title of the window }
			Windows.WS_OVERLAPPEDWINDOW and (not Windows.WS_THICKFRAME),
									{ style of the window }
			0, 0,					{ position of the window }
			10,{temp}				{ size of the window }
			10,{temp}
			0,						{ handle of the parent window }
			0,						{ handle of the menu }
			myhinst,				{ handle of the instance }
			NIL );					{ pointer to window arguments }

	With r do begin
		r.left   := 0;
		r.top    := 0;
		r.right  := (FONTW * MaxCol);
		r.bottom := (FONTH * MaxRow);
	end;
	Windows.AdjustWindowRect( @r, Windows.WS_OVERLAPPEDWINDOW and (not Windows.WS_THICKFRAME), FALSE{nomenu} );
	if ((-1 = WIN_X) and (-1 = WIN_Y)) then begin
		hDeskWnd := Windows.GetDesktopWindow;
		Windows.GetWindowRect( hDeskWnd, @deskRect );
		WIN_X := ((deskRect.right - (r.right - r.left) ) div 2);
		WIN_Y := ((deskRect.bottom - (r.bottom - r.top) ) div 2);
	end;
	Windows.SetWindowPos( myhwnd, 0, WIN_X, WIN_Y, (r.right - r.left), (r.bottom - r.top),
		Windows.SWP_NOACTIVATE or Windows.SWP_NOMOVE or Windows.SWP_NOZORDER {or Windows.SWP_NOREDRAW} );
	Windows.MoveWindow( myhwnd, WIN_X, WIN_Y, (r.right - r.left), (r.bottom - r.top), FALSE );
	Windows.ShowWindow( myhwnd, Windows.SW_SHOWDEFAULT );
	Windows.UpdateWindow( myhwnd );

end;

{----------------------------------------------------------------------------------}

Procedure RedrawScreen;
var
	myhdc: Windows.HDC;
begin
	if (0 = myhwnd) then exit;
	myhdc := Windows.GetDC( myhwnd );
	w32crt.RedrawArea( 0,0, (MaxCol - 1),(MaxRow - 1) );
	Windows.BitBlt( myhdc, 0,0, (MaxCol * FONTW), (MaxRow * FONTH), backhdc, 0, 0, Windows.SRCCOPY );
	Windows.ReleaseDC( myhwnd, myhdc );
end;

Procedure ClrEol;
var
	x: Integer;
	bufptr: Integer;
	m: Integer;
begin
	if bWindowClosing then exit;

	bufptr := (cursory * MaxCol + cursorx);
	for x := cursorx to clip.X2 do begin
		for m := 0 to (MaxMultibyteLen - 1) do begin
			virtualscreen[bufptr * MaxMultibyteLen + m] := #$0;
		end;
		fgcol[bufptr] := currfgc;
		bgcol[bufptr] := currbgc;
		bufptr := (bufptr + 1);
	end;
end;


Procedure ClrScr;
var
	x, y: Integer;
	bufptr: Integer;
	m: Integer;
begin
	if bWindowClosing then exit;

	for y := clip.Y1 to clip.Y2 do begin
		bufptr := (y * MaxCol + clip.X1);
		for x := clip.X1 to clip.X2 do begin
			for m := 0 to (MaxMultibyteLen - 1) do begin
				virtualscreen[bufptr * MaxMultibyteLen + m] := #$0;
			end;
			fgcol[bufptr] := currfgc;
			bgcol[bufptr] := currbgc;
			bufptr := (bufptr + 1);
		end;
	end;
end;

Procedure CursorOn;
var
	myhdc: Windows.HDC;
	mybrush: Windows.HBRUSH;
	r: Windows.RECT;
begin
	if (0 = myhwnd) then exit;
	myhdc := Windows.GetDC( myhwnd );

	mybrush := Windows.CreateSolidBrush( colortable[currfgc] );
	Windows.SetRect( @r, (cursorx * FONTW), (cursory * FONTH), ((cursorx + 1) * FONTW), ((cursory + 1) * FONTH) );
	Windows.FillRect( myhdc, r, mybrush );
	Windows.DeleteObject( mybrush );

	Windows.ReleaseDC( myhwnd, myhdc );
end;

Procedure CursorOff;
begin
end;

Procedure Delay(MS: Word);
begin
	Windows.Sleep( MS );
end;

Procedure GotoXY( X: Byte; Y: Byte );
begin
	cursorx := (X - 1 + clip.X1);
	cursory := (Y - 1 + clip.Y1);
	if (cursorx < clip.X1) then cursorx := clip.X1
	else if (clip.X2 < cursorx) then cursorx := clip.X2;
	if (cursory < clip.Y1) then cursory := clip.Y1
	else if (clip.Y2 < cursory) then cursory := clip.Y2;
end;

Function KeyPressed: Boolean;
var
	mymsg: Windows.MSG;
begin
	if w32crt.KeybufEmpty then begin
		if (Windows.PeekMessage(@mymsg, 0, Windows.WM_KEYFIRST, Windows.WM_KEYLAST, Windows.PM_REMOVE)) then begin
			Windows.TranslateMessage( @mymsg );
			Windows.DispatchMessage( @mymsg );
		end;
	end;
	KeyPressed := not(w32crt.KeybufEmpty);
end;

Function ReadKeyOrig( var b: Boolean ): Char;
var
	mymsg: Windows.MSG;
begin
	while w32crt.KeybufEmpty do begin
		if (Windows.PeekMessage(@mymsg, 0, 0, 0, Windows.PM_NOREMOVE)) then begin
			if not(Windows.GetMessage(@mymsg, 0, 0, 0)) then break;
			Windows.TranslateMessage( @mymsg );
			Windows.DispatchMessage( @mymsg );
		end
		else Windows.Sleep( 10 );
	end;
	ReadKeyOrig := w32crt.KeybufGet( b );
end;

Function ReadKey: Char;
var
	c: Char;
	b: Boolean;
begin
	c := w32crt.ReadKeyOrig( b );
	if ( b and (KBD_UNICODE_FLAG <= c) ) then exit(KBD_NONE);
	ReadKey := c;
end;

Function GetKeyEvent: Char;
var
	c: Char;
	b: Boolean;
begin
	c := w32crt.ReadKeyOrig( b );
	if ( b and (KBD_UNICODE_FLAG <= c) ) then exit(KBD_NONE);
	GetKeyEvent := c;
end;

Procedure TextBackground( CL: Byte );
begin
	currbgc := CL;
end;

Procedure TextColor( CL: Byte );
begin
	currfgc := CL;
end;

Function WhereX: Byte;
begin
	WhereX := (cursorx - clip.X1 + 1);
end;

Function WhereY: Byte;
begin
	WhereY := (cursory - clip.Y1 + 1);
end;

Procedure Window( AX1, AY1, AX2, AY2: Byte );
var
	tmp: Integer;
begin
	clip.X1 := (AX1 - 1);
	clip.Y1 := (AY1 - 1);
	clip.X2 := (AX2 - 1);
	clip.Y2 := (AY2 - 1);
	if (clip.X2 < clip.X1) then begin tmp := clip.X1; clip.X1 := clip.X2; clip.X2 := tmp; end;
	if (clip.Y2 < clip.Y1) then begin tmp := clip.Y1; clip.Y1 := clip.Y2; clip.Y2 := tmp; end;
	if (clip.X1 < 0) then clip.X1 := 0;
	if (clip.Y1 < 0) then clip.Y1 := 0;
	if ((MaxCol - 1) < clip.X2) then clip.X2 := (MaxCol - 1);
	if ((MaxRow - 1) < clip.Y2) then clip.Y2 := (MaxRow - 1);
	cursorx := clip.X1;
	cursory := clip.Y1;
end;

Procedure NormVideo;
begin
	currfgc := LightGray;
	currbgc := Black;
	w32crt.Window( 0,0, (MaxCol - 1),(MaxRow - 1) );
	w32crt.ClrScr;
end;

Procedure ScrollOneLine;
var
	x, y: Integer;
	bufptr, bufptr2: Integer;
	m: Integer;
begin
	if bWindowClosing then exit;

	if (clip.Y1 < clip.Y2) then begin
		for y := (clip.Y1 + 1) to clip.Y2 do begin
			bufptr := ((y - 1) * MaxCol + clip.X1);
			bufptr2 := (y * MaxCol + clip.X1);
			for x := clip.X1 to clip.X2 do begin
				for m := 0 to (MaxMultibyteLen - 1) do begin
					virtualscreen[bufptr * MaxMultibyteLen + m] := virtualscreen[bufptr2 * MaxMultibyteLen + m];
				end;
				fgcol[bufptr] := fgcol[bufptr2];
				bgcol[bufptr] := bgcol[bufptr2];
				inc( bufptr );
				inc( bufptr2 );
			end;
		end;
		{ w32crt.RedrawArea( clip.X1,clip.Y1, clip.X2,(clip.Y2 - 1) ); }
		w32crt.ClrEol;
	end;
end;

Procedure Write( const Str: String; const Redraw: Boolean );
var
	len, strptr, xlen, xptr: Integer;
	bufptr: Integer;
	m: Integer;
	u: rect;
	MBCharLen: Integer;
	myhdc: Windows.HDC;
begin
	if bWindowClosing then exit;

	len := Length(Str);
	if (0 = len) then exit;

	u.X1 := cursorx;
	u.Y1 := cursory;
	u.X2 := cursorx;
	u.Y2 := cursory;
	strptr := 1;
	repeat
		xlen := (clip.X2 - cursorx + 1);
		if (len < xlen) then xlen := len;

		bufptr := (cursory * MaxCol + cursorx);
		xptr := 1;
		while (xptr <= xlen) do begin
			if (Length(Str) < strptr) then begin
				break;
			end;
			MBCharLen := LengthUTF8Char( Str[strptr] );
			if (len < MBCharLen) then begin
				{ If the last char is incompleted multibyte character, exit loop. }
				len := -1;
				break;
			end;
			if (1 < MBCharLen) then begin
				if ((xptr + 1) <= xlen) then begin
					{ multibyte character }
					len := (len - MBCharLen);
					for m := 0 to (MBCharLen - 1) do begin
						virtualscreen[bufptr * MaxMultibyteLen + m] := Str[strptr];
						inc( strptr );
					end;
					fgcol[bufptr] := currfgc;
					bgcol[bufptr] := currbgc;
					inc( cursorx );
					inc( bufptr );
					inc( xptr );
					virtualscreen[bufptr * MaxMultibyteLen] := #$0;
					fgcol[bufptr] := currfgc;
					bgcol[bufptr] := currbgc;
					inc( cursorx );
					inc( bufptr );
					inc( xptr );
				end else begin
					{ first byte of multibyte char is placed at last of line }
					while (cursorx <= clip.X2) do begin
						for m := 0 to (MBCharLen - 1) do begin
							virtualscreen[bufptr * MaxMultibyteLen + m] := #$0;
						end;
						fgcol[bufptr] := currfgc;
						bgcol[bufptr] := currbgc;
						inc( cursorx );
						inc( bufptr );
					end;
					cursorx := 9999;
					break;
				end;
			end else begin
				{ single byte character }
				dec( len );
				for m := 0 to (MBCharLen - 1) do begin
					virtualscreen[bufptr * MaxMultibyteLen + m] := #$0;
				end;
				virtualscreen[bufptr * MaxMultibyteLen] := Str[strptr];
				inc( strptr );
				fgcol[bufptr] := currfgc;
				bgcol[bufptr] := currbgc;
				inc( cursorx );
				inc( bufptr );
				inc( xptr );
			end;
		end;
		if (u.X2 < cursorx) then u.X2 := cursorx;

		if (clip.X2 < cursorx) then begin
			cursorx := clip.X1;
			cursory := (cursory + 1);
			u.X2 := clip.X2;
			if (clip.Y2 < cursory) then begin
				cursory := clip.Y2;
				{ w32crt.ScrollOneLine; }
				{ w32crt.ClrEol; }
				u.Y1 := clip.Y1;
				u.Y2 := clip.Y2;
			end;
		end;
		if (cursorx < u.X1) then u.X1 := cursorx;
		u.Y2 := cursory;

	until (len <= 0);

	if Redraw then begin
		{ w32crt.RedrawArea( u.X1, u.Y1, u.X2, u.Y2 ); }
		if (0 <> myhwnd) then begin
			myhdc := Windows.GetDC( myhwnd );
			w32crt.RedrawArea( u.X1, u.Y1, u.X2, u.Y2 );
			Windows.BitBlt( myhdc, (u.X1 * FONTW), (u.Y1 * FONTH), (u.X2 * FONTW), (u.Y2 * FONTH), backhdc, (u.X1 * FONTW), (u.Y1 * FONTH), Windows.SRCCOPY );
			Windows.ReleaseDC( myhwnd, myhdc );
		end;
	end;
end;

Procedure Write( const Str: String );
begin
	w32crt.Write( Str, False );
end;

Procedure WriteChar( c: Char );
var
	m: Integer;
	bufptr: Integer;
begin
	if bWindowClosing then exit;

	bufptr := (cursory * MaxCol + cursorx);
	{ if ((virtualscreen[bufptr * MaxMultibyteLen] <> c) or (fgcol[bufptr] <> currfgc) or (bgcol[bufptr] <> currbgc)) then begin }
		for m := 0 to (MaxMultibyteLen - 1) do begin
			virtualscreen[bufptr * MaxMultibyteLen + m] := #$0;
		end;
		virtualscreen[bufptr * MaxMultibyteLen] := c;
		fgcol[bufptr] := currfgc;
		bgcol[bufptr] := currbgc;
		{ w32crt.RedrawArea( cursorx, cursory, cursorx, cursory ); }
	{ end; }
	cursorx := (cursorx + 1);
	if (clip.X2 < cursorx) then begin
		cursorx := clip.X1;
		cursory := (cursory + 1);
		if (clip.Y2 < cursory) then begin
			cursory := clip.Y2;
			{ w32crt.ScrollOneLine; }
			{ w32crt.ClrEol; }
		end;
	end;
end;


Procedure WriteLn( const Str: String );
begin
	if ('' <> Str) then w32crt.Write( Str );
	cursorx := clip.X1;
	inc( cursory );
	if (clip.Y2 < cursory) then begin
		cursory := clip.Y2;
		w32crt.ScrollOneLine;
		w32crt.ClrEol;
	end;
end;

Function GetLine: String;
var
	dummy: Boolean;
begin
	GetLine := w32crt.GetLine('', dummy);
end;

Function GetLine( it: String ): String;
var
	dummy: Boolean;
begin
	GetLine := w32crt.GetLine(it, dummy);
end;

Function GetLine( it: String; var canceled: Boolean ): String;
const
{$IFDEF IBMGraphics}
	CARETCHR = #$16;
{$ELSE}
	CARETCHR = '_';
{$ENDIF}
var
	buf: String;
	c: Char;
	b: Boolean;
	x, y: Integer;
	len, mxl: Integer;
	clen: Integer;
	i: Integer;
	tmp: String;
	j: Integer;
	tlen: Integer;
begin
	canceled := False;

	buf := it;
	len := Length(buf);

	x := w32crt.WhereX;
	y := w32crt.WhereY;
	mxl := ((clip.X2 - clip.X1 + 1) - (x - 1) - 3);

	w32crt.Write( (buf + CARETCHR + '  '), True );
	w32crt.ClrEol;
	repeat
		c := w32crt.ReadKeyOrig( b );
		if (KBD_BS = c) then begin
			{ Backspace }
			clen := Length(TailUTF8Char(buf));
			buf := Copy( buf, 1, (Length(buf) - clen) );
			len := (len - clen);
		end else if (KBD_CTRL_U = c) then begin
			{ Clear Line }
			buf := DupeString( ' ' , len );
			w32crt.GotoXY( x, y );
			w32crt.Write( (buf + '   '), True );
			w32crt.ClrEol;
			buf := '';
			len := 0;
		end else if (KBD_ESC = c) then begin
			{ Esc - cancelled }
			buf := '';
			c := KBD_RET; {exit loop}
			canceled := True;
		end else if b and TextISO646_AllowableCheck(c) then begin
			if (len < mxl) then begin
				buf := (buf + c);
				inc( len );
			end;
		end else if b then begin
			tmp := c;
			while w32crt.KeyPressed do begin
				c := w32crt.ReadKeyOrig( b );
				tmp := tmp + c;
			end;
			tmp := AnsiToUTF8( tmp );
			tlen := Length(tmp);
			j := 0;
			while (j < tlen) do begin
				inc( j );
				c := tmp[j];
				if IsUTF8CharLeadByte(c) then begin
					clen := LengthUTF8Char( c );
					if (clen <= 1) then begin
					end else begin
						if ((len + clen) < mxl) then begin
							buf := (buf + c);
							i := (clen - 1);
							while (0 < i) do begin
								inc( j );
								c := tmp[j];
								buf := (buf + c);
								dec( i );
							end;
							len := (len + clen);
						end else begin
							i := (clen - 1);
							while (0 < i) do begin
								inc( j );
								dec( i );
							end;
						end;
					end;
				end;
			end;
		end;
		w32crt.GotoXY( x, y );
		w32crt.Write( (buf + CARETCHR + '  '), True );
		w32crt.ClrEol;
	until ((KBD_RET = c) or (KBD_LF = c));

	GetLine := buf;
end;

Procedure CrtInit( W,H: Integer );
begin
	if bWindowClosing then begin
		MaxCol := W;
		MaxRow := H;
		virtualscreen := AllocMem( MaxCol * MaxRow * MaxMultibyteLen );
		fgcol := AllocMem( MaxCol * MaxRow );
		bgcol := AllocMem( MaxCol * MaxRow );
		if ((NIL = virtualscreen) or (NIL = fgcol) or (NIL = bgcol)) then begin
			ErrorMessage('ERROR- w32crt. Out of memory.');
			halt(255);
		end;
		bWindowClosing := False;
	end;
	w32crt.MyMain;
	w32crt.NormVideo;
end;

Function DisposeWindow: LongInt;
var
	mymsg: Windows.MSG;
begin
	if not(bWindowClosing) then begin
		{ w32crt.SaveWinCfg; }
		bWindowClosing := true;
		Windows.SendMessage( myhwnd, Windows.WM_CLOSE, 0, 0 );
		{ while (0 < Windows.GetMessage(@mymsg, 0, 0, 0)) do begin}
		while Windows.GetMessage( @mymsg, 0, 0, 0 ) do begin
			{ no dispatch. }
		end;
		FreeMem( bgcol );
		FreeMem( fgcol );
		FreeMem( virtualscreen );
		bgcol := NIL;
		fgcol := NIL;
		virtualscreen := NIL;
		DisposeWindow := mymsg.wParam;
	end else begin
		DisposeWindow := 0;
	end;
end;

{----------------------------------------------------------------------------------}
initialization
begin
{$IFDEF DEBUG}
	ErrorMessage_fork('DEBUG: w32crt.pp');
{$ENDIF DEBUG}
end;

finalization
begin
{$IFDEF DEBUG}
	ErrorMessage_fork('DEBUG: w32crt.pp(finalization)');
{$ENDIF DEBUG}
end;

end.
