#!/usr/bin/env perl

#:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
# This file is part of G-language Genome Analysis Environment package
#
#     Copyright (C) 2001-2007 Keio University
#:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
# 
#   $Id: Primitive.pm,v 1.1 2002/07/30 17:44:27 gaou Exp $
#
# G-language GAE is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
# 
# G-language GAE is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
# See the GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public
# License along with G-language GAE -- see the file COPYING.
# If not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
# 
#END_HEADER
#

package G::Seq::Primitive;

use strict;
use G::Messenger;
use SubOpt;

use vars qw($VERSION @ISA @EXPORT);

use SelfLoader;

require Exporter;

@ISA = qw(Exporter);

@EXPORT = qw(
	     complement
	     translate
	     splitprintseq
	     to_fasta
	     DoubleHelix
);


#::::::::::::::::::::::::::::::
#          Constants
#::::::::::::::::::::::::::::::

my %CodonTable = (
	       'gac', 'D', 'caa', 'Q', 'gca', 'A', 'ctg', 'L',
	       'gat', 'D', 'cag', 'Q', 'gcc', 'A', 'ctt', 'L',
	       'gaa', 'E', 'agc', 'S', 'gcg', 'A', 'ata', 'I',
	       'gag', 'E', 'agt', 'S', 'gct', 'A', 'atc', 'I',
	       'aga', 'R', 'tca', 'S', 'gga', 'G', 'att', 'I',
	       'agg', 'R', 'tcc', 'S', 'ggc', 'G', 'cca', 'P',
	       'cga', 'R', 'tcg', 'S', 'ggg', 'G', 'ccc', 'P',
	       'cgc', 'R', 'tct', 'S', 'ggt', 'G', 'ccg', 'P',
	       'cgg', 'R', 'aca', 'T', 'gta', 'V', 'cct', 'P',
	       'cgt', 'R', 'acc', 'T', 'gtc', 'V', 'atg', 'M',
	       'aaa', 'K', 'acg', 'T', 'gtg', 'V', 'tgg', 'W',
	       'aag', 'K', 'act', 'T', 'gtt', 'V', 'tgc', 'C',
	       'cac', 'H', 'tac', 'Y', 'tta', 'L', 'tgt', 'C',
	       'cat', 'H', 'tat', 'Y', 'ttg', 'L', 'taa', '/',
	       'aac', 'N', 'ttc', 'F', 'cta', 'L', 'tag', '/',
	       'aat', 'N', 'ttt', 'F', 'ctc', 'L', 'tga', '/'
		  );

sub translate {
    my $seq = lc(shift);
    my $amino = '';

    while(3 <= length($seq)){
	my $codon = substr($seq, 0, 3);
	substr($seq, 0, 3) = '';
	if ($codon =~ /[^atgc]/){
	    $amino .= 'X';
	}else{
	    $amino .= $CodonTable{$codon};
	}
    }

    msg_error("Translation: illegal length.\n") if(length($seq));

    return $amino;
}


__DATA__

#::::::::::::::::::::::::::::::
#        Methods Start
#::::::::::::::::::::::::::::::

sub complement {
    my $nuc = reverse(shift);

    $nuc =~ tr
	[acgturymkdhbvwsnACGTURYMKDHBVWSN]
	[tgcaayrkmhdvbwsnTGCAAYRKMHDVBWSN];

    return $nuc;
}


sub splitprintseq {
    my $seq = shift;
    my $len = shift || 60;
    my $ret = '';

    while(length $seq){
	$ret .= substr($seq, 0, $len) . "\n";
	substr($seq, 0, $len) = '';
    }
    
    return $ret;
}


sub to_fasta{
    SubOpt::opt_default(length=>60, filename=>"out.fasta", name=>"sequence");
    my @args = SubOpt::opt_get(@_);
    my $gb = SubOpt::opt_as_gb(shift @args);
    my $filename = SubOpt::opt_val("filename");
    my $name = SubOpt::opt_val("name");
    my $length = SubOpt::opt_val("length");

    $name = $gb->{LOCUS}->{id} if($name eq 'sequence' && length $gb->{LOCUS}->{id});

    open(OUT, ">$filename") || die($!);
    printf OUT ">%s\n%s", $name, splitprintseq($gb->{SEQ}, $length);
    close(OUT);

    return $filename;
}

sub DoubleHelix{
    SubOpt::opt_default(speed=>0.05);
    my @args = SubOpt::opt_get(@_);  
    my $gb = SubOpt::opt_as_gb(shift @args);
    my $speed = SubOpt::opt_val("speed");

    $| = 1;

    my $i;
    my (@offset) = qw/1 0 0 0 1 2 3 4 5 5 4 3 2 1 0 0 0 1/;
    my (@dist)   = qw/0 2 3 4 4 4 3 2 0 0 2 3 4 4 4 3 2 0/;

    foreach my $base (split(//, $gb->{SEQ})){
        print "         ", q/ /x$offset[($i%scalar(@offset))];
	print uc($base);
	$i ++;
	print q/-/x$dist[($i%scalar(@dist))];
	print uc(complement($base)), "\n";
	select(undef,undef,undef,$speed);
    }
}




1;



__END__


=head1 NAME

G::Seq::Eliminate - Perl extension for blah blah blah

=head1 SYNOPSIS

  use G::Seq::Primitive;
  blah blah blah

=head1 DESCRIPTION

Stub documentation for G::Seq::Eliminate was created by h2xs. It looks like the
author of the extension was negligent enough to leave the stub
unedited.

Blah blah blah.

=head1 AUTHOR

A. U. Thor, a.u.thor@a.galaxy.far.far.away

=head1 SEE ALSO

perl(1).

=cut
