#!/usr/bin/env perl

#:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
# This file is part of G-language Genome Analysis Environment package
#
#     Copyright (C) 2001-2007 Keio University
#:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
# 
# G-language GAE is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
# 
# G-language GAE is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
# See the GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public
# License along with G-language GAE -- see the file COPYING.
# If not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
# 
#END_HEADER
#
# written by Kazuharu Arakawa <gaou@sfc.keio.ac.jp> at
# G-language Project, Institute for Advanced Biosciences, Keio University.
#

package G::Messenger;

use strict;
use Cwd;
use Data::Dumper;

if ('MSWin32' eq $^O){
    eval(require Win32::InternetExplorer::Window);
    eval(require Image::Size);
}

require Exporter;

our @ISA = qw(Exporter);
our @EXPORT = qw(
	     msg_interface
	     msg_ask_interface
	     msg_send
	     msg_error
	     msg_system_console
	     msg_percent
	     msg_progress
	     msg_term_console
	     msg_gimv
	     msg_set_gimv
	     say
	     p
	     puts
);

#::::::::::::::::::::::::::::::
#          Variables
#::::::::::::::::::::::::::::::

my %data;

$data{"interface"} = 'STDOUT';
my $console;
my $progress;
my $term;
my $gimv = 'gimv';

#::::::::::::::::::::::::::::::
#           Methods
#::::::::::::::::::::::::::::::

sub p{
    print Dumper(@_), "\n";
}

sub puts{
    print @_, "\n";
}

sub say{
    print join(',', @_), "\n";
}

sub msg_gimv{
    my $file = shift;
    my $time = 'gimv' . time();

    if ($data{"interface"} eq 'Inspire'){
	&$gimv($file);
    }else{
	if('MSWin32' eq $^O){
	    my $dir = getcwd() unless (substr($file, 0, 1) eq '/');
	    my $filename = $dir . '/' . $file;
	    if ($filename =~ /[htm|html]$/){
		$data{$time} = Win32::InternetExplorer::Window->new();
	    }else{
		my ($x, $y) = Image::Size::imgsize($filename);
		$data{$time} = Win32::InternetExplorer::Window->new(
								    height=>$y+60,
								    width=>$x+50
								    );
	    }
	    $data{$time}->display('\G-language\title.jpg');
	    $data{$time}->display_wait($filename);
	}else{
	    $gimv = 'mozilla' if ($file =~ /html$/ && $gimv eq 'gimv');
	    $gimv = 'mozilla' if ($file =~ /svg$/ && $gimv eq 'gimv');
	    $file = 'file://' . getcwd() . '/' . $file if ($gimv eq 'mozilla');
	    system("$gimv $file &");
	}
    }
}

sub msg_term_console{
    $term = shift;
}

sub msg_system_console{
    $console = shift;
}

sub msg_progress{
    $progress = shift;
}

sub msg_set_gimv{
    $gimv = shift;
}

sub msg_percent {
    my $percent = shift;

    if ($data{"interface"} eq 'GUI'){
	$progress->update($percent);
    }elsif ($data{"interface"} eq 'Wx'){
	$progress->SetValue($percent);
        $progress->Refresh();
    }
}

sub msg_interface{
    $data{"interface"} = shift;
}

sub msg_ask_interface{
    return $data{"interface"};
}

sub msg_send{
    my @msg = @_;

    if ($data{"interface"} eq 'GUI'){
	$data{"term"} .= join('',@msg);
	$term->insert('','','',join('',@msg));
    }elsif ($data{"interface"} eq 'Wx'){
	my $tmp = join('', @msg);
	$data{"term"} .= $tmp;
	$term->WriteText(scalar($tmp));
    }elsif ($data{'interface'} eq 'Inspire'){
	my $tmp = join('', @msg);
	$data{"term"} .= $tmp;
	&$term($tmp);
    }else{
	print STDOUT @msg;
    }
}

sub msg_error{
    my @msg = @_;

    if ($data{"interface"} eq 'GUI'){
	$data{"system"} .= join('', @msg);
	$console->insert('','','',join('', @msg));
    }elsif ($data{"interface"} eq 'Wx'){
	my $tmp = join('', @msg);
	$data{"system"} .= $tmp;
	$console->WriteText(scalar($tmp));
    }elsif ($data{'interface'} eq 'Inspire'){
	my $tmp = join('', @msg);
	$data{"system"} .= $tmp;
	&$console($tmp);
    }else{
	print STDERR @msg;
    }
}


1;
__END__


=head1 NAME

  G::Messenger.pm G-language GAE Messaging API

=head1 INTRODUCTION

  One of the key features of the G-language Genome Analysis Environment is the 
  ease of use and development of different user interfaces. With very little 
  modification, a script intended for commandline Perl use can be transformed
  into a cross-platform GUI application using wxWindows, or a web application 
  using INSPIRE. The Messaging API G::Messenger.pm is what makes this possible.
  All output in the software system is passed to the interface provided by this
  API, and Messenger interprets the current user interface to give correct form
  of output. It is essential to create Odyssey Subroutines using this API, and 
  to create new user interfaces.

=head1 SYNOPSIS

  use G::Messenger;

  &msg_send("this is an output"); #prints message to STDOUT
  &msg_error("this is an error"); #prints message to STDERR
  &msg_gimv("graph.png");         #shows the image via gimv (G-language IMage Viewer)

=head1 DESCRIPTION

  Methods used in Odyssey:

    msg_send(<message, can be an array of messages>)
        Prints out a regular output message. The output is STDOUT in CUI mode,
        and text output in Gtk+, wxWindows, and INSPIRE modes.
        Synonym of "print".

    msg_error(<message, can be an array of messages>)
	Prints out an error or system message. The output is STDERR in CUI mode,
        and system console in Gtk+, wxWindows, and INSPIRE modes.
        Synonym of "print STDOUT".

    msg_gimv(<image file name>)
        Shows the image via gimv (G-language IMage Viewer). The gimv program will
        startup in Gtk+ and wxWindows mode, and only the file path is passed on
        in INSPIRE mode.
        Synonym of "system('gimv image.png')".

  Methods used in Engines:

    msg_percent(<decimal>)
        Sets the progress bar to the given percentage.

  Methods used in User Interfaces (INSPIRE):

    msg_interface(<interface>)
        Sets the interface that the system currently uses. 'GUI' for Gtk+ mode,
        'Wx' for wxWindows mode, 'Inspire' for INSPIRE mode, and none or 'STDOUT'
        for CUI mode. All Messenger output bases the interpretation of the current
        user interface on the value set with this function.

    msg_system_console(<reference of system console>)
        In Gtk+ and wxWindows mode, system console widget instance should be supplied.
        In INSPIRE mode, the reference to a subroutine that interpret the job of system
        console should be supplied. Messages sent with msg_error() are passed to the 
        given reference.

    msg_term_console(<reference of term console>)
        In Gtk+ and wxWindows mode, term console widget instance should be supplied.
        In INSPIRE mode, the reference to a subroutine that interpret the job of term
        console should be supplied. Messages sent with msg_send() are passed to the 
        given reference.

    msg_progress(<reference of progress bar>)
        In Gtk+ and wxWindows mode, progress bar widget instance should be supplied.
        In INSPIRE mode, the reference to a subroutine that interpret the job of progress
        bar may be supplied. Messages sent with msg_percent() are passed to the 
        given reference.

    msg_set_gimv(<reference of gimv subroutine>)
        In INSPIRE mode, the reference to a subroutine that interpret the job of gimv
        should be supplied. Messages sent with msg_gimv() are passed to the given 
        reference.

    msg_ask_interface()
        Returns the current mode. 


=head1 AUTHOR

  Kazuharu Arakawa, gaou@sfc.keio.ac.jp

=head1 SEE ALSO

perl(1).

=cut



