#!/usr/bin/env perl

#:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
# This file is part of G-language Genome Analysis Environment package
#
#     Copyright (C) 2001-2007 Keio University
#:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
# 
#   $Id: Util.pm,v 1.1.1.1 2002/04/02 20:25:43 gaou Exp $
#
# G-language GAE is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
# 
# G-language GAE is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
# See the GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public
# License along with G-language GAE -- see the file COPYING.
# If not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
# 
#END_HEADER
#
# written by Kazuharu Arakawa <gaou@sfc.keio.ac.jp> at
# G-language Project, Institute for Advanced Biosciences, Keio University.
#

package G::Seq::GenomeMap;

use SubOpt;
use G::Messenger;
use G::Tools::GPAC;
use G::Tools::GMap;
use G::DB::SDB;

use strict;
use Cwd;
use GD;
use SVG;
use SelfLoader;


require Exporter;

our @ISA = qw(Exporter);

our @EXPORT = qw(
		 genome_map
		 genome_map2
		 genome_map3	 
		 plasmid_map
		 seq2png
		 );


#:::::::::::::::::::::::::::::::::
#       Perldoc
#:::::::::::::::::::::::::::::::::


=head1 NAME

  G::Seq::GenomeMap - Collection of methods for creating genome maps.

=head1 DESCRIPTION

    This class is a part of G-language Genome Analysis Environment, 
    collecting methods for creating genome maps.

=cut

#::::::::::::::::::::::::::::::
#        Constants
#::::::::::::::::::::::::::::::

my %COG_fcode = (
		 J=>"Translation, ribosomal structure and biogenesis",
		 K=>"Transcription",
		 L=>"DNA replication, recombination and repair",
		 D=>"Cell division and chromosome partitioning",
		 O=>"Posttranslational modification, protein turnover, chaperones",
		 M=>"Cell envelope biogenesis, outer membrane",
		 N=>"Cell motility and secretion",
		 P=>"Inorganic ion transport and metabolism",
		 T=>"Signal transduction mechanisms",
		 C=>"Energy production and conservation",
		 G=>"Carbohydrate transport and metabolism",
		 E=>"Amino acid transport and metabolism",
		 F=>"Nucleotide transport and metabolism",
		 H=>"Coenzyme metabolism",
		 I=>"Lipid metabolism",
		 Q=>"Secondary metabolites biosynthesis, transport and catabolism",
		 R=>"General function prediction only",
		 S=>"Function unknown",
		 '-'=>"Non COG"
		 );

my %COG_fcolor = (
		  J=>"plum",
		  K=>"fuchsia",
		  L=>"pink",
		  D=>"lightgreen",
		  O=>"green",
		  M=>"khaki",
		  N=>"greenyellow",
		  P=>"darkkhaki",
		  T=>"cyan",
		  C=>"blue",
		  G=>"mediumturquoise",
		  E=>"lightskyblue",
		  F=>"mediumpurple",
		  H=>"aqua",
		  I=>"blueviolet",
		  Q=>"lightskyblue",
		  R=>"gainsboro",
		  S=>"darkgrey",
		  '-'=>"aliceblue"
		  );

__DATA__

#::::::::::::::::::::::::::::::
#        Methods Start
#::::::::::::::::::::::::::::::

=head2 genome_map

  Name: genome_map   -   draws the map of the genome

  Description:
    This method creates a map of the genome, showing the local 
    nucleotide contents and positions of genes.
    A is shown in red, T is shown in green, 
    G is shown in yellow, and C is shown in blue. 

    This method generates the entire genome map using 
    multiple 800x600 pixel image in PNG (via GD library).

  Usage:
    NULL = genome_map($gb);

 Options:
    -name      print gene name (default: 1)
    -window    number of nucleotides represented by one pixel (default: 50)
    -amp       height of the nucleotide content graph (default: 1.5)
    -output    "g" for graph, "show" for display (default: "show") 

  Author: 
    Kazuharu Arakawa (gaou@sfc.keio.ac.jp)

  History:
    20010906-01 initial posting

=cut

sub genome_map {
    &opt_default(output=>"show", name=>1, window=>50, amp=>1.5);
    my @args = opt_get(@_);
    my $gb = opt_as_gb(shift @args);
    my $acnum = $gb->{LOCUS}->{id};
    my $output = opt_val("output");
    my $name = opt_val("name");
    my $filename;
    my $topmargin = 30;
    my $sidemargin = 80;
    my $hblock = 100;
    my $vblock = 10;
    my $page = 1;
    my $start;
    my $width = 800;
    my $height = 600;
    my $i = 0;
    my $cds = 1;
    my $window=opt_val("window");
    my $amp = opt_val("amp");

    mkdir ("graph", 0777);
    for ($start = 1; $start <= length($gb->{SEQ}); $start += $window * 700 * 10){ 
	my $end = $start + 10 * $window * 700 - 1;
	
	# GD constant
	my $im = new GD::Image($width, $height);
	my $white = $im->colorAllocate(255,255,255);
	my $black = $im->colorAllocate(0,0,0);
	my $gray = $im->colorAllocate(180,180,180);  
	my $red = $im->colorAllocate(255,0,0);       #A
	my $yellow = $im->colorAllocate(255,255,0);  #G
	my $green = $im->colorAllocate(0,150,0);     #T
	my $blue = $im->colorAllocate(0,0,255);      #C
	my $aqua = $im->colorAllocate(120, 160, 255);

	my $gred = $im->colorAllocate(255,150,150);       #A for graph
	my $gyellow = $im->colorAllocate(255,255,50);     #G for graph
	my $ggreen = $im->colorAllocate(150,150,150);     #T for graph
	my $gblue = $im->colorAllocate(150,150,255);      #C for graph


	# Draw Base Graph
	for ($i = $sidemargin; $i <= $sidemargin + $hblock * 7; $i += $hblock){
	    $im->line($i, $topmargin, $i, $topmargin + 5 * 11 * $vblock, 
		      $gray);
	}
	for ($i = $topmargin; $i <= $topmargin + $vblock * 5 * 11; $i += 
	     $vblock){
	    $im->line($sidemargin, $i, $sidemargin + 7 * $hblock, $i, $gray);
	}
	for ($i = $topmargin + $vblock * 5; 
	     $i < $topmargin + $vblock * 5 * 11; 
	     $i += $vblock * 5){
	    $im->line($sidemargin, $i - 1, $sidemargin + 7 * $hblock, $i - 1, 
		      $black);
	    $im->line($sidemargin, $i + 1, $sidemargin + 7 * $hblock, $i + 1, 
		      $black);
	}
	$im->string(gdSmallFont, $width - 110, 5, 
		    "G-language Project", $black);
	$im->string(gdSmallFont, $width - 110 - 50, 5, "A", $red);
	$im->string(gdSmallFont, $width - 110 - 40, 5, "T", $green);
	$im->string(gdSmallFont, $width - 110 - 30, 5, "G", $yellow);
	$im->string(gdSmallFont, $width - 110 - 20, 5, "C", $blue);

	my $j = 0;
	for ($i = $topmargin + $vblock * 5; 
	     $i <= $topmargin + $vblock * 5 * 10; 
	     $i += $vblock * 5){
	    my $num = $start + $j * $window * 700;
	    $im->string(gdTinyFont, 10, $i, "$num", $black);
	    $j ++;
	}
	$im->string(gdSmallFont, 5,  5, "$acnum : from $start to $end", 
		    $black);

	my ($pa, $pt, $pg, $pc, $num, $color);
	my $locus = 0;
	for ($i = $start - 1; $i <= $start + 700 * 10 * $window - 1; $i += $window){
	    last if ($i + $window >= length($gb->{SEQ}));
	    my $seq = $gb->getseq($i, $i + $window - 1);
	    my $a = $seq =~ tr/a/a/;
	    my $t = $seq =~ tr/t/t/;
	    my $g = $seq =~ tr/g/g/;
	    my $c = $seq =~ tr/c/c/;

	    # Draw DNA
	    if ($a >= $t && $a >= $g && $a >= $c){
		my $num = int($a / $window * 100);
		$color = $red;
	    }elsif ($t >= $a && $t >= $g && $t >= $c){
		my $num = int($t / $window * 100);
		$color = $green;
	    }elsif ($g >= $a && $g >= $t && $g >= $c){
		my $num = int($g / $window * 100);
		$color = $yellow;
	    }elsif ($c >= $a && $c >= $t && $c >= $g){
		my $num = int($c / $window * 100);
		$color = $blue;
	    }
	    $im->setPixel($sidemargin + 1 + $locus % (700), 
			  $topmargin + (int($locus / 700) + 1) * $vblock * 5,
			  $color);

	    my $dist = 7;
	    if ($locus % $dist == $dist - 1){
		# Draw A content graph
		$num = int($a / int ($window * $amp) * 100) + 5;
		$im->line($sidemargin - $dist + $locus % (700), $pa,
			  $sidemargin + $locus % (700),
			  $topmargin - $num
			  + (int($locus / 700) + 1) * $vblock * 5,
			  $gred);
		$pa = $topmargin - $num +(int($locus / 700) + 1)
		    * $vblock * 5;

		# Draw T content graph
		$num = int($t / int ($window * $amp) * 100) + 5;
		$im->line($sidemargin - $dist + $locus % (700), $pt,
			  $sidemargin + $locus % (700),
			  $topmargin - $num
			  + (int($locus / 700) + 1) * $vblock * 5,
			  $ggreen);
		$pt = $topmargin - $num +(int($locus / 700) + 1)
		    * $vblock * 5;

		# Draw G content graph
		$num = int($g / int ($window * $amp) * 100) + 5;
		$im->line($sidemargin - $dist + $locus % (700), $pg,
			  $sidemargin + $locus % (700),
			  $topmargin - $num
			  + (int($locus / 700) + 1) * $vblock * 5,
			  $gyellow);
		$pg = $topmargin - $num +(int($locus / 700) + 1)
		    * $vblock * 5;

		# Draw C content graph
		$num = int($c / int ($window * $amp) * 100) + 5;
		$im->line($sidemargin - $dist + $locus % (700), $pc,
			  $sidemargin + $locus % (700),
			  $topmargin - $num
			  + (int($locus / 700) + 1) * $vblock * 5,
			  $gblue);
		$pc = $topmargin - $num +(int($locus / 700) + 1)
		    * $vblock * 5;
	    }elsif($locus % 700 == 0){
		$num = int($a / int ($window * $amp) * 100) + 5;
		$pa = $topmargin - $num +(int($locus / 700) + 1)
		    * $vblock * 5;
		$num = int($t / int ($window * $amp) * 100) + 5;
		$pt = $topmargin - $num +(int($locus / 700) + 1)
		    * $vblock * 5;
		$num = int($g / int ($window * $amp) * 100) + 5;
		$pg = $topmargin - $num +(int($locus / 700) + 1)
		    * $vblock * 5;
		$num = int($c / int ($window * $amp) * 100) + 5;
		$pc = $topmargin - $num +(int($locus / 700) + 1)
		    * $vblock * 5;
	    }
	    $locus ++;
	}

	# Draw Genes
	my $flag = 0;
	my $before = -5000;
	my $before2 = -10000;
	while (defined %{$gb->{"CDS$cds"}}){
	    my $cdsstart = $gb->{"CDS$cds"}->{start};
	    my $cdsend = $gb->{"CDS$cds"}->{end};
	    my $cdsdir = $gb->{"CDS$cds"}->{direction};
	    my $cdsdiff = $cdsstart - $before;
	    my $cdsdiff2 = $cdsstart - $before2;
	    if ($flag == 0){
		if (int($cdsdiff / $window) < 20){
		    $flag = 1;
		}
	    }elsif ($flag == 1){
		if (int($cdsdiff / $window) < 20){
		    if (int($cdsdiff2 / $window) < 20){
			$flag = 2;
		    }else{
			$flag = 0;
		    }
		}else{
		    $flag = 0;
		}
	    }elsif ($flag == 2){
		if (int($cdsdiff2 / $window) < 20){
		    $flag = 1;
		}else{
		    $flag = 0;
		}
	    }
		
	    if ($cdsstart < $start && $cdsend >$start){
		my $dif1 = -3; 
		my $dif2 = -2 - 3;
		if ($cdsdir eq 'complement'){
		    $dif1 *= -1;
		    $dif2 *= -1;
		}
		my $k;
		for ($k = 1; $k <= $cdsend - $start; $k ++){
		    my $l = int ($k / $window); 
		    $im->line($sidemargin + 1 + $l % 700,
			      $topmargin + $dif1 + (int($l/700)+1)*$vblock * 5,
			      $sidemargin + 1 + $l % 700,
			      $topmargin + $dif2 + (int($l/700)+1)*$vblock * 5,
			      $aqua);
		}
		$cds ++;
	    }else{
		last if ($cdsstart >= $end);
		my $feat = $gb->{"CDS$cds"}->{feature};
		my $genename = $gb->{"FEATURE$feat"}->{gene};
		
		my $dif1 = -3; 
		my $dif2 = -2 - 3;
		if ($cdsdir eq 'complement'){
		    $dif1 *= -1;
		    $dif2 *= -1;
		}

		my $k;
		for ($k = $cdsstart-$start; $k <= $cdsend - $start; $k += $window){
		    last if ($k + $start > $end);
		    my $l = int ($k / $window);
		    $im->line($sidemargin + 1 + $l % 700,
			      $topmargin + $dif1 + (int($l/700)+1)*$vblock * 5,
			      $sidemargin + 1 + $l % 700,
			      $topmargin + $dif2 + (int($l/700)+1)*$vblock * 5,
			      $aqua);
		}
		last if ($k + $start > $end);
		
		$k = int(($cdsstart - $start)/$window);
		
		$dif1 = -2; 
		$dif2 = -2 - 4;
		my $dif3 = -2 - 4 - 9 + (-9 * $flag);
		if ($cdsdir eq 'complement'){
		    $dif1 *= -1;
		    $dif2 *= -1;
		    $dif3 *= -1;
		    $dif3 -= 7;
		}
		$im->line($sidemargin + 1 + $k % 700,
			  $topmargin + $dif1 + (int($k/700)+1)*$vblock * 5,
			  $sidemargin + 1 + $k % 700,
			  $topmargin + $dif2 + (int($k/700)+1)*$vblock * 5,
			  $black);

		$im->string(gdTinyFont, $sidemargin + 1 + $k %700,
			    $topmargin + $dif3 + (int($k/700)+1)*$vblock * 5,
			    "$genename", $black) if ($name);

		if ($cdsend > $end){
		    last;
		}else{
		    $before = $cdsstart;
		    $before2 = $before;
		    $cds ++;
		}
	    }
	}

	open(OUT, '>graph/' . $acnum . '-'. $page . '.png');
	binmode OUT;
	print OUT $im->png;
	close(OUT);
        msg_gimv("graph/$acnum" . '-' . $page . '.png') if ($output eq 'show');
	$page ++;
    }
}

=head2 genome_map2

  Description:
    This method creates a map of the genome, showing the local 
    nucleotide contents and positions of genes using SVG.
    A is shown in red, T is shown in green, 
    G is shown in yellow, and C is shown in blue. 

    Map of the specified position (with -start and -end options)
    is generated in SVG format.

  Usage:
    NULL = genome_map2($gb);

 Options:
    -window    length of one pixel (default: 50)
    -type      type of features to print (default: CDS)
    -output    "g" for graph, "show" for display (default: "show") 
    -start     start position (default: 1)
    -end       end position (default: 100000)
    -filename  output file name (default: genome_map.svg)
    -ptt       ptt filename
    
  Author: 
    Kazuharu Arakawa (gaou@sfc.keio.ac.jp)

  History:
    20020906-01 initial posting

=cut


sub genome_map2 {
    &opt_default(output=>"show", window=>50, type=>"CDS", 
		 start=>1, end=>100000, filename=>"genome_map.svg", ptt=>'', cgi=>0);
    my @args = opt_get(@_);
    my $gb = shift @args;
    my $acnum = $gb->{LOCUS}->{id};
    my $output = opt_val("output");
    my $filename = opt_val("filename");
    my $topmargin = 60;
    my $sidemargin = 80;
    my $hblock = 100;
    my $vblock = 10;
    my $page = 1;
    my $i = 0;
    my $cds = 1;
    my $window=opt_val("window");
    my $start = opt_val("start");
    my $end = opt_val("end");
    my $ptt = opt_val("ptt");
    my $cgi = opt_val("cgi");

    my @type = split(/ /, opt_val("type"));
    my $j2 = 0;
    my $width = (int(($end - $start) / $hblock / $window) + 1) * $hblock  + $sidemargin * 2;
    my $height = $hblock/2 + $topmargin * 2 + scalar(@type) * $hblock - $vblock;

    eval{
	$ptt = set_gpac($gb, -ptt=>$ptt);
	 };

    unless ($cgi){
	sdb_save(\$ptt, "ptt");
	sdb_save($gb, "gb");
    }

    my $svg = SVG->new(width=>$width, height=>$height);
    
    # Draw Base Graph
    for ($i = $sidemargin; $i <= $width - $sidemargin; $i += $hblock){
	$svg->line(id=>"bvline$i", x1=>$i, 
		   y1=>$topmargin, x2=>$i, y2=>($hblock * (scalar(@type) + 1) + $vblock),
		   style=>{
		       stroke=>"darkgray", 'stroke-width'=>1,
		       'stroke-opacity'=>0.2
		       });
    }
    for ($i = $sidemargin; $i < $width - $sidemargin; $i += $hblock){
	$svg->text(
		   id=>"num$i", 
		   x=>$i, y=>($topmargin - 5), fill=>"black", 
		   'font-size'=>8,
		   )->cdata(((($i - $sidemargin) + 1) * $window - 49 + $start - 1) . " bp");
    }
    for ($i = $topmargin; $i <= $topmargin + $vblock * 5 ; $i += $vblock){
	$svg->line(id=>"bhline$i", x1=>$sidemargin, 
		   y1=>$i, x2=>($width - $sidemargin), y2=>$i,
		   style=>{
		       stroke=>"darkgray", 'stroke-width'=>1,
		       'stroke-opacity'=>0.2
		       });
    }
    for ($i = $topmargin + $vblock * 5; 
	 $i <= $topmargin + $hblock * (scalar(@type) + 1); $i += $hblock){
	$svg->line(id=>"bhlline$i", x1=>$sidemargin, 
		   y1=>$i, x2=>($width - $sidemargin), y2=>$i,
		   style=>{
		       stroke=>"darkgray", 'stroke-width'=>1,
		       'stroke-opacity'=>0.2
		       });
    }
    for ($i = $topmargin + $vblock * 5 + $hblock / 2; 
	 $i <= $topmargin + $hblock * scalar(@type); $i += $hblock){
	$svg->line(id=>"dnau$i", x1=>$sidemargin, 
		   y1=>($i - 1), x2=>($width - $sidemargin), y2=>($i - 1),
		   style=>{
		       stroke=>"black", 'stroke-width'=>1,
		       'stroke-opacity'=>1
		       });
	$svg->line(id=>"dnad$i", x1=>$sidemargin, 
		   y1=>($i + 1), x2=>($width - $sidemargin), y2=>($i + 1),
		   style=>{
		       stroke=>"black", 'stroke-width'=>1,
		       'stroke-opacity'=>1
		       });
    }

    $svg->text(
	       id=>"A", 
	       x=>20, y=>($topmargin + $vblock * 3), fill=>"red", 
	       'font-size'=>10,
	       )->cdata("A");
    $svg->text(
	       id=>"T", 
	       x=>30, y=>($topmargin + $vblock * 3), fill=>"green", 
	       'font-size'=>10,
	       )->cdata("T");
    $svg->text(
	       id=>"G", 
	       x=>40, y=>($topmargin + $vblock * 3), fill=>"gold", 
	       'font-size'=>10,
	       )->cdata("G");
    $svg->text(
	       id=>"C", 
	       x=>50, y=>($topmargin + $vblock * 3), fill=>"blue", 
	       'font-size'=>10,
	       )->cdata("C");
    
    my ($pa, $pt, $pg, $pc, $num, $color);
    my (@gra, @grt, @grc, @grg);
    my $locus = 0;
    for ($i = $start - 1; $i <= $end; $i += $window){
	my $seq = $gb->getseq($i, $i + $window * 2 - 1);
	my $a = $seq =~ tr/a/a/;
	my $t = $seq =~ tr/t/t/;
	my $g = $seq =~ tr/g/g/;
	my $c = $seq =~ tr/c/c/;
	my $x = $seq =~ tr/x/x/;
	my $n = $seq =~ tr/n/n/;
	
	if ($locus % 2 != 1){
    	    # Draw DNA
	    if ($a >= $t && $a >= $g && $a >= $c){
		my $num = int($a / $window * 100);
		$color = 'red';
		$color = 'white' if ($a == 0);
	    }elsif ($t >= $a && $t >= $g && $t >= $c){
		my $num = int($t / $window * 100);
		$color = 'green';
	    }elsif ($g >= $a && $g >= $t && $g >= $c){
		my $num = int($g / $window * 100);
		$color = 'yellow';
	    }elsif ($c >= $a && $c >= $t && $c >= $g){
		my $num = int($c / $window * 100);
		$color = 'blue';
	    }

	    my $j;
	    for ($j = $topmargin + $vblock * 5 + $hblock / 2; 
		 $j <= $topmargin + $hblock * scalar(@type); 
		 $j += $hblock
		 ){

		last if ($locus >= int($width - $sidemargin * 2) - 3);
		$j2 ++;

		$svg->line(id=>"dnac$j-$j2", x1=>($sidemargin + 1 + $locus), 
			   y1=>$j, x2=>($sidemargin + 3 + $locus), y2=>$j,
			   style=>{
			       stroke=>$color, 'stroke-width'=>1,
			       'stroke-opacity'=>1
			       });
	    }
	}
    
	my $dist = 5;
	if ($locus % $dist == $dist -1 || $locus == 0){
	    last if ($a == 0 && $t == 0 && $c == 0 && $g == 0 && $x == 0 && $n == 0);
	    push (@gra, sprintf("%d,%d", $sidemargin + $locus,
				$topmargin + $vblock * 5 - int($a / 2 / $window * 100 ) + 5));
	    push (@grt, sprintf("%d,%d", $sidemargin + $locus,
				$topmargin + $vblock * 5 - int($t / 2 / $window * 100 ) + 5));
	    push (@grg, sprintf("%d,%d", $sidemargin + $locus,
				$topmargin + $vblock * 5 - int($g / 2 / $window * 100 ) + 5));
	    push (@grc, sprintf("%d,%d", $sidemargin + $locus,
				$topmargin + $vblock * 5 - int($c / 2 / $window * 100 ) + 5));
	}

	$locus ++;
    }

    $svg->polyline(id=>"gra", points=>join(" ", @gra), 
	       style=>{
		   stroke=>'red', 'stroke-width'=>1,
		   'stroke-opacity'=>0.2, fill=>'none'
		   });
    $svg->polyline(id=>"grt", points=>join(" ", @grt), 
	       style=>{
		   stroke=>'green', 'stroke-width'=>1,
		   'stroke-opacity'=>0.2, fill=>'none'
		   });
    $svg->polyline(id=>"grg", points=>join(" ", @grg), 
	       style=>{
		   stroke=>'yellow', 'stroke-width'=>1,
		   'stroke-opacity'=>0.2, fill=>'none'
		   });
    $svg->polyline(id=>"grc", points=>join(" ", @grc), 
	       style=>{
		   stroke=>'blue', 'stroke-width'=>1,
		   'stroke-opacity'=>0.2, fill=>'none'
		   });


    # Draw Genes
    my $flag = 0;
    my $before = -5000;
    my $before2 = -10000;
    my @colors = qw(J K L D O M N P T C G E F H I Q R S -);
    
    my $col = 0;
    foreach my $pat (@type){
	$svg->text(
		   id=>$pat, 
		   x=>20, y=>($topmargin + $vblock * 5 + $hblock * $col + $hblock/2 + 3),
		   'font-size'=>10,
		   )->cdata($pat);
	
	foreach my $cds ($gb->feature()){
	    next unless($gb->{$cds}->{end} >= $start && $gb->{$cds}->{start} <= $end);
	    next unless($gb->{$cds}->{type} eq $pat);
	    my $cdsstart = $gb->{$cds}->{start};
	    my $cdsend = $gb->{$cds}->{end};
	    my $cdsdir = $gb->{$cds}->{direction};
	    my $cdsdiff = $cdsstart - $before;
	    my $genename = $gb->{$cds}->{gene};
	    my $code = $gb->{$cds}->{code} || $colors[$col];

	    my $dif1 = -10;
	    my $sign = -1;
	    $dif1 = -25 if ($cdsdiff / $window < 15);
	    
	    if ($cdsdir eq 'complement'){
		$dif1 *= -1;
		$sign *= -1;
	    }
	    
	    $cdsstart = $start if ($cdsstart < $start);
	    $cdsend = $end if ($cdsend > $end);
	    
	    if ($cdsdir eq 'direct'){
		$svg->rect(
			   id=>"liner$cds", x=>($sidemargin + ($cdsstart - $start)/$window), 
			   y=>($topmargin + $dif1 + $vblock * 5 + $hblock / 2 - 3 + $hblock * $col),
			   height=>(abs($dif1) + 3),
			   width=>(($cdsend - $cdsstart)/$window),
			   style=>{
			       stroke=>"gray", 'stroke-width'=>1,
			       'stroke-opacity'=>0.2, 'fill'=>'none'
			       });
	    }else{
		$svg->rect(
			   id=>"liner$cds", x=>($sidemargin + ($cdsstart - $start)/$window), 
			   y=>($topmargin + $vblock * 5 + $hblock / 2 - 1 + $hblock * $col),
			   height=>(abs($dif1) + 3),
			   width=>(($cdsend - $cdsstart)/$window),
			   style=>{
			       stroke=>"gray", 'stroke-width'=>1,
			       'stroke-opacity'=>0.2, 'fill'=>'none'
			       });
	    }
	    
	    $svg->anchor(
			 -href=>'http://localhost/g-language/genome_view.cgi?pos=' . $cds,
			 -target=>'_blank'
			 )->rect(
				 id=>"line$cds", x=>($sidemargin + ($cdsstart - $start)/$window), 
				 y=>($topmargin + $dif1 + $vblock * 5 + $hblock / 2 - 3 + $hblock * $col),
				 height=>5,
				 width=>(($cdsend - $cdsstart)/$window),
				 style=>{
				     stroke=>"green", 'stroke-width'=>1,
				     'stroke-opacity'=>0.2, 'fill'=>$COG_fcolor{$code},
				     'fill-opacity'=>0.2
				     });
	    
	    $svg->text(
		       id=>"cds$cds", 
		       x=>($sidemargin + ($cdsstart - $start)/$window), 
		       y=>($topmargin + (abs($dif1) + 6) * $sign + $vblock * 5 + $hblock / 2 + 1  + $hblock * $col), 
		       fill=>$COG_fcolor{$code}, 
		       'font-size'=>5,
		       )->cdata($genename);
	    
	    $before = $cdsstart;
	}
	$col ++;
    }

    my $topline = $svg->line(id=>"top", x1=>0, y1=>10, x2=>$width, y2=>10,
			     style=>{
				 stroke=>"plum", 'stroke-width'=>3,
				 'stroke-opacity'=>0.3
				 });
    
    $topline->animate(
		      attributeName=>"stroke", 
		      values=>join(';', values(%COG_fcolor)), dur=>"90s",
		      repeatDur=>'indefinite'
		      );
    
    my $bottomline = $svg->line(id=>"bottom", x1=>0, y1=>($height-10), x2=>$width, y2=>($height-10),
				style=>{
				    stroke=>"plum", 'stroke-width'=>3,
				    'stroke-opacity'=>0.3
				    });
    
    $bottomline->animate(
			 attributeName=>"stroke", 
			 values=>join(';', values(%COG_fcolor)), dur=>"90s",
			 repeatDur=>'indefinite'
			 );
    
    $svg->anchor(
		 -href=>"http://www.g-language.org/",
		 -target=>"_blank"
		 )->text(
			 id=>"credits", 
			 x=>($width - 300), y=>($height - 15), fill=>"darkgray", 
			 'font-size'=>8, 'font-style'=>'italic',
			 )->cdata(
				  "generated by genome_map2, G-language Genome Analysis Environment"
				  );
    
    $svg->text(
	       id=>"locus1", 
	       x=>30, y=>26, fill=>"navy", 
	       'font-size'=>10,
	       )->cdata("Accession Number: ", $gb->{LOCUS}->{id});
    
    $svg->text(
	       id=>"locus2", 
	       x=>250, y=>26, fill=>"navy", 
	       'font-size'=>10,
	       )->cdata("Organism: ", $gb->{FEATURE0}->{organism});
    
    $svg->text(
	       id=>"locus3", 
	       x=>500, y=>26, fill=>"navy", 
	       'font-size'=>10,
	       )->cdata(length($gb->{SEQ}) . " bp");
    

    my $leftarrow = $svg->anchor(
				 -href=>'http://localhost/g-language/genome_view.cgi?pos=' . ($start - 50000)
				 )->polygon(
				   id=>"larrow", 
				   points=>sprintf("%d,%d %d,%d %d,%d", 50,
						   $topmargin + $vblock * 5 + $hblock * scalar(@type) - 10,
						   50,
						   $topmargin + $vblock * 5 + $hblock * scalar(@type) + 20,
						   15, 
						   $topmargin + $vblock * 5 + $hblock * scalar(@type) + 5
						   ),
				   style=>{
				       fill=>"plum", stroke=>"plum", 'stroke-width'=>5,
				       'fill-opacity'=>0.1, 'stroke-opacity'=>0.3
				       }
				   );
    
    $leftarrow->animate(
		      attributeName=>"stroke", 
		      values=>join(';', reverse(values(%COG_fcolor))), dur=>"90s",
		      repeatDur=>'indefinite'
		      );

    $leftarrow->animate(
		      attributeName=>"fill", 
		      values=>join(';', reverse(values(%COG_fcolor))), dur=>"90s",
		      repeatDur=>'indefinite'
		      );

    my $rightarrow = $svg->anchor(
				 -href=>'http://localhost/g-language/genome_view.cgi?pos=' . ($end + 50001)
				 )->polygon(
				   id=>"rarrow", 
				   points=>sprintf("%d,%d %d,%d %d,%d", $width - 50,
						   $topmargin + $vblock * 5 + $hblock * scalar(@type) - 10,
						   $width - 50,
						   $topmargin + $vblock * 5 + $hblock * scalar(@type) + 20,
						   $width - 15, 
						   $topmargin + $vblock * 5 + $hblock * scalar(@type) + 5
						   ),
				   style=>{
				       fill=>"plum", stroke=>"plum", 'stroke-width'=>5,
				       'fill-opacity'=>0.1, 'stroke-opacity'=>0.3
				       }
				   );
    
    $rightarrow->animate(
		      attributeName=>"stroke", 
		      values=>join(';', reverse(values(%COG_fcolor))), dur=>"90s",
		      repeatDur=>'indefinite'
		      );

    $rightarrow->animate(
		      attributeName=>"fill", 
		      values=>join(';', reverse(values(%COG_fcolor))), dur=>"90s",
		      repeatDur=>'indefinite'
		      );

    mkdir ("graph", 0777);
    open(OUT, '>graph/' . $filename) || msg_error($!);
    print OUT $svg->xmlify;
    close(OUT);
    msg_gimv('graph/' . $filename) if ($output eq 'show');
    
    return 1;
}







=head2 genome_map3

  Name: genome_map3   -   draws the map of the genome (version 3)

  Description:
    This method creates a map of the genome, showing the local 
    nucleotide contents and positions of genes.
    A is shown in red, T is shown in green, 
    G is shown in yellow, and C is shown in blue. 

    Entire genome is prepresented in a very large PNG image
    (8192x8192 pixel by default), and the number of nucleotides
    represented by one pixel is automatically calculated. Then 
    the large image is converted to a zoomable image usinig 
    Google Map API.

  Usage:
    NULL = genome_map($gb);

 Options:
    -width          image width (default: 8192)
    -height         image height (default: 8192)
    -output         "g" for graph, "show" for display (default: "show") 
    -datafilename   coordinate file name (default: "data.coord")

  Author: 
    Kazuharu Arakawa (gaou@sfc.keio.ac.jp)

  History:
   20070628-01 added -datafilename option
   20070623-01 initial posting

=cut

sub genome_map3 {
    &opt_default(output=>"show", width=>8192, height=>8192, datafilename=>"data.coord");
    my @args = opt_get(@_);
    my $gb = opt_as_gb(shift @args);
    my $acnum = $gb->{LOCUS}->{id};
    my $datafilename = opt_val("datafilename");
    my $output = opt_val("output");
    my $filename;
    my $topmargin = 50;
    my $sidemargin = 150;
    my $hblock = 100;
    my $vblock = 10;
    my $width = opt_val("width");
    my $height = opt_val("height");

    my $numHBlock = int(($width - $sidemargin - 50) / $hblock);
    my $numVBlock = int(($height - $topmargin - 50) / ($vblock * 5));
    my $window = int(length($gb->{SEQ}) / ($numHBlock * $numVBlock * $hblock)) + 1;

    msg_error("Image size: $width x $height pixels.\nUsing Window size = $window bp to fit the whole genome...\n\n");
    msg_error("Map generation will take a few minutes. Please make yourself comfortable until the image is generated ;-\)\n\n");

    my $start = 1;
    my $end = length $gb->{SEQ};
    my $i = 0;
    my $cds = 1;

    mkdir ("graph", 0777);
    mkdir ("data", 0777);
    open(OUT, '>' . 'data/' . $datafilename) || die($!);

    # GD constant
    my $im = new GD::Image($width, $height);
    my $white = $im->colorAllocate(255,255,255);
    my $black = $im->colorAllocate(0,0,0);
    my $gray = $im->colorAllocate(180,180,180);  
    my $red = $im->colorAllocate(255,0,0);       #A
    my $yellow = $im->colorAllocate(255,255,0);  #G
    my $green = $im->colorAllocate(0,150,0);     #T
    my $blue = $im->colorAllocate(0,0,255);      #C
    my $aqua = $im->colorAllocate(120, 160, 255);
    my $rrna = $im->colorAllocate(255, 160, 160);
    my $trna = $im->colorAllocate(160, 255, 160);

    my $gred = $im->colorAllocate(255,200,200);       #A for graph
    my $gyellow = $im->colorAllocate(255,255,100);    #G for graph
    my $ggreen = $im->colorAllocate(180,220,180);     #T for graph
    my $gblue = $im->colorAllocate(200,200,255);      #C for graph
    

    # Draw Base Graph
    for ($i = $sidemargin; $i <= $sidemargin + $hblock * $numHBlock; $i += $hblock){
	$im->line($i, $topmargin, $i, $topmargin + 5 * ($numVBlock + 1) * $vblock, $gray);
    }
    for ($i = $topmargin; $i <= $topmargin + $vblock * 5 * ($numVBlock + 1); $i += $vblock){
	$im->line($sidemargin, $i, $sidemargin + $numHBlock * $hblock, $i, $gray);
    }
    for ($i = $topmargin + $vblock * 5; $i < $topmargin + $vblock * 5 * ($numVBlock + 1); $i += $vblock * 5){
	$im->line($sidemargin, $i - 1, $sidemargin + $numHBlock * $hblock, $i - 1, $black);
	$im->line($sidemargin, $i + 1, $sidemargin + $numHBlock * $hblock, $i + 1, $black);
    }

    $im->string(gdGiantFont, $width - 210, 5, "G-language Project", $black);
    $im->string(gdGiantFont, $width - 210 - 50, 5, "A", $red);
    $im->string(gdGiantFont, $width - 210 - 40, 5, "T", $green);
    $im->string(gdGiantFont, $width - 210 - 30, 5, "G", $yellow);
    $im->string(gdGiantFont, $width - 210 - 20, 5, "C", $blue);

    my $j = 0;
    for ($i = $topmargin + $vblock * 5; $i <= $topmargin + $vblock * 5 * $numVBlock; $i += $vblock * 5){
	my $num = $start + $j * $window * $numHBlock * $numVBlock * $hblock;
	$im->string(gdLargeFont, 10, $i, "$num", $black);
	$j ++;
    }
    $im->string(gdGiantFont, 50,  5, "$acnum : from $start to $end", $black);

    my ($pa, $pt, $pg, $pc, $num, $color);
    my $locus = 0;
    for ($i = $start - 1; $i + $window <= $end; $i += $window){
	my $seq = $gb->getseq($i, $i + $window - 1);
	my $a = $seq =~ tr/a/a/;
	my $t = $seq =~ tr/t/t/;
	my $g = $seq =~ tr/g/g/;
	my $c = $seq =~ tr/c/c/;

	# Draw DNA
	if ($a >= $t && $a >= $g && $a >= $c){
	    my $num = int($a / $window * 100);
	    $color = $red;
	}elsif ($t >= $a && $t >= $g && $t >= $c){
	    my $num = int($t / $window * 100);
	    $color = $green;
	}elsif ($g >= $a && $g >= $t && $g >= $c){
	    my $num = int($g / $window * 100);
	    $color = $yellow;
	}elsif ($c >= $a && $c >= $t && $c >= $g){
	    my $num = int($c / $window * 100);
	    $color = $blue;
	}
	$im->setPixel($sidemargin + 1 + $locus % (($hblock * $numHBlock)), $topmargin + (int($locus / ($hblock * $numHBlock)) + 1) * $vblock * 5, $color);

	my $dist = 7;
	if (($locus % (($hblock * $numHBlock))) % $dist == $dist - 1){
	    # Draw A content graph
	    $num = int(($a / $window) * $vblock * 3.8) + 2;

	    $im->line($sidemargin - $dist + 1 + $locus % (($hblock * $numHBlock)), $pa, 
		      $sidemargin + $locus % (($hblock * $numHBlock)),
		      $topmargin - $num + (int($locus / ($hblock * $numHBlock)) + 1) * $vblock * 5, $gred);
	    $pa = $topmargin - $num + (int($locus / ($hblock * $numHBlock)) + 1) * $vblock * 5;
	    
	    # Draw T content graph
	    $num = int(($t / $window) * $vblock * 3.8) + 2;
	    $im->line($sidemargin - $dist + 1 + $locus % (($hblock * $numHBlock)), $pt, 
		      $sidemargin + $locus % (($hblock * $numHBlock)),
		      $topmargin - $num + (int($locus / ($hblock * $numHBlock)) + 1) * $vblock * 5, $ggreen);
	    $pt = $topmargin - $num +(int($locus / ($hblock * $numHBlock)) + 1) * $vblock * 5;
	    
	    # Draw G content graph
	    $num = int(($g / $window) * $vblock * 3.8) + 2;
	    $im->line($sidemargin - $dist + 1 + $locus % (($hblock * $numHBlock)), 
		      $pg, $sidemargin + $locus % (($hblock * $numHBlock)),
		      $topmargin - $num + (int($locus / ($hblock * $numHBlock)) + 1) * $vblock * 5, $gyellow);
	    $pg = $topmargin - $num +(int($locus / ($hblock * $numHBlock)) + 1) * $vblock * 5;
	    
	    # Draw C content graph
	    $num = int(($c / $window) * $vblock * 3.8) + 2;
	    $im->line($sidemargin - $dist + 1 + $locus % (($hblock * $numHBlock)), 
		      $pc, $sidemargin + $locus % (($hblock * $numHBlock)),
		      $topmargin - $num + (int($locus / ($hblock * $numHBlock)) + 1) * $vblock * 5, $gblue);
	    $pc = $topmargin - $num +(int($locus / ($hblock * $numHBlock)) + 1) * $vblock * 5;

	}elsif($locus % ($hblock * $numHBlock) == 0){

	    $num = int(($a / $window) * $vblock * 3.8) + 2;
	    $pa = $topmargin - $num +(int($locus / ($hblock * $numHBlock)) + 1) * $vblock * 5;
	    $num = int(($t / $window) * $vblock * 3.8) + 2;
	    $pt = $topmargin - $num +(int($locus / ($hblock * $numHBlock)) + 1) * $vblock * 5;
	    $num = int(($g / $window) * $vblock * 3.8) + 2;
	    $pg = $topmargin - $num +(int($locus / ($hblock * $numHBlock)) + 1) * $vblock * 5;
	    $num = int(($c / $window) * $vblock * 3.8) + 2;
	    $pc = $topmargin - $num +(int($locus / ($hblock * $numHBlock)) + 1) * $vblock * 5;
	}
	$locus ++;
    }

    # Draw Genes
    my $flag = 0;
    my $before = -5000;
    my $before2 = -10000;
    foreach my $feat ($gb->feature()){
	next unless($gb->{$feat}->{type} eq 'CDS' || $gb->{$feat}->{type} eq 'rRNA' || $gb->{$feat}->{type} eq 'tRNA');
	my $cdsstart = $gb->{$feat}->{start};
	my $cdsend = $gb->{$feat}->{end};
	my $cdsdir = $gb->{$feat}->{direction};
	my $cdsdiff = $cdsstart - $before;
	my $cdsdiff2 = $cdsstart - $before2;
	my $genename = $gb->{$feat}->{gene};

	if ($flag == 0){
	    if (int($cdsdiff / $window) < 25){
		$flag = 1;
	    }
	}elsif ($flag == 1){
	    if (int($cdsdiff / $window) < 25){
		if (int($cdsdiff2 / $window) < 25){
		    $flag = 2;
		}else{
		    $flag = 0;
		}
		$flag = 0;
	    }else{
		$flag = 0;
	    }
	}elsif ($flag == 2){
	    if (int($cdsdiff2 / $window) < 25){
		$flag = 1;
	    }else{
		$flag = 0;
	    }
	}

	my $blockcolor = $aqua;
	$blockcolor = $rrna if ($gb->{$feat}->{type} eq 'rRNA');
	$blockcolor = $trna if ($gb->{$feat}->{type} eq 'tRNA');

	unless ($cdsstart >= $end){
	    
	    my $dif1 = -2; 
	    my $dif2 = -7 - 2;
	    if ($cdsdir eq 'complement'){
		$dif1 *= -1;
		$dif2 *= -1;
	    }

	    if($cdsdir eq 'complement'){
		printf OUT "%s\t%s\t%s\t%s\t%s\n", $genename, $sidemargin + ($cdsstart / $window) % ($hblock * $numHBlock),
		$topmargin + $dif1 + (int(($cdsstart / $window) / ($hblock * $numHBlock))+1) * $vblock * 5,
		$sidemargin + ($cdsend / $window) % ($hblock * $numHBlock),
		$topmargin + $dif2 + (int(($cdsend / $window) / ($hblock * $numHBlock))+1) * $vblock * 5;
	    }else{
		printf OUT "%s\t%s\t%s\t%s\t%s\n", $genename, $sidemargin + ($cdsstart / $window) % ($hblock * $numHBlock),
		$topmargin + $dif2 + (int(($cdsstart / $window) / ($hblock * $numHBlock))+1) * $vblock * 5,
		$sidemargin + ($cdsend / $window) % ($hblock * $numHBlock),
		$topmargin + $dif1 + (int(($cdsend / $window) / ($hblock * $numHBlock))+1) * $vblock * 5;
	    }


	    my $k;
	    for ($k = $cdsstart-$start; $k <= $cdsend - $start; $k += $window){
		last if ($k + $start > $end);
		my $l = int ($k / $window);
		$im->line($sidemargin + 1 + $l % ($hblock * $numHBlock),
			  $topmargin + $dif1 + (int($l/($hblock * $numHBlock))+1)*$vblock * 5,
			  $sidemargin + 1 + $l % ($hblock * $numHBlock),
			  $topmargin + $dif2 + (int($l/($hblock * $numHBlock))+1)*$vblock * 5,
			  $blockcolor);
	    }
	    last if ($k + $start > $end);
	    
	    $k = int(($cdsstart - $start)/$window);
	    
	    $dif1 = -2; 
	    $dif2 = -2 - 11;
	    my $dif3 = -2 - 15 - 9 + (-12 * $flag);
	    if ($cdsdir eq 'complement'){
		$dif1 *= -1;
		$dif2 *= -1;
		$dif3 *= -1;
		$dif3 -= 15;
	    }
	    $im->line($sidemargin + 1 + $k % ($hblock * $numHBlock),
		      $topmargin + $dif1 + (int($k/($hblock * $numHBlock))+1)*$vblock * 5,
		      $sidemargin + 1 + $k % ($hblock * $numHBlock),
		      $topmargin + $dif2 + (int($k/($hblock * $numHBlock))+1)*$vblock * 5,
		      $black);
	    
	    $im->string(gdMediumBoldFont, $sidemargin + 1 + $k %($hblock * $numHBlock),
			$topmargin + $dif3 + (int($k/($hblock * $numHBlock))+1)*$vblock * 5,
			"$genename", $black);
	    
	    if ($cdsend > $end){
		last;
	    }else{
		$before = $cdsstart;
		$before2 = $before;
	    }
	}
    }
    close(OUT);

    open(IMAGE, '>graph/' . $acnum . '.png');
    binmode IMAGE;
    print IMAGE $im->png;
    close(IMAGE);

    generateGMap("graph/$acnum.png");
}






=head2 seq2png

 Description:
  Converts a sequence to a png image, by representing nucleotide 
  sequences with representative pixels. 
  A is shown in red, T is shown in green, 
  G is shown in yellow, and C is shown in blue. 

  Usage: 
    NULL = seq2png(G instance); 

 Options:
    -width     width of the image (default:640)
    -window    window size of a sequence to represent each pixel (default: 20)
    -filename  output filename (default:'seq.png')
    -output    "g" for graph, "show" for display


  Author: 
    Kazuharu Arakawa (gaou@sfc.keio.ac.jp)

  History:
    20060714 added -window option
    20010906 update with options
    20010830 initial posting

=cut




sub seq2png {
    &opt_default(width=>640, filename=>"seq.png", output=>"show", window=>"20");
    my @args = opt_get(@_);
    my $gb = opt_as_gb(shift @args);
    my $width = opt_val("width");
    my $window = opt_val("window");
    my $output = opt_val("output");
    my $filename = opt_val("filename");
    my $height = int((length($gb->{SEQ})/$window)/$width)+1;
    my $im = new GD::Image($width, $height);

    my $white = $im->colorAllocate(255,255,255);
    my $red = $im->colorAllocate(255,0,0);
    my $yellow = $im->colorAllocate(255,255,0);
    my $green = $im->colorAllocate(0,150,0);
    my $blue = $im->colorAllocate(0,0,255);

    my ($x, $y);
    my $count = 0;
    for ($y = 0; $y <= $height; $y ++){
	for ($x = 0; $x <= $width; $x ++){
	    my $color=$white;

	    my $g = substr($gb->{SEQ}, $count * $window, $window) =~ tr/g/g/;
	    my $c = substr($gb->{SEQ}, $count * $window, $window) =~ tr/c/c/;
	    my $a = substr($gb->{SEQ}, $count * $window, $window) =~ tr/a/a/;
	    my $t = substr($gb->{SEQ}, $count * $window, $window) =~ tr/t/t/;

	    if($g >= $c && $g >= $a && $g >= $t){
		$color=$yellow;
	    }elsif($c >= $g && $c >= $a && $c >= $t){
		$color = $blue;
	    }elsif($a >= $g && $a >= $c && $a >= $t){
		$color = $red;
	    }elsif($t >= $g && $t >= $c && $t >= $a){
		$color = $green;
	    }
	    
	    $im->setPixel($x,$y,$color);
	    last if ($count == length($gb->{SEQ}));
	    $count ++;
	}
    }

    mkdir ('graph', 0777);
    open(OUT, '>graph/' . $filename);
    binmode OUT;
    print OUT $im->png;
    close(OUT);

    msg_gimv("graph/$filename") if ($output eq 'show');
}




=head2 plasmid_map

  Description:
   This method creates a plasmid map image with SVG.

  Usage:
    NULL = plasmid_map($gb);

 Options:
   -ptt       ptt file name
   -output    g|show (default:show)
   -filename  output file name (default:plasmid_map.svg)

  Author: 
    Kazuharu Arakawa (gaou@sfc.keio.ac.jp)

  History:
   20020905-01 initial posting

=cut


sub plasmid_map{
    opt_default(ptt=>'', output=>'show', filename=>"plasmid_map.svg", cgi=>0);
    my @args = opt_get(@_);
    my $gb = shift;
    my $ptt = opt_val("ptt");
    my $output = opt_val("output");
    my $filename = opt_val("filename");
    my $cgi = opt_val("cgi");

    eval{
	$ptt = set_gpac($gb, -ptt=>$ptt);
	 };

    unless ($cgi){
	sdb_save(\$ptt, "ptt");
	sdb_save($gb, "gb");
    }

    my $svg = SVG->new(width=>640, height=>400);
    my $pi = atan2(1,1) * 4;
    my $seqlen = length $gb->{SEQ};
    my $maxlen;
    foreach my $cds ($gb->cds()){
	my $tmplen = length ($gb->get_geneseq($cds));
	$maxlen = $tmplen if ($tmplen > $maxlen);
    }
    
    foreach my $cds ($gb->cds()){
	my $id = $gb->{$cds}->{gene};
	my $gene = $gb->get_geneseq($cds);
	my $length = length $gene;
	my $gc = int(log($maxlen) / log($length) * 30);
	
	my $rad = 2 * $pi * $gb->{$cds}->{start} / $seqlen * -1 + $pi;
	
	my ($x1, $x2, $y1, $y2);
	if ($gb->{$cds}->{direction} eq 'direct'){
	    $x1 = (100 + 5) * sin($rad) + 200;
	    $y1 = (100 + 5) * cos($rad) + 200;
	    $x2 = (100 + $gc) * sin($rad) + 200;
	    $y2 = (100 + $gc) * cos($rad) + 200;
	}else{
	    $x1 = (100 - $gc) * sin($rad) + 200;
	    $y1 = (100 - $gc) * cos($rad) + 200;
	    $x2 = (100 - 5) * sin($rad) + 200;
	    $y2 = (100 - 5) * cos($rad) + 200;
	}
	
	my $stroke = $COG_fcolor{$gb->{$cds}->{code}};
	
	my $cdsstart = $gb->{$cds}->{start};
	$svg->anchor(
		     -href=>'http://localhost/g-language/genome_view.cgi?pos=' . $cdsstart,
		     -target=>"_blank"
		     )->line(
			     id=>$id, x1=>$x1, 
			     y1=>$y1, x2=>$x2, y2=>$y2,
			     style=>{
				 stroke=>$stroke, 'stroke-width'=>1,
				 'stroke-opacity'=>0.3
				 });
    }
    
    $svg->circle(id=>'genome', cx=>200, cy=>200, r=>100, 
		 style=>{
		     stroke=>'blue', fill=>'none', 'stroke-width'=>5,
		     'stroke-opacity'=>0.2
		     });
    
    my $x = 400;
    my $y = 123;
    my $y2 = 130;
    
    $svg->anchor(
		 -href=>"http://www.ncbi.nih.gov/cgi-bin/COG/palox?fun=all",
		 -target=>"_blank"
		 )->text(
			 id=>"COG link", 
			 x=>($x - 20), y=>($y2 - 30), 
			 'font-size'=>9
			 )->cdata(
				  "Gene Classification based on NCBI COG functional categories"
				  );
    
    foreach my $category (qw(J K L D O M N P T C G E F H I Q R S -)){
	$svg->rect(id=>$category, x=>$x, y=>$y, width=>9, height=>8,
		   style=>{
		       stroke=>'none', fill=>$COG_fcolor{$category},
		       opacity=>0.3
		       });
	
	my $id = $category . '_text';
	
	$svg->text(id=>$id, x=>($x + 15), y=>$y2, 'font-size'=>7)->cdata($COG_fcode{$category});
	
	$y += 12;
	$y2 += 12;
    }
    
    my $topline = $svg->line(id=>"top", x1=>0, y1=>10, x2=>640, y2=>10,
			     style=>{
				 stroke=>"blue", 'stroke-width'=>3,
				 'stroke-opacity'=>0.3
				 });
    
    $topline->animate(
		      attributeName=>"stroke", 
		      values=>join(';', values(%COG_fcolor)), dur=>"90s",
		      repeatDur=>'indefinite'
		      );
    
    my $bottomline = $svg->line(id=>"bottom", x1=>0, y1=>390, x2=>640, y2=>390,
				style=>{
				    stroke=>"blue", 'stroke-width'=>3,
				    'stroke-opacity'=>0.3
				    });
    
    $bottomline->animate(
			 attributeName=>"stroke", 
			 values=>join(';', values(%COG_fcolor)), dur=>"90s",
			 repeatDur=>'indefinite'
			 );
    
    $svg->anchor(
		 -href=>"http://www.g-language.org/",
		 -target=>"_blank"
		 )->text(
			 id=>"credits", 
			 x=>380, y=>385, fill=>"darkgray", 
			 'font-size'=>8, 'font-style'=>'italic',
			 )->cdata(
				  "generated by plasmid map, G-language Genome Analysis Environment"
				  );
    
    $svg->text(
	       id=>"locus", 
	       x=>30, y=>30, fill=>"navy", 
	       'font-size'=>10,
	       )->cdata("Accession Number: ", $gb->{LOCUS}->{id});
    
    $svg->text(
	       id=>"locus", 
	       x=>250, y=>30, fill=>"navy", 
	       'font-size'=>10,
	       )->cdata("Organism: ", $gb->{FEATURE0}->{organism});
    
    $svg->text(
	       id=>"locus", 
	       x=>500, y=>30, fill=>"navy", 
	       'font-size'=>10,
	       )->cdata("$seqlen bp");
    
    mkdir('graph', 0777);
    open(OUT, '>graph/' . $filename) || msg_error($!);
    print OUT $svg->xmlify;
    close(OUT);

    msg_gimv('graph/' . $filename) if ($output eq "show");

    return 1;
}






1;
