#:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
# This file is part of G-language Genome Analysis Environment package
#
#     Copyright (C) 2001-2010 Keio University
#:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
# 
#   $Id: EPCR.pm,v 1.1.1.1 2002/04/02 20:25:44 gaou Exp $
#
# G-language GAE is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
# 
# G-language GAE is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
# See the GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public
# License along with G-language GAE -- see the file COPYING.
# If not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
# 
#END_HEADER
#

package G::Tools::EMBOSS;

use SubOpt;
use G::Messenger;

use strict;
use base qw(Exporter);
use SelfLoader;

our @EXPORT = qw(
		 seqret
		 );


__DATA__


#:::::::::::::::::::::::::::::::::
#       Perldoc
#:::::::::::::::::::::::::::::::::


=head1 NAME

    G::Tools::EMBOSS - Wrapper interface to EMBOSS software suite

=head1 DESCRIPTION

    This class is a part of G-language Genome Analysis Environment, 
    collecting interfaces to EMBOSS software suite.

=cut



#:::::::::::::::::::::::::::::::::
#       Let the code begin...
#:::::::::::::::::::::::::::::::::


=head2 seqret

 Name: seqret   -   reads and writes (returns) sequences (interface to EMBOSS)

 Description:
    This is a wrapper around EMBOSS Explorer web service provided by 
    http://emboss.bioinformatics.nl/.
    Seqret command of EMBOSS allows sequence retrieval from public databases
    and format interconversion. 

    See http://emboss.bioinformatics.nl/cgi-bin/emboss/help/seqret
    for supported input/output formats.

    Supported databases are:
      ensemblp, ipi, ipi_arath, ipi_brare, ipi_human, ipi_mouse, ipi_rat
      myco_pep, myco_pep_all, nbrf, pdb, pir, refseqp, sw, swissprot, test
      trembl, uniprot, uniprotsrs, uniref, univar, em, embl, emblann, emblannnew
      emblsrs, emblwgs, emnew, emwgs, epd, refseq

 Usage:
    seqret("database:entry");

 Options:
   [-sequence]    sequence filename or input USA
   [-outseq]      output filename and optional format
   -osformat      output format
   -feature       flag to retrieve feature information
   -firstonly     flag to read only one sequence
   -help          display help messages
  

 References:
   1. Rice P, Longden I, Bleasby A (2000) "EMBOSS: the European Molecular Biology Open
      Software Suite.", Trends Genet., 16(6):276-277.

 Author: 
    Kazuki Oshita (t07122ko@sfc.keio.ac.jp)
    Kazuharu Arakawa (gaou@sfc.keio.ac.jp)

 History:
   20080212-01 initial posting

=cut


sub seqret{
    require CGI;
    require LWP::UserAgent;
    require HTML::Form;

    my @args = opt_get(@_);
    my $USA = opt_val("sequence") || shift @args;
    my $feature = opt_val("feature");
    my $firstonly = opt_val("firstonly");
    my $osformat = opt_val("osformat");
    my $help = opt_val("help");
    my $outseq = opt_val("outseq") || shift @args;

    if($help){
	print << '__HELP__';

	   Standard (Mandatory) qualifiers:
  [-sequence]          seqall     (Gapped) sequence(s) filename and optional
                                  format, or reference (input USA)
  [-outseq]            seqoutall  [<sequence>.<format>] Sequence set(s)
                                  filename and optional format (output USA)

   Additional (Optional) qualifiers: (none)
   Advanced (Unprompted) qualifiers:
   -feature            boolean    Use feature information
   -firstonly          boolean    Read one sequence and stop

   General qualifiers:
   -help               boolean    Report command line options. More
                                  information on associated and general
                                  qualifiers can be found with -help -verbose
__HELP__
	exit;
    }

    msg_error("Reads and writes (returns) sequences\n");

    if(length($USA) < 1){
	msg_error("Input (gapped) sequence(s): ");
	$USA = <STDIN>;
	chomp($USA);
    }

    unless(length($osformat)){
        if ($USA =~ /(.+)::(.+)/) {
	    ($osformat, $USA) = ($1, $2);
        }
        if ($outseq =~ /(.+)::(.+)/) {
    	    ($osformat, $outseq) = ($1, $2);
        }
        if ($outseq =~ /(.+):(.+)/) {
	    ($osformat, $outseq) = ($1,$2);;
        }
    }

    my $ua = LWP::UserAgent->new;
    my $request = $ua->get("http://emboss.bioinformatics.nl/cgi-bin/emboss/seqret");
    my $form = HTML::Form->parse($request->decoded_content, $request->base);

    if (-e $USA) {
	$form->value("sequence.file","$USA");
    } else {
	$form->value("sequence.db","$USA");
    }

    $form->value('outseq',"${osformat}::outseq") if $osformat;
    $form->value('feature','1') if $feature;
    $form->value('firstonly','1') if $firstonly;

    my $res = $ua->request($form->click('_run'));
    while ($res->is_redirect) {
	my $url = $res->header('Location');
	$res = $ua->request(HTTP::Request->new(GET => $url));
    }

    my $query = CGI->new;
    my $result = $res->content();
    $result =~ m/<pre>(.+)<\/pre>/s;
    $result = $query->unescapeHTML($1);
    while ($result =~ /<pre>/) {
	$result =~ m/^(.+)<\/pre>.+<pre>(.+)/sg;
	$result = "$1\n$2";
    }

    while ($result =~ /Warning/) {
	$result =~ /(Warning.+)/;
	msg_error($1,"\n");
	$result =~ /$1\n+(.+)/s;
	$result = $1;
    }

    if ($result =~ /with -auto/) {
	my @error = split(/:/,$USA);
	msg_error("Error: Failed to open filename \"$error[0]\"\n");
	msg_error("Error: Unable to read sequence \"$USA\"\n");
	msg_error("Died: seqret terminated: Bad value for '-sequence' and no prompt\n");
    } else {
	$result =~ /\s+(.{6})/;
	my $name = $1;
        chomp($name);
	$osformat = "fasta" unless $osformat;
	
	if ($outseq eq "") {
	    msg_error("output sequence(s) [$name.$osformat]: ");
	    $outseq = <STDIN>;
	    chomp($outseq);
	}
	
	if (lc($outseq) eq "stdout") {
	    msg_send("$result\n");
	} elsif ($outseq) {
	    open(RESULT,">$outseq");
	    print RESULT "$result";
	    close(RESULT);
	} else {
	    open(RESULT,">$name.$osformat");
	    print RESULT "$result";
	    close(RESULT);
	}
    }
}





1;
