#:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
# This file is part of G-language Genome Analysis Environment package
#
#     Copyright (C) 2001-2011 Keio University
#:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
# 
#   $Id: Rcmd.pm,v 1.3 2002/07/30 17:40:56 gaou Exp $
#
# G-language GAE is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
# 
# G-language GAE is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
# See the GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public
# License along with G-language GAE -- see the file COPYING.
# If not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
# 
#END_HEADER
#
# written by Kazuharu Arakawa <gaou@sfc.keio.ac.jp> at
# G-language Project, Institute for Advanced Biosciences, Keio University.
#

package Rcmd;

use strict;
use base qw(Rcmd::Clustering Rcmd::Normality);
use autouse 'Carp'=>qw(croak);
use autouse 'File::Copy'=>qw(copy);

use File::Temp;
our $VERSION = '1.22';

######################################
#       Let the code begin...
######################################

sub new {
    my $pkg = shift;
    my $save = shift || '--no-save'; 
    my $output = shift;

    my $uniq = time() . rand(1000);
    my $cmd = "/tmp/$uniq.cmd";
    my $log = "/tmp/$uniq.log";

    my $tmpcmd = "/tmp/$uniq.tmpcmd";
    my $tmplog = "/tmp/$uniq.tmplog";

    my $rPath;
    eval {
	$rPath = `/usr/bin/which R`;
    };

    unless(length($rPath)){
	croak("R language not found in your system.");
	return;
    }

    my $this = {};

    $this->{cmd}        = $cmd; 
    $this->{log}        = $log;
    $this->{tmpcmd}     = $tmpcmd;
    $this->{tmplog}     = $tmplog;
    $this->{sessioncmd} = $cmd;
    $this->{sessionlog} = $log;
    $this->{output}     = $output;
    $this->{save}       = $save;
    $this->{tmpdat}     =[];

    return bless $this;
}

sub set_mode{
    my $this = shift;
    my $option = shift;

    if($option eq 'tmp'){
	$this->{cmd} = $this->{tmpcmd};
	$this->{log} = $this->{tmplog};
    }elsif($option eq 'silent'){
	$this->{silent} = 1;
    }else{
	unlink($this->{tmpcmd});
	unlink($this->{tmplog});
	$this->{cmd} = $this->{sessioncmd};
	$this->{log} = $this->{sessionlog};
    }

}

sub DESTROY{
    my $this = shift;
    if (length $this->{output}){
	copy($this->{cmd}, $this->{output});
    }
    unlink $this->{cmd};
    unlink $this->{log};
    unlink $this->{tmpcmd};
    unlink $this->{tmplog};

}

sub array{
    my $this = shift;
    my $val = shift;
    my @array = @_;

    my $fh = File::Temp->new(SUFFIX=>'.dat');
    print $fh join(",", @array);

    open(my $fh2, '>>' . $this->{cmd});
    print $fh2 "$val <- scan\(\"" . $fh->filename . "\", sep\=\",\"\)\n";
    close($fh2);
    push(@{$this->{tmpdat}}, $fh);
}

sub sarray{
    my $this = shift;
    my $val = shift;
    my @array = @_;

    my $fh = File::Temp->new(SUFFIX=>'.dat');
    print $fh join(" ", @array), "\n";

    open(my $fh2, '>>' . $this->{cmd});
    print $fh2 "$val <- scan\(\"" . $fh->filename . "\"\, character\(\)\)\n";
    close($fh2);
    push(@{$this->{tmpdat}}, $fh);
}
    

sub exec{
    my $this = shift;
    my @tmp = @_;
    my $tmprequest = join("\n", @_, '');
    my $request = '';

    foreach my $line (split(/\n/, $tmprequest)){
	if (length $line > 1023){
	    $line =~ s/,/,\n/g;
	}
	$request .= $line . "\n";
    }
    my $data = '';

    open(my $fh, '>>' . $this->{cmd});
    print $fh $request;
    close($fh);

    system("/usr/bin/env R " . $this->{save} . " --slave < "
	   . $this->{cmd} . " > " . $this->{log} . ' 2>&1');

    open(my $fh2, $this->{log});
    while(<$fh2>){
	if (/\[(\d+).*\] +(.*)/){
	    if ($1 > 1){
		$data .= ' ' . $2;
	    }else{
		$data = $2;
	    }
	}elsif(/Error/){
	    print STDERR $_;
	    while(<$fh2>){
		print STDERR $_;
	    }
	    die("Error in R, exiting...\n");
	}elsif(/Warning/){
	    unless($this->{silent} == 1){
		print STDERR $_;
		while(<$fh2>){
		    print STDERR $_;
		    last;
		}
		warn("Warning in R...\n");
	    }
	}elsif(/^Read \d+ items/){

	}else{
	    unless($this->{silent} == 1){
		print STDERR $_;
	    }
	}

    }
    close($fh2);

    if (wantarray()){
	return split(/\s+/,$data);
    }else{
	return $data;
    }
}




1;
__END__

##############################
#       Perldoc
##############################

=head1 NAME

Rcmd - Perl interface for R language

=head1 SYNOPSIS

  use Rcmd;

  $rcmd = new Rcmd;

  @result = $rcmd->exec("<R commands>","<next R command>");  

=head1 DESCRIPTION

=cut 

=head2 $rcmd = new Rcmd()

  Name: $rcmd = new Rcmd()   -   create an instance of R language session

  This module is a simple interface to open-source R statistics language
  availabe under GPL at http://www.r-project.org/.

  Rcmd enables Perl manipulation of the R language by simply executing them
  through $rcmd->exec() function. Input is an array of R commands.

  ex:

  print $rcmd->exec(
                  "x = 5", 
                  "y = 4", 
                  "z = x * y", 
                  "z"
		   );

  Returned values are always an array. Therefore, in case the returned value
  is only one, the value is accessible as:

  @val = $rcmd->exec("y");

  print $val[0];

  All the values are saved in each session. Thus,

    $val1 = $rcmd->exec( "x = 5" , "x" );
    $val2 = $rcmd->exec( "x");

  will output "5" for both $val1 and $val2.

  Obviously, it is also possible to use perl variables, as:

    $i = 3;

    print $rcmd->exec("x = x * $i","x");

  The strength of R graphing abilities can be accessed as:

  @array = $rcmd->exec(
		       "postscript(\"/tmp/out.ps\")", 
		       "x = c(1:10)",
		       "y = c(3,6,3,5,8,0,1,9,2,6)",
		       "plot(x,y)",
		       "z = lsfit(x,y)",
		       "abline(z)",
		       "y"
		       );

  system("gs /tmp/out.ps");

  You can easily add Perl array data by $rcmd->array() for numerical data, or
  by $rcmd->sarray() for character data. For example:

    $rcmd->array("array1", 1, 2, 3, 4, 5);

  or

    $rcmd->sarray("array2", "a", "b", "c", "d", "e");

  Note: We recommend using R/S Perl (http://www.omegahat.org/RSPerl/) instead
  of this module in UNIX environment. Follow instructions of the above web site.

=cut 

=head1 AUTHOR

Kazuharu Arakawa, gaou@sfc.keio.ac.jp

=cut
