#:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
# This file is part of G-language Genome Analysis Environment package
#
#     Copyright (C) 2001-2014 Keio University
#:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
# 
# G-language GAE is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
# 
# G-language GAE is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
# See the GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public
# License along with G-language GAE -- see the file COPYING.
# If not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
# 
#END_HEADER
#
# written by Kazuharu Arakawa <gaou@sfc.keio.ac.jp> at
# G-language Project, Institute for Advanced Biosciences, Keio University.
#

package G::Shell::Log;

use strict;
use base qw(Exporter);

our @EXPORT = qw(
		 make_log
		 );

#::::::::::::::::::::::::::::::
#          Perldoc
#::::::::::::::::::::::::::::::

=head1 NAME

  G::Shell::Log - G-language Shell helper module for session logging.

=head1 DESCRIPTION

  This class is a part of G-language Genome Analysis Environment, 
  providing functions for the logging of sessions in G-language Shell.

=cut


#::::::::::::::::::::::::::::::
#        Constants
#::::::::::::::::::::::::::::::

my $G_INTERNAL_LOG_HEADER = 
'
  my $G_INTERNAL_HASH = sdb_load("G_INTERNAL_OLD");
  foreach  my $G_INTERNAL_TMP1 (keys(%{$$G_INTERNAL_HASH})){
    my $G_INTERNAL_TMP2 = $$G_INTERNAL_HASH->{$G_INTERNAL_TMP1};
    if($G_INTERNAL_TMP1 =~ /^\$/){
        $G_INTERNAL_TMP1 =~ s/\$//g;
        ${"$G_INTERNAL_TMP1"} = $G_INTERNAL_TMP2;
    }
    if($G_INTERNAL_TMP1 =~ /^\@/){
        $G_INTERNAL_TMP1 =~ s/\@//g;
        @{"$G_INTERNAL_TMP1"} = @$G_INTERNAL_TMP2;
    }
    if($G_INTERNAL_TMP1 =~ /^\%/){
        $G_INTERNAL_TMP1 =~ s/\%//g;
        %{"$G_INTERNAL_TMP1"} = %$G_INTERNAL_TMP2;
    }
  }
';


#::::::::::::::::::::::::::::::
#    Let the code begin...
#::::::::::::::::::::::::::::::

=head2 makelog

  Name: makelog   -   activate logging in G-language Shell

  Description:
    G-language Shell logging command. "makelog" command 
    activates logging (and "rmlog" deactivates logging) 
    of the G-language Shell sessions.

    Alternatively, you can activate logging by specifying 
    '-l' option when starting G-language Shell.

    Log file will be saved in the working directory with the
    following name:
      G-languageSessionLog-$year-$month-$day.$hour-$minute-$second.pl

    This is a working Perl script file, complete with saved
    session workspace.

  Author: 
    Kazuharu Arakawa (gaou@sfc.keio.ac.jp)

  History:
   20070606-01 initial posting

=cut

sub make_log {
    require G::IO;

    my $G_INTERNAL_LOG = shift;

    my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = localtime(time);
    $mon ++;
    $year += 1900;

    my $filename = sprintf "G-languageSessionLog-%d-%d-%d.%d-%d-%d.pl", 
    $year, $mon, $mday, $hour, $min, $sec;

    open(FILE,">$filename") ||die($!);
    print FILE qq/
# This script is generated by G-language Shell 
# Date: $year-$mon-$mday $hour:$min:$sec
# Version: $G::IO::VERSION

use G;
use G::Shell::Log;

#::::::::::::::::::::::::::::::::::::::::::::
#    HEADER (do not edit the code below)
#::::::::::::::::::::::::::::::::::::::::::::

$G_INTERNAL_LOG_HEADER;

#::::::::::::::::::::::::::::::
#    Let the code begin...
#::::::::::::::::::::::::::::::

$G_INTERNAL_LOG/;

    close(FILE);

    return $filename;
}


1;

